#! /usr/bin/env python
# ==========================================================================
# Show analysis results for paper
#
# Copyright (C) 2024-2025 Juergen Knoedlseder
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# ==========================================================================
import os
import math
import gammalib
from pipelines import fitutils
from scipy     import stats


# =================== #
# Get result filename #
# =================== #
def get_result_filename(band, brems, ic, ia='ia-g-plaw', dge='plaws', prefitted='prefitted'):
    """
    Get result filename

    Parameters
    ----------
    band : str
        Energy band ('low', 'high', 'full')
    brems : str
        Bremsstrahlungs component ('none','map','gmap')
    ic : str
        Inverse Compton component ('none','map','gmap',[2D name])
    ia : str, optional
        In-flight annihilation component string
    dge : str, optional
        DGE spectral fitting string
    prefitted : str, optional
        Use prefitted results?

    Returns
    -------
    filename : str
        Result XML filename
    """
    # Set filename
    if band == 'low':
        energies = '00750-03000keV_4bins_wo26Al'
    elif band == 'high':
        energies = '03000-30000keV_4bins'
    elif band == 'full':
        energies = '16bins_wo26Al'
    if brems == 'map' or ic == 'map':
        map = 'map'
    elif brems == 'gmap' or ic == 'gmap':
        map = 'gmap'
    else:
        map = ic
    suffix = map
    if brems != 'none':
        suffix += '-brems-map'
    if ic != 'none':
        suffix += '-ic-map'
    if prefitted == None:
        fitmode = ''
    else:
        fitmode = '_%s' % (prefitted)
    filename = 'results_com_240x140_drwnorm2_%s_conv-%s_%s-' \
               'g14-3c9-3c3-p18-p05-cr-ve-x1-ca-ls-%s-' \
               'isofix_%s_bgdlixf_ni11%s.xml' % (energies, map, ia, suffix, dge, fitmode)

    # If file does not exist then try prepending results/
    if not os.path.isfile(filename):
        alt_filename = 'results/%s' % (filename)
        if os.path.isfile(alt_filename):
            filename = alt_filename

    # Return filename
    return filename


# =============== #
# Get upper limit #
# =============== #
def get_upper_limit(filename, source='IA', suffix=None):
    """
    Get upper limit

    Parameters
    ----------
    filename : str
        Result XML filename

    Returns
    -------
    ulimit : float
        Upper limit on parameter valu (None if file not found)
    """
    # Initialise upper limit
    ulimit = None

    # Convert filename into upper limit filename
    extension = '%s' % gammalib.tolower(source)
    if suffix != None:
        extension += '-%s' % (suffix)
    filename = filename.replace('results_','ulimit_').replace('.xml','_%s.log' % extension)

    # Continue only if file exists
    if os.path.isfile(filename):

        # Open logfile
        f = open(filename, 'r')

        # Loop over lines
        for line in f:

            # Extract number of iterations
            pos = line.find('Upper limit param. value ..:')
            if pos != -1:
                ulimit = float(line[pos+28:])
                break

        # Close file
        f.close()

    # ... otherwise print warning
    else:
        print('*** No upper limit file "%s" found.' % (filename))

    # Return upper limit
    return ulimit


# ===================== #
# Show IA model results #
# ===================== #
def show_ia_model(model, emin, emax, band, ulimit):
    """
    Show result

    Parameters
    ----------
    model : gammalib.GSkyModel
        IA sky model
    emin : gammalib.GEnergy
        Minimum energy
    emax : gammalib.GEnergy
        Maximum energy
    band : string
        Energy band
    ulimit : float
        Upper limit on Sigma (None if file not found)
    """
    # Set constants
    dgcpc  = 8178.0   # Distance of Galactic centre in pc (Gravity collaboration, 2019)
    s2fwhm = 2.35482
    ef2lum = gammalib.fourpi * (dgcpc * gammalib.pc2cm) * (dgcpc * gammalib.pc2cm)

    # Set energy band dependent scaling factors
    if band == 'low':
        scale_pre   = 1e-4
        scale_flux  = 1e-4
        scale_eflux = 1e-10
        scale_lum   = 1e36
    elif band == 'high':
        scale_pre   = 1e-6
        scale_flux  = 1e-5
        scale_eflux = 1e-10
        scale_lum   = 1e36
    elif band == 'full':
        if model.name() == 'IA':
            scale_pre   = 1e-5
            scale_flux  = 1e-4
            scale_eflux = 1e-10
            scale_lum   = 1e36
        elif model.name() == 'GC':
            scale_pre   = 1e-6
            scale_flux  = 1e-5
            scale_eflux = 1e-10
            scale_lum   = 1e36
        else:
            scale_pre   = 1e-4
            scale_flux  = 1e-4
            scale_eflux = 1e-10
            scale_lum   = 1e36

    # Print TS value
    print(' %s%.1f (%.1f sigma)' % (gammalib.parformat('Test statistic',1),
                                    model.ts(), math.sqrt(model.ts())))

    # Print location
    print(' %s%4.1f +/- %.1f deg' % (gammalib.parformat('GLON',1),
                                     model['GLON'].value(), model['GLON'].error()))
    print(' %s%4.1f +/- %.1f deg' % (gammalib.parformat('GLAT',1),
                                     model['GLAT'].value(), model['GLAT'].error()))

    # Optionally print extension
    if model.has_par('Sigma'):
        if ulimit != None:
            print(' %s<%4.1f deg' % (gammalib.parformat('Sigma',1), ulimit))
            print(' %s<%4.1f deg' % (gammalib.parformat('FWHM',1), s2fwhm*ulimit))
        else:
            if model['Sigma'].error() > 0.0:
                print(' %s%4.1f +/- %.1f deg' % (gammalib.parformat('Sigma',1),
                                                 model['Sigma'].value(),
                                                 model['Sigma'].error()))
                print(' %s%4.1f +/- %.1f deg' % (gammalib.parformat('FWHM',1),
                                                 s2fwhm*model['Sigma'].value(),
                                                 s2fwhm*model['Sigma'].error()))

            else:
                print(' %s%4.1f deg' % (gammalib.parformat('Sigma',1),
                                        model['Sigma'].value()))
                print(' %s%4.1f deg' % (gammalib.parformat('FWHM',1),
                                        s2fwhm*model['Sigma'].value()))

    # Print pivot energy, prefactor, spectral index, cutoff and curvature
    if model.has_par('PivotEnergy'):
        print(' %s%4.1f MeV' % (gammalib.parformat('Pivot energy',1), model['PivotEnergy'].value()))
    if model.has_par('Prefactor'):
        print(' %s%4.1f +/- %.1f x %.0e ph cm^-2 s^-1 MeV^-1' % (gammalib.parformat('Prefactor',1),
                                                                 model['Prefactor'].value()/scale_pre,
                                                                 model['Prefactor'].error()/scale_pre,
                                                                 scale_pre))
    if model.spectral().has_par('Normalization'):
        print(' %s%4.1f +/- %.1f x %.0e ph cm^-2 s^-1' % (gammalib.parformat('Normalization',1),
                                                          model.spectral()['Normalization'].value()/scale_flux,
                                                          model.spectral()['Normalization'].error()/scale_flux,
                                                          scale_flux))
    if model.has_par('Index'):
        print(' %s%4.1f +/- %.1f' % (gammalib.parformat('Spectral index',1),
                                     -model['Index'].value(), model['Index'].error()))
    if model.has_par('Energy'):
        print(' %s%4.1f +/- %.1f MeV' % (gammalib.parformat('Injection energy',1),
                                         model['Energy'].value(), model['Energy'].error()))
    if model.has_par('MaxEnergy'):
        print(' %s%4.1f +/- %.1f MeV' % (gammalib.parformat('Max. injection energy',1),
                                         model['MaxEnergy'].value(), model['MaxEnergy'].error()))
    if model.has_par('CutoffEnergy'):
        print(' %s%4.1f +/- %.1f MeV' % (gammalib.parformat('Cutoff Energy',1),
                                         model['CutoffEnergy'].value(),
                                         model['CutoffEnergy'].error()))
    if model.has_par('Curvature'):
        print(' %s%4.1f +/- %.1f' % (gammalib.parformat('Curvature',1),
                                     model['Curvature'].value(),
                                     model['Curvature'].error()))

    # Print photon and energy fluxes
    print(' %s%4.1f +/- %.1f x %.0e ph cm^-2 s^-1' % (gammalib.parformat('Photon flux',1),
                                                      model.flux(emin,emax)/scale_flux,
                                                      model.flux_error(emin,emax)/scale_flux,
                                                      scale_flux))
    print(' %s%4.1f +/- %.1f x %.0e erg cm^-2 s^-1' % (gammalib.parformat('Energy flux',1),
                                                       model.eflux(emin,emax)/scale_eflux,
                                                       model.eflux_error(emin,emax)/scale_eflux,
                                                       scale_eflux))

    # Print GC luminosity
    print(' %s%4.1f +/- %.1f x %.0e erg s^-1' % (gammalib.parformat('Luminosity at GC',1),
                                                 ef2lum*model.eflux(emin,emax)/scale_lum,
                                                 ef2lum*model.eflux_error(emin,emax)/scale_lum,
                                                 scale_lum))

    # Return
    return


# ============================================ #
# Show diffuse Galactic emission model results #
# ============================================ #
def show_dge_model(model, emin, emax, band, dl=60.0, db=20.0):
    """
    Show result

    Parameters
    ----------
    model : gammalib.GSkyModel
        IA sky model
    emin : gammalib.GEnergy
        Minimum energy
    emax : gammalib.GEnergy
        Maximum energy
    band : string
        Energy band
    dl : float, optional
        Longitude box length for flux integration
    db : float, optional
        Latitude box width for flux integration
    """
    # Set constants
    dgcpc  = 8178.0   # Distance of Galactic centre in pc (Gravity collaboration, 2019)
    s2fwhm = 2.35482
    ef2lum = gammalib.fourpi * (dgcpc * gammalib.pc2cm) * (dgcpc * gammalib.pc2cm)

    # Setup sky region for integration
    region = gammalib.GSkyRegionRectangle(266.40499, -28.93617, dl, db, -58.59867)

    # Set energy band dependent scaling factors
    if band == 'low':
        scale_flux  = 1e-4
        scale_eflux = 1e-10
        scale_lum   = 1e36
    elif band == 'high':
        scale_flux  = 1e-4
        scale_eflux = 1e-9
        scale_lum   = 1e36
    elif band == 'full':
        if model.name() == 'Inverse Compton disk':
            scale_flux  = 1e-4
            scale_eflux = 1e-9
        elif model.name() == 'Inverse Compton':
            scale_flux  = 1e-4
            scale_eflux = 1e-9
        else:
            scale_flux  = 1e-4
            scale_eflux = 1e-9

    # Print TS value
    print(' %s%.1f (%.1f sigma)' % (gammalib.parformat('Test statistic',1),
                                    model.ts(), math.sqrt(model.ts())))

    # Print spectral index and curvature
    if model.has_par('Index'):
        print(' %s%4.1f +/- %.1f' % (gammalib.parformat('Spectral index',1),
                                     -model['Index'].value(), model['Index'].error()))
    if model.has_par('Curvature'):
        print(' %s%4.1f +/- %.1f' % (gammalib.parformat('Curvature',1),
                                     model['Curvature'].value(), model['Curvature'].error()))

    # Print photon and energy flux
    print(' %s%4.1f +/- %.1f x %.0e ph cm^-2 s^-1' % (gammalib.parformat('Photon flux',1),
                                                      model.flux(region,emin,emax)/scale_flux,
                                                      model.flux_error(region,emin,emax)/scale_flux,
                                                      scale_flux))
    print(' %s%4.1f +/- %.1f x %.0e erg cm^-2 s^-1' % (gammalib.parformat('Energy flux',1),
                                                       model.eflux(region,emin,emax)/scale_eflux,
                                                       model.eflux_error(region,emin,emax)/scale_eflux,
                                                       scale_eflux))

    # Return
    return


# ================================= #
# Generate LaTeX point source table #
# ================================= #
def latex_ptsrc_table(models, latexname, emin, emax):
    """
    Generate LaTeX point source table

    Parameters
    ----------
    models : gammalib.GModels
        Model container
    latexname : string
        LaTeX table filename
    emin : gammalib.GEnergy
        Minimum energy
    emax : gammalib.GEnergy
        Maximum energy
    """
    # Print header
    print('=== %s ===' % (latexname))

    # Scales
    scale_pre   = 1e-6
    scale_flux  = 1e-4
    scale_eflux = 1e-10

    # Set list of point sources
    sources = [{'name': 'PKS1830', 'latexname': 'PKS~1830--210'},
               {'name': 'LS5039',  'latexname': 'LS~5039'},
               {'name': 'CygX1',   'latexname': 'Cyg~X-1'},
               {'name': 'Vela',    'latexname': 'Vela pulsar'},
               {'name': 'PKS0506', 'latexname': 'PKS~0506--612'},
               {'name': '3C273',   'latexname': '3C~273'},
               {'name': '3C279',   'latexname': '3C~279'},
               {'name': 'CenA',    'latexname': 'Cen~A'},
               {'name': 'GRO1411', 'latexname': 'GRO~J1411--64'},
               {'name': 'Circ',    'latexname': 'PSR~J1513--5908'}]

    # Get pivot energy
    epivot = 0.0
    for source in sources:
        if models.contains(source['name']):
            value = models[source['name']]['PivotEnergy'].value()
            if epivot == 0:
                epivot = value
            else:
                if epivot != value:
                    print('*** Pivot energy %3.1f for source %s differs from previous pivot energy %3.1f.' % (value, source['name'], epivot))

    # Open LaTeX file
    f = open(latexname, 'w')

    # Write table header
    f.write('\\begin{table}\n')
    f.write('\\centering\n')
    f.write('\\footnotesize\n')
    f.write('\\caption{\n')
    f.write('\\textbf{Point sources fitted in the analysis.}\n')
    f.write('$l$ and $b$ specify the Galactic longitudes and latitudes of the point sources, \n')
    f.write('TS gives the Test Statistic of the source detection, $k$ are the power law prefactors,\n')
    f.write('and $\Gamma$ are the power law indices.\n')
    f.write('Power-law prefactors $k$ are in units of $10^{%d}$ ph cm$^{-2}$ s$^{-1}$ MeV$^{-1}$ \n' % (math.log10(scale_pre)))
    f.write('and specified for a pivot energy of %3.1f MeV.\n' % (epivot))
    f.write('Photon and energy fluxes are specified for the 0.75--30 MeV energy band, with photon fluxes in units\n')
    f.write('of $10^{%d}$ ph cm$^{-2}$ s$^{-1}$ \n' % (math.log10(scale_flux)))
    f.write('and energy fluxes in units of $10^{%d}$ erg cm$^{-2}$ s$^{-1}$.}\n' % (math.log10(scale_eflux)))
    f.write('\\label{tab:supp-ptsrc}\n')
    f.write('\\centering\n')
    f.write('\\begin{tabular}{lccccccc}\n')
    f.write('\\hline\n')
    f.write('Source & $l$ & $b$ & TS & $k$ & $\Gamma$ & ')
    f.write('Photon flux & Energy flux \\\\\n')
    f.write('& (deg) & (deg) & & (ph cm$^{-2}$ s$^{-1}$ MeV$^{-1}$) & & ')
    f.write('(ph cm$^{-2}$ s$^{-1}$) & (erg cm$^{-2}$ s$^{-1}$) \\\\\n')
    f.write('\\hline\n')

    # Loop over sources
    for source in sources:
        if models.contains(source['name']):
            model = models[source['name']]
            f.write('%s & ' % (source['latexname']))
            f.write('%.2f & ' % (model['GLON'].value()))
            f.write('%.2f & ' % (model['GLAT'].value()))
            f.write('%.1f & ' % (model.ts()))
            f.write('%.1f$\pm$%.1f & ' % (model['Prefactor'].value()/scale_pre,
                                          model['Prefactor'].error()/scale_pre))
            f.write('%.1f$\pm$%.1f & ' % (-model['Index'].value(), model['Index'].error()))
            f.write('%.1f$\pm$%.1f & ' % (model.flux(emin,emax)/scale_flux,
                                          model.flux_error(emin,emax)/scale_flux))
            f.write('%.1f$\pm$%.1f '   % (model.eflux(emin,emax)/scale_eflux,
                                          model.eflux_error(emin,emax)/scale_eflux))
            f.write('\\\\\n')

    # Write table trailer
    f.write('\\hline\n')
    f.write('\\end{tabular}\n')
    f.write('\\end{table}\n')

    # Close LaTeX file
    f.close()

    # Return
    return


# ================================================= #
# Generate LaTeX table for different 511 keV models #
# ================================================= #
def latex_models_table(lows, highs, latexname):
    """
    Generate LaTeX table for different 511 keV models

    Parameters
    ----------
    lows : list of dict
        Dictionaries for low-energy bands
    highs : list of dict
        Dictionaries for high-energy bands
    latexname : string
        LaTeX table filename
    """
    # Set constants
    s2fwhm = 2.35482

    # Print header
    print('=== %s ===' % (latexname))

    # Open LaTeX file
    f = open(latexname, 'w')

    # Write table header
    f.write('\\begin{table}\n')
    f.write('\\footnotesize\n')
    f.write('\\caption{\\small\n')
    f.write('{\\bf Fit results using different Galactic bulge emission models.}\n')
    f.write('Results in the top section are for the 0.75--3 MeV band, results in the ')
    f.write('bottom section for the 3--30 MeV\n')
    f.write('band.\n')
    f.write('The column 2$\\Delta$logL indicates twice the worsening of the maximum ')
    f.write('log-likelihood values with\n')
    f.write('respect to the best-fitting 2-dimensional symmetric Gaussian with free ')
    f.write('spatial parameters.\n')
    f.write('95\% confidence level upper limits on the prefactor $k$ and the fluxes are ')
    f.write('given for non-significant\n')
    f.write('components assuming a spectral index of 2.0.\n')
    f.write('\\label{tab:supp-511keV-fit}}\n')
    f.write('\\centering\n')
    f.write('\\begin{tabular}{l c c c c c c c}\n')
    f.write('\\hline\\hline\n')
    f.write('Model & 2$\\Delta$logL & Component & TS & k & $\\Gamma$ & Ph.~flux & En.~flux \\\\\n')
    f.write('\\hline\n')

    # Loop over energy bands
    for i, results in enumerate([lows, highs]):

        # Set energy band information
        if i == 0:
            band        = 'low'
            scale_pre   = 1e-4
            scale_flux  = 1e-5
            scale_eflux = 1e-10
            emin = gammalib.GEnergy(0.75,'MeV')
            emax = gammalib.GEnergy(3.00,'MeV')
            f.write('\\multicolumn{8}{c}{----- 0.75--3 MeV -----} \\\\\n')
        else:
            band        = 'high'
            scale_pre   = 1e-6
            scale_flux  = 1e-5
            scale_eflux = 1e-10
            emin = gammalib.GEnergy(3.00,'MeV')
            emax = gammalib.GEnergy(30.00,'MeV')
            f.write('\\multicolumn{8}{c}{----- 3--30 MeV -----} \\\\\n')

        # Loop over model fit results
        for k, result in enumerate(results):

            # Set prefitted option
            if 'pre' in result:
                prefitted = result['pre']
            else:
                prefitted = 'prefitted'

            # Get result and log filenames
            filename = get_result_filename(band, 'none', result['ic'], ia=result['ia'], prefitted=prefitted)
            logname  = filename.replace('.xml', '.log')

            # Skip if filename not found
            skip = False
            if not os.path.isfile(filename):
                print('*** File "%s" not found.' % (filename))
                skip = True
            if not os.path.isfile(logname):
                print('*** File "%s" not found.' % (logname))
                skip = True
            if skip:
                continue

            # Load result
            models = gammalib.GModels(filename)

            # Get log filename for reference result
            refname = get_result_filename(band, 'none', result['ic']).replace('.xml', '.log')

            # Get results from log file
            log = fitutils.extract_ctlike_results(logname)
            ref = fitutils.extract_ctlike_results(refname)

            # Skip if result does not contain IA model
            if not models.contains('IA'):
                print('*** No model "IA" in file "%s".' % (filename))
                continue

            # Get model components
            model_ia = None
            model_nb = None
            model_bb = None
            model_gc = None
            if i == 1 and 'Skinner' in result['name'] and models.contains('IA') and not models.contains('GC'):
                model_gc = models['IA']
                if models.contains('BB'):
                    model_bb = models['BB']
            else:
                if models.contains('BB'):
                    model_nb = models['IA']
                    model_bb = models['BB']
                else:
                    model_ia = models['IA']
                if models.contains('GC'):
                    model_gc = models['GC']

            # Put model components in list
            components = [model_ia, model_gc, model_nb, model_bb]
            if 'Bouchet' in result['name']:
                names = ['Bulge',  'Core',  'Inner bulge',  'Outer bulge']
            else:
                names = ['Bulge',  'Core',  'Narrow bulge', 'Broad bulge']

            # Loop over model components
            first = True
            for j, model in enumerate(components):

                # Skip component if it does not exist
                if model == None:
                    continue

                # If model component is not significant then search upper prefactor limit
                ulimit = None
                if model['Index'].error() == 0.0:
                    ulimit = get_upper_limit(filename, model.name())
                if ulimit != None:
                    model['Prefactor'].value(ulimit)
                    model['Prefactor'].error(0.0)
                pre         = model['Prefactor'].value()/scale_pre
                pre_error   = model['Prefactor'].error()/scale_pre
                flux        = model.flux(emin,emax)/scale_flux
                flux_error  = model.flux_error(emin,emax)/scale_flux
                eflux       = model.eflux(emin,emax)/scale_eflux
                eflux_error = model.eflux_error(emin,emax)/scale_eflux

                # Write data
                if first:
                    f.write('%s & ' % (result['name']))
                    ts = 2.0*(log['logL'] - ref['logL'])
                    df = ref['Nfree'] - log['Nfree']
                    if ts > 0.0:
                        f.write('+%.1f & ' % (ts))
                    else:
                        f.write('%.1f & ' % (ts))
                    if df > 0:
                        sigma = stats.norm.isf(0.5*stats.chi2.sf(abs(ts), df))
                    elif df < 0:
                        sigma = stats.norm.isf(0.5*stats.chi2.sf(abs(ts), -df))
                    else:
                        sigma = 0.0
                    if sigma != 0.0:
                        if ts > 0.0:
                            print('%s%.2f (%d)' % (gammalib.parformat(result['name']), sigma, df))
                        else:
                            print('%s-%.2f (%d)' % (gammalib.parformat(result['name']), sigma, df))
                    else:
                        print('%s (%d)' % (gammalib.parformat(result['name']), df))
                    first = False
                else:
                    f.write('&& ')
                f.write('%s & ' % (names[j]))
                f.write('%.1f & ' % (model.ts()))
                if pre_error > 0.0:
                    f.write('%.1f$\pm$%.1f & ' % (pre, pre_error))
                else:
                    f.write('$<$%.1f & ' % (pre))
                if model['Index'].error() > 0.0:
                    f.write('%.1f$\pm$%.1f & ' % (-model['Index'].value(), model['Index'].error()))
                else:
                    f.write('%.1f & ' % (-model['Index'].value()))
                if band == 'low':
                    fmt_value  = '%.0f$\pm$%.0f & '
                    fmt_ulimit = '$<$%.0f & '
                else:
                    fmt_value  = '%.1f$\pm$%.1f & '
                    fmt_ulimit = '$<$%.1f & '
                if flux_error > 0.0:
                    f.write(fmt_value % (flux, flux_error))
                else:
                    f.write(fmt_ulimit % (flux))
                if eflux_error > 0.0:
                    f.write('%.1f$\pm$%.1f ' % (eflux, eflux_error))
                else:
                    f.write('$<$%.1f ' % (eflux))
                f.write('\\\\\n')

        # Write separator
        if i == 0:
            f.write('\\hline\n')
        else:
            f.write('\\hline\\hline\n')

    # Write table trailer
    f.write('\\end{tabular}\n')
    f.write('\\footnotesize{\n')
    f.write('\\begin{flushleft}\n')
    f.write('For the 0.75--3 MeV band, $k$ is in units of $10^{-4}$ ph cm$^{-2}$ s$^{-1}$ MeV$^{-1}$ ')
    f.write('and derived\n')
    f.write('for a pivot energy of 1.5 MeV; for the 3--30 MeV band, $k$ is in units of ')
    f.write('$10^{-6}$ ph cm$^{-2}$ s$^{-1}$ MeV$^{-1}$\n')
    f.write('and derived for a pivot energy of 9.5 MeV.\n')
    f.write('Photon fluxes are in units of $10^{-5}$ ph cm$^{-2}$ s$^{-1}$ and energy fluxes\n')
    f.write('in units of $10^{-10}$ erg cm$^{-2}$ s$^{-1}$.\n')
    f.write('\\end{flushleft}\n')
    f.write('}\n')
    f.write('\\end{table}\n')

    # Close LaTeX file
    f.close()

    # Return
    return


# ============================================= #
# Generate LaTeX table for temporal variability #
# ============================================= #
def latex_temporal_table(lows, highs, latexname):
    """
    Generate LaTeX table for temporal variability

    Parameters
    ----------
    lows : list of dict
        Dictionaries for low-energy bands
    highs : list of dict
        Dictionaries for high-energy bands
    latexname : string
        LaTeX table filename
    """
    # Set constants
    s2fwhm = 2.35482

    # Print header
    print('=== %s ===' % (latexname))

    # Open LaTeX file
    f = open(latexname, 'w')

    # Write table header
    f.write('\\begin{table}\n')
    f.write('\\footnotesize\n')
    f.write('\\caption{\\small\n')
    f.write('{\\bf Fit results for four temporal bins.}\n')
    f.write('Results in the top section are for the 0.75--3 MeV band, results in the ')
    f.write('bottom section for the 3--30 MeV\n')
    f.write('band.\n')
    f.write('\\label{tab:supp-temporal-fit}}\n')
    f.write('\\centering\n')
    f.write('\\begin{tabular}{l c c c c c}\n')
    f.write('\\hline\\hline\n')
    f.write('MJD & TS & k & $\\Gamma$ & Photon flux & Energy flux \\\\\n')
    f.write('\\hline\n')

    # Loop over energy bands
    for i, results in enumerate([lows, highs]):

        # Set energy band information
        if i == 0:
            band        = 'low'
            scale_pre   = 1e-4
            scale_flux  = 1e-5
            scale_eflux = 1e-10
            emin = gammalib.GEnergy(0.75,'MeV')
            emax = gammalib.GEnergy(3.00,'MeV')
            f.write('\\multicolumn{6}{c}{----- 0.75--3 MeV -----} \\\\\n')
        else:
            band        = 'high'
            scale_pre   = 1e-6
            scale_flux  = 1e-5
            scale_eflux = 1e-10
            emin = gammalib.GEnergy(3.00,'MeV')
            emax = gammalib.GEnergy(30.00,'MeV')
            f.write('\\multicolumn{6}{c}{----- 3--30 MeV -----} \\\\\n')

        # Loop over model fit results
        for k, result in enumerate(results):

            # Get result and log filenames
            resname  = get_result_filename(band, 'none', result['ic'], ia=result['ia'])
            filename = 'temporal/%s/%s' % (result['bin'], resname)
            logname  = filename.replace('.xml', '.log')

            # Skip if filename not found
            skip = False
            if not os.path.isfile(filename):
                print('*** File "%s" not found.' % (filename))
                skip = True
            if not os.path.isfile(logname):
                print('*** File "%s" not found.' % (logname))
                skip = True
            if skip:
                continue

            # Load result
            models = gammalib.GModels(filename)

            # Get results from log file
            log = fitutils.extract_ctlike_results(logname)

            # Skip if result does not contain IA model
            if not models.contains('IA'):
                print('*** No model "IA" in file "%s".' % (filename))
                continue

            # Get model
            model = models['IA']

            # Write data
            f.write('%s & ' % (result['name']))
            f.write('%.1f & ' % (model.ts()))
            f.write('%.1f$\pm$%.1f & ' % (model['Prefactor'].value()/scale_pre,
                                          model['Prefactor'].error()/scale_pre))
            f.write('%.1f$\pm$%.1f & ' % (-model['Index'].value(), model['Index'].error()))
            if band == 'low':
                f.write('%.0f$\pm$%.0f & ' % (model.flux(emin,emax)/scale_flux,
                                              model.flux_error(emin,emax)/scale_flux))
            else:
                f.write('%.1f$\pm$%.1f & ' % (model.flux(emin,emax)/scale_flux,
                                              model.flux_error(emin,emax)/scale_flux))
            f.write('%.1f$\pm$%.1f ' % (model.eflux(emin,emax)/scale_eflux,
                                        model.eflux_error(emin,emax)/scale_eflux))
            f.write('\\\\\n')

        # Write separator
        if i == 0:
            f.write('\\hline\n')
        else:
            f.write('\\hline\\hline\n')

    # Write table trailer
    f.write('\\end{tabular}\n')
    f.write('\\footnotesize{\n')
    f.write('\\begin{flushleft}\n')
    f.write('For the 0.75--3 MeV band, $k$ is in units of $10^{-4}$ ph cm$^{-2}$ s$^{-1}$ MeV$^{-1}$ ')
    f.write('and derived\n')
    f.write('for a pivot energy of 1.5 MeV; for the 3--30 MeV band, $k$ is in units of ')
    f.write('$10^{-6}$ ph cm$^{-2}$ s$^{-1}$ MeV$^{-1}$\n')
    f.write('and derived for a pivot energy of 9.5 MeV.\n')
    f.write('Photon fluxes are in units of $10^{-5}$ ph cm$^{-2}$ s$^{-1}$ and energy fluxes\n')
    f.write('in units of $10^{-10}$ erg cm$^{-2}$ s$^{-1}$.\n')
    f.write('\\end{flushleft}\n')
    f.write('}\n')
    f.write('\\end{table}\n')

    # Close LaTeX file
    f.close()

    # Return
    return


# ============================================= #
# Generate LaTeX table for temporal variability #
# ============================================= #
def latex_table1(lows, highs, latexname):
    """
    Generate LaTeX table for temporal variability

    Parameters
    ----------
    lows : list of dict
        Dictionaries for low-energy bands
    highs : list of dict
        Dictionaries for high-energy bands
    latexname : string
        LaTeX table filename
    """
    # Print header
    print('=== %s ===' % (latexname))

    # Open LaTeX file
    f = open(latexname, 'w')

    # Write table header
    f.write('\\begin{table}\n')
    f.write('\\caption{Fitted excess fluxes in units of $10^{-5}$ \\pflux\\ for different time intervals.\n')
    f.write('\\label{table:flux}\n')
    f.write('\\centering\n')
    f.write('\\begin{tabular}{c c c}\n')
    f.write('\\hline\\hline\n')
    f.write('MJD & $0.75-3$~MeV & $3-30$~MeV \\\\\n')
    f.write('\\hline\n')

    # Loop over dictionaries
    for i, _ in enumerate(lows):

        # Get low energy flux
        filename   = get_result_filename('low', 'none', lows[i]['ic'], ia=lows[i]['ia'])
        if lows[i]['bin'] != None:
            filename = 'temporal/%s/%s' % (lows[i]['bin'], filename)
        models     = gammalib.GModels(filename)
        model      = models['IA']
        emin       = gammalib.GEnergy(0.75,'MeV')
        emax       = gammalib.GEnergy(3.00,'MeV')
        scale_flux = 1e-5
        flux_low   = '$%.1f\pm%.1f$' % (model.flux(emin,emax)/scale_flux,
                                        model.flux_error(emin,emax)/scale_flux)

        # Get high energy flux
        filename   = get_result_filename('high', 'none', highs[i]['ic'], ia=highs[i]['ia'])
        if highs[i]['bin'] != None:
            filename = 'temporal/%s/%s' % (highs[i]['bin'], filename)
        models     = gammalib.GModels(filename)
        model      = models['IA']
        emin       = gammalib.GEnergy(3.00,'MeV')
        emax       = gammalib.GEnergy(30.00,'MeV')
        scale_flux = 1e-5
        flux_high  = '$%.1f\pm%.1f$' % (model.flux(emin,emax)/scale_flux,
                                        model.flux_error(emin,emax)/scale_flux)

        # Write data
        f.write('%s & '     % (lows[i]['name']))
        f.write('%s & '     % (flux_low))
        f.write('%s \\\\\n' % (flux_high))

    # Write table trailer
    f.write('\\hline\n')
    f.write('\\end{tabular}\n')
    f.write('\\end{table}\n')

    # Close LaTeX file
    f.close()

    # Return
    return


# =========== #
# Show result #
# =========== #
def show_result(filename, latexname):
    """
    Show result

    Parameters
    ----------
    filename : string
        Result filename
    latexname : string
        LaTeX table filename
    """
    # Set energy boundaries
    if '00750-03000keV' in filename:
        emin = gammalib.GEnergy(0.75,'MeV')
        emax = gammalib.GEnergy(3.00,'MeV')
        band = 'low'
    elif '03000-30000keV' in filename:
        emin = gammalib.GEnergy(3.00,'MeV')
        emax = gammalib.GEnergy(30.00,'MeV')
        band = 'high'
    else:
        emin = gammalib.GEnergy(0.75,'MeV')
        emax = gammalib.GEnergy(30.00,'MeV')
        band = 'full'

    # Get log filename
    logname = filename.replace('.xml', '.log')

    # Skip if filename not found
    skip = False
    if not os.path.isfile(filename):
        print('*** File "%s" not found.' % (filename))
        skip = True
    if not os.path.isfile(logname):
        print('*** File "%s" not found.' % (logname))
        skip = True
    if skip:
        return

    # Get results from log file
    log = fitutils.extract_ctlike_results(logname)

    # Print results
    print('%s%s' % (gammalib.parformat('Fit status'), log['status']))
    print('%s%.3f' % (gammalib.parformat('Log-likelihood'), log['logL']))
    print('%s%.3f' % (gammalib.parformat('Nobs'), log['Nobs']))
    print('%s%.3f' % (gammalib.parformat('Npred'), log['Npred']))
    print('%s%.3f' % (gammalib.parformat('Nres'), log['Nobs']-log['Npred']))

    # Get models
    models = gammalib.GModels(filename)

    # Get upper limit for Sigma
    if band == 'high':
        ulimit = get_upper_limit(filename, suffix='sigma')
    else:
        ulimit = None

    # Extract components
    if models.contains('IA'):
        if models.contains('BB'):
            print('%s%s' % (gammalib.parformat('IA component'),'Narrow bulge (NB)'))
            show_ia_model(models['IA'], emin, emax, band, ulimit)
            print('%s%s' % (gammalib.parformat('IA component'),'Broad bulge (BB)'))
            show_ia_model(models['BB'], emin, emax, band, None)
        elif models.contains('GC'):
            print('%s%s' % (gammalib.parformat('IA component'),'Bulge (IA)'))
            show_ia_model(models['IA'], emin, emax, band, ulimit)
            print('%s%s' % (gammalib.parformat('IA component'),'Galactic centre (GC)'))
            show_ia_model(models['GC'], emin, emax, band, None)
        else:
            print('%s%s' % (gammalib.parformat('IA component'),'Bulge (IA)'))
            show_ia_model(models['IA'], emin, emax, band, ulimit)

    # Extract diffuse Galactic emissions
    if models.contains('Inverse Compton disk'):
        print('%s%s' % (gammalib.parformat('DGE component'),'Inverse Compton disk'))
        show_dge_model(models['Inverse Compton disk'], emin, emax, band)
    if models.contains('Inverse Compton'):
        print('%s%s' % (gammalib.parformat('DGE component'),'Inverse Compton'))
        show_dge_model(models['Inverse Compton'], emin, emax, band)

    # Generate point source result table
    if latexname != None:
        latex_ptsrc_table(models, latexname, emin, emax)

    # Return
    return


# ================ #
# Show fit results #
# ================ #
def show_fit_results():
    """
    Show fit results
    """
    # Set result filenames
    filelow  = get_result_filename('low',  'none', gre_low)
    filehigh = get_result_filename('high', 'none', gre_high)
    fileall  = get_result_filename('full', 'none', gre_full, ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-logp', dge='logps', prefitted=None)
    filefree = get_result_filename('full', 'none', gre_full, ia='gc-plaw-ia-g-logp', dge='logps')
    filemono = get_result_filename('full', 'none', gre_full, ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-mono0.10', dge='logps', prefitted=None)
    filedm   = get_result_filename('full', 'none', gre_full, ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-dm0.10', dge='logps', prefitted=None)
    filecont = get_result_filename('full', 'none', gre_full, ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-cont10.10', dge='logps', prefitted=None)

    # Set result list dictionary
    results = [{'name': 'Low-energy band analysis',                    'filename': filelow,  'latex': 'ptsrc_low.tex'},
               {'name': 'High-energy band analysis',                   'filename': filehigh, 'latex': 'ptsrc_high.tex'},
               {'name': 'Full band analysis (Fixed spatial to bands)', 'filename': fileall,  'latex': 'ptsrc_full.tex'},
               {'name': 'Full band analysis (Free spatial)',           'filename': filefree, 'latex': None},
               {'name': 'Full band analysis (Mono-energetic Bulge)',   'filename': filemono, 'latex': None},
               {'name': 'Full band analysis (Dark matter Bulge)',      'filename': filedm,   'latex': None},
               {'name': 'Full band analysis (Continuum Bulge)',        'filename': filecont, 'latex': None},
              ]

    # Show results
    for result in results:
        print('=== %s ===' % result['name'])
        show_result(result['filename'], result['latex'])

    # Return
    return


# ============== #
# Create Table 1 #
# ============== #
def create_table1():
    """
    Create Table 1
    """
    # Set lists for LaTeX supplementary table for temporal variability
    lows  = [{'name':    'Full period', 'bin':             None, 'ic': gre_low, 'ia': 'ia-g-plaw'},
             {'name': '48393 -- 48926', 'bin': 'mjd48393-48926', 'ic': gre_low, 'ia': 'ia-g-l1.4-b0.2r3.3-plaw'},
             {'name': '48926 -- 49459', 'bin': 'mjd48926-49459', 'ic': gre_low, 'ia': 'ia-g-l1.4-b0.2r3.3-plaw'},
             {'name': '49459 -- 49992', 'bin': 'mjd49459-49992', 'ic': gre_low, 'ia': 'ia-g-l1.4-b0.2r3.3-plaw'},
             {'name': '49992 -- 50526', 'bin': 'mjd49992-50526', 'ic': gre_low, 'ia': 'ia-g-l1.4-b0.2r3.3-plaw'}]
    highs = [{'name':    'Full period', 'bin':             None, 'ic': gre_high, 'ia': 'ia-g-plaw'},
             {'name': '48393 -- 48926', 'bin': 'mjd48393-48926', 'ic': gre_high, 'ia': 'ia-l-0.12+b0.66-plaw'},
             {'name': '48926 -- 49459', 'bin': 'mjd48926-49459', 'ic': gre_high, 'ia': 'ia-l-0.12+b0.66-plaw'},
             {'name': '49459 -- 49992', 'bin': 'mjd49459-49992', 'ic': gre_high, 'ia': 'ia-l-0.12+b0.66-plaw'},
             {'name': '49992 -- 50526', 'bin': 'mjd49992-50526', 'ic': gre_high, 'ia': 'ia-l-0.12+b0.66-plaw'}]

    # Generate LaTeX supplementary table for temporal variability
    latex_table1(lows, highs, 'table1.tex')

    # Return
    return


# ============== #
# Create Table 2 #
# ============== #
def create_table2():
    """
    Create Table 2
    """
    # Set lists for LaTeX supplementary table for different 511 keV models
    lows  = [{'name': 'Skinner et al.~(2014)', 'ic': gre_low, 'pre': 'p',
              'ia'  : 'bb-g-l-0.0-b-0.0r8.7-plawx-gc-l-0.06-b0.05-plawx-ia-g-l1.1-b0.2r2.5-plaw'}]
    highs = [{'name': 'Skinner et al.~(2014)', 'ic': gre_high, 'pre': 'p',
              'ia'  : 'bb-g-l-0.0-b-0.0r8.7-plawx-gc-l-0.06-b0.05-plaw-ia-g-l1.1-b0.2r2.5-plawx'}]

    # Generate LaTeX table for different 511 keV models
    latex_models_table(lows, highs, 'table2.tex')

    # Return
    return


# =========================== #
# Show results of IA analysis #
# =========================== #
def show_ia_results():
    """
    Show results of IA analysis
    """
    # Set GREs
    global gre_low
    global gre_high
    global gre_sym_low
    global gre_sym_high
    global gre_full
    gre_low      = 'icmap35.0-80.0-03.0'
    gre_high     = 'icmap50.0-70.0-06.0'
    gre_sym_low  = 'icmap60.0-60.0-02.0'
    gre_sym_high = 'icmap60.0-60.0-06.0'
    gre_full     = 'icmaps358003-507006'

    # Show fit results
    show_fit_results()

    # Create tables
    create_table1()
    create_table2()

    # Return
    return


# ======================== #
# Main routine entry point #
# ======================== #
if __name__ == '__main__':

    # Show IA results
    show_ia_results()
