#! /usr/bin/env python
# ==========================================================================
# Shows SED of Core component
#
# Copyright (C) 2024-2025 Juergen Knoedlseder
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# ==========================================================================
import os
import sys
import math
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D
import gammalib
import cscripts


# ============ #
# Get filename #
# ============ #
def get_filename(band, brems, ic, prefix, ia='ia-g-plaw', dge='logps',
                 prefitted=False, source='ia', filetype='.fits'):
    """
    Get butterfly filename

    Parameters
    ----------
    band : str
        Energy band ('low', 'high', 'full')
    brems : str
        Bremsstrahlungs component ('none','map','gmap')
    ic : str
        Inverse Compton component ('none','map','gmap',[2D name])
    prefix : str
        Filename prefix
    ia : str, optional
        In-flight annihilation component string
    dge : str, optional
        DGE spectral fitting string
    prefitted : boolean, optional
        Use prefitted results?
    source : str, optional
        Resource string
    filetype : str, optional
        File type

    Returns
    -------
    filename : str
        Result XML filename
    """
    # Set filename
    if band == 'low':
        energies = '00750-03000keV_4bins_wo26Al'
    elif band == 'high':
        energies = '03000-30000keV_4bins'
    elif band == 'full':
        energies = '16bins_wo26Al'
    if brems == 'map' or ic == 'map':
        map = 'map'
    elif brems == 'gmap' or ic == 'gmap':
        map = 'gmap'
    else:
        map = ic
    suffix = map
    if brems != 'none':
        suffix += '-brems-map'
    if ic != 'none':
        suffix += '-ic-map'
    if prefitted:
        fitmode = '_prefitted'
    else:
        fitmode = ''
    if source != None:
        fitmode += '_%s' % (source)
    filename = '%s_com_240x140_drwnorm2_%s_conv-%s_%s-' \
               'g14-3c9-3c3-p18-p05-cr-ve-x1-ca-ls-%s-' \
               'isofix_%s_bgdlixf_ni11%s%s' % (prefix, energies, map, ia, suffix, dge, fitmode, filetype)

    # If file does not exist then try prepending results/
    if not os.path.isfile(filename):
        alt_filename = 'results/%s' % (filename)
        if os.path.isfile(alt_filename):
            filename = alt_filename

    # Return filename
    return filename


# ============== #
# Get FGL source #
# ============== #
def get_fgl_source(name):
    """
    Get FGL source

    Parameters
    ----------
    name : str
        Source name
    """
    # Initialise source
    source = None

    # Open 4FGL-DR3 catalogue
    cat = gammalib.GFits('$COMGITROOT/catalogues/fermi-lat-4fgl-dr3-v31.fits')

    # Gather energy bands from table
    ebounds  = gammalib.GEbounds()
    emins    = []
    emaxs    = []
    table    = cat['EnergyBounds']
    col_emin = table['LowerEnergy']
    col_emax = table['UpperEnergy']
    for i in range(table.nrows()):
        emin = col_emin[i]
        emax = col_emax[i]
        if emin not in emins and emax not in emaxs:
            emins.append(emin)
            emaxs.append(emax)
            ebounds.append(gammalib.GEnergy(emin,'MeV'),gammalib.GEnergy(emax,'MeV'))

    # Get catalogue table
    table = cat['LAT_Point_Source_Catalog']

    # Find source row
    col_name = table['Source_Name']
    for i in range(col_name.nrows()):
        if col_name[i] == name:

            # Extract information
            nbands      = ebounds.size()
            glon        = table['GLON'][i]
            glat        = table['GLAT'][i]
            spectype    = table['SpectrumType'][i]
            epivot      = table['Pivot_Energy'][i]
            pl_pre      = table['PL_Flux_Density'][i]
            pl_pre_err  = table['Unc_PL_Flux_Density'][i]
            pl_inx      = table['PL_Index'][i]
            pl_inx_err  = table['Unc_PL_Index'][i]
            lp_pre      = table['LP_Flux_Density'][i]
            lp_pre_err  = table['Unc_LP_Flux_Density'][i]
            lp_inx      = table['LP_Index'][i]
            lp_inx_err  = table['Unc_LP_Index'][i]
            lp_beta     = table['LP_beta'][i]
            lp_beta_err = table['Unc_LP_beta'][i]
            fluxes      = [table['Flux_Band'].real(i,k)      for k in range(nbands)]
            fluxes_low  = [-table['Unc_Flux_Band'].real(i,k) for k in range(0,2*nbands,2)]
            fluxes_high = [table['Unc_Flux_Band'].real(i,k) for k in range(1,2*nbands,2)]
            sed         = [table['nuFnu_Band'].real(i,k)    for k in range(nbands)]

            # Set upper limit in fluxes_high slot
            for k in range(nbands):
                if math.isnan(fluxes_low[k]):
                    fluxes_high[k] = fluxes[k] + 2.0 * fluxes_high[k]

            # Derive information
            sed_low  = []
            sed_high = []
            for k in range(nbands):
                sed_low.append(sed[k] * fluxes_low[k] / fluxes[k])
                sed_high.append(sed[k] * fluxes_high[k] / fluxes[k])

            # Build spectrum
            if 'PowerLaw' in spectype:
                spectrum = gammalib.GModelSpectralPlaw(pl_pre, -pl_inx, gammalib.GEnergy(epivot, 'MeV'))
                spectrum['Prefactor'].error(pl_pre_err)
                spectrum['Index'].error(pl_inx_err)
            else:
                spectrum = gammalib.GModelSpectralLogParabola(lp_pre, -lp_inx, gammalib.GEnergy(epivot, 'MeV'), -lp_beta)
                spectrum['Prefactor'].error(lp_pre_err)
                spectrum['Index'].error(lp_inx_err)
                spectrum['Curvature'].error(lp_beta_err)

            # Build record
            source = {'name'    :     name,
                      'glon'    :     glon,
                      'glat'    :     glat,
                      'spectrum': spectrum,
                      'ebounds' :  ebounds,
                      'fluxes'  :   fluxes, 'fluxes_low': fluxes_low, 'fluxes_high': fluxes_high,
                      'sed'     :      sed,    'sed_low'   : sed_low,    'sed_high'   : sed_high}

            # Break loop
            break

    # Return source
    return source


# =============== #
# Get HGPS source #
# =============== #
def get_hgps_source(name):
    """
    Get FGL source

    Parameters
    ----------
    name : str
        Source name
    """
    # Initialise source
    source = None

    # Open HGPS catalogue
    cat = gammalib.GFits('$COMGITROOT/catalogues/hgps_catalog_v1.fits')

    # Get catalogue table
    table = cat['HGPS_SOURCES']

    # Find source row
    col_name = table['Source_Name']
    for i in range(col_name.nrows()):
        if col_name[i] == name:

            # Extract information
            nbands      = table['N_Flux_Points'][i]
            glon        = table['GLON'][i]
            glat        = table['GLAT'][i]
            model       = table['Spectral_Model'][i]
            energies    = [table['Flux_Points_Energy'].real(i,k)*1.0e6       for k in range(nbands)]
            emins       = [table['Flux_Points_Energy_Min'].real(i,k)         for k in range(nbands)]
            emaxs       = [table['Flux_Points_Energy_Max'].real(i,k)         for k in range(nbands)]
            fluxes      = [table['Flux_Points_Flux'].real(i,k)*1.0e-6        for k in range(nbands)]
            fluxes_low  = [table['Flux_Points_Flux_Err_Lo'].real(i,k)*1.0e-6 for k in range(nbands)]
            fluxes_high = [table['Flux_Points_Flux_Err_Hi'].real(i,k)*1.0e-6 for k in range(nbands)]
            fluxes_ul   = [table['Flux_Points_Flux_UL'].real(i,k)*1.0e-6     for k in range(nbands)]
            fluxes_isul = [table['Flux_Points_Flux_Is_UL'].integer(i,k)      for k in range(nbands)]

            # Build energy bands from table
            ebounds = gammalib.GEbounds()
            for k in range(nbands):
                ebounds.append(gammalib.GEnergy(emins[k],'TeV'),gammalib.GEnergy(emaxs[k],'TeV'))

            # Derive SED information
            sed      = []
            sed_low  = []
            sed_high = []
            for k in range(nbands):
                conv = energies[k] * energies[k] * gammalib.MeV2erg
                sed.append(fluxes[k] * conv)
                if fluxes_isul[k] == 1:
                    sed_low.append(float('nan'))
                    sed_high.append(fluxes_ul[k] * conv)
                else:
                    sed_low.append(fluxes_low[k] * conv)
                    sed_high.append(fluxes_high[k] * conv)

            # Build spectrum
            if model == 'PL':
                epivot   = table['Energy_Spec_PL_Pivot'][i]
                pre      = table['Flux_Spec_PL_Diff_Pivot'][i] / 1.0e6
                pre_err  = table['Flux_Spec_PL_Diff_Pivot_Err'][i] / 1.0e6
                inx      = table['Index_Spec_PL'][i]
                inx_err  = table['Index_Spec_PL_Err'][i]
                spectrum = gammalib.GModelSpectralPlaw(pre, -inx, gammalib.GEnergy(epivot, 'TeV'))
                spectrum['Prefactor'].error(pre_err)
                spectrum['Index'].error(inx_err)
            elif model == 'ECPL':
                epivot   = table['Energy_Spec_ECPL_Pivot'][i]
                pre      = table['Flux_Spec_ECPL_Diff_Pivot'][i] * 1.0e-6
                pre_err  = table['Flux_Spec_ECPL_Diff_Pivot_Err'][i] * 1.0e-6
                inx      = table['Index_Spec_ECPL'][i]
                inx_err  = table['Index_Spec_ECPL_Err'][i]
                lam      = table['Lambda_Spec_ECPL'][i] * 1.0e-6
                lam_err  = table['Lambda_Spec_ECPL_Err'][i] * 1.0e-6
                spectrum = gammalib.GModelSpectralExpInvPlaw(pre, -inx, gammalib.GEnergy(epivot, 'TeV'), lam)
                spectrum['Prefactor'].error(pre_err)
                spectrum['Index'].error(inx_err)
                spectrum['InverseCutoffEnergy'].error(lam_err)
            else:
                spectrum = None

            # Build record
            source = {'name'    :     name,
                      'glon'    :     glon,
                      'glat'    :     glat,
                      'spectrum': spectrum,
                      'ebounds' :  ebounds,
                      'fluxes'  :   fluxes, 'fluxes_low': fluxes_low, 'fluxes_high': fluxes_high,
                      'sed'     :      sed,    'sed_low'   : sed_low,    'sed_high'   : sed_high}

            # Break loop
            break

    # Return source
    return source


# ===================================== #
# Extract the spectrum info from a file #
# ===================================== #
def get_spectrum_file(filename, source='IA'):
    """
    Extract the spectrum info from a file for plotting

    Parameters
    ----------
    filename : str
        Name of spectrum FITS file
    source : str, optional
        Source name

    Returns
    -------
    spec : dict
        Python dictionary defining spectral plot parameters
    """
    # Initialise dictionary
    spec = None

    # Get filename
    fname = gammalib.GFilename(filename)

    # If filename is a FITS file then extract spectrum from FITS file
    if fname.is_fits():
        fits = gammalib.GFits(filename)
        if fits.table(1).string('OBJECT') == source:
            spec = get_spectrum_fits(fits)

    # ... otherwise extract spectrum from model
    else:
        models = gammalib.GModels(filename)
        if models.contains(source):
            model  = models[source].spectral()
            spec   = get_spectrum_model(model)

    # Return dictionary
    return spec


# ============================================= #
# Extract the spectrum info from a GFits object #
# ============================================= #
def get_spectrum_fits(fits, tslimit=1.0):
    """
    Extract the spectrum info from a GFits object

    Parameters
    ----------
    fits : `~gammalib.GFits`
        Spectral GFits object
    tslimit : float, optional
        TS limit for data points

    Returns
    -------
    spec : dict
        Python dictionary defining spectral plot parameters
    """
    # Read spectrum objects
    table    = fits.table(1)
    c_energy = table['e_ref']
    c_ed     = table['e_min']
    c_eu     = table['e_max']
    c_flux   = table['ref_e2dnde']
    c_norm   = table['norm']
    c_eflux  = table['norm_err']
    c_upper  = table['norm_ul']
    c_ts     = table['ts']

    # Initialise arrays to be filled
    spec = {
        'energies'    : [],
        'flux'        : [],
        'ed_engs'     : [],
        'eu_engs'     : [],
        'e_flux'      : [],
        'ul_energies' : [],
        'ul_ed_engs'  : [],
        'ul_eu_engs'  : [],
        'ul_flux'     : [],
        'yerr'        : [],
    }

    # Determine if we can load the delta-log-likelihood profiles
    has_sedtype = table.has_card('SED_TYPE')
    load_dll    = False
    if has_sedtype:
        seds = table.card('SED_TYPE').string().split(',')

    # Loop over rows of the file
    nrows = table.nrows()
    for row in range(nrows):

        # Get Test Statistic, flux and flux error
        ts    = c_ts.real(row)
        norm  = c_norm.real(row)
        flx   = norm * c_flux.real(row)
        e_flx = flx * c_eflux.real(row)

        # If Test Statistic is larger than 1 and flux error is smaller than
        # flux then append flux plots ...
        if ts > tslimit and e_flx < flx:
            spec['energies'].append(c_energy.real(row)*1.0e6)
            spec['flux'].append(flx)
            spec['ed_engs'].append(c_ed.real(row)*1.0e6)
            spec['eu_engs'].append(c_eu.real(row)*1.0e6)
            spec['e_flux'].append(e_flx)

        # ... otherwise append upper limit
        else:
            spec['ul_energies'].append(c_energy.real(row)*1.0e6)
            spec['ul_flux'].append(flx*c_upper.real(row))
            spec['ul_ed_engs'].append(c_ed.real(row)*1.0e6)
            spec['ul_eu_engs'].append(c_eu.real(row)*1.0e6)

    # Set upper limit errors
    spec['yerr'] = [0.3 * x for x in spec['ul_flux']]

    # Return dictionary
    return spec


# ============================================ #
# Extract the spectrum from GModelSpectralBins #
# ============================================ #
def get_spectrum_model(model):
    """
    Extract the spectrum info from a GFits object

    Parameters
    ----------
    model : `~gammalib.GModelSpectralBins`
        Spectral bins model

    Returns
    -------
    spec : dict
        Python dictionary defining spectral plot parameters
    """
    # Initialise arrays to be filled
    spec = {
        'energies'    : [],
        'flux'        : [],
        'ed_engs'     : [],
        'eu_engs'     : [],
        'e_flux'      : [],
        'ul_energies' : [],
        'ul_ed_engs'  : [],
        'ul_eu_engs'  : [],
        'ul_flux'     : [],
        'yerr'        : [],
    }

    # Loop over bins
    nbins = model.bins()
    for bin in range(nbins):
        emin   = model.emin(bin).MeV()
        emax   = model.emax(bin).MeV()
        flux   = model.intensity(bin)
        error  = model.error(bin)
        emean  = math.sqrt(emin*emax)
        ed_eng = emean - emin
        eu_eng = emax - emean
        norm   = emean * emean * gammalib.MeV2erg
        flux  *= norm
        error *= norm
        spec['energies'].append(emean)
        spec['flux'].append(flux)
        spec['ed_engs'].append(ed_eng)
        spec['eu_engs'].append(eu_eng)
        spec['e_flux'].append(error)

    # Return dictionary
    return spec


# ======== #
# Plot SED #
# ======== #
def plot_sed(ax, model, erg=False, emin=0.75, emax=30.0, label='', color='red', butterfly=False):
    """
    Plot Spectral Energy Distribution

    Parameters
    ----------
    ax : pyplot
        Frame for plot
    model : `~gammalib.GSkyModel()`
        Sky model
    erg : boolean, optional
        Plot in erg cm^-2 s-1?
    emin : float, optional
        Minimum energy (MeV)
    emax : float, optional
        Maximum energy (MeV)
    label : string, optional
        Plot label string
    color : string, optional
        Color string
    butterfly : boolean, optional
        Show butterfly based on uncertainties
    """
    # Set flux conversion factor
    if erg:
        conv = gammalib.MeV2erg
    else:
        conv = 1.0

    # If spectral model is a node function then evaluate model at nodes.
    if model.spectral().classname() == 'GModelSpectralNodes':

        # Get number of nodes
        nodes = model.spectral().nodes()

        # Extract node energies
        energies = [model.spectral().energy(i) for i in range(nodes)]

        # Evaluate model at node energies
        y = [model.spectral().eval(energy) * \
             energy.MeV() * energy.MeV() * conv for energy in energies]

        # Evaluate errors
        yerr = []
        for i in range(nodes):
            intensity = model.spectral().intensity(i)
            error     = model.spectral().error(i)
            if intensity != 0.0:
                yerr.append(y[i] * error / intensity)
            else:
                yerr.append(0.0)

        # Plot SED
        x = [energy.MeV() for energy in energies]
        ax.errorbar(x, y, yerr=yerr, fmt='o', color=color, label=label)

    # If spectral model is a node function then evaluate model at nodes.
    elif model.spectral().classname() == 'GModelSpectralBins':

        # Get number of bins
        bins = model.spectral().bins()

        # Extract bin energies
        energies = [gammalib.GEnergy(math.sqrt(model.spectral().emin(i).MeV() *
                                               model.spectral().emax(i).MeV()), 'MeV') \
                                               for i in range(bins)]

        # Set energy error in MeV
        xerr_low = [energies[i].MeV() - model.spectral().emin(i).MeV() for i in range(bins)]
        xerr_up  = [model.spectral().emax(i).MeV() - energies[i].MeV() for i in range(bins)]

        # Evaluate model at bin energies
        y = [model.spectral().eval(energy) * \
             energy.MeV() * energy.MeV() * conv for energy in energies]

        # Evaluate errors
        yerr = []
        for i in range(bins):
            intensity = model.spectral().intensity(i)
            error     = model.spectral().error(i)
            if intensity != 0.0:
                yerr.append(y[i] * error / intensity)
            else:
                yerr.append(0.0)

        # Plot SED
        x = [energy.MeV() for energy in energies]
        ax.errorbar(x, y, xerr=[xerr_low, xerr_up], yerr=yerr, fmt='o', color=color, label=label)

    # Otherwise evaluate the model at 100 logarithmically spaced energies
    else:

        # If requested then show butterfly
        if butterfly:
            plot_butterfly_error(ax, model.spectral(), emin, emax, erg=erg, label=label, color=color)

        # ... otherwise simply plot a line
        else:

            # Set energies
            energies = gammalib.GEnergies(100, gammalib.GEnergy(emin, 'MeV'),
                                               gammalib.GEnergy(emax, 'MeV'))

            # Evaluate spectrum
            y = [model.spectral().eval(energy) * energy.MeV() * energy.MeV() * conv for energy in energies]

            # Plot spectrum
            x = [energy.MeV() for energy in energies]
            ax.plot(x, y, color=color, label=label)

    # Return
    return


# =========== #
# Plot result #
# =========== #
def plot_result(ax, filename, color='red', linestyle='-', linewidth=2, source='IA',
                emin=0.75, emax=30.0, n=100, label=None):
    """
    Plot result

    Parameters
    ----------
    ax : pyplot
        Plotting frame
    filename : str
        Butterfly FITS file
    color : str, optional
        Color
    linestyle : str, optional
        Linestyle
    linewidth : float, optional
        Line width
    source : str, optional
        Source name
    label : str, optional
        Label
    """
    # Set energies
    energies = gammalib.GEnergies(n, gammalib.GEnergy(emin, 'MeV'),
                                     gammalib.GEnergy(emax, 'MeV'))

    # Load models
    #print(filename)
    models = gammalib.GModels(filename)

    # Continue only if models contain source
    if models.contains(source):

        # Get spectral model
        model = models[source].spectral()

        # Set x vector
        x = [energy.MeV() for energy in energies]

        # Set y vector
        y = [model.eval(energy) * energy.MeV() * energy.MeV() * gammalib.MeV2erg for energy in energies]

        # Plot line
        ax.plot(x, y, color=color, linestyle=linestyle, linewidth=linewidth, label=label, zorder=-3)

    # Return
    return


# ============== #
# Plot butterfly #
# ============== #
def plot_butterfly(ax, filename, color='red', linestyle='-', linewidth=2, alpha=0.5, label=None):
    """
    Plot butterfly diagram

    Parameters
    ----------
    ax : pyplot
        Plotting frame
    filename : str
        Butterfly FITS file
    color : str, optional
        Color
    linestyle : str, optional
        Linestyle
    linewidth : float, optional
        Line width
    alpha : float, optional
        Alpha for butterfly (0 = no butterfly)
    label : str, optional
        Label
    """
    # Initialise arrays to be filled
    butterfly = {'butterfly_x' : [],
                 'butterfly_y' : [],
                 'line_x'      : [],
                 'line_y'      : []}

    # Open FITS file
    fits = gammalib.GFits(filename)

    # Get sensitivity table
    table = fits.table('BUTTERFLY')

    # Get relevant columns
    c_energy        = table['ENERGY']
    c_intensity     = table['INTENSITY']
    c_intensity_min = table['INTENSITY_MIN']
    c_intensity_max = table['INTENSITY_MAX']

    # Fill vectors
    nrows = table.nrows()
    for row in range(nrows):

        # Get conversion coefficient TeV -> erg
        conv = c_energy[row] * c_energy[row] * 1.0e6 * gammalib.MeV2erg

        # Compute upper edge of confidence band
        butterfly['butterfly_x'].append(c_energy[row]*1.0e6)
        butterfly['butterfly_y'].append(c_intensity_max[row] * conv)

        # Set line values
        butterfly['line_x'].append(c_energy[row]*1.0e6)
        butterfly['line_y'].append(c_intensity[row] * conv)

    # Loop over the rows backwards to compute the lower edge of the
    # confidence band
    for row in range(nrows-1,-1,-1):
        conv      = c_energy[row] * c_energy[row] * 1.0e6 * gammalib.MeV2erg
        low_error = max(c_intensity_min[row] * conv, 1e-26)
        butterfly['butterfly_x'].append(c_energy[row]*1.0e6)
        butterfly['butterfly_y'].append(low_error)

    # Plot butterfly
    if alpha > 0.0:
        ax.fill(butterfly['butterfly_x'], butterfly['butterfly_y'], color=color, alpha=alpha)

    # Plot line
    ax.plot(butterfly['line_x'], butterfly['line_y'], color=color, linestyle=linestyle,
            linewidth=linewidth, label=label, zorder=-3)

    # Return
    return


# ==================================== #
# Plot butterfly based on model errors #
# ==================================== #
def plot_butterfly_error(ax, model, emin, emax, n=100, erg=True, label=None, linestyle='-', linewidth=1,
                         color='red', alpha=0.5):
    """
    Plot butterfly based on model errors

    Parametes
    ---------
    ax : pyplot
        Plotting frame
    model : `~gammalib.GSpectralModel()`
        Spectral model
    emin : float
        Minimum energy (MeV)
    emax : float
        Maximum energy (MeV)
    n : int, optional
        Number of energies used for plotting
    erg : boolean, optional
        Plot in erg cm^-2 s-1?
    label : string, optional
        Plot label string
    linestyle : str, optional
        Line style
    linewidth : float, optional
        Line width
    color : string, optional
        Color string
    alpha : float, optional
        Alpha for butterfly
    """
    # Set flux conversion factor
    if erg:
        conv = gammalib.MeV2erg
    else:
        conv = 1.0

    # Set energies
    energies = gammalib.GEnergies(n, gammalib.GEnergy(emin, 'MeV'),
                                     gammalib.GEnergy(emax, 'MeV'))

    # Set x vector
    x = [energy.MeV() for energy in energies]

    # Set y vector
    y = [model.eval(energy) * energy.MeV() * energy.MeV() * conv for energy in energies]

    # Extract names of all free paramters in model
    pars = []
    for par in model:
        if par.is_free():
            pars.append(par.name())

    # Compute number of parameter change combinations
    ncomb = int(math.pow(2,len(pars))+0.5)

    # Loop over all combinations and append a model with changed parameters
    models = []
    for i in range(ncomb):
        mod = model.copy()
        k   = 1
        for par in pars:
            mod[par].remove_range()
            pos = (i & k == k)
            if pos:
                mod[par].value(model[par].value()+model[par].error())
            else:
                mod[par].value(model[par].value()-model[par].error())
            k *= 2
        models.append(mod)

    # Build butterfly vectors
    butter_x = []
    butter_y = []
    y_min    = []
    for i, energy in enumerate(energies):
        ymin = 1.0e30
        ymax = 0.0
        for m in models:
            y_val = m.eval(energy) * energy.MeV() * energy.MeV() * conv
            if y_val < ymin:
                ymin = y_val
            if y_val > ymax:
                ymax = y_val
        y_min.append(ymin)
        butter_x.append(x[i])
        butter_y.append(ymax)
    for i in range(n-1,-1,-1):
        butter_x.append(x[i])
        butter_y.append(y_min[i])

    # Plot model
    ax.plot(x, y, color=color, label=label, linestyle=linestyle, linewidth=linewidth)

    # Plot butterfly
    if alpha > 0.0:
        ax.fill(butter_x, butter_y, color=color, alpha=alpha)

    # Return
    return


# ========================= #
# Plot catalogue source SED #
# ========================= #
def plot_source_sed(ax, source, erg=True, color='blue', label=None, markersize=3, linewidth=1):
    """
    Plot catalogue source SED

    Parametes
    ---------
    ax : pyplot
        Plotting frame
    erg : boolean, optional
        Plot in erg cm^-2 s-1?
    color : str, optional
        Plotting color
    label : str, optional
        Plot label
    markersize : float, optional
        Marker size
    linewidth : float, optional
        Line width
    """
    # Set flux conversion factor
    if erg:
        conv1 = gammalib.MeV2erg
        conv2 = 1.0
    else:
        conv1 = 1.0
        conv2 = gammalib.erg2MeV

    # Initialise vectors
    x        = []
    y        = []
    xerrl    = []
    xerru    = []
    yerrl    = []
    yerru    = []
    ul_x     = []
    ul_y     = []
    ul_xerrl = []
    ul_xerru = []
    ul_yerr  = []

    # Get number of energy bins
    n = source['ebounds'].size()

    # Loop over energies
    for i in range(n):

        # Get flux and errors
        flux      = source['sed'][i]      * conv2
        flux_low  = source['sed_low'][i]  * conv2
        flux_high = source['sed_high'][i] * conv2

        # Handle upper limits
        if math.isnan(flux_low):
            emean = source['ebounds'].elogmean(i).MeV()
            ul_x.append(emean)
            ul_y.append(flux_high)
            ul_xerrl.append(emean - source['ebounds'].emin(i).MeV())
            ul_xerru.append(source['ebounds'].emax(i).MeV() - emean)
            ul_yerr.append(0.3*flux_high)

        # ... otherwise handle flux points
        else:
            emean = source['ebounds'].elogmean(i).MeV()
            x.append(emean)
            y.append(flux)
            yerrl.append(flux_low)
            yerru.append(flux_high)
            xerrl.append(emean - source['ebounds'].emin(i).MeV())
            xerru.append(source['ebounds'].emax(i).MeV() - emean)

    # Plot fluxes
    ax.errorbar(x, y, xerr=[xerrl,xerru], yerr=[yerrl,yerru], marker='o',
                color=color, markersize=markersize, linewidth=linewidth,
                linestyle='None', label=label)

    # Optionally plot upper limits
    if len(ul_x) > 0:
        ax.errorbar(ul_x, ul_y, xerr=[ul_xerrl,ul_xerru], yerr=ul_yerr, marker='o',
                    uplims=True, color=color, markersize=markersize, linewidth=linewidth,
                    linestyle='None')

    # Return
    return


# =============================== #
# Plot catalogue source butterfly #
# =============================== #
def plot_source_butterfly(ax, source, erg=False, color='blue', label=''):
    """
    Plot catalogue source butterfly

    Parametes
    ---------
    ax : pyplot
        Plotting frame
    source : dict
        Source dictionary
    erg : boolean, optional
        Plot in erg cm^-2 s-1?
    color : str, optional
        Plotting color
    label : str, optional
        Plot label
    """
    # Extract source attributes
    emin = source['ebounds'].emin().MeV()
    emax = source['ebounds'].emax().MeV()
    if label == '':
        label = source['name']

    # Plot butterfly
    plot_butterfly_error(ax, source['spectrum'], emin, emax, erg=erg, label=label, color=color)

    # Determine photon and energy flux
    flux  = source['spectrum'].flux(gammalib.GEnergy(0.511,'MeV'),gammalib.GEnergy(100.0,'TeV'))
    eflux = source['spectrum'].eflux(gammalib.GEnergy(0.511,'MeV'),gammalib.GEnergy(100.0,'TeV'))
    print('Total 511 keV - 100 TeV photon flux in %s: %e ph/cm2/s'  % (source['name'], flux))
    print('Total 511 keV - 100 TeV energy flux in %s: %e erg/cm2/s' % (source['name'], eflux))

    # Convert to luminosity for a source at the Galactic centre
    d = 8178.0 * gammalib.pc2cm  # Distance of Galactic centre in pc (Gravity collaboration, 2019)
    a = 4.0 * gammalib.pi * (d*d)
    print('Total 511 keV - 100 TeV  luminosity in %s: %e erg/s' % (source['name'], eflux*a))

    # Return
    return


# ======================== #
# Plot Fermi/LAT butterfly #
# ======================== #
def plot_fermi_butterfly(ax, source, check='', erg=False, color='blue', label='Fermi/LAT (4FGL-DR3)'):
    """
    Plot Fermi/LAT butterfly of 4FGL J1745.6-2859

    Parametes
    ---------
    ax : pyplot
        Plotting frame
    source : dict
        Source dictionary
    check : str, optional
        Check options
    erg : boolean, optional
        Plot in erg cm^-2 s-1?
    color : str, optional
        Plotting color
    label : str, optional
        Plot label
    """
    # 4FGL-DR3
    plot_butterfly_error(ax, source['spectrum'], 60.0, 500000.0, erg=erg, label=label, color=color)

    # Determine photon and energy flux
    flux  = source['spectrum'].flux(gammalib.GEnergy(0.511,'MeV'),gammalib.GEnergy(100.0,'TeV'))
    eflux = source['spectrum'].eflux(gammalib.GEnergy(0.511,'MeV'),gammalib.GEnergy(100.0,'TeV'))
    print('Total 511 keV - 100 TeV photon flux in %s: %e ph/cm2/s'  % (source['name'], flux))
    print('Total 511 keV - 100 TeV energy flux in %s: %e erg/cm2/s' % (source['name'], eflux))

    # Convert to luminosity for a source at the Galactic centre
    d = 8178.0 * gammalib.pc2cm  # Distance of Galactic centre in pc (Gravity collaboration, 2019)
    a = 4.0 * gammalib.pi * (d*d)
    print('Total 511 keV - 100 TeV  luminosity in %s: %e erg/s' % (source['name'], eflux*a))

    # Set curvature conversion
    if 'log10' in check:
        conv = 1.0/2.303
    else:
        conv = 1.0

    # Universal model
    if 'afardo' in check:
        spectrum = gammalib.GModelSpectralLogParabola(2.53e-12, -2.59, gammalib.GEnergy(4134.0, 'MeV'), -0.260*conv)
        spectrum['Prefactor'].error(0.05e-12)
        spectrum['Index'].error(0.03)
        spectrum['Curvature'].error(0.011)
        plot_butterfly_error(ax, spectrum, 100.0, 500000.0, erg=erg, label='Universal model: 100 MeV - 500 GeV', color='black')

    # Split model
    if 'afardo' in check:
        # 60 - 300 MeV
        spectrum = gammalib.GModelSpectralPlaw(13.3e-12, -1.745, gammalib.GEnergy(2454.0, 'MeV'))
        spectrum['Prefactor'].error(2.7e-12)
        spectrum['Index'].error(0.073)
        plot_butterfly_error(ax, spectrum, 60.0, 300.0, erg=erg, label='60 - 300 MeV', color='green')

        # 300 MeV - 3 GeV (I believe there is a typo in the prefactor value)
        if 'fixpre' in check:
            pre = 2.49e-12
        else:
            pre = 4.49e-12
        spectrum = gammalib.GModelSpectralLogParabola(pre, -2.58, gammalib.GEnergy(4134.0, 'MeV'), -0.234*conv)
        spectrum['Prefactor'].error(0.09e-12)
        spectrum['Index'].error(0.03)
        spectrum['Curvature'].error(0.022)
        plot_butterfly_error(ax, spectrum, 300.0, 3000.0, erg=erg, label='300 MeV - 3 GeV', color='blue')

        # 3-10 GeV
        spectrum = gammalib.GModelSpectralLogParabola(2.52e-12, -2.79, gammalib.GEnergy(4134.0, 'MeV'), -0.234*conv)
        spectrum['Prefactor'].error(0.07e-12)
        spectrum['Index'].error(0.10)
        spectrum['Curvature'].error(0.021)
        plot_butterfly_error(ax, spectrum, 3000.0, 10000.0, erg=erg, label='3 - 10 GeV', color='red')

        # 10-500 GeV
        spectrum = gammalib.GModelSpectralLogParabola(2.27e-12, -2.33, gammalib.GEnergy(4134.0, 'MeV'), -0.234*conv)
        spectrum['Prefactor'].error(0.06e-12)
        spectrum['Index'].error(0.06)
        spectrum['Curvature'].error(0.021)
        plot_butterfly_error(ax, spectrum, 10000.0, 500000.0, erg=erg, label='10 - 500 GeV', color='grey')

    # Return
    return


# =================== #
# Plot HESS butterfly #
# =================== #
def plot_hess_butterfly(ax, source, erg=False, color='green', label='HESS (HESS J1745-290)', choice='HGPS'):
    """
    Plot butterfly of HESS J1745-290

    Parametes
    ---------
    ax : pyplot
        Plotting frame
    source : dict
        Source dictionary
    check : str, optional
        Check options
    erg : boolean, optional
        Plot in erg cm^-2 s-1?
    color : str, optional
        Plotting color
    label : str, optional
        Plot label
    """
    # HGPS
    if 'HGPS' in choice:
        spectrum = gammalib.GModelSpectralExpInvPlaw(2.55e-18, -2.14, gammalib.GEnergy(1.0, 'TeV'), 9.345794e-8)
        spectrum['Prefactor'].error(0.04e-18)
        spectrum['Index'].error(0.02)
        spectrum['InverseCutoffEnergy'].error(1.746877e-8)
        plot_butterfly_error(ax, spectrum, 200000.0, 50000000.0, erg=erg, label=label, color=color)

    # Aharonian et al. (2009)
    else:
        spectrum = gammalib.GModelSpectralExpPlaw(2.55e-18, -2.10, gammalib.GEnergy(1.0, 'TeV'),
                                                                   gammalib.GEnergy(14.7, 'TeV'))
        spectrum['Prefactor'].error(0.06e-18)
        spectrum['Index'].error(0.04)
        spectrum['CutoffEnergy'].error(3.41)
        plot_butterfly_error(ax, spectrum, 200000.0, 50000000.0, erg=erg, label=label, color=color)

    # Determine photon and energy flux
    flux  = spectrum.flux(gammalib.GEnergy(0.511,'MeV'),gammalib.GEnergy(100.0,'TeV'))
    eflux = spectrum.eflux(gammalib.GEnergy(0.511,'MeV'),gammalib.GEnergy(100.0,'TeV'))
    print('Total 511 keV - 100 TeV photon flux in %s: %e ph/cm2/s'  % ('HESS J1745-290', flux))
    print('Total 511 keV - 100 TeV energy flux in %s: %e erg/cm2/s' % ('HESS J1745-290', eflux))

    # Convert to luminosity for a source at the Galactic centre
    d = 8178.0 * gammalib.pc2cm  # Distance of Galactic centre in pc (Gravity collaboration, 2019)
    a = 4.0 * gammalib.pi * (d*d)
    print('Total 511 keV - 100 TeV  luminosity in %s: %e erg/s' % ('HESS J1745-290', eflux*a))

    # Return
    return


# ======================================== #
# Plot Churazov et al. (2011) upper limits #
# ======================================== #
def plot_churazov(ax, color='black', label='INTEGRAL/SPI (Churazov et al. 2011)', markersize=3, linewidth=1):
    """
    Plot Churazov et al. (2011) upper limits

    Parameters
    ----------
    ax : pyplot
        Plotting frame
    color : str, optional
        Color of upper limits
    label : str, optional
        Label
    markersize : float, optional
        Marker size
    linewidth : float, optional
        Line width
    """
    # Set energy bins
    emin  = [0.520, 0.770, 2.000, 4.000]
    emax  = [0.770, 1.000, 4.000, 7.000]
    emean = [math.sqrt(emin[i]*emax[i]) for i in range(4)]

    # Set upper limits
    ulimit = [0.0001660, 0.0005132, 0.0005916, 0.0009975]

    # Convert MeV cm-2 s-1 to erg cm-2 s-1
    ulimit = [x * gammalib.MeV2erg for x in ulimit]

    # Set energy uncertainites
    xmin = [emean[i]-emin[i] for i in range(4)]
    xmax = [emax[i]-emean[i] for i in range(4)]

    # Set length of upper limit arrows
    yerr = [0.3 * x for x in ulimit]

    # Plot upper limits
    ax.errorbar(emean, ulimit, yerr=yerr, xerr=[xmin, xmax], uplims=True, marker='o', color=color,
                linestyle='none', markersize=markersize, linewidth=linewidth)

    # Return
    return


# =============================================== #
# Plot Malyshev et al. (2015) PICsIT upper limits #
# =============================================== #
def plot_malyshev(ax, color='black', label='INTEGRAL/PICsIT (Malyshev et al. (2015)', markersize=3, linewidth=1):
    """
    Plot Malyshev et al. (2015) PICsIT upper limits

    Parameters
    ----------
    ax : pyplot
        Plotting frame
    color : str, optional
        Color of upper limits
    label : str, optional
        Label
    markersize : float, optional
        Marker size
    linewidth : float, optional
        Line width
    """
    # Set energy bins
    emin  = [0.300, 0.514, 1.000, 2.000]
    emax  = [0.514, 1.000, 2.000, 4.000]
    emean = [math.sqrt(emin[i]*emax[i]) for i in range(4)]

    # Set upper limits
    ulimit = [8.14e-12, 2.45e-11, 5.99e-11, 1.48e-10]

    # Set energy uncertainites
    xmin = [emean[i]-emin[i] for i in range(4)]
    xmax = [emax[i]-emean[i] for i in range(4)]

    # Set length of upper limit arrows
    yerr = [0.6 * x for x in ulimit]

    # Plot upper limits
    ax.errorbar(emean, ulimit, yerr=yerr, xerr=[xmin, xmax], uplims=True, marker='o', color=color,
                linestyle='none', markersize=markersize, linewidth=linewidth, zorder=-3)

    # Return
    return


# ================ #
# Plot table model #
# ================ #
def plot_table_model(ax, filename, source='GC', color='blue', emin=0.75, emax=1.0e6, alpha=0.0):
    """
    Plot table model

    Parameters
    ----------
    ax : pyplot
        Plotting frame
    filename : str
        Table model file
    source : str, optional
        Source name
    color : str, optional
        Color
    emin : float, optional
        Minimum energy (MeV)
    emax : float, optional
        Maximum energy (MeV)
    alpha : float, optional
        Alpha for butterfly
    """
    # Set components
    components = [{'name': 'Inverse Compton',        'suffix' : 'ic', 'linestyle': 'dotted'},
                  {'name': 'In-flight annihilation', 'suffix' : 'ia', 'linestyle': 'dashed'},
                  {'name': 'Bremsstrahlung',         'suffix' : 'eb', 'linestyle': 'dashdot'}]

    # Load models
    models = gammalib.GModels(filename)

    # Continue only if models contain source
    if models.contains(source):

        # Get spectral model
        spectrum = models[source].spectral()

        # Plot butterfly
        plot_butterfly_error(ax, spectrum, emin, emax, n=1000, color=color, alpha=alpha)

        # Prepare for plotting of components
        xml       = gammalib.GXmlElement()
        spectrum.write(xml)
        tablefile = xml.attribute('file')

        # Plot components
        for component in components:

            # Create table model component filename
            comp_file = tablefile.replace('.fits', '_%s.fits' % component['suffix'])
            if not os.path.isfile(comp_file):
                print('*** File "%s" not found.' % (comp_file))
                continue

            # Create table model for component
            xml.attribute('file', comp_file)
            comp_spec = gammalib.GModelSpectralTable(xml)

            # Plot component butterfly
            plot_butterfly_error(ax, comp_spec, emin, emax, n=1000, color=color,
                                 linestyle=component['linestyle'], alpha=alpha)

    # Return components
    return components


# =============================== #
# Plot systematic uncertainty box #
# =============================== #
def plot_systematics(ax, source='IA', color='red', alpha=0.4, width=0.025, display='bars', spectrum=True):
    """
    Plot systematic uncertainty box

    Parameters
    ----------
    ax : pyplot
        Plotting frame
    source : string, optional
        Source
    color : str, optional
        Colour
    alpha : float, optional
        Alpha
    width : float, optional
        Uncertainty bar width
    display : str, optional
        Display style ('box' or 'bars')
    spectrum : boolean, optional
        Use spectrum instead of results
    """
    # Set SED string
    sed = '%s-%s-%s-%s' % ('gc-l-0.12+b0.66', 'bins3', 'ia-g-l1.4-b0.2r3.3', 'bins8')

    # Set different DGEs
    dges = [{'brems': 'none', 'ic': 'icmaps358003-507006'},
            {'brems': 'none', 'ic': 'icmaps60-06_60-02'},
            {'brems': 'none', 'ic': 'map'},
            {'brems': 'none', 'ic': 'gmap'},
            {'brems': 'map',  'ic': 'map'},
            {'brems': 'gmap', 'ic': 'gmap'}]

    # Initialise energies and minimum/maximum fluxes
    energies    = None
    flux_min    = None
    flux_max    = None
    ul_energies = []
    ul_ed_engs  = []
    ul_eu_engs  = []

    # Loop over DGEs
    for dge in dges:

        # Get result filename
        if spectrum:
            filename = get_filename('full', dge['brems'], dge['ic'], prefix='spectrum_bins', ia=sed,
                                    source=gammalib.tolower(source), filetype='.fits')
        else:
            filename = get_filename('full', dge['brems'], dge['ic'], prefix='results', ia=sed,
                                    source=None, filetype='.xml')

        # Continue only if file exists
        if os.path.isfile(filename):

            # Get spectrum
            spec = get_spectrum_file(filename, source=gammalib.toupper(source).lstrip('68_'))

            # Continue only if spectrum was found
            if spec != None:

                # Build vectors from results
                if spectrum:
                    spec_energies = []
                    #spec_flux     = []
                    spec_flux_max = []
                    spec_flux_min = []
                    for i in range(len(spec['energies'])):
                        spec_energies.append(spec['energies'][i])
                        #spec_flux.append(spec['flux'][i])
                        if display == 'bars':
                            spec_flux_min.append(spec['flux'][i]-spec['e_flux'][i])
                            spec_flux_max.append(spec['flux'][i]+spec['e_flux'][i])
                        else:
                            spec_flux_min.append(spec['flux'][i])
                            spec_flux_max.append(spec['flux'][i])
                    for i in range(len(spec['ul_energies'])):
                        ul_energies.append(spec['ul_energies'][i])
                        ul_ed_engs.append(spec['ul_ed_engs'][i])
                        ul_eu_engs.append(spec['ul_eu_engs'][i])
                        spec_energies.append(spec['ul_energies'][i])
                        #spec_flux.append(spec['ul_flux'][i])
                        spec_flux_min.append(spec['ul_flux'][i])
                        spec_flux_max.append(spec['ul_flux'][i])
                else:
                    spec_energies = spec['energies']
                    #spec_flux     = spec['flux']
                    spec_flux_min = spec['flux']
                    spec_flux_max = spec['flux']

                # Determine minimum and maximum flux values
                if flux_min == None:
                    energies = spec_energies
                    flux_min = list(spec_flux_min)
                    flux_max = list(spec_flux_max)
                else:
                    for i, flux in enumerate(spec_flux_min):
                        if flux < flux_min[i]:
                            flux_min[i] = flux
                    for i, flux in enumerate(spec_flux_max):
                        if flux > flux_max[i]:
                            flux_max[i] = flux

            # ... otherwise indicate that no spectrum was found
            else:
                print('--- No spectrum found for source "%s" in file "%s"' % (source,filename))

        # ... otherwise indicate that file is missing
        else:
            print('--- %s' % filename)

    # Plot uncertainty box
    if energies != None:
        if display == 'bars':
            for i, energy in enumerate(energies):
                if 'IA' in source and i > 2:
                    k          = ul_energies.index(energy)
                    energy_min = ul_ed_engs[k]
                    energy_max = ul_eu_engs[k]
                    yerr_min   = 0.3 * flux_min[i]
                    yerr_max   = 0.3 * flux_max[i]
                    eb1        = ax.errorbar([energy], [flux_min[i]], xerr=[[energy_min], [energy_max]],
                                             yerr=[yerr_min], uplims=True, color=color, alpha=alpha)
                    eb2        = ax.errorbar([energy], [flux_max[i]], xerr=[[energy_min], [energy_max]],
                                             yerr=[yerr_max], uplims=True, color=color, alpha=alpha)
                    eb1[-1][0].set_linestyle('--')
                    eb1[-1][1].set_linestyle('--')
                    eb2[-1][0].set_linestyle('--')
                    eb2[-1][1].set_linestyle('--')
                else:
                    ax.plot([energy, energy], [flux_min[i], flux_max[i]], linestyle='dashed', color=color, alpha=0.5)
        else:
            for i, energy in enumerate(energies):
                x = [(1.0-width)*energy, (1.0+width)*energy, (1.0+width)*energy, (1.0-width)*energy, (1.0-width)*energy]
                y = [flux_min[i], flux_min[i], flux_max[i], flux_max[i], flux_min[i]]
                if alpha > 0.0:
                    ax.fill(x, y, color=color, alpha=alpha, linewidth=0)
                else:
                    ax.plot(x, y, color=color, linewidth=1)

    # Return
    return


# ========================== #
# Show SED of Core component #
# ========================== #
def show_core(ax, flux=1.0e-3, color='blue', markersize=3, linewidth=1):
    """
    Show SED of Core component

    Parameters
    ----------
    ax : pyplot
        Plotting frame
    flux : float, optional
        Flux normalisation
    color : string, optional
        Colour for plots
    markersize : float, optional
        Marker size
    linewidth : float, optional
        Line width
    """
    # Initialise handles
    handles = []

    # Set parameters
    gc        = 'gc-l-0.12+b0.66'
    ia        = 'ia-g-l1.4-b0.2r3.3'
    icmaps    = 'icmaps358003-507006'
    fgl_color = 'green'
    spi_color = 'black'

    # Set SED filename
    comptel = '%s-%s-%s-%s' % (gc, 'bins3', ia, 'bins8')
    sedname = get_filename('full', 'none', icmaps, 'spectrum_bins', ia=comptel, source='68_gc')

    # Set table model fit
    model     = '%s-%s-%s-%s' % (gc, 'plaw-r3-n1000-fix-0.10', ia, 'logp')
    modelname = get_filename('full', 'none', icmaps, 'results', ia=model, source='mwl4gfl',
                             filetype='.xml')

    # Exit if SED file not found
    if not os.path.isfile(sedname):
        print('*** File "%s" not found.' % (sedname))
        return

    # Load COMPTEL SED
    sed = get_spectrum_file(sedname, source='GC')

    # Plot COMPTEL SED
    ax.errorbar(sed['energies'], sed['flux'],
                yerr=sed['e_flux'], xerr=[sed['ed_engs'], sed['eu_engs']],
                marker='o', color=color, linestyle='None',
                markersize=markersize, linewidth=linewidth)
    if len(sed['ul_energies']) > 0:
        ax.errorbar(sed['ul_energies'], sed['ul_flux'],
                    yerr=sed['yerr'], xerr=[sed['ul_ed_engs'], sed['ul_eu_engs']],
                    uplims=True, marker='o', color=color, linestyle='None',
                    markersize=markersize, linewidth=linewidth)

    # Append legend
    handles.append(Line2D([0], [0], marker='o', color=color, markersize=markersize,
                   linestyle='None', label='COMPTEL (Core)'))

    # Plot systematic uncertainty
    plot_systematics(ax, source='68_GC', color=color)

    # Plot INTEGRAL/PICsIT upper limits
    plot_malyshev(ax, color=spi_color, label='INTEGRAL/PICsIT (Malyshev et al. 2015)')

    # Append legend
    handles.append(Line2D([0], [0], marker='o', color=spi_color, markersize=markersize,
                   linestyle='None', label='INTEGRAL/PICsIT (Malyshev et al. 2015)'))

    # Plot 4FGL J1745.6-2859 SED
    fgl = get_fgl_source('4FGL J1745.6-2859')
    if fgl != None:

        # Plot SED
        plot_source_sed(ax, fgl, color=fgl_color, markersize=markersize)

        # Append legend
        handles.append(Line2D([0], [0], marker='o', color=fgl_color, markersize=markersize,
                       linestyle='None', label='Fermi/LAT (4FGL J1745.6-2859)'))

    # Plot table model
    if os.path.isfile(modelname):

        # Plot table model
        components = plot_table_model(ax, modelname, color=color)

        # Append legend
        handles.append(Line2D([0], [0], label='Total', color=color, linestyle='-',
                       linewidth=linewidth))
        for component in components:
            handles.append(Line2D([0], [0], label=component['name'], color=color,
                           linestyle=component['linestyle'], linewidth=linewidth))

    else:
        print('*** File "%s" not found.' % (modelname))

    # Set attributes
    ax.set_xscale('log')
    ax.set_yscale('log')
    ax.set_xlim([0.2,1.2e6])
    ax.set_ylim([1e-13,1e-9])
    ax.set_xlabel('Energy (MeV)', fontsize=6)
    ax.set_ylabel(r'E$^2$ $\times$ Flux (erg cm$^{-2}$ s$^{-1}$)', fontsize=6)

    # Add legends
    ax.legend(handles=handles, loc='lower right', fontsize=5)

    # Return
    return


# ========================== #
# Show SED of Core component #
# ========================== #
def show_core_sed():
    """
    Show SED of Core component
    """
    # Set figure size
    figwidth  = 91.5 / 25.4;
    figheight = 80.0 / 25.4;

    # Create figure
    fig = plt.figure(figsize=(figwidth,figheight))
    fig.subplots_adjust(left=0.15, right=0.98, top=0.98, bottom=0.112, wspace=0.2)

    # Create plotting frames
    ax1 = fig.add_subplot(111)

    # Show SED of Core component
    show_core(ax1)

    # Set attributes
    ax1.tick_params(axis='both', which='major', labelsize=6)
    ax1.tick_params(axis='both', which='minor', labelsize=5)

    # Show plot
    plt.show()

    # Save figure
    fig.savefig('fig5.pdf', dpi=300)

    # Return
    return


# ======================== #
# Main routine entry point #
# ======================== #
if __name__ == '__main__':

    # Show SED of components
    show_core_sed()
