Require Import List Classical Nat Logic Lia ExtensionalityFacts Streams stdpp.list.
Import ListNotations.
From Coq 
Require Import ssreflect. 
From CC
  Require Import lib. 

(** ** Participants *)

Definition participant := nat.
Definition participant_eqb (n1 n2 : participant) : bool := Nat.eqb n1 n2.  
Lemma participant_eqb_correct : forall n1 n2, participant_eqb n1 n2 = true <-> n1 = n2.
Proof. 
  intros. split; intro.
  unfold participant_eqb in H.
  now apply Nat.eqb_eq.
  now apply Nat.eqb_eq.
Qed.

Lemma participant_eqb_refl : forall n, participant_eqb n n = true.
Proof. 
 intros. assert (n = n) by tauto.
 apply Nat.eqb_refl.
Qed. 

Lemma participant_eqb_comm : forall n1 n2, participant_eqb n1 n2 = participant_eqb n2 n1.
Proof.
  intros.
  apply Nat.eqb_sym.
Qed. 

Lemma participant_eqb_no : forall n1 n2, n1 <> n2 -> participant_eqb n1 n2 = false.
Proof.
intros.
 case_eq (participant_eqb n1 n2); auto.
intros.
apply participant_eqb_correct in H0; congruence.
Qed.  

Definition participant_eq_dec : forall n1 n2 : participant, {n1 = n2}+{n1 <> n2}.
Proof.
intros.
case_eq (participant_eqb n1 n2); intros.
- left; apply participant_eqb_correct; assumption.
- right.
  intro.
  apply participant_eqb_correct in H0.
  congruence.
Defined.

(** ** Asynchronous Alphabet *)

Definition message := Type. 

Inductive AsyncAlphabetType :=
| Snd : participant -> participant -> message -> AsyncAlphabetType
| Rcv : participant -> participant -> message -> AsyncAlphabetType.

Definition sender_async (a: AsyncAlphabetType) :=
  match a with
  | Snd p q v => p
  | Rcv p q v => p
  end.

Definition receiver_async (a: AsyncAlphabetType) :=
  match a with
  | Snd p q v => q
  | Rcv p q v => q
  end.

Definition value_async (a: AsyncAlphabetType) :=
  match a with
  | Snd p q v => v
  | Rcv p q v => v
  end.

Definition sender_receiver_neq_async (x : AsyncAlphabetType) :=
  sender_async x <> receiver_async x.

Definition AsyncAlphabet : Type :=
  { x : AsyncAlphabetType | sender_receiver_neq_async x}. 

Definition AsyncAlphabet_proj1 (x : AsyncAlphabet) : AsyncAlphabetType :=
  match x with
| exist _ b _ => b 
  end.

Coercion AsyncAlphabet_proj1 : AsyncAlphabet >-> AsyncAlphabetType. 

Definition is_snd (x : AsyncAlphabet) : Prop :=
  match AsyncAlphabet_proj1 x with
  | Snd p q v => True
  | Rcv p q v => False
  end.

Definition is_rcv (x : AsyncAlphabet) : Prop :=
  match AsyncAlphabet_proj1 x with
  | Snd p q v => False
  | Rcv p q v => True
  end.

Lemma snd_rcv_not :
  forall (x : AsyncAlphabet),
    is_snd x <-> ~ (is_rcv x). 
Proof.
  intros x; split; destruct x; intros; destruct x; try easy.  
  unfold not in H. apply H.
  reflexivity. 
Qed.

Lemma rcv_snd_not :
  forall (x : AsyncAlphabet),
    is_rcv x <-> ~ (is_snd x). 
Proof.
  intros x; split; destruct x; intros; destruct x; try easy.  
  unfold not in H. apply H.
  reflexivity. 
Qed.

Lemma snd_or_rcv :
  forall (x : AsyncAlphabet),
    is_snd x \/ is_rcv x. 
Proof.
  destruct x as [x H_neq].
  destruct x as [p q m | p q m].
  left; easy. right; easy.
Qed.

Definition is_sndb (x : AsyncAlphabet) : bool :=
  match AsyncAlphabet_proj1 x with
  | Snd p q v => true
  | Rcv p q v => false
  end.

Definition is_rcvb (x : AsyncAlphabet) : bool :=
  match AsyncAlphabet_proj1 x with
  | Snd p q v => false
  | Rcv p q v => true
  end.

Lemma snd_rcv_notb :
  forall (x : AsyncAlphabet),
    is_sndb x = negb (is_rcvb x). 
Proof.
  intros x; destruct x; intros; destruct x; try easy.  
Qed.

Lemma rcv_snd_notb :
  forall (x : AsyncAlphabet),
    is_rcvb x = negb (is_sndb x). 
Proof.
  intros x; destruct x; intros; destruct x; try easy.  
Qed.

(** ** Synchronous Alphabet *) 
Inductive SyncAlphabetType :=
| Event : participant -> participant -> message -> SyncAlphabetType. 

Definition sender_receiver_neq_sync (x : SyncAlphabetType) :=
  match x with
  | Event p q v => p <> q
  end. 

Definition SyncAlphabet : Type :=
  { x : SyncAlphabetType | sender_receiver_neq_sync x}. 

Definition SyncAlphabet_proj1 (x : SyncAlphabet) : SyncAlphabetType :=
  match x with
| exist _ b _ => b 
  end.

Program Definition SyncAlphabet_proj2 (x : SyncAlphabet) : sender_receiver_neq_sync (SyncAlphabet_proj1 x) := _. 
Next Obligation.
  intros.
  destruct x. 
  simpl. assumption.
Defined.

Coercion SyncAlphabet_proj1 : SyncAlphabet >-> SyncAlphabetType. 

Definition sender_sync (a: SyncAlphabet) :=
  match SyncAlphabet_proj1 a with
  | Event p q v => p
  end.

Definition receiver_sync (a: SyncAlphabet) :=
  match SyncAlphabet_proj1 a with
  | Event p q v => q
  end.

Definition value_sync (a: SyncAlphabet) :=
  match SyncAlphabet_proj1 a with
  | Event p q v => v
  end.

(** ** Asynchronous and Synchronous Words *) 
Definition FinAsyncWord := list AsyncAlphabet.
Definition FinSyncWord := list SyncAlphabet. 
Definition InfAsyncWord := Stream AsyncAlphabet.
Definition InfSyncWord := Stream SyncAlphabet.

(** ** Split *) 
Program Definition convert_neq (x : SyncAlphabetType) (P : sender_receiver_neq_sync x) : sender_receiver_neq_async (Snd (sender_sync x) (receiver_sync x) (value_sync x)) := _.
Next Obligation. 
  intros.
  destruct x as [p q v].
  simpl. eauto.
Defined.

Program Definition sync_to_async_neq (x : SyncAlphabet) : sender_receiver_neq_async (Snd (sender_sync x) (receiver_sync x) (value_sync x)) := _.
Next Obligation. 
  intros.
  destruct x as [p q v].
  simpl. eauto.
Defined.

Program Definition async_to_sync (x : AsyncAlphabet) : SyncAlphabet := _.
Next Obligation. 
  intro. 
  destruct x as [x H_neq].
  destruct x as [p q m | p q m].
  assert (H_neq_y : sender_receiver_neq_sync (Event p q m)) by easy.
  remember (exist _ (Event p q m) H_neq_y) as y.
  exists (Event p q m). assumption.
  assert (H_neq_y : sender_receiver_neq_sync (Event p q m)) by easy.
  remember (exist _ (Event p q m) H_neq_y) as y.
  exists (Event p q m). assumption. 
Defined.

Program Definition split_symbol_snd : SyncAlphabet -> AsyncAlphabet :=
  fun (x : {x : SyncAlphabetType | sender_receiver_neq_sync x}) =>
    exist _ (Snd (sender_sync x) (receiver_sync x) (value_sync x)) _.
Next Obligation.
  intros.
  destruct x.
  assumption.
Defined.

Program Definition split_symbol_rcv : SyncAlphabet -> AsyncAlphabet :=
  fun (x : {x : SyncAlphabetType | sender_receiver_neq_sync x}) =>
    exist _ (Rcv (sender_sync x) (receiver_sync x) (value_sync x)) _.
Next Obligation.
  intros.
  destruct x.
  assumption.
Defined.
Next Obligation. 
  intros. 
  destruct x.
  unfold sender_receiver_neq_async. simpl.
  unfold sender_receiver_neq_sync in s.
  destruct x.
  unfold sender_sync, receiver_sync; simpl.
  assumption.
Defined.

(* Even though we can define this in terms of the above two definitions,
   keeping it as is for now to save one unfolding in proofs *) 
Program Definition split_symbol : SyncAlphabet -> list AsyncAlphabet :=
  fun (x : {x : SyncAlphabetType | sender_receiver_neq_sync x}) =>
    [exist _ (Snd (sender_sync x) (receiver_sync x) (value_sync x)) _] ++
      [exist _ (Rcv (sender_sync x) (receiver_sync x) (value_sync x)) _].  
Next Obligation.
  intros.
  destruct x.
  simpl. unfold sender_receiver_neq_async. simpl.
  unfold sender_receiver_neq_sync in s.
  destruct x.
  unfold sender_sync, receiver_sync.
  simpl. assumption.
Defined.
Next Obligation.
  intros.
  destruct x.
  simpl. unfold sender_receiver_neq_async. simpl.
  unfold sender_receiver_neq_sync in s.
  destruct x.
  unfold sender_sync, receiver_sync.
  simpl. assumption.
Defined. 
Next Obligation.
  intros. unfold sender_receiver_neq_async.
  destruct x. unfold sender_receiver_neq_sync in s.
  simpl in *. destruct x. simpl. assumption.
Defined. 
Next Obligation.
  intros. unfold sender_receiver_neq_async.
  destruct x. unfold sender_receiver_neq_sync in s.
  simpl in *. destruct x. simpl. assumption.
Defined. 
(* Why are there so many obligations?? *) 
(* For some reason match expressions behave weirdly with program definitions *) 

Program Definition unsplit_symbol : AsyncAlphabet -> SyncAlphabet :=
  fun (x : {x : AsyncAlphabetType | sender_receiver_neq_async x}) =>
    exist _ (Event (sender_async x) (receiver_async x) (value_async x)) _. 
Next Obligation.
  intros.
  destruct x. simpl.
  unfold sender_receiver_neq_sync in s.
  unfold sender_receiver_neq_async. simpl.
  destruct x. simpl. assumption.
  simpl.
  unfold sender_receiver_neq_async in s. 
  simpl. assumption.
Defined.

Definition split (w: FinSyncWord) : FinAsyncWord :=
  flat_map split_symbol w.

CoFixpoint split_inf (w : InfSyncWord) : InfAsyncWord :=
  match w with
  | Cons e w' => Cons (split_symbol_snd e) (Cons (split_symbol_rcv e) (split_inf w'))
  end. 

(* Basic facts about split *)
Lemma split_app :
  forall (w1 w2 : FinSyncWord),
    split (w1 ++ w2) = split w1 ++ split w2. 
Proof.
  intros. unfold split.
  now rewrite flat_map_app.
Qed.

Lemma split_singleton_false :
  forall (w : FinSyncWord) (x : AsyncAlphabet),
    split w = [x] -> False. 
Proof.
  intros.
  destruct w.
  inversion H. inversion H.
Qed.

Lemma split_length_even :
  forall (w : FinSyncWord),
    Nat.Even (length (split w)).  
Proof.
  intro w.
  induction w using rev_ind.
  - simpl. exists 0. lia.
  - destruct IHw as [n IHw].
    exists (S n).
    rewrite split_app.
    rewrite app_length.
    rewrite IHw. simpl.
    lia.
Qed.

Lemma split_nil :
  forall (w : FinSyncWord),
    split w = [] -> w = []. 
Proof.
  intros.
  destruct w.
  inversion H. reflexivity.
  simpl in H. inversion H. 
Qed.

Lemma split_length : 
  forall (w : FinSyncWord),
    length (split w) = 2 * length w. 
Proof.
  intros.
  induction w as [|a w IHw].
  - simpl. lia.
  - simpl. rewrite IHw. lia. 
Qed.

Lemma split_injective_step :
  forall (w1 w2 : FinSyncWord) (a1 a2 : SyncAlphabet),
    split (a1 :: w1) = split (a2 :: w2) ->
    a1 = a2 /\ split w1 = split w2. 
Proof.
  intros.
  destruct a1 as [a1 H_neq1]. 
  destruct a2 as [a2 H_neq2].
  destruct a1 as [p1 q1 m1].
  destruct a2 as [p2 q2 m2].
  inversion H.
  unfold sender_sync, receiver_sync, value_sync in *.
  simpl in *. subst.
  split. f_equal. apply proof_irrelevance.
  reflexivity.
Qed.

Lemma split_injective :
  forall (w1 w2 : FinSyncWord),
    split w1 = split w2 ->
    w1 = w2. 
Proof. 
  intros w1 w2 H_eq.
  generalize dependent w1.
  induction w2 as [|a2 w2 IHw2]; intros.
  - simpl in H_eq.
    now apply split_nil in H_eq.
  - destruct w1 as [| a1 w1].
    * simpl in H_eq. symmetry in H_eq.
      inversion H_eq.
    * apply split_injective_step in H_eq.
      destruct H_eq.
      spec IHw2 w1 H0.
      subst. reflexivity.
Qed.

Lemma split_app_false :
  forall (w1 w2 : FinSyncWord) (x : AsyncAlphabet), 
  split w1 = split w2 ++ [x] -> False. 
Proof.
  intros.
  assert (H_length1 := split_length_even w1). 
  assert (H_length2 := split_length_even w2).
  rewrite H in H_length1.
  rewrite app_length in H_length1.
  simpl in H_length1.
  destruct H_length2 as [m2 H_length2].
  destruct H_length1 as [m1 H_length1].
  lia.
Qed.

Lemma prefix_split_prefix_iff :
  forall (w1 w2 : FinSyncWord),
    prefix w1 w2 <->
      prefix (split w1) (split w2). 
Proof.
  intros w1 w2; split; intro H_pref. 
  - induction w2 as [|a w2 IHw2] using rev_ind.
    * inversion H_pref. 
      symmetry in H. apply app_eq_nil in H.
      destruct H.
      rewrite H. reflexivity.
    * apply prefix_app_tail_or in H_pref.
      destruct H_pref.
      spec IHw2 H.
      rewrite split_app.
      now apply prefix_app_r.
      rewrite H. reflexivity.
  - induction w2 as [|a w2 IHw2] using rev_ind. 
    * inversion H_pref.
      simpl in H. 
      unfold split in H.
      symmetry in H. apply app_eq_nil in H.
      destruct H.
      enough (w1 = []).
      rewrite H1. reflexivity.
      destruct w1. reflexivity.
      simpl in H. inversion H. 
    * rewrite split_app in H_pref.
      destruct a as [a H_neq].
      destruct a as [p q m].
      unfold split at 3 in H_pref.
      unfold split_symbol in H_pref. simpl in H_pref.
      unfold sender_sync, receiver_sync, value_sync in H_pref.
      simpl in H_pref.
      (* Have to do this the brute force way *) 
      assert (H_equiv : split w1 `prefix_of` split w2 ++ [Snd p q m ↾ H_neq] ++ [Rcv p q m ↾ H_neq]) by easy.
      (* Getting rid of the bad one *) 
      clear H_pref.
      rewrite app_assoc in H_equiv.
      apply prefix_app_tail_or in H_equiv.
      destruct H_equiv.
      ** (* In the case that split w1 <= split w2 ++ Snd p q m *)
        (* We do further case analysis on this prefix relation *)
        apply prefix_app_tail_or in H.
        destruct H.
        (* In the case that split w1 <= split w2 *)
        (* We can use IHw2 directly *)
        spec IHw2 H.
        now apply prefix_app_r.
        now apply split_app_false in H. 
      ** (* In the case that split w1 = split (w2 + Event p q m) *)
        clear IHw2.
        (* We can appeal to the injectivity of split to show that w1 = w2 ++ Event p q m *)
        assert (H_rewrite : (split w2 ++ [Snd p q m ↾ H_neq]) ++ [Rcv p q m ↾ H_neq] = split (w2 ++ [Event p q m ↾ H_neq])).
        { rewrite split_app.
          rewrite app_assoc.
          unfold split_symbol.
          unfold sender_sync, receiver_sync, value_sync.
          simpl. rewrite app_nil_r.
          rewrite <- app_assoc.
          reflexivity. }
        rewrite H_rewrite in H.
        apply split_injective in H. 
        subst. reflexivity.
Qed. 

Lemma split_inf_inf_split_hd :
  forall (hd : SyncAlphabet) (tl : InfSyncWord), 
    stream_to_list (split_inf (Cons hd tl)) 2 = split (stream_to_list (Cons hd tl) 1). 
Proof.
  intros hd tl.
  simpl. 
  unfold sender_sync, receiver_sync, value_sync; simpl.
  reflexivity.
Qed.

Lemma split_inf_inf_split :
  forall (run : InfSyncWord) (k : nat), 
    stream_to_list (split_inf run) (2 * k) = split (stream_to_list run k).
Proof.
  intros run k. 
  simpl.
  generalize dependent run. 
  induction k; intros.  
  - simpl.
    reflexivity.
  - destruct run.
    rewrite stream_to_list_behead.
    simpl.
    destruct s.
    destruct x as [p q m].
    unfold split_symbol_rcv, split_symbol_snd.
    unfold sender_sync, receiver_sync, value_sync.
    simpl.
    replace (k + S (k + 0)) with (S (k + k)) by lia.
    assert (H_rewrite := stream_to_list_behead). 
    remember (Rcv p q m ↾ s) as x_rcv.
    spec H_rewrite AsyncAlphabet x_rcv.
    spec H_rewrite (split_inf run).
    spec H_rewrite (k+k).
    rewrite H_rewrite.
    rewrite Nat.add_0_r in IHk.
    spec IHk run.
    rewrite IHk. easy.
Qed. 

Lemma Str_nth_zero_cons :
  forall (A : Type) (a b : A) (s : Stream A),
    Str_nth 0 (Cons a s) = b ->
    a = b. 
Proof.                                    
  intros. unfold Str_nth in H. simpl in H. assumption.
Qed.

Lemma singleton_eq_means_eq :
  forall (A : Type) (a b : A),
    [a] = [b] -> a = b. 
Proof. 
  intros. inversion H. easy.
Qed.

Lemma doubleton_eq_means_eq :
  forall (A : Type) (a b c d : A),
    [a;b] = [c;d] -> a = c /\ b = d. 
Proof. 
  intros. inversion H. subst.
  easy.
Qed. 

(* Be wary of indexing here *)
(* Very tricksy lemma! *) 
Lemma Str_nth_split_inf_inf_split :
  forall (run_inf : InfSyncWord) (i : nat) (p q : participant) (m : message) (H_neq : sender_receiver_neq_sync (Event p q m)) (H_neq_snd : sender_receiver_neq_async (Snd p q m)) (H_neq_rcv : sender_receiver_neq_async (Rcv p q m)),
    Str_nth i run_inf = (Event p q m ↾ H_neq) -> 
    Str_nth (2 * i) (split_inf run_inf) = (Snd p q m ↾ H_neq_snd) /\
      Str_nth (2 * i + 1) (split_inf run_inf) = (Rcv p q m ↾ H_neq_rcv). 
Proof. 
  intros.
  assert (H_helper := split_inf_inf_split).
  assert (H_helper' := stream_to_list_S_Str_nth_app AsyncAlphabet).
  assert (H_helper'' := stream_to_list_S_Str_nth_app AsyncAlphabet).
  assert (H_eq_p : Snd p q m ↾ H_neq = Snd p q m ↾ H_neq_snd) by sigma_equal.
  assert (H_eq_q : Rcv p q m ↾ H_neq = Rcv p q m ↾ H_neq_rcv) by sigma_equal.
  destruct i.
  - destruct run_inf.
    apply Str_nth_zero_cons in H.
    subst. simpl.
    (* assert (H_useful := split_inf_inf_split_hd).  *)
    spec H_helper' 0 (split_inf (Cons (Event p q m ↾ H_neq) run_inf)).
    spec H_helper'' 1 (split_inf (Cons (Event p q m ↾ H_neq) run_inf)).
    simpl in *.
    unfold split_symbol_snd, split_symbol_rcv in *. 
    simpl in *.
    unfold sender_sync, receiver_sync, value_sync in *.
    simpl in *.
    split.
    simpl in H_helper'.
    symmetry.
    apply singleton_eq_means_eq in H_helper'.
    rewrite <- H_eq_p. assumption.
    apply doubleton_eq_means_eq in H_helper''.
    symmetry. rewrite <- H_eq_q. tauto.
  - remember (Datatypes.S i) as k.
    spec H_helper' ((2*k)+1) (split_inf run_inf).
    spec H_helper'' (2*k) (split_inf run_inf).
    replace (S (2*k)) with (2*k + 1) in H_helper'' by lia.
    rewrite H_helper'' in H_helper'.
    spec H_helper run_inf (k+1).
    replace (2*(k+1)) with (S (2*k+1)) in H_helper by lia.
    rewrite H_helper in H_helper'.
    clear H_helper H_helper''.
    assert (H_helper := stream_to_list_S_Str_nth_app SyncAlphabet).
    spec H_helper k run_inf.
    rewrite H in H_helper.
    replace (S k) with (k+1) in H_helper by lia.
    rewrite H_helper in H_helper'.
    rewrite split_app in H_helper'.
    rewrite <- split_inf_inf_split in H_helper'.
    rewrite <- app_assoc in H_helper'.
    apply app_inv_head in H_helper'.
    symmetry in H_helper'.
    unfold split in H_helper'.
    unfold split_symbol in H_helper'.
    simpl in H_helper'.
    unfold sender_sync, receiver_sync, value_sync in H_helper'.
    simpl in H_helper'. inversion H_helper'.
    split. replace (k + (k + 0)) with (2 * k) in H1 by lia.
    assert (H_eq : Snd p q m ↾ H_neq = Snd p q m ↾ H_neq_snd) by sigma_equal.
    rewrite <- H_eq. assumption.
    replace (k + (k + 0) + 1) with (2 * k + 1) in H2 by lia.
    assert (H_eq : Rcv p q m ↾ H_neq = Rcv p q m ↾ H_neq_rcv) by sigma_equal.
    rewrite <- H_eq. assumption.
Qed. 

Lemma Str_nth_split_inf_eq_split_Str_nth :
   forall (run_inf : InfSyncWord) (i : nat),
     [Str_nth (2 * i) (split_inf run_inf)] ++
       [Str_nth (Datatypes.S (2 * i)) (split_inf run_inf)] =
       split [Str_nth i run_inf].
Proof.
  intros. 
  assert (H_helper := Str_nth_split_inf_inf_split run_inf i).
  remember (Str_nth i run_inf) as x.
  destruct x as [[p q m] H_neq].
  assert (H_neq_snd : sender_receiver_neq_async (Snd p q m)) by easy. 
  assert (H_neq_rcv : sender_receiver_neq_async (Rcv p q m)) by easy.
  spec H_helper p q m H_neq H_neq_snd H_neq_rcv.
  spec H_helper. reflexivity.
  destruct H_helper.
  rewrite H.
  replace (S (2 * i)) with (2 * i + 1) by lia.
  rewrite H0.
  unfold split, split_symbol.
  unfold sender_sync, receiver_sync, value_sync.
  simpl. f_equal. sigma_equal. f_equal. sigma_equal.
Qed. 

(** ** Participant Projections *) 
Definition ParticipantAsyncAlphabet (p : participant) :=
  { x : AsyncAlphabet | sender_async x = p \/ receiver_async x = p}.

(* Apparently subtyping is not really doable here *)
Definition wproj_symbol (r : participant) (x : AsyncAlphabet) : list (AsyncAlphabet) :=
  if (is_sndb x && participant_eqb (sender_async x) r) ||
       (is_rcvb x && participant_eqb (receiver_async x) r)
  then [x] else [].

Definition wproj (w : FinAsyncWord) (r : participant) : FinAsyncWord :=
  flat_map (wproj_symbol r) w.

Lemma wproj_receiver_eq :
  forall (p q r : participant) (m : message) (H_neq : sender_receiver_neq_async (Rcv p q m)),
    q = r -> 
    wproj [exist _ (Rcv p q m) H_neq] r = [exist _ (Rcv p q m) H_neq]. 
Proof.
  intros.
  unfold wproj.
  unfold wproj_symbol; simpl.
  replace (participant_eqb q r) with true.
  now rewrite app_nil_r.
  symmetry.
  now apply participant_eqb_correct.
Qed.

Lemma wproj_symbol_receiver_eq :
  forall (p q r : participant) (m : message) (H_neq : sender_receiver_neq_async (Rcv p q m)),
    q = r -> 
    wproj_symbol r (exist _ (Rcv p q m) H_neq) = [exist _ (Rcv p q m) H_neq]. 
Proof.
  intros.
  unfold wproj.
  unfold wproj_symbol; simpl.
  replace (participant_eqb q r) with true.
  reflexivity. 
  symmetry.
  now apply participant_eqb_correct.
Qed.

Lemma wproj_receiver_neq :
  forall (p q r : participant) (m : message) (H_neq : sender_receiver_neq_async (Rcv p q m)),
    q <> r -> 
    wproj [exist _ (Rcv p q m) H_neq] r = []. 
Proof.
  intros.
  unfold wproj.
  unfold wproj_symbol; simpl.
  replace (participant_eqb q r) with false.
  now rewrite app_nil_r.
  symmetry.
  now apply participant_eqb_no.
Qed.

Lemma wproj_symbol_receiver_neq :
  forall (p q r : participant) (m : message) (H_neq : sender_receiver_neq_async (Rcv p q m)),
    q <> r -> 
    wproj_symbol r (exist _ (Rcv p q m) H_neq) = []. 
Proof.
  intros.
  unfold wproj.
  unfold wproj_symbol; simpl.
  replace (participant_eqb q r) with false.
  reflexivity. symmetry.
  now apply participant_eqb_no.
Qed.

Lemma wproj_sender_eq :
  forall (p q r : participant) (m : message) (H_neq : sender_receiver_neq_async (Snd p q m)),
    p = r -> 
    wproj [exist _ (Snd p q m) H_neq] r = [exist _ (Snd p q m) H_neq]. 
Proof.
  intros.
  unfold wproj.
  unfold wproj_symbol; simpl.
  replace (participant_eqb p r) with true.
  now rewrite app_nil_r.
  symmetry.
  now apply participant_eqb_correct.
Qed.

Lemma wproj_symbol_sender_eq :
  forall (p q r : participant) (m : message) (H_neq : sender_receiver_neq_async (Snd p q m)),
    p = r -> 
    wproj_symbol r (exist _ (Snd p q m) H_neq) = [exist _ (Snd p q m) H_neq]. 
Proof.
  intros.
  unfold wproj.
  unfold wproj_symbol; simpl.
  replace (participant_eqb p r) with true.
  reflexivity. 
  symmetry.
  now apply participant_eqb_correct.
Qed.

Lemma wproj_sender_neq :
  forall (p q r : participant) (m : message) (H_neq : sender_receiver_neq_async (Snd p q m)),
    p <> r -> 
    wproj [exist _ (Snd p q m) H_neq] r = []. 
Proof.
  intros.
  unfold wproj.
  unfold wproj_symbol; simpl.
  replace (participant_eqb p r) with false.
  now rewrite app_nil_r.
  symmetry.
  now apply participant_eqb_no.
Qed.

Lemma wproj_symbol_sender_neq :
  forall (p q r : participant) (m : message) (H_neq : sender_receiver_neq_async (Snd p q m)),
    p <> r -> 
    wproj_symbol r (exist _ (Snd p q m) H_neq) = []. 
Proof.
  intros.
  unfold wproj.
  unfold wproj_symbol; simpl.
  replace (participant_eqb p r) with false.
  reflexivity. symmetry.
  now apply participant_eqb_no.
Qed.

Lemma wproj_app :
  forall (w1 w2 : FinAsyncWord) (r : participant),
    wproj (w1 ++ w2) r = wproj w1 r ++ wproj w2 r.
Proof. 
  intros w1 w2 r. 
  unfold wproj at 1.
  now rewrite flat_map_app.
Qed.

Lemma wproj_symbol_idempotent :
  forall (x : AsyncAlphabet) (r : participant),
    wproj (wproj_symbol r x) r = wproj_symbol r x. 
Proof.     
  intros [x H_neq] r.
  destruct x as [p q m | p q m].
  - destruct (classic (p = r)).
    * unfold wproj_symbol.
      simpl.
      replace (participant_eqb p r) with true.
      simpl.
      unfold wproj_symbol.
      simpl.
      replace (participant_eqb p r) with true.
      simpl. reflexivity.
      symmetry. now apply participant_eqb_correct.
      symmetry. now apply participant_eqb_correct.
    * unfold wproj_symbol.
      simpl.
      replace (participant_eqb p r) with false.
      simpl.
      unfold wproj_symbol.
      simpl.
      replace (participant_eqb p r) with false. 
      simpl. reflexivity.
      symmetry. now apply participant_eqb_no.
      symmetry. now apply participant_eqb_no.
  - destruct (classic (q = r)).
    * unfold wproj_symbol.
      simpl.
      replace (participant_eqb q r) with true.
      simpl.
      unfold wproj_symbol.
      simpl.
      replace (participant_eqb q r) with true.
      simpl. reflexivity.
      symmetry. now apply participant_eqb_correct.
      symmetry. now apply participant_eqb_correct.
    * unfold wproj_symbol.
      simpl.
      replace (participant_eqb q r) with false.
      simpl.
      unfold wproj_symbol.
      simpl.
      replace (participant_eqb q r) with false.
      simpl. reflexivity.
      symmetry. now apply participant_eqb_no.
      symmetry. now apply participant_eqb_no.
Qed.

Lemma wproj_in :
  forall (w : FinAsyncWord) (p : participant) (x : AsyncAlphabet),
    In x (wproj w p) -> In x w. 
Proof.
  intros w p x H_in.
  induction w as [|a w IHw] using rev_ind.
  - inversion H_in.
  - rewrite wproj_app in H_in.
    apply in_app_or in H_in.
    destruct H_in.
    spec IHw H.
    apply in_or_app. tauto.
    apply in_or_app. right.
    destruct a as [a H_neq].
    destruct a as [q r m | q r m].
    destruct (classic (q = p)).
    apply (wproj_sender_eq q r p m H_neq) in H0.
    now rewrite H0 in H. 
    apply (wproj_sender_neq q r p m H_neq) in H0.
    rewrite H0 in H.
    inversion H.
    destruct (classic (r = p)).
    apply (wproj_receiver_eq q r p m H_neq) in H0.
    now rewrite H0 in H.
    apply (wproj_receiver_neq q r p m H_neq) in H0.
    now rewrite H0 in H.
Qed.

Lemma wproj_nil_means_prefix_wproj_nil :
  forall (w w' : list AsyncAlphabet) (p : participant),
    wproj w p = [] -> 
    prefix w' w ->
    wproj w' p = []. 
Proof.
  intros w w' p H_nil H_pref. 
  apply prefix_exists_suffix in H_pref.
  destruct H_pref as [w'_suf H_split].
  subst. rewrite wproj_app in H_nil.
  apply app_eq_nil in H_nil.
  tauto.
Qed.

Lemma wproj_idempotent :
  forall (w : FinAsyncWord) (p : participant),
    wproj (wproj w p) p = wproj w p.
Proof. 
  intros. 
  induction w as [|a w IHw].
  - reflexivity.
  - simpl.
    rewrite wproj_app.
    rewrite IHw.
    apply app_inv_tail_iff.
    now apply wproj_symbol_idempotent. 
Qed.

Lemma wproj_split_app :
  forall (w1 w2 : FinSyncWord) (p : participant),
    wproj (split (w1 ++ w2)) p =
      wproj (split w1) p ++ wproj (split w2) p. 
Proof.
  intros.
  unfold split, wproj.
  repeat rewrite flat_map_app.
  easy.
Qed.

Lemma wproj_app_exists_app :
  forall (w w1 w2 : FinAsyncWord) (r : participant),
    wproj w r = w1 ++ w2 ->
    exists (v1 v2 : FinAsyncWord),
      w = v1 ++ v2 /\
        wproj v1 r = w1 /\
        wproj v2 r = w2. 
Proof.
  intros.
  generalize dependent w1. 
  generalize dependent w2.
  induction w using rev_ind; intros. 
  - exists [], []. inversion H.
    symmetry in H1.
    apply app_nil in H1.
    split; easy.
  - destruct x as [x H_neq].
    destruct x as [p q m | p q m].
    * destruct (classic (p = r)).
      ** rewrite wproj_app in H.
         apply (wproj_sender_eq p q r m H_neq) in H0.
         rewrite H0 in H.
         destruct (destruct_list_last _ w2).
         exists (w ++ [Snd p q m ↾ H_neq]), [].
         split. rewrite app_nil_r. reflexivity.
         split.
         rewrite wproj_app. rewrite H1 in H.
         rewrite app_nil_r in H.
         now rewrite H0. 
         rewrite H1. easy.
         destruct H1 as [x2 [l2 H1]].
         rewrite H1 in H.
         rewrite app_assoc in H.
         apply app_inj_tail_iff in H. 
         destruct H.
         spec IHw l2 w1 H.
         destruct IHw as [v1 [v2 IHw]].
         exists v1, (v2 ++ [Snd p q m ↾ H_neq]).
         split. destruct IHw. rewrite H3.
         now rewrite app_assoc.
         split. tauto. rewrite H1.
         destruct IHw. destruct H4.
         rewrite <- H5.
         rewrite wproj_app. rewrite <- H2.
         rewrite H0. reflexivity.
      ** rewrite wproj_app in H.
         apply (wproj_sender_neq p q r m H_neq) in H0.
         rewrite H0 in H.
         rewrite app_nil_r in H.
         spec IHw w2 w1. 
         spec IHw H.
         destruct IHw as [v1 [v2 IHw]].
         exists v1, (v2 ++ [Snd p q m ↾ H_neq]). 
         split. 
         rewrite app_assoc.
         apply app_inj_tail_iff. split. tauto. reflexivity.
         split. tauto.
         rewrite wproj_app.
         rewrite H0.
         rewrite app_nil_r.
         tauto.
    * destruct (classic (q = r)).
      rewrite wproj_app in H.
      apply (wproj_receiver_eq p q r m H_neq) in H0.
      rewrite H0 in H.
      destruct (destruct_list_last _ w2).
      exists (w ++ [Rcv p q m ↾ H_neq]), [].
      split. rewrite app_nil_r. reflexivity.
      split.
      rewrite wproj_app. rewrite H1 in H.
      rewrite app_nil_r in H.
      now rewrite H0. 
      rewrite H1. easy.
      destruct H1 as [x2 [l2 H1]].
      rewrite H1 in H.
      rewrite app_assoc in H.
      apply app_inj_tail_iff in H. 
      destruct H.
      spec IHw l2 w1 H.
      destruct IHw as [v1 [v2 IHw]].
      exists v1, (v2 ++ [Rcv p q m ↾ H_neq]).
      split. destruct IHw. rewrite H3.
      now rewrite app_assoc.
      split. tauto. rewrite H1.
      destruct IHw. destruct H4.
      rewrite <- H5.
      rewrite wproj_app. rewrite <- H2.
      rewrite H0. reflexivity.
      ** rewrite wproj_app in H.
         apply (wproj_receiver_neq p q r m H_neq) in H0.
         rewrite H0 in H.
         rewrite app_nil_r in H.
         spec IHw w2 w1. 
         spec IHw H.
         destruct IHw as [v1 [v2 IHw]].
         exists v1, (v2 ++ [Rcv p q m ↾ H_neq]). 
         split. 
         rewrite app_assoc.
         apply app_inj_tail_iff. split. tauto. reflexivity.
         split. tauto.
         rewrite wproj_app.
         rewrite H0.
         rewrite app_nil_r.
         tauto.
Qed.

Lemma wproj_preserves_prefix :
  forall (w1 w2 : FinAsyncWord) (p : participant),
    prefix w1 w2 ->
    prefix (wproj w1 p) (wproj w2 p). 
Proof.
  intros w1 w2 p H_prefix. 
  generalize dependent w1.
  induction w2 as [|a w2 IHw2] using rev_ind; intros. 
  - apply prefix_nil_inv in H_prefix. rewrite H_prefix.
    reflexivity.
  - apply prefix_app_tail_or in H_prefix.
    destruct H_prefix.
    spec IHw2 w1 H.
    unfold wproj at 2; rewrite flat_map_app.
    now apply prefix_app_r.
    rewrite H.
    reflexivity.
Qed.

Lemma wproj_sender_inv :
  forall (x : SyncAlphabet) (r : participant) (x' : AsyncAlphabet),
    wproj (split_symbol x) r = [x'] -> 
    is_snd x' ->
    (sender_sync x) = r.
Proof. 
  intros x r x' H_wproj H_snd.
  destruct x as [x H_neq].
  destruct x as [p q m].
  unfold split_symbol, wproj, wproj_symbol in H_wproj. 
  simpl in H_wproj.
  unfold sender_sync, receiver_sync, value_sync in H_wproj. 
  simpl in H_wproj.
  destruct (classic (p = r)).
  unfold sender_sync. assumption.
  replace (participant_eqb p r) with false in H_wproj.
  simpl in H_wproj.
  destruct (participant_eqb q r). inversion H_wproj.
  subst. inversion H_snd.
  rewrite app_nil_r in H_wproj. 
  inversion H_wproj.
  symmetry. now apply participant_eqb_no.
Qed.

Lemma wproj_receiver_inv : 
    forall (x : SyncAlphabet) (r : participant) (x' : AsyncAlphabet),
    wproj (split_symbol x) r = [x'] -> 
    is_rcv x' ->
    (receiver_sync x) = r.
Proof.
  intros x r x' H_wproj H_snd.
  destruct x as [x H_neq].
  destruct x as [p q m].
  unfold split_symbol, wproj, wproj_symbol in H_wproj. 
  simpl in H_wproj.
  unfold sender_sync, receiver_sync, value_sync in H_wproj. 
  simpl in H_wproj.
  destruct (classic (q = r)).
  unfold receiver_sync. assumption.
  replace (participant_eqb q r) with false in H_wproj.
  simpl in H_wproj.
  destruct (participant_eqb p r). inversion H_wproj.
  subst. inversion H_snd.
  rewrite app_nil_r in H_wproj. 
  inversion H_wproj.
  symmetry. now apply participant_eqb_no.
Qed. 

Lemma in_wproj_means_eq :
  forall (w : FinAsyncWord) (p : participant) (x : AsyncAlphabet),
    In x (wproj w p) ->
    wproj_symbol p x = [x].  
Proof.
  intros w r x H_in.
  apply in_flat_map in H_in.
  destruct H_in as [x' [H_in' H_in]].
  destruct x as [x H_neq].
  destruct x as [p q m | p q m]. 
  - (* When x is a send event *)
    destruct x' as [x' H_neq']. 
    destruct x' as [p' q' m' | p' q' m'].
    * unfold wproj_symbol in H_in.
      simpl in H_in.
      destruct (classic (p' = r)).
      ** replace (participant_eqb p' r) with true in H_in.
         simpl in H_in.
         destruct H_in.
         inversion H0.
         unfold wproj_symbol. simpl.
         replace (participant_eqb p r) with true.
         simpl. reflexivity.
         symmetry.
         rewrite <- H2. 
         now apply participant_eqb_correct.
         contradiction.
         symmetry.
         now apply participant_eqb_correct.
      ** replace (participant_eqb p' r) with false in H_in.
         simpl in H_in.
         contradiction.
         replace (participant_eqb p' r) with false in H_in.
         simpl in H_in.
         contradiction.
         replace (participant_eqb p' r) with false in H_in.
         simpl in H_in.
         contradiction.
         symmetry.
         now apply participant_eqb_no.
    * unfold wproj_symbol in H_in.
      simpl in H_in.
      destruct (participant_eqb q' r).
      inversion H_in.
      inversion H.
      inversion H. inversion H_in.
  - (* When x is receive event *)
    destruct x' as [x' H_neq']. 
    destruct x' as [p' q' m' | p' q' m'].
    * (* When x' is a send event *)
      unfold wproj_symbol in H_in.
      simpl in H_in.
      destruct (participant_eqb p' r).
      inversion H_in.
      inversion H.
      inversion H. inversion H_in. 
    * unfold wproj_symbol in H_in.
      simpl in H_in.
      destruct (classic (q' = r)).
      ** replace (participant_eqb q' r) with true in H_in.
         simpl in H_in.
         destruct H_in.
         inversion H0.
         unfold wproj_symbol; simpl.
         replace (participant_eqb q r) with true.
         reflexivity.
         symmetry.
         rewrite <- H3. 
         now apply participant_eqb_correct.
         contradiction. 
         symmetry.
         now apply participant_eqb_correct.
      ** replace (participant_eqb q' r) with false in H_in.
         simpl in H_in.
         contradiction.
         replace (participant_eqb q' r) with false in H_in.
         simpl in H_in.
         contradiction.
         replace (participant_eqb q' r) with false in H_in.
         simpl in H_in.
         contradiction.
         symmetry.
         now apply participant_eqb_no.
Qed.

Definition is_active (p : participant) (x : AsyncAlphabet) :=
  ((is_snd x -> sender_async x = p) /\
     (is_rcv x -> receiver_async x = p)).

Lemma in_wproj_means_active :
  forall (w : FinAsyncWord) (p : participant),
    Forall (is_active p) (wproj w p). 
Proof.
  intros w r.
  induction w as [|a w IHw].
  - easy.
  - simpl. rewrite Forall_app.
    split. 2 : assumption.
    unfold wproj_symbol; simpl.
    destruct a as [[p q m | p q m] H_neq]; simpl. 
    destruct (classic (p = r)).
    replace (participant_eqb p r) with true.
    simpl.
    apply Forall_cons. 
    split. 2 : easy.
    rewrite <- H.
    easy.
    symmetry; now apply participant_eqb_correct.
    replace (participant_eqb p r) with false.
    simpl. easy.
    symmetry; now apply participant_eqb_no.
    destruct (classic (q = r)).
    replace (participant_eqb q r) with true.
    simpl.
    apply Forall_cons. 
    split. 2 : easy.
    rewrite <- H.
    easy.
    symmetry; now apply participant_eqb_correct.
    replace (participant_eqb q r) with false.
    simpl. easy.
    symmetry; now apply participant_eqb_no.
Qed.

Lemma wproj_no_effect :
  forall (w : FinAsyncWord) (p : participant),
    Forall (is_active p) w ->
    wproj w p = w. 
Proof. 
  intros w p.
  induction w as [|a w IHw].
  - reflexivity.
  - intros.
    apply Forall_cons_1 in H.
    spec IHw.
    tauto.
    simpl. rewrite IHw.
    destruct H as [H_active _].
    destruct H_active as [H_snd H_rcv].
    destruct a as [[q r m | q r m] H_neq].
    unfold wproj_symbol; simpl.
    spec H_snd. easy.
    simpl in H_snd.
    apply participant_eqb_correct in H_snd. rewrite H_snd.
    reflexivity.
    spec H_rcv. easy.
    simpl in H_rcv.
    apply participant_eqb_correct in H_rcv.
    unfold wproj_symbol; simpl; rewrite H_rcv; reflexivity.
Qed.

Lemma split_case :
  forall (p q r : participant) (m : message) (H_neq : sender_receiver_neq_sync (Event p q m)),
    (wproj (split [exist _ (Event p q m) H_neq]) r = [exist _ (Snd p q m) H_neq]) \/
      (wproj (split [exist _ (Event p q m) H_neq]) r = [exist _ (Rcv p q m) H_neq]) \/
      (wproj (split [exist _ (Event p q m) H_neq]) r = []) .
Proof.                                              
  intros p q r m H_neq. 
  destruct (classic (p = r \/ q = r)). 
  - destruct H.
    * left. 
      unfold split; simpl. 
      unfold sender_sync, receiver_sync, value_sync; simpl.
      rewrite app_nil_r.
      unfold wproj_symbol.
      simpl.
      replace (participant_eqb p r) with true.
      replace (participant_eqb q r) with false.
      simpl. reflexivity.
      symmetry.
      rewrite H in H_neq.
      unfold sender_receiver_neq_sync in H_neq.
      rewrite participant_eqb_comm.
      now apply participant_eqb_no.
      symmetry; now apply participant_eqb_correct.
    * right; left.
      unfold split; simpl. 
      unfold sender_sync, receiver_sync, value_sync; simpl.
      rewrite app_nil_r.
      unfold wproj_symbol.
      simpl.
      replace (participant_eqb p r) with false.
      replace (participant_eqb q r) with true.
      simpl. reflexivity.
      symmetry; now apply participant_eqb_correct. 
      symmetry.
      rewrite H in H_neq.
      unfold sender_receiver_neq_sync in H_neq.
      rewrite participant_eqb_comm.
      now apply participant_eqb_no.
  - right; right.
    apply not_or_and in H. 
    unfold split; simpl. 
    unfold sender_sync, receiver_sync, value_sync; simpl.
    rewrite app_nil_r.
    unfold wproj_symbol.
    simpl.
    replace (participant_eqb p r) with false.
    replace (participant_eqb q r) with false.
    simpl. reflexivity.
    destruct H.
    symmetry; now apply participant_eqb_no.
    symmetry; now apply participant_eqb_no. 
Qed.

Lemma wproj_split_symbol_eq_snd_inv :
  forall (x : SyncAlphabet) (p q : participant) (m : message) (H_neq : sender_receiver_neq_async (Snd p q m)), 
    wproj (split_symbol x) p = [Snd p q m ↾ H_neq] ->
    exists (H_neq' : sender_receiver_neq_sync (Event p q m)), 
    x = exist _ (Event p q m) H_neq'. 
Proof.                               
  intros x p q m H_neq H_eq.
  assert (witness : sender_receiver_neq_sync (Event p q m)) by easy. 
  exists witness.
  destruct x as [x H_neq_x]. 
  destruct x as [p0 q0 m0]. 
  simpl in H_eq.
  unfold sender_sync, receiver_sync, value_sync in H_eq.
  destruct (classic (p0 = p)); simpl in H_eq; subst. 
  rewrite wproj_symbol_sender_eq in H_eq.
  reflexivity.
  destruct (classic (q0 = q)); simpl in H_eq; subst. 
  rewrite wproj_symbol_receiver_neq in H_eq. easy.
  destruct (classic (m0 = m)); subst.
  sigma_equal.
  unnil H_eq. inversion H_eq. contradiction.
  rewrite wproj_symbol_receiver_neq in H_eq. easy.
  unnil H_eq. inversion H_eq. contradiction.
  rewrite wproj_symbol_sender_neq in H_eq. easy.
  destruct (classic (q0 = q)); simpl in H_eq; subst. 
  rewrite wproj_symbol_receiver_neq in H_eq. easy.
  inversion H_eq.
  destruct (classic (q0 = p)); simpl in H_eq; subst. 
  rewrite wproj_symbol_receiver_eq in H_eq. easy.
  inversion H_eq.
  rewrite wproj_symbol_receiver_neq in H_eq. easy.
  inversion H_eq.
Qed. 

Lemma wproj_split_symbol_eq_rcv_inv :
  forall (x : SyncAlphabet) (p q : participant) (m : message) (H_neq : sender_receiver_neq_async (Rcv p q m)), 
    wproj (split_symbol x) q = [Rcv p q m ↾ H_neq] ->
    exists (H_neq' : sender_receiver_neq_sync (Event p q m)), 
    x = exist _ (Event p q m) H_neq'. 
Proof.                               
  intros x p q m H_neq H_eq.
  assert (witness : sender_receiver_neq_sync (Event p q m)) by easy. 
  exists witness.
  destruct x as [x H_neq_x]. 
  destruct x as [p0 q0 m0]. 
  simpl in H_eq.
  unfold sender_sync, receiver_sync, value_sync in H_eq.
  destruct (classic (p0 = p)); simpl in H_eq; subst. 
  rewrite wproj_symbol_sender_neq in H_eq.
  easy.  
  destruct (classic (q0 = q)); simpl in H_eq; subst. 
  rewrite wproj_symbol_receiver_eq in H_eq. easy.
  destruct (classic (m0 = m)); subst.
  sigma_equal.
  unnil H_eq. inversion H_eq. contradiction.
  rewrite wproj_symbol_receiver_neq in H_eq. easy.
  unnil H_eq. inversion H_eq.
  destruct (classic (q0 = q)); simpl in H_eq; subst. 
  rewrite wproj_symbol_receiver_eq in H_eq. easy.
  rewrite wproj_symbol_sender_neq in H_eq. easy.
  unnil H_eq. inversion H_eq. contradiction.
  rewrite wproj_symbol_receiver_neq in H_eq. easy.
  destruct (classic (p0 = q)); simpl in H_eq; subst. 
  rewrite wproj_symbol_sender_eq in H_eq. easy.
  inversion H_eq.
  rewrite wproj_symbol_sender_neq in H_eq. easy.
  inversion H_eq.
Qed. 

(** ** Message projections **) 
Definition mproj_snd_symbol (p q : participant) (x : AsyncAlphabet) : list message :=
  if (participant_eqb (sender_async x) p) &&
     (participant_eqb (receiver_async x) q) &&
     (is_sndb x)
  then [value_async x]
  else []. 

Definition mproj_snd (w : list AsyncAlphabet) (p q : participant) : list message :=
  flat_map (mproj_snd_symbol p q) w. 

Lemma mproj_snd_app :
  forall (w1 w2 : FinAsyncWord) (p q : participant),
    mproj_snd (w1 ++ w2) p q =
      mproj_snd w1 p q ++ mproj_snd w2 p q. 
Proof.
  intros w1 w2 p q.
  unfold mproj_snd.
  now rewrite flat_map_app.
Qed.

Lemma mproj_snd_rcv :
  forall (x : AsyncAlphabet),
    is_rcvb x = true -> 
    flat_map (mproj_snd_symbol (sender_async x) (receiver_async x)) [x] = [].
Proof. 
  intros.
  unfold mproj_snd_symbol. 
  simpl.
  rewrite app_nil_r.
  rewrite snd_rcv_notb. 
  rewrite H. simpl. 
  rewrite andb_false_r. 
  reflexivity.
Qed.
           
Lemma mproj_snd_snd :
  forall (x : AsyncAlphabet),
    is_sndb x = true -> 
    flat_map (mproj_snd_symbol (sender_async x) (receiver_async x)) [x] = [value_async x].
Proof. 
  intros.
  unfold mproj_snd_symbol. 
  simpl.
  rewrite app_nil_r.
  rewrite H. 
  simpl. now repeat rewrite participant_eqb_refl.
Qed.

Definition mproj_rcv_symbol (p q : participant) (x : AsyncAlphabet) : list message :=
  if (participant_eqb (sender_async x) p) &&
     (participant_eqb (receiver_async x) q) &&
     (is_rcvb x)
  then [value_async x]
  else []. 

Definition mproj_rcv (w : list AsyncAlphabet) (p q : participant) : list message :=
  flat_map (mproj_rcv_symbol p q) w.

Lemma mproj_rcv_app :
  forall (w1 w2 : FinAsyncWord) (p q : participant),
    mproj_rcv (w1 ++ w2) p q =
      mproj_rcv w1 p q ++ mproj_rcv w2 p q. 
Proof.
  intros w1 w2 p q.
  unfold mproj_rcv.
  now rewrite flat_map_app.
Qed.

Lemma mproj_rcv_snd :
  forall (x : AsyncAlphabet),
    is_sndb x = true -> 
    flat_map (mproj_rcv_symbol (sender_async x) (receiver_async x)) [x] = [].                                                                                           
Proof. 
  intros.
  unfold mproj_rcv_symbol. 
  simpl.
  rewrite app_nil_r.
  rewrite rcv_snd_notb.
  rewrite H. simpl.  
  rewrite andb_false_r.
  reflexivity.
Qed.

Lemma mproj_rcv_snd_eq :
  forall (p q : participant) (m : message) (H_neq_async : sender_receiver_neq_async (Snd p q m)), 
    mproj_rcv_symbol p q (Snd p q m ↾ H_neq_async) = []. 
Proof.         
  intros.
  unfold mproj_rcv_symbol. 
  simpl.
  rewrite rcv_snd_notb.
  rewrite andb_false_r.
  reflexivity. 
Qed.

Lemma mproj_rcv_snd_neq :
  forall (p q p' q' : participant) (m : message) (H_neq_async : sender_receiver_neq_async (Snd p q m)), 
    mproj_rcv_symbol p' q' (Snd p q m ↾ H_neq_async) = []. 
Proof.         
  intros.
  unfold mproj_rcv_symbol. 
  simpl.
  rewrite rcv_snd_notb.
  rewrite andb_false_r.
  reflexivity. 
Qed.

Lemma mproj_snd_rcv_eq :
  forall (p q : participant) (m : message) (H_neq_async : sender_receiver_neq_async (Rcv p q m)), 
    mproj_snd_symbol p q (Rcv p q m ↾ H_neq_async) = []. 
Proof.         
  intros.
  unfold mproj_snd_symbol. 
  simpl.
  rewrite snd_rcv_notb.
  rewrite andb_false_r.
  reflexivity. 
Qed.

Lemma mproj_snd_rcv_neq :
  forall (p q p' q': participant) (m : message) (H_neq_async : sender_receiver_neq_async (Rcv p q m)), 
    mproj_snd_symbol p' q' (Rcv p q m ↾ H_neq_async) = []. 
Proof.         
  intros.
  unfold mproj_snd_symbol. 
  simpl.
  rewrite snd_rcv_notb.
  rewrite andb_false_r.
  reflexivity. 
Qed.

Lemma mproj_snd_snd_eq :
  forall (p q : participant) (m : message) (H_neq_async : sender_receiver_neq_async (Snd p q m)), 
    mproj_snd_symbol p q (Snd p q m ↾ H_neq_async) = [m]. 
Proof.         
  intros.
  unfold mproj_snd_symbol. 
  simpl.
  do 2 rewrite participant_eqb_refl.
  replace (is_sndb (Snd p q m ↾ H_neq_async)) with true by easy.
  simpl. reflexivity. 
Qed.

Lemma mproj_snd_snd_neq :
  forall (p q p' q': participant) (m : message) (H_neq_async : sender_receiver_neq_async (Snd p q m)),
    ~ (p = p' /\ q = q') -> 
    mproj_snd_symbol p' q' (Snd p q m ↾ H_neq_async) = []. 
Proof.         
  intros.
  unfold mproj_snd_symbol. 
  simpl.
  replace (participant_eqb p p' && participant_eqb q q') with false.
  simpl. reflexivity.
  symmetry. rewrite andb_false_iff.
  apply not_and_or in H.
  destruct H. left.
  now apply participant_eqb_no. 
  right. now apply participant_eqb_no.
Qed.

Lemma mproj_rcv_rcv_eq :
  forall (p q : participant) (m : message) (H_neq_async : sender_receiver_neq_async (Rcv p q m)), 
    mproj_rcv_symbol p q (Rcv p q m ↾ H_neq_async) = [m]. 
Proof.         
  intros.
  unfold mproj_rcv_symbol. 
  simpl.
  do 2 rewrite participant_eqb_refl.
  replace (is_rcvb (Rcv p q m ↾ H_neq_async)) with true by easy.
  simpl. reflexivity. 
Qed.

Lemma mproj_rcv_rcv_neq :
  forall (p q p' q': participant) (m : message) (H_neq_async : sender_receiver_neq_async (Rcv p q m)),
    ~ (p = p' /\ q = q') -> 
    mproj_rcv_symbol p' q' (Rcv p q m ↾ H_neq_async) = []. 
Proof.         
  intros.
  unfold mproj_rcv_symbol. 
  simpl.
  replace (participant_eqb p p' && participant_eqb q q') with false.
  simpl. reflexivity.
  symmetry. rewrite andb_false_iff.
  apply not_and_or in H.
  destruct H. left.
  now apply participant_eqb_no. 
  right. now apply participant_eqb_no.
Qed.

Lemma mproj_rcv_rcv :
  forall (x : AsyncAlphabet),
    is_rcvb x = true ->
    flat_map (mproj_rcv_symbol (sender_async x) (receiver_async x)) [x] = [value_async x]. 
Proof. 
  intros.
  unfold mproj_rcv_symbol. 
  simpl.
  rewrite app_nil_r.
  rewrite H. 
  now repeat rewrite participant_eqb_refl. 
Qed.

Lemma mproj_rcv_extension :
  forall (w : list AsyncAlphabet) (p q : participant) (x : AsyncAlphabet),
    is_sndb x = true -> 
    mproj_rcv w p q = mproj_rcv (w ++ [x]) p q. 
Proof.     
  intros w p q x H_snd.
  unfold mproj_rcv; rewrite flat_map_app; simpl. 
  rewrite app_nil_r.
  assert (mproj_rcv_symbol p q x = []).
  { unfold mproj_rcv_symbol.
    assert (is_rcvb x = false).
    { rewrite rcv_snd_notb. now rewrite H_snd. }
    rewrite H. simpl.
    rewrite andb_false_r. reflexivity. }
  rewrite H.
  rewrite app_nil_r.
  reflexivity.
Qed.

Lemma mproj_snd_extension :
  forall (w : list AsyncAlphabet) (p q : participant) (x : AsyncAlphabet),
    is_rcvb x = true -> 
    mproj_snd w p q = mproj_snd (w ++ [x]) p q. 
Proof.     
  intros w p q x H_snd.
  unfold mproj_snd; rewrite flat_map_app; simpl. 
  rewrite app_nil_r.
  assert (mproj_snd_symbol p q x = []).
  { unfold mproj_snd_symbol.
    assert (is_sndb x = false).
    { rewrite snd_rcv_notb. now rewrite H_snd. }
    rewrite H. simpl.
    rewrite andb_false_r. reflexivity. }
  rewrite H.
  rewrite app_nil_r.
  reflexivity.
Qed.

Lemma mproj_snd_wproj_idempotent :
  forall (w : list AsyncAlphabet) (p q : participant),
    mproj_snd w p q =
      mproj_snd (wproj w p) p q.
Proof. 
  intros w p q. 
  induction w as [| a w IHw] using rev_ind.
  - reflexivity.
  - repeat rewrite mproj_snd_app.
    rewrite IHw.
    rewrite wproj_app.
    rewrite mproj_snd_app.
    rewrite app_inv_head_iff.  
    unfold wproj_symbol.
    destruct a as [a H_neq].  
    destruct a as [p' q' m | p' q' m].
    * (* When a is a send event *)
      destruct (classic (p' = p)).
      ** destruct (classic (q' = q)).
         *** unfold wproj, mproj_snd.
             unfold wproj_symbol, mproj_snd_symbol.
             unfold sender_async, receiver_async, value_async.
             simpl.
             replace (participant_eqb p' p) with true by
               (symmetry; now apply participant_eqb_correct). 
             replace (participant_eqb q' q) with true by
               (symmetry; now apply participant_eqb_correct). 
             simpl.
             replace (participant_eqb p' p) with true by
               (symmetry; now apply participant_eqb_correct). 
             replace (participant_eqb q' q) with true by
               (symmetry; now apply participant_eqb_correct). 
             easy.
         *** unfold wproj, mproj_snd.
             unfold wproj_symbol, mproj_snd_symbol.
             unfold sender_async, receiver_async, value_async.
             simpl.
             replace (participant_eqb p' p) with true by
               (symmetry; now apply participant_eqb_correct). 
             replace (participant_eqb q' q) with false by
               (symmetry; now apply participant_eqb_no). 
             simpl.
             replace (participant_eqb p' p) with true by
               (symmetry; now apply participant_eqb_correct). 
             replace (participant_eqb q' q) with false by
               (symmetry; now apply participant_eqb_no). 
             easy.
      ** destruct (classic (q' = q)).
         *** unfold wproj, mproj_snd.
             unfold wproj_symbol, mproj_snd_symbol.
             unfold sender_async, receiver_async, value_async.
             simpl.
             replace (participant_eqb p' p) with false by
               (symmetry; now apply participant_eqb_no). 
             replace (participant_eqb q' q) with true by
               (symmetry; now apply participant_eqb_correct). 
             simpl.
             replace (participant_eqb p' p) with false by
               (symmetry; now apply participant_eqb_no). 
             replace (participant_eqb q' q) with true by
               (symmetry; now apply participant_eqb_correct). 
             easy.
         *** unfold wproj, mproj_snd.
             unfold wproj_symbol, mproj_snd_symbol.
             unfold sender_async, receiver_async, value_async.
             simpl.
             replace (participant_eqb p' p) with false by
               (symmetry; now apply participant_eqb_no). 
             replace (participant_eqb q' q) with false by
               (symmetry; now apply participant_eqb_no). 
             simpl.
             replace (participant_eqb p' p) with false by
               (symmetry; now apply participant_eqb_no). 
             replace (participant_eqb q' q) with false by
               (symmetry; now apply participant_eqb_no). 
             easy.
    * (* When a is a receive event *)
      unfold wproj, mproj_snd.
      unfold wproj_symbol, mproj_snd_symbol.
      simpl.
      assert (H_false : is_sndb (Rcv p' q' m ↾ H_neq) = false).
      easy.
      rewrite H_false. rewrite andb_false_r. 
      destruct (participant_eqb q' p).
      simpl.
      rewrite H_false.
      rewrite andb_false_r. reflexivity.
      simpl. reflexivity.
Qed.

Lemma mproj_rcv_wproj_idempotent :
  forall (w : list AsyncAlphabet) (p q : participant),
    mproj_rcv w p q =
      mproj_rcv (wproj w q) p q.
Proof. 
  intros w p q. 
  induction w as [| a w IHw] using rev_ind.
  - reflexivity.
  - repeat rewrite mproj_rcv_app.
    rewrite IHw.
    rewrite wproj_app.
    rewrite mproj_rcv_app.
    rewrite app_inv_head_iff.  
    unfold wproj_symbol.
    destruct a as [a H_neq]. 
    destruct a as [p' q' m | p' q' m].
    * (* When a is a send event *)
      unfold wproj, mproj_rcv. 
      unfold wproj_symbol, mproj_rcv_symbol.
      simpl.
      assert (H_false : is_rcvb (Snd p' q' m ↾ H_neq) = false).
      easy.
      rewrite H_false. rewrite andb_false_r. 
      destruct (participant_eqb p' q).
      simpl.
      rewrite H_false.
      rewrite andb_false_r. reflexivity.
      simpl. reflexivity.  
    * (* When a is a receive event *)
      destruct (classic (p' = p)).
      ** destruct (classic (q' = q)).
         *** unfold wproj, mproj_rcv.
             unfold wproj_symbol, mproj_rcv_symbol.
             unfold sender_async, receiver_async, value_async.
             simpl.
             replace (participant_eqb p' p) with true by
               (symmetry; now apply participant_eqb_correct). 
             replace (participant_eqb q' q) with true by
               (symmetry; now apply participant_eqb_correct). 
             simpl.
             replace (participant_eqb p' p) with true by
               (symmetry; now apply participant_eqb_correct). 
             replace (participant_eqb q' q) with true by
               (symmetry; now apply participant_eqb_correct). 
             easy.
         *** unfold wproj, mproj_rcv.
             unfold wproj_symbol, mproj_rcv_symbol.
             unfold sender_async, receiver_async, value_async.
             simpl.
             replace (participant_eqb p' p) with true by
               (symmetry; now apply participant_eqb_correct). 
             replace (participant_eqb q' q) with false by
               (symmetry; now apply participant_eqb_no). 
             simpl.
             replace (participant_eqb p' p) with true by
               (symmetry; now apply participant_eqb_correct). 
             replace (participant_eqb q' q) with false by
               (symmetry; now apply participant_eqb_no). 
             easy.
      ** destruct (classic (q' = q)).
         *** unfold wproj, mproj_rcv.
             unfold wproj_symbol, mproj_rcv_symbol.
             unfold sender_async, receiver_async, value_async.
             simpl.
             replace (participant_eqb p' p) with false by
               (symmetry; now apply participant_eqb_no). 
             replace (participant_eqb q' q) with true by
               (symmetry; now apply participant_eqb_correct). 
             simpl.
             replace (participant_eqb p' p) with false by
               (symmetry; now apply participant_eqb_no). 
             replace (participant_eqb q' q) with true by
               (symmetry; now apply participant_eqb_correct). 
             easy.
         *** unfold wproj, mproj_rcv.
             unfold wproj_symbol, mproj_rcv_symbol.
             unfold sender_async, receiver_async, value_async.
             simpl.
             replace (participant_eqb p' p) with false by
               (symmetry; now apply participant_eqb_no). 
             replace (participant_eqb q' q) with false by
               (symmetry; now apply participant_eqb_no). 
             simpl.
             replace (participant_eqb p' p) with false by
               (symmetry; now apply participant_eqb_no). 
             replace (participant_eqb q' q) with false by
               (symmetry; now apply participant_eqb_no). 
             easy.
Qed.

(* If two words' role projections satisfy the prefix relation, then applying any message projection with that role active preserves the prefix relation *)
Lemma wproj_nil_means_mproj_snd_nil :
  forall (w : list AsyncAlphabet) (p : participant),
    wproj w p = [] -> 
    forall (q : participant),
      p <> q -> 
      mproj_snd w p q = []. 
Proof.
  intros w p H_nil q H_neq. 
  induction w as [|a w IHw] using rev_ind.
  - reflexivity.
  - rewrite wproj_app in H_nil.
    apply app_eq_nil in H_nil.
    destruct H_nil. spec IHw H.
    rewrite mproj_snd_app.
    rewrite IHw.
    unfold mproj_snd, mproj_snd_symbol. 
    destruct a as [a H_neq_a]. 
    destruct a as [p' q' m' | p' q' m'].
    * simpl.
      destruct (classic (p' = p)).
      simpl. 
      ** (* In this case we find a contradiction to H0 *)
        exfalso. 
        unfold wproj, wproj_symbol in H0.
        simpl in H0. 
        replace (participant_eqb p' p) with true in H0.
        simpl in H0.
        inversion H0.
        symmetry; now apply participant_eqb_correct.
      ** replace (participant_eqb p' p) with false.
         simpl. reflexivity.
         symmetry; now apply participant_eqb_no.
    * simpl.
      assert (H_rewrite : is_sndb (Rcv p' q' m' ↾ H_neq_a) = false) by easy.
      rewrite H_rewrite. 
      rewrite andb_false_r.       
      reflexivity. 
Qed.

Lemma wproj_nil_means_mproj_rcv_nil :
  forall (w : list AsyncAlphabet) (p : participant),
    wproj w p = [] -> 
    forall (q : participant),
      q <> p -> 
      mproj_rcv w q p = []. 
Proof.
  intros w p H_nil q H_neq. 
  induction w as [|a w IHw] using rev_ind.
  - reflexivity. 
  - rewrite wproj_app in H_nil.
    apply app_eq_nil in H_nil.
    destruct H_nil. spec IHw H.
    rewrite mproj_rcv_app.
    rewrite IHw. 
    unfold mproj_rcv, mproj_rcv_symbol. 
    destruct a as [a H_neq_a]. 
    destruct a as [p' q' m' | p' q' m'].
    * simpl.
      assert (H_rewrite :  is_rcvb (Snd p' q' m' ↾ H_neq_a) = false) by easy. 
      rewrite H_rewrite. 
      rewrite andb_false_r.       
      reflexivity.
    * simpl. destruct (classic (p = q')).
      simpl. 
      ** (* In this case we find a contradiction to H0 *)
        exfalso. 
        unfold wproj, wproj_symbol in H0.
        simpl in H0. 
        replace (participant_eqb q' p) with true in H0.
        simpl in H0.
        inversion H0.
        symmetry; now apply participant_eqb_correct.
      ** replace (participant_eqb q' p) with false.
         rewrite andb_false_r. 
         simpl. reflexivity.
         symmetry; now apply participant_eqb_no.
Qed.

Lemma mproj_snd_preserves_active_wproj_prefix :
  forall (w1 w2 : list AsyncAlphabet) (p : participant),
    prefix (wproj w1 p) (wproj w2 p) ->
    forall (q : participant),
      p <> q -> 
      prefix (mproj_snd w1 p q) (mproj_snd w2 p q).
Proof. 
  intros w1 w2 p H_pref q H_neq.
  remember (length w2) as n. 
  generalize dependent w2.
  generalize dependent w1.
  induction n as [|n IHn]; intros.  
  - symmetry in Heqn.
    apply nil_length_inv in Heqn.
    rewrite Heqn in H_pref. simpl in H_pref.
    inversion H_pref.
    symmetry in H.
    apply app_eq_nil in H.
    destruct H.
    assert (H_helper := wproj_nil_means_mproj_snd_nil).
    spec H_helper w1 p H q H_neq.
    rewrite H_helper.
    apply prefix_nil. 
  - destruct (destruct_list_last _ w2).
    rewrite H in Heqn; inversion Heqn; lia.
    destruct H as [a2 [w2_rest H_eq_w2]]; subst. 
    rewrite mproj_snd_app.
    rewrite wproj_app in H_pref.
    simpl in H_pref.
    rewrite app_nil_r in H_pref.
    (* We do case analysis on what wproj_symbol p a2 winds up being *) 
    destruct a2 as [a2 H_neq_a2].
    destruct a2 as [p' q' m' | p' q' m'].  
    * (* In the case that a2 is a send event *) 
      destruct (classic (p' = p)).
      ** (* When p' = p, wproj_symbol p a2 is a singleton *)
        unfold wproj_symbol in H_pref. simpl in H_pref.
         replace (participant_eqb p' p) with true in H_pref.
         2 : symmetry; now apply participant_eqb_correct. 
         simpl in H_pref.
         apply prefix_app_tail_or in H_pref.
         destruct H_pref.
         *** spec IHn w1 w2_rest H0. 
             spec IHn.
             rewrite app_length in Heqn.
             simpl in Heqn. lia.
             now apply prefix_app_r. 
         *** rewrite mproj_snd_wproj_idempotent.
             rewrite H0.
             rewrite mproj_snd_app.
             rewrite (mproj_snd_wproj_idempotent w2_rest).
             reflexivity.
      ** unfold wproj_symbol in H_pref. simpl in H_pref.
         replace (participant_eqb p' p) with false in H_pref.
         2 : symmetry; now apply participant_eqb_no. 
         simpl in H_pref.
         rewrite app_nil_r in H_pref.
         spec IHn w1 w2_rest H_pref.
         spec IHn.
         rewrite app_length in Heqn. simpl in Heqn.
         lia.
         now apply prefix_app_r.
    * (* In the case that a2 is a receive event *)
      destruct (classic (q' = p)).
      ** (* When q' = p, wproj_symbol p a2 is a singleton *)
        unfold wproj_symbol in H_pref. simpl in H_pref.
         replace (participant_eqb q' p) with true in H_pref.
         2 : symmetry; now apply participant_eqb_correct. 
         simpl in H_pref.
         apply prefix_app_tail_or in H_pref.
         destruct H_pref.
         *** spec IHn w1 w2_rest H0. 
             spec IHn.
             rewrite app_length in Heqn.
             simpl in Heqn. lia.
             now apply prefix_app_r. 
         *** rewrite mproj_snd_wproj_idempotent.
             rewrite H0.
             rewrite mproj_snd_app.
             rewrite (mproj_snd_wproj_idempotent w2_rest).
             reflexivity.
      ** unfold wproj_symbol in H_pref. simpl in H_pref.
         replace (participant_eqb q' p) with false in H_pref.
         2 : symmetry; now apply participant_eqb_no. 
         simpl in H_pref.
         rewrite app_nil_r in H_pref.
         spec IHn w1 w2_rest H_pref.
         spec IHn.
         rewrite app_length in Heqn. simpl in Heqn.
         lia.
         now apply prefix_app_r. 
Qed.

Lemma mproj_rcv_preserves_active_wproj_prefix :
  forall (w1 w2 : list AsyncAlphabet) (p : participant),
    prefix (wproj w1 p) (wproj w2 p) ->
    forall (q : participant),
      p <> q -> 
      prefix (mproj_rcv w1 q p) (mproj_rcv w2 q p).
Proof. 
  intros w1 w2 p H_pref q H_neq.
  remember (length w2) as n. 
  generalize dependent w2.
  generalize dependent w1.
  induction n as [|n IHn]; intros.  
  - symmetry in Heqn.
    apply nil_length_inv in Heqn.
    rewrite Heqn in H_pref. simpl in H_pref.
    inversion H_pref.
    symmetry in H.
    apply app_eq_nil in H.
    destruct H.
    assert (H_helper := wproj_nil_means_mproj_rcv_nil).
    spec H_helper w1 p H q.
    spec H_helper. easy. rewrite H_helper.
    apply prefix_nil. 
  - destruct (destruct_list_last _ w2).
    rewrite H in Heqn; inversion Heqn; lia.
    destruct H as [a2 [w2_rest H_eq_w2]]; subst. 
    rewrite mproj_rcv_app.
    rewrite wproj_app in H_pref.
    simpl in H_pref.
    rewrite app_nil_r in H_pref.
    (* We do case analysis on what wproj_symbol p a2 winds up being *) 
    destruct a2 as [a2 H_neq_a2].
    destruct a2 as [p' q' m' | p' q' m'].  
    * (* In the case that a2 is a send event *) 
      destruct (classic (p' = p)).
      ** (* When p' = p, wproj_symbol p a2 is a singleton *)
        unfold wproj_symbol in H_pref. simpl in H_pref.
         replace (participant_eqb p' p) with true in H_pref.
         2 : symmetry; now apply participant_eqb_correct. 
         simpl in H_pref.
         apply prefix_app_tail_or in H_pref.
         destruct H_pref.
         *** spec IHn w1 w2_rest H0. 
             spec IHn.
             rewrite app_length in Heqn.
             simpl in Heqn. lia.
             now apply prefix_app_r. 
         *** rewrite mproj_rcv_wproj_idempotent.
             rewrite H0.
             rewrite mproj_rcv_app.
             rewrite (mproj_rcv_wproj_idempotent w2_rest).
             reflexivity.
      ** unfold wproj_symbol in H_pref. simpl in H_pref.
         replace (participant_eqb p' p) with false in H_pref.
         2 : symmetry; now apply participant_eqb_no. 
         simpl in H_pref.
         rewrite app_nil_r in H_pref.
         spec IHn w1 w2_rest H_pref.
         spec IHn.
         rewrite app_length in Heqn. simpl in Heqn.
         lia.
         now apply prefix_app_r.
    * (* In the case that a2 is a receive event *)
      destruct (classic (q' = p)).
      ** (* When q' = p, wproj_symbol p a2 is a singleton *)
        unfold wproj_symbol in H_pref. simpl in H_pref.
         replace (participant_eqb q' p) with true in H_pref.
         2 : symmetry; now apply participant_eqb_correct. 
         simpl in H_pref.
         apply prefix_app_tail_or in H_pref.
         destruct H_pref.
         *** spec IHn w1 w2_rest H0. 
             spec IHn.
             rewrite app_length in Heqn.
             simpl in Heqn. lia.
             now apply prefix_app_r. 
         *** rewrite mproj_rcv_wproj_idempotent.
             rewrite H0.
             rewrite mproj_rcv_app.
             rewrite (mproj_rcv_wproj_idempotent w2_rest).
             reflexivity.
      ** unfold wproj_symbol in H_pref. simpl in H_pref.
         replace (participant_eqb q' p) with false in H_pref.
         2 : symmetry; now apply participant_eqb_no. 
         simpl in H_pref.
         rewrite app_nil_r in H_pref.
         spec IHn w1 w2_rest H_pref.
         spec IHn.
         rewrite app_length in Heqn. simpl in Heqn.
         lia.
         now apply prefix_app_r. 
Qed.

Lemma wproj_split_symbol_async_to_sync_eq :
  forall (x : AsyncAlphabet) (p : participant),
    is_active p x -> 
    wproj (split_symbol (async_to_sync x)) p = [x].
Proof. 
  intros. 
  destruct x as [x H_neq].
  destruct x as [p' q' m' | p' q' m'].
  - destruct H as [H_snd _].
    spec H_snd. easy.
    unfold sender_async in H_snd.
    simpl in H_snd.
    subst. 
    unfold async_to_sync, split_symbol, wproj, wproj_symbol. 
    simpl. unfold sender_sync, receiver_sync, value_sync.
    simpl. subst.
    rewrite participant_eqb_refl. 
    simpl.
    replace (participant_eqb q' p) with false.
    simpl. reflexivity.
    symmetry. now apply participant_eqb_no.
  - destruct H as [_ H_rcv].
    spec H_rcv. easy.
    unfold receiver_async in H_rcv.
    simpl in H_rcv.
    subst.
    unfold async_to_sync, split_symbol, wproj, wproj_symbol. 
    simpl. unfold sender_sync, receiver_sync, value_sync.
    simpl. subst.
    rewrite participant_eqb_refl. 
    simpl.
    replace (participant_eqb p' p) with false.
    simpl. reflexivity.
    symmetry. now apply participant_eqb_no.
Qed.

Lemma wproj_split_async_to_sync_eq :
  forall (x : AsyncAlphabet) (p : participant),
    is_active p x -> 
    wproj (split [async_to_sync x]) p = [x].
Proof. 
  intros. 
  destruct x as [x H_neq].
  destruct x as [p' q' m' | p' q' m'].
  - destruct H as [H_snd _].
    spec H_snd. easy.
    unfold sender_async in H_snd.
    simpl in H_snd.
    subst. 
    unfold async_to_sync, split_symbol, wproj, wproj_symbol. 
    simpl. unfold sender_sync, receiver_sync, value_sync.
    simpl. subst.
    rewrite participant_eqb_refl. 
    simpl.
    replace (participant_eqb q' p) with false.
    simpl. reflexivity.
    symmetry. now apply participant_eqb_no.
  - destruct H as [_ H_rcv].
    spec H_rcv. easy.
    unfold receiver_async in H_rcv.
    simpl in H_rcv.
    subst.
    unfold async_to_sync, split_symbol, wproj, wproj_symbol. 
    simpl. unfold sender_sync, receiver_sync, value_sync.
    simpl. subst.
    rewrite participant_eqb_refl. 
    simpl.
    replace (participant_eqb p' p) with false.
    simpl. reflexivity.
    symmetry. now apply participant_eqb_no.
Qed.

Lemma mproj_rcv_split_async_to_sync_eq :
  forall (x : AsyncAlphabet) (p q : participant),
    is_rcv x ->
    sender_async x = p -> 
    receiver_async x = q -> 
    mproj_rcv (split [async_to_sync x]) p q = [value_async x].
Proof. 
  intros. 
  destruct x as [x H_neq].
  destruct x as [p' q' m' | p' q' m'].
  - inversion H.
  - unfold sender_async in H0.
    simpl in H0.
    unfold receiver_async in H1.
    simpl in H1. subst.
    unfold async_to_sync, split_symbol, wproj, wproj_symbol. 
    simpl. unfold sender_sync, receiver_sync, value_sync.
    simpl.
    unfold mproj_rcv_symbol. simpl.
    repeat rewrite participant_eqb_refl. 
    simpl.
    reflexivity.
Qed. 

Lemma mproj_snd_split_async_to_sync_eq :
  forall (x : AsyncAlphabet) (p q : participant),
    is_snd x ->
    sender_async x = p -> 
    receiver_async x = q -> 
    mproj_snd (split [async_to_sync x]) p q = [value_async x].
Proof. 
  intros. 
  destruct x as [x H_neq].
  destruct x as [p' q' m' | p' q' m'].
  - unfold sender_async in H0.
    simpl in H0.
    unfold receiver_async in H1.
    simpl in H1. subst.
    unfold async_to_sync, split_symbol, wproj, wproj_symbol. 
    simpl. unfold sender_sync, receiver_sync, value_sync.
    simpl.
    unfold mproj_snd_symbol. simpl.
    repeat rewrite participant_eqb_refl. 
    simpl.
    reflexivity. 
  - inversion H.
Qed.

Lemma wproj_prefix_stream_to_list_max :
  forall (w : FinAsyncWord) (q : participant) (rho : InfSyncWord) (i j : nat), 
    wproj w q `prefix_of` wproj (split (stream_to_list rho i)) q ->
    j >= i -> 
    wproj w q `prefix_of` wproj (split (stream_to_list rho j)) q.
Proof. 
  intros w q rho i j H_pref H_geq.
  assert (H_useful := stream_to_list_prefix). 
  spec H_useful SyncAlphabet rho i j H_geq.
  apply prefix_split_prefix_iff in H_useful.
  eapply (wproj_preserves_prefix _ _ q) in H_useful.
  eapply PreOrder_Transitive. exact H_pref. assumption.
Qed.

Tactic Notation "clean" hyp(H) :=
  repeat rewrite split_app in H;
  simpl in H;
  repeat rewrite wproj_app in H;
  simpl in H;
  unfold sender_sync, receiver_sync, value_sync in H;
  simpl in H;
  try (rewrite wproj_symbol_sender_eq in H; easy);
  try (rewrite wproj_symbol_receiver_eq in H; easy);
  try (rewrite wproj_symbol_sender_neq in H; easy);
  try (rewrite wproj_symbol_receiver_neq in H; easy); 
  try rewrite app_nil_r in H; try rewrite app_nil_l in H;
  try rewrite app_nil_r in H; try rewrite app_nil_l in H;
  try rewrite app_nil_r in H; try rewrite app_nil_l in H;
  try rewrite app_nil_r in H; try rewrite app_nil_l in H;
  simpl. 

Tactic Notation "clean" :=
  repeat rewrite split_app;
  simpl;
  repeat rewrite wproj_app;
  simpl;
  unfold sender_sync, receiver_sync, value_sync;
  simpl;
  try (rewrite wproj_symbol_sender_eq; easy);
  try (rewrite wproj_symbol_receiver_eq; easy);
  try (rewrite wproj_symbol_sender_neq; easy);
  try (rewrite wproj_symbol_receiver_neq; easy); 
  try rewrite app_nil_r; try rewrite app_nil_l;
  try rewrite app_nil_r; try rewrite app_nil_l;
  try rewrite app_nil_r; try rewrite app_nil_l;
  try rewrite app_nil_r; try rewrite app_nil_l;
  simpl.

Tactic Notation "clean_mproj" :=
  try repeat rewrite mproj_rcv_app;
  try repeat rewrite mproj_snd_app;
  simpl;
  clean;
  try rewrite mproj_snd_snd_eq;
  try tauto; try easy; try reflexivity; 
  try rewrite  mproj_snd_snd_neq;
  try tauto; try easy; try reflexivity; 
  try rewrite   mproj_rcv_rcv_eq;
  try tauto; try easy; try reflexivity; 
  try rewrite   mproj_rcv_rcv_neq;
  try tauto; try easy; try reflexivity; 
  try rewrite   mproj_snd_rcv_eq;
  try tauto; try easy; try reflexivity; 
  try rewrite   mproj_snd_rcv_neq;
  try tauto; try easy; try reflexivity; 
  try rewrite   mproj_rcv_snd_eq;
  try tauto; try easy; try reflexivity; 
  try rewrite  mproj_rcv_snd_neq;
  try tauto; try easy; try reflexivity;
  clean. 

Tactic Notation "clean_mproj" hyp(H) :=
  repeat rewrite mproj_rcv_app mproj_snd_app in H;
  simpl in H;
  clean H;
  try rewrite mproj_snd_snd_eq in H;
  try easy; try reflexivity; 
  try rewrite  mproj_snd_snd_neq in H;
  try easy; try reflexivity; 
  try rewrite   mproj_rcv_rcv_eq in H;
  try easy; try reflexivity; 
  try rewrite   mproj_rcv_rcv_neq in H;
  try easy; try reflexivity; 
  try rewrite   mproj_snd_rcv_eq in H;
  try easy; try reflexivity; 
  try rewrite   mproj_snd_rcv_neq in H;
  try easy; try reflexivity; 
  try rewrite   mproj_rcv_snd_eq in H;
  try easy; try reflexivity; 
  try rewrite   mproj_rcv_snd_neq in H;
  try easy; try reflexivity; 
  clean H. 
