Require Import List Classical Nat Logic Lia FunctionalExtensionality ExtensionalityFacts Streams stdpp.list.
Import ListNotations. 
From Coq  
Require Import ssreflect. 
      
From CC     
  Require Import lib structures lts clts channel_compliant channel_complete protocol closure_conditions run canonical_implementation nmc_soundness rcc_soundness.


Section SCC.

Context {State : Type} {LocalState : Type}. 

(* Lemma 4.15: Send events preserve run prefixes *)
(* Intuitively, this lemma says that firing send events does not change what already happened *) 
(* If you take any possible run in I(w), and find the maximal prefix alpha that agrees with w according to sender p, then that same alpha is still compliant after extending w with x *)
(* This is a more precise statement that peels away the intersection set business, which talks about runs in terms of nasty existentials *)

Lemma in_wproj_split_run_means_async_to_sync_in_run :
  forall (x : AsyncAlphabet) (rho : FinSyncWord),
    In x (split rho) ->
    exists (l1 l2 : FinSyncWord) (y : SyncAlphabet),
      rho = l1 ++ [y] ++ l2 /\
        y = async_to_sync x. 
Proof.
  intros x rho H_in.
  induction rho as [|a rho IHrho] using rev_ind.
  - simpl in H_in.
    contradiction.
  - rewrite split_app in H_in.
    apply in_app_or in H_in.
    destruct H_in as [H_old | H_new].
    * spec IHrho H_old.
      destruct IHrho as [l1 [l2 [y [H_split H_eq]]]].
      exists l1, (l2 ++ [a]), y.
      split.
      rewrite H_split.
      rewrite <- app_assoc. 
      rewrite <- app_assoc. reflexivity. assumption.
    * clear IHrho.
      exists rho, [], a.
      split.
      now rewrite app_nil_r.
      destruct a as [a H_neq]. 
      destruct a as [p q m].
      unfold async_to_sync. 
      clean H_new. destruct H_new.
      subst. simpl. 
      sigma_equal. destruct H. 
      subst. simpl. sigma_equal.
      contradiction.
Qed.

(* A consequence of SCC is that if a role can still take a send action, then the trace cannot be terminated *) 
(* For this we rely on sink finality of S to show a contradiction *) 
Lemma send_extension_means_non_terminated_trace :
  forall (S : @LTS SyncAlphabet State) (T : CLTS) (w : FinAsyncWord) (x : AsyncAlphabet), 
    GCLTS S ->
    @SCC State S -> 
    canonical_implementation S T -> 
    is_snd x ->
    @is_clts_trace LocalState T (w ++ [x]) ->
    @is_finite_clts_word LocalState T w ->
    False. 
Proof.
  intros S T w x H_GCLTS H_SCC H_canonical H_snd H_trace_wx H_word.
  assert (H_df : deadlock_free S). { unfold GCLTS in *; tauto. } 
  destruct H_word as [c_w [H_reach_w H_final_w]]. 
  assert (H_trace_w : @is_clts_trace LocalState T w).
  { exists c_w. assumption. }
  assert (H_sink_final : sink_final S) by (unfold GCLTS in H_GCLTS; tauto).
  (* Establishing that x = p!q:m *) 
  destruct x as [x H_neq].  
  destruct x as [p q m | p q m].
  2 : { inversion H_snd. }
  clear H_snd.
  remember (get_local_state c_w p) as s. 
  (* Reasoning: from the fact that wx is an canonical implementation trace, we can obtain a run in S that matches wx, and a state that is simultaneously reachable with s for rho *)
  (* Then we say that rho must be extensible to also send x *)
  (* And find a contradiction to S being sink final *)
  unfold SCC in H_SCC.  
  (* But first we need to concoct all the ingredients to instantiate SCC with *)
  (* Begin: Obtaining a global transition from p's transition on x *) 
  destruct H_trace_wx as [c_wx H_reach_wx].
  assert (H_helper := @canonical_implementation_local_transition_means_global_transition State LocalState). 
  spec H_helper S T (wproj w p) (Snd p q m ↾ H_neq) p (get_local_state c_w p) (get_local_state c_wx p).
  spec H_helper H_df H_canonical. 
  spec H_helper. 
  (* Establishing that T_p reaches (get_local_state c_w p) on (wproj w p) *)
  { 
    apply (clts_reachable_means_implementation_reachable T w c_w H_reach_w p).
  }
  spec H_helper.
  (* Establishing that T_p reaches (get_local_state c_wx p) on (wproj wx p) *)
  {
    assert (H_useful := clts_reachable_means_implementation_reachable T (w ++ [Snd p q m ↾ H_neq]) c_wx H_reach_wx p).
    clean H_useful.
  }
  (* End: Obtaining a global transition from p's transition on x *)
  destruct H_helper as [s1 [s2 [rho_p [H_reach_rho_p [H_rho_p_eq H_transition]]]]].
  assert (H_neq_sync : sender_receiver_neq_sync (Event p q m)) by easy.
  assert (H_eq : (async_to_sync
                    (Snd p q m ↾ H_neq)) = (exist _ (Event p q m) H_neq_sync)).
  { unfold async_to_sync.  simpl.
    sigma_equal. } 
  rewrite H_eq in H_transition. 
  clear H_eq.
  (* s1 -> s2 is the transition we obtained *) 
  (* Now we can instantiate the first parts of SCC *)
  spec H_SCC s1 s2 (Event p q m ↾ H_neq_sync) H_transition.
  (* Begin: Obtaining a simultaneously reachable state as s1 *)
  (* In particular, we want to find a *final* simultaneously reachable state *) 
  (* Such a state must exist because s is final in c_w *)
  spec H_canonical p.
  destruct H_canonical as [H_finite_words _].
  spec H_finite_words (wproj w p). 
  destruct H_finite_words as [H_finite_words _].
  spec H_finite_words.
  { (* We need to show that wproj w p is final in T_p *)
    exists s. split.
    rewrite Heqs.
     eapply clts_reachable_means_implementation_reachable.
     exact H_reach_w.
     unfold Configuration_final in H_final_w.
     destruct H_final_w as [H_final_state _].
     spec H_final_state p.
     rewrite Heqs. assumption. }
  (* From the fact that p is in a final state in c_w,
     we can obtain a finite protocol word in S whose projection onto p is equal to wproj w p *) 
  (* And we actually don't need to care what any of the other roles are doing in this word w_fin *)
  destruct H_finite_words as [w_fin [H_w_fin H_eq_w_fin]].
  (* From this finite protocol word we can obtain a final state *)
  (* And this final state we shall show to have outgoing transitions *)
  destruct H_w_fin as [w_fin_run [H_reach_w_fin_run [H_role H_cc]]]. 
  destruct H_reach_w_fin_run as [s_w_fin_run [H_reach_w_fin_run H_final_w_fin_run]].
  (* Now we have our finality premise, H_final_w_fin_run *)
  spec H_SCC s_w_fin_run.
  spec H_SCC.
  exists (wproj w p). split; unfold reachable_for, reachable_for_on.
  exists rho_p. tauto.
  exists w_fin_run. split. 
  tauto. 
  clean.
  rewrite <- H_eq_w_fin.
  spec H_role p.
  assumption. 
  (* Now we can finally show that this final state s_w_fin_run has outgoing transitions and contradict sink finality *)
  destruct H_SCC as [s2' H_reach_s2'].
  clean H_reach_s2'. 
  rewrite wproj_symbol_sender_eq in H_reach_s2'.
  reflexivity.
  destruct H_reach_s2' as [rho_seg [H_reach_rho_seg H_about_rho_seg]].
  destruct rho_seg. 
  { simpl in H_about_rho_seg.
    subst. 
    inversion H_about_rho_seg. }
  apply lts.Reachable_behead in H_reach_rho_seg. 
  destruct H_reach_rho_seg as [s' [H_reach_rho_seg' H_transition']].
  (* Contradicting sink finality *)
  spec H_sink_final s_w_fin_run s' s0.
  spec H_sink_final.
  split. exact H_final_w_fin_run. exact H_transition'.
  contradiction.
Qed. 

(* Another consequence of SCC and sink finality is that if a sender has exhausted all actions along a run, yet it can still take an action, then the run cannot be final *) 
Lemma send_extension_means_exhausted_run_non_final :
  forall (S : @LTS SyncAlphabet State) (T : CLTS) (w : FinAsyncWord) (x : AsyncAlphabet) (rho_fin : FinSyncWord) (s : State),
    GCLTS S ->
    @SCC State S -> 
    canonical_implementation S T -> 
    @is_clts_trace LocalState T w ->
    possible_run_prefix S rho_fin w ->
    @lts.Reachable SyncAlphabet State S (s0 S) rho_fin s -> 
    (* If the sender has exhausted all actions along this run *) 
    wproj (split rho_fin) (sender_async x) = wproj w (sender_async x) ->
    is_snd x ->
    @is_clts_trace LocalState T (w ++ [x]) ->
    final S s ->
    False. 
Proof.
  intros S T w x rho_fin s H_GCLTS H_SCC H_canonical H_trace_w [H_run H_compliant] H_reach H_exhaust H_snd H_trace_wx H_final.
  assert (H_df : deadlock_free S). { unfold GCLTS in *; tauto. } 
  assert (H_sink_final : sink_final S) by (unfold GCLTS in H_GCLTS; tauto). 

  spec H_sink_final s.
  (* Reasoning: from the fact that wx is an canonical implementation trace, we can obtain a run in S that matches wx, and a state that is simultaneously reachable with s for rho *)
  (* Then we say that rho must be extensible to also send x *)
  (* And find a contradiction to S being sink final *)
  (* Establishing that x = p!q:m *) 
  destruct x as [x H_neq].  
  destruct x as [p q m | p q m].
  2 : { inversion H_snd. }
  clear H_snd. 
  (* We find an extension to alpha using SCC directly *) 
  unfold SCC in H_SCC.  
  (* But first we need to concoct all the ingredients to instantiate SCC with *)
  (* Begin: Obtaining a global transition from p's transition on x *) 
  destruct H_trace_w as [c_w H_reach_w].
  destruct H_trace_wx as [c_wx H_reach_wx].
  assert (H_helper := @canonical_implementation_local_transition_means_global_transition State LocalState). 
  spec H_helper S T (wproj w p) (Snd p q m ↾ H_neq) p (get_local_state c_w p) (get_local_state c_wx p). spec H_helper H_df H_canonical. 
  spec H_helper. 
  (* Establishing that T_p reaches (get_local_state c_w p) on (wproj w p) *)
  { 
    apply (clts_reachable_means_implementation_reachable T w c_w H_reach_w p).
  }
  spec H_helper.
  (* Establishing that T_p reaches (get_local_state c_wx p) on (wproj wx p) *)
  {
    assert (H_useful := clts_reachable_means_implementation_reachable T (w ++ [Snd p q m ↾ H_neq]) c_wx H_reach_wx p).
    clean H_useful.
  }
  (* End: Obtaining a global transition from p's transition on x *)
  destruct H_helper as [s1 [s2 [rho_p [H_reach_rho_p [H_rho_p_eq H_transition]]]]].
  assert (H_neq_sync : sender_receiver_neq_sync (Event p q m)) by easy.
  assert (H_eq : (async_to_sync
                    (Snd p q m ↾ H_neq)) = (exist _ (Event p q m) H_neq_sync)).
  { unfold async_to_sync.  simpl.
    sigma_equal. } 
  rewrite H_eq in H_transition. 
  clear H_eq.
  (* s1 -> s2 is the transition we obtained *) 
  (* Now we can instantiate the first parts of SCC *)
  spec H_SCC s1 s2 (Event p q m ↾ H_neq_sync) H_transition.
  (* Begin: Obtaining a simultaneously reachable state as s1 *)
  (* This state comes from the existing possible run, specifically from alpha *)
  spec H_SCC s. 
  spec H_SCC.
  exists (wproj w p). split; unfold reachable_for, reachable_for_on.
  exists rho_p. tauto.
  exists rho_fin. tauto.
  destruct H_SCC as [s2' H_reach_s2'].
  clean H_reach_s2'. 
  rewrite wproj_symbol_sender_eq in H_reach_s2'.
  reflexivity.
  destruct H_reach_s2' as [rho_seg [H_reach_rho_seg H_about_rho_seg]].
  destruct rho_seg. 
  { simpl in H_about_rho_seg.
    subst. 
    inversion H_about_rho_seg. }
  apply lts.Reachable_behead in H_reach_rho_seg. 
  destruct H_reach_rho_seg as [s' [H_reach_rho_seg' H_transition']].
  spec H_sink_final s' s0.
  spec H_sink_final.
  split. exact H_final. tauto.
  contradiction.
Qed. 

(* General lemma saying that runs can always be extended to match a send action *) 
Lemma send_extension_exists_run_extension :
forall (S : @LTS SyncAlphabet State) (T : CLTS) (w : FinAsyncWord) (x : AsyncAlphabet) (rho_fin : FinSyncWord),
    GCLTS S ->
    @NMC State S -> 
    @SCC State S -> 
    @RCC State S ->
    canonical_implementation S T -> 
    @is_clts_trace LocalState T w ->
    possible_run_prefix S rho_fin w ->
    (* If the sender has exhausted all actions along this run *) 
    wproj (split rho_fin) (sender_async x) = wproj w (sender_async x) ->
    is_snd x ->
    @is_clts_trace LocalState T (w ++ [x]) ->
    exists (beta : FinSyncWord),
      possible_run_prefix S (rho_fin ++ beta) (w ++ [x]). 
Proof.
  intros S T w x rho_fin H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_w [H_run H_compliant] H_exhaust H_snd H_trace_wx.
  assert (H_df : deadlock_free S). { unfold GCLTS in *; tauto. } 
  (* Establishing that x = p!q:m *) 
  destruct x as [x H_neq].  
  destruct x as [p q m | p q m].
  2 : { inversion H_snd. }
  clear H_snd. 
  (* We find an extension to alpha using SCC directly *) 
  unfold SCC in H_SCC.  
  (* But first we need to concoct all the ingredients to instantiate SCC with *)
  (* Begin: Obtaining a global transition from p's transition on x *) 
  destruct H_trace_w as [c_w H_reach_w].
  destruct H_trace_wx as [c_wx H_reach_wx].
  assert (H_helper := @canonical_implementation_local_transition_means_global_transition State LocalState). 
  spec H_helper S T (wproj w p) (Snd p q m ↾ H_neq) p (get_local_state c_w p) (get_local_state c_wx p). spec H_helper H_df H_canonical. 
  spec H_helper. 
  (* Establishing that T_p reaches (get_local_state c_w p) on (wproj w p) *)
  { 
    apply (clts_reachable_means_implementation_reachable T w c_w H_reach_w p).
  }
  spec H_helper.
  (* Establishing that T_p reaches (get_local_state c_wx p) on (wproj wx p) *)
  {
    assert (H_useful := clts_reachable_means_implementation_reachable T (w ++ [Snd p q m ↾ H_neq]) c_wx H_reach_wx p).
    clean H_useful.
  }
  (* End: Obtaining a global transition from p's transition on x *)
  destruct H_helper as [s1 [s2 [rho_p [H_reach_rho_p [H_rho_p_eq H_transition]]]]].
  assert (H_neq_sync : sender_receiver_neq_sync (Event p q m)) by easy.
  assert (H_eq : (async_to_sync
                    (Snd p q m ↾ H_neq)) = (exist _ (Event p q m) H_neq_sync)).
  { unfold async_to_sync.  simpl.
    sigma_equal. } 
  rewrite H_eq in H_transition. 
  clear H_eq.
  (* s1 -> s2 is the transition we obtained *) 
  (* Now we can instantiate the first parts of SCC *)
  spec H_SCC s1 s2 (Event p q m ↾ H_neq_sync) H_transition.
  (* Begin: Obtaining a simultaneously reachable state as s1 *)
  (* This state comes from the existing possible run, specifically from alpha *) 
  assert (H_trace_alpha : @is_trace SyncAlphabet State S rho_fin).
  {
    destruct H_run as [s_rho_fin H_reach_rho_fin]. 
    exists s_rho_fin. tauto.  } 
  destruct H_trace_alpha as [s_alpha H_reach_alpha].
  (* End: Obtaining a simultaneously reachable state as s1, s_alpha *)
  (* Now we can instantiate the next part of SCC *)
  spec H_SCC s_alpha.
  spec H_SCC.
  (* Establishing simultaneous reachability of s_alpha and s1 *)
  { exists (wproj w p).
    unfold sender_sync, reachable_for, reachable_for_on.
    split.
    exists rho_p. split. assumption.
    simpl. assumption.
    unfold reachable_for, reachable_for_on. 
    exists rho_fin. split.
    assumption. simpl.
    assumption. } 
  (* Finished instantiating H_SCC, from which we obtain s2' *)
  (* We know from SCC that s2' is reachable on x from s_alpha for p *)
  destruct H_SCC as [s2' H_reach_s2'].
  unfold reachable_for_on in H_reach_s2'.
  destruct H_reach_s2' as [beta [H_reach_alphabeta H_beta_p]].  
  (* From SCC we obtain s2' and this run segment beta,
     during which p's only action is to do p!q:m *)
  (* H_beta_p tells us that alphabeta is compliant with wx for p *)
  (* Cleaning up H_reach_s2' *)
  unfold split_symbol in H_beta_p.
  clean H_beta_p.
  rewrite wproj_symbol_sender_eq in H_beta_p. reflexivity.
  rewrite wproj_symbol_receiver_neq in H_beta_p. easy.
  unnil H_beta_p.
  exists beta.
  split.
  exists s2'. 
  eapply lts.Reachable_app.  exact H_reach_alpha.
  exact H_reach_alphabeta.
  intro p0.
  destruct (classic (p = p0)).
  subst p0.
  clean.
  rewrite wproj_symbol_sender_eq. reflexivity.
  simpl in H_exhaust.
  rewrite H_exhaust.
  apply prefix_app.
  rewrite H_beta_p.
  assert (H_rewrite : [Snd p q m ↾ H_neq] = [Snd p q m ↾ H_neq_sync]). 
  f_equal. sigma_equal.
  rewrite H_rewrite. reflexivity. 
  clean.
  rewrite wproj_symbol_sender_neq. assumption.
  rewrite app_nil_r.
  spec H_compliant p0.
  apply prefix_app_r. assumption.
Qed. 

(* This lemma is a special case of the previous one when the relevant event can be tacked on directly *) 
(* In the case that rho_fin <= rho_fin_max *)
(* Note that here we actually need the fact that rho_fin is a finite maximal run *) 
(* In order to show that it must contain a next action matching x *)
(* Otherwise a possible run prefix could just stop short at p's actions in w *)
(* Update: this is not true, just need that rho_fin is strictly longer than alpha *) 
(* Reminder: this extension proof uses both sender-driven choice and no mixed choice *) 
Lemma send_extension_exists_direct_run_extension :
forall (S : @LTS SyncAlphabet State) (T : CLTS) (w : FinAsyncWord) (x : AsyncAlphabet) (rho_fin : FinSyncWord) (alpha : FinSyncWord),
    GCLTS S ->
    @NMC State S -> 
    @SCC State S -> 
    @RCC State S ->
    canonical_implementation S T -> 
    @is_clts_trace LocalState T w ->
    possible_run_prefix S rho_fin w -> 
    is_alpha rho_fin alpha w (sender_async x) ->
    (* Instead of requiring that rho_fin is maximal,
       this is actually all that you need *) 
    alpha <> rho_fin -> 
    is_snd x ->
    @is_clts_trace LocalState T (w ++ [x]) ->
    @is_trace SyncAlphabet State S (alpha ++ [async_to_sync x]). 
Proof.        
  intros S T w x rho_fin alpha H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_w H_possible H_max H_neq_alpha_rho_fin H_snd H_trace_wx. 
  assert (H_df : deadlock_free S). { unfold GCLTS in *; tauto. } (* Establishing that x = p!q:m *) 
  destruct x as [x H_neq].  
  destruct x as [p q m | p q m].
  2 : { inversion H_snd. }
  clear H_snd. 
  (* We find an extension to alpha using SCC directly *) 
  unfold SCC in H_SCC.  
  (* But first we need to concoct all the ingredients to instantiate SCC with *)
  (* Begin: Obtaining a global transition from p's transition on x *) 
  destruct H_trace_w as [c_w H_reach_w].
  destruct H_trace_wx as [c_wx H_reach_wx].
  assert (H_helper := @canonical_implementation_local_transition_means_global_transition State LocalState). 
  spec H_helper S T (wproj w p) (Snd p q m ↾ H_neq) p (get_local_state c_w p) (get_local_state c_wx p). spec H_helper H_df H_canonical. 
  spec H_helper. 
  (* Establishing that T_p reaches (get_local_state c_w p) on (wproj w p) *)
  { 
    apply (clts_reachable_means_implementation_reachable T w c_w H_reach_w p).
  }
  spec H_helper.
  (* Establishing that T_p reaches (get_local_state c_wx p) on (wproj wx p) *)
  {
    assert (H_useful := clts_reachable_means_implementation_reachable T (w ++ [Snd p q m ↾ H_neq]) c_wx H_reach_wx p).
    clean H_useful.
  }
  (* End: Obtaining a global transition from p's transition on x *)
  destruct H_helper as [s1 [s2 [rho_p [H_reach_rho_p [H_rho_p_eq H_transition]]]]].
  assert (H_neq_sync : sender_receiver_neq_sync (Event p q m)) by easy.
  assert (H_eq : (async_to_sync
                    (Snd p q m ↾ H_neq)) = (exist _ (Event p q m) H_neq_sync)).
  { unfold async_to_sync.  simpl.
    sigma_equal. } 
  rewrite H_eq in H_transition. 
  clear H_eq.
  (* s1 -> s2 is the transition we obtained *) 
  (* Now we can instantiate the first parts of SCC *)
  spec H_SCC s1 s2 (Event p q m ↾ H_neq_sync) H_transition.
  (* Begin: Obtaining a simultaneously reachable state as s1 *)
  (* This state comes from the existing possible run, specifically from alpha *) 
  assert (H_trace_alpha : @is_trace SyncAlphabet State S alpha).
  { destruct H_max as [H_pref_alpha [H_eq H_max]].
    destruct H_possible as [[s_rho_fin H_reach_rho_fin] _].
    eapply lts_trace_prefix_closed with rho_fin. 
    exists s_rho_fin. tauto. assumption. } 
  destruct H_trace_alpha as [s_alpha H_reach_alpha].
  (* End: Obtaining a simultaneously reachable state as s1, s_alpha *)
  (* Now we can instantiate the next part of SCC *)
  spec H_SCC s_alpha.
  spec H_SCC.
  (* Establishing simultaneous reachability of s_alpha and s1 *)
  { exists (wproj w p).
    unfold sender_sync, reachable_for, reachable_for_on.
    split.
    exists rho_p. split. assumption.
    simpl. assumption.
    unfold reachable_for, reachable_for_on. 
    exists alpha. split.
    assumption. simpl.
    unfold is_alpha in H_max.
    destruct H_max as [_ [H_goal _]]. 
    simpl in H_goal. now symmetry. }
  (* Finished instantiating H_SCC, from which we obtain s2' *)
  (* We know from SCC that s2' is reachable on x from s_alpha for p *)
  destruct H_SCC as [s2' H_reach_s2'].
  unfold reachable_for_on in H_reach_s2'.
  destruct H_reach_s2' as [beta [H_reach_alphabeta H_beta_p]].  
  (* From SCC we obtain s2' and this run segment beta,
     during which p's only action is to do p!q:m *)
  (* H_beta_p tells us that alphabeta is compliant with wx for p *)
  (* Cleaning up H_reach_s2' *)
  unfold split_symbol in H_beta_p.
  clean H_beta_p.
  rewrite wproj_symbol_sender_eq in H_beta_p. reflexivity.
  rewrite wproj_symbol_receiver_neq in H_beta_p. easy.
  unnil H_beta_p.
  (* Now we want to "find" Event p q m in beta *) 
  (* Obtaining the first synchronous event in beta *) 
  destruct beta as [| y beta].
  { (* Discharging the case where beta is empty *)
    (* Trivially beta cannot be empty, because it has to at least contain Event p q m *) 
    simpl in H_beta_p.
    inversion H_beta_p. }
  (* Now we have established that alpha ++ y ++ beta is a run prefix *) 
  (* We want to show that actually, the head of beta is Event p q m *)
  (* The reasoning goes as follows:
     Attention! This is where we need some extra facts about rho_fin
     rho_fin has to include at least one more action by p
     In other words, it cannot be the case that wproj (split rho_fin) p = wproj w p, otherwise we cannot draw any conclusions about the next action in alpha
     It is not strictly required for rho_fin to be a maximal run, it just has to be the case that rho_fin contains at least one more action for p that is not contained in w *) 
  
  (* In other words, we show that alpha is also maximal for alpha ++ y :: beta *)
  (* For this we need to appeal to the fact that alpha is maximal in rho_fin, and also critically s_alpha's outgoing transitions have p as the sender *)
  (* So whatever condition we need to impose on rho_fin is used here *) 
  (* And certainly imposing that rho_fin is maximal is too strong *)
  (* Dealing with rho_fin = alpha ++ y_fin ++ beta_fin *)
  apply lts.Reachable_behead in H_reach_alphabeta. 
  destruct H_reach_alphabeta as [s_alphay [H_reach_beta H_transition_y]]. 
  assert (H_max_alpha_copy := H_max).
  destruct H_max as [H_pref_alpha [H_eq_alpha H_max_alpha]].
  apply prefix_exists_suffix in H_pref_alpha.
  destruct H_pref_alpha as [rho_fin_suf H_rho_fin_split].
  destruct rho_fin_suf as [| y_fin rho_fin_suf].
  { (* Here we just need to show that it cannot be the case that alpha = rho_fin *)
    (* So actually we can just require this as a premise *)
    rewrite app_nil_r in H_rho_fin_split.
    symmetry in H_rho_fin_split. contradiction. 
  }
  (* Now we have established that rho_fin = alpha ++ y_fin ++ rho_fin_suf *)
  (* We can show that y_fin has p as its active role *)
  assert (H_useful := alpha_next_active (alpha ++ y_fin :: rho_fin_suf) w p alpha y_fin rho_fin_suf).
  spec H_useful. easy.
  spec H_useful. rewrite H_rho_fin_split in H_max_alpha_copy.
  assumption.
  (* H_useful gives us this fact *) 
  (* Cleanup and bookkeeping *)
  assert (H_reach_alpha_y_fin : @is_trace SyncAlphabet State S (alpha ++ [y_fin])). 
  {
    destruct H_possible as [H_rho_fin_run _].
    destruct H_rho_fin_run as [s_rho_fin H_reach_rho_fin].  
    eapply lts_trace_prefix_closed with rho_fin.
    exists s_rho_fin. assumption.
    rewrite H_rho_fin_split.
    replace (y_fin :: rho_fin_suf) with ([y_fin] ++ rho_fin_suf) by easy.
    apply prefix_app.
    apply prefix_app_r. reflexivity. }
  destruct H_reach_alpha_y_fin as [s_alphayfin H_reach_alphayfin].
  apply lts.Reachable_unwind in H_reach_alphayfin.
  destruct H_reach_alphayfin as [s_alpha' [H_reach_alpha' H_transition_yfin]].
  assert (H_eq : s_alpha' = s_alpha). 
  { eapply deterministic_word.
    destruct H_GCLTS as [H_det _].
    exact H_det.
    exact H_reach_alpha'.
    exact H_reach_alpha. }
  (* Now we finally do case analysis on whether p is the sender or receiver in y_fin *) 
  destruct H_useful as [H_true | H_false].
  - (* In the case that p is the sender in y_fin *)
    (* Then all other outgoing transitions from alpha must also have p as their sender, by the fact that S is sender-driven *)
    (* And because y is an outgoing transition, y must have p as its sender *)
    (* Then y must be Event p q m *)
    (* And we get our extension *)
    destruct H_GCLTS as [_ [H_sender_driven _]]. 
    spec H_sender_driven s_alpha s_alphay s_alphayfin.
    spec H_sender_driven y y_fin H_transition_y.
    spec H_sender_driven.
    rewrite <- H_eq. assumption.
    (* Now we have established that sender_sync y = sender_sync y_fin = p *)
    assert (H_eq_y_fin : y = Event p q m ↾ H_neq_sync). 
    { destruct y as [[p' q' m'] H_neq_y]. 
      simpl in H_sender_driven.
      rewrite H_true in H_sender_driven.
      clean H_sender_driven.
      clean H_beta_p.
      rewrite wproj_symbol_sender_eq in H_beta_p. assumption.
      rewrite wproj_symbol_receiver_neq in H_beta_p. 
      { subst. easy. }
      rewrite app_nil_l in H_beta_p.
      inversion H_beta_p. subst.
      sigma_equal. }
    apply lts_trace_prefix_closed with (alpha ++ y :: beta).
    { exists s2'.
      apply lts.Reachable_app with s_alpha.
      now rewrite H_eq in H_reach_alpha'.
      replace (y :: beta) with ([y] ++ beta) by easy.
      apply lts.Reachable_app with s_alphay.
      rewrite <- H_eq.
      replace [y] with ([] ++ [y]) by now rewrite <- app_nil_l.
      apply lts.Reachable_step with s_alpha.
      rewrite H_eq. apply lts.Reachable_refl.
      assumption. assumption. }
    rewrite H_eq_y_fin.
    apply prefix_app.
    assert (Event p q m ↾ H_neq_sync = Event p q m ↾ H_neq) by sigma_equal.
    rewrite H. clear H.
    apply singleton_prefix_cons.
  - (* In the case that p is the receiver in y_fin *)
    (* We find a contradiction using NMC *)
    destruct y_fin as [y_fin H_neq_y_fin]. 
    destruct y_fin as [p' q' m'].
    simpl in H_false.
    subst.
    spec H_NMC s1 s2 s_alpha s_alphayfin.
    spec H_NMC q' q p' m m' H_neq_sync.
    assert (H_neq_y_fin' : sender_receiver_neq_sync (Event p' q' m')).  { subst. easy. }
    spec H_NMC H_neq_y_fin'.
    spec H_NMC H_transition.
    assert (H_eq : transition S s_alpha (Event p' q' m' ↾ H_neq_y_fin) = transition S s_alpha (Event p' q' m' ↾ H_neq_y_fin')).
    { f_equal. sigma_equal. }
    rewrite H_eq in H_transition_yfin.
    spec H_NMC H_transition_yfin.
    spec H_NMC.
    exists (wproj (split rho_p) q').
    split.
    exists rho_p. split. assumption.
    reflexivity.
    exists alpha. split. assumption.
    clean H_eq_alpha. rewrite <- H_eq_alpha.
    clean H_rho_p_eq.
    rewrite H_rho_p_eq. reflexivity.
    contradiction.
Qed. 

Lemma alpha_next_active' :
  ∀ (run alpha : FinSyncWord) (w : FinAsyncWord) 
    (p : participant), 
    is_alpha run alpha w p ->
    run <> alpha ->
    exists (y : SyncAlphabet) (suf : FinSyncWord),
      run = alpha ++ [y] ++ suf /\
        (sender_sync y = p ∨ receiver_sync y = p). 
Proof.
  intros run alpha w p H_alpha H_neq.
  destruct H_alpha as [H_pref [H_eq H_max]].
  apply prefix_exists_suffix in H_pref.
  destruct H_pref as [beta H_split].
  destruct beta. rewrite app_nil_r in H_split.
  contradiction.
  exists s, beta. split.
  easy.
  destruct (classic (sender_sync s = p ∨ receiver_sync s = p)).
  assumption.
  exfalso. 
  apply not_or_and in H.
  spec H_max (alpha ++ [s]). 
  spec H_max. rewrite H_split.
  apply app_singleton_prefix_app_cons. 
  spec H_max.
  destruct s as [s H_neq_s]. 
  destruct s as [p' q' m'].
  clean H.
  clean. rewrite wproj_symbol_sender_neq. tauto.
  rewrite wproj_symbol_receiver_neq. tauto.
  unnil. assumption.
  apply prefix_app_not in H_max. contradiction.
  easy.
Qed.

Lemma send_receive_mismatch_false :
forall (S : @LTS SyncAlphabet State) (T : CLTS) (w : FinAsyncWord) (x : AsyncAlphabet) (alpha : FinSyncWord) (y : SyncAlphabet),
    GCLTS S ->
    @NMC State S -> 
    canonical_implementation S T -> 
    @is_clts_trace LocalState T w ->
    wproj (split alpha) (sender_async x) = wproj w (sender_async x) -> 
    @is_trace SyncAlphabet State S (alpha ++ [y]) ->
    receiver_sync y = sender_async x -> 
    is_snd x ->
    @is_clts_trace LocalState T (w ++ [x]) ->
    False. 
Proof.
  intros S T w x alpha y H_GCLTS H_NMC H_canonical H_trace_w H_eq H_run_alphay H_role H_snd H_trace_wx.
  destruct y as [y H_neq_y].
  destruct y as [p q m].
  destruct x as [x H_neq_x]. 
  destruct x as [ p' q' m' | p' q' m'].
  2 : inversion H_snd.
  clean H_role. subst q.
  clear H_snd. simpl in H_eq.
  remember (wproj (split [Event p p' m ↾ H_neq_y]) p') as x'. 
  clean Heqx'. rewrite wproj_symbol_sender_neq in Heqx'.
  easy. rewrite wproj_symbol_receiver_eq in Heqx'.
  reflexivity. 
  rewrite app_nil_l in Heqx'.
  assert (H_rewrite1 : (wproj w p' ++ [Snd p' q' m' ↾ H_neq_x]) = (wproj (w ++ [Snd p' q' m' ↾ H_neq_x]) p')).
  clean. 
  assert (H_rewrite2 : (wproj (split alpha) p' ++ [Rcv p p' m ↾ H_neq_y]) = (wproj (split (alpha ++ [Event p p' m ↾ H_neq_y])) p')).
  clean. rewrite wproj_symbol_receiver_eq.
  reflexivity.
  rewrite wproj_symbol_sender_neq. easy.
  rewrite app_nil_l. reflexivity.
  assert (H_useful := NMC_implies_no_mixed_choice S T (wproj w p') (Snd p' q' m' ↾ H_neq_x) (Rcv p p' m ↾ H_neq_y) p' H_GCLTS H_NMC H_canonical).
  spec H_useful. 
  {
    destruct H_trace_wx as [c_wx H_reach_wx].
    exists (get_local_state c_wx p').
    clean.
    rewrite H_rewrite1. 
    eapply clts_reachable_means_implementation_reachable.
    exact H_reach_wx.
  }
  spec H_useful.
  { 
    rewrite <- H_eq.
    assert (H_useful' := @canonical_implementation_prefixes_include_protocol_prefixes State LocalState).
    spec H_useful' S T H_canonical (split (alpha ++ [Event p p' m ↾ H_neq_y])).
    spec H_useful'. 
    destruct H_run_alphay as [s_alphay H_reach_alphay].
    eapply split_run_protocol_prefix.
    unfold GCLTS in H_GCLTS. tauto.
    exact H_reach_alphay. 
    destruct H_useful' as [c_alphay H_reach_split_alphay].
    exists (get_local_state c_alphay p').
    rewrite H_rewrite2.
    eapply clts_reachable_means_implementation_reachable.
    exact H_reach_split_alphay.
  }
  spec H_useful. 
  easy. 
  inversion H_useful.
Qed.

(* Now this property diverges from the receive counterpart, which says that any possible run prefix that contains all the actions in w also contain all the actions in wx *) 
(* In the send case the pattern is rather: there exists a possible run prefix that starts with alpha_p that contains all the actions in wx *)
Lemma send_preserves_run_prefixes_finite :
forall (S : @LTS SyncAlphabet State) (T : CLTS) (w : FinAsyncWord) (x : AsyncAlphabet) (rho_fin : FinSyncWord) (alpha : FinSyncWord),
    GCLTS S ->
    @NMC State S -> 
    @SCC State S -> 
    @RCC State S ->
    canonical_implementation S T -> 
    @is_clts_trace LocalState T w ->
    possible_run_prefix S rho_fin w ->
    is_alpha rho_fin alpha w (sender_async x) ->
    (* Important and just strong enough premise to add *)
    alpha <> rho_fin -> 
    is_snd x ->
    @is_clts_trace LocalState T (w ++ [x]) ->
    exists (rho_fin' : FinSyncWord),
      prefix alpha rho_fin' /\
        alpha <> rho_fin' /\ 
        possible_run_prefix S rho_fin' (w ++ [x]). 
Proof.       
  intros S T w x rho_fin alpha H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_w H_possible H_max H_neq_alpha_rho_fin H_snd H_trace_wx. 
  remember (length w) as n.  
  generalize dependent w.
  generalize dependent x.
  generalize dependent rho_fin.
  generalize dependent alpha.
  induction n as [n' IHn'] using lt_wf_ind; intros. 
  (* Outer induction: strong induction on the length of wx about possible run prefixes *)
  (* Establishing that x = p!q:m *) 
  destruct x as [x H_neq].  
  destruct x as [p q m | p q m].
  2 : { inversion H_snd. }
  clear H_snd.
  simpl in *. 
  (* In the first step, we extend alpha with an action that agrees with x *)
  assert (H_candidate := send_extension_exists_direct_run_extension S T w (Snd p q m ↾ H_neq) rho_fin alpha H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_w H_possible H_max H_neq_alpha_rho_fin). 
  spec H_candidate. easy.
  spec H_candidate H_trace_wx.
  clean H_candidate.
  remember (Event p q m ↾ H_neq) as y. 
  (* What we have established is that there exists a direct transition label l from alpha to make a run/synchronous trace, and obviously this run prefix agrees with p *)
  (* Now what we need to establish is that every symbol in w can agree with the run prefix alpha ++ [y], because by construction this run prefix settles compliance for p *) 
  (* So if we can just show that all other symbols are also compliant with it, then we are good *)
  (* First we establish that w actually does not agree with alpha, otherwise we can just use it *) 
  destruct (classic (possible_run_prefix S (alpha) w)) as [H_done | H_undone].
  { exists (alpha ++ [y]). 
    split. now apply prefix_app_r.
    split. intro H_false.
    symmetry in H_false. apply list_app_identity_means_nil in H_false.
    inversion H_false.
    split. assumption.
    intros p0.
    destruct (classic (p = p0)).
    { subst. clean.
      unfold is_alpha in H_max.
      destruct H_max as [_ [H_goal _]].
      rewrite <- H_goal.
      simpl.
      rewrite wproj_symbol_sender_eq. reflexivity.
      apply prefix_app. 
      now apply prefix_app_r. 
    }
    { rewrite wproj_app.
      simpl. rewrite wproj_symbol_sender_neq. assumption.
      unnil.
      destruct H_done as [_ H_compliant].
      rewrite wproj_split_app.
      apply prefix_app_r.
      now spec H_compliant p0. }
    (* Proving the additional condition *)
  }
  apply not_and_or in H_undone.
  destruct H_undone as [H_false | H_disagree].
  { exfalso. apply H_false.
    eapply lts_trace_prefix_closed.  exact H_candidate.
    now apply prefix_app_r. } 
  (* Specifically, it must be the case that some participant does not agree with alpha *)
  (* Which means that w cannot be empty *) 
  (* Now we have established that w is non-empty, we state our inner claim that we prove by induction, which state that all prefixes of w agree with our candidate run prefix *)
  assert (H_inner : forall (w' : FinAsyncWord),
             prefix w' w ->
             (exists (beta' : FinSyncWord),
                 possible_run_prefix S (alpha ++ [Event p q m ↾ H_neq] ++ beta') w')). 
  { intros w' H_pref.
    induction w' as [|z' w' IHw'] using rev_ind.
    - (* Base case *)
      exists []. split.
      rewrite app_nil_r. rewrite <- Heqy.
      assumption. intros.
      apply prefix_nil. 
    - (* Induction step *)
      (* First we do case analysis on whether z' is a send or receive event *)
      (* We hope to establish off the bat that beta_w' is nonempty *)
      (* Currently solved this problem by explicitly including it in the inductive property *) 
      (* From IHw' we obtain a possible run prefix for (a'::w'),
       which is alpha ++ [y] ++ beta_w' *)
      (* Our obligation is to construct a possible run prefix for w'z' *)
      destruct z' as [z' H_neq_z'].
      destruct z' as [p' q' m' | p' q' m'].
      2 : { (* Receive case *)
        (* In the receive case, we know that the same run can be reused *)
        (* Using the induction hypothesis *) 
        remember (Rcv p' q' m' ↾ H_neq_z') as z'. 
        spec IHw'.
        { apply prefix_app_l in H_pref. assumption. }
        destruct IHw' as [beta_w' H_possible_w'].
        assert (H_helper := rcv_possible_run_prefix_still_possible S T w' (alpha ++ [Event p q m ↾ H_neq] ++ beta_w') H_GCLTS H_NMC H_SCC H_RCC H_canonical). 
        spec H_helper.
        { eapply clts_trace_prefix_closed with w.
          assumption. 
          apply prefix_app_l in H_pref. assumption. }
        spec H_helper H_possible_w' (Rcv p' q' m' ↾ H_neq_z').
        spec H_helper. easy.
        spec H_helper.
        {
          eapply clts_trace_prefix_closed with w.
          assumption.
          rewrite Heqz' in H_pref. assumption. 
        }
        exists beta_w'.
        rewrite Heqz'. 
        assumption. 
        (* And the receive case is done, easy as that *)
      }
      remember (Snd p' q' m' ↾ H_neq_z') as z'.
      (* Using the induction hypothesis in the send case *) 
      spec IHw'.
      { apply prefix_app_l in H_pref. assumption. }
      destruct IHw' as [beta_w' H_possible_w'].
      (* First we establish the important fact that for every role,
         if its part in w does not agree with alpha, then its point of disagreement comes after alpha *) 

      assert (H_point : forall (p : participant), (~ (prefix (wproj w p) (wproj (split (alpha)) p))) ->
                                             (prefix (wproj (split alpha) p) (wproj w p))). 
      { intros p0 H_disagree_p0.
        assert (H_useful := prefix_weak_total (wproj w p0) (wproj (split alpha) p0) (wproj (split rho_fin) p0)).
        spec H_useful.
        destruct H_possible as [_ H_compliant].
        now spec H_compliant p0.
        destruct H_max as [H_pref_alpha _].
        apply prefix_split_prefix_iff in H_pref_alpha.
        apply (wproj_preserves_prefix _ _ p0) in H_pref_alpha.
        spec H_useful H_pref_alpha.
        destruct H_useful. contradiction. assumption.
      }
      (* Natural thing to first do case analysis on is whether the old run is still possible for w'z' *) 
      destruct (classic (possible_run_prefix S (alpha ++ [Event p q m ↾ H_neq] ++ beta_w') (w' ++ [Snd p' q' m' ↾ H_neq_z']))) as [H_old | H_new].
      * (* In the case that the old one still works *)
        exists beta_w'.
        rewrite Heqz'. assumption.
      * (* Otherwise, it must be z' that is causing trouble *)
        apply not_and_or in H_new.
        destruct H_new as [H_false | H_disagree_z'].
        { exfalso. apply H_false.
          unfold possible_run_prefix in H_possible_w'.
          tauto. }
        apply not_all_ex_not in H_disagree_z'. 
        destruct H_disagree_z' as [p_dis H_disagree_z'].
        (* Now we establish that p_dis must be p', because nobody else should care *)
        assert (H_eq : p_dis = p').
        {
          destruct (classic (p_dis = p')).
          assumption.
          exfalso.
          apply H_disagree_z'.
          rewrite wproj_app. simpl.
          rewrite wproj_symbol_sender_neq.
          now apply not_eq_sym.
          unnil.
          destruct H_possible_w' as [_ H_compliant_w'].
          spec H_compliant_w' p_dis.
          assumption. }
        rewrite H_eq in H_disagree_z'.
        clear H_eq.
        (* We have now established that w'z' onto p' is not a prefix of rho_c onto p' *) 
        (* This could be due to two reasons: either there is a z'' in rho_c that does not match z',
           or rho_c simply isn't "long enough" and doesn't contain enough actions by p' *)
        (* We handle each case separately *)
        destruct (classic (wproj w' p' = wproj (split (alpha ++ [Event p q m ↾ H_neq] ++ beta_w')) p')) as [H_not_enough | H_disagreeing_z'].
        ** (* In the case that there are not enough actions, we can simply extend rho_c with something that makes it compliant with z', which is all that we need to care about *)
          (* Show that rho_c cannot be final, because z' exists *)
          (* This uses SCC and sink-finality *)
          (* Get an extension *)
          (* Now that the extension exists we can invoke some cousin of send_extension_exists_run_extension to get our requisite run *)
          (* Similarity spotted *)
          assert (H_useful := send_extension_exists_run_extension S T w' z' (alpha ++ [Event p q m ↾ H_neq] ++ beta_w') H_GCLTS H_NMC H_SCC H_RCC H_canonical).
          spec H_useful. 
          { eapply clts_trace_prefix_closed with w.
            assumption. 
            apply prefix_app_l in H_pref.
            assumption. }
          spec H_useful H_possible_w'.
          rewrite Heqz' in H_useful.
          spec H_useful. 
          { simpl.
            rewrite cons_middle in H_useful.
            now symmetry. } 
          spec H_useful. easy. 
          spec H_useful.
          { eapply clts_trace_prefix_closed with w.
            assumption. 
            rewrite Heqz' in H_pref.
            assumption.
          }
          destruct H_useful as [beta_easy H_possible_easy].
          rewrite Heqz'. exists (beta_w' ++ beta_easy).
          repeat rewrite <- app_assoc in H_possible_easy.
          assumption. 
        ** (* In the case that there are enough actions, but there is just a disagreement with z', we can split up the run's actions onto p' to find the incriminating one *) 
          (* And then we can argue that the incriminating one must occur after alpha *)
          assert (H_compliant_w'_p' : prefix (wproj w' p') (wproj (split (alpha ++ [Event p q m ↾ H_neq] ++ beta_w')) p')). 
          { destruct H_possible_w' as [_ H_compliant_w'].
            spec H_compliant_w' p'.
            assumption. }
          assert (H_useful := about_not_prefix _ (wproj w' p') (wproj (split (alpha ++ [Event p q m ↾ H_neq] ++ beta_w')) p') (Snd p' q' m' ↾ H_neq_z')). 
          spec H_useful H_compliant_w'_p'.
          spec H_useful.
          { intro H. apply H_disagree_z'.
            rewrite wproj_app. simpl.
            rewrite wproj_symbol_sender_eq. reflexivity.
            now rewrite app_nil_r. }
          spec H_useful.
          { (* Showing that the lengths are > *)
            (* Only need to rule out the case where the lengths are equal *)
            apply strict_prefix_means_length_lt. 
            destruct H_possible_w' as [_ H_compliant_w'].
            now spec H_compliant_w' p'.
            assumption. }
          (* Obtaining the splitting of rho_c into the parts for w', z', and the rest *)
          destruct H_useful as [not_z' [suf_p' [H_split_rho_c_p' H_about_not_z']]].
           (* Finding alpha_p' *)
          assert (H_useful := finite_unique_splitting (alpha ++ [y] ++ beta_w') w' p').
          rewrite Heqy in H_useful.
          spec H_useful H_compliant_w'_p'.
          destruct H_useful as [alpha_p' H_alpha_p'].
          (* Right after finding alpha_p' we do case analysis on whether it is equal to the run thus far *) 
          destruct (classic (alpha_p' = alpha ++ [y] ++ beta_w')) as [H_contra | H_fine].
          {
            (* This is actually the same case as before, where we can use the common premise that
               wproj w' p' = wproj (split (alpha ++ [Event p q m ↾ H_neq] ++ beta_w')) p' *) 
            (* So either we can find a run directly, or we can contradict SCC and sink finality *)
            assert (H_useful := send_extension_exists_run_extension S T w' z' (alpha ++ [Event p q m ↾ H_neq] ++ beta_w') H_GCLTS H_NMC H_SCC H_RCC H_canonical).
            spec H_useful. 
            { eapply clts_trace_prefix_closed with w.
              assumption. 
              apply prefix_app_l in H_pref.
              assumption. }
            spec H_useful H_possible_w'.
            rewrite Heqz' in H_useful.
            spec H_useful. 
            { simpl.
              rewrite cons_middle.
              rewrite Heqy in H_contra.
              destruct H_alpha_p' as [_ [H_rewrite _]].
              rewrite H_rewrite.
              symmetry.
              rewrite H_contra.
              reflexivity.
            } 
            spec H_useful. easy. 
            spec H_useful.
            { eapply clts_trace_prefix_closed with w.
              assumption. 
              rewrite Heqz' in H_pref.
              assumption.
            }
            destruct H_useful as [beta_easy H_possible_easy].
            rewrite Heqz'. exists (beta_w' ++ beta_easy).
            repeat rewrite <- app_assoc in H_possible_easy.
            assumption. 
          }
          (* H_fine now gets used all over the place later *)
          (* Now we can move on *)
          (* Case analysis on the relationship between alpha_p' and (alpha ++ beta1 ++ [y] *) 
          (* We appeal to the fact that they are both prefixes of the entire run, alpha ++ [y] ++ beta_w' *)
          (* To establish that one must be a prefix of the other *)
          assert (H_helper := prefix_weak_total alpha_p' (alpha ++ [y]) (alpha ++ [y] ++ beta_w')).
          spec H_helper.
          { unfold is_alpha in H_alpha_p'.
            rewrite Heqy. tauto.
          }
          spec H_helper.
          { repeat apply prefix_app. 
            now apply prefix_app_r.
          }
          
          (* In the case that alpha ++ [y] <= alpha_p', we are good *) 
          (* However, in the case that alpha_p' <= alpha ++ [y], we need to do further case analysis *)
          
          (* Case analysis on p''s alpha versus p's alpha *)        
          
          (* But really we want to do three cases,
             (1) alpha ++ [y] = alpha_p'
             (2) alpha_p' < alpha ++ [y]
             (3) alpha ++ [y] < alpha_p' *)
          (* Why did we want to do three cases? *)
          (* Actually we want to push the equality into the right case for convenience *)
          assert (H_good_cases : (alpha_p' `prefix_of` alpha ++ [y] /\ alpha_p' <> alpha ++ [y]) ∨ (alpha ++ [y] `prefix_of` alpha_p')).
          { destruct (classic (alpha_p' = alpha ++ [y])).
            subst. tauto. tauto. }           
          clear H_helper.
          (* destruct H_helper as [H_p_longer | H_p'_longer].  *)
          destruct H_good_cases as [H_p'_shorter | H_p'_longer]. 
          *** (* In the case that alpha_p' <= alpha ++ [y] *)
            destruct H_p'_shorter as [H_p'_shorter H_p'_strict]. 
            (* Now there is no more corner case where IHn' gets used *)
            (* What happens when alpha_p' = alpha ? *)
            (* This is such a bizarre case *)
            (* We first establish that alpha_p' <> alpha *) 
            destruct (classic (alpha_p' = alpha)) as [H_bizarre | H_p'_even_stricter]. 
            {
              subst alpha_p'. 
              (* Discharging the bizarro case *)
              (* We have to first establish that p' = p *)
              (* p' cannot be equal to q because of NMC *)
              (* Then we find a contradiction using H_pref and H_alpha the OG *)
              assert (H_useful := alpha_next_active).
              spec H_useful (alpha ++ [y] ++ beta_w') w' p' alpha y beta_w'.
              spec H_useful. reflexivity. 
              spec H_useful. rewrite Heqy.
              exact H_alpha_p'.
              destruct H_useful as [H_sender_p' | H_receiver_p']. 

              {
                (* When p' is the sender in y *)
                rewrite Heqy in H_sender_p'.
                unfold sender_sync in H_sender_p'.
                simpl in H_sender_p'.
                subst p.
                (* We find a contradiction to H_pref *)
                destruct H_alpha_p' as [_ [H_eq1 _]].
                destruct H_max as [_ [H_eq2 _]].
                apply (wproj_preserves_prefix _ _ p') in H_pref.
                rewrite wproj_app in H_pref.
                rewrite H_eq2 in H_pref.
                rewrite <- H_eq1 in H_pref.
                rewrite Heqz' in H_pref.
                simpl in H_pref.
                rewrite wproj_symbol_sender_eq in H_pref.
                reflexivity.
                rewrite app_nil_r in H_pref.
                apply prefix_app_not in H_pref.
                contradiction.
                easy. 
              } 
              {
                (* When p' is the receiver in y *)
                (* We invoke another fact that says that when an canonical implementation can do a receive by some role, there cannot exist a run in S that prescribes a send *)
                assert (H_useful := send_receive_mismatch_false S T w' z' alpha y H_GCLTS H_NMC H_canonical). 
                spec H_useful.
                { eapply clts_trace_prefix_closed with w.
                  assumption.
                  apply prefix_app_l in H_pref.
                  assumption. }
                spec H_useful.
                rewrite Heqz'. simpl.
                unfold is_alpha in H_alpha_p'.
                symmetry. tauto.
                spec H_useful H_candidate.
                spec H_useful.
                { now subst. }
                spec H_useful.
                { now subst. }
                spec H_useful.
                {
                  eapply clts_trace_prefix_closed with w.
                  assumption.
                  assumption. 
                }                  
                contradiction.
              }
            } (* Now there are truly no more corner cases where IHn' gets used *)
            {
              (* In the case where alpha_p' <> alpha *)
              assert (H_p'_shorter_copy := H_p'_shorter).
              apply prefix_app_tail_or in H_p'_shorter_copy. 
              destruct H_p'_shorter_copy as [H_p'_even_shorter | H_false].
              2 : { contradiction. }
              (* When alpha_p' < alpha *)
              clear IHn' Heqn. 
              exists beta_w'. 
              split.
              (* Showing that our new run is a synchronous trace of S *)
              {
                rewrite <- Heqy.
                unfold possible_run_prefix in H_possible_w'.
                rewrite Heqy. tauto. }
              (* Showing that every role is compliant with our new run *)
              {
                intros p0.
                destruct (classic (p' = p0)).
                {
                  (* In the case that we are dealing with p' *)
                  (* Here is where we need to argue that the symbol following alpha_p' in alpha ++ [y] is Event p' q' m' *)
                  (* Here is where we finally use H_point *) 
                  spec H_point p'.
                  spec H_point.
                  intro H_false.
                  (* We find a contradiction between w'z' not agreeing with alpha ++ [y], yet miraculously w agreeing with alpha *)
                  {
                    apply (wproj_preserves_prefix _ _ p') in H_pref.
                    assert (H_step : prefix (wproj (w' ++ [z']) p') (wproj (split alpha) p')) by now apply PreOrder_Transitive with (wproj w p').
                    apply (prefix_app_r (wproj (w' ++ [z']) p') (wproj (split alpha) p') (wproj (split ([Event p q m ↾ H_neq] ++ beta_w')) p')) in H_step. 
                    rewrite <- wproj_split_app in H_step.
                    rewrite Heqz' in H_step.
                    contradiction.
                  } 
                  (* Having used H_point to establish that nothing in alpha disagrees with w for p' *)
                  (* First we need to obtain the next symbol following alpha_p' in alpha *)
                  (* It must exist because alpha_p' < alpha *)
                  apply prefix_exists_suffix in H_p'_even_shorter.  
                  destruct H_p'_even_shorter as [alpha_p'_suf H_split_alpha].
                  destruct alpha_p'_suf as [|alpha_p'_next alpha_p'_suf].
                  {
                    rewrite app_nil_r in H_split_alpha.
                    symmetry in H_split_alpha. 
                    contradiction. }
                  assert (H_useful := alpha_next_active).
                  spec H_useful (alpha ++ [Event p q m ↾ H_neq] ++ beta_w') w' p' alpha_p' alpha_p'_next (alpha_p'_suf ++ [Event p q m ↾ H_neq] ++ beta_w').
                  spec H_useful.
                  rewrite H_split_alpha. 
                  simpl. rewrite <- app_assoc.
                  simpl. reflexivity. 
                  spec H_useful H_alpha_p'.
                  (* We have found the next action after alpha_p' in alpha, alpha_p'_next *)
                  (* And established that p' is active in it *)
                  (* So now we have 
                     wproj (split alpha_p') p' < wproj (split alpha) p' <= wproj w p' *) 
                  (* I think I see it *)
                  assert (H_step1 : prefix (wproj (split (alpha_p' ++ [alpha_p'_next])) p') (wproj w p')).
                  { eapply PreOrder_Transitive with (wproj (split alpha) p').
                    enough (prefix (alpha_p' ++ [alpha_p'_next]) alpha).
                    apply prefix_split_prefix_iff in H0.
                    apply (wproj_preserves_prefix _ _ p') in H0. 
                    assumption.
                    rewrite H_split_alpha.
                    apply prefix_app. 
                    apply singleton_prefix_cons. assumption.
                  }  
                  (* Now we find a contradiction from H_step1 *) 
                  rewrite wproj_split_app in H_step1. 
                  unfold is_alpha in H_alpha_p'.
                  replace (wproj (split alpha_p') p') with (wproj w' p') in H_step1 by tauto.
                  apply prefix_exists_suffix in H_pref. destruct H_pref as [w_rest H_split_w].
                  rewrite H_split_w in H_step1.
                  rewrite wproj_app in H_step1. 
                  rewrite wproj_app in H_step1.
                  rewrite <- app_assoc in H_step1.
                  apply prefix_app_inv in H_step1.
                  { 
                    destruct alpha_p'_next as [alpha_p'_next H_neq_alpha_p'_next]. 
                    destruct alpha_p'_next as [p1 p2 m1]. 
                    clean H_useful. clean H_step1.
                    destruct H_useful.
                    { (* In the case that the senders are the same *)
                      (* Then we can show that the run agrees with p0 *)
                      subst p1. subst p0.
                      rewrite wproj_symbol_sender_eq in H_step1; try reflexivity. 
                      rewrite Heqz' in H_step1.
                      rewrite wproj_symbol_sender_eq in H_step1; try reflexivity.
                      clean H_step1. 
                      inversion H_step1. inversion H.
                      subst m1. subst p2. subst.
                      clear x H H3.
                      (* Now everything looks matchy matchy in the conclusion *)
                      clean.
                      replace (wproj w' p') with (wproj (split alpha_p') p').
                      2 : symmetry; tauto.
                      apply prefix_app_r.
                      apply prefix_app.
                      apply prefix_app_r.
                      assert (H_rewrite : wproj_symbol p' (Snd p' q' m' ↾ H_neq_alpha_p'_next) = wproj_symbol p' (Snd p' q' m' ↾ H_neq_z')).  f_equal. sigma_equal.
                      now rewrite H_rewrite.
                    }
                    {
                      (* In the case that the receiver is the same *)
                      (* Then we prove a contradiction to z' being a send event *)
                      subst p0. subst p2. exfalso.
                      rewrite wproj_symbol_receiver_eq in H_step1.
                      reflexivity.
                      rewrite wproj_symbol_sender_neq in H_step1; try easy.
                      unnil H_step1.
                      rewrite Heqz' in H_step1. 
                      rewrite wproj_symbol_sender_eq in H_step1.
                      reflexivity. simpl in H_step1.
                      inversion H_step1. inversion H.
                    }
                  } (* Where on earth am I in the proof *)
                }
                (* In the case that we are not dealing with p' *)
                { 
                  rewrite Heqz'.
                  rewrite wproj_app. 
                  simpl. rewrite wproj_symbol_sender_neq.
                  assumption.
                  unnil. 
                  destruct H_possible_w' as [_ H_compliant_w'].
                  now spec H_compliant_w' p0.
                }
              }
            }
            *** (* Case (3) : alpha ++ [y] < alpha_p' *)
              (* In the case that alpha ++ [y] < alpha_p' *)
              (* Then we can use the outer induction hypothesis *)
              (* And we can prove that there exists a next action in alpha ++ [y] that must not disagree with z' *)
              { spec IHn' (length w'). 
                spec IHn'.
                { simpl. apply prefix_length in H_pref.
                  simpl in H_pref. rewrite app_length in H_pref.
                  simpl in H_pref. lia. }
                spec IHn' alpha_p' (alpha ++ [y] ++ beta_w').
                spec IHn' H_fine. 
                spec IHn' (Snd p' q' m' ↾ H_neq_z').  
                spec IHn'. 
                easy. spec IHn' w'. 
                spec IHn'.
                {
                  eapply clts_trace_prefix_closed with w.
                  assumption. 
                  apply prefix_app_l in H_pref. assumption. }
                rewrite <- Heqy in H_possible_w'.
                rewrite <- Heqy in H_alpha_p'. spec IHn' H_possible_w' H_alpha_p'.
                spec IHn'.
                { eapply clts_trace_prefix_closed with w.
                  assumption. 
                  rewrite <- Heqz'. assumption. }
                spec IHn'. reflexivity.  
                destruct IHn' as [rho_fin' [H_pref_alpha_p' [H_neq_alpha_p'_rho_fin' H_possible_rho_fin']]]. 
                (* Obtaining the witness for beta' *)
                apply prefix_exists_suffix in H_p'_longer.
                destruct H_p'_longer as [alphay_extra H_split_alpha_p'].
                apply prefix_exists_suffix in H_pref_alpha_p'. 
                destruct H_pref_alpha_p' as [beta' H_split_rho_fin'].
                exists (alphay_extra ++ beta').
                split. 
                { 
                  rewrite <- Heqy.
                  rewrite H_split_alpha_p' in H_split_rho_fin'.
                  rewrite <- app_assoc in H_split_rho_fin'.
                  rewrite <- app_assoc in H_split_rho_fin'.
                  unfold possible_run_prefix in H_possible_rho_fin'.
                  rewrite H_split_rho_fin' in H_possible_rho_fin'.
                  tauto. } 
                (* Showing that every role is compliant with our new run *)
                {
                  intros p0.
                  destruct (classic (p' = p0)).
                  {
                    (* In the case that we are dealing with p' *)
                    subst.
                    destruct H_possible_rho_fin' as [_ H_compliant_rho_fin'].
                    spec H_compliant_rho_fin' p0.
                    rewrite <- app_assoc in H_compliant_rho_fin'.
                    rewrite <- app_assoc in H_compliant_rho_fin'.
                    exact H_compliant_rho_fin'. }
                  {
                    (* In the case that we are not dealing with p' *)
                    (* We can appeal to the rho_fin' we obtained from the induction hypothesis directly *)
                    destruct H_possible_rho_fin' as [_ H_compliant_rho_fin'].
                    subst.
                    clean. 
                    rewrite wproj_symbol_sender_neq. assumption.
                    spec H_compliant_rho_fin' p0.
                    clean H_compliant_rho_fin'.
                    rewrite wproj_symbol_sender_neq in H_compliant_rho_fin'.
                    assumption.
                    unnil H_compliant_rho_fin'.
                    rewrite app_nil_r in H_compliant_rho_fin'.
                    repeat rewrite <- app_assoc in H_compliant_rho_fin'.
                    clean. exact H_compliant_rho_fin'. }
                }
              }
  }
  (** Finished proving the inner fact by induction *) 
  (* Now that we have established this inner claim by induction,
     we can use it to finish off our proof obligation for the outer induction *)
  spec H_inner w. 
  spec H_inner. easy.
  destruct H_inner as [beta' [H_run_new_candidate H_compliant_new_candidate]]. 
  exists (alpha ++ [y] ++ beta').
  split.
  now apply prefix_app_r. 
  split.
  intro H_false. 
  symmetry in H_false.
  apply (list_app_identity_means_nil _ alpha ([y] ++ beta')) in H_false.
  inversion H_false.
  split.
  {
    rewrite Heqy.
    unfold possible_run_prefix in H_compliant_new_candidate.
    tauto. }
  (* Case analysis on which role's compliance we are obligated to show *)
  {
    intro p0.
    destruct (classic (p = p0)).
    { (* In case it's p *)
      unfold is_alpha in H_max. 
      rewrite wproj_app.
      subst.
      replace (wproj w p0) with (wproj (split alpha) p0).
      rewrite wproj_split_app.
      apply prefix_app. clean.
      now apply prefix_app_r.
      symmetry. tauto. } 
    { (* In case it's not p *)
      spec H_compliant_new_candidate p0.
      rewrite wproj_app.
      simpl. rewrite wproj_symbol_sender_neq.
      assumption.
      do 2 rewrite app_nil_r.
      simpl in H_compliant_new_candidate.
      rewrite Heqy. assumption.
  } }
Qed.

(* Actually this lemma is all we need, and the case analysis is trivial to manage *) 
Lemma send_preserves_run_prefixes_finite_case_analysis :
  forall (S : @LTS SyncAlphabet State) (T : CLTS) (w : FinAsyncWord) (x : AsyncAlphabet) (rho_fin : FinSyncWord) (alpha : FinSyncWord)
    (H_GCLTS : GCLTS S)
    (H_NMC : @NMC State S)
    (H_SCC : @SCC State S)
    (H_RCC : @RCC State S)
    (H_canonical : canonical_implementation S T)
    (H_trace_w : @is_clts_trace LocalState T w),
    possible_run_prefix S rho_fin w ->
    is_alpha rho_fin alpha w (sender_async x) ->
    is_snd x ->
    @is_clts_trace LocalState T (w ++ [x]) ->
    exists (rho_fin' : FinSyncWord),
      prefix alpha rho_fin' /\
        possible_run_prefix S rho_fin' (w ++ [x]). 
Proof.    
  intros S T w x rho_fin alpha H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_wx H_possible H_alpha H_snd H_trace.  
  destruct (classic (alpha = rho_fin)).
  - (* In the case that alpha = rho_fin *)
    (* We can just extend rho_fin directly *)
    subst. 
    assert (H_useful := send_extension_exists_run_extension S T w x rho_fin H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_wx H_possible).
    spec H_useful. 
    { 
      unfold is_alpha in H_alpha. 
      symmetry. tauto. 
    } 
    spec H_useful H_snd H_trace.
    destruct H_useful as [beta [H_possible_new]].
    exists (rho_fin ++ beta). split.
    apply prefix_app_r. reflexivity.
    split. assumption. assumption.
  - (* In the case that alpha <> rho_fin *)
    (* We can use the existing lemmas *)
    assert (H_useful := send_preserves_run_prefixes_finite S T w x rho_fin alpha H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_wx H_possible H_alpha H H_snd H_trace).
    destruct H_useful as [rho_fin' [H_pref' [_ H_possible']]].
    exists rho_fin'. tauto.
Qed.

Lemma infinite_possible_run_exists_possible_run_prefix_neq_alpha :
  forall  (S : @LTS SyncAlphabet State)
     (T : CLTS)
     (w : FinAsyncWord)
     (H_GCLTS : GCLTS S)
     (H_NMC : @NMC State S)
     (H_SCC : @SCC State S)
     (H_RCC : @RCC State S)
     (H_canonical : canonical_implementation S T)
     (H_trace_w : @is_clts_trace LocalState T w)
     (rho : InfSyncWord)
     (H_infinite_possible_rho : infinite_possible_run S rho w)
     (p q : participant)
     (m : message)
     (H_neq : sender_receiver_neq_async (Snd p q m))
     (H_snd : is_snd (Snd p q m ↾ H_neq))
     (H_trace_wx : @is_clts_trace LocalState T (w ++ [Snd p q m ↾ H_neq])),
  exists (pref : FinSyncWord),
    possible_run_prefix S pref w /\
      forall (alpha : FinSyncWord),
        is_alpha pref alpha w p ->
        alpha <> pref.
Proof. 
  intros.
  assert (H_helper := infinite_possible_run_means_finite_possible_run_prefix S rho w H_infinite_possible_rho). 
  destruct H_helper as [pref [H_possible H_pref_rho]].
  assert (H_alpha_p := finite_unique_splitting pref w p). 
  spec H_alpha_p.
  {
      destruct H_possible as [_ H_compliant].
      now spec H_compliant p.
  }
  destruct H_alpha_p as [alpha H_alpha_p].
  (* We do case analysis to see if we can reuse alpha *)  
  destruct (classic (pref = alpha)). 
  * (* In case they are equal, we can appeal to SCC to find a longer extension *)
    subst alpha. 
    assert (H_hopeful := send_extension_exists_run_extension S T w (Snd p q m ↾ H_neq) pref H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_w H_possible). 
    spec H_hopeful.
    {
      unfold is_alpha in H_alpha_p.
      destruct H_alpha_p as [_ [H_rewrite _]].
    simpl. rewrite H_rewrite. reflexivity. }
    spec H_hopeful. easy.
    spec H_hopeful H_trace_wx.
    destruct H_hopeful as [beta H_new_possible].
    assert (H_new_alpha_p := finite_unique_splitting (pref ++ beta) w p). 
    spec H_new_alpha_p.
    {
      destruct H_new_possible as [_ H_new_compliant].
      spec H_new_compliant p.
      clean H_new_compliant.
      apply prefix_app_l in H_new_compliant.
      clean. assumption. }
    destruct H_new_alpha_p as [new_alpha H_new_alpha_p].
    exists (pref++ beta). split.
    assert (H_copy := H_new_possible). 
    eapply prefix_preserves_possible_run in H_new_possible.
    exact H_new_possible. now apply prefix_app_r.
    intros. 
    assert (alpha = new_alpha).
    { eapply is_alpha_unique.
      exact S. exact H. exact H_new_alpha_p. }
    subst new_alpha. 
    intro H_false. subst.
    unfold is_alpha in H.
    destruct H as [_ [H_false _]].
    unfold is_alpha in H_alpha_p.
    destruct H_alpha_p as [_ [H_rewrite _]].
    unfold possible_run_prefix in H_new_possible.
    destruct H_new_possible as [_ H_new_compliant].
    spec H_new_compliant p.
    rewrite <- H_false in H_new_compliant.
    clean H_new_compliant.
    rewrite wproj_symbol_sender_eq in H_new_compliant.
    reflexivity.
    eapply prefix_app_not. 
    2 : exact H_new_compliant.
    easy.
  * (* In the case that they are not equal, we can just reuse the existing ones *)
    exists pref. split. assumption.
    intros alpha0 H_alpha0.
    assert (alpha = alpha0).
    { eapply is_alpha_unique.
      exact S. exact H_alpha_p. exact H_alpha0.  }
    subst alpha0. easy.
Qed. 
  
(* New proof idea, Maine *)
(* Trying to prove the send lemma by contraposition *)
(* Update: this does not work *) 

(* This is the proof that can hopefully be simplified *)
(* Update: proof cannot be simplified *)


Lemma I_set_non_empty_inductive_snd :
  forall (S : @LTS SyncAlphabet State) (T : CLTS) (w : FinAsyncWord),
    GCLTS S ->
    @NMC State S -> 
    @SCC State S ->
    @RCC State S ->
    canonical_implementation S T -> 
    @is_clts_trace LocalState T w ->
    I_set_non_empty S w -> 
    forall (x : AsyncAlphabet),
      is_snd x ->
      @is_clts_trace LocalState T (w ++ [x]) ->
      I_set_non_empty S (w ++ [x]). 
Proof.
  intros S T w H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_w IHw x H_snd H_trace_wx.
  destruct x as [x H_neq]. 
  destruct x as [p q m | p q m].
  2 : inversion H_snd.
  (** Coq-obligatory case analysis on whether w's run is finite or infinite *)
  (** Except that now we have encapsulated the reasoning to a single run prefix, thereby delaying the acknowledgement of this difference until the last possible moment **)
  (** In each case we take the maximal run, finite or infinite from w, and extend it using the deadlock freedom of S, to get either a finite or infinite run for wx **)
  assert (H_df : deadlock_free S). 
  { unfold GCLTS in H_GCLTS; tauto. }
  destruct IHw as [H_fin | H_inf]. 
  - (** In the finite case **)
    destruct H_fin as [rho H_finite_possible_rho].
    (* Here we don't want to lose information about rho being maximal,
       because we need this to show that alpha <> rho *) 
    assert (H_possible_rho := @finite_possible_run_is_possible_run_prefix State).
    spec H_possible_rho S w rho H_finite_possible_rho. 
    assert (H_alpha_p := finite_unique_splitting rho w p). 
    spec H_alpha_p.
    {
      destruct H_finite_possible_rho as [_ H_compliant].
      now spec H_compliant p.
    }
    destruct H_alpha_p as [alpha H_alpha_p].
    assert (H_helper := send_preserves_run_prefixes_finite S T w (Snd p q m ↾ H_neq) rho alpha H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_w H_possible_rho H_alpha_p).
    spec H_helper.
    {
      intro H_false. 
      subst alpha. (* It cannot be the case that p can take a step, but the run is maximal *)
      assert (H_contra := send_extension_means_exhausted_run_non_final).
      spec H_contra S T w (Snd p q m ↾ H_neq) rho.
      destruct H_finite_possible_rho as [[s_rho [H_reach_rho H_max_rho]] _]. 
      spec H_contra s_rho H_GCLTS H_SCC H_canonical H_trace_w H_possible_rho H_reach_rho. 
      spec H_contra.
      simpl. unfold is_alpha in H_alpha_p.
      symmetry. tauto. spec H_contra. easy. spec H_contra H_trace_wx.
      apply H_contra. assumption.
    }
    spec H_helper. easy. spec H_helper H_trace_wx.
    destruct H_helper as [rho_pref [H_pref_alpha [H_alpha_neq H_possible_rho_pref]]]. 
    (* From our core correctness lemma for send events, we obtain a run prefix *) 
    (* We now appeal to deadlock freedom of S to extend this run prefix *) 
    destruct H_possible_rho_pref as [[s_rho_pref H_reach_rho_pref] H_compliant_rho_pref]. 
    spec H_df s_rho_pref rho_pref H_reach_rho_pref.
    destruct H_df as [H_fin | H_inf].
    * destruct H_fin as [run_fin [H_max_run_fin H_pref_rho_pref]].
      left. exists run_fin. split. assumption.
      intro p0. spec H_compliant_rho_pref p0.
      apply PreOrder_Transitive with (wproj (split rho_pref) p0).
      assumption.
      apply wproj_preserves_prefix.
      apply prefix_split_prefix_iff.
      assumption.
    * destruct H_inf as [rho_inf [H_run_inf [i H_pref]]].
      right. exists rho_inf. split. assumption.
      intro p0. spec H_compliant_rho_pref p0.
      exists i.
      apply PreOrder_Transitive with (wproj (split rho_pref) p0).
      assumption.
      apply wproj_preserves_prefix.
      apply prefix_split_prefix_iff.
      subst. reflexivity.
  - (** In the infinite case **)
    destruct H_inf as [rho H_infinite_possible_rho].
    (* In the infinite case we simply apply this conversion lemma to extract a finite possible run prefix *)
    (* However, since we require a possible run prefix whose alpha is strictly not equal to itself,
       we need to invoke a special lemma that gives the desired stronger property *) 
    assert (H_helper := infinite_possible_run_exists_possible_run_prefix_neq_alpha S T w H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_w rho H_infinite_possible_rho p q m H_neq).
    spec H_helper. easy.
    spec H_helper H_trace_wx.
    destruct H_helper as [pref [H_possible H_pref_rho]].
    assert (H_alpha_p := finite_unique_splitting pref w p). 
    spec H_alpha_p.
    {
      destruct H_possible as [_ H_compliant].
      now spec H_compliant p.
    }
    destruct H_alpha_p as [alpha H_alpha_p]. 
    spec H_pref_rho alpha H_alpha_p.
    (* Now we magically have this inequality to instantiate our key lemma with *) 
    assert (H_helper := send_preserves_run_prefixes_finite S T w (Snd p q m ↾ H_neq) pref alpha H_GCLTS H_NMC H_SCC H_RCC H_canonical H_trace_w H_possible H_alpha_p H_pref_rho).
    spec H_helper.
    easy. spec H_helper H_trace_wx.
    destruct H_helper as [rho_pref [H_pref_alpha [H_alpha_neq H_possible_rho_pref]]]. 
    (* From our core correctness lemma for send events, we obtain a run prefix *) 
    (* We now appeal to deadlock freedom of S to extend this run prefix *) 
    destruct H_possible_rho_pref as [[s_rho_pref H_reach_rho_pref] H_compliant_rho_pref]. 
    spec H_df s_rho_pref rho_pref H_reach_rho_pref.
    destruct H_df as [H_fin | H_inf].
    * destruct H_fin as [run_fin [H_max_run_fin H_pref_rho_pref]].
      left. exists run_fin. split. assumption.
      intro p0. spec H_compliant_rho_pref p0.
      apply PreOrder_Transitive with (wproj (split rho_pref) p0).
      assumption.
      apply wproj_preserves_prefix.
      apply prefix_split_prefix_iff.
      assumption.
    * destruct H_inf as [rho_inf [H_run_inf [i H_pref]]].
      right. exists rho_inf. split. assumption.
      intro p0. spec H_compliant_rho_pref p0.
      exists i.
      apply PreOrder_Transitive with (wproj (split rho_pref) p0).
      assumption.
      apply wproj_preserves_prefix.
      apply prefix_split_prefix_iff.
      subst. reflexivity.
Qed. 

End SCC. 
