Require Import Arith List Classical Nat Logic Lia FunctionalExtensionality ExtensionalityFacts Streams stdpp.list.
Import ListNotations. 
From Coq 
Require Import ssreflect. 
    
From CC    
  Require Import lib structures lts clts protocol channel_compliant channel_complete.

(** Formalization of compliant runs, maximal run prefixes, and facts about runs **) 
(* Definitions in this file are specific to the precisness theorem *) 
Section Run.

Context {State : Type} {LocalState : Type}.
  
(** Reminder to self that is_run as a definition does not type check
   so the disjunction with finite or infinite runs must be included explicitly **)
Definition finite_possible_run (S : @LTS SyncAlphabet State) (run : FinSyncWord) (w : FinAsyncWord) :=
  @is_finite_maximal_run SyncAlphabet State S run /\ 
  forall (p : participant),
    prefix (wproj w p) (wproj (split run) p). 

Definition infinite_possible_run (S : @LTS SyncAlphabet State) (run : InfSyncWord) (w : FinAsyncWord) :=
  @is_infinite_run SyncAlphabet State S run /\ 
    forall (p : participant),
      exists (i : nat), 
    prefix (wproj w p) (wproj (split (stream_to_list run i)) p).

(* Turns out that the following weaker notion of a possible run prefix is extremely handy in mediating between the two disparate definitions *) 
Definition possible_run_prefix (S : @LTS SyncAlphabet State) (run : FinSyncWord) (w : FinAsyncWord) :=
  (* We just weaken the requirement that the run is maximal to simply saying that it is a run, i.e. it is a synchronous trace in the global LTS *) 
  @is_trace SyncAlphabet State S run /\ 
  forall (p : participant),
    prefix (wproj w p) (wproj (split run) p).

Definition I_set_non_empty (S : @LTS SyncAlphabet State) (w : FinAsyncWord) :=
  (exists (run : FinSyncWord),
      finite_possible_run S run w)
  \/
    (exists (run : InfSyncWord),
        infinite_possible_run S run w).

(* Naming this maximal run alpha so as not to overload maximality in the LTS sense *)
Definition is_alpha (run alpha : FinSyncWord) (w : FinAsyncWord) (p : participant) :=
  prefix alpha run /\ 
  wproj w p = wproj (split alpha) p /\
    (forall (u : FinSyncWord),
        prefix u run ->
        wproj w p = wproj (split u) p ->
        prefix u alpha).

(* We define an analogous notion of maximal prefix matching some word for asynchronous words, not for the splitting of synchronous runs *) 
Definition is_alpha_async (w_fin alpha : FinAsyncWord) (w : FinAsyncWord) (p : participant) :=
  prefix alpha w_fin /\ 
  wproj w p = wproj alpha p /\
    (forall (u : FinAsyncWord),
        prefix u w_fin ->
        wproj w p = wproj u p ->
        prefix u alpha).

Lemma is_alpha_unique :
  forall (S : @LTS SyncAlphabet State) (rho alpha1 alpha2 : FinSyncWord) (w : FinAsyncWord) (p : participant),
    is_alpha rho alpha1 w p ->
    is_alpha rho alpha2 w p ->
    alpha1 = alpha2. 
Proof.
  intros S rho alpha1 alpha2 w p H_alpha1 H_alpha2.
  destruct H_alpha1 as [H_pref1 [H_eq1 H_max1]].
  destruct H_alpha2 as [H_pref2 [H_eq2 H_max2]]. 
  assert (H_useful := prefix_weak_total alpha1 alpha2 rho).
  spec H_useful H_pref1 H_pref2.
  destruct (classic (alpha1 = alpha2)).
  - assumption.
  - exfalso.
    destruct H_useful.
    * (* Contradiction: if alpha1 < alpha2, then alpha1 cannot be alpha *)
      spec H_max1 alpha2 H_pref2 H_eq2.
      apply symmetric_prefix_means_eq in H0.
      symmetry in H0. contradiction.
      assumption.
    * spec H_max2 alpha1 H_pref1 H_eq1.
      apply symmetric_prefix_means_eq in H0.
      contradiction.
      assumption.
Qed.

Lemma prefix_inf_infinite_possible_run_is_trace :
  forall (S : @LTS SyncAlphabet State) (run : InfSyncWord) (pref : FinSyncWord),
    @is_infinite_run SyncAlphabet State S run -> 
    prefix_inf pref run ->
    @is_trace SyncAlphabet State S pref.
Proof. 
  intros S run pref H_run [i H_pref].
  generalize dependent pref. 
  induction i as [|i IHi]; intros.
  - exists (s0 S).
    rewrite stream_to_list_zero in H_pref.
    subst.
    now apply lts.Reachable_refl.
  - destruct (destruct_list_last _ pref).
    { (* Discharging the pref empty case *)
      exists (s0 S).
      rewrite H. 
      now apply lts.Reachable_refl.
    } 
    destruct H as [pref_last [pref_pref H_split_pref]].
    assert (H_eq_pref : pref_pref = stream_to_list run i).
    { rewrite H_split_pref in H_pref.
      symmetry.
      eapply stream_to_list_app. exact H_pref. }
    spec IHi pref_pref.
    spec IHi.
    easy. 
    rewrite H_split_pref. Print is_infinite_run. 
    spec H_run i.
    destruct H_run as [s_i [s_i' [H_reach_i H_transition]]].
    exists s_i'. eapply lts.Reachable_step with s_i.
    rewrite H_eq_pref. assumption.
    assert (H_pref_last : Str_nth i run = pref_last).
    { rewrite stream_to_list_S_Str_nth_app in H_pref.
      rewrite H_split_pref in H_pref.
      apply app_inj_tail in H_pref.
      tauto. }
    now rewrite <- H_pref_last.
Qed.

Lemma prefix_preserves_finite_possible_run :
  forall (S : @LTS SyncAlphabet State) (rho : FinSyncWord) (w : FinAsyncWord), 
    finite_possible_run S rho w ->
    forall (w' : FinAsyncWord),
      prefix w' w ->
      finite_possible_run S rho w'. 
Proof.
  intros S rho w [H_max H_role] w' H_pref.
  split.
  assumption.
  intro p.
  spec H_role p.
  eapply PreOrder_Transitive with (wproj w p).
  now apply wproj_preserves_prefix.
  assumption.
Qed.

Lemma prefix_preserves_infinite_possible_run :
  forall (S : @LTS SyncAlphabet State) (rho : InfSyncWord) (w : FinAsyncWord), 
    infinite_possible_run S rho w ->
    forall (w' : FinAsyncWord),
      prefix w' w ->
      infinite_possible_run S rho w'. 
Proof.
  intros S rho w [H_max H_role] w' H_pref.
  split.
  assumption.
  intro p.
  spec H_role p.
  destruct H_role as [i H_role].
  exists i. eapply PreOrder_Transitive with (wproj w p).
  now apply wproj_preserves_prefix.
  assumption.
Qed.

Lemma prefix_preserves_possible_run :
  forall (S : @LTS SyncAlphabet State) (rho : FinSyncWord) (w : FinAsyncWord), 
    possible_run_prefix S rho w ->
    forall (w' : FinAsyncWord),
      prefix w' w ->
      possible_run_prefix S rho w'. 
Proof.
  intros S rho w [H_trace H_role] w' H_pref.
  split.
  assumption.
  intro p.
  spec H_role p.
  eapply PreOrder_Transitive with (wproj w p).
  now apply wproj_preserves_prefix.
  assumption.
Qed. 

(* Obtaining a finite possible run prefix from an infinite possible run: for the consequent I want is_finite_maximal_run but without the maximal part *) 
Lemma infinite_possible_run_means_finite_possible_run_prefix :
  forall (S : @LTS SyncAlphabet State) (run_inf : InfSyncWord) (w : FinAsyncWord), 
    infinite_possible_run S run_inf w ->
    exists (run_fin : FinSyncWord),
      possible_run_prefix S run_fin w /\
        prefix_inf run_fin run_inf. 
Proof.
  intros S run_inf w H_inf.
  induction w as [|a w IHw] using rev_ind; intros. 
  - exists [].
    split.
    split. exists (s0 S). now apply lts.Reachable_refl.
    easy.
    exists 0. apply stream_to_list_zero.
  - spec IHw.
    eapply prefix_preserves_infinite_possible_run. 
    exact H_inf.
    now apply prefix_app_r.
    destruct IHw as [run_w [H_possible_run_w H_pref_run_w]].
    destruct H_possible_run_w as [H_run_w H_compliant_run_w]. 
    destruct a as [a H_neq].
    destruct a as [p q m | p q m].
    (* Now we do case analysis on whether the active role in a is covered already *)
    * (* In the case that a = Snd p q m, the active role is p *)
      destruct H_pref_run_w as [i H_pref_run_w]. 
      destruct H_inf as [H_run_inf H_role_run_inf].
      spec H_role_run_inf p.
      destruct H_role_run_inf as [i_p H_pref_p].
      remember (max i_p i) as max_i.
      exists (stream_to_list run_inf max_i).
      split.
      split.
      eapply prefix_inf_infinite_possible_run_is_trace.
      exact H_run_inf.
      exists max_i. reflexivity.
      intro p0.
      destruct (classic (p = p0)).
      { subst.
        eapply PreOrder_Transitive with (wproj
                (split
                   (stream_to_list
                      run_inf
                      (i_p)))
                p0).
        assumption.
        apply wproj_preserves_prefix.
        apply prefix_split_prefix_iff.
        apply stream_to_list_prefix.
        lia. }
      { spec H_compliant_run_w p0.
        clean.
        rewrite wproj_symbol_sender_neq. easy.
        rewrite app_nil_r.
        rewrite <- H_pref_run_w in H_compliant_run_w. 
        eapply PreOrder_Transitive with (wproj
                       (split
                       (stream_to_list
                       run_inf i))
                       p0).
        assumption.
        apply wproj_preserves_prefix.
        apply prefix_split_prefix_iff.
        apply stream_to_list_prefix.
        lia. }
      exists max_i. reflexivity.
    * (* In the case that a = Rcv p q m, the active role is q *)
      destruct H_pref_run_w as [i H_pref_run_w]. 
      destruct H_inf as [H_run_inf H_role_run_inf].
      spec H_role_run_inf q. 
      destruct H_role_run_inf as [i_q H_pref_q]. 
      remember (max i_q i) as max_i.
      exists (stream_to_list run_inf max_i).
      split.
      split.
      eapply prefix_inf_infinite_possible_run_is_trace.
      exact H_run_inf.
      exists max_i. reflexivity.
      intro p0.
      destruct (classic (q = p0)).
      { subst.
        eapply PreOrder_Transitive with (wproj
                (split
                   (stream_to_list
                      run_inf
                      (i_q)))
                p0).
        assumption.
        apply wproj_preserves_prefix.
        apply prefix_split_prefix_iff.
        apply stream_to_list_prefix.
        lia. }
      { spec H_compliant_run_w p0.
        clean.
        rewrite wproj_symbol_receiver_neq. easy.
        rewrite app_nil_r.
        rewrite <- H_pref_run_w in H_compliant_run_w. 
        eapply PreOrder_Transitive with (wproj
                       (split
                       (stream_to_list
                       run_inf i))
                       p0).
        assumption.
        apply wproj_preserves_prefix.
        apply prefix_split_prefix_iff.
        apply stream_to_list_prefix.
        lia. }
      exists max_i. reflexivity.
Qed. 

(* Finding the unique splitting, in lemma form *) 
Lemma finite_unique_splitting :
  forall (run : FinSyncWord) (w : FinAsyncWord) (p : participant),
    prefix (wproj w p) (wproj (split run) p) -> 
    exists (alpha : FinSyncWord),
      is_alpha run alpha w p. 
Proof.         
  intros run w p. 
  remember (length run) as n.
  generalize dependent run.
  induction n as [|n IHn]; intros run Heqn H_compliant.
  - (* Base case: both alpha and beta are [] *)
    symmetry in Heqn. apply nil_length_inv in Heqn.
    rewrite Heqn in H_compliant.
    simpl in H_compliant.
    apply prefix_nil_inv in H_compliant.
    exists [].
    repeat split; try easy.
    apply prefix_nil.
    intros u H_pref H_eq.
    rewrite Heqn in H_pref.
    apply prefix_nil_inv in H_pref.
    now rewrite H_pref. 
  - (* Induction step *)
    (* Induction hypothesis holds for all runs of length n *)
    destruct (destruct_list_last _ run).  
    * rewrite H in H_compliant.
      simpl in H_compliant.
      apply prefix_nil_inv in H_compliant.
      exists [].
      repeat split; try easy.
      apply prefix_nil.
      intros u H_pref H_eq.
      now rewrite H in H_pref.
    * (* We have a run of length n+1 that w is compliant with *)
      (* First we do outermost case analysis on whether p is involved in tl *) 
      destruct H as [tl [run' H_eq]].
      spec IHn run'. spec IHn.
      rewrite H_eq in Heqn.
      rewrite app_length in Heqn.
      simpl in Heqn. lia.
      assert (H_case := split_case).
      destruct tl as [tl H_neq]. 
      destruct tl as [p' q' m].
      spec H_case p' q' p m H_neq.
      (* Outermost case analysis on whether p is involved in tl *) 
      destruct H_case as [H_snd | [H_rcv | H_nil]].
      ** (* Either w is already compliant with the first n symbols of run, or it is only compliant with all of run' *)
        (* In the first case, the same maximal prefix and suffix can be chosen *) 
        (* In the second case, the maximal prefix must contain tl, otherwise we fall back into the first case *)
        destruct (classic (wproj w p `prefix_of` wproj (split run') p)). 
         *** (* In the case that w is already compliant with the first n symbols of run *)
           (* Then alpha is still the maximal prefix of run' for w *) 
           (* We can use the induction hypothesis *) 
           spec IHn H.
           destruct IHn as [alpha [H_glue [H_alpha H_max]]]. 
           exists alpha.
           repeat split; try assumption.
           rewrite H_eq.
           now apply prefix_app_r. 
           (* H_max says that alpha is the maximal prefix of run' that is per-role equivalent with w *) 
           (* Need to show that alpha is still the maximal prefix of run' ++ [tl] that is per-role equivalent with w *)
           intros u H_pref H_split. 
           apply H_max.
           2 : assumption.
           (* What remains is to show that u is a prefix of run' *)
           (* We know that u is a prefix of run' ++ [tl] *)
           rewrite H_eq in H_pref.
           apply prefix_app_tail_or in H_pref.
           destruct H_pref as [H_pref | H_eq_run].
           assumption.
           (* Need to prove a contradiction in the case that u = run *)
           rewrite H_split in H. 
           rewrite H_eq_run in H.
           rewrite wproj_split_app in H.
           exfalso.
           apply (prefix_app_not (wproj (split run') p) (wproj (split [Event p' q' m ↾ H_neq]) p)).
           now rewrite H_snd.
           assumption.
         *** (* In the case that wproj w p is only compliant with run' ++ [tl] *)
           (* Then it must be the case that actually wproj (split run') p is a prefix of wproj w p *)
           (* And the maximal prefix is the entire run *)
           assert (H_fact : wproj w p = wproj (split run) p). 
           { (* Except that we are dealing with wproj (split _) _) instead of directly with lists *)
             rewrite H_eq in H_compliant.
             rewrite wproj_split_app in H_compliant. 
             rewrite H_snd in H_compliant.
             apply prefix_app_tail_or in H_compliant. 
             destruct H_compliant.
             contradiction.
             rewrite H0.
             rewrite H_eq.
             rewrite wproj_split_app.
             rewrite H_snd. reflexivity.
           }
           exists run. split; try easy.
      ** destruct (classic (wproj w p `prefix_of` wproj (split run') p)). 
         *** (* Identical reasoning to the send case above *)
           spec IHn H.
           destruct IHn as [alpha [H_glue [H_alpha H_max]]]. 
           exists alpha.
           repeat split; try assumption.
           rewrite H_eq.
           now apply prefix_app_r. 
           intros u H_pref H_split. 
           apply H_max.
           2 : assumption.  
           rewrite H_eq in H_pref.
           apply prefix_app_tail_or in H_pref.
           destruct H_pref as [H_pref | H_eq_run].
           assumption.
           rewrite H_split in H. 
           rewrite H_eq_run in H.
           rewrite wproj_split_app in H.
           exfalso.
           apply (prefix_app_not (wproj (split run') p) (wproj (split [Event p' q' m ↾ H_neq]) p)).
           now rewrite H_rcv.
           assumption.
         *** (* In the case that wproj w p is only compliant with run' ++ [tl] *)
           (* Then it must be the case that actually wproj (split run') p is a prefix of wproj w p *)
           (* And the maximal prefix is the entire run *)
           assert (H_fact : wproj w p = wproj (split run) p). 
           { (* Except that we are dealing with wproj (split _) _) instead of directly with lists *)
             rewrite H_eq in H_compliant.
             rewrite wproj_split_app in H_compliant. 
             rewrite H_rcv in H_compliant.
             apply prefix_app_tail_or in H_compliant. 
             destruct H_compliant.
             contradiction.
             rewrite H0.
             rewrite H_eq.
             rewrite wproj_split_app.
             rewrite H_rcv. reflexivity.
           }
           exists run. split; try easy.
      ** (* In the case that tl is irrelevant to p *)
        (* It must be the case that w is already compliant with the first n symbols *)
        (* But the question is whether alpha = run', in which case the new alpha must include the extra symbol *) 
        (* So we do a different case analysis here *)
        spec IHn.
        rewrite H_eq in H_compliant.
        rewrite wproj_split_app in H_compliant.
        rewrite H_nil in H_compliant.
        rewrite app_nil_r in H_compliant.
        assumption.
        destruct IHn as [alpha [H_glue [H_alpha H_max]]]. 
        destruct (classic (alpha = run')).
        *** (* In the case that alpha = run', then the new alpha needs to be run' ++ [tl] *)
          exists run.
          split.
          reflexivity. 
          split.
          rewrite H_eq.
          rewrite wproj_split_app.
          rewrite H_nil.
          rewrite <- H.
          now rewrite app_nil_r.
          intros u H_pref H_split.
          assumption.
        *** (* In the case that alpha <> run', then we can reuse alpha from before *)
          exists alpha.
          split.
          rewrite H_eq. 
          now apply prefix_app_r.
          split. assumption.
          intros u H_pref H_split.
          apply H_max. 
          2 : assumption.
          rewrite H_eq in H_pref.
          apply prefix_app_tail_or in H_pref.
          destruct H_pref.
          assumption.
          spec H_max run'. spec H_max.
          reflexivity.
          spec H_max. rewrite H0 in H_split.
          rewrite wproj_split_app in H_split.
          rewrite H_nil in H_split.
          rewrite app_nil_r in H_split.
          assumption. rewrite H0 in H_split.
          rewrite H0.
          assert (H_useful := symmetric_prefix_means_eq run' alpha H_max H_glue).
          symmetry in H_useful. contradiction. 
Qed.

(* Finding the unique splitting of a word with respect to another word, in lemma form *) 
Lemma finite_unique_splitting_word :
  forall (w_fin : FinAsyncWord) (w : FinAsyncWord) (p : participant),
    prefix (wproj w p) (wproj w_fin p) -> 
    exists (alpha : FinAsyncWord),
      is_alpha_async w_fin alpha w p. 
Proof.         
  intros run w p.  
  remember (length run) as n.
  generalize dependent run.
  induction n as [|n IHn]; intros run Heqn H_compliant.
  - (* Base case: both alpha and beta are [] *)
    symmetry in Heqn. apply nil_length_inv in Heqn.
    rewrite Heqn in H_compliant.
    simpl in H_compliant.
    apply prefix_nil_inv in H_compliant.
    exists [].
    repeat split; try easy.
    apply prefix_nil.
    intros u H_pref H_eq.
    rewrite Heqn in H_pref.
    apply prefix_nil_inv in H_pref.
    now rewrite H_pref. 
  - (* Induction step *)
    (* Induction hypothesis holds for all runs of length n *)
    destruct (destruct_list_last _ run).  
    * rewrite H in H_compliant.
      simpl in H_compliant.
      apply prefix_nil_inv in H_compliant.
      exists [].
      repeat split; try easy.
      apply prefix_nil.
      intros u H_pref H_eq.
      now rewrite H in H_pref.
    * (* We have a run of length n+1 that w is compliant with *)
      (* First we do outermost case analysis on whether p is involved in tl *) 
      destruct H as [tl [run' H_eq]].
      spec IHn run'. spec IHn.
      rewrite H_eq in Heqn. 
      rewrite app_length in Heqn.
      simpl in Heqn. lia.
      destruct tl as [tl H_neq]. 
      destruct tl as [p' q' m | p' q' m]. 
      ** (* Second-to-outermost case analysis on whether p is involved in tl *) 
        destruct (classic (p' = p)) as [H_active | H_inactive]. 
        *** subst.
            (* Either w is already compliant with the first n symbols of run, or it is only compliant with all of run' *)
            (* In the first case, the same maximal prefix and suffix can be chosen *) 
            (* In the second case, the maximal prefix must contain tl, otherwise we fall back into the first case *)
            destruct (classic (wproj w p `prefix_of` wproj run' p)) as [H | H]. 
            **** (* In the case that w is already compliant with the first n symbols of run *)
              (* Then alpha is still the maximal prefix of run' for w *) 
              (* We can use the induction hypothesis *) 
              spec IHn H.
              destruct IHn as [alpha [H_glue [H_alpha H_max]]]. 
              exists alpha.
              repeat split; try assumption.
              now apply prefix_app_r. 
              (* H_max says that alpha is the maximal prefix of run' that is per-role equivalent with w *) 
              (* Need to show that alpha is still the maximal prefix of run' ++ [tl] that is per-role equivalent with w *)
              intros u H_pref H_split. 
              apply H_max.
              2 : assumption.
              (* What remains is to show that u is a prefix of run' *)
              (* We know that u is a prefix of run' ++ [tl] *)
              apply prefix_app_tail_or in H_pref.
              destruct H_pref as [H_pref | H_eq_run].
              assumption.
              (* Need to prove a contradiction in the case that u = run *)
              rewrite H_split in H. 
              rewrite H_eq_run in H.
              rewrite wproj_app in H.
              exfalso.
              apply (prefix_app_not (wproj run' p) (wproj [Snd p q' m ↾ H_neq] p)). 
              rewrite wproj_sender_eq.
              reflexivity. 
              easy. 
              assumption.
            **** (* In the case that wproj w p is only compliant with run' ++ [tl] *)
              (* Then it must be the case that actually wproj (split run') p is a prefix of wproj w p *)
              (* And the maximal prefix is the entire run *)
              assert (H_fact : wproj w p = wproj (run' ++ [Snd p q' m ↾ H_neq]) p). 
              { (* Except that we are dealing with wproj (split _) _) instead of directly with lists *)
                rewrite wproj_app in H_compliant. 
                rewrite wproj_sender_eq in H_compliant.
                reflexivity.
                apply prefix_app_tail_or in H_compliant. 
                destruct H_compliant.
                contradiction.
                rewrite H0.
                rewrite wproj_app.
                rewrite wproj_sender_eq. reflexivity.
                reflexivity. }
              exists (run' ++ [Snd p q' m ↾ H_neq]). split; try easy.
        *** (* In the case that tl is irrelevant to p *)
          (* It must be the case that w is already compliant with the first n symbols *)
          (* But the question is whether alpha = run', in which case the new alpha must include the extra symbol *) 
          (* So we do a different case analysis here *)
          spec IHn.
          rewrite H_eq in H_compliant.
          rewrite wproj_app in H_compliant.
          rewrite wproj_sender_neq in H_compliant.
          assumption. rewrite app_nil_r in H_compliant.
          assumption.
          destruct IHn as [alpha [H_glue [H_alpha H_max]]]. 
          destruct (classic (alpha = run')).
          **** (* In the case that alpha = run', then the new alpha needs to be run' ++ [tl] *)
            exists run.
            split.
            reflexivity. 
            split.
            rewrite H_eq.
            rewrite wproj_app.
            rewrite wproj_sender_neq.
            assumption. rewrite <- H.
            now rewrite app_nil_r.
            intros u H_pref H_split.
            assumption.
          **** (* In the case that alpha <> run', then we can reuse alpha from before *)
            exists alpha.
            split.
            rewrite H_eq. 
            now apply prefix_app_r.
            split. assumption.
            intros u H_pref H_split.
            apply H_max. 
            2 : assumption.
            rewrite H_eq in H_pref.
            apply prefix_app_tail_or in H_pref.
            destruct H_pref.
            assumption.
            spec H_max run'. spec H_max.
            reflexivity.
            spec H_max. rewrite H0 in H_split.
            rewrite wproj_app in H_split.
            rewrite wproj_sender_neq in H_split.
            assumption. rewrite app_nil_r in H_split.
            assumption. rewrite H0 in H_split.
            rewrite H0.
            assert (H_useful := symmetric_prefix_means_eq run' alpha H_max H_glue).
            symmetry in H_useful. contradiction.
      ** (* Basically we repeat the same reasoning but swapping q' for p' *)
        (* Second-to-outermost case analysis on whether p is involved in tl *) 
        destruct (classic (q' = p)) as [H_active | H_inactive]. 
        *** subst.
            (* Either w is already compliant with the first n symbols of run, or it is only compliant with all of run' *)
            (* In the first case, the same maximal prefix and suffix can be chosen *) 
            (* In the second case, the maximal prefix must contain tl, otherwise we fall back into the first case *)
            destruct (classic (wproj w p `prefix_of` wproj run' p)) as [H | H]. 
            **** (* In the case that w is already compliant with the first n symbols of run *)
              (* Then alpha is still the maximal prefix of run' for w *) 
              (* We can use the induction hypothesis *) 
              spec IHn H.
              destruct IHn as [alpha [H_glue [H_alpha H_max]]]. 
              exists alpha.
              repeat split; try assumption.
              now apply prefix_app_r. 
              (* H_max says that alpha is the maximal prefix of run' that is per-role equivalent with w *) 
              (* Need to show that alpha is still the maximal prefix of run' ++ [tl] that is per-role equivalent with w *)
              intros u H_pref H_split. 
              apply H_max.
              2 : assumption.
              (* What remains is to show that u is a prefix of run' *)
              (* We know that u is a prefix of run' ++ [tl] *)
              apply prefix_app_tail_or in H_pref.
              destruct H_pref as [H_pref | H_eq_run].
              assumption.
              (* Need to prove a contradiction in the case that u = run *)
              rewrite H_split in H. 
              rewrite H_eq_run in H.
              rewrite wproj_app in H.
              exfalso.
              apply (prefix_app_not (wproj run' p) (wproj [Rcv p' p m ↾ H_neq] p)). 
              rewrite wproj_receiver_eq.
              reflexivity. 
              easy. 
              assumption.
            **** (* In the case that wproj w p is only compliant with run' ++ [tl] *)
              (* Then it must be the case that actually wproj (split run') p is a prefix of wproj w p *)
              (* And the maximal prefix is the entire run *)
              assert (H_fact : wproj w p = wproj (run' ++ [Rcv p' p m ↾ H_neq]) p). 
              { (* Except that we are dealing with wproj (split _) _) instead of directly with lists *)
                rewrite wproj_app in H_compliant. 
                rewrite wproj_receiver_eq in H_compliant.
                reflexivity.
                apply prefix_app_tail_or in H_compliant. 
                destruct H_compliant.
                contradiction.
                rewrite H0.
                rewrite wproj_app.
                rewrite wproj_receiver_eq. reflexivity.
                reflexivity. }
              exists (run' ++ [Rcv p' p m ↾ H_neq]). split; try easy.
        *** (* In the case that tl is irrelevant to p *)
          (* It must be the case that w is already compliant with the first n symbols *)
          (* But the question is whether alpha = run', in which case the new alpha must include the extra symbol *) 
          (* So we do a different case analysis here *)
          spec IHn.
          rewrite H_eq in H_compliant.
          rewrite wproj_app in H_compliant.
          rewrite wproj_receiver_neq in H_compliant.
          assumption. rewrite app_nil_r in H_compliant.
          assumption.
          destruct IHn as [alpha [H_glue [H_alpha H_max]]]. 
          destruct (classic (alpha = run')).
          **** (* In the case that alpha = run', then the new alpha needs to be run' ++ [tl] *)
            exists run.
            split.
            reflexivity. 
            split.
            rewrite H_eq.
            rewrite wproj_app.
            rewrite wproj_receiver_neq.
            assumption. rewrite <- H.
            now rewrite app_nil_r.
            intros u H_pref H_split.
            assumption.
          **** (* In the case that alpha <> run', then we can reuse alpha from before *)
            exists alpha.
            split.
            rewrite H_eq. 
            now apply prefix_app_r.
            split. assumption.
            intros u H_pref H_split.
            apply H_max. 
            2 : assumption.
            rewrite H_eq in H_pref.
            apply prefix_app_tail_or in H_pref.
            destruct H_pref.
            assumption.
            spec H_max run'. spec H_max.
            reflexivity.
            spec H_max. rewrite H0 in H_split.
            rewrite wproj_app in H_split.
            rewrite wproj_receiver_neq in H_split.
            assumption. rewrite app_nil_r in H_split.
            assumption. rewrite H0 in H_split.
            rewrite H0.
            assert (H_useful := symmetric_prefix_means_eq run' alpha H_max H_glue).
            symmetry in H_useful. contradiction.
Qed. 
(* Relatively easy adaptation from the previous lemma *) 

Lemma alpha_next_active :
  forall (run : FinSyncWord) (w : FinAsyncWord) (p : participant) (alpha : FinSyncWord) (y : SyncAlphabet) (beta : FinSyncWord),
    run = alpha ++ [y] ++ beta ->
    is_alpha run alpha w p ->
    sender_sync y = p \/ receiver_sync y = p. 
Proof. 
  intros run w p alpha y beta H_split H_alpha.
  destruct y as [y H_neq].
  destruct y as [p' q' m'].
  unfold sender_sync, receiver_sync; simpl.
  destruct (classic (p' = p \/ q' = p)).
  - assumption.
  - (* We find a contradiction to the maximality of alpha *)
    apply not_or_and in H.
    destruct H_alpha as [H_pref [H_eq H_max]].
    spec H_max (alpha ++ [Event p' q' m' ↾ H_neq]).
    spec H_max. rewrite H_split.
    apply prefix_app.
    now apply prefix_app_r.
    spec H_max.
    clean.
    rewrite wproj_symbol_sender_neq. tauto.
    rewrite wproj_symbol_receiver_neq. tauto.
    unnil. assumption.
    apply prefix_app_not in H_max. contradiction.
    easy.
Qed. 

Lemma incomplete_run_means_max_neq_run :
  forall (w v : FinAsyncWord) (p : participant) (alpha rho : FinSyncWord),
    prefix (wproj w p) (wproj (split rho) p) ->
    prefix v (wproj w p) -> 
    is_alpha rho alpha v p -> 
    length v < length (wproj w p) ->
    alpha = rho ->
    False. 
Proof. 
  intros w v p alpha rho H_compliant H_pref_w2 [H_pref [H_alpha H_max]] H_length H_eq.
  apply prefix_exists_suffix in H_pref. 
  destruct H_pref as [w2_rest H_w1_w2].
  destruct w2_rest as [|y w2_rest].
  (* In the case that y is empty, find a contradiction to H_w1_w2 *) 
  2 : { rewrite H_eq in H_w1_w2.
        eapply eq_app_false. 
        exact H_w1_w2. }
  destruct H_pref_w2. 
  destruct x. 
  - (* In the case that rho = alpha *)
    rewrite app_nil_r in H.
    rewrite H in H_length.
    lia.
  - (* In the case that rho is longer than alpha *)
    rewrite H in H_compliant. 
    rewrite <- H_eq in H_compliant.
    assert (H_helper : prefix (wproj (v ++ a :: x) p) (wproj (split alpha) p)).
    { replace (wproj (split alpha) p) with (wproj (wproj (split alpha) p) p).
      apply wproj_preserves_prefix.
      exact H_compliant.
      now rewrite wproj_idempotent. } 
    rewrite wproj_app in H_helper.
    apply (prefix_app_not (wproj v p) (wproj (a::x) p)). 
    simpl.
    assert (H_helper' := in_wproj_means_eq).
    spec H_helper' w p a.
    spec H_helper'.
    rewrite H.
    apply in_elt.
    rewrite H_helper'. easy.
    rewrite <- H_alpha in H_helper. 
    assumption.
Qed.

(* The next action following the maximal run prefix for a word must be the synchronous counterpart of the asynchronous symbol *) 
Lemma finite_unique_splitting_next_active :
  forall (run : FinSyncWord) (w : FinAsyncWord) (x : AsyncAlphabet) (p : participant) (alpha : FinSyncWord) (y : SyncAlphabet) (beta : FinSyncWord),
    prefix (w ++ [x]) (wproj (split run) p) -> 
    is_alpha run alpha w p ->
    run = alpha ++ [y] ++ beta ->
    y = async_to_sync x. 
Proof. 
  intros run w x p alpha y beta H_compliant [H_pref [H_eq H_max]] H_split.
  destruct (classic (y = async_to_sync x)). 
  - easy. 
  - exfalso.
    (* Direct proof by contradiction *)
    assert (H_pref' := H_compliant).
    apply (wproj_preserves_prefix _ _ p) in H_pref'.
    rewrite wproj_idempotent in H_pref'.
    rewrite H_split in H_pref'.
    rewrite wproj_app in H_pref'.
    repeat rewrite wproj_split_app in H_pref'.
    rewrite <- H_eq in H_pref'.
    apply prefix_app_inv in H_pref'.
    assert (H_eq' : wproj [x] p = [x]).
    { assert (H_helper := in_wproj_means_active).
      spec H_helper (split run) p.
      rewrite Forall_forall in H_helper.
      spec H_helper x.
      spec H_helper.
      apply prefix_exists_suffix in H_compliant.
      destruct H_compliant as [v H_run_split].
      rewrite H_run_split.
      apply elem_of_list_In.
      apply in_or_app.
      left.
      apply in_or_app.
      right. simpl. tauto.
      destruct x as [x H_neq].
      destruct x as [q r m | q r m].
      destruct H_helper as [H_snd _].
      spec H_snd. easy.
      simpl in H_snd.
      assert (H_helper := wproj_sender_eq q r p m H_neq H_snd).
      exact H_helper.
      destruct H_helper as [_ H_rcv].
      spec H_rcv. easy.
      simpl in H_rcv.
      assert (H_helper := wproj_receiver_eq q r p m H_neq H_rcv).
      now rewrite H_helper. } 
    rewrite H_eq' in H_pref'. clear H_eq'.
    destruct y as [y H_neq_y].
    destruct y as [q r m ]. 
    destruct (split_case q r p m H_neq_y).
    (* In the two cases where wproj (split x) is non-empty,
       find a contradiction to y not matching x *) 
    { rewrite H0 in H_pref'.
      inversion H_pref'.  
      inversion H1. rewrite <- H3 in H.
      simpl in H. contradiction. }
    destruct H0.
    { rewrite H0 in H_pref'.
      inversion H_pref'.  
      inversion H1. rewrite <- H3 in H.
      simpl in H. contradiction. }
    (* In the case that wproj (split x) is empty,
       find a contradiction to the maximality of alpha *)
    { spec H_max (alpha ++ [Event q r m ↾ H_neq_y]). 
      spec H_max. 
      rewrite H_split.
      rewrite app_assoc. apply prefix_app_r.
      reflexivity.
      spec H_max. rewrite wproj_split_app.
      rewrite H0.
      rewrite app_nil_r. 
      assumption.
      apply prefix_app_not in H_max. contradiction.
      easy. } 
Qed.


Lemma finite_unique_splitting_word_next_active :
  forall (run : FinAsyncWord) (w : FinAsyncWord) (x : AsyncAlphabet) (p : participant) (alpha : FinAsyncWord) (y : AsyncAlphabet) (beta : FinAsyncWord),
    prefix (w ++ [x]) (wproj run p) -> 
    is_alpha_async run alpha w p ->
    run = alpha ++ [y] ++ beta ->
    y = x. 
Proof. 
  intros run w x p alpha y beta H_compliant [H_pref [H_eq H_max]] H_split.
  destruct (classic (y = x)). 
  - easy. 
  - exfalso.
    (* Direct proof by contradiction *)
    assert (H_pref' := H_compliant).
    apply (wproj_preserves_prefix _ _ p) in H_pref'.
    rewrite wproj_idempotent in H_pref'.
    rewrite H_split in H_pref'.
    rewrite wproj_app in H_pref'.
    repeat rewrite wproj_app in H_pref'.
    rewrite <- H_eq in H_pref'.
    apply prefix_app_inv in H_pref'.
    assert (H_eq' : wproj [x] p = [x]).
    { assert (H_helper := in_wproj_means_active).
      spec H_helper run p.
      rewrite Forall_forall in H_helper.
      spec H_helper x.
      spec H_helper.
      apply prefix_exists_suffix in H_compliant.
      destruct H_compliant as [v H_run_split].
      rewrite H_run_split.
      apply elem_of_list_In.
      apply in_or_app.
      left.
      apply in_or_app.
      right. simpl. tauto.
      destruct x as [x H_neq].
      destruct x as [q r m | q r m].
      destruct H_helper as [H_snd _].
      spec H_snd. easy.
      simpl in H_snd.
      assert (H_helper := wproj_sender_eq q r p m H_neq H_snd).
      exact H_helper.
      destruct H_helper as [_ H_rcv].
      spec H_rcv. easy.
      simpl in H_rcv.
      assert (H_helper := wproj_receiver_eq q r p m H_neq H_rcv).
      now rewrite H_helper. } 
    rewrite H_eq' in H_pref'. clear H_eq'.
    destruct y as [y H_neq_y].
    destruct y as [q r m | q r m]. 
    * destruct (classic (q = p)). 
      (* In the case that wproj (split x) is non-empty,
       find a contradiction to y not matching x *) 
      { subst. 
        rewrite wproj_sender_eq in H_pref'. reflexivity.
        inversion H_pref'.
        inversion H0. subst. contradiction. }
      (* In the case that wproj (split x) is empty,
       find a contradiction to the maximality of alpha *)
      { spec H_max (alpha ++ [Snd q r m ↾ H_neq_y]). 
        spec H_max. 
        rewrite H_split.
        rewrite app_assoc. apply prefix_app_r.
        reflexivity.
        spec H_max. rewrite wproj_app.
        rewrite wproj_sender_neq.
        assumption. 
        rewrite app_nil_r. 
        assumption.
        apply prefix_app_not in H_max. contradiction.
        easy. }  
    * destruct (classic (r = p)).
      (* In the case that wproj (split x) is non-empty,
       find a contradiction to y not matching x *) 
      { subst. 
        rewrite wproj_receiver_eq in H_pref'. reflexivity.
        inversion H_pref'.
        inversion H0. subst. contradiction. }
      (* In the case that wproj (split x) is empty,
       find a contradiction to the maximality of alpha *)
      { spec H_max (alpha ++ [Rcv q r m ↾ H_neq_y]). 
        spec H_max. 
        rewrite H_split.
        rewrite app_assoc. apply prefix_app_r.
        reflexivity.
        spec H_max. rewrite wproj_app.
        rewrite wproj_receiver_neq.
        assumption. 
        rewrite app_nil_r. 
        assumption.
        apply prefix_app_not in H_max. contradiction.
        easy. }  
Qed.

Lemma run_for_word_means_run_prefix_for_word_prefix :
  forall (S : @LTS SyncAlphabet State) (w : FinAsyncWord) (rho : FinSyncWord) (s : State) (p : participant),
    @lts.Reachable SyncAlphabet State S (s0 S) rho s ->
    per_role_identical (split rho) w -> 
    channel_compliant w ->  
    forall (w' : FinAsyncWord),
      prefix w' w ->
      exists (rho' : FinSyncWord),
        prefix rho' rho /\
          wproj (split rho') p = wproj w' p. 
Proof.
  (* This lemma is redundant now but I can't be bothered to change its usage in context so here we go *) 
  intros S w rho s p H_rho H_role H_cc w' H_pref .
  assert (H_useful := finite_unique_splitting). 
  spec H_useful rho w' p.
  spec H_useful. spec H_role p.
  eapply PreOrder_Transitive with (wproj w p).
  apply (wproj_preserves_prefix _ _ p) in H_pref.
  assumption. now rewrite H_role.
  destruct H_useful as [alpha H_alpha].
  unfold is_alpha in H_alpha.
  exists alpha. split.
  tauto.
  easy.
Qed. 

Lemma prefix_app_finite_unique_splitting_elaborate :
  forall (rho : FinSyncWord) (w : FinAsyncWord) (x : AsyncAlphabet) (p : participant),
    prefix (w ++ [x]) (wproj (split rho) p) ->
    exists (alpha : FinSyncWord) (y : SyncAlphabet) (beta : FinSyncWord),
      rho = alpha ++ [y] ++ beta /\
        is_alpha rho alpha w p /\
        wproj (split_symbol y) p = [x].
Proof. 
  intros.
  assert (H_eq : wproj w p = w). 
  {
    assert (H_useful2 := wproj_no_effect w p).
    spec H_useful2.
    apply prefix_exists_suffix in H.
    destruct H as [v H_eq].
    assert (H_helper := in_wproj_means_active).
    spec H_helper (split rho) p.
    rewrite H_eq in H_helper.
    rewrite <- app_assoc in H_helper.
    apply Forall_app in H_helper.
    tauto. assumption. }
  assert (H_eq' : wproj (w ++ [x]) p = w ++ [x]). 
  {
    assert (H_useful2 := wproj_no_effect (w ++ [x]) p).
    spec H_useful2.
    apply prefix_exists_suffix in H.
    destruct H as [v' H_eq'].
    assert (H_helper := in_wproj_means_active).
    spec H_helper (split rho) p.
    rewrite H_eq' in H_helper.
    apply Forall_app in H_helper.
    tauto. assumption. }
  assert (H_pref : prefix (wproj w p) (wproj (split rho) p)).
  {
    rewrite <- H_eq in H.
    eapply PreOrder_Transitive with (wproj w p ++ [x]).
    now apply prefix_app_r. assumption. }
  assert (H_useful := finite_unique_splitting). 
  spec H_useful rho w p H_pref.
  destruct H_useful as [alpha H_max].
  exists alpha.
  assert (H_max_copy := H_max). 
  destruct H_max as [H_pref_alpha [H_alpha H_max_alpha]]. 
  apply prefix_exists_suffix in H_pref_alpha.
  destruct H_pref_alpha as [beta H_pref_alpha].
  destruct beta as [|y beta].
  { 
    (* Why can't it be the case that beta is empty? *)
    (* In this case, alpha = rho *) 
    (* But p has not completed all its actions in rho *)
    (* So we can prove a contradiction *)
    (* This helper lemma took forever to state correctly *)
    rewrite app_nil_r in H_pref_alpha. 
    assert (H_helper := incomplete_run_means_max_neq_run).
    spec H_helper (w ++ [x]) w p alpha rho. 
    spec H_helper.
    now rewrite H_eq'. 
    spec H_helper.
    rewrite <- H_eq at 1. 
    rewrite wproj_app.
    now apply prefix_app_r.
    spec H_helper. assumption. 
    spec H_helper.
    rewrite H_eq'.
    rewrite app_length. simpl. lia.
    spec H_helper.
    easy.
    contradiction. }
  assert (H_next_matches_x := finite_unique_splitting_next_active rho w x p alpha y beta).  
  spec H_next_matches_x H H_max_copy H_pref_alpha.  
  exists y, beta. split. assumption.
  split. assumption.
  rewrite H_next_matches_x.
  apply wproj_split_async_to_sync_eq.
  assert (H_helper := in_wproj_means_active (w ++ [x]) p).
  eapply Forall_forall in H_helper. 
  exact H_helper. rewrite H_eq'.
  apply elem_of_app. right.
  apply elem_of_list_In. 
  apply in_eq.
Qed.

(* In every CLTS configuration reached on some word with a non-empty channel between two roles, for all compliant runs with the word, the sender is farther along than the receiver *)
Lemma channel_non_empty_means_alpha_neq :
  forall (T : CLTS) (w : FinAsyncWord) (c_w : Configuration) (p q : participant) (m : message) (ls : list message),
    @Reachable LocalState T (c0 T) w c_w ->
    p <> q -> 
    get_channel_contents c_w p q = m :: ls ->
    forall (S : LTS) (run alpha1 alpha2 : FinSyncWord),
      @is_trace SyncAlphabet State S run ->
      is_alpha run alpha1 w p ->
      is_alpha run alpha2 w q ->
      alpha1 <> alpha2.
Proof. 
  intros T w c_w p q m ls H_reach H_neq H_chan_pq S rho rho_p rho_q H_run H_max_rho_p H_max_rho_q H_false.
  subst.
  (* Now we prove a contradiction *)
  (* How can it be the case that the sender's run is equal to the receiver's run? *)
  assert (H_helper1 := mproj_snd_preserves_active_wproj_prefix).
  assert (H_helper2 := mproj_rcv_preserves_active_wproj_prefix).
  eapply about_clts_trace_configuration_channel_contents in H_reach. 
  2 : exact H_neq.
  spec H_helper1 w (split rho_q) p.
  spec H_helper1.
  { unfold is_alpha in H_max_rho_p.
    destruct H_max_rho_p as [H_pref_rho [H_role_p H_max]].
    now rewrite H_role_p. }
  spec H_helper1 q H_neq.
  rewrite <- H_reach in H_helper1.
  enough (mproj_snd (split rho_q) p q = mproj_rcv w p q).
  assert (H_helper3 : prefix (mproj_snd (split rho_q) p q) (mproj_snd (split rho_q) p q)).
  { apply mproj_snd_preserves_active_wproj_prefix.
    apply wproj_preserves_prefix.
    apply prefix_split_prefix_iff. reflexivity. 
    exact H_neq. }
  eapply PreOrder_Transitive in H_helper1.
  spec H_helper1 H_helper3.
  rewrite H in H_helper1.
  apply prefix_app_not with (mproj_rcv w p q) (get_channel_contents c_w p q).
  rewrite H_chan_pq. easy.
  assumption.
  (* We just need to show this equality *)
  assert (H_helper := split_word_channel_complete rho_q). 
  spec H_helper p q H_neq.
  rewrite <- H_helper.
  rewrite mproj_rcv_wproj_idempotent.
  destruct H_max_rho_q as [H_pref_rho [H_role_eq H_max_rho_q]].
  rewrite <- H_role_eq.
  rewrite <- mproj_rcv_wproj_idempotent.
  reflexivity.
Qed.   

Lemma receive_extension_channel_compliant_means_alpha_neq :
  forall (w : FinAsyncWord) (p q : participant) (m : message) (H_neq : sender_receiver_neq_async (Rcv p q m)),
    channel_compliant (w ++ [Rcv p q m ↾ H_neq]) ->
    forall (S : LTS) (run alpha1 alpha2 : FinSyncWord),
      @is_trace SyncAlphabet State S run ->
      is_alpha run alpha1 w p ->
      is_alpha run alpha2 w q ->
      alpha1 <> alpha2.
Proof.
  intros w p q m H_neq H_cc_wx S rho rho_p rho_q H_run H_max_rho_p H_max_rho_q H_false.
  subst.
  (* Now we prove a contradiction *)
  (* How can it be the case that the sender's run is equal to the receiver's run? *)
  assert (H_helper1 := mproj_snd_preserves_active_wproj_prefix).
  assert (H_helper2 := mproj_rcv_preserves_active_wproj_prefix).
  (* The only difference between this lemma and the above should be to appeal to channel compliance of w instead of channel-nonemptiness of w *)
  spec H_cc_wx (w ++ [Rcv p q m ↾ H_neq]). 
  spec H_cc_wx. reflexivity. 
  spec H_cc_wx p q H_neq. apply prefix_exists_suffix in H_cc_wx.
  destruct H_cc_wx as [rest H_split]. 
  rewrite mproj_snd_app mproj_rcv_app in H_split.
  simpl in H_split.
  rewrite mproj_snd_rcv_eq in H_split.
  rewrite mproj_rcv_rcv_eq in H_split.
  unnil H_split. 
  rewrite app_nil_r in H_split.
  rewrite <- app_assoc in H_split.
  (* Now m :: rest = get_channel_contents w p q *)
  spec H_helper1 w (split rho_q) p.
  spec H_helper1.
  { unfold is_alpha in H_max_rho_p.
    destruct H_max_rho_p as [H_pref_rho [H_role_p H_max]].
    now rewrite H_role_p. }
  spec H_helper1 q H_neq.
  rewrite H_split in H_helper1.
  enough (mproj_snd (split rho_q) p q = mproj_rcv w p q).
  assert (H_helper3 : prefix (mproj_snd (split rho_q) p q) (mproj_snd (split rho_q) p q)).
  { apply mproj_snd_preserves_active_wproj_prefix.
    apply wproj_preserves_prefix.
    apply prefix_split_prefix_iff. reflexivity. 
    exact H_neq. }
  eapply PreOrder_Transitive in H_helper1.
  spec H_helper1 H_helper3.
  rewrite H in H_helper1.
  apply prefix_app_not with (mproj_rcv w p q) ([m] ++ rest).
  easy. assumption. 
  (* We just need to show this equality *)
  assert (H_helper := split_word_channel_complete rho_q). 
  spec H_helper p q H_neq.
  rewrite <- H_helper.
  rewrite mproj_rcv_wproj_idempotent.
  destruct H_max_rho_q as [H_pref_rho [H_role_eq H_max_rho_q]].
  rewrite <- H_role_eq.
  rewrite <- mproj_rcv_wproj_idempotent.
  reflexivity.
Qed. 
(* Voila! *) 

Lemma channel_non_empty_means_longer_compliant_prefix :
  forall (T : CLTS) (w : FinAsyncWord) (c_w : Configuration) (p q : participant) (m : message) (ls : list message),
    @Reachable LocalState T (c0 T) w c_w ->
    p <> q -> 
    get_channel_contents c_w p q = m :: ls ->
    forall (S : LTS) (run alpha1 alpha2 : FinSyncWord),
      @is_trace SyncAlphabet State S run ->
      is_alpha run alpha1 w p ->
      is_alpha run alpha2 w q ->
      prefix alpha2 alpha1 /\
        alpha1 <> alpha2. 
Proof. 
  intros T w c_w p q m ls H_reach H_neq H_chan_pq S rho alpha_p alpha_q H_run H_alpha_p H_alpha_q.
  (* First we establish that because alpha_p and alpha_q are prefixes of rho, one must be the prefix of the other *)
  assert (H_weak := prefix_weak_total alpha_p alpha_q rho). 
  spec H_weak.  
  { unfold is_alpha in H_alpha_p. tauto. }
  spec H_weak.
  { unfold is_alpha in H_alpha_q. tauto. }
  destruct H_weak as [H_wrong | H_right].
  (* The second disjunct, alpha_q <= alpha_p, matches our goal, except we need to additionally prove strict prefixhood *) 
  (* We show that alpha_p <> alpha_q by appealing to a helper lemma which says that if the channel in a word is non-empty, then the sender and receiver cannot have equal alphas *) 
  2 : { split. assumption.
        intro H_false.
        eapply channel_non_empty_means_alpha_neq. 
        exact H_reach. exact H_neq.
        exact H_chan_pq.
        exact H_run. exact H_alpha_p.
        exact H_alpha_q. assumption. }
  (* Now we prove a contradiction *)
  exfalso.
  (* Now we show a contradiction in the case that alpha_p <= alpha_q *)
  assert (H_helper1 := mproj_snd_preserves_active_wproj_prefix).
  assert (H_helper2 := mproj_rcv_preserves_active_wproj_prefix).
  eapply about_clts_trace_configuration_channel_contents in H_reach.
  2 : exact H_neq.
  spec H_helper1 w (split alpha_p) p.
  spec H_helper1.
  { unfold is_alpha in H_alpha_p.
    destruct H_alpha_p as [H_pref_rho [H_role_p H_max]].
    now rewrite H_role_p. }
  spec H_helper1 q H_neq.
  rewrite <- H_reach in H_helper1.
  enough (mproj_snd (split alpha_q) p q = mproj_rcv w p q).
  assert (H_helper3 : prefix (mproj_snd (split alpha_p) p q) (mproj_snd (split alpha_q) p q)).
  { apply mproj_snd_preserves_active_wproj_prefix.
    apply wproj_preserves_prefix.
    apply prefix_split_prefix_iff. exact H_wrong.
    exact H_neq. }
  eapply PreOrder_Transitive in H_helper1.
  spec H_helper1 H_helper3.
  rewrite H in H_helper1.
  apply prefix_app_not with (mproj_rcv w p q) (get_channel_contents c_w p q).
  rewrite H_chan_pq. easy.
  assumption.
  (* We just need to show this equality *)
  assert (H_helper := split_word_channel_complete alpha_q). 
  spec H_helper p q H_neq.
  rewrite <- H_helper.
  rewrite mproj_rcv_wproj_idempotent.
  destruct H_alpha_q as [H_pref_rho [H_role_eq H_alpha_q]].
  rewrite <- H_role_eq.
  rewrite <- mproj_rcv_wproj_idempotent.
  reflexivity.
Qed. 


Lemma channel_compliant_means_longer_compliant_prefix :
  forall (w : FinAsyncWord) (p q : participant) (m : message) (H_neq : sender_receiver_neq_async (Rcv p q m)),
    channel_compliant (w ++ [Rcv p q m ↾ H_neq]) ->
    forall (S : LTS) (run alpha1 alpha2 : FinSyncWord),
      @is_trace SyncAlphabet State S run ->
      is_alpha run alpha1 w p ->
      is_alpha run alpha2 w q ->
      prefix alpha2 alpha1 /\
        alpha1 <> alpha2.
Proof.
  intros w p q m H_neq H_cc_wx S rho rho_p rho_q H_run H_max_rho_p H_max_rho_q.
  (* First we establish that because rho_p and rho_q are prefixes of rho, one must be the prefix of the other *)
  assert (H_weak := prefix_weak_total rho_p rho_q rho).
  spec H_weak.
  { unfold is_alpha in H_max_rho_p. tauto. }
  spec H_weak.
  { unfold is_alpha in H_max_rho_q. tauto. }
  destruct H_weak as [H_wrong | H_right].
  2 : { split. assumption.
        intro H_false.
        eapply receive_extension_channel_compliant_means_alpha_neq.
        exact H_cc_wx. exact H_run.
        exact H_max_rho_p.
        exact H_max_rho_q. assumption. } 
  (* Now we prove a contradiction *)
  exfalso.
  (* How can it be the case that the sender's run is a prefix of the receiver's run? *)
  assert (H_helper1 := mproj_snd_preserves_active_wproj_prefix).
  assert (H_helper2 := mproj_rcv_preserves_active_wproj_prefix).
  (* Begin copypasta of same lines to manipulate out rest/get_channel_contents *)
  spec H_cc_wx (w ++ [Rcv p q m ↾ H_neq]). 
  spec H_cc_wx. reflexivity. 
  spec H_cc_wx p q H_neq. apply prefix_exists_suffix in H_cc_wx.
  destruct H_cc_wx as [rest H_split]. 
  rewrite mproj_snd_app mproj_rcv_app in H_split.
  simpl in H_split.
  rewrite mproj_snd_rcv_eq in H_split.
  rewrite mproj_rcv_rcv_eq in H_split.
  unnil H_split. 
  rewrite app_nil_r in H_split.
  rewrite <- app_assoc in H_split.
  (* End copypasta of same lines to manipulate out rest/get_channel_contents *) 
  spec H_helper1 w (split rho_p) p.
  spec H_helper1.
  { unfold is_alpha in H_max_rho_p.
    destruct H_max_rho_p as [H_pref_rho [H_role_p H_max]].
    now rewrite H_role_p. }
  spec H_helper1 q H_neq.
  rewrite H_split in H_helper1.
  enough (mproj_snd (split rho_q) p q = mproj_rcv w p q).
  assert (H_helper3 : prefix (mproj_snd (split rho_p) p q) (mproj_snd (split rho_q) p q)).
  { apply mproj_snd_preserves_active_wproj_prefix.
    apply wproj_preserves_prefix.
    apply prefix_split_prefix_iff. exact H_wrong.
    exact H_neq. }
  eapply PreOrder_Transitive in H_helper1.
  spec H_helper1 H_helper3.
  rewrite H in H_helper1.
  apply prefix_app_not with (mproj_rcv w p q) ([m] ++ rest). 
  easy. 
  assumption.
  (* We just need to show this equality *)
  assert (H_helper := split_word_channel_complete rho_q).
  spec H_helper p q H_neq.
  rewrite <- H_helper.
  rewrite mproj_rcv_wproj_idempotent.
  destruct H_max_rho_q as [H_pref_rho [H_role_eq H_max_rho_q]].
  rewrite <- H_role_eq.
  rewrite <- mproj_rcv_wproj_idempotent.
  reflexivity.
Qed.


Lemma channel_compliant_receiver_prefix_means_sender_prefix : 
  forall (w : FinAsyncWord) (p q : participant) (m : message) (H_neq : sender_receiver_neq_async (Rcv p q m)), 
    channel_compliant (w ++ [Rcv p q m ↾ H_neq]) -> 
    forall (S : @LTS SyncAlphabet State) (alpha beta : FinSyncWord),
      possible_run_prefix S (alpha ++ beta) (w ++ [Rcv p q m ↾ H_neq]) -> 
      prefix (wproj (split alpha) q) (wproj w q) ->
      prefix (wproj (split alpha) p) (wproj w p). 
Proof.
  intros w p q m H_neq H_cc_wx S alpha beta [H_trace H_compliant] H_pref.
  assert (H_useful := channel_compliant_means_longer_compliant_prefix). 
  spec H_useful w p q m H_neq H_cc_wx S (alpha ++ beta).
  (* Finding the unique splittings for p and q *) 
  assert (H_alpha_p := finite_unique_splitting (alpha ++ beta) w p).  
  assert (H_alpha_q := finite_unique_splitting (alpha ++ beta) w q).
  spec H_alpha_p. 
  { spec H_compliant p.
    rewrite wproj_app in H_compliant.
    simpl in H_compliant.
    rewrite wproj_symbol_receiver_neq in H_compliant.
    easy. unnil H_compliant. assumption.
  }
  spec H_alpha_q.
  {
    spec H_compliant q.
    rewrite wproj_app in H_compliant.
    simpl in H_compliant.
    rewrite wproj_symbol_receiver_eq in H_compliant.
    easy.
    rewrite app_nil_r in H_compliant.
    now apply prefix_app_l in H_compliant.
  }
  destruct H_alpha_p as [alpha_p H_alpha_p].
  destruct H_alpha_q as [alpha_q H_alpha_q].
  spec H_useful alpha_p alpha_q H_trace H_alpha_p H_alpha_q. 
  destruct H_useful as [H_pref_alpha H_strict].
  destruct H_alpha_p as [_ [H_eq_p _]]. 
  destruct H_alpha_q as [H_pref_alpha_q [H_eq_q H_max_q]].
  (* Now we just need to do some transitivity reasoning *)
  (* It suffices to show that alpha <= alpha_p *) 
  enough (prefix alpha alpha_p).
  { apply prefix_split_prefix_iff in H.
    eapply (wproj_preserves_prefix _ _ p) in H.
    rewrite H_eq_p.
    assumption.
  }
  (* To show that alpha <= alpha_p, it further suffices to show that alpha <= alpha_q *)
  enough (prefix alpha alpha_q).
  {
    now apply PreOrder_Transitive with alpha_q.
  }
  (* Showing that alpha <= alpha_q *)
  destruct (classic (wproj (split alpha) q = wproj w q)).
  { (* In the case that they're equal, appeal to maximality of alpha_q *)
    spec H_max_q alpha. spec H_max_q.
    now apply prefix_app_r.
    symmetry in H. now spec H_max_q H.
  }
  { (* In the case that alpha < alpha_q *)
    rewrite H_eq_q in H_pref. 
    assert (H_helper := prefix_weak_total alpha alpha_q (alpha ++ beta)). 
    spec H_helper. now apply prefix_app_r.
    spec H_helper. tauto.
    destruct H_helper. assumption.
    apply prefix_split_prefix_iff in H0.
    eapply (wproj_preserves_prefix _ _ q) in H0.
    apply symmetric_prefix_means_eq in H_pref.
    rewrite <- H_eq_q in H_pref.
    rewrite H_pref in H. contradiction.
    assumption.
  }
Qed.

Lemma split_infinite_run_is_infinite_word :
  forall (S : @LTS SyncAlphabet State) (run : InfSyncWord),
    deadlock_free S ->
    @is_infinite_run SyncAlphabet State S run -> 
    @is_infinite_protocol_word State S (split_inf run). 
Proof.     
  intros S run H_df H_run. 
  intro j. 
  (* Now depending on whether j is odd or even,
     we want to find an index into the infinite run
     such that there exists a suffix appended to the first j symbols
     that satisfies channel compliance and per-role identicality *)
  (* This index will roughly be half of j *) 
  assert (H_or := Nat.orb_even_odd j). 
  apply orb_prop in H_or.
  destruct H_or.
  - (* In the case that j is even *)
    (* We needn't append anything *)
    apply Nat.even_spec in H.
    destruct H as [k H_even].
    exists (stream_to_list run k), [].
    rewrite app_nil_r.
    rewrite H_even.
    enough (H_eq : stream_to_list (split_inf run) (2 * k) = split (stream_to_list run k)).
    rewrite H_eq.
    split. 
    spec H_run k. destruct H_run as [s1 [s2 [H_reach H_trans]]].
    eapply deadlock_free_lts_trace_prefix_iff. exact H_df.
    exists s1. assumption. split. intro. reflexivity.
    apply split_word_channel_compliant.
    apply split_inf_inf_split.
  - (* In the case that j is odd *)
    (* We need to append the rogue receive event *)
    apply Nat.odd_spec in H.
    destruct H as [k H_odd].
    exists (stream_to_list run (k+1)). 
    remember (Str_nth (2*k + 1) (split_inf run)) as x.
    exists [x]. subst j.
    (* We need a lemma that says the first 2k elements of a split_inf run is equal to the splitting of the first k elements of the infinite run *)
    assert (H_helper := split_inf_inf_split). 
    spec H_helper run (k+1).
    assert (H_eq : stream_to_list run k ++ [Str_nth k run] = stream_to_list run (k + 1)).
    { replace (k+1) with (Datatypes.S k).
      rewrite stream_to_list_S_Str_nth_app.
      reflexivity. lia. }
    rewrite <- H_eq in H_helper. 
    rewrite split_app in H_helper.
    assert (H_helper' := stream_to_list_S_Str_nth_app).
    spec H_helper' AsyncAlphabet ((2*k)+1) (split_inf run).
    replace (2*(k+1)) with (Datatypes.S (2*k+1)) in H_helper.
    2 : lia. rewrite H_helper' in H_helper.
    rewrite Heqx.
    rewrite H_helper.
    split.
    spec H_run (k+1).
    destruct H_run as [s1 [s2 [H_reach H_trans]]].
    eapply deadlock_free_lts_trace_prefix_iff. exact H_df.
    exists s1. assumption. 
    split. intros p0. replace (k+1) with (Datatypes.S k). 2 : lia.
    rewrite stream_to_list_S_Str_nth_app.
    clean. reflexivity.
    rewrite <- split_app.
    apply split_word_channel_compliant.
Qed.


Lemma split_run_protocol_prefix :
  forall (S : @LTS SyncAlphabet State) (rho : FinSyncWord) (s : State),
    deadlock_free S -> 
    @lts.Reachable SyncAlphabet State S (s0 S) rho s ->
    is_protocol_prefix S (split rho). 
Proof.
  intros S rho s H_df H_reach.
  assert (H_df_copy := H_df). 
  spec H_df s rho H_reach.
  destruct H_df as [H_fin | H_inf]. 
  - left.
    destruct H_fin as [run [H_max H_pref]].
    exists (split run). split.
    exists run. split. assumption. split.
    intro p. reflexivity.
    apply split_word_channel_compliant.
    now apply prefix_split_prefix_iff. 
  - right.
    destruct H_inf as [run [H_max H_pref]].
    (* Need a custom split_inf for infinite words *) 
    exists (split_inf run).
    split.
    now apply split_infinite_run_is_infinite_word. 
    destruct H_pref as [i H_pref]. 
    exists (2*i).
    assert (H_useful := split_inf_inf_split).
    spec H_useful run i.
    rewrite H_useful.
    rewrite H_pref.
    reflexivity.
Qed.

Lemma split_finite_run_is_finite_word :
  forall (S : @LTS SyncAlphabet State) (run : FinSyncWord),
    @is_finite_maximal_run SyncAlphabet State S run -> 
    @is_finite_protocol_word State S (split run). 
Proof.     
  intros S run H_run.
  exists run. 
  split. 
  assumption. 
  split. now intro p.
  now apply split_word_channel_compliant.
Qed. 

Lemma split_run_finite_possible_run_prefix :
  forall (S : @LTS SyncAlphabet State) (run : FinSyncWord) (w : FinSyncWord),
    GCLTS S ->
    @is_trace SyncAlphabet State S run ->
    @is_trace SyncAlphabet State S w -> 
    per_role_prefix (split w) (split run) -> 
    prefix w run. 
Proof. 
  intros S run w H_GCLTS H_trace_run H_trace_w H_role.
  induction w as [|a w IHw] using rev_ind; intros.
  - apply prefix_nil.
  - spec IHw.
    eapply lts_trace_prefix_closed_step. exact H_trace_w.
    spec IHw.
    intro p. spec H_role p.
    rewrite split_app in H_role.
    rewrite wproj_app in H_role.
    apply prefix_app_l in H_role.
    assumption.
    destruct (classic (prefix (w ++ [a]) run)).
    * assumption.
    * assert (H_useful := about_not_prefix _ w run a IHw H).
      spec H_useful. 
      { assert (H_length := prefix_length _ _ IHw).
        assert (length w < length run \/ length w = length run) by lia. 
        destruct H0. assumption.
        (* Why can't it be the case that w and run are equal length? *)
        (* Thankfully here we have the extra premise that w < run *) 
        (* Otherwise this would be a real pain to show *)
        (* From IHw and H0 we obtain that w = run *)
        apply prefix_length_eq in IHw.
        subst.
        destruct a as [a H_neq].
        destruct a as [p q m].
        spec H_role p.
        rewrite wproj_split_app in H_role.
        destruct (split_case p q p m H_neq).
        rewrite H1 in H_role.
        apply prefix_snoc_not in H_role. contradiction.
        destruct H1.
        rewrite H1 in H_role.
        apply prefix_snoc_not in H_role. contradiction.
        unfold wproj, wproj_symbol in H1. 
        simpl in H1.
        unfold sender_sync in H1; simpl in H1.
        rewrite participant_eqb_refl in H1. simpl in H1.
        inversion H1. lia. }  
      destruct H_useful as [y [suf [H_split H_neq]]].
      clear H.
      exfalso.
      (* First, we establish from the fact that w and rho are both synchronous runs of S,
         and the fact that S is sender-driven, 
         that a and y must have the same sender *) 
      assert (H_sender_eq : sender_sync a = sender_sync y). 
      {(* Obtaining the first transition for a *) 
        destruct H_GCLTS as [H_det [H_sender_driven _]].
        destruct H_trace_w as [s_wa H_reach_s_wa].
        apply lts.Reachable_unwind in H_reach_s_wa.
        destruct H_reach_s_wa as [s_w' [H_reach_s_w' H_transition1]].
        spec H_sender_driven s_w' s_wa.
        (* Obtaining the second transition for y *)
        assert (H_trace_wy : @is_trace SyncAlphabet State S (w ++ [y])).
        { eapply lts_trace_prefix_closed.
          exact H_trace_run.
          rewrite H_split.
          rewrite app_assoc.
          now apply prefix_app_r. } 
        destruct H_trace_wy as [s_wy H_reach_s_wy].
        apply lts.Reachable_unwind in H_reach_s_wy.
        destruct H_reach_s_wy as [s_w'' [H_reach_s_w'' H_transition2]].
        assert (H_eq : s_w' = s_w'').
        { eapply deterministic_word.
          exact H_det. exact H_reach_s_w'. exact H_reach_s_w''. }
        subst.
        spec H_sender_driven s_wy a y H_transition1 H_transition2.
        assumption. }
      (* Now we prove a contradiction using H_role and H_neq *) 
      (* Specifically, we instantiate H_role with one of the active roles in a *)
      destruct a as [a H_neq_a].
      destruct a as [p q m].
      spec H_role p.
      (* Now we massage H_role into its components *)
      rewrite H_split in H_role.
      rewrite wproj_split_app in H_role. 
      rewrite app_assoc in H_role.
      rewrite wproj_split_app in H_role.
      rewrite wproj_split_app in H_role.
      rewrite <- app_assoc in H_role.
      apply (prefix_app_inv (wproj (split w) p) (wproj (split [Event p q m ↾ H_neq_a]) p) ((wproj (split [y]) p) ++ wproj (split suf) p)) in H_role. 
      unfold sender_sync at 1 in H_sender_eq. simpl in H_sender_eq.
      destruct (split_case p q p m H_neq_a).
      destruct y as [y H_neq_y]. 
      destruct y as [p' q' m'].
      destruct (split_case p' q' p m' H_neq_y); unfold sender_sync in H_sender_eq; simpl in H_sender_eq. 
      { rewrite H H0 in H_role.
        inversion H_role.
        inversion H1.
        subst.
        apply H_neq.
        sigma_equal.
      }
      destruct H0.
      {
        inversion H0.
        unfold sender_sync, receiver_sync, value_sync in H2.
        simpl in H2.
        subst. unfold wproj_symbol at 1 in H2. simpl in H2.
        rewrite participant_eqb_refl in H2. simpl in H2.
        inversion H2.
      }
      { unfold wproj, wproj_symbol in H0. simpl in H0.
        unfold sender_sync, receiver_sync, value_sync in H0.
        simpl in H0.
        replace (participant_eqb p' p) with true in H0. simpl in H0.
        inversion H0. symmetry. now apply participant_eqb_correct.
      }
      destruct H.
      { unfold wproj, wproj_symbol in H.
        simpl in H.
        unfold sender_sync, receiver_sync, value_sync in H.
        simpl in H.
        rewrite participant_eqb_refl in H. simpl in H.
        inversion H. }
      { unfold wproj, wproj_symbol in H.
        simpl in H.
        unfold wproj, wproj_symbol in H.
        simpl in H.
        unfold sender_sync, receiver_sync, value_sync in H.
        simpl in H.
        rewrite participant_eqb_refl in H. simpl in H.
        inversion H. }
Qed.

Lemma possible_run_prefix_nil_inv :
  forall (w : FinAsyncWord),
    per_role_prefix w [] -> w = []. 
Proof.
  intros w H_possible. 
  destruct w.
  reflexivity.
  destruct a as [a H_neq].
  destruct a as [p q m | p q m].
  spec H_possible p.
  simpl in H_possible.
  rewrite wproj_symbol_sender_eq in H_possible.
  reflexivity. inversion H_possible.
  inversion H.
  spec H_possible q.
  simpl in H_possible.
  rewrite wproj_symbol_receiver_eq in H_possible.
  reflexivity. inversion H_possible.
  inversion H.
Qed.

(** Critical lemma for RCC soundness, intuition is basically declaratively snipping out some compliant run prefix's "parts" in a word **) 
(* This lemma is saying that if w is compliant with run rho, and I take some prefix of rho alpha, then I can make a w' that remains channel compliant and is compliant with beta, and furthermore if some role had already completed its actions in alpha, then all of them get removed *) 
Lemma possible_run_prefix_exists_possible_run_suffix :
  forall (S : @LTS SyncAlphabet State) (rho : FinSyncWord) (w : FinAsyncWord) (alpha beta : FinSyncWord) (s : State),
    deterministic S -> 
    possible_run_prefix S rho w ->
    channel_compliant w ->
    alpha ++ beta = rho ->
    @lts.Reachable SyncAlphabet State S (s0 S) alpha s -> 
    exists (w' : FinAsyncWord),
      possible_run_prefix (reinitial_S s S) beta w' /\
        channel_compliant w' /\
        (* Every participant who hadn't completed all actions in alpha in w is left with no actions*) 
        (forall (p : participant),
            prefix (wproj w p) (wproj (split alpha) p) ->
            wproj w' p = []) /\
        (* Every participant who had completed all actions in alpha is left with only the actions from beta *)
        (forall (p : participant),
            prefix (wproj (split alpha) p) (wproj w p) ->
            wproj w p = wproj (split alpha) p ++ wproj w' p). 
Proof.
  intros S rho w alpha beta s H_det [H_trace_rho H_compliant_rho] H_cc_w H_split_rho H_reach_alpha. 
  (* We instead rather want to prove this by induction on w! *)
  induction w as [|x w IHw] using rev_ind.
  - exists []. split.
    split. unfold is_trace.
    destruct H_trace_rho as [s_rho H_reach_rho].
    exists s_rho.
    rewrite <- H_split_rho in H_reach_rho. 
    apply lts.Reachable_app_inv in H_reach_rho.
    destruct H_reach_rho as [s_alpha [H_reach_alpha' H_reach_beta]].
    assert (s_alpha = s).
    { eapply deterministic_word.
      exact H_det.
      exact H_reach_alpha'.
      exact H_reach_alpha.
    }
    subst s_alpha.
    simpl. apply Reachable_means_Reachable_reinitial in H_reach_beta.
    assumption. 
    intros. simpl. apply prefix_nil.
    split. easy.
    split.
    intros. easy. intros.
    simpl in H. inversion H.
    symmetry in H0.
    apply app_eq_nil in H0.
    destruct H0; subst. 
    simpl. rewrite H0. easy.
  - (* Using the induction hypothesis *)
    spec IHw.
    {
      intro p0.
      spec H_compliant_rho p0.
      rewrite wproj_app in H_compliant_rho.
      apply prefix_app_l in H_compliant_rho.
      assumption.
    }
    spec IHw.
    {
      eapply prefix_preserves_channel_compliance.
      exact H_cc_w.
      now apply prefix_app_r.
    }
    destruct IHw as [w' [H_possible_abeta [H_cc_w' [H_w'_lazy H_w'_diligent]]]].
    (* Proof strategy: w' is w with all the present actions from alpha removed *) 
    (* We want to construct w'' which is wx with all the present actions from alpha removed *)
    (* So either we need to remove x, or we don't *)
    destruct x as [x H_neq].
    destruct x as [p q m | p q m].
    * (* In the case that x = Snd p q m *)
      (* Then we need to inspect wproj w p *)
      (* And compare its relation to wproj (split alpha) p *)
      (* In case that everything in alpha already got removed, then x can stay *)
      (* In the case that x needs to get removed, which corresponds to wproj w p ++ x <= wproj (split alpha) p *)
      (* Then we need to remove it *)
      assert (H_or := prefix_weak_total (wproj w p ++ [Snd p q m ↾ H_neq]) (wproj (split alpha) p) (wproj (split rho) p)).
      spec H_or.
      {
        spec H_compliant_rho p.
        clean H_compliant_rho.
      }
      spec H_or.
      { 
        apply wproj_preserves_prefix.
        apply prefix_split_prefix_iff.
        rewrite <- H_split_rho.
        now apply prefix_app_r.
      }
      assert (H_case : (prefix (wproj w p ++ [Snd p q m ↾ H_neq]) (wproj (split alpha) p)) \/ ((prefix (wproj (split alpha) p) (wproj w p ++ [Snd p q m ↾ H_neq]) /\ (wproj (split alpha) p <> (wproj w p ++ [Snd p q m ↾ H_neq]))))).
      {
        destruct H_or. 
        destruct (classic (wproj (split alpha) p = (wproj w p ++ [Snd p q m ↾ H_neq]))).
        tauto. tauto.
        destruct (classic (wproj (split alpha) p = (wproj w p ++ [Snd p q m ↾ H_neq]))).
        left. rewrite H0. reflexivity.
        tauto.
      }
      clear H_or. 
      destruct H_case as [H_fine | H_excise].
      ** exists w'.   
         split. assumption. split. assumption. 
         split.
         { intros p0.
           destruct (classic (p0 = p)).
           subst. clean. rewrite wproj_symbol_sender_eq.
         reflexivity.
         intros.
         spec H_w'_lazy p.
         apply H_w'_lazy.
         apply prefix_app_l in H.
         assumption.
         clean. rewrite wproj_symbol_sender_neq. easy.
         intros. spec H_w'_lazy p0.
         apply H_w'_lazy.
         rewrite app_nil_r in H0. easy.
         }
         { 
         intros p0.
         destruct (classic (p0 = p)).
         subst p0.
         spec H_w'_lazy p.
         spec H_w'_lazy.
         apply prefix_app_l in H_fine.
         assumption.
         rewrite H_w'_lazy.
         clean. rewrite wproj_symbol_sender_eq.
         reflexivity. 
         intros.
         apply symmetric_prefix_means_eq in H_fine.
         rewrite <- H_fine. 
         reflexivity. 
         assumption.
         intros. clean.
         rewrite wproj_symbol_sender_neq. easy.
         rewrite app_nil_r.
         spec H_w'_diligent p0.
         spec H_w'_diligent.
         clean H0.
         rewrite wproj_symbol_sender_neq in H0. easy.
         unnil H0. 
         rewrite app_nil_r in H0. assumption.
         assumption. }
      ** (* In the case where we need to remove x from w' *)
        destruct H_excise as [H_pref H_neq_w].
        (* Now we need to do further case analysis on whether alpha's bits were removed by w already *) 
        (* In which case we still don't need to remove anything *)
        apply prefix_app_tail_or in H_pref.
        destruct H_pref. 
        *** (* If all of alpha's bits were removed in w' *)
          assert (H_p_diligent : wproj w p = wproj (split alpha) p ++ wproj w' p). 
          { spec H_w'_diligent p.
            spec H_w'_diligent. assumption. assumption. }
          exists (w' ++ [Snd p q m ↾ H_neq]); repeat split.  
          { (* Proving trace-hood *)
            unfold possible_run_prefix in H_possible_abeta.
            tauto. 
          }
          { (* Proving per-role compliance *)
            intro p0.
            destruct (classic (p = p0)).
            subst. spec H_w'_diligent p0.
            spec H_w'_diligent H.
            spec H_compliant_rho p0.
            clean H_compliant_rho. 
            rewrite H_w'_diligent in H_compliant_rho.
            rewrite <- app_assoc in H_compliant_rho.
            apply prefix_app_inv in H_compliant_rho.
            rewrite wproj_symbol_sender_eq in H_compliant_rho.
            reflexivity.
            clean.
            clean.
            rewrite wproj_symbol_sender_neq.
            easy. rewrite app_nil_r.
            destruct H_possible_abeta as [_ H_compliant_beta].
            spec H_compliant_beta p0. assumption.
          }
          { (* Proving channel compliance *) 
            apply snd_extension_preserves_channel_compliance.
            assumption. easy.
          }
          { (* Proving about the lazy roles *) 
            intros p0.
            destruct (classic (p = p0)).
            subst p0. 
            intro H_false. 
            clean H_false.
            rewrite H_p_diligent in H_false.
            rewrite <- app_assoc in H_false.
            rewrite <- app_nil_r in H_false.
            eapply (prefix_app_inv ( wproj (split alpha) p) (wproj w' p ++ wproj_symbol p (Snd p q m ↾ H_neq)) []) in H_false.
            simpl in H_false.
            rewrite wproj_symbol_sender_eq in H_false.
            reflexivity.
            apply prefix_nil_inv in H_false.
            apply app_eq_nil in H_false. 
            easy.
            clean.
            rewrite wproj_symbol_sender_neq.
            easy.
            rewrite app_nil_r.
            intros.
            spec H_w'_lazy p0 H1.
            rewrite app_nil_r. easy.
          }
          { (* Proving about diligent roles *)
            intros p0.
            destruct (classic (p = p0)).
            subst p0.
            intros. 
            clean. rewrite H_p_diligent.
            rewrite <- app_assoc.
            reflexivity.
            intros.
            clean.
            rewrite wproj_symbol_sender_neq.
            easy.
            spec H_w'_diligent p0. spec H_w'_diligent.
            clean H1.
            rewrite wproj_symbol_sender_neq in H1.
            easy.  rewrite app_nil_r in H1.
            assumption.
            repeat rewrite app_nil_r.
            assumption.
          }
        *** contradiction.
    * (* Basically just change every occurrence of q into p *)
      (* In the case that x = Rcv p q m *) 
      (* Then we need to inspect wproj w q *)
      (* And compare its relation to wproj (split alpha) p *)
      (* In case that everything in alpha already got removed, then x can stay *)
      (* In the case that x needs to get removed, which corresponds to wproj w p ++ x <= wproj (split alpha) p *)
      (* Then we need to remove it *)
      assert (H_or := prefix_weak_total (wproj w q ++ [Rcv p q m ↾ H_neq]) (wproj (split alpha) q) (wproj (split rho) q)).
      spec H_or.
      {
        spec H_compliant_rho q.
        clean H_compliant_rho.
      }
      spec H_or.
      { 
        apply wproj_preserves_prefix.
        apply prefix_split_prefix_iff.
        rewrite <- H_split_rho.
        now apply prefix_app_r.
      }
      assert (H_case : (prefix (wproj w q ++ [Rcv p q m ↾ H_neq]) (wproj (split alpha) q)) \/ ((prefix (wproj (split alpha) q) (wproj w q ++ [Rcv p q m ↾ H_neq]) /\ (wproj (split alpha) q <> (wproj w q ++ [Rcv p q m ↾ H_neq]))))).
      {
        destruct H_or. 
        destruct (classic (wproj (split alpha) q = (wproj w q ++ [Rcv p q m ↾ H_neq]))).
        tauto. tauto.
        destruct (classic (wproj (split alpha) q = (wproj w q ++ [Rcv p q m ↾ H_neq]))).
        left. rewrite H0. reflexivity.
        tauto.
      }
      clear H_or. 
      destruct H_case as [H_fine | H_excise].
      ** exists w'.   
         split. assumption. split. assumption. 
         split.
         { intros p0.
           destruct (classic (p0 = q)).
           subst. clean. rewrite wproj_symbol_receiver_eq.
         reflexivity.
         intros.
         spec H_w'_lazy q.
         apply H_w'_lazy.
         apply prefix_app_l in H.
         assumption.
         clean. rewrite wproj_symbol_receiver_neq. easy.
         intros. spec H_w'_lazy p0.
         apply H_w'_lazy.
         rewrite app_nil_r in H0. easy.
         }
         { 
         intros p0.
         destruct (classic (p0 = q)).
         subst p0.
         spec H_w'_lazy q.
         spec H_w'_lazy.
         apply prefix_app_l in H_fine.
         assumption.
         rewrite H_w'_lazy.
         clean. rewrite wproj_symbol_receiver_eq.
         reflexivity. 
         intros.
         apply symmetric_prefix_means_eq in H_fine.
         rewrite <- H_fine. 
         subst.
         reflexivity.
         assumption.
         intros. clean.
         rewrite wproj_symbol_receiver_neq. easy.
         rewrite app_nil_r.
         spec H_w'_diligent p0.
         spec H_w'_diligent.
         clean H0.
         rewrite wproj_symbol_receiver_neq in H0. easy.
         rewrite app_nil_r in H0. 
         assumption.
         assumption.
         }
      ** (* In the case where we need to remove x from w' *)
        destruct H_excise as [H_pref H_neq_w].
        (* Now we need to do further case analysis on whether alpha's bits were removed by w already *) 
        (* In which case we still don't need to remove anything *)
        apply prefix_app_tail_or in H_pref.
        destruct H_pref. 
        *** (* If all of alpha's bits were removed in w' *)
          assert (H_p_diligent : wproj w q = wproj (split alpha) q ++ wproj w' q). 
          { spec H_w'_diligent q. 
            spec H_w'_diligent. assumption. assumption. }
          exists (w' ++ [Rcv p q m ↾ H_neq]); repeat split.  
          { (* Proving trace-hood *)            unfold possible_run_prefix in H_possible_abeta.
            tauto. 
          }
          { (* Proving per-role compliance *)
            intro p0.
            destruct (classic (q = p0)).
            subst. spec H_w'_diligent p0.
            spec H_w'_diligent H.
            spec H_compliant_rho p0.
            clean H_compliant_rho.
            rewrite H_w'_diligent in H_compliant_rho.
            rewrite <- app_assoc in H_compliant_rho.
            apply prefix_app_inv in H_compliant_rho.
            rewrite wproj_symbol_receiver_eq in H_compliant_rho.
            reflexivity.
            clean.
            clean. 
            rewrite wproj_symbol_receiver_neq.
            easy. rewrite app_nil_r.
            destruct H_possible_abeta as [_ H_compliant_beta].
            spec H_compliant_beta p0. assumption.
          }
          { (* Proving channel compliance *)
            intros w0 H_pref0 p0 q0 H_neq0. 
            apply prefix_app_tail_or in H_pref0.
            destruct H_pref0.
            now spec H_cc_w' w0 H0 p0 q0 H_neq0.
            destruct (classic (p = p0 /\ q = q0)). 
            {
              (* In the case that we are dealing with p and q *)
              destruct H1. subst p0 q0.  
              (* First we establish that the only thing that has changed is mproj_rcv w0 p q *) 
              assert (H_step1 : mproj_snd w0 p q = mproj_snd w' p q).
              { rewrite mproj_snd_wproj_idempotent.
                rewrite (mproj_snd_wproj_idempotent w' p q). 
                rewrite H0.
                clean.
                rewrite wproj_symbol_receiver_neq. easy.
                rewrite app_nil_r.
                rewrite <- mproj_snd_wproj_idempotent.
                reflexivity.
              }
              rewrite H_step1.
              (* Critical step, where we need the additional fact that alpha must also be a prefix for p *) 
              assert (H_step2 : mproj_snd w p q = mproj_snd (split alpha) p q ++ mproj_snd w' p q).
              {
                enough (prefix (wproj (split alpha) p) (wproj w p)).
                spec H_w'_diligent p H1.
                rewrite mproj_snd_wproj_idempotent.
                rewrite H_w'_diligent.
                rewrite mproj_snd_app.
                rewrite <- mproj_snd_wproj_idempotent.
                apply app_inv_head_iff.
                rewrite <- mproj_snd_wproj_idempotent.
                reflexivity.
                (* This should follow from the channel compliance of w *)
                eapply channel_compliant_receiver_prefix_means_sender_prefix. 
                exact H_cc_w.
                split. rewrite <- H_split_rho in H_trace_rho.
                exact H_trace_rho.
                rewrite H_split_rho. assumption.
                assumption.
              }
              
              spec H_cc_w (w ++ [Rcv p q m ↾ H_neq]). spec H_cc_w.
              reflexivity. 
              spec H_cc_w p q H_neq.
              rewrite mproj_rcv_app mproj_snd_app in H_cc_w.
              simpl in H_cc_w.
              rewrite mproj_rcv_rcv_eq in H_cc_w.
              rewrite mproj_snd_rcv_eq in H_cc_w.
              repeat rewrite app_nil_r in H_cc_w.
              rewrite H_step2 in H_cc_w.
              rewrite mproj_rcv_wproj_idempotent in H_cc_w.
              rewrite H_p_diligent in H_cc_w.  (* Misnamed? *)
              (* Now we're almost there *)
              rewrite mproj_rcv_app in H_cc_w.
              rewrite <- mproj_rcv_wproj_idempotent in H_cc_w.
              replace ( mproj_snd (split alpha) p q) with ( mproj_rcv (split alpha) p q) in H_cc_w.
              2 : { eapply split_word_channel_complete. easy. }
              rewrite <- app_assoc in H_cc_w. apply prefix_app_inv in H_cc_w. 
              rewrite H0.
              rewrite mproj_rcv_app.
              simpl.
              rewrite mproj_rcv_rcv_eq.
              rewrite app_nil_r.
              rewrite mproj_rcv_wproj_idempotent.
              assumption. (* Done!!! *)
            }
            { (* In the case that we are not dealing with p,q, nothing changes *)
              rewrite H0.
              clean_mproj.
              spec H_cc_w' w'. spec H_cc_w'.
              reflexivity.
              spec H_cc_w' p0 q0 H_neq0.
              assumption.
            }
          } (* Complete! *) 
          { (* Proving about the lazy roles *) 
            intros p0.
            destruct (classic (q = p0)).
            subst p0. 
            intro H_false. 
            clean H_false.
            rewrite H_p_diligent in H_false.
            rewrite <- app_assoc in H_false.
            rewrite <- app_nil_r in H_false.
            eapply (prefix_app_inv ( wproj (split alpha) q) (wproj w' q ++ wproj_symbol q (Rcv p q m ↾ H_neq)) []) in H_false.
            simpl in H_false.
            rewrite wproj_symbol_receiver_eq in H_false.
            reflexivity.
            apply prefix_nil_inv in H_false.
            apply app_eq_nil in H_false. 
            easy.
            clean.
            rewrite wproj_symbol_receiver_neq.
            easy.
            rewrite app_nil_r.
            intros.
            spec H_w'_lazy p0 H1.
            rewrite app_nil_r. easy.
          }
          { (* Proving about diligent roles *)
            intros p0.
            destruct (classic (q = p0)).
            subst p0.
            intros. 
            clean. rewrite H_p_diligent.
            rewrite <- app_assoc.
            reflexivity.
            intros.
            clean.
            rewrite wproj_symbol_receiver_neq.
            easy.
            spec H_w'_diligent p0. spec H_w'_diligent.
            clean H1.
            rewrite wproj_symbol_receiver_neq in H1.
            easy.  rewrite app_nil_r in H1.
            assumption.
            repeat rewrite app_nil_r.
            assumption.
          }
        *** contradiction. 
Qed.      

(* Clusterfuggle of this lemma *)
(* Restoring previous attempt *)
(* Attention! Previously the post-condition merely required channel complete *) 
(* As it turns out, channel complete does not imply channel compliant *)
(* So we really need the stronger post-condition of channel compliance *)
Lemma possible_run_prefix_channel_compliant_word_extension :
  forall (S : @LTS SyncAlphabet State) (w : FinAsyncWord) (rho_fin : FinSyncWord),
    channel_compliant w ->
    possible_run_prefix S rho_fin w ->
    exists (u : FinAsyncWord),
      channel_compliant (w ++ u) /\
        per_role_identical (w ++ u) (split rho_fin). 
Proof.
  intros S w rho_fin H_cc H_possible_rho_fin. 
  induction w as [|x w' IHw'] using rev_ind.  
  - exists (split rho_fin). split; rewrite app_nil_l. 
    apply split_word_channel_compliant.
    now intro p.
  - (* Plan for constructing u' for w'x :
       Step 1: Obtain u for w' from IHw'
       Step 2: Let p be the active role in x; find the maximal prefix of u with respect to [] for p, let this maximal prefix be u1
       Step 3: Show that u1 does not contain any actions from p, and that the next symbol in u following u1 must be   x, thereby splitting u = u1xu2 
       Step 4: Let u' be u1u2, and show that this retains channel compliance and per role-identicality *)
    (* Step 1 *)
    spec IHw'.
    { apply prefix_preserves_channel_compliance with (w' ++ [x]).
      exact H_cc. now apply prefix_app_r.
    }
    spec IHw'. 
    { apply prefix_preserves_possible_run with (w' ++ [x]).
      exact H_possible_rho_fin.
      now apply prefix_app_r. }
    destruct IHw' as [u [H_cc_u H_role_u]]. 
    (* Step 2 *) 
    destruct H_possible_rho_fin as [H_max_rho_fin H_role_rho_fin].
    destruct x as [x H_neq].
    destruct x as [p q m | p q m].
    * (* x = Snd p q m *)
      assert (H_step2 := finite_unique_splitting_word u [] p).
      spec H_step2.
      simpl. apply prefix_nil.  
      destruct H_step2 as [u1 H_max_prefix_u1].
      assert (H_max_prefix_u1_copy := H_max_prefix_u1).
      destruct H_max_prefix_u1 as [H_pref_u1 [H_empty_u1 H_max_u1]].
      apply prefix_exists_suffix in H_pref_u1.
      destruct H_pref_u1 as [u2 H_split_u].
      destruct u2 as [| y u2]. 
      (* Discharging the case where u2 is empty *)
      { rewrite app_nil_r in H_split_u.
        subst.
        spec H_role_u p.
        clean H_role_u.
        simpl in H_empty_u1.
        rewrite <- H_empty_u1 in H_role_u.
        rewrite app_nil_r in H_role_u.
        spec H_role_rho_fin p.
        rewrite <- H_role_u in H_role_rho_fin.
        clean H_role_rho_fin.
        rewrite wproj_symbol_sender_eq in H_role_rho_fin.
        reflexivity.
        apply prefix_app_not in H_role_rho_fin.
        contradiction. easy. }
      (* Step 3 *) 
      assert (H_step2' := finite_unique_splitting_word_next_active u [] (Snd p q m ↾ H_neq) p u1).
      spec H_step2' y u2.
      spec H_step2'.
      { rewrite app_nil_l. 
        spec H_role_rho_fin p.
        clean H_role_rho_fin.
        spec H_role_u p.
        rewrite <- H_role_u in H_role_rho_fin.
        clean H_role_rho_fin.
        apply prefix_app_inv in H_role_rho_fin.
        rewrite wproj_symbol_sender_eq in H_role_rho_fin.
        reflexivity.
        assumption. }
      spec H_step2' H_max_prefix_u1_copy H_split_u. 
      subst. 
      (* Step 4 *) 
      exists (u1 ++ u2).
      split. 
      (* Showing channel-compliance *)
      { simpl in H_empty_u1.
        symmetry in H_empty_u1.  
        now apply channel_compliant_fastforward_snd_suffix.
      }
      (* Showing per role-identicality *) 
      { intros p0.
        spec H_role_u p0.
        rewrite <- H_role_u.
        destruct (classic (p = p0)).
        ** (* In the case that we are dealing with p *)
          subst.
          clean.
          simpl in H_empty_u1.
          clean.
          rewrite <- H_empty_u1.
          simpl.
          rewrite wproj_symbol_sender_eq. reflexivity.
          now rewrite app_assoc.
        ** (* In the case that we are not dealing with p *)
          clean.
          rewrite wproj_symbol_sender_neq. 
          assumption.
          unnil. reflexivity. }
    * (* x = Rcv p q m *)
      assert (H_step2 := finite_unique_splitting_word u [] q).
      spec H_step2.
      simpl. apply prefix_nil.  
      destruct H_step2 as [u1 H_max_prefix_u1].
      assert (H_max_prefix_u1_copy := H_max_prefix_u1).
      destruct H_max_prefix_u1 as [H_pref_u1 [H_empty_u1 H_max_u1]].
      apply prefix_exists_suffix in H_pref_u1.
      destruct H_pref_u1 as [u2 H_split_u].
      destruct u2 as [| y u2]. 
      (* Discharging the case where u2 is empty *)
      { rewrite app_nil_r in H_split_u.
        subst.
        spec H_role_u q.
        clean H_role_u.
        simpl in H_empty_u1.
        rewrite <- H_empty_u1 in H_role_u.
        rewrite app_nil_r in H_role_u.
        spec H_role_rho_fin q.
        rewrite <- H_role_u in H_role_rho_fin.
        clean H_role_rho_fin.
        rewrite wproj_symbol_receiver_eq in H_role_rho_fin.
        reflexivity.
        apply prefix_app_not in H_role_rho_fin.
        contradiction. easy. }
      (* Step 3 *) 
      assert (H_step2' := finite_unique_splitting_word_next_active u [] (Rcv p q m ↾ H_neq) q u1).
      spec H_step2' y u2.
      spec H_step2'.
      { rewrite app_nil_l. 
        spec H_role_rho_fin q.
        clean H_role_rho_fin.
        spec H_role_u q.
        rewrite <- H_role_u in H_role_rho_fin.
        clean H_role_rho_fin.
        apply prefix_app_inv in H_role_rho_fin.
        rewrite wproj_symbol_receiver_eq in H_role_rho_fin.
        reflexivity.
        assumption. }
      spec H_step2' H_max_prefix_u1_copy H_split_u. 
      subst. 
      (* Step 4 *) 
      exists (u1 ++ u2).
      split. 
      (* Showing channel-compliance *)
      { eapply channel_compliant_fastforward_rcv_suffix.
        simpl in H_empty_u1. now symmetry.
        2 : exact H_cc_u.
        {
          spec H_cc (w' ++ [Rcv p q m ↾ H_neq]).
          spec H_cc. reflexivity. 
          spec H_cc p q H_neq.
          clean_mproj H_cc. assumption.
        } 
      }  
      (* Showing per role-identicality *) 
      { intros p0.
        spec H_role_u p0.
        rewrite <- H_role_u.
        destruct (classic (q = p0)).
        ** (* In the case that we are dealing with p *)
          subst.
          clean.
          simpl in H_empty_u1.
          clean.
          rewrite <- H_empty_u1.
          simpl.
          rewrite wproj_symbol_receiver_eq. reflexivity.
          now rewrite app_assoc.
        ** (* In the case that we are not dealing with p *)
          clean.
          rewrite wproj_symbol_receiver_neq. 
          assumption.
          unnil. reflexivity. }
Qed. 
 
Lemma finite_possible_run_means_possible_run_prefix :
  forall (S : @LTS SyncAlphabet State) (rho : FinSyncWord) (w : FinAsyncWord),
    finite_possible_run S rho w ->
    possible_run_prefix S rho w. 
Proof.
  intros S rho w [[s [H_reach _]] H_role].
  split. exists s. assumption. assumption.
Qed.

Lemma protocol_prefix_exists_possible_run_prefix :
  forall (S : @LTS SyncAlphabet State) (w : FinAsyncWord),
    deadlock_free S -> 
    is_protocol_prefix S w ->
    exists (rho : FinSyncWord),
      possible_run_prefix S rho w.
Proof. 
  intros S w H_df H_pref. 
  destruct H_pref as [H_fin | H_inf].
  (* If w is the prefix of a finite word *)
  - destruct H_fin as [w_fin [H_word H_pref]].
    destruct H_word as [rho_fin [H_max_rho [H_role H_cc]]]. 
    exists rho_fin. split.
    destruct H_max_rho as [s_rho [H_reach_rho H_fin_rho]].
    exists s_rho. tauto. intro p.
    spec H_role p. rewrite H_role.
    now apply (wproj_preserves_prefix _ _ p) in H_pref.
  - (* If w is the prefix of an infinite word *)
    destruct H_inf as [w_inf [H_word H_pref]].
    destruct H_pref as [i H_pref].
    spec H_word i.
    destruct H_word as [rho_inf [v [H_rho [H_role_rho H_cc]]]].
    exists rho_inf. 
    split.
    eapply deadlock_free_lts_trace_prefix_iff. exact H_df. tauto.
    intro p. spec H_role_rho p.
    rewrite <- H_pref.
    rewrite <- H_role_rho.
    rewrite wproj_app.
    now apply prefix_app_r.
Qed.

Lemma possible_run_prefix_extension_finite :
  forall (S : @LTS SyncAlphabet State) (rho : FinSyncWord) (w : FinAsyncWord),
    possible_run_prefix S rho w ->
    forall (rho' : FinSyncWord),
      prefix rho rho' ->
      @is_trace SyncAlphabet State S rho' -> 
      possible_run_prefix S rho' w. 
Proof. 
  intros S rho w [H_trace_rho H_compliant_rho] rho' H_trace_rho' H_pref.
  split.
  assumption.
  intro p.
  spec H_compliant_rho p.
  apply PreOrder_Transitive with (wproj (split rho) p).
  assumption.
  apply prefix_split_prefix_iff in H_trace_rho'.
  now apply wproj_preserves_prefix.
Qed.

(* This lemma says that if you take a run and a word such that the word is channel compliant and has completed all actions in the run, you can glue it to the splitting of an infinite run suffix to obtain an infinite word in S *) 
Lemma stream_app_infinite_word :
  forall (S : @LTS SyncAlphabet State) (pref : FinSyncWord) (suf_inf : InfSyncWord) (w : FinAsyncWord),
    deadlock_free S -> 
    channel_compliant w ->
    per_role_identical w (split pref) ->
    @is_infinite_run SyncAlphabet State S (cons_list_stream pref suf_inf) ->
    is_infinite_protocol_word S (cons_list_stream w (split_inf suf_inf)).
Proof.     
  intros S pref suf_inf w H_df H_cc H_role H_run.
  remember (cons_list_stream pref suf_inf) as rho_inf.
  intro i. 
  remember (stream_to_list
              (cons_list_stream w (split_inf suf_inf)) i) as inf_pref. 
  (* For every index into the candidate infinite word, we must find a index into rho_inf that satisfies this property *)
  (* We can invoke our extensibility lemma from above, we just need to find a finite run prefix that is possible for this word prefix *) 
  destruct (classic (i <= length w)). 
  {
    (* In the case that i <= length w *)
    (* Then we are dealing with a prefix of w, and we can just use pref *) 
    exists (stream_to_list rho_inf (length pref)).
    assert (H_possible : possible_run_prefix S pref w). 
    {
      split.
      spec H_run (length pref).
      destruct H_run as [s_pref [s_pref' [H_reach_pref _]]].
      exists s_pref.
      subst rho_inf.
      rewrite cons_list_stream_to_list_length in H_reach_pref.
      assumption.
      intro p. subst.
      spec H_role p. now rewrite H_role. }
    (* Now when i <= length, we don't actually need the extensibility lemma *) 
    (* We can just use the remaining suffix in i *)
    assert (H_useful : prefix inf_pref w).
    {
      rewrite Heqinf_pref.
      assert (H_helpful := stream_to_list_prefix). 
      spec H_helpful AsyncAlphabet (cons_list_stream w (split_inf suf_inf)) i (length w).
      spec H_helpful. lia.
      enough (w = stream_to_list (cons_list_stream w (split_inf suf_inf)) (length w)).
      rewrite H0. rewrite <- H0 at 1.
      assumption.
      symmetry. 
      apply cons_list_stream_to_list_length. }
    apply prefix_exists_suffix in H_useful.
    destruct H_useful as [w_suf H_split_w].
    exists w_suf.
    rewrite <- H_split_w.
    split.
    rewrite Heqrho_inf.
    rewrite cons_list_stream_to_list_length. 
    { destruct H_possible as [H_trace _].
      eapply deadlock_free_lts_trace_prefix_iff. exact H_df. assumption. }
    split. rewrite Heqrho_inf.
    rewrite cons_list_stream_to_list_length. assumption. assumption.
  }
  {
    (* In the case that i > length w *)
    rename H into H_gt. 
    remember (i - length w) as j.
    assert (H_or := Nat.orb_even_odd j). 
    apply orb_prop in H_or.
    destruct H_or as [H | H].
    - (* In the case that j is even *)
      (* We needn't append anything *)
      apply Nat.even_spec in H.
      destruct H as [k H_even].
      exists (stream_to_list rho_inf (length pref + k)). 
      exists [].
      rewrite app_nil_r.
      assert (inf_pref = w ++ stream_to_list (split_inf suf_inf) (i - length w)).
      { rewrite Heqinf_pref.
        assert (H_helper := stream_to_list_cons_list_stream_length).
        spec H_helper AsyncAlphabet w (split_inf suf_inf) (i - length w).
        replace (length w + (i - length w)) with i in H_helper.
        rewrite H_helper. reflexivity.
        lia.
      } 
      split.
      (* Showing that this run prefix is a run in S *)
      { spec H_run (length pref + k).
        destruct H_run as [s1 [s2 [H_run _]]].
        eapply deadlock_free_lts_trace_prefix_iff. exact H_df. exists s1. assumption. }
      split. 
      (* Showing per role identicality *) 
      { intro p. 
        rewrite H.
        rewrite wproj_app.
        assert (H_eq : stream_to_list (split_inf rho_inf) (2 * k) = split (stream_to_list rho_inf k)).
        { apply split_inf_inf_split. }
        rewrite Heqrho_inf.  
        assert (stream_to_list (cons_list_stream pref suf_inf)
                  (length pref + k) = pref ++ stream_to_list suf_inf k).
        { assert (H_helper := stream_to_list_cons_list_stream_length).
          spec H_helper SyncAlphabet pref suf_inf k.
          rewrite H_helper. reflexivity.
        }
        rewrite H0.
        clean.
        spec H_role p. rewrite H_role.
        apply (app_inv_head_iff (wproj (split pref) p)). 
        rewrite <- Heqj. rewrite H_even.
        f_equal.
        rewrite split_inf_inf_split. reflexivity.
      }
      {
        (* Showing channel compliance *)
        rewrite H. 
        rewrite <- Heqj.
        rewrite H_even.
        rewrite split_inf_inf_split.
        eapply per_role_identical_word_app_split_word_channel_compliant. 
        exact H_role. exact H_cc.
      } 
    - (* In the case that j is odd *)
      (* We need to append the rogue receive event *)
      apply Nat.odd_spec in H.
      destruct H as [k H_odd].
      remember (Str_nth k suf_inf) as a. 
      destruct a as [a H_neq_a].
      destruct a as [p q m].
      assert (H_neq_rcv : sender_receiver_neq_async (Rcv p q m)) by easy.
      exists (stream_to_list rho_inf (length pref + k+1)).
      (* exists (length pref + k+1). *) 
      exists [exist _ (Rcv p q m) H_neq_rcv].
      symmetry in Heqa. 
      assert (H_useful := Str_nth_split_inf_inf_split suf_inf k p q m H_neq_a H_neq_rcv H_neq_rcv Heqa). 
      assert (inf_pref = w ++ stream_to_list (split_inf suf_inf) (i - length w)).
      { rewrite Heqinf_pref.
        assert (H_helper := stream_to_list_cons_list_stream_length).
        spec H_helper AsyncAlphabet w (split_inf suf_inf) (i - length w).
        replace (length w + (i - length w)) with i in H_helper.
        rewrite H_helper. reflexivity.
        lia.
      }
      split.
      (* Showing is trace *) 
      {
        spec H_run (length pref + k+1).
        destruct H_run as [s1 [_ [H_reach _]]].
        eapply deadlock_free_lts_trace_prefix_iff. exact H_df. exists s1. assumption. }
      split.
      (* Showing per role identicality *) 
      { intro p0. 
        rewrite H.
        rewrite wproj_app.
        assert (H_eq : stream_to_list (split_inf rho_inf) (2 * k) = split (stream_to_list rho_inf k)).
        { apply split_inf_inf_split. }
        rewrite Heqrho_inf.  
        assert (stream_to_list (cons_list_stream pref suf_inf)
                  (length pref + k + 1) = pref ++ stream_to_list suf_inf (k + 1)).
        { assert (H_helper := stream_to_list_cons_list_stream_length).
          spec H_helper SyncAlphabet pref suf_inf (k + 1).
          rewrite Nat.add_assoc in H_helper. rewrite H_helper. reflexivity.
        }
        rewrite H0.
        clean. 
        spec H_role p0.
        rewrite H_role.
        apply (app_inv_head_iff (wproj (split pref) p0)). 
        rewrite <- Heqj. rewrite H_odd.
        f_equal.
        rewrite <- app_assoc.
        apply (app_inv_head_iff (wproj (split pref) p0)).
        symmetry in Heqa. 
        replace (2 * k + 1) with (Datatypes.S (2 * k)) by lia.
        rewrite stream_to_list_S_Str_nth_app.
        replace (k + 1) with (Datatypes.S k) by lia.
        rewrite stream_to_list_S_Str_nth_app.
        rewrite wproj_split_app.
        rewrite <- Heqa. 
        rewrite split_inf_inf_split.
        rewrite wproj_app. rewrite <- app_assoc.
        apply app_inv_head_iff.
        destruct H_useful as [H_useful _].
        rewrite H_useful.
        simpl.
        clean. 
        f_equal. f_equal. sigma_equal.
        f_equal. f_equal. sigma_equal.
      }
      {
        (* Showing channel compliance *)
        rewrite H. 
        rewrite <- Heqj.
        rewrite H_odd. 
        assert (H_helper := per_role_identical_word_app_split_word_channel_compliant). 
        spec H_helper w pref.
        assert (H_eq : stream_to_list (split_inf suf_inf) (2 * k + 1) ++ [Rcv p q m ↾ H_neq_rcv] = stream_to_list (split_inf suf_inf) (2 * k + 2)).
        { replace (2 * k + 2) with (Datatypes.S (2 * k + 1)) by lia.
          rewrite stream_to_list_S_Str_nth_app.
          destruct H_useful as [_ H_useful].
          rewrite H_useful. reflexivity.
        }
        replace (2 * k + 2) with (2 * (k + 1)) in H_eq by lia.
        rewrite split_inf_inf_split in H_eq.
        spec H_helper (stream_to_list suf_inf (k+1)).
        spec H_helper H_role H_cc.
        rewrite <- H_eq in H_helper.
        rewrite <- app_assoc. assumption. 
      }
  }
Qed. 
(* Vanquished the annoying drudgery *)

(* Note: this lemma's premise can be weakened to possible run prefix *) 
Lemma channel_compliant_I_non_empty_implies_prefix :
  forall (S : @LTS SyncAlphabet State) (rho : FinSyncWord) (w : FinAsyncWord),
    deadlock_free S -> 
    channel_compliant w ->
    possible_run_prefix S rho w ->
    @is_protocol_prefix State S w. 
Proof.
  (** Proof strategy : need to find a protocol word that w is a prefix of
      Step 1: Extend rho to a maximal run that is either finite or infinite
      Step 2: In the case that rho is part of a finite maximal run, invoke a lemma that says w can be extended to be per role complete for this finite maximal run, giving us wu 
      Step 3: The finite word witness we want is obtained as wu, and the prefix relation holds by construction
      Step 4: Show that wu is a finite protocol word *) 
  intros S rho w H_df H_cc H_possible.
  (* assert (H_df : deadlock_free S) by (unfold GCLTS in H_GCLTS; tauto). *)
  assert (H_possible_copy := H_possible). 
  destruct H_possible as [H_run H_compliant].
  destruct H_run as [s_rho H_reach_rho].
  assert (H_df_copy := H_df). 
  spec H_df s_rho rho H_reach_rho.
  (* From the fact that S is deadlock-free and rho is a run in S,
     we obtain either a finite suffix or infinite suffix for rho *) 
  destruct H_df as [[rho_fin [H_max H_pref]] | [rho_inf [H_inf H_pref]]].
  {
    (* In the case that rho is part of a finite run *)
    left.
    (* Then we want to find the finite protocol word that w is a prefix of *)
    (* Using this extension lemma *)
    assert (H_useful := possible_run_prefix_channel_compliant_word_extension).
    (* Attention! Want to instantiate this with rho_fin and not rho *) 
    spec H_useful S w rho_fin H_cc.
    spec H_useful.
    apply possible_run_prefix_extension_finite with rho. 
    assumption. assumption. 
    { destruct H_max as [s_rho_fin [H_reach_rho_fin H_max_rho_fin]].
      exists s_rho_fin.
      assumption. }
    destruct H_useful as [u [H_cc_wu H_role_wu]].
    exists (w ++ u). split.
    exists rho_fin. split. tauto.
    split.
    now intro p. assumption.
    now apply prefix_app_r.
  }
  {
    (* In the case that rho is part of an infinite run *)
    right.
    (* We want to find an infinite protocol word that w is a prefix of *)
    assert (H_useful := possible_run_prefix_channel_compliant_word_extension).
    (* Attention! Want to instantiate this with rho this time *)
    spec H_useful S w rho H_cc. 
    spec H_useful.
    apply possible_run_prefix_extension_finite with rho. 
    assumption. reflexivity.
    { exists s_rho.  assumption. }
    destruct H_useful as [u [H_cc_wu H_role_wu]].
    (* Now we construct our infinite word by gluing wu to split of the remainder of rho_inf after rho *)
    destruct H_pref as [i_rho H_split_rho_inf]. 
    exists (cons_list_stream (w ++ u) (split_inf (Str_nth_tl i_rho rho_inf))).
    split. 
    (* Showing that the amalgamation is actually an infinite protocol word *)
    eapply stream_app_infinite_word.
    exact H_df_copy. assumption. 
    exact H_role_wu. 
    rewrite <- H_split_rho_inf. 
    rewrite cons_list_stream_index_glue. 
    assumption. 
    exists (length w).
    rewrite cons_list_stream_app.
    rewrite cons_list_stream_to_list_length.
    reflexivity.
  }
Qed.

Lemma not_disagreeing:
  forall (S : @LTS SyncAlphabet State) (w : FinAsyncWord) (rho_fin : FinSyncWord) (alpha : FinSyncWord),
    possible_run_prefix S rho_fin w ->
    prefix alpha rho_fin -> 
    forall (q : participant),
      prefix (wproj w q) (wproj (split alpha) q) \/
        prefix (wproj (split alpha) q) (wproj w q). 
Proof. 
  intros S w rho_fin alpha [H_trace H_possible_run] H_pref.
  intro q.
  eapply prefix_weak_total with (wproj (split rho_fin) q).
  exact (H_possible_run q).
  apply wproj_preserves_prefix.
  apply prefix_split_prefix_iff.
  assumption.
Qed.

Lemma finite_possible_run_is_possible_run_prefix :
  forall (S : @LTS SyncAlphabet State) (w : FinAsyncWord) (rho_fin : FinSyncWord),
    finite_possible_run S rho_fin w ->
    possible_run_prefix S rho_fin w. 
Proof. 
  intros S w rho_fin H_fin.
  destruct H_fin as [H_fin H_role].
  destruct H_fin as [s_rho [H_reach H_max]].
  split. exists s_rho. assumption.
  assumption.
Qed.

Lemma possible_run_prefix_exists_nil :
  forall (S : @LTS SyncAlphabet State),
    GCLTS S ->
    exists (rho : FinSyncWord),
      possible_run_prefix S rho []. 
(* Note that here we cannot prove that this possible run prefix is not alpha for all participants, 
   because there may be runs in the protocol that no participants are part of! *)
Proof.     
  intros S H_GCLTS.
  destruct H_GCLTS as [_ [_ [_ [_ H_run]]]].
  destruct H_run as [H_fin | H_inf]. 
  - destruct H_fin as [rho H_fin].
    exists rho.
    split. eapply finite_possible_run_is_possible_run_prefix with [].
    split. assumption. intros.
    eapply prefix_nil.
    intros. simpl. apply prefix_nil.
  - destruct H_inf as [rho H_inf].
    assert (H_useful := infinite_possible_run_means_finite_possible_run_prefix S rho).
    spec H_useful (@nil AsyncAlphabet).
    spec H_useful. split. assumption.
    intro p. 
    exists 0.  simpl. reflexivity.
    destruct H_useful as [run_fin [H_possible H_pref]].
    exists run_fin. easy.
Qed.


End Run. 
