Require Import List Classical Nat Logic Lia ExtensionalityFacts Streams stdpp.list.
Import ListNotations. 
From Coq 
Require Import ssreflect. 
From CC
       Require Import lib structures lts clts. 


(** Protocol model and semantics **)
Section Protocol.

Context {State : Type} {LocalState : Type}. 

Definition sender_driven (S : LTS) :=
  forall (s s1 s2 : State) (x1 x2 : SyncAlphabet),
    transition S s x1 s1 ->
    transition S s x2 s2 ->
    sender_sync x1 = sender_sync x2. 

Definition GCLTS (S : LTS) :=
  deterministic S /\
    sender_driven S /\
    sink_final S /\
    deadlock_free S /\
    exists_run S.

Definition per_role_identical (w1 w2 : FinAsyncWord) :=
  forall (p : participant),
    wproj w1 p = wproj w2 p. 

Definition per_role_prefix (w1 w2 : FinAsyncWord) :=
  forall (p : participant),
    prefix (wproj w1 p) (wproj w2 p). 

Definition is_finite_protocol_word (S : LTS) (w : list AsyncAlphabet):=
  exists (rho : FinSyncWord),
    @is_finite_maximal_run SyncAlphabet State S rho
    /\
      per_role_identical (split rho) w
    /\
      channel_compliant w. 

Definition is_infinite_protocol_word (S : @LTS SyncAlphabet State) (w : InfAsyncWord) :=
  forall (i : nat),
  exists (rho : FinSyncWord) (v : FinAsyncWord),
    @is_lts_prefix SyncAlphabet State S rho /\
      per_role_identical (stream_to_list w i ++ v) (split rho) /\
      channel_compliant (stream_to_list w i ++ v). 

Definition is_protocol_prefix (S : @LTS SyncAlphabet State) (w : FinAsyncWord) :=
  (exists (w' : FinAsyncWord), @is_finite_protocol_word S w' /\ prefix w w')
  \/
    (exists (w' : InfAsyncWord), @is_infinite_protocol_word S w' /\ (exists (i : nat), stream_to_list w' i = w)).

Definition clts_includes_protocol_language {LocalState : Type} (S : @LTS SyncAlphabet State) (T : CLTS) :=
  (forall (w : FinAsyncWord),
      @is_finite_protocol_word S w -> @is_finite_clts_word LocalState T w)
  /\
    (forall (w : InfAsyncWord),
        @is_infinite_protocol_word S w -> @is_infinite_clts_word LocalState T w).

Definition protocol_includes_clts_language {LocalState : Type} (S : @LTS SyncAlphabet State) (T : CLTS) :=
  (forall (w : FinAsyncWord),
      @is_finite_clts_word LocalState T w -> @is_finite_protocol_word S w)
  /\
    (forall (w : InfAsyncWord),
        @is_infinite_clts_word LocalState T w -> @is_infinite_protocol_word S w).

Definition protocol_fidelity {LocalState : Type} (S : @LTS SyncAlphabet State) (T : CLTS) :=
  @clts_includes_protocol_language LocalState S T /\ @protocol_includes_clts_language LocalState S T. 
 
Definition implements {LocalState : Type} (S : @LTS SyncAlphabet State) (T : CLTS) :=
  protocol_fidelity S T 
  /\
    @clts_deadlock_free LocalState T. 

Definition implementable {LocalState : Type} (S : @LTS SyncAlphabet State) :=
  exists (T : CLTS),
    @implements LocalState S T. 

(** Facts about protocols and their semantics**)
Lemma protocol_fidelity_means_prefixes_eq :
  forall (S : @LTS SyncAlphabet State) (T : CLTS) (w : FinAsyncWord),
    implements S T ->
    is_protocol_prefix S w <->
      @is_clts_prefix LocalState T w.
Proof.
  intros S T w H_impl.
  destruct H_impl as [H_pf H_df].
  split; intro H_pref. 
  - destruct H_pref as [H_fin | H_inf].
    ** destruct H_fin as [w_fin [H_word H_pref]].
       left. exists w_fin. 
       split; try assumption.
       destruct H_pf as [[H_incl _] _]. 
       now apply H_incl. 
    ** assert (H_inf_copy := H_inf).
       destruct H_inf as [w_inf [H_word H_pref]].
       right. exists w_inf. split; try assumption. 
       destruct H_pf as [[_ H_incl] _].
       now apply H_incl.
  - destruct H_pref as [H_fin | H_inf]. 
    ** destruct H_fin as [w_fin [H_word H_pref]].
       left. exists w_fin. 
       split; try assumption.
       destruct H_pf as [_ [H_incl _]]. 
       now apply H_incl. 
    ** assert (H_inf_copy := H_inf).
       destruct H_inf as [w_inf [H_word H_pref]].
       right. exists w_inf. split; try assumption. 
       destruct H_pf as [_ [_ H_incl]].
       now apply H_incl.
Qed.

Lemma deadlock_free_means_state_final_or_extensible :
  forall (S : @LTS SyncAlphabet State),
    deterministic S -> 
    deadlock_free S ->
    forall (rho : FinSyncWord) (s : State),
      @lts.Reachable SyncAlphabet State S (s0 S) rho s -> 
      (exists (s' : State) (x : SyncAlphabet),
          transition S s x s')
      \/
        final S s. 
Proof. 
  intros S H_det H_df rho s H_reach. 
  spec H_df s rho H_reach. 
  destruct H_df as [[run_fin [H_max H_pref]] | [run_inf [H_inf H_pref]]]. 
  - destruct (classic (rho = run_fin)).
    * subst rho.
      clear H_pref.
      right.
      unfold is_finite_maximal_run in H_max.
      destruct H_max as [s1 [H_reach1 H_final1]].
      assert (s = s1).
      eapply (deterministic_word S s s1 run_fin).
      exact H_det. exact H_reach. exact H_reach1.
      subst s1. assumption.
    * apply prefix_exists_suffix in H_pref.
      destruct H_pref as [suf H_split_run_fin].
      destruct suf.
      rewrite app_nil_r in H_split_run_fin.
      symmetry in H_split_run_fin. contradiction.
      left.
      assert (@is_trace SyncAlphabet State S (rho ++ [s0])).
      { eapply lts_trace_prefix_closed with run_fin.
        destruct H_max as [sf [H_reach_f H_fin]].
        exists sf. assumption.
        rewrite H_split_run_fin.
        apply prefix_app.
        simpl. apply singleton_prefix_cons.
      }
      destruct H0 as [s' H_reach_rhos0].
      exists s', s0.
      apply lts.Reachable_unwind in H_reach_rhos0.
      destruct H_reach_rhos0 as [s'' [H_reach' H_transition]].
      assert (s'' = s).
      { now apply (deterministic_word S s'' s rho). }
      subst. assumption.
  - destruct H_pref as [i H_eq].
    spec H_inf i. 
    destruct H_inf as [s1 [s2 [H_reach' H_transition]]]. 
    left. exists s2, (Str_nth i run_inf).
    assert (s1 = s).
    eapply deterministic_word. exact H_det.
    exact H_reach'. subst rho.
    exact H_reach. subst.
    assumption.
Qed.


(** Defining LTS with initial states re-assigned **) 
Definition reinitial_S (s : State) (S : @LTS SyncAlphabet State) : @LTS SyncAlphabet State :=
  {| transition := transition S;
    s0 := s;
    final := final S |}.

(** Facts about reinitialized LTS **) 
Lemma Reachable_means_Reachable_reinitial :
  forall (S : @LTS SyncAlphabet State)
    (s1 s2 : State) (run : FinSyncWord),
    @lts.Reachable SyncAlphabet State S s1 run s2 <->
    @lts.Reachable SyncAlphabet State (reinitial_S s1 S) s1 run s2. 
Proof. 
  intros.
  generalize dependent s1.
  generalize dependent s2.
  induction run using rev_ind; intros. 
  - split; intro. 
    * apply Reachable_nil_inv in H.
      subst.
      apply lts.Reachable_refl.
    * apply Reachable_nil_inv in H.
      subst.
      apply lts.Reachable_refl.
  - split; intro.
    * apply lts.Reachable_unwind in H.
      destruct H as [s' [H_s' H_transition]].
      spec IHrun s' s1.
      destruct IHrun as [IHrun _].
      spec IHrun H_s'.
      eapply lts.Reachable_step.
      exact IHrun.
      simpl. assumption.
    * apply lts.Reachable_unwind in H.
      destruct H as [s' [H_s' H_transition]].
      spec IHrun s' s1.
      destruct IHrun as [_ IHrun].
      spec IHrun H_s'.
      eapply lts.Reachable_step.
      exact IHrun.
      simpl. assumption.
Qed.

Lemma Reachable_means_Reachable_reinitial_swap :
  forall (S : @LTS SyncAlphabet State)
    (s1 s2 s3 : State) (run : FinSyncWord),
    @lts.Reachable SyncAlphabet State S s2 run s3 <->
    @lts.Reachable SyncAlphabet State (reinitial_S s1 S) s2 run s3. 
Proof. 
  intros.
  generalize dependent s3.
  generalize dependent s2.
  generalize dependent s1. 
  induction run using rev_ind; intros. 
  - split; intro. 
    * apply Reachable_nil_inv in H.
      subst.
      apply lts.Reachable_refl.
    * apply Reachable_nil_inv in H.
      subst.
      apply lts.Reachable_refl.
  - split; intro.
    * apply lts.Reachable_unwind in H.
      destruct H as [s' [H_s' H_transition]].
      spec IHrun s1 s2 s'.
      destruct IHrun as [IHrun _].
      spec IHrun H_s'.
      eapply lts.Reachable_step.
      exact IHrun.
      simpl. assumption.
    * apply lts.Reachable_unwind in H.
      destruct H as [s' [H_s' H_transition]].
      spec IHrun s1 s2 s'.
      destruct IHrun as [_ IHrun].
      spec IHrun H_s'.
      eapply lts.Reachable_step.
      exact IHrun.
      simpl. assumption.
Qed. 
                                        
Lemma reinitial_S_preserves_deadlock_freedom :
  forall (S : @LTS SyncAlphabet State) (w : FinSyncWord) (s : State),
    deterministic S -> 
    deadlock_free S ->
    (* This premise is needed for this property to be true *) 
    @lts.Reachable SyncAlphabet State S (s0 S) w s ->
    deadlock_free (reinitial_S s S). 
Proof.     
  intros S w s H_det H_df H_reach.
  intros s' w' H_reach_s'.
  simpl in H_reach_s'.  
  spec H_df s' (w ++ w'). 
  spec H_df.
  (* And here is where that premise gets used *)
  (* We know that s0 -w-> s in S and s -w'-> s' in S' *)
  (* Therefore s0 -w++w'-> s' in S *)
  { 
    eapply Reachable_app.
    exact H_reach.
    (* Silently convert one S to the other *) 
    apply Reachable_means_Reachable_reinitial in H_reach_s'.
    assumption. 
  } 
  destruct H_df as [H_fin | H_inf].
  - destruct H_fin as [run_fin [H_max H_pref]].
    left. 
    apply prefix_exists_suffix in H_pref. 
    destruct H_pref as [run_suf H_split_run_fin].
    exists (w' ++ run_suf). split.
    destruct H_max as [s_run_fin [H_reach_run_fin H_max_run_fin]].
    exists s_run_fin. split. 
    eapply lts.Reachable_app.
    simpl. exact H_reach_s'.
    rewrite H_split_run_fin in H_reach_run_fin.
    apply lts.Reachable_app_inv in H_reach_run_fin.
    destruct H_reach_run_fin as [s'' [H_reach_ww' H_reach_run_suf]].
    apply lts.Reachable_app_inv in H_reach_ww'.
    destruct H_reach_ww' as [s_w [H_reach_w H_reach_s2']].
    assert (s = s_w).
    {
      eapply deterministic_word.
      exact H_det. 
      exact H_reach. exact H_reach_w. }
    subst s_w. 
    assert (s' = s'').
    {
      assert (H_goal : lts.Reachable S (s0 S) (w ++ w') s').
      {
        eapply lts.Reachable_app.
        exact H_reach.
        apply Reachable_means_Reachable_reinitial in H_reach_s'.
        exact H_reach_s'.
      }
      eapply deterministic_word.
      exact H_det.
      exact H_goal. 
      eapply lts.Reachable_app.
      exact H_reach. assumption.
    } 
    subst s''.
    rewrite (Reachable_means_Reachable_reinitial_swap S s s' s_run_fin) in H_reach_run_suf.
    assumption.
    simpl. assumption.
    now apply prefix_app_r.
  - destruct H_inf as [old_run [H_max H_pref]]. 
    right.
    destruct H_pref as [i H_split_old_run]. 
    remember (stream_to_list old_run i) as ww'. 
    assert (H_length_ww'_split : length ww' = length w + length w').
    { rewrite H_split_old_run.  
      apply app_length.
    }
    assert (H_length_ww' : length ww' = i).
    {
      rewrite Heqww'.
      apply stream_to_list_length.
    }
    remember (cons_list_stream w' (Str_nth_tl i old_run)) as new_run.
    assert (H_old_new : cons_list_stream w new_run = old_run). 
    { rewrite Heqnew_run.
      assert (H_useful := cons_list_stream_app).
      spec H_useful SyncAlphabet w w'. 
      rewrite <- H_length_ww'.
      rewrite H_length_ww'_split.
      spec H_useful (Str_nth_tl (length w + length w') old_run).
      rewrite <- H_useful.
      rewrite <- H_split_old_run. 
      rewrite Heqww'.
      rewrite <- H_length_ww'_split. 
      rewrite H_length_ww'.
      assert (H_useful' := cons_list_stream_index_glue). 
      spec H_useful' SyncAlphabet old_run i.
      assumption. }       
    exists new_run.
    split.
    intro k.
    simpl.
    assert (stream_to_list old_run (length w + k) = w ++ stream_to_list new_run k). 
    {
      
      assert (H_pref_w : stream_to_list old_run (length w) = w).
      {
        rewrite <- H_old_new. 
        assert (H_useful :=stream_to_list_cons_list_stream_length).
        spec H_useful SyncAlphabet w new_run 0.
        rewrite Nat.add_0_r in H_useful.
        rewrite H_useful.
        rewrite stream_to_list_zero.
        now rewrite app_nil_r. 
      }
      rewrite <- H_old_new.
      assert (H_useful :=stream_to_list_cons_list_stream_length).
      spec H_useful SyncAlphabet w new_run k.
      rewrite H_useful.
      reflexivity. 
    }
    spec H_max (length w + k).
    destruct H_max as [s_wwk [s_wwk' [H_reach_wwk H_transition_wwk]]].
    exists s_wwk, s_wwk'.
    split.
    rewrite H in H_reach_wwk.
    apply lts.Reachable_app_inv in H_reach_wwk. 
    destruct H_reach_wwk as [s_w [H_reach_w H_reach_s2']].
    assert (s = s_w). 
    {
      eapply deterministic_word.
      exact H_det. 
      exact H_reach. exact H_reach_w.
    }
    subst s_w.
    rewrite Reachable_means_Reachable_reinitial in H_reach_s2'.
    assumption.
    rewrite (Reachable_means_Reachable_reinitial_swap S s (s0 S) s_wwk) in H_reach_wwk.
    rewrite Heqnew_run.
    enough (Str_nth k
       (cons_list_stream w'
          (Str_nth_tl i old_run)) =
              (Str_nth
                          (length w + k)
                          old_run)). 
    rewrite H0. assumption.
    assert (H_rewrite := cons_list_stream_Str_nth).
    spec H_rewrite SyncAlphabet w (cons_list_stream w' (Str_nth_tl i old_run)) k.
    rewrite H_rewrite.
    rewrite <- Heqnew_run. rewrite H_old_new. 
    reflexivity. 
    exists (length w').
    rewrite Heqnew_run.
    apply cons_list_stream_to_list_length.
Qed.

End Protocol. 
