Require Import List Classical Nat Logic Lia ExtensionalityFacts Streams stdpp.list.
Import ListNotations.
From Coq
  Require Import ssreflect.
From CC
  Require Import lib structures.

(** ** LTS *)
Record LTS {A : Type} {State : Type} :=
  mkLTS {
      (* state : StateType -> Prop; *)
      transition : State -> A -> State -> Prop;
      s0 : State;
      final : State -> Prop;
    }.

Inductive Reachable {A : Type} {State : Type} (P : @LTS A State) : State -> list A -> State -> Prop :=
| Reachable_refl : forall (s : State), Reachable P s [] s
| Reachable_step : forall (s1 s2 s3 : State) (w : list A) (x : A),
    Reachable P s1 w s2 ->
    transition P s2 x s3 ->
    Reachable P s1 (w ++ [x]) s3.

Definition is_trace {A : Type} {State : Type} {T : @LTS A State} (w : list A) :=
  exists (s : State), @Reachable A State T (s0 T) w s.

Definition is_finite_maximal_run {A : Type} {State : Type} {T : @LTS A State} (w : list A) :=
  exists (s : State), @Reachable A State T (s0 T) w s /\ final T s. 

Definition is_infinite_run {A : Type} {State : Type} {T : @LTS A State} (w : Stream A) :=
  forall (i : nat),
  exists (s1 s2 : State),
    @Reachable A State T (s0 T) (stream_to_list w i) s1 /\ transition T s1 (Str_nth i w) s2.

Definition is_finite_word {A : Type} {State : Type} {T : @LTS A State} (w : list A) :=
  @is_finite_maximal_run A State T w.

Definition is_infinite_word {A : Type} {State : Type} {T : @LTS A State} (w : Stream A) :=
  @is_infinite_run A State T w.

Definition is_lts_prefix {A : Type} {State : Type} {T : @LTS A State} (w : list A) :=
  (exists (w_fin : list A), @is_finite_word A State T w_fin /\ prefix w w_fin)
  \/
    (exists (w_inf : Stream A), @is_infinite_word A State T w_inf /\ prefix_inf w w_inf). 


(* Basic facts about generic LTS *) 

Lemma Reachable_nil_inv :
  forall {A : Type} {State : Type} (P : @LTS A State) (s1 s2 : State),
    @Reachable A State P s1 [] s2 ->
    s1 = s2. 
Proof.
  intros. 
  inversion H. easy.
  apply app_nil in H0. destruct H0.
  inversion H5.
Qed.

Lemma Reachable_singleton :
  forall {A : Type} {State : Type} (P : @LTS A State) (s1 s2 : State) (x : A),
    transition P s1 x s2 ->
    @Reachable A State P s1 [x] s2. 
    
Proof.
  intros A P State s1 s2 x H.
  simpl. rewrite <- (app_nil_l [x]).
  eapply lts.Reachable_step with s1. 
  apply Reachable_refl. assumption.
Qed.

Lemma Reachable_singleton_inv :
  forall {A : Type} {State : Type} (P : @LTS A State) (s1 s2 : State) (x : A),
    @Reachable A State P s1 [x] s2 -> 
    transition P s1 x s2. 
Proof.
  intros A State P s1 s2 x H.
  inversion H. 
  apply app_singleton in H0.
  destruct H0.
  destruct H0.
  inversion H5.
  rewrite H0 in H1.
  apply Reachable_nil_inv in H1.
  rewrite H1. now rewrite <- H7.
  destruct H0.
  inversion H5.
Qed.

Lemma Reachable_unwind :
  forall {A : Type} {State : Type} (P : @LTS A State) (s0 s : State) (w : list A) (x : A),
    @Reachable A State P s0 (w ++ [x]) s ->
    exists (s' : State),
      @Reachable A State P s0 w s' /\
        transition P s' x s.
Proof.   
  intros A State P s0 s w x H_reach.
  (* destruct H as [H_reach H_neq].  *)
  inversion H_reach; simpl. 
  - symmetry in H1. apply app_eq_nil in H1.
    destruct H1.
    inversion H2. 
  - apply app_inj_tail in H.
    destruct H.
    exists s2. rewrite <- H. split. assumption. rewrite <- H4; assumption. 
Qed.

Lemma Reachable_app :
  forall {A : Type} {State : Type} (P : @LTS A State) (s1 s2 s3 : State) (w1 w2 : list A),
    @Reachable A State P s1 w1 s2 ->
    @Reachable A State P s2 w2 s3 ->
    @Reachable A State P s1 (w1 ++ w2) s3. 
Proof.   
  intros A State P s1 s2 s3 w1 w2 H_reach1 H_reach2.
  generalize dependent s3.
  induction w2 as [|a2 w2 IHw2] using rev_ind; intros.
  - apply Reachable_nil_inv in H_reach2.
    subst. now rewrite app_nil_r.
  - apply Reachable_unwind in H_reach2. 
    destruct H_reach2 as [s2' [H_reach_s2' H_transition]].
    spec IHw2 s2' H_reach_s2'.
    rewrite app_assoc.
    eapply Reachable_step. exact IHw2.
    assumption.
Qed.

Lemma Reachable_behead :
  forall {A : Type} {State : Type} (P : @LTS A State) (s0 s : State) (w : list A) (x : A),
    @Reachable A State P s0 (x :: w) s ->
    exists (s' : State),
      @Reachable A State P s' w s /\
        transition P s0 x s'.
Proof.   
  intros A State P s0 s w x H_reach.
  generalize dependent s. 
  generalize dependent x.
  generalize dependent s0.
  remember (length w) as n.
  generalize dependent w.
  induction n as [|n IHn]; intros. 
  - symmetry in Heqn.
    apply nil_length_inv in Heqn.
    rewrite Heqn. 
    exists s.  split.
    apply Reachable_refl.
    rewrite Heqn in H_reach. 
    now apply Reachable_singleton_inv in H_reach.
  - destruct (destruct_list_last _ w).
    * rewrite H in Heqn. 
      inversion Heqn.
    * destruct H as [a [ls H_ls]]. 
      spec IHn ls. spec IHn.
      rewrite H_ls in Heqn. 
      rewrite app_length in Heqn. simpl in Heqn. lia.
      rewrite H_ls in H_reach.
      simpl in H_reach.
      inversion H_reach; subst.
      replace ( x :: ls ++ [a]) with ((x :: ls) ++ [a]) in H.
      apply (app_inj_tail_iff w0 (x :: ls)) in H. destruct H. subst.
      spec IHn s1 x s3. 
      spec IHn H0. destruct IHn as [s' H1 H4].
      exists s'. split. destruct H1. eapply Reachable_step.
      exact H.  exact H2. tauto. tauto.
Qed. 

Lemma Reachable_app_inv :
  forall {A : Type} {State : Type} (P : @LTS A State) (s1 s3 : State) (w1 w2 : list A),
    @Reachable A State P s1 (w1 ++ w2) s3 ->
    exists (s2 : State),     
      @Reachable A State P s1 w1 s2 /\ 
        @Reachable A State P s2 w2 s3. 
Proof.
  intros. 
  generalize dependent w2.
  induction w1 using rev_ind; intros.
  - exists s1.  split. apply Reachable_refl.
    rewrite app_nil_l in H. assumption.
  - spec IHw1 (x :: w2).
    spec IHw1.
    rewrite <- app_assoc in H. simpl in H.
    assumption.
    destruct IHw1.
    destruct H0.
    apply Reachable_behead in H1.
    destruct H1.
    exists x1. split.
    eapply Reachable_step.
    exact H0. tauto. tauto.
Qed. 

Lemma lts_trace_prefix_closed_step :
  forall {A : Type} {State : Type} {T : LTS} (w : list A) (x : A),
    @is_trace A State T (w ++ [x]) -> @is_trace A State T w. 
Proof.
  intros A State T w x H.
  unfold is_trace in *.
  destruct H as [s H_reach].
  inversion H_reach.
  - symmetry in H1. apply app_eq_nil in H1.
    destruct H1.
    inversion H2. 
  - apply app_inj_tail in H.
    destruct H.
    exists s2. rewrite <- H. assumption.
Qed.

Lemma lts_trace_prefix_closed :
  forall {A : Type} {State : Type} {T : LTS} (w : list A),
    @is_trace A State T w -> 
    forall (w' : list A),
      prefix w' w ->
      @is_trace A State T w'. 
Proof.
  intros A State T w H. 
  induction w using rev_ind.
  - intros w' H_prefix. 
    apply prefix_nil_inv in H_prefix. 
    now rewrite H_prefix.
  - intros w'' H_prefix'.
    destruct (classic (@is_trace A State T w)) as [H_pos | H_neg].
    * spec IHw H_pos w''.
      assert (H_disj : prefix w'' w \/ w'' = w ++ [x]) by apply (prefix_app_tail_or w w'' x H_prefix'). 
      destruct H_disj as [H_prefix | H_eq].
      ** spec IHw H_prefix. assumption.
      ** rewrite H_eq. assumption.
    * apply lts_trace_prefix_closed_step in H.
      contradiction.
Qed.

Lemma lts_finite_maximal_word_prefix_closed :
  forall {A : Type} {State : Type} {T : LTS} (w : list A),
    @is_finite_maximal_run A State T w -> 
    forall (w' : list A),
      prefix w' w ->
      @is_trace A State T w'. 
Proof.
  intros A State T w H w' H_pref. 
  eapply lts_trace_prefix_closed with w.
  destruct H as [s [H_reach H_max]].
  exists s. assumption.
  assumption.
Qed.

(** Protocol-agnostic properties of LTSs **)
Definition deterministic {A : Type} {State : Type} (P : @LTS A State) :=
  forall (s1 s2 s3 : State) (a : A),
    transition P s1 a s2 ->
    transition P s1 a s3 ->
    s2 = s3.

Lemma deterministic_word :
  forall {A : Type} {State : Type} (P : @LTS A State) (s1 s2 : State) (w : list A),
    deterministic P -> 
    @Reachable A State P (s0 P) w s1 ->
    @Reachable A State P (s0 P) w s2 ->
    s1 = s2.
Proof. 
  intros A State P s1 s2 w H_det H_reach1 H_reach2. 
  remember (length w) as n.
  generalize dependent w.
  generalize dependent s2.
  generalize dependent s1.
  induction n; intros.
  - symmetry in Heqn.
    apply nil_length_inv in Heqn.
    rewrite Heqn in H_reach1, H_reach2.
    inversion H_reach1.
    inversion H_reach2.
    reflexivity.
    apply app_nil in H0.
    easy. apply app_nil in H. easy.
  - destruct (destruct_list_last _ w).
    * rewrite H in Heqn.
      inversion Heqn.
    * destruct H as [x [w' H_eq]].
      rewrite H_eq in H_reach1, H_reach2. 
      apply Reachable_unwind in H_reach1, H_reach2.
      destruct H_reach1 as [s1' [H_reach1 H_transition1]].
      destruct H_reach2 as [s2' [H_reach2 H_transition2]].
      spec IHn s1' s2' w' H_reach1 H_reach2.
      spec IHn.
      rewrite H_eq in Heqn. 
      rewrite app_length in Heqn. simpl in Heqn. lia.
      spec H_det s1' s1 s2 x H_transition1. 
      spec H_det. (* fuz wuz here *)
      rewrite IHn. assumption.
      assumption.
Qed.

Lemma reachable_extension_transition : 
  forall {A : Type} {State : Type} {T : @LTS A State} (w : list A) (x : A) (s1 s2 : State),
    @deterministic A State T -> 
    @Reachable A State T (s0 T) w s1 ->
    @Reachable A State T (s0 T) (w ++ [x]) s2 ->
    transition T s1 x s2. 
Proof.
  intros A State T w x s1 s2 H_det H_reach1 H_reach2. 
  inversion H_reach2.
  - symmetry in H1. apply app_eq_nil in H1.
    destruct H1.
    inversion H2. 
  - apply app_inj_tail in H.
    destruct H.
    rewrite H4 in H2. 
    assert (s1 = s4).
    { apply (deterministic_word _ s1 s4 w H_det H_reach1). rewrite <- H. assumption. }
    rewrite H5. assumption. 
Qed.

Definition sink_final {A : Type} {State : Type} (P : @LTS A State) :=
  forall (s s' : State) (x : A),
    final P s /\ transition P s x s' -> False.

Definition deadlock_free {A : Type} {State : Type} (P : @LTS A State) :=
  forall (s : State) (w : list A),
    @lts.Reachable A State P (s0 P) w s ->
    (exists (run : list A), @is_finite_maximal_run A State P run /\ prefix w run) \/
      (exists (run : Stream A), @is_infinite_run A State P run /\ exists (i : nat), stream_to_list run i = w).

Definition exists_run {A : Type} {State : Type} (P : @LTS A State) := 
  (exists (rho : list A), @is_finite_maximal_run A State P rho)
  \/
    (exists (rho : Stream A), @is_infinite_run A State P rho). 
                           
Lemma deadlock_free_lts_trace_prefix_iff :
  forall {A : Type} {State : Type} {T : LTS} (w : list A),
    deadlock_free T ->
    @is_trace A State T w <-> @is_lts_prefix A State T w. 
Proof.
  intros A State T w H_df.
  split.
  intros [s H_reach]. unfold is_lts_prefix.
  spec H_df s w H_reach. 
  assumption.
  intros.
  destruct H as [H_fin | H_inf].
  destruct H_fin as [w_fin [H_word H_pref]].
  eapply lts_trace_prefix_closed. 
  destruct H_word as [s [H_reach H_fin]].
  exists s. exact H_reach. assumption. destruct H_inf as [w_inf [H_word H_pref]].
  destruct H_pref as [i H_pref].
  spec H_word i. destruct H_word as [s [H_reach H_trans]].
  exists s. rewrite <- H_pref. tauto.
Qed. 
