Require Import List Classical Nat Logic Lia ExtensionalityFacts FunctionalExtensionality Streams stdpp.list.
Import ListNotations. 
From Coq 
  Require Import ssreflect.
From CC 
  Require Import lib structures lts.

(** ** CLTS *)
 
Definition Configuration {State : Type} : Type := (participant -> State) * (participant -> participant -> list message). 

Definition get_local_state {State : Type} (c : @Configuration State) (p : participant) : State :=
  match c with
  | (s, xi) => s p
  end. 

Definition get_channel_contents {State : Type} (c : @Configuration State) (p q : participant) : list message :=
  match c with
  | (s, xi) => xi p q
  end.

Definition message_at_head {State : Type} (c : @Configuration State) (p q : participant) (m : message) : Prop :=
  match (get_channel_contents c p q) with
  | m :: _ => True
  | _ => False
  end.

Definition states_unchanged_except {State : Type} (c1 c2 : @Configuration State) (p : participant) : Prop :=
  forall (q : participant),
    q <> p -> get_local_state c1 q = get_local_state c2 q.

Definition channels_unchanged_except {State : Type} (c1 c2 : @Configuration State) (p q : participant) : Prop :=
  forall (r s : participant),
    r <> p \/ s <> q -> get_channel_contents c1 r s = get_channel_contents c2 r s. 

Definition message_sent {State : Type} (c1 c2 : @Configuration State) (p q : participant) (m : message) : Prop :=
  channels_unchanged_except c1 c2 p q /\
  get_channel_contents c2 p q =
  get_channel_contents c1 p q ++ [m].

Definition message_received {State : Type} (c1 c2 : @Configuration State) (p q : participant) (m : message) : Prop :=
  channels_unchanged_except c1 c2 p q /\
    exists (l' : list message), get_channel_contents c1 p q = m :: l' /\ get_channel_contents c2 p q = l'. 

(* Adding in the property about local implementations' alphabets as an axiom of CLTS for now, instead of defining a dependent type for role alphabets *)
Record CLTS {State : Type} :=
  mkCLTS {
      implementations : participant -> @LTS AsyncAlphabet State;
      c0 : Configuration;
      about_c0 : (forall (p : participant), s0 (implementations p) = (get_local_state c0 p))
                 /\
                   (forall (p q : participant), get_channel_contents c0 p q = []);
      deterministic_implementations : forall (p : participant), deterministic (implementations p);
      about_implementations_alphabet : ∀ (w : FinAsyncWord) (p : participant) (s_w : State),
        @lts.Reachable AsyncAlphabet State (implementations p) (s0 (implementations p)) w s_w ->
        Forall (is_active p) w;
    }. 

Definition is_local_transition {State : Type} {T : @CLTS State} (c1 c2 : Configuration) (p : participant) (x : AsyncAlphabet) : Prop :=
  (transition (implementations T p)) (get_local_state c1 p) x (get_local_state c2 p). 

Definition Configuration_snd {State : Type} {T : @CLTS State} (c1 : Configuration) (x : AsyncAlphabet) (c2 : Configuration) :=
    states_unchanged_except c1 c2 (sender_async x) /\ 
    @is_local_transition State T c1 c2 (sender_async x) x /\ 
    message_sent c1 c2 (sender_async x) (receiver_async x) (value_async x).

Definition Configuration_rcv {State : Type} {T : @CLTS State} (c1 : Configuration) (x : AsyncAlphabet) (c2 : Configuration) :=
    states_unchanged_except c1 c2 (receiver_async x) /\ 
    @is_local_transition State T c1 c2 (receiver_async x) x /\ 
    message_received c1 c2 (sender_async x) (receiver_async x) (value_async x).

Definition Configuration_step {State : Type} {T : @CLTS State} (c1 : Configuration) (x : AsyncAlphabet) (c2 : Configuration) :=
  (is_snd x /\ @Configuration_snd State T c1 x c2) \/
  (is_rcv x /\ @Configuration_rcv State T c1 x c2). 

Definition Configuration_final  {State : Type} {T : @CLTS State} (c : Configuration) : Prop :=
  (forall (p : participant), final (implementations T p) (get_local_state c p))
  /\
    (forall (p q : participant), p <> q -> get_channel_contents c p q = []).

Inductive Reachable  {State : Type} {T : @CLTS State} : Configuration -> list AsyncAlphabet -> Configuration -> Prop :=
| Reachable_init : forall (c : Configuration) (x : AsyncAlphabet),
    (@Configuration_step State T (c0 T) x c) ->
    Reachable (c0 T) [x] c
| Reachable_refl : forall (c : Configuration), Reachable c [] c 
| Reachable_step : forall (c1 c2 c3 : Configuration) (w : list AsyncAlphabet) (x : AsyncAlphabet),
    Reachable c1 w c2 ->
    @Configuration_step State T c2 x c3 ->
    Reachable c1 (w ++ [x]) c3.

Definition is_clts_trace {State : Type} {T : @CLTS State} (w : FinAsyncWord) :=
  exists (c : Configuration), @Reachable State T (c0 T) w c.

Definition is_finite_clts_word {State : Type} {T : @CLTS State} (w : FinAsyncWord) :=
  exists (c : Configuration), @Reachable State T (c0 T) w c /\ @Configuration_final State T c.

Definition is_infinite_clts_word {State : Type} {T : @CLTS State} (w : InfAsyncWord) :=
  forall (i : nat),
  exists (c : Configuration),
    @Reachable State T (c0 T) (stream_to_list w i) c. 

Definition is_clts_prefix {State : Type} {T : @CLTS State} (w : FinAsyncWord) :=
  (exists (w_fin : FinAsyncWord), @is_finite_clts_word State T w_fin /\ prefix w w_fin)
  \/
    (exists (w_inf : InfAsyncWord), @is_infinite_clts_word State T w_inf /\ (exists (i : nat), stream_to_list w_inf i = w)). 

(* Stating CLTS deadlock freedom on trace extensions directly 
   instead of on states and then proving it on traces *) 
Definition clts_deadlock_free {State : Type} (T : @CLTS State) :=
  forall (c : Configuration) (w : FinAsyncWord),
    @Reachable State T (c0 T) w c ->
    (exists (w_fin : FinAsyncWord), @is_finite_clts_word State T w_fin /\ prefix w w_fin) \/
      (exists (w_inf : InfAsyncWord), @is_infinite_clts_word State T w_inf /\ prefix_inf w w_inf). 
    
Definition channel_compliant (w : FinAsyncWord) :=
  forall (w' : FinAsyncWord),
    prefix w' w ->
    forall (p q : participant),
      p <> q ->
      prefix (mproj_rcv w' p q) (mproj_snd w' p q). 

Definition channel_complete (w : FinAsyncWord) :=
  forall (p q : participant),
    p <> q ->
    mproj_rcv w p q = mproj_snd w p q.

(** Facts about CLTS **) 
Lemma Reachable_unwind :
  forall {State : Type} {T : @CLTS State} (c1 c2 : Configuration) (w : list AsyncAlphabet) (x : AsyncAlphabet),
    @Reachable State T c1 (w ++ [x]) c2 ->
    exists (c' : Configuration),
      @Reachable State T c1 w c' /\ @Configuration_step State T c' x c2. 
Proof.     
  intros State T c1 c2 w x H_reach.
  (* destruct H as [H_reach H_neq].  *)
  inversion H_reach; simpl. 
  - symmetry in H.
      rewrite <- app_nil_l in H. 
      apply app_inj_tail in H.
      destruct H.
      exists (c0 T). split.
      rewrite H.
      apply Reachable_refl. 
      now rewrite H3.
  - symmetry in H1. apply app_eq_nil in H1.
    destruct H1.
    inversion H2. 
  - apply app_inj_tail in H.
    destruct H.
    exists c4. rewrite <- H. split. assumption. rewrite <- H4; assumption. 
Qed.

Lemma deterministic_clts :
  forall {State : Type} (T : @CLTS State),
  forall (w : FinAsyncWord) (c1 c2 : Configuration),
    @Reachable State T (c0 T) w c1 ->
    @Reachable State T (c0 T) w c2 ->
    c1 = c2.
Proof. 
  intros State T w c1 c2 H_reach1 H_reach2.
  remember (length w) as n.
  generalize dependent w. 
  generalize dependent c1.
  generalize dependent c2.
  induction n; intros.
  - symmetry in Heqn.
    apply nil_length_inv in Heqn.
    rewrite Heqn in H_reach1, H_reach2.
    inversion H_reach1. inversion H_reach2.
    easy.
    apply app_nil in H0.
    destruct H0. inversion H6.
    apply app_nil in H.
    destruct H.
    inversion H4.
  - destruct (destruct_list_last _ w).
    * rewrite H in Heqn.
      inversion Heqn.
    * destruct H as [a [w' H_eq]].
      rewrite H_eq in H_reach1. 
      apply Reachable_unwind in H_reach1.
      destruct H_reach1 as [c_w' [H_reach1 H_step1]].
      rewrite H_eq in H_reach2.
      apply Reachable_unwind in H_reach2.
      destruct H_reach2 as [c_w'' [H_reach2 H_step2]].
      spec IHn c_w'' c_w' w' H_reach1 H_reach2.
      spec IHn. rewrite H_eq in Heqn.
      rewrite app_length in Heqn.
      simpl in Heqn. lia.
      destruct a.
      rewrite <- IHn in H_reach2, H_step2. 
      clear IHn.
      destruct c_w' as [s_w' xi_w']. 
      destruct x as [p q m | p q m].
      ** (* In the case that x is p!q:m *) 
        destruct H_step1 as [H_snd1 | H_false1].
        (* Discharging the mismatching event type case *) 
        2 : { 
          destruct H_false1.
          inversion H. }
        destruct H_step2 as [H_snd2 | H_false2].
        2 : {
          destruct H_false2.
          inversion H. }
        destruct H_snd1 as [_ H_snd1].
        destruct H_snd2 as [_ H_snd2].
        destruct H_snd1 as [H_states1 [H_active1 H_chan1]]. 
        destruct H_snd2 as [H_states2 [H_active2 H_chan2]].
        destruct c1 as [s_c1 xi_c1];
          destruct c2 as [s_c2 xi_c2].
        enough (s_c1 = s_c2 /\ xi_c1 = xi_c2).
        destruct H. rewrite H. rewrite H0.
        reflexivity.
        split.
        apply functional_extensionality.
        intro r.
        destruct (classic (r = p)).
        (* In the case that r = p *)
        (* Showing that the state maps are equal *) 
        {
          assert (H_useful :=  deterministic_implementations T).
          spec H_useful p.
          spec H_useful (s_w' p) (s_c1 p) (s_c2 p) (Snd p q m ↾ s).
          spec H_useful H_active1 H_active2.
          rewrite H.
          assumption.
        } 
        (* In the case that r != p *) 
        (* Showing that the state maps are equal *)
        {
          spec H_states1 r H.
          spec H_states2 r H.
          replace (s_c1 r) with (get_local_state (s_c1, xi_c1) r) by easy.
          replace (s_c2 r) with (get_local_state (s_c2, xi_c2) r) by easy.
          rewrite <- H_states1, H_states2.
          easy.
        }
        apply functional_extensionality. 
        intro r.
        apply functional_extensionality.
        intro r'.
        destruct (classic (r = p /\ r' = q)).
        (* In the case that r,r' = p,q *)
        (* Showing that the channel maps are equal *) 
        {
          destruct H_chan1 as [_ H_chan1]. 
          destruct H_chan2 as [_ H_chan2].
          simpl in H_chan1, H_chan2.
          destruct H.
          rewrite H H0.
          rewrite H_chan2.
          rewrite H_chan1. reflexivity. }
        (* In the case that r,r' != p,q *)
        (* Showing that the channel maps are equal *)
        {
          apply not_and_or in H.
          destruct H_chan1 as [H_chan1 _].
          destruct H_chan2 as [H_chan2 _].
          spec H_chan1 r r'.
          spec H_chan1. easy.
          spec H_chan2 r r'.
          spec H_chan2. easy.
          simpl in H_chan1, H_chan2. 
          rewrite <- H_chan1, H_chan2.
          reflexivity. }
      ** (* In the case that x is q?p:m *) 
        destruct H_step1 as [H_false1 | H_rcv1].
        (* Discharging the mismatching event type case *) 
        destruct H_false1.
        inversion H. 
        destruct H_step2 as [H_false2 | H_rcv2].
        destruct H_false2.
        inversion H. 
        destruct H_rcv1 as [_ H_rcv1].
        destruct H_rcv2 as [_ H_rcv2].
        destruct H_rcv1 as [H_states1 [H_active1 H_chan1]]. 
        destruct H_rcv2 as [H_states2 [H_active2 H_chan2]].
        destruct c1 as [s_c1 xi_c1];
          destruct c2 as [s_c2 xi_c2].
        enough (s_c1 = s_c2 /\ xi_c1 = xi_c2).
        destruct H. rewrite H. rewrite H0.
        reflexivity.
        split.
        apply functional_extensionality.
        intro r.
        destruct (classic (r = q)).
        (* In the case that r = q *)
        (* Showing that the state maps are equal *) 
        {
          assert (H_useful :=  deterministic_implementations T).
          spec H_useful q.
          spec H_useful (s_w' q) (s_c1 q) (s_c2 q) (Rcv p q m ↾ s).
          spec H_useful H_active1 H_active2.
          rewrite H.
          assumption.
        } 
        (* In the case that r != q *) 
        (* Showing that the state maps are equal *)
        {
          spec H_states1 r H.
          spec H_states2 r H.
          replace (s_c1 r) with (get_local_state (s_c1, xi_c1) r) by easy.
          replace (s_c2 r) with (get_local_state (s_c2, xi_c2) r) by easy.
          rewrite <- H_states1, H_states2.
          easy.
        }
        apply functional_extensionality. 
        intro r.
        apply functional_extensionality.
        intro r'.
        destruct (classic (r = p /\ r' = q)).
        (* In the case that r,r' = p,q *)
        (* Showing that the channel maps are equal *) 
        {
          destruct H_chan1 as [_ H_chan1]. 
          destruct H_chan2 as [_ H_chan2].
          simpl in H_chan1, H_chan2.
          destruct H_chan1 as [l1 [H_eq1 H_eq1']].
          destruct H_chan2 as [l2 [H_eq2 H_eq2']].
          destruct H.
          rewrite H H0. 
          rewrite H_eq1' H_eq2'.
          inversion H_eq1; inversion H_eq2.
          rewrite H2 in H3.
          inversion H3. easy. } 
        (* In the case that r,r' != p,q *)
        (* Showing that the channel maps are equal *)
        {
          apply not_and_or in H.
          destruct H_chan1 as [H_chan1 _].
          destruct H_chan2 as [H_chan2 _].
          spec H_chan1 r r'.
          spec H_chan1. easy.
          spec H_chan2 r r'.
          spec H_chan2. easy.
          simpl in H_chan1, H_chan2. 
          rewrite <- H_chan1, H_chan2.
          reflexivity. }
Qed.

Lemma clts_trace_prefix_closed_step :
  forall {State : Type} {T : @CLTS State} (w : FinAsyncWord) (x : AsyncAlphabet),
    @is_clts_trace State T (w ++ [x]) -> @is_clts_trace State T w. 
Proof.
  intros State T w x H.
  unfold is_clts_trace in *.
  destruct H as [c H_reach].
  inversion H_reach.
  - symmetry in H.
    rewrite <- app_nil_l in H. 
    apply app_inj_tail in H.
    exists (c0 T).
    destruct H as [H _].
    rewrite H; apply Reachable_refl.
  - symmetry in H1. apply app_eq_nil in H1.
    destruct H1.
    inversion H2. 
  - apply app_inj_tail in H.
    destruct H.
    exists c2. rewrite <- H. assumption.
Qed.

Lemma clts_trace_prefix_closed :
  forall {State : Type} {T : @CLTS State} (w : FinAsyncWord),
    @is_clts_trace State T w -> 
    forall (w' : FinAsyncWord),
      prefix w' w ->
      @is_clts_trace State T w'. 
Proof.
  intros State T w H. 
  induction w using rev_ind.
  - intros w' H_prefix. 
    apply prefix_nil_inv in H_prefix. 
    now rewrite H_prefix.
  - intros w'' H_prefix'.
    destruct (classic (@is_clts_trace State T w)) as [H_pos | H_neg].
    * spec IHw H_pos w''.
      assert (H_disj : prefix w'' w \/ w'' = w ++ [x]) by apply (prefix_app_tail_or w w'' x H_prefix'). 
      destruct H_disj as [H_prefix | H_eq].
      ** spec IHw H_prefix. assumption.
      ** rewrite H_eq. assumption.
    * apply clts_trace_prefix_closed_step in H.
      contradiction.
Qed.

Lemma deadlock_free_clts_trace_prefix_iff :
  forall {State : Type} {T : @CLTS State} (w : FinAsyncWord),
    clts_deadlock_free T -> 
    @is_clts_trace State T w <->
    @is_clts_prefix State T w.
Proof.
  intros State T w H_df.
  split.
  - intro H_trace. 
    destruct H_trace as [s H_reach]. 
    spec H_df s w H_reach.
    destruct H_df as [[w_fin [H_fin H_pref]] | [w_inf [H_inf H_pref]]].
    left. exists w_fin. tauto.
    right. exists w_inf. tauto.
  - intro H_pref.
    destruct H_pref as [H_fin | H_inf].
    destruct H_fin as [w_fin [H_word H_pref]].
    eapply clts_trace_prefix_closed.
    destruct H_word as [s_w [H_word H_fin]].
    exists s_w.  
    exact H_word. assumption.
    destruct H_inf as [w_inf [H_word H_pref]].
    destruct H_pref as [i H_pref].
    spec H_word i.
    destruct H_word as [c H_reach].
    exists c. rewrite <- H_pref. assumption.
Qed. 

Lemma receive_first_event_false :
  forall {State : Type} {T : @CLTS State} (c : Configuration) (x : AsyncAlphabet),
    is_rcv x -> 
    @Configuration_step State T (c0 T) x c -> False. 
Proof.
  intros State T c x H_event H_step.
  destruct H_step as [H_snd | H_rcv]. 
  - (* Send case, contradiction *)
    unfold Configuration_snd in H_snd.
    destruct H_snd as [H_snd [H_transition H_sent]].
    apply snd_rcv_not in H_snd. 
    contradiction. 
  - unfold Configuration_rcv in H_rcv.
    destruct H_rcv as [H_rcv [H_states [H_transition H_sent]]].
    unfold message_received in H_sent.
    destruct H_sent as [_ [l' [H_pre _]]].
    assert (H_contra := about_c0 T).
    destruct H_contra as [_ H_contra].
    spec H_contra (sender_async x) (receiver_async x).
    rewrite H_pre in H_contra.
    inversion H_contra.
Qed.

Lemma reach_epsilon_identity :
  forall {State : Type} {T : @CLTS State} (c1 c2 : Configuration) ,
    @Reachable State T c1 [] c2 -> c1 = c2. 
Proof.
  intros State T c1 c2 H_reach. 
  inversion H_reach.
  - reflexivity.
  - apply app_eq_nil in H.
    destruct H. inversion H4.
Qed.

Lemma reach_receive_first_false :
  forall {State : Type} {T : @CLTS State} (c : Configuration) (x : AsyncAlphabet),
    is_rcv x -> 
    @Reachable State T (c0 T) [x] c -> False. 
Proof.
  intros State T c x H_rcv H_reach. 
  inversion H_reach.
  - apply receive_first_event_false in H0. contradiction. assumption. 
  - apply app_eq_unit in H.
    destruct H. 
    destruct H. 
    rewrite H in H0.
    apply reach_epsilon_identity in H0.
    rewrite <- H0 in H2. 
    apply (@receive_first_event_false State T c x).
    inversion H4. 
    rewrite <- H6.
    now rewrite H6. 
    inversion H4. now rewrite <- H6.
    destruct H. inversion H4.
Qed.

Lemma about_clts_trace_configuration_channel_contents :
  forall {State : Type} {T : @CLTS State} (w : FinAsyncWord) (c : Configuration),
    @Reachable State T (c0 T) w c ->
    forall (p q : participant),
      p <> q ->
      (mproj_rcv w p q) ++ get_channel_contents c p q = (mproj_snd w p q). 
Proof.
  intros State T w c H_reach p q H_neq.
  generalize dependent c. 
  induction w as [| x w' IHw] using rev_ind.
  - intros c H_reach.
    assert (H_empty : get_channel_contents c p q = []).
    { apply reach_epsilon_identity in H_reach.
      rewrite <- H_reach.
      apply about_c0. }
    rewrite H_empty.
    reflexivity.
  - intros c. intro H_reach.
    apply Reachable_unwind in H_reach. 
    destruct H_reach as [c' [H_reach H_step]].
    spec IHw c' H_reach.
    destruct x as [x H_x].
    destruct x as [p' q' v | p' q' v].
    * (* Send case *)
      destruct (classic (p = p' /\ q = q')).
      (* In case the send action p!q:m *) 
      (* Then both sides get a m message appended to them *)
      ** destruct H as [H_p H_q].
         (* rewrite <- H_p in *.  *)
         (* rewrite <- H_q in *. *)
         (* clear H_p H_q. *)
         (* LHS gets a [v] *) 
         remember (exist _ (Snd p' q' v) H_x) as x.  
         assert (H_LHS :  mproj_rcv (w' ++ [x]) p q ++ get_channel_contents c p q =
                            mproj_rcv (w') p q ++ get_channel_contents c' p q ++ [v]).
         { unfold mproj_rcv; rewrite flat_map_app.
           rewrite <- app_assoc.
           assert (flat_map (mproj_rcv_symbol p' q') [x] = []). 
           destruct x. rewrite Heqx.
           eapply mproj_rcv_snd. 
           easy. rewrite <- H_p in H. rewrite <- H_q in H.
           rewrite H. clear H. rewrite app_assoc. rewrite app_nil_r.
           destruct H_step as [H_snd | H_rcv].
           destruct H_snd as [_ [H_unchanged [H_local [H_sent H_chan]]]].
           simpl in *.
           rewrite Heqx in H_chan. 
           simpl in H_chan. rewrite H_p. rewrite H_q. rewrite H_chan.
           reflexivity.
           inversion H_rcv. rewrite Heqx in H. inversion H.
         }
         (* RHS gets a [v] *)
         assert (H_RHS : mproj_snd (w' ++ [x]) p q = mproj_snd w' p q ++ [v]).
         { unfold mproj_snd; rewrite flat_map_app.
           assert (flat_map (mproj_snd_symbol p' q') [x] = [v]).
           destruct x. 
           rewrite Heqx.
           eapply mproj_snd_snd.
           easy. rewrite <- H_p in H. rewrite <- H_q in H.
           rewrite H. clear H.
           reflexivity.
         }
         rewrite H_LHS. 
         rewrite H_RHS.
         rewrite app_assoc. rewrite IHw. 
         reflexivity.
      ** (* In case the send action is not between p and q *) 
        (* Then both sides stay the same *)
        apply not_and_or in H.
        assert (H_inter : participant_eqb p' p && participant_eqb q' q = false).
        { apply andb_false_iff.
          destruct H.
          left.
          rewrite participant_eqb_comm.  
          apply participant_eqb_no in H.
          assumption.
          right.
          rewrite participant_eqb_comm.
          apply participant_eqb_no in H. assumption. }
        (* LHS gets nothing *)
        remember (exist _ (Snd p' q' v) H_x) as x.  
        assert (H_LHS :  mproj_rcv (w' ++ [x]) p q ++ get_channel_contents c p q =
                           mproj_rcv (w') p q ++ get_channel_contents c' p q).
        { unfold mproj_rcv; rewrite flat_map_app.
          rewrite <- app_assoc.
          unfold mproj_rcv_symbol at 2; simpl.
          rewrite Heqx. simpl. rewrite H_inter. 
          clear H_inter. rewrite andb_false_l. simpl.
          destruct H_step as [H_snd | H_rcv].
          destruct H_snd as [_ [H_unchanged [H_local [H_sent H_chan]]]].
          simpl in *.
          spec H_sent p q.
          rewrite Heqx in H_sent. 
          simpl in H_sent. spec H_sent H. 
          rewrite H_sent. reflexivity.
          inversion H_rcv. rewrite Heqx in H0. inversion H0. 
        }
        (* RHS gets nothing *)
        assert (H_RHS :  mproj_snd (w' ++ [x]) p q =
                           mproj_snd (w') p q).
        { unfold mproj_snd; rewrite flat_map_app.
          unfold mproj_snd_symbol at 2; simpl.
          rewrite Heqx. simpl.
          rewrite H_inter.
          rewrite andb_false_l.
          repeat rewrite app_nil_r. reflexivity. }
        rewrite H_LHS. rewrite H_RHS.
        assumption.
    * (* Receive case *) 
      destruct (classic (p = p' /\ q = q')).
      (* In case the receive action q?p:m *) 
      (* Then both sides remain the same *)
      ** destruct H as [H_p H_q].
         (* LHS remains the same *)
         remember (exist _ (Rcv p' q' v) H_x) as x.  
         assert (H_LHS :  mproj_rcv (w' ++ [x]) p q ++ get_channel_contents c p q =
                            mproj_rcv (w') p q ++ get_channel_contents c' p q).
         { unfold mproj_rcv; rewrite flat_map_app.
           rewrite <- app_assoc.
           assert (flat_map (mproj_rcv_symbol p' q') [x] = [v]). 
           destruct x. rewrite Heqx.
           eapply mproj_rcv_rcv. 
           easy. rewrite <- H_p in H. rewrite <- H_q in H.
           rewrite H. clear H. rewrite app_assoc. 
           destruct H_step as [H_snd | H_rcv].
           rewrite Heqx in H_snd; inversion H_snd. inversion H. 
           destruct H_rcv as [_ [H_unchanged [H_local [H_received H_chan]]]].
           simpl in *.
           destruct H_chan as [l' [H_chan_pre H_chan_post]].
           rewrite Heqx in H_chan_post. 
           simpl in H_chan_post.
           rewrite H_p. rewrite H_q. rewrite H_chan_post.
           rewrite Heqx in H_chan_pre.
           simpl in H_chan_pre.
           rewrite H_chan_pre.
           rewrite <- app_assoc. simpl.
           reflexivity. 
         } 
         (* RHS gets nothing *)
         assert (H_RHS :  mproj_snd (w' ++ [x]) p q =
                           mproj_snd (w') p q).
         { unfold mproj_snd; rewrite flat_map_app.
           assert (flat_map (mproj_snd_symbol p q) [x] = []). 
           destruct x. rewrite Heqx.
           rewrite H_p. rewrite H_q.
           eapply mproj_snd_rcv. easy.
           rewrite H. rewrite app_nil_r.
           reflexivity. }
         rewrite H_LHS H_RHS.
         assumption. 
      ** (* In case the receive action is not between p and q *) 
        (* Then both sides stay the same *)
        apply not_and_or in H.
        assert (H_inter : participant_eqb p' p && participant_eqb q' q = false).
        { apply andb_false_iff.
          destruct H.
          left.
          apply participant_eqb_no in H.
          rewrite participant_eqb_comm in H.  assumption.
          right.
          apply participant_eqb_no in H.
          rewrite participant_eqb_comm in H. assumption. }
        remember (exist _ (Rcv p' q' v) H_x) as x.  
        (* LHS gets nothing *)
        assert (H_LHS :  mproj_rcv (w' ++ [x]) p q ++ get_channel_contents c p q =
                            mproj_rcv (w') p q ++ get_channel_contents c' p q).
        { unfold mproj_rcv; rewrite flat_map_app.
          rewrite <- app_assoc.
          unfold mproj_rcv_symbol at 2; simpl.
          rewrite Heqx. rewrite H_inter. 
          clear H_inter. rewrite andb_false_l. simpl.
          destruct H_step as [H_snd | H_rcv].
          inversion H_snd. rewrite Heqx in H0. inversion H0.
          destruct H_rcv as [_ [H_unchanged [H_local [H_received H_chan]]]].
          simpl in *.
          spec H_received p q.
          rewrite Heqx in H_received.  
          simpl in H_received. 
          spec H_received H. rewrite H_received. reflexivity.
        } 
        (* RHS gets nothing *)
        assert (H_RHS :  mproj_snd (w' ++ [x]) p q =
                           mproj_snd (w') p q).
        { unfold mproj_snd; rewrite flat_map_app.
          unfold mproj_snd_symbol at 2; simpl.
          rewrite Heqx. rewrite H_inter. 
          rewrite andb_false_l.
          repeat rewrite app_nil_r. reflexivity. }
        rewrite H_LHS. rewrite H_RHS.
        assumption.
Qed. 

Lemma clts_trace_channel_compliant :
  forall {State : Type} {T : @CLTS State} (w : list AsyncAlphabet),
    @is_clts_trace State T w -> channel_compliant w. 
Proof. 
  intros State T w H.
  intros w' H_prefix p q H_neq.
  assert (H_trace := H). 
  destruct H as [c H_reach].
  assert (H_inter := @about_clts_trace_configuration_channel_contents State T).
  spec H_inter w'.
  assert (H_trace' := clts_trace_prefix_closed w H_trace w' H_prefix).
  destruct H_trace' as [c' H_reach'].
  spec H_inter c' H_reach' p q H_neq.
  rewrite <- H_inter. 
  apply prefix_app_r.
  reflexivity.
Qed.

Lemma clts_trace_participant_projection :
  forall {State : Type} {T : @CLTS State} (w : FinAsyncWord) (c : Configuration),
    @Reachable State T (c0 T) w c ->
    forall (p : participant),
      @lts.Reachable AsyncAlphabet State (implementations T p) (s0 (implementations T p)) (wproj w p)
        (get_local_state c p). 
Proof.
  intros State T w c H_reach p.
  generalize dependent c. 
  induction w as [|a w IHw] using rev_ind; intros.
  - (* Base case *)
    inversion H_reach.
    simpl.
    assert (H_c0 := about_c0 T).
    destruct H_c0 as [H_c0 _].
    spec H_c0 p.
    rewrite H_c0.
    apply lts.Reachable_refl.
    apply app_eq_nil in H.
    easy.
  - (* Inductive step *)
    apply Reachable_unwind in H_reach.
    destruct H_reach as [c_w [H_reach H_step]].
    spec IHw c_w H_reach.
    destruct a as [a H_neq].
    destruct a as [p' q' v' | p' q' v'].
    * (* In the case that p'!q':v' *)
      destruct (classic (p' = p)).
      ** (* In the case that p' is the active sender *)
        destruct H_step as [H_snd | H_rcv].
        destruct H_snd as [_ H_step].
        destruct H_step as [H_unchanged [H_state H_chan]].
        simpl in H_state.
        unfold wproj.
        rewrite flat_map_app.
        simpl. unfold wproj_symbol at 2.
        simpl.
        replace (participant_eqb p' p) with true.
        2: symmetry; 
        now apply participant_eqb_correct.
        simpl. apply lts.Reachable_step with (get_local_state c_w p).
        exact IHw.
        unfold is_local_transition in H_state.
        rewrite <- H. 
        exact H_state.
        destruct H_rcv as [H_false _].
        inversion H_false.
      ** (* In the case that p' is uninvolved *)
        destruct H_step as [H_snd | H_rcv].
        destruct H_snd as [_ H_step].
        destruct H_step as [H_unchanged [H_state H_chan]].
        unfold wproj; rewrite flat_map_app; simpl.
        rewrite app_nil_r.
        unfold wproj_symbol at 2; simpl.
        replace (participant_eqb p' p) with false.
        2 : symmetry; now apply participant_eqb_no.
        simpl.
        rewrite app_nil_r.
        spec H_unchanged p.
        simpl in H_unchanged. 
        spec H_unchanged. 
        intro. symmetry in H0. contradiction.
        rewrite <- H_unchanged.
        exact IHw.
        destruct H_rcv as [H_false _]; inversion H_false.
    * (* In the case that q'?p':v' *)
      destruct (classic (q' = p)).
      ** (* In the case that q' is the active receiver *)
        destruct H_step as [H_snd | H_rcv].
        destruct H_snd as [H_false _].
        inversion H_false.
        destruct H_rcv as [_ H_step].
        destruct H_step as [H_unchanged [H_state H_chan]].
        simpl in H_state.
        unfold wproj.
        rewrite flat_map_app.
        simpl. unfold wproj_symbol at 2.
        simpl.
        replace (participant_eqb q' p) with true.
        2: symmetry; 
        now apply participant_eqb_correct.
        simpl.
        apply lts.Reachable_step with (get_local_state c_w p).
        exact IHw.
        unfold is_local_transition in H_state.
        rewrite <- H. 
        exact H_state.
      ** (* In the case that q' is uninvolved *)
        destruct H_step as [H_snd | H_rcv].
        destruct H_snd as [H_false _].
        inversion H_false. 
        destruct H_rcv as [_ H_step].
        destruct H_step as [H_unchanged [H_state H_chan]].
        unfold wproj; rewrite flat_map_app; simpl.
        rewrite app_nil_r.
        unfold wproj_symbol at 2; simpl.
        replace (participant_eqb q' p) with false.
        2 : symmetry; now apply participant_eqb_no.
        rewrite app_nil_r.
        spec H_unchanged p.
        simpl in H_unchanged. 
        spec H_unchanged. 
        intro. symmetry in H0. contradiction.
        rewrite <- H_unchanged.
        exact IHw.
Qed.

Lemma about_clts_configuration_step :
  forall {State : Type} (T : @CLTS State) (w : FinAsyncWord) (x : AsyncAlphabet) (c1 c2 : Configuration),
    @Reachable State T (c0 T) w c1 -> 
    @Reachable State T (c0 T) (w ++ [x]) c2 ->
    @Configuration_step State T c1 x c2. 
Proof. 
  intros State T w x c1 c2 H_reach1 H_reach2. 
  inversion H_reach2.
  -  assert (w = []).
    { destruct w.  reflexivity.
      simpl in H.
      inversion H.
      symmetry in H4. 
      apply app_eq_nil in H4.
      destruct H4. inversion H4. }
    rewrite H1 in H.  
    simpl in H.
    inversion H.
    rewrite H1 in H_reach2.
    simpl in H_reach2.
    rewrite H1 in H_reach1. apply reach_epsilon_identity in H_reach1.
    rewrite <- H_reach1. rewrite <- H4. assumption.
  - symmetry in H1. apply app_eq_nil in H1.
    destruct H1. inversion H2.
  - apply app_inj_tail in H. destruct H.
    rewrite H in H0.
    assert (c1 = c4) by now eapply (deterministic_clts T) with w.
    rewrite <- H4. 
    now rewrite H5.
Qed.

Lemma clts_step_snd_means_local_transition :
  forall {State : Type} (T : @CLTS State) (w : FinAsyncWord) (p q : participant) (v : message)
    (H_pq : sender_receiver_neq_async (Snd p q v))
    (c1 c2 : Configuration),
    @Reachable State T (c0 T) w c1 -> 
    @Reachable State T (c0 T) (w ++ [exist _ (Snd p q v) H_pq]) c2 ->
    @transition AsyncAlphabet State (implementations T p) (get_local_state c1 p) (exist _ (Snd p q v) H_pq) (get_local_state c2 p). 
Proof.
  intros State T w p q v H_pq c1 c2 H_reach1 H_reach2.
  assert (H_config_step := about_clts_configuration_step T w (exist _ (Snd p q v) H_pq) c1 c2 H_reach1 H_reach2).
  destruct H_config_step. destruct H. destruct H0. destruct H1. assumption.
  destruct H. inversion H.
Qed.

Lemma clts_step_rcv_means_local_transition :
  forall {State : Type} (T : @CLTS State) (w : FinAsyncWord) (p q : participant) (v : message)
    (H_pq : sender_receiver_neq_async (Rcv p q v))
    (c1 c2 : Configuration),
    @Reachable State T (c0 T) w c1 -> 
    @Reachable State T (c0 T) (w ++ [exist _ (Rcv p q v) H_pq]) c2 ->
    @transition AsyncAlphabet State (implementations T q) (get_local_state c1 q) (exist _ (Rcv p q v) H_pq) (get_local_state c2 q). 
Proof.
  intros State T w p q v H_pq c1 c2 H_reach1 H_reach2.
  assert (H_config_step := about_clts_configuration_step T w (exist _ (Rcv p q v) H_pq) c1 c2 H_reach1 H_reach2).
  destruct H_config_step. destruct H. inversion H.
  destruct H. destruct H0. destruct H1.
  simpl in H1. assumption.
Qed.

Lemma clts_reachable_means_implementation_reachable :
  forall {State : Type} (T : @CLTS State) (w : FinAsyncWord) (c : Configuration),
    @Reachable State T (c0 T) w c ->
    forall (p : participant),
      @lts.Reachable AsyncAlphabet State (implementations T p) (s0 (implementations T p)) (wproj w p) (get_local_state c p). 
Proof.
  intros State T w c H_clts p.
  generalize dependent c. 
  induction w using rev_ind; intros c_wx H_clts. 
  - (* Base case *)
    assert (H_rewrite := about_c0 T).
    destruct H_rewrite as [H_rewrite _]. 
    spec H_rewrite p. rewrite H_rewrite.
    apply reach_epsilon_identity in H_clts.
    rewrite <- H_clts. apply lts.Reachable_refl. 
  - (* Inductive step *) 
    (* From the induction hypothesis, we know that (wproj w p) is a prefix in T_p *)
    (* Want to show using Reachable_step that (wproj wx p) is also a prefix in T_p *) 
    unfold wproj; rewrite flat_map_app; simpl; rewrite app_nil_r.  
    assert (H_reach_w : @is_clts_trace State T w).  
    { apply (clts_trace_prefix_closed_step w x).
      exists c_wx. assumption. }
    destruct H_reach_w as [c_w H_reach_w].
    spec IHw c_w H_reach_w. 
    (* If p is uninvolved in x, then we can use IHw right away *)
    destruct x as [x H_neq].
    destruct x as [r s v | r s v].  
    * (* In the case that x = r!s:v *) 
      destruct (classic (p = r)) as [H_p_eq_r | H_p_neq_r].
      ** (* In the case that p = r *)
        destruct (classic (p = s)) as [H_p_eq_s | H_p_neq_s].
        *** (* In the case that p = s *)
          (* Immediate contradiction *) 
          rewrite H_p_eq_r in H_p_eq_s. contradiction. 
        *** (* In the case that p != s *)
          (* Then p is the sender in Snd r s, and we obtain the post-state for p from c_wx *)
          (* First peeling out the x which p is involved in *) 
          unfold wproj_symbol at 2. 
          simpl.
          assert (H_p_eq_r' := H_p_eq_r). 
          symmetry in H_p_eq_r.  
          apply participant_eqb_correct in H_p_eq_r.
          apply participant_eqb_no in H_p_neq_s.
          rewrite participant_eqb_comm in H_p_neq_s.
          rewrite H_p_eq_r. simpl. 
          simpl.
          (* Now transforming the goal using Reachable_step *) 
          rewrite H_p_eq_r'.
          apply (lts.Reachable_step (implementations T r) (s0 (implementations T r)) (get_local_state c_w r) (get_local_state c_wx r) (wproj w r) (exist _ (Snd r s v) H_neq)).
          rewrite H_p_eq_r' in IHw. 
          assumption. 
          apply (clts_step_snd_means_local_transition T w r s); try assumption. 
      ** (* In the case that p != r *)
        destruct (classic (p = s)) as [H_p_eq_s | H_p_neq_s]. 
        *** (* In the case that p = s *)
          (* Then p is the receiver in Snd r s, and the pre- and post-state for p is the same *)
          (* First getting rid of x *) 
          unfold wproj_symbol at 2.  
          simpl.
          assert (H_p_neq_r' := H_p_neq_r). 
          apply participant_eqb_no in H_p_neq_r.
          rewrite participant_eqb_comm in H_p_neq_r. 
          rewrite H_p_neq_r. simpl.
          rewrite app_nil_r.
          (* Establishing that p's state in c_w and c_wx are equal *)
          assert (H_eq : get_local_state c_wx s = get_local_state c_w s).
          { assert (H_helper := about_clts_configuration_step T w (exist _ (Snd r s v) H_neq) c_w c_wx).
            spec H_helper H_reach_w H_clts.
            destruct H_helper as [H_send | H_receive].
            destruct H_send as [_ [H_unchanged [H_local H_channels]]].
            spec H_unchanged p.
            simpl in H_unchanged. spec H_unchanged H_p_neq_r'.
            symmetry. rewrite <- H_p_eq_s. assumption. 
            destruct H_receive as [H_false _].
            inversion H_false. }
          (* Finally invoking IHw *) 
          rewrite H_p_eq_s. rewrite H_eq. 
          rewrite <- H_p_eq_s. assumption.
        *** (* In the case that p != s *)
          (* Then p is completely uninvolved in Snd r s, and the pre- and post-state for p is the same *)
          (* First getting rid of x *)
          unfold wproj_symbol at 2.  
          simpl.
          assert (H_p_neq_r' := H_p_neq_r). 
          apply participant_eqb_no in H_p_neq_r.
          rewrite participant_eqb_comm in H_p_neq_r. 
          rewrite H_p_neq_r. simpl.
          rewrite app_nil_r.
          (* Establishing that p's state in c_w and c_wx are equal *)
          assert (H_eq : get_local_state c_wx p = get_local_state c_w p).
          { assert (H_helper := about_clts_configuration_step T w (exist _ (Snd r s v) H_neq) c_w c_wx).
            spec H_helper H_reach_w H_clts.
            destruct H_helper as [H_send | H_receive].
            destruct H_send as [_ [H_unchanged [H_local H_channels]]].
            spec H_unchanged p.
            simpl in H_unchanged. spec H_unchanged H_p_neq_r'.
            symmetry. assumption.
            destruct H_receive as [H_false _].
            inversion H_false. }
          (* Finally invoking IHw *) 
          rewrite H_eq. assumption. 
    * (* In the case that x = s?r:v *) 
      destruct (classic (p = r)) as [H_p_eq_r | H_p_neq_r].
      ** (* In the case that p = r *)
        destruct (classic (p = s)) as [H_p_eq_s | H_p_neq_s].
        *** (* In the case that p = s *)
          (* Immediate contradiction *) 
          rewrite H_p_eq_r in H_p_eq_s. contradiction. 
        *** (* In the case that p != s *)
          (* Then p is the sender in Rcv r s, and the pre- and post-state for p is the same *)
          (* First getting rid of x *)
          unfold wproj_symbol at 2.  
          simpl.
          assert (H_p_neq_s' := H_p_neq_s). 
          apply participant_eqb_no in H_p_neq_s.
          rewrite participant_eqb_comm in H_p_neq_s. 
          rewrite H_p_neq_s. simpl.
          rewrite app_nil_r.
          (* Establishing that p's state in c_w and c_wx are equal *)
          assert (H_eq : get_local_state c_wx p = get_local_state c_w p).
          { assert (H_helper := about_clts_configuration_step T w (exist _ (Rcv r s v) H_neq) c_w c_wx).
            spec H_helper H_reach_w H_clts.
            destruct H_helper as [H_send | H_receive].
            destruct H_send as [H_false _].
            inversion H_false.
            destruct H_receive as [_[H_unchanged [H_local H_channels]]].
            spec H_unchanged p.
            simpl in H_unchanged. spec H_unchanged H_p_neq_s'.
            symmetry. assumption.
          }
          (* Finally invoking IHw *) 
          rewrite H_eq. assumption.  
      ** (* In the case that p != r *)
        destruct (classic (p = s)) as [H_p_eq_s | H_p_neq_s]. 
        *** (* In the case that p = s *)
          (* Then p is the receiver in Rcv r s, and we obtain the post-state for p from c_wx  *)
          (* First peeling out x which p is involved in *)
          unfold wproj_symbol at 2. 
          simpl.
          assert (H_p_eq_s' := H_p_eq_s).
          symmetry in H_p_eq_s.  
          apply participant_eqb_correct in H_p_eq_s.
          rewrite H_p_eq_s.
          (* Now transforming the goal using Reachable_step *) 
          rewrite H_p_eq_s'.
          apply (lts.Reachable_step (implementations T s) (s0 (implementations T s)) (get_local_state c_w s) (get_local_state c_wx s) (wproj w s) (exist _ (Rcv r s v) H_neq)).
          rewrite H_p_eq_s' in IHw. 
          assumption. 
          apply (clts_step_rcv_means_local_transition T w r s); try assumption. 
        *** (* In the case that p != s *)
          (* Then p is uninvolved in Rcv r s, and the pre- and post-state for p is the same *)
          unfold wproj_symbol at 2.  
          simpl.
          assert (H_p_neq_r' := H_p_neq_r). 
          assert (H_p_neq_s' := H_p_neq_s).
          apply participant_eqb_no in H_p_neq_s.
          rewrite participant_eqb_comm in H_p_neq_s. 
          rewrite H_p_neq_s. simpl.
          rewrite app_nil_r.
          (* Establishing that p's state in c_w and c_wx are equal *)
          assert (H_eq : get_local_state c_wx p = get_local_state c_w p).
          { assert (H_helper := about_clts_configuration_step T w (exist _ (Rcv r s v) H_neq) c_w c_wx).
            spec H_helper H_reach_w H_clts.
            destruct H_helper as [H_send | H_receive].
            destruct H_send as [H_false _]. 
            inversion H_false.
            destruct H_receive as [_ [H_unchanged [H_local H_channels]]].
            spec H_unchanged p.
            simpl in H_unchanged. spec H_unchanged H_p_neq_s'.
            symmetry. assumption. }
          (* Finally invoking IHw *) 
          rewrite H_eq. assumption.
Qed.

Lemma clts_trace_snd_extension_sufficient :
  forall {State : Type} (T : @CLTS State) (w : FinAsyncWord) (p q : participant) (v : message)
    (H_pq : sender_receiver_neq_async (Snd p q v)),
    @is_clts_trace State T w -> 
    @lts.is_trace AsyncAlphabet State (implementations T p) (wproj w p ++ [exist _ (Snd p q v) H_pq]) -> 
    @is_clts_trace State T (w ++ [exist _ (Snd p q v) H_pq]).
Proof.
  intros State T w p q v H_pq H_trace H_local.
  (* First, obtaining a configuration for w *)
  destruct H_trace as [c_w H_trace_w].
  (* In the case that x = p!q:v *) 
  destruct H_local as [s_p H_reach_p]. 
  (* Constructing the post-configuration state map and channel contents *) 
  remember (fun (q : participant) => if participant_eqb q p then s_p else (get_local_state c_w q)) as s_wx.
  remember (fun (r s : participant) => if participant_eqb p r && participant_eqb q s then get_channel_contents c_w p q ++ [v] else get_channel_contents c_w r s) as xi_wx.
  exists (s_wx, xi_wx).
  apply (Reachable_step (c0 T) c_w (s_wx, xi_wx)).
  assumption.
  left.
  repeat split.
  (* Showing that all the conjuncts for Configuration_snd hold of this post-configuration *)
  ** (* Obligation: all states except p's are unchanged *)
    simpl. unfold states_unchanged_except.
    intros s H_neq_p.
    rewrite Heqs_wx.
    simpl.
    apply participant_eqb_no in H_neq_p.
    rewrite H_neq_p.
    reflexivity.
  ** (* Obligation: the CLTS transition is permitted by p's local transition *) 
    unfold is_local_transition. simpl. 
    apply (reachable_extension_transition (wproj w p)).
    apply (deterministic_implementations T p).
    eapply clts_reachable_means_implementation_reachable; try assumption. 
    rewrite Heqs_wx.
    replace (participant_eqb p p) with true.
    assumption. 
    symmetry; now apply participant_eqb_correct.
  ** (* Obligation: all channels except (p,q) are unchanged *)
    rewrite Heqxi_wx. simpl.
    intros p' q' H_neq_or.
    unfold get_channel_contents.
    destruct c_w as [s_w xi_w]. 
    replace (participant_eqb p p' && participant_eqb q q') with false.
    reflexivity.
    symmetry. apply andb_false_iff.
    destruct H_neq_or. left.
    rewrite participant_eqb_comm. now apply participant_eqb_no.
    right. rewrite participant_eqb_comm. now apply participant_eqb_no.
  ** (* Obligation: channel (p,q) is correctly updated *)
    simpl. rewrite Heqxi_wx.
    do 2 rewrite participant_eqb_refl. simpl.
    reflexivity.
Qed.

Lemma clts_trace_rcv_extension_sufficient :
  forall {State : Type} (T : @CLTS State) (w : FinAsyncWord) (p q : participant) (v : message)
    (H_pq : sender_receiver_neq_async (Snd p q v))
    (c_w : Configuration),
    @Reachable State T (c0 T) w c_w -> 
    @lts.is_trace AsyncAlphabet State (implementations T q) (wproj w q ++ [exist _ (Rcv p q v) H_pq]) ->
    (exists (rest : list message), get_channel_contents c_w p q = v :: rest) -> 
    @is_clts_trace State T (w ++ [exist _ (Rcv p q v) H_pq]).
Proof.
  intros State T w p q v H_pq c_w H_trace H_local H_behead.
  (* In the case that x = q?p:v, now the active role is q *)
  (* Obtaining the post-state for q after performing q?p:m *) 
  destruct H_local as [s_q H_reach].
  destruct H_behead as [rest H_chan_pq]. 
  (* Constructing the post-configuration state map and channel contents *) 
  remember (fun (p : participant) => if participant_eqb p q then s_q else (get_local_state c_w p)) as s_wx.
  remember (fun (r s : participant) => if participant_eqb p r && participant_eqb q s then rest else get_channel_contents c_w r s) as xi_wx.
  exists (s_wx, xi_wx).
  apply (Reachable_step (c0 T) c_w (s_wx, xi_wx)).
  assumption.
  right. repeat split.
  (* Showing that all the conjuncts for Configuration_rcv hold of this post-configuration *) 
  ** (* Obligation: all states except q's are unchanged *)
    (* Same as before *) 
    simpl. unfold states_unchanged_except.
    intros s H_neq_p.
    rewrite Heqs_wx.
    simpl.
    apply participant_eqb_no in H_neq_p.
    rewrite H_neq_p.
    reflexivity.
  ** (* Obligation: the CLTS transition is permitted by p's local transition *)
    (* Same as before *) 
    unfold is_local_transition. simpl. 
    apply (reachable_extension_transition (wproj w q)).
    apply (deterministic_implementations T q).
    eapply clts_reachable_means_implementation_reachable; try assumption. 
    rewrite Heqs_wx.
    replace (participant_eqb q q) with true.
    assumption. 
    symmetry; now apply participant_eqb_correct.
  ** (* Obligation: all channels except (p,q) are unchanged *)
    (* Same as before *) 
    rewrite Heqxi_wx. simpl.
    intros p' q' H_neq_or.
    unfold get_channel_contents.
    destruct c_w as [s_w xi_w]. 
    replace (participant_eqb p p' && participant_eqb q q') with false.
    reflexivity.
    symmetry. apply andb_false_iff.
    destruct H_neq_or. left.
    rewrite participant_eqb_comm. now apply participant_eqb_no.
    right. rewrite participant_eqb_comm. now apply participant_eqb_no.
  ** (* Obligation: channel (p,q) is correctly updated *)
    simpl. exists rest. split.
    easy. rewrite Heqxi_wx.
    do 2 rewrite participant_eqb_refl. simpl.
    reflexivity.
Qed.

Lemma clts_deadlock_free_trace_is_prefix :
  forall {State : Type} (T : @CLTS State) (w : FinAsyncWord),
    clts_deadlock_free T -> 
    @is_clts_trace State T w ->
    @is_clts_prefix State T w. 
Proof.
  intros State T w H_df H_trace.
  destruct H_trace as [c_w H_reach_w].
  spec H_df c_w w H_reach_w.
  destruct H_df as [[w_fin H_fin] | [w_inf H_inf]].
  left. exists w_fin.  assumption.
  right. exists w_inf. split. tauto.
  unfold prefix_inf in H_inf.
  tauto.
Qed. 
