Require Import List Classical Nat Logic Lia ExtensionalityFacts Streams stdpp.list. 
Import ListNotations. 
From Coq 
Require Import ssreflect. 
  
From CC
  Require Import lib structures lts clts protocol channel_complete.

(** Facts about channel compliance **)
(* Trivial fact about prefixes preserving channel compliance *) 
Lemma prefix_preserves_channel_compliance :
  forall (w1 w2 : FinAsyncWord),
    channel_compliant w2 ->
    prefix w1 w2 ->
    channel_compliant w1. 
Proof.
  intros w1 w2 H_cc H_pref.
  intros w' H_pref' p q H_neq.
  spec H_cc w'.
  spec H_cc.
  eapply PreOrder_Transitive.
  exact H_pref'. exact H_pref.
  spec H_cc p q H_neq.
  assumption.
Qed.

(* Extending a channel compliant word with a send action preserves channel compliance *)
Lemma snd_extension_preserves_channel_compliance :
  forall (w : FinAsyncWord),
    channel_compliant w ->
    forall (x : AsyncAlphabet),
      is_snd x -> 
      channel_compliant (w ++ [x]). 
Proof.
  intros w H_cc x H_rcv.
  intros w' H_prefix p q H_neq.
  apply prefix_app_tail_or in H_prefix.
  destruct H_prefix as [H_old | H_new].
  spec H_cc w' H_old p q H_neq. assumption.
  rewrite H_new.
  destruct x as [x H_neq_async].
  destruct x as [p' q' m' | p' q' m'].
  - (* In the send case *)
    destruct (classic (p' = p /\ q' = q)).
    ** (* In the case that p,q = p',q' *)
      destruct H as [H_eq_p H_eq_q].
      unfold mproj_rcv.
      rewrite flat_map_app.
      unfold mproj_rcv_symbol at 2. simpl.
      replace (participant_eqb p' p) with true.
      replace (participant_eqb q' q) with true.
      simpl.
      unfold mproj_snd.
      rewrite flat_map_app.
      unfold mproj_snd_symbol at 2. simpl.
      replace (participant_eqb p' p) with true.
      replace (participant_eqb q' q) with true.
      simpl.
      rewrite app_nil_r.
      spec H_cc w.
      assert (H_pre : prefix w w) by reflexivity.
      spec H_cc H_pre.
      spec H_cc p q H_neq.
      now apply prefix_app_r. 
      symmetry; now apply participant_eqb_correct.
      symmetry; now apply participant_eqb_correct.
      symmetry; now apply participant_eqb_correct.
      symmetry; now apply participant_eqb_correct.
    ** (* In the case that p,q <> p',q' *)
      unfold mproj_rcv.
      rewrite flat_map_app.
      unfold mproj_rcv_symbol at 2. simpl.
      replace (participant_eqb p' p && participant_eqb q' q) with false. 
      simpl.
      unfold mproj_snd.
      rewrite flat_map_app.
      unfold mproj_snd_symbol at 2. simpl.
      replace (participant_eqb p' p && participant_eqb q' q) with false. 
      simpl.
      rewrite app_nil_r.
      spec H_cc w.
      assert (H_pre : prefix w w) by reflexivity.
      spec H_cc H_pre.
      spec H_cc p q H_neq.  
      now apply prefix_app_r.  
      symmetry. apply andb_false_iff.
      apply not_and_or in H.
      destruct H. left.
      now apply participant_eqb_no.
      right.
      now apply participant_eqb_no.
      symmetry. apply andb_false_iff.
      apply not_and_or in H.
      destruct H. left.
      now apply participant_eqb_no.
      right.
      now apply participant_eqb_no.
  - (* In the send case, we find a contradiction immediately *) 
    inversion H_rcv.
Qed.

Lemma snd_extension_preserves_channel_compliance_hd :
  forall (w : FinAsyncWord),
    channel_compliant w ->
    forall (x : AsyncAlphabet),
      is_snd x ->
      wproj w (receiver_async x) = [] -> 
      channel_compliant (x :: w).  
Proof.
  intros w H_cc x H_snd H_nil.
  intros w' H_prefix p q H_neq.
  destruct w' as [|x' w'].
  { simpl. reflexivity. }
  assert (H_eq : x' = x). 
  {   inversion H_prefix. inversion H. easy. }
  assert (H_prefix_cons := H_prefix).
  apply prefix_cons_inv_2 in H_prefix_cons. 
  rewrite H_eq in H_prefix.
  rewrite H_eq.
  clear H_eq.
  destruct x as [x H_neq_async].
  destruct x as [p' q' m' | p' q' m'].
  2: inversion H_snd. 
  (* We only need to reason about the send case *)
  destruct (classic (p' = p /\ q' = q)).
  ** (* In the case that p,q = p',q' *)
    destruct H as [H_eq_p H_eq_q].
    subst. simpl. 
    rewrite mproj_rcv_snd_eq.
    rewrite app_nil_l. 
    rewrite mproj_snd_snd_eq.
    replace (mproj_rcv w' p q) with (mproj_rcv (wproj w' q) p q) by now rewrite <- mproj_rcv_wproj_idempotent.
    simpl in H_nil.
    rewrite (wproj_nil_means_prefix_wproj_nil w w' q); try assumption. 
    simpl. apply prefix_nil.
  ** (* In the case that p,q <> p',q' *)
    (* Then we can remove both individual events
       and appeal to the fact that prefixes preserve channel compliance *) 
    apply not_and_or in H.
    unfold mproj_rcv.
    simpl.
    rewrite mproj_rcv_snd_neq.
    rewrite app_nil_l.
    unfold mproj_snd_symbol; simpl. 
    replace (participant_eqb p' p && participant_eqb q' q) with false.
    rewrite andb_false_l. simpl.
    apply prefix_cons_inv_2 in H_prefix.
    assert (H_cc_w' := prefix_preserves_channel_compliance w' w H_cc H_prefix).
    spec H_cc_w' w'. spec H_cc_w'.
    reflexivity. spec H_cc_w' p q H_neq.
    assumption.
    symmetry. rewrite andb_false_iff.
    destruct H.
    left.
    now apply participant_eqb_no.
    right.
    now apply participant_eqb_no.
Qed. 

(* Trying to find a strong enough characterization about prefixes of split words *) 
Lemma split_word_prefix_snd_rcv :
  forall (w : FinSyncWord) (w' : FinAsyncWord),
    prefix w' (split w) ->
    forall (p q : participant),
      p <> q ->
      mproj_rcv w' p q = mproj_snd w' p q \/
        exists (m : message),
          mproj_rcv w' p q ++ [m] = mproj_snd w' p q. 
Proof.
  intro w.
  induction w as [|a ls IHw] using rev_ind; intros w' H_prefix p q H_neq.
  - inversion H_prefix. inversion H.
    symmetry in H1. apply app_nil in H1.
    destruct H1. rewrite H0.
    left. reflexivity.
  - unfold split in H_prefix.
    rewrite flat_map_app in H_prefix.
    destruct a as [[p' q' m'] H_neq_sync].
    unfold flat_map at 2 in H_prefix.
    simpl in H_prefix.
    unfold sender_sync, receiver_sync, value_sync in H_prefix.
    simpl in H_prefix.
    assert (H_rewrite : flat_map split_symbol ls ++
               [Snd p' q' m' ↾ H_neq_sync; Rcv p' q' m' ↾ H_neq_sync] =
      (flat_map split_symbol ls ++
         [Snd p' q' m' ↾ H_neq_sync] ++ [Rcv p' q' m' ↾ H_neq_sync])) by easy.
    rewrite H_rewrite in H_prefix.
    clear H_rewrite.
    assert (H_helper := @prefix_app_tail_or AsyncAlphabet).
    spec H_helper (flat_map split_symbol ls ++ [Snd p' q' m' ↾ H_neq_sync]) w'.
    (* Reminder to self: this weird trick works but inlining the construction in spec does not *) 
    remember (exist _ (Rcv p' q' m') H_neq_sync) as x_rcv. 
    spec H_helper x_rcv.
    rewrite app_assoc in H_prefix.
    spec H_helper H_prefix.
    destruct H_helper as [H_or1 | H_or2].
    apply prefix_app_tail_or in H_or1.
    destruct H_or1 as [H_or1 | H_or3].
    (* Now that we have all the cases *)
    * (* In the case that w' is a prefix of split ls *)
      (* Reuse the induction hypothesis directly *)
      spec IHw w' H_or1 p q H_neq.
      assumption. 
    * (* In the case that w' is split ls ++ p!q:m *) 
      rewrite H_or3. 
      (* Depending on whether p,q = p',q' *)
      (* We pick a different branch *) 
      destruct (classic (p' = p /\ q' = q)).
      ** (* In the case that p,q = p',q' *)
        right. 
        destruct H as [H_eq_p H_eq_q].
        unfold mproj_rcv.
        rewrite flat_map_app.
        unfold mproj_rcv_symbol at 2. simpl.
        replace (participant_eqb p' p) with true.
        replace (participant_eqb q' q) with true.
        simpl.
        unfold mproj_snd.
        rewrite flat_map_app.
        unfold mproj_snd_symbol at 2. simpl.
        replace (participant_eqb p' p) with true.
        replace (participant_eqb q' q) with true.
        simpl.
        rewrite app_nil_r.
        exists m'.
        assert (H_eq := split_word_channel_complete). 
        spec H_eq ls.
        unfold channel_complete in H_eq.
        spec H_eq p q H_neq.
        unfold mproj_rcv in H_eq. rewrite H_eq.
        reflexivity.
        symmetry; now apply participant_eqb_correct.
        symmetry; now apply participant_eqb_correct.
        symmetry; now apply participant_eqb_correct.
        symmetry; now apply participant_eqb_correct.
      ** (* In the case that p,q <> p',q' *)
        left.
        unfold mproj_rcv.
        rewrite flat_map_app.
        unfold mproj_rcv_symbol at 2. simpl.
        replace (participant_eqb p' p && participant_eqb q' q) with false.
        simpl.
        unfold mproj_snd.
        rewrite flat_map_app.
        unfold mproj_snd_symbol at 2. simpl.
        replace (participant_eqb p' p && participant_eqb q' q) with false.
        simpl.
        repeat rewrite app_nil_r.
        assert (H_eq := split_word_channel_complete). 
        spec H_eq ls.
        unfold channel_complete in H_eq.
        spec H_eq p q H_neq.
        unfold mproj_rcv in H_eq. rewrite H_eq.
        reflexivity.
        apply not_and_or in H. 
        destruct H.
        now rewrite participant_eqb_no.
        rewrite (participant_eqb_no q'). assumption.
        now rewrite andb_false_r.
        apply not_and_or in H. 
        destruct H.
        now rewrite participant_eqb_no.
        rewrite (participant_eqb_no q'). assumption.
        now rewrite andb_false_r.
    * (* In the case that w' is split ls ++ p!q:m ++ q?p:m *)
      rewrite H_or2. 
      (* Depending on whether p,q = p',q' *)
      (* We pick a different branch *) 
      destruct (classic (p' = p /\ q' = q)).
      ** (* In the case that p,q = p',q' *)
        destruct H as [H_eq_p H_eq_q].
        rewrite Heqx_rcv. 
        unfold mproj_rcv.
        repeat rewrite flat_map_app. 
        unfold mproj_rcv_symbol.
        simpl.
        replace (participant_eqb p' p && participant_eqb q' q) with true.
        simpl.
        unfold mproj_snd.
        repeat rewrite flat_map_app. 
        unfold mproj_snd_symbol. simpl.
        replace (participant_eqb p' p && participant_eqb q' q) with true.
        simpl.
        repeat rewrite app_nil_r.
        left. 
        assert (H_eq := split_word_channel_complete). 
        spec H_eq ls.
        unfold channel_complete in H_eq.
        spec H_eq p q H_neq.
        unfold mproj_rcv in H_eq. rewrite H_eq.
        reflexivity.
        symmetry; rewrite andb_true_iff.
        split; now apply participant_eqb_correct.
        symmetry; rewrite andb_true_iff. 
        split; now apply participant_eqb_correct.
      ** (* In the case that p,q <> p',q' *)
        left.
        rewrite Heqx_rcv. 
        unfold mproj_rcv.
        repeat rewrite flat_map_app.
        unfold mproj_rcv_symbol. simpl.
        replace (participant_eqb p' p && participant_eqb q' q) with false.
        simpl.
        unfold mproj_snd.
        repeat rewrite flat_map_app.
        unfold mproj_snd_symbol. simpl.
        replace (participant_eqb p' p && participant_eqb q' q) with false.
        simpl.
        repeat rewrite app_nil_r.
        assert (H_eq := split_word_channel_complete). 
        spec H_eq ls.
        unfold channel_complete in H_eq.
        spec H_eq p q H_neq.
        unfold mproj_rcv in H_eq. rewrite H_eq.
        reflexivity.
        apply not_and_or in H. 
        destruct H.
        symmetry; rewrite andb_false_iff.
        left; now rewrite participant_eqb_no.
        symmetry; rewrite andb_false_iff.
        right; now rewrite (participant_eqb_no q').
        apply not_and_or in H. 
        destruct H.
        symmetry; rewrite andb_false_iff.
        left; now rewrite participant_eqb_no.
        symmetry; rewrite andb_false_iff.
        right; now rewrite (participant_eqb_no q').
Qed. 

(* Showing that split synchronous words are channel compliant by construction *)
(* This requires on the disjunction established about the structure of
   prefixes of split synchronous words *) 
Lemma split_word_channel_compliant :
  forall (w : FinSyncWord),
    channel_compliant (split w). 
Proof.
  intros w w' H_prefix p q H_neq.
  assert (H_helper := split_word_prefix_snd_rcv).
  spec H_helper w w' H_prefix.
  spec H_helper p q H_neq.
  destruct H_helper as [H_eq | H_one_off].
  - rewrite H_eq. reflexivity.
  - destruct H_one_off as [m H_eq].
    rewrite <- H_eq.
    now apply prefix_app_r.
Qed.

Lemma split_word_app_channel_compliant_word_channel_compliant :
  forall (w : FinSyncWord) (w' : FinAsyncWord),
    channel_compliant w' -> 
    channel_compliant (split w ++ w'). 
Proof.
  intros w w' H_cc.
  intro w_pref. 
  intros H_pref p q H_neq.
  assert (H_cc_w : channel_complete (split w)) by apply split_word_channel_complete. 
  destruct (classic (prefix w_pref (split w))).
  - (* In the case that w_pref <= split w *)
    assert (H_helper := prefix_preserves_channel_compliance w_pref (split w)).
    spec H_helper.
    apply split_word_channel_compliant.
    spec H_helper H.
    spec H_helper w_pref.
    spec H_helper. reflexivity.
    now spec H_helper p q H_neq.
  - (* In the case that split w <= w_pref *)
    assert (H_helper := prefix_not_prefix_means_prefix (split w) w_pref w' []). 
    rewrite app_nil_r in H_helper.
    spec H_helper H_pref H.
    apply prefix_exists_suffix in H_helper.
    destruct H_helper as [w_suf H_split].
    rewrite H_split.
    rewrite mproj_rcv_app.
    rewrite mproj_snd_app.
    spec H_cc_w p q H_neq. 
    rewrite H_cc_w.
    (* Getting rid of the first segment *)
    apply prefix_app.
    assert (H_pref_w_suf : prefix w_suf w').
    { rewrite H_split in H_pref.
      apply prefix_app_inv in H_pref. assumption. }
    assert (H_cc_w_suf : channel_compliant w_suf).
    { now apply prefix_preserves_channel_compliance with w'. }
    spec H_cc_w_suf w_suf. spec H_cc_w_suf.
    reflexivity.
    now spec H_cc_w_suf p q H_neq.
Qed.

(* Important : what channel complete really desires to say is per-role identicality to some run *) 
(* As a definition, channel completeness by itself does not imply channel compliant! *)
(* So it is somewhat of a useless definition *) 
(* Rather the property that implies channel compliance is per-role-identicality to a split run *)
Lemma per_role_identical_word_app_split_word_channel_compliant :
  forall (w1 : FinAsyncWord) (rho1 : FinSyncWord) (rho2 : FinSyncWord),
    per_role_identical w1 (split rho1) ->
    channel_compliant w1 -> 
    channel_compliant (w1 ++ split rho2). 
Proof.
  intros w1 rho1 rho2 H_role H_cc. 
  intro w_pref. 
  intros H_pref p q H_neq.
  assert (H_cc_rho1 : channel_complete (split rho1)) by apply split_word_channel_complete. 
  assert (H_cc_rho2' : channel_compliant (split rho2)) by apply split_word_channel_compliant. 
  destruct (classic (prefix w_pref w1)).
  - spec H_cc w_pref H p q H_neq.
    assumption.
  - assert (H_or := prefix_weak_total w_pref w1 (w1 ++ split rho2)).
    spec H_or H_pref. spec H_or. now apply prefix_app_r.
    destruct H_or. contradiction.
    apply prefix_exists_suffix in H0.
    destruct H0 as [w_suf H_split_w_pref].
    rewrite H_split_w_pref.
    clean_mproj.
    rewrite mproj_rcv_wproj_idempotent.
    rewrite mproj_snd_wproj_idempotent.
    rewrite (H_role p).
    rewrite (H_role q).
    rewrite <- mproj_rcv_wproj_idempotent.
    rewrite <- mproj_snd_wproj_idempotent.
    replace (mproj_rcv (split rho1) p q) with (mproj_snd (split rho1) p q).
    2 : { spec H_cc_rho1 p q H_neq. now symmetry. }
    assert (H2 : prefix w_suf (split rho2)).
    { rewrite H_split_w_pref in H_pref.
      apply prefix_app_inv in H_pref. assumption. }
    apply prefix_app.
    eapply prefix_preserves_channel_compliance.
    exact H_cc_rho2'. exact H2. reflexivity. easy.
Qed. 
                                                                             
(* This proof is utterly trivialized by the new language definition *) 
Lemma protocol_prefix_channel_compliant :
  forall {State : Type} (S : @LTS SyncAlphabet State) (w : FinAsyncWord),
    is_protocol_prefix S w ->
    channel_compliant w. 
Proof.
  intros State S w H_prefix.
  destruct H_prefix as [H_fin | H_inf].
  - destruct H_fin as [w_fin [H_fin H_pref]].
    destruct H_fin as [rho [H_max [H_role H_cc]]].
    now apply prefix_preserves_channel_compliance with w_fin.
  - destruct H_inf as [w_inf [H_inf [i H_pref]]].
    spec H_inf i.
    destruct H_inf as [rho [v [H_trace [H_role H_cc]]]].  
    apply prefix_preserves_channel_compliance with (stream_to_list w_inf i ++ v).
    assumption.
    rewrite H_pref.
    now apply prefix_app_r.
Qed.
(* Nicely all the streams and lists business is ignored by the proof here *)
(* Wow! *)

Lemma snd_rcv_extension_preserves_channel_compliance :
  forall (w : FinAsyncWord),
    channel_compliant w ->
    forall (r q : participant) (m' : message)
      (H_rq_async : sender_receiver_neq_async (Snd r q m'))
      (H_rq_async' : sender_receiver_neq_async (Rcv r q m')),
      wproj w q = [] -> 
      channel_compliant ((exist _ (Snd r q m') H_rq_async) :: w ++ [exist _ (Rcv r q m') H_rq_async']).  
Proof.
  intros w H_cc r q m' H_rq_async H_rq_async' H_nil.
  intros w' H_prefix p' q' H_neq.
  destruct w' as [|x' w'].
  { simpl. reflexivity. }
  assert (H_eq : x' = (Snd r q m' ↾ H_rq_async)). 
  { inversion H_prefix. inversion H. easy. }
  assert (H_prefix_cons := H_prefix).
  apply prefix_cons_inv_2 in H_prefix_cons. 
  rewrite H_eq in H_prefix.
  rewrite H_eq.
  clear H_eq.
  destruct (classic (p' = r /\ q' = q)). 
  ** (* In the case that r,q = p',q' *)
    destruct H as [H_eq_p H_eq_q].
    subst. simpl.
    (* The LHS is [] *)
    rewrite mproj_rcv_snd_eq.
    rewrite app_nil_l. 
    apply prefix_app_tail_or in H_prefix_cons.
    destruct H_prefix_cons as [H_prefix_w' | H_eq_w']. 
    *** (* When w' <= w *)
      (* Then the LHS is [] *) 
      assert (H_rewrite : mproj_rcv w' r q = []).
      { assert (H_w'_nil : wproj w' q = []).
        { eapply wproj_nil_means_prefix_wproj_nil.
          exact H_nil. exact H_prefix_w'. }
        eapply (wproj_nil_means_mproj_rcv_nil w' q).
        exact H_w'_nil.
        exact H_neq.
      }
      rewrite H_rewrite.
      apply prefix_nil.
    *** (* When w' = w ++ Rcv r q m' *)
      rewrite H_eq_w'.
      rewrite mproj_rcv_app.
      assert (H_rewrite : mproj_rcv w r q = []).
      {
        eapply (wproj_nil_means_mproj_rcv_nil w q).
        exact H_nil. 
        exact H_neq.
      }
      rewrite H_rewrite.
      unfold mproj_rcv.
      simpl.
      rewrite mproj_rcv_rcv_eq. 
      rewrite app_nil_r. 
      rewrite mproj_snd_snd_eq.
      now apply prefix_app_r.
  ** (* In the case that r,q <> p', q' *) 
    simpl.
    rewrite (mproj_rcv_snd_neq r q p' q' m' H_rq_async).
    rewrite app_nil_l.
    unfold mproj_snd_symbol.
    simpl.
    replace (participant_eqb r p' && participant_eqb q q') with false.
    2 : { 
      symmetry. apply andb_false_iff.
      apply not_and_or in H.
      destruct H.
      left. rewrite participant_eqb_comm.
      now rewrite participant_eqb_no.
      right. rewrite participant_eqb_comm.
      now rewrite participant_eqb_no. }
    unfold is_sndb; simpl.
    apply prefix_app_tail_or in H_prefix_cons.
    destruct H_prefix_cons as [H_prefix_w' | H_eq_w']. 
    *** (* When w' <= w *)
      (* Then we appeal to the channel compliance of w' *) 
      assert (H_cc_w' : channel_compliant w') by now apply (prefix_preserves_channel_compliance w' w).
      spec H_cc_w' w'.
      spec H_cc_w'. reflexivity.
      spec H_cc_w' p' q' H_neq.
      assumption.
    *** (* When w' = w ++ Rcv r q m' *)
      (* Then we appeal to the channel compliance of w *) 
      rewrite H_eq_w'.
      rewrite mproj_rcv_app.
      simpl. 
      unfold mproj_rcv_symbol.
      simpl.
      replace (participant_eqb r p' && participant_eqb q q') with false.
      simpl. rewrite app_nil_r.
      rewrite mproj_snd_app.
      spec H_cc w.
      spec H_cc. reflexivity.
      spec H_cc p' q' H_neq.
      now apply prefix_app_r.
      symmetry.
      apply andb_false_iff.
      apply not_and_or in H.
      destruct H.
      left. rewrite participant_eqb_comm.
      now rewrite participant_eqb_no.
      right. rewrite participant_eqb_comm.
      now rewrite participant_eqb_no.
Qed. 

Lemma frankenstein_channel_compliant :
  forall (w_pref w_suf : FinAsyncWord)
    (r q : participant) (m' : message)
    (H_rq_async : sender_receiver_neq_async (Snd r q m'))
    (H_rq_async' : sender_receiver_neq_async (Rcv r q m'))
    (H_cc_w : channel_compliant (w_pref ++ w_suf))
    (H_nil : wproj w_pref q = []),  
    channel_compliant ([Snd r q m' ↾ H_rq_async] ++ w_pref ++ [Rcv r q m' ↾ H_rq_async'] ++ w_suf).
Proof.
  intros.  
  intros w' H_prefix p' q' H_neq.
  destruct w' as [|x' w'].
  { simpl. reflexivity. }
  assert (H_eq : x' = (Snd r q m' ↾ H_rq_async)). 
  { inversion H_prefix. inversion H. easy. }
  assert (H_prefix_cons := H_prefix).
  apply prefix_cons_inv_2 in H_prefix_cons. 
  rewrite H_eq in H_prefix.
  rewrite H_eq.
  clear H_eq. 
  destruct (classic (p' = r /\ q' = q)). 
  ** (* In the case that r,q = p',q' *)
    destruct H as [H_eq_p H_eq_q].
    subst. simpl.
    (* The LHS is [] *)
    rewrite mproj_rcv_snd_eq.
    rewrite app_nil_l. 
    (* Now we can't apply prefix_app_tail_or to H_prefix_cons *)
    (* Instead we just do case analysis on whether w' <= w_pref ++ [Rcv r q m' ↾ H_rq_async'] *)
    destruct (classic (prefix w' (w_pref ++ [Rcv r q m' ↾ H_rq_async']))) as [H_yes | H_no].
    *** (* In the case that w' <= w_pref ++ [Rcv r q m' ↾ H_rq_async'] *)
      (* Proceed as before *)
      apply prefix_app_tail_or in H_yes.
      destruct H_yes as [H_prefix_w' | H_eq_w']. 
      **** (* When w' <= w *)
        (* Then the LHS is [] *) 
        assert (H_rewrite : mproj_rcv w' r q = []).
        { assert (H_w'_nil : wproj w' q = []).
          { eapply wproj_nil_means_prefix_wproj_nil.
            exact H_nil. exact H_prefix_w'. } 
          eapply (wproj_nil_means_mproj_rcv_nil w' q).
          exact H_w'_nil.
          exact H_neq.
        }
        rewrite H_rewrite.
        apply prefix_nil.
      **** (* When w' = w_pref ++ [Rcv r q m' ↾ H_rq_async'] *) 
        rewrite H_eq_w'.
        rewrite mproj_rcv_app.
        assert (H_rewrite : mproj_rcv w_pref r q = []).
        {
          eapply (wproj_nil_means_mproj_rcv_nil w_pref q).
          exact H_nil. exact H_neq. 
        }
        rewrite H_rewrite.
        rewrite app_nil_l. 
        simpl. rewrite mproj_rcv_rcv_eq. 
        rewrite app_nil_r. 
        rewrite mproj_snd_snd_eq.
        now apply prefix_app_r.
    *** (* In the case that w' is not a prefix of w_pref ++ [Rcv r q m' ↾ H_rq_async'] *)
      (* Then the opposite must be true: w_pref ++ [Rcv r q m' ↾ H_rq_async'] <= w' *)
      assert (H_useful := prefix_weak_total w' (w_pref ++ [Rcv r q m' ↾ H_rq_async'])  (w_pref ++ [Rcv r q m' ↾ H_rq_async'] ++ w_suf) H_prefix_cons).
      spec H_useful. rewrite app_assoc. now apply prefix_app_r.
      destruct H_useful.
      contradiction.
      (* Having established that w_pref ++ [Rcv r q m' ↾ H_rq_async'] <= w' in H *)
      apply prefix_exists_suffix in H.
      destruct H as [w'_suf H_split_w'].
      rewrite <- app_assoc in H_split_w'.
      (* Now we have split up w' into three parts *)
      (* Now we can piecewise reason about mproj *)  
      rewrite H_split_w'.
      repeat rewrite mproj_rcv_app.
      repeat rewrite mproj_snd_app.
      simpl.
      rewrite mproj_rcv_rcv_eq.
      rewrite app_nil_r.
      rewrite mproj_snd_snd_eq.
      rewrite app_nil_r.
      rewrite mproj_snd_rcv_eq.
      rewrite app_nil_l.
      assert (H_rewrite : mproj_rcv w_pref r q = []).
      {
        eapply (wproj_nil_means_mproj_rcv_nil w_pref q).
        exact H_nil. exact H_neq. 
      } 
      rewrite H_rewrite.
      rewrite app_nil_l. 
      apply prefix_app.
      (* Now we want to reintroduce H_rewrite *)
      rewrite <- (app_nil_l (mproj_rcv w'_suf r q)).
      rewrite <- H_rewrite.
      rewrite <- mproj_rcv_app.
      rewrite <- mproj_snd_app.
      assert (H_w'_suf_pref : prefix w'_suf w_suf). 
      {
        rewrite H_split_w' in H_prefix.
        apply prefix_app_inv with ([Snd r q m' ↾ H_rq_async] ++ w_pref ++ [Rcv r q m' ↾ H_rq_async']).
        repeat rewrite <- app_assoc.
        exact H_prefix. }
      assert (H_cc' : channel_compliant (w_pref ++ w'_suf)).
      { apply prefix_preserves_channel_compliance with (w_pref ++ w_suf).
        exact H_cc_w.
        apply prefix_app.
        exact H_w'_suf_pref. }
      spec H_cc' (w_pref ++ w'_suf).
      spec H_cc'. reflexivity.
      now spec H_cc' r q H_neq.
  ** (* In the case that r,q <> p',q' *)
    (* Establishing this rewrite *) 
    assert (H_rewrite : (participant_eqb r p' && participant_eqb q q') = false). 
    { rewrite andb_false_iff.
      apply not_and_or in H.
      destruct H. left.
      rewrite participant_eqb_comm.
      now rewrite participant_eqb_no.
      right.
      rewrite participant_eqb_comm.
      now rewrite participant_eqb_no. }
    simpl.
    rewrite mproj_rcv_snd_neq.
    rewrite app_nil_l.
    unfold mproj_snd_symbol. simpl.
    rewrite H_rewrite. 
    simpl.
destruct (classic (prefix w' (w_pref ++ [Rcv r q m' ↾ H_rq_async']))) as [H_yes | H_no].
    *** (* In the case that w' <= w_pref ++ [Rcv r q m' ↾ H_rq_async'] *)
      (* Proceed as before *)
      (* Establishing that w_pref is channel compliant *) 
      assert (H_cc_w_pref : channel_compliant w_pref).
      {
        eapply prefix_preserves_channel_compliance.
        exact H_cc_w.
        now apply prefix_app_r.
      }
      apply prefix_app_tail_or in H_yes.
      destruct H_yes as [H_prefix_w' | H_eq_w']. 
      **** (* When w' <= w_pref *)
        (* Then we appeal to the channel compliance of w_pref *)
        now spec H_cc_w_pref w' H_prefix_w' p' q' H_neq.
      **** (* When w' = wpref ++ [Rcv r q m' ↾ H_rq_async'] *)
        rewrite H_eq_w'.
        rewrite mproj_rcv_app.
        rewrite mproj_snd_app.
        simpl.
        unfold mproj_rcv_symbol.
        simpl.
        rewrite H_rewrite. rewrite app_nil_r.
        apply prefix_app_r.
        spec H_cc_w_pref w_pref.
        spec H_cc_w_pref. reflexivity.
        rewrite app_nil_r.
        now spec H_cc_w_pref p' q' H_neq.
    *** (* In the case that w_pref ++ [Rcv r q m' ↾ H_rq_async'] <= w' *)
      (* Then the opposite must be true: w_pref ++ [Rcv r q m' ↾ H_rq_async'] <= w' *) 
      assert (H_useful := prefix_weak_total w' (w_pref ++ [Rcv r q m' ↾ H_rq_async'])  (w_pref ++ [Rcv r q m' ↾ H_rq_async'] ++ w_suf) H_prefix_cons).
      spec H_useful. rewrite app_assoc. now apply prefix_app_r.
      destruct H_useful.
      contradiction.
      (* Having established that w_pref ++ [Rcv r q m' ↾ H_rq_async'] <= w' in H *)
      apply prefix_exists_suffix in H0.
      destruct H0 as [w'_suf H_split_w'].
      rewrite <- app_assoc in H_split_w'.
      (* Now we have split up w' into three parts *)
      (* Now we can piecewise reason about mproj *) 
      rewrite H_split_w'.
      repeat rewrite mproj_rcv_app.
      repeat rewrite mproj_snd_app.
      simpl.
      (* Which consists of first removing the middle symbols *)
      unfold mproj_rcv_symbol.
      simpl. rewrite H_rewrite.
      simpl.
      rewrite mproj_snd_rcv_neq. 
      do 2 rewrite app_nil_l.
      (* Boils down to showing that w_pref ++ w'_suf is channel_compliant *)
      assert (H_cc_goal : channel_compliant (w_pref ++ w'_suf)). 
      { 
        apply prefix_preserves_channel_compliance with (w_pref ++ w_suf).
        exact H_cc_w.
        apply prefix_app.
        (* Which further boils down to showing that w'_suf <= w_suf *)
        rewrite H_split_w' in H_prefix_cons.
        apply prefix_app_inv in H_prefix_cons.
        now apply prefix_app_inv in H_prefix_cons. }
      rewrite <- mproj_rcv_app.
      rewrite <- mproj_snd_app.
      spec H_cc_goal (w_pref ++ w'_suf). spec H_cc_goal.
      reflexivity. 
      now spec H_cc_goal p' q' H_neq.
Qed.

(* Wow this is actually provable *) 
Lemma split_run_snd_channel_compliant_app_channel_compliant : 
  forall (w : FinAsyncWord) 
    (q r : participant)
    (m' : message)
    (H_rq_async : sender_receiver_neq_async (Snd r q m'))
    (rho1 : FinSyncWord),
    channel_compliant w ->
    wproj w q = [] -> 
    channel_compliant (split rho1 ++ [Snd r q m' ↾ H_rq_async] ++ w). 
Proof.   
  intros w q r m' H_rq_async rho1 H_cc_w H_empty. 
  intros w0 H_pref0 p0 q0 H_neq0.
  assert (H_cc_rho1 : channel_compliant (split rho1)) by apply split_word_channel_compliant. 
  assert (H_cc_rho1' : channel_complete (split rho1)) by apply split_word_channel_complete.
  assert (H_case := case_prefix_app_cons_app _ w0 (split rho1) w (Snd r q m' ↾ H_rq_async) H_pref0). 
  destruct H_case as [H_pref | [H_eq | H_pref]]. 
  - spec H_cc_rho1 w0 H_pref p0 q0 H_neq0. 
    assumption. 
  - subst w0.
    rewrite mproj_rcv_app mproj_snd_app.
    replace (mproj_rcv (split rho1) p0 q0) with (mproj_snd (split rho1) p0 q0).
    2 : { symmetry. now spec H_cc_rho1' p0 q0 H_neq0. }
    apply prefix_app.
    destruct (classic (r = p0 /\ q = q0)). 
    * destruct H.
      subst p0 q0.
      clean_mproj.
      apply prefix_nil.
    * clean_mproj. 
  -  apply prefix_exists_suffix in H_pref.
     destruct H_pref as [w0_suf H_split_w0]. 
     assert (H_about_w0_suf : prefix w0_suf w). 
      {
        rewrite H_split_w0 in H_pref0.
        rewrite <- app_assoc in H_pref0.
        apply prefix_app_inv in H_pref0.
        clean H_pref0. 
        now apply prefix_cons_inv_2 in H_pref0.
      }
      rewrite H_split_w0.
      rewrite <- app_assoc. 
      rewrite mproj_rcv_app mproj_snd_app.
      replace (mproj_rcv (split rho1) p0 q0) with (mproj_snd (split rho1) p0 q0).
      2 : { symmetry. now spec H_cc_rho1' p0 q0 H_neq0. }
      apply prefix_app.
      destruct (classic (r = p0 /\ q = q0)).
     * destruct H.
       subst p0 q0.
       clean_mproj.
       apply (wproj_preserves_prefix _ _ q) in H_about_w0_suf.
       rewrite H_empty in H_about_w0_suf.
       apply prefix_nil_inv in H_about_w0_suf.
       rewrite mproj_rcv_wproj_idempotent.
       rewrite H_about_w0_suf. simpl. apply prefix_nil.
    * clean_mproj.  
      spec H_cc_w w0_suf H_about_w0_suf p0 q0 H_neq0.
      assumption.
Qed.


Lemma channel_compliant_fastforward_rcv_suffix :
  forall (p q : participant)
    (m : message)
    (H_neq : sender_receiver_neq_async (Rcv p q m))
    (w : list AsyncAlphabet)
    (u1 : FinAsyncWord)
    (u2 : list AsyncAlphabet)
    (H_empty_u1 : wproj u1 q = [])
    (H_w_pq : mproj_rcv w p q ++ [m]
           `prefix_of` mproj_snd w p q)
    (H_cc_u : channel_compliant (w ++ u1 ++ Rcv p q m ↾ H_neq :: u2)),
    channel_compliant ((w ++ [Rcv p q m ↾ H_neq]) ++ u1 ++ u2).
Proof.
  intros.
  intros w0 H_pref0 p0 q0 H_neq0.
  rewrite <- app_assoc in H_pref0.
  assert (H_case := case_prefix_app_cons_app _ w0 w (u1 ++ u2) (Rcv p q m ↾ H_neq) H_pref0).
  assert (H_cc_w : channel_compliant w).
    { eapply prefix_preserves_channel_compliance.
      exact H_cc_u.
      repeat apply prefix_app_r.
      reflexivity.
    }
  destruct H_case as [H_pref | [H_eq | H_pref]]. 
  - 
    spec H_cc_w w0 H_pref p0 q0 H_neq0. 
    assumption. 
  - subst w0.
    rewrite mproj_rcv_app mproj_snd_app.
    destruct (classic (p = p0 /\ q = q0)). 
    * destruct H.
      subst p0 q0.
      clean_mproj. 
      assumption. 
    * clean_mproj. 
      spec H_cc_w w.
      spec H_cc_w. reflexivity.
      spec H_cc_w p0 q0 H_neq0. assumption.
  -  apply prefix_exists_suffix in H_pref.
     destruct H_pref as [w0_suf H_split_w0]. 
     rewrite <- app_assoc in H_split_w0.
     rewrite H_split_w0.
     assert (H_about_w0_suf : prefix w0_suf (u1 ++ u2)). 
     {
       rewrite H_split_w0 in H_pref0.
       now repeat apply prefix_app_inv in H_pref0.
     }
     destruct (classic (p = p0 /\ q = q0)).
     * destruct H. subst p0 q0.
       (* So if w0_suf <= u1 then this is easy *)
       (* But if u1 <= w0_suf, and w0_suf contains some bits from u2 *)
       (* Then...how? *)
       (* Then we should be able to appeal to the channel compliance of the original word, from H_cc_u *)
       (* Now if w0_suf onto p were conveniently empty, this would be easy *)
       assert (H_case := prefix_weak_total u1 w0_suf (u1 ++ u2)).
       spec H_case. now apply prefix_app_r.
       spec H_case H_about_w0_suf.
       destruct H_case as [H_hard | H_easy]. 
       ** apply prefix_exists_suffix in H_hard.
          destruct H_hard as [u2_part H_split_w0_suf].
          assert (H_about_u2_part : prefix u2_part u2).
          { rewrite H_split_w0_suf in H_about_w0_suf. 
            now apply prefix_app_inv in H_about_w0_suf.
          }
          (* Hopefully this works *) 
          spec H_cc_u (w ++ u1 ++ [Rcv p q m ↾ H_neq] ++ u2_part).
          spec H_cc_u.
          repeat apply prefix_app. simpl.
          apply prefix_cons. assumption.
          spec H_cc_u p q H_neq0.
          assert (H_eq_q : wproj w0 q = wproj (w ++ u1 ++ [Rcv p q m ↾ H_neq] ++ w0_suf) q).
          {
            rewrite H_split_w0.
            clean. apply app_inv_head_iff.
            rewrite H_empty_u1.
            easy.
          }
          rewrite <- H_split_w0 at 1. 
          rewrite mproj_rcv_wproj_idempotent.
          rewrite H_eq_q.
          rewrite H_split_w0_suf.
          assert (H_eq_p : mproj_snd (w ++ u1 ++ [Rcv p q m ↾ H_neq] ++ u2_part) p q = mproj_snd (w ++ [Rcv p q m ↾ H_neq] ++ u1 ++ u2_part) p q).
          {
            clean_mproj. (* Wow! *) 
          }
          rewrite H_eq_p in H_cc_u.
          rewrite <- mproj_rcv_wproj_idempotent.
          assert (H_eq_q' : mproj_rcv (w ++ u1 ++ [Rcv p q m ↾ H_neq] ++ u1 ++ u2_part) p q = mproj_rcv (w ++ u1 ++ [Rcv p q m ↾ H_neq] ++ u2_part) p q).
          {
            clean_mproj.
            repeat apply app_inv_head_iff.
            rewrite mproj_rcv_wproj_idempotent.
            rewrite H_empty_u1.
            simpl. reflexivity.
          }
          rewrite H_eq_q'.
          assumption. (* Triumph! *) 
       ** assert (H_about_w0_suf' : mproj_rcv w0_suf p q = []).
          {
            apply (wproj_preserves_prefix _ _ q) in H_easy.
            rewrite H_empty_u1 in H_easy.
            apply prefix_nil_inv in H_easy.
            rewrite mproj_rcv_wproj_idempotent.
            rewrite H_easy.
            reflexivity.
          }
          clean_mproj.
          rewrite H_about_w0_suf'.
          now apply prefix_app_r. 
     * (* In this case, nothing changes *)
       clean_mproj. (* Gets rid of the irrelevant action magically! *)
       (* Although we still need to do case analysis on the status of w0_suf because how we instantiate H_cc_u will depend on it *)
       assert (H_case := prefix_weak_total u1 w0_suf (u1 ++ u2)).
       spec H_case. now apply prefix_app_r.
       spec H_case H_about_w0_suf.
       destruct H_case as [H_hard | H_easy]. 
       ** apply prefix_exists_suffix in H_hard.
          destruct H_hard as [u2_part H_split_w0_suf].
          assert (H_about_u2_part : prefix u2_part u2).
          { rewrite H_split_w0_suf in H_about_w0_suf. 
            now apply prefix_app_inv in H_about_w0_suf.
          }
          (* Hopefully this works *) 
          spec H_cc_u (w ++ u1 ++ [Rcv p q m ↾ H_neq] ++ u2_part).
          spec H_cc_u.
          repeat apply prefix_app. simpl.
          apply prefix_cons. assumption.
          spec H_cc_u p0 q0 H_neq0.
          clean_mproj H_cc_u.
          rewrite H_split_w0_suf.
          clean_mproj.
       ** spec H_cc_u (w ++ w0_suf). 
          spec H_cc_u.
          repeat apply prefix_app.
          now apply prefix_app_r. 
          spec H_cc_u p0 q0 H_neq0.
          clean_mproj H_cc_u.
Qed.

Lemma channel_compliant_fastforward_snd_suffix :
  forall (p q : participant)
    (m : message)
    (H_neq : sender_receiver_neq_async (Snd p q m))
    (w : list AsyncAlphabet)
    (u1 : FinAsyncWord)
    (u2 : list AsyncAlphabet)
    (H_empty_u1 : wproj u1 p = [])
    (H_cc_u : channel_compliant (w ++ u1 ++ Snd p q m ↾ H_neq :: u2)),
    channel_compliant ((w ++ [Snd p q m ↾ H_neq]) ++ u1 ++ u2).
Proof.
  intros. 
  intros w0 H_pref0 p0 q0 H_neq0.
  rewrite <- app_assoc in H_pref0.
  assert (H_case := case_prefix_app_cons_app _ w0 w (u1 ++ u2) (Snd p q m ↾ H_neq) H_pref0).
  assert (H_cc_w : channel_compliant w).
    { eapply prefix_preserves_channel_compliance.
      exact H_cc_u.
      repeat apply prefix_app_r.
      reflexivity.
    }
  destruct H_case as [H_pref | [H_eq | H_pref]]. 
  - spec H_cc_w w0 H_pref p0 q0 H_neq0. 
    assumption. 
  - subst w0.
    rewrite mproj_rcv_app mproj_snd_app.
    destruct (classic (p = p0 /\ q = q0)). 
    * destruct H.
      subst p0 q0.
      clean_mproj. 
      spec H_cc_w w. spec H_cc_w.
      reflexivity. spec H_cc_w p q H_neq.
      now apply prefix_app_r. 
    * clean_mproj. 
      spec H_cc_w w.
      spec H_cc_w. reflexivity.
      spec H_cc_w p0 q0 H_neq0. assumption.
  -  apply prefix_exists_suffix in H_pref.
     destruct H_pref as [w0_suf H_split_w0]. 
     rewrite <- app_assoc in H_split_w0.
     rewrite H_split_w0.
     assert (H_about_w0_suf : prefix w0_suf (u1 ++ u2)). 
     {
       rewrite H_split_w0 in H_pref0.
       now repeat apply prefix_app_inv in H_pref0.
     }
     destruct (classic (p = p0 /\ q = q0)).
     * destruct H. subst p0 q0.
       (* So if w0_suf <= u1 then this is easy *)
       (* But if u1 <= w0_suf, and w0_suf contains some bits from u2 *)
       (* Then...how? *)
       (* Then we should be able to appeal to the channel compliance of the original word, from H_cc_u *)
       (* Now if w0_suf onto p were conveniently empty, this would be easy *)
       assert (H_case := prefix_weak_total u1 w0_suf (u1 ++ u2)).
       spec H_case. now apply prefix_app_r.
       spec H_case H_about_w0_suf.
       destruct H_case as [H_hard | H_easy].  
       ** apply prefix_exists_suffix in H_hard.
          destruct H_hard as [u2_part H_split_w0_suf].
          assert (H_about_u2_part : prefix u2_part u2).
          { rewrite H_split_w0_suf in H_about_w0_suf. 
            now apply prefix_app_inv in H_about_w0_suf.
          }
          (* Hopefully this works *) 
          spec H_cc_u (w ++ u1 ++ [Snd p q m ↾ H_neq] ++ u2_part).
          spec H_cc_u.
          repeat apply prefix_app. simpl.
          apply prefix_cons. assumption.
          spec H_cc_u p q H_neq0.
          assert (H_eq_p : wproj w0 p = wproj (w ++ u1 ++ [Snd p q m ↾ H_neq] ++ u2_part) p).
          {
            rewrite H_split_w0.
            clean. apply app_inv_head_iff.
            rewrite H_split_w0_suf. 
            rewrite H_empty_u1.
            rewrite app_nil_l. rewrite wproj_app.
            rewrite H_empty_u1. easy.
          } 
          rewrite <- H_split_w0 at 2. 
          rewrite mproj_snd_wproj_idempotent.
          rewrite H_eq_p.
          rewrite H_split_w0_suf.
          assert (H_eq_q : mproj_rcv (w ++ u1 ++ [Snd p q m ↾ H_neq] ++ u2_part) p q = mproj_rcv (w ++ [Snd p q m ↾ H_neq] ++ u1 ++ u2_part) p q).
          {
            clean_mproj. (* Wow! *) 
          }
          rewrite H_eq_q in H_cc_u.
          rewrite <- mproj_snd_wproj_idempotent.
          assumption.
       ** assert (H_about_w0_suf' : mproj_snd w0_suf p q = []). 
          {
            apply (wproj_preserves_prefix _ _ p) in H_easy.
            rewrite H_empty_u1 in H_easy.
            apply prefix_nil_inv in H_easy.
            rewrite mproj_snd_wproj_idempotent.
            rewrite H_easy.
            reflexivity.
          }
          spec H_cc_u (w ++ w0_suf).
          spec H_cc_u.
          apply prefix_app.
          now apply prefix_app_r.
          spec H_cc_u p q H_neq.
          rewrite mproj_snd_app in H_cc_u.  
          rewrite H_about_w0_suf' in H_cc_u.
          clean_mproj H_cc_u.
          clean_mproj. rewrite mproj_rcv_app in H_cc_u.
          now apply prefix_app_r.
     * (* In this case, nothing changes *)
       clean_mproj. (* Gets rid of the irrelevant action magically! *)
       (* Although we still need to do case analysis on the status of w0_suf because how we instantiate H_cc_u will depend on it *)
       assert (H_case := prefix_weak_total u1 w0_suf (u1 ++ u2)).
       spec H_case. now apply prefix_app_r.
       spec H_case H_about_w0_suf.
       destruct H_case as [H_hard | H_easy]. 
       ** apply prefix_exists_suffix in H_hard.
          destruct H_hard as [u2_part H_split_w0_suf].
          assert (H_about_u2_part : prefix u2_part u2).
          { rewrite H_split_w0_suf in H_about_w0_suf. 
            now apply prefix_app_inv in H_about_w0_suf.
          }
          (* Hopefully this works *) 
          spec H_cc_u (w ++ u1 ++ [Snd p q m ↾ H_neq] ++ u2_part).
          spec H_cc_u.
          repeat apply prefix_app. simpl.
          apply prefix_cons. assumption.
          spec H_cc_u p0 q0 H_neq0.
          clean_mproj H_cc_u.
          rewrite H_split_w0_suf.
          clean_mproj.
       ** spec H_cc_u (w ++ w0_suf). 
          spec H_cc_u.
          repeat apply prefix_app.
          now apply prefix_app_r. 
          spec H_cc_u p0 q0 H_neq0.
          clean_mproj H_cc_u.
Qed.

