Require Import List Classical Nat Logic Lia ExtensionalityFacts Streams stdpp.list.
Import ListNotations.
From Coq  
  Require Import ssreflect.
Require Import Logic.FunctionalExtensionality Logic.PropExtensionality.
     
From CC  
  Require Import lib structures lts clts protocol channel_complete channel_compliant run closure_conditions.

(** Canonical implementations **) 

Section Canonical.
 
Context {State : Type}. 

Section Definitions.

  Context {LocalState : Type}.
  
(** Declarative definition of canonical implementations **) 
Definition canonical_implementation_local (S : LTS) (p : participant) (T_p : LTS) :=
  (* Every finite word in L(T_p) has a finite word counterpart in L(S) *)
  (forall (w : FinAsyncWord),
      (@is_finite_word AsyncAlphabet LocalState T_p w -> 
       (exists (w' : FinAsyncWord),
           @is_finite_protocol_word State S w' /\
             wproj w' p = w))
      /\
        (* Every finite word in L(S) has its projection in L(T_p) *)
        (@is_finite_protocol_word State S w ->
         @is_finite_word AsyncAlphabet _ T_p (wproj w p)))
  /\
    (* Every finite prefix in L(T_p) has a finite prefix counterpart in L(S) *)
    (forall (w : FinAsyncWord), 
        (@is_trace AsyncAlphabet _ T_p w ->
        (exists (w' : FinAsyncWord),
            @is_protocol_prefix State S w' /\
              wproj w' p = w))
        /\
          (* Every finite prefix in protocol S has its projection in the prefix set of T_p *)
          (is_protocol_prefix S w ->
           @is_trace AsyncAlphabet _ T_p (wproj w p))). 
        
Definition canonical_implementation (S : LTS) (T: CLTS) :=
  forall (p : participant),
    @canonical_implementation_local S p (implementations T p). 

(** Facts about canonical implementations **)
Lemma canonical_implementation_locally_reachable_means_globally_reachable :
  ∀ (S : LTS) (T : CLTS) (w : FinAsyncWord) (p : participant) (s_w : LocalState),
    deadlock_free S -> 
    canonical_implementation S T -> 
    lts.Reachable (implementations T p) (s0 (implementations T p)) (wproj w p) s_w ->
    exists (s : State) (v : FinSyncWord),
      lts.Reachable S (s0 S) v s /\
        wproj (split v) p = (wproj w p). 
Proof.
  intros S T w p s_w H_df H_canonical H_reach_local. 
  spec H_canonical p. 
  destruct H_canonical as [_ H_canonical_pref].
  spec H_canonical_pref (wproj w p). 
  destruct H_canonical_pref as [H_canonical_pref _].
  spec H_canonical_pref.
  exists s_w. assumption.
  destruct H_canonical_pref as [w' [H_pref H_eq]]. 
  (* We know that w' is a protocol prefix, 
     and we want to construct a synchronous word whose split onto p is w' *)
  destruct H_pref as [H_fin | H_inf].
  - (* In the case that w is a prefix of a finite word w_fin *)
    destruct H_fin as [w_fin [H_fin H_pref]].
    unfold is_finite_protocol_word in H_fin.
    destruct H_fin as [rho_fin [H_max [H_role H_cc]]]. 
    (* Want to find a synchronous word whose split onto p is equal to w *) 
    (* And we know that w_fin is a finite protocol word from run w_fin_run *)
    (* And w is a prefix of w_fin *) 
    destruct H_max as [s_rho_fin [H_reach H_max]].
    assert (H_useful := run_for_word_means_run_prefix_for_word_prefix S w_fin rho_fin s_rho_fin p H_reach H_role H_cc w' H_pref).
    destruct H_useful as [run' [H_run' H_proj_run']].
    assert (H_helper : @is_trace SyncAlphabet State S run').  
    { eapply lts_trace_prefix_closed.
      exists s_rho_fin. exact H_reach. assumption. }
    destruct H_helper as [s_run' H_reach_run'].
    exists s_run', run'. split. assumption.
    rewrite <- H_eq. assumption.
  - (* In the case that w is a prefix of an infinite word w_inf *)
    destruct H_inf as [w_inf [H_inf [i H_eq_i]]].
    unfold is_infinite_protocol_word in H_inf.
    spec H_inf i. 
    destruct H_inf as [rho_inf [u [H_trace [H_role H_cc]]]]. 
    assert (H_useful := run_for_word_means_run_prefix_for_word_prefix S).
    spec H_useful (stream_to_list w_inf i ++ u).
    spec H_useful rho_inf.
    eapply deadlock_free_lts_trace_prefix_iff in H_trace. 
    2 : exact H_df. destruct H_trace as [s_rho_inf H_trace].
    spec H_useful s_rho_inf p H_trace.
    spec H_useful. intro q.
    symmetry. spec H_role q. assumption.
    spec H_useful H_cc.
    spec H_useful w'. spec H_useful.
    rewrite <- H_eq_i.
    apply prefix_app_r. reflexivity.
    destruct H_useful as [run' [H_run' H_proj_run']].
    assert (H_helper : @is_trace SyncAlphabet State S run').  
    { eapply lts_trace_prefix_closed.
      exists s_rho_inf. exact H_trace. assumption. } 
    destruct H_helper as [s_run' H_reach_run'].
    exists s_run', run'. split. assumption.
    rewrite <- H_eq. assumption. 
Qed. 

Lemma canonical_implementation_local_transition_means_global_transition :
  ∀ (S : LTS) (T : CLTS) (w : FinAsyncWord) (x : AsyncAlphabet) (p : participant) (s_w s_wx : LocalState),
    deadlock_free S -> 
    canonical_implementation S T -> 
    @lts.Reachable AsyncAlphabet _ (implementations T p) (s0 (implementations T p)) w s_w ->
    @lts.Reachable AsyncAlphabet _ (implementations T p) (s0 (implementations T p)) (w ++ [x]) s_wx ->
    exists (s1 s2 : State) (v : FinSyncWord),
      @lts.Reachable SyncAlphabet State S (s0 S) v s1 /\
        wproj (split v) p = w /\
        transition S s1 (async_to_sync x) s2. 
Proof.
  intros S T w x p s_w s_wx H_df H_canonical H_reach_w H_reach_wx.
  (* This fact gets used all over the place later *) 
  assert (H_idem_w : wproj w p = w). 
  {
    rewrite wproj_no_effect.
    now apply (about_implementations_alphabet T w p s_w).
    reflexivity. }
  assert (H_idem_wx : wproj (w ++ [x]) p = w ++ [x]). 
  {
    rewrite wproj_no_effect.
    now apply (about_implementations_alphabet T (w ++ [x]) p s_wx).
    reflexivity. }
  assert (H_idem_x : wproj [x] p = [x]). 
  {
    rewrite wproj_no_effect.
    apply (about_implementations_alphabet T (w ++ [x]) p s_wx) in H_reach_wx.
    apply List.Forall_app in H_reach_wx.
    tauto. 
    reflexivity. }
  (* First we establish that there exists a run whose projection is wx onto p *) 
  assert (H_useful := canonical_implementation_locally_reachable_means_globally_reachable). 
  spec H_useful S T (w ++ [x]) p s_wx H_df H_canonical.
  spec H_useful.
  { rewrite wproj_app.
    rewrite H_idem_x. 
    rewrite H_idem_w. 
    exact H_reach_wx.  }
  destruct H_useful as [s_rho [rho [H_reach_rho H_about_rho]]].
  (* Now we find the unique splitting of the run for p with respect to w *)
  assert (H_split := prefix_app_finite_unique_splitting_elaborate).  
  spec H_split rho w x p. spec H_split.
  rewrite H_about_rho.
  rewrite H_idem_wx.
  reflexivity.
  destruct H_split as [alpha [y [beta [H_split [H_max H_role_eq]]]]].
  (* Now we establish that the next action is related to x *)
  assert (H_active_y := finite_unique_splitting_next_active). 
  spec H_active_y rho w x p alpha y beta.
  spec H_active_y.
  rewrite H_about_rho.
  rewrite H_idem_wx. 
  reflexivity.
  spec H_active_y H_max H_split.
  (* Now we establish that alpha ++ [y] is a trace in S *)
  assert (H_trace_alpha_y : @is_trace SyncAlphabet State S (alpha ++ [y])).
  { eapply lts_trace_prefix_closed.
    exists s_rho. 
    exact H_reach_rho.
    rewrite H_split.
    rewrite app_assoc. 
    apply prefix_app_r.
    reflexivity. }
  (* Now we "unwind" alpha ++ [y] to reveal the transition *)
  destruct H_trace_alpha_y as [s_alpha_y H_reach_alpha_y].
  eapply lts.Reachable_unwind in H_reach_alpha_y.
  destruct H_reach_alpha_y as [s_alpha' [H_reach_alpha' H_transition]].
  (* Now we instantiate s1 and s2 with s_alpha' and s_alpha_y *) 
  exists s_alpha', s_alpha_y, alpha.
  split. assumption. 
  split.
  destruct H_max as [H_pref_alpha [H_role_eq_alpha H_max_alpha]].
  rewrite <- H_role_eq_alpha.
  assumption.
  rewrite <- H_active_y.
  assumption.
Qed.

Lemma canonical_implementation_prefixes_include_protocol_prefixes :
  forall (S : LTS) (T : CLTS),
    @canonical_implementation S T ->
    forall (w : FinAsyncWord),
      is_protocol_prefix S w ->
      @is_clts_trace _ T w. 
Proof. 
  intros S T H_canonical w H_prefix. 
  induction w as [| x w' IHw'] using rev_ind. 
  - unfold is_clts_trace. 
    exists (c0 T).
    apply Reachable_refl. 
  - (* Proof idea: apply induction hypothesis to obtain a configuration for w', 
     then use the fact that T is an canonical implementation to construct a configuration
     for w'x, performing case analysis on whether x is a send or receive event *)
    (* First, obtaining a configuration for w' *)
    assert (H_prefix' : is_protocol_prefix S w'). 
    { unfold is_protocol_prefix in H_prefix.
      destruct H_prefix as [H_prefix_fin | H_prefix_inf]. 
      (* Either w'x comes from a finite or infinite word, 
     and w' will be a prefix of the same respective word type *) 
      * destruct H_prefix_fin as [u [H_member H_prefix]].
        left. (* Matching the case for w'x *)
        exists u. 
        split. assumption.
        now apply (prefix_app_l w' u [x]).
      * destruct H_prefix_inf as [u [H_member H_prefix]].
        right. exists u. split. assumption.
        destruct H_prefix as [i H_prefix].
        (* Want to use i-1 as the witness, need to first establish i > 0 *)
        destruct i as [| i'].
        inversion H_prefix. 
        apply app_cons_not_nil in H0. contradiction.
        (* Having established that i = S i', we can use i' *)
        exists i'. now apply stream_to_list_app with x. (* Need an intermediate fact about stream_to_list *) }
    (* Having established that w' is a prefix in L(S), we can now use IHw' *) 
    spec IHw' H_prefix'.
    (* Now we want to obtain a configuration on w' *)
    destruct IHw' as [c_w' IHw'].
    (* Case analysis on whether x is a send or receive event *)
    (* The way we construct c_w'x will depend on whether x is a send or receive event *) 
    destruct x as [x H_neq]. destruct x as [p q v | p q v].
    * (* In the case that x = p!q:v *) 
      spec H_canonical p.
      (* We want to use the second conjunct from canonicalness which talks about prefixes *)
      destruct H_canonical as [_ H_canonical_prefix].
      spec H_canonical_prefix (w' ++ [exist _ (Snd p q v) H_neq]).
      (* We want to use the direction which concludes that wx is a local prefix for p *) 
      destruct H_canonical_prefix as [_ H_canonical_prefix].
      spec H_canonical_prefix.
      assumption. 
      (* Obtaining the post-state for p after performing p!q:m *) 
      destruct H_canonical_prefix as [s_p'' H_reach].
      apply clts_trace_snd_extension_sufficient; try assumption.  
      now exists c_w'.
      exists s_p''. unfold wproj in H_reach; rewrite flat_map_app in H_reach.
      unfold wproj_symbol at 2 in H_reach.
      simpl in H_reach. 
      replace (participant_eqb p p) with true in H_reach by (symmetry; now apply participant_eqb_refl).
      simpl in H_reach.
      assumption.
    * (* In the case that x = q?p:v, now the active role is q *)
      spec H_canonical q.
      assert (H_prefix_copy := H_prefix). 
      (* We want to use the second conjunct from canonicalness which talks about prefixes *)
      destruct H_canonical as [_ H_canonical_prefix].
      spec H_canonical_prefix (w' ++ [exist _ (Rcv p q v) H_neq]).
      (* We want to use the direction which concludes that wx is a local prefix for p *) 
      destruct H_canonical_prefix as [_ H_canonical_prefix].
      spec H_canonical_prefix.
      assumption. 
      (* Obtaining the post-state for q after performing q?p:m *) 
      destruct H_canonical_prefix as [s_q'' H_reach].
      eapply clts_trace_rcv_extension_sufficient; try assumption. 
      exact IHw'.
      exists s_q''.
      unfold wproj in H_reach; rewrite flat_map_app in H_reach.
      unfold wproj_symbol at 2 in H_reach.
      simpl in H_reach. 
      replace (participant_eqb q q) with true in H_reach by (symmetry; now apply participant_eqb_refl).
      simpl in H_reach.
      assumption.
      (* Need to behead the channel (p,q) in order to define the channel contents *)
      (** Need to show that not only is channel (w',p,q) not empty,
          but additionally its head message is v and not any other message value *) 
      (** Need the fact that protocol prefixes are all channel-compliant,
          and also CLTS traces satisfy the stronger equality of rcv + channel = snd, 
          only then can we establish that because w'x is a protocol prefix,
          the channel between p and q in c_w' must contain v at its head *)
      remember (get_channel_contents c_w' p q) as chan. 
      destruct chan.
      { (* Now we need to prove a contradiction here *) 
        (* We get a contradiction from the fact that w'x is channel compliant *)
        (* And the characterization of buffer contents for CLTS configurations *)
        assert (H_cc : channel_compliant (w' ++ [Rcv p q v ↾ H_neq])) by now apply (protocol_prefix_channel_compliant S).
        (* assert (H_cc' : channel_compliant w') by now apply (protocol_prefix_channel_compliant S).  *)
        assert (H_about := @about_clts_trace_configuration_channel_contents _ T). 
        spec H_about w' c_w' IHw' p q H_neq.
        spec H_cc (w' ++ [Rcv p q v ↾ H_neq]).
        spec H_cc.
        reflexivity. 
        spec H_cc p q H_neq.
        replace (mproj_rcv (w' ++ [Rcv p q v ↾ H_neq]) p q) with
          (mproj_rcv w' p q ++ [v]) in H_cc.
        2: 
          { unfold mproj_rcv at 2.
            rewrite flat_map_app.
            assert (H_helper := mproj_rcv_rcv).
            spec H_helper (Rcv p q v ↾ H_neq).
            spec H_helper. easy.
            rewrite H_helper.
            unfold value_async; simpl.
            reflexivity. }
        replace (mproj_snd (w' ++ [Rcv p q v ↾ H_neq]) p q) with
          (mproj_snd w' p q) in H_cc.
        2: { assert (H_helper := mproj_snd_rcv).
             spec H_helper (Rcv p q v ↾ H_neq).
             spec H_helper. easy.
             unfold mproj_snd at 2.
             rewrite flat_map_app.
             rewrite H_helper.
             rewrite app_nil_r. reflexivity. }
        rewrite <- Heqchan in H_about. rewrite app_nil_r in H_about.
        rewrite H_about in H_cc.
        apply prefix_snoc_not in H_cc. contradiction. }
      (* Now that we have shown that the channel between p and q in c_w' cannot be empty,
         we need to show that m = v *)
      assert (H_head_msg : m = v). 
      {
        (* We know that w'x is channel compliant because all protocol prefixes are channel compliant *) 
        assert (H_cc : channel_compliant (w' ++ [Rcv p q v ↾ H_neq])) by now apply (protocol_prefix_channel_compliant S). 
        spec H_cc (w' ++ [Rcv p q v ↾ H_neq]).
        spec H_cc.
        reflexivity. 
        spec H_cc p q H_neq.
        replace (mproj_rcv (w' ++ [Rcv p q v ↾ H_neq]) p q) with
          (mproj_rcv w' p q ++ [v]) in H_cc.
        2: 
          { unfold mproj_rcv at 2.
            rewrite flat_map_app.
            assert (H_helper := mproj_rcv_rcv).
            spec H_helper (Rcv p q v ↾ H_neq).
            spec H_helper. easy.
            rewrite H_helper.
            unfold value_async; simpl.
            reflexivity. }
        replace (mproj_snd (w' ++ [Rcv p q v ↾ H_neq]) p q) with
          (mproj_snd w' p q) in H_cc.
        2: { assert (H_helper := mproj_snd_rcv).
             spec H_helper (Rcv p q v ↾ H_neq).
             spec H_helper. easy.
             unfold mproj_snd at 2.
             rewrite flat_map_app.
             rewrite H_helper.
             rewrite app_nil_r. reflexivity. }
        (* We know that w' is a CLTS trace and thus satisfies a stronger property *)
        assert (H_about := @about_clts_trace_configuration_channel_contents _ T). 
        spec H_about w' c_w' IHw' p q H_neq.
        rewrite <- H_about in H_cc.
        rewrite <- Heqchan in H_cc.
        (* Now we can show that m = v *)
        apply prefix_app_inv in H_cc. 
        inversion H_cc. inversion H. easy. }
      exists chan. now rewrite H_head_msg.
Qed.

Lemma canonical_implementation_language_includes_protocol_language :
  forall (S : LTS) (T : CLTS),
    @canonical_implementation S T ->
    clts_includes_protocol_language S T.
Proof.
  intros S T H_canonical. 
  split; intros w H_word.
  - (* Finite case *)
    (* First we establish that w is a trace in CLTS T *)  
    assert (H_helper := @canonical_implementation_prefixes_include_protocol_prefixes S T H_canonical).
    spec H_helper w.
    spec H_helper.
    left.
    exists w. split. assumption. reflexivity.
    (* Now we need to show that not only is it a trace,
       it is a finite maximal trace,
       meaning that all states are final
       and all channels are empty *) 
    destruct H_helper as [c_w H_reach].
    exists c_w. split. assumption.
    assert (H_word_copy := H_word). 
    destruct H_word as [run [H_max [H_role H_cc]]]. 
    destruct H_max as [s_w [H_reach_s H_final]].
    split. 
    * (* Showing that all states are final *)
      intro p.
      spec H_canonical p. 
      destruct H_canonical as [H_canonical_fin _].
      spec H_canonical_fin w.
      apply H_canonical_fin in H_word_copy.
      unfold is_finite_word in H_word_copy. 
      destruct H_word_copy as [s_p_w [H_reach_p H_final_p]].
      assert (H_eq : s_p_w = get_local_state c_w p).
      { (* This should follow from the fact that p's implementation is deterministic *)
       assert (H_det_local := deterministic_word (implementations T p)). 
       spec H_det_local s_p_w (get_local_state c_w p).
       spec H_det_local (wproj w p).
       spec H_det_local.
       apply (deterministic_implementations T).
       spec H_det_local H_reach_p. 
       spec H_det_local.
       apply clts_reachable_means_implementation_reachable;
          try assumption. 
       assumption. }
    rewrite <- H_eq.
    exact H_final_p.
    * (* Showing that all channels are empty *)
      assert (H_cc' := split_word_channel_complete run).
      (* Need a lemma that states Swappable preserves channel completeness *)
      (* Intuitively this is true because Swappable is a permutation and channel_completeness is a filter equality *)
      assert (H_cc_w : channel_complete w) by now apply (@finite_protocol_word_channel_complete _ S).
      intros p q H_neq.
      spec H_cc_w p q H_neq. 
      assert (H_about_chan := @about_clts_trace_configuration_channel_contents _ T w c_w H_reach p q H_neq).
      rewrite H_cc_w in H_about_chan.
      rewrite <- app_nil_r in H_about_chan.
      apply app_inv_head in H_about_chan. 
      assumption.
  - (* Infinite case *)
    intros i.
    (* To show that w is an infinite word, we need to show that every prefix is a trace in the CLTS *)
    (* Let (stream_to_list w i) be an arbitrary prefix *)
    (* We know that there exists a run such that for every finite prefix of w,
       there exists a longer prefix of split run that is swappable with w *)
    assert (H_helper := canonical_implementation_prefixes_include_protocol_prefixes S T H_canonical).
    spec H_helper (stream_to_list w i).
    spec H_helper.
    right. 
    exists w. split. assumption.
    exists i. reflexivity.
    destruct H_helper as [c H_helper].
    exists c. assumption.
Qed.

End Definitions. 

(** Canonical implementation existence proof **) 
(* For the preciseness theorem, we only need to show the existence of canonical implementations for GCLTS *)

(* First observation: canonical_implementation_local is too strong because it refers to protocol words and prefixes, i.e. the asynchronous semantics of S treated as a protocol *)
(* Because of the definition of asynchronous semantics, the following definition that treats S as an LTS naively should suffice *)

(* Note that because the alphabets of the naive LTS S and S_p are different, each iff direction must be stated separately *) 
Definition canonical_implementation_local_naive {LocalState : Type} (S : LTS) (p : participant) (S_p : LTS) := 
  (* Every finite word in S_p has a corresponding finite word in S *)
  (∀ w : FinAsyncWord, 
     (@is_finite_word AsyncAlphabet LocalState S_p w
      → ∃ w' : FinSyncWord,
          @is_finite_word SyncAlphabet State S w'
          ∧ wproj (split w') p = w))
  ∧
    (* Every finite word in S has a corresponding finite word in S_p *)
    (forall w : FinSyncWord, 
       (@is_finite_word SyncAlphabet State S w
        → @is_finite_word AsyncAlphabet LocalState S_p (wproj (split w) p)))
  ∧
    (* Every prefix in S_p has a corresponding prefix in S *) 
    (∀ w : FinAsyncWord,
        (@is_trace AsyncAlphabet LocalState S_p w
         → ∃ w' : FinSyncWord,
            @is_trace SyncAlphabet State S w'
            ∧ wproj (split w') p = w))
  ∧
    (* Every finite prefix in S has a corresponding finite prefix in S_p *) 
    (forall w : FinSyncWord, 
        (@is_trace SyncAlphabet State S w
         → @is_trace AsyncAlphabet LocalState S_p (wproj (split w) p))). 

(* Proving that this treatment of canonical implementations is indeed sufficient for the version that is used in all the proofs *)
(* This sufficiency indeed exploits the deadlock freedom of S *) 
Lemma canonical_implementation_local_naive_sufficient :
  forall {LocalState : Type} (S : LTS) (p : participant) (S_p : LTS),
    deadlock_free S -> 
    @canonical_implementation_local_naive LocalState S p S_p ->
    @canonical_implementation_local LocalState S p S_p. 
Proof.
  intros LocalState S p S_p H_df H.
  split.
  - intro w. split.
    * intro H_word.
      destruct H as [H _].
      spec H w H_word.
      destruct H as [w' [H_word_w' H_about_w']].
      exists (split w'). 
      split.
      apply split_finite_run_is_finite_word.
      easy. easy.
    * intro H_word.
      destruct H as [_ [H _]].
      destruct H_word as [run H_about_run].
      spec H run. spec H. easy.
      destruct H_about_run as [_ [H_about_run _]].
      spec H_about_run p. rewrite <- H_about_run.
      assumption.
  - intro w. split.
    * intro H_trace.
      destruct H as [_ [_ [H _]]].
      spec H w H_trace.
      destruct H as [w' [H_trace_w' H_split_w']].
      exists (split w').
      split.
      destruct H_trace_w' as [s_w' H_reach_s_w'].
      eapply split_run_protocol_prefix.
      assumption.
      exact H_reach_s_w'. 
      assumption.
    * intro H_prefix.
      destruct H as [_ [_ [_ H]]].
      (* We want to find a synchronous prefix for instantiating H *) 
      (* To do this we appeal to some other lemmas *)
      assert (H_useful1 := @protocol_prefix_exists_possible_run_prefix State).
      spec H_useful1 S w H_df H_prefix.
      destruct H_useful1 as [rho H_possible_rho].
      destruct H_possible_rho as [H_trace_rho H_participant_rho].
      (* From rho we want to get a prefix for p *)
      spec H_participant_rho p.
      assert (H_useful2 := finite_unique_splitting).
      spec H_useful2 rho w p H_participant_rho.
      destruct H_useful2 as [alpha H_alpha].
      unfold is_alpha in H_alpha.
      spec H alpha.
      spec H.
      eapply lts_trace_prefix_closed. exact H_trace_rho.
      easy. destruct H_alpha as [_ [H_rewrite _]].
      rewrite H_rewrite.
      assumption.
Qed. 

(* Having established the sufficiency of canonical_implementation_local_naive, we can now use this as a target specification for synthesizing canonical implementations *) 

(* Now we want to define the projection of an LTS onto a participant *)

Definition initial_subset_construction_state (S : @LTS SyncAlphabet State) (p : participant) : State -> Prop :=
  (* The initial state is the set of all states reachable on epsilon from s0 *)
  (fun s => exists (w : list SyncAlphabet), lts.Reachable S (s0 S) w s /\ wproj (split w) p = []). 

Definition subset_construction_state (S : @LTS SyncAlphabet State) (p : participant) : (State -> Prop) -> Prop :=
  (* A subset construction state is non-empty *) 
  (fun lstate => (exists (s : State), lstate s)). 

Definition subset_construction_transition_relation (S : @LTS SyncAlphabet State) (p : participant) : (State -> Prop) -> AsyncAlphabet -> (State -> Prop) -> Prop :=
  fun lstate1 a lstate2 => is_active p a 
                        /\ subset_construction_state S p lstate1
                        /\ subset_construction_state S p lstate2
                        /\ forall (s' : State), lstate2 s' <->
                                            (* Either it's an immediate next state on async_to_sync a *)
                                            (exists (s : State), lstate1 s /\ transition S s (async_to_sync a) s') \/
                                              (* Or it's epsilon-reachable from an immediate next state *)
                                              (exists (s s_inter : State), lstate1 s /\ transition S s (async_to_sync a) s_inter /\ exists (v_epsilon : list SyncAlphabet), lts.Reachable S s_inter v_epsilon s' /\ wproj (split v_epsilon) p = []). 


Definition final_subset_construction_state (S : @LTS SyncAlphabet State) (p : participant) : (State -> Prop) -> Prop :=
  fun lstate => subset_construction_state S p lstate /\ exists (s : State), lstate s /\ final S s. 

Definition LTS_participant_subset_construction (S : @LTS SyncAlphabet State) (p : participant) : @LTS AsyncAlphabet (State -> Prop) :=
  mkLTS AsyncAlphabet (State -> Prop)
    (subset_construction_transition_relation S p)
    (initial_subset_construction_state S p)
    (final_subset_construction_state S p). 

(* Now we prove directly that the subset construction LTS is a naive canonical implementation *)

Lemma subset_construction_reachable_state_nonempty :
  forall (S : @LTS SyncAlphabet State)
    (p : participant)
    (ls : State -> Prop)
    (w : FinAsyncWord),
    lts.Reachable (LTS_participant_subset_construction S p) (s0 (LTS_participant_subset_construction S p)) w ls ->
    exists (s : State),
      ls s. 
Proof.
  intros.
  generalize dependent ls.
  induction w as [|x w' IHw'] using rev_ind; intros ls H_reach.
  - inversion H_reach; subst; simpl.
    exists (s0 S). 
    unfold initial_subset_construction_state.
    exists []. split. apply lts.Reachable_refl. easy.
    rewrite app_nil in H. destruct H as [_ H].
    inversion H. 
  -  apply lts.Reachable_app_inv in H_reach.
    destruct H_reach as [ls_w' [H_reach_ls_w' H_reach_ls_w]].
    spec IHw' ls_w' H_reach_ls_w'.  
    destruct IHw' as [s_w' H_about_s_w'].
    apply lts.Reachable_singleton_inv in H_reach_ls_w.
    unfold transition in H_reach_ls_w.
    unfold LTS_participant_subset_construction in H_reach_ls_w.
    unfold subset_construction_transition_relation in H_reach_ls_w.
    destruct H_reach_ls_w as [_ [_ [H_about_ls _]]].
    unfold subset_construction_state in H_about_ls.
    tauto.
Qed.

(* First we show that all words in the subset construction are over the participant's alphabet *) 
Lemma about_subset_construction_alphabet :
  ∀ (S : @LTS SyncAlphabet State) (p : participant) (w : FinAsyncWord) (ls_w : State -> Prop),
        @lts.Reachable AsyncAlphabet (State -> Prop) (LTS_participant_subset_construction S p) (s0 (LTS_participant_subset_construction S p)) w ls_w ->
        Forall (is_active p) w. 
Proof.         
  intros S p w.
  induction w as [|x w' IHw'] using rev_ind; intros ls_w H_reach.
  - (* Base case *)
    apply Forall_nil. easy.
  - (* Induction step *)
    (* First we establish that there exists a state in s_w *)
    assert (H_useful := subset_construction_reachable_state_nonempty S p ls_w (w' ++ [x]) H_reach). 
    destruct H_useful as [s_w H_s_w_in_ls_w].
    (* Now we unpeel ls_w to reveal the state reached on w' *)
    apply lts.Reachable_app_inv in H_reach. 
    destruct H_reach as [ls_w' [H_reach_ls_w' H_reach_ls_w]].
    spec IHw' ls_w' H_reach_ls_w'.  
    apply Reachable_singleton_inv in H_reach_ls_w.
    unfold transition in H_reach_ls_w.
    unfold LTS_participant_subset_construction in H_reach_ls_w.
    unfold subset_construction_transition_relation in H_reach_ls_w.
    destruct H_reach_ls_w as [H_active [_ [_ H_transition]]].
    apply Forall_app. 
    split. assumption.
    apply Forall_singleton. assumption.
Qed.

(* Helpful fact showing that transitions in the subset construction means there exist immediate transitions in S *)
Lemma subset_construction_reachable_transition_means_S_transition :
  ∀ (S : @LTS SyncAlphabet State) (p : participant) (w : FinAsyncWord) (x : AsyncAlphabet) (ls1 ls2 : State -> Prop),
    lts.Reachable (LTS_participant_subset_construction S p) (s0 (LTS_participant_subset_construction S p)) w ls1 -> 
    transition (LTS_participant_subset_construction S p) ls1 x ls2 ->
    exists (s1 s2 : State),
      transition S s1 (async_to_sync x) s2 /\
        ls1 s1 /\
        ls2 s2. 
Proof.
  intros S p w x ls1 ls2 H_reach H_transition.
  unfold transition in H_transition. 
  unfold LTS_participant_subset_construction in H_transition.
  unfold subset_construction_transition_relation in H_transition. 
  (* First we show that because ls2 is reachable, it is non-empty *) 
  assert (H_useful := subset_construction_reachable_state_nonempty S p ls2 (w ++ [x])).
  spec H_useful.
  eapply lts.Reachable_step. 
  exact H_reach. exact H_transition.
  destruct H_useful as [s2 H_s2_in_ls2].
  (* There is a state s2 in ls2 *) 
  destruct H_transition as [_ [_ [_ H_transition]]].
  assert (H_transition_copy := H_transition).
  spec H_transition s2. destruct H_transition as [H_transition _].
  spec H_transition H_s2_in_ls2.
  destruct H_transition as [H_immediate | H_epsilon].
  - destruct H_immediate as [s1 [H_s1_in_ls1 H_transition]].
    exists s1, s2. easy.  
  - destruct H_epsilon as [s1 [s_inter [H_s1_in_ls1 [H_transition [v_epsilon [H_reach_v_epsilon H_about_v_epsilon]]]]]]. 
    spec H_transition_copy s_inter.
    destruct H_transition_copy as [_ H_transition_copy].
    spec H_transition_copy.
    left. exists s1. easy. exists s1, s_inter. 
    easy. 
Qed.

 Lemma subset_construction_reachable_transition_fact :
  ∀ (S : @LTS SyncAlphabet State) (p : participant) (w : FinAsyncWord) (x : AsyncAlphabet) (ls1 ls2 : State -> Prop),
    lts.Reachable (LTS_participant_subset_construction S p) (s0 (LTS_participant_subset_construction S p)) w ls1 -> 
    transition (LTS_participant_subset_construction S p) ls1 x ls2 ->
    forall (s2 : State),
      ls2 s2 -> 
      (exists (s1 : State), ls1 s1 /\ transition S s1 (async_to_sync x) s2) \/
        (exists (s1 s_inter : State), ls1 s1 /\ transition S s1 (async_to_sync x) s_inter /\ 
                                  exists (v_epsilon : list SyncAlphabet), lts.Reachable S s_inter v_epsilon s2 /\ wproj (split v_epsilon) p = []). 
 Proof.
   intros S p w x ls1 ls2 H_reach H_transition s2 H_s2_in_ls2. 
   destruct H_transition as [_ [_ [_ H_transition]]].
   spec H_transition s2.
   destruct H_transition as [H_transition _].
   apply H_transition. assumption.
 Qed.

(* We also show that states in the subset construction are epsilon-closed *) 
Lemma subset_construction_states_epsilon_closed :
  ∀ (S : @LTS SyncAlphabet State) (p : participant) (w : FinAsyncWord) (ls_w : State -> Prop),
    @lts.Reachable AsyncAlphabet (State -> Prop) (LTS_participant_subset_construction S p) (s0 (LTS_participant_subset_construction S p)) w ls_w ->
    forall (s1 s2 : State),
      ls_w s1 ->
      (exists (v : FinSyncWord), lts.Reachable S s1 v s2 /\ wproj (split v) p = []) ->
      ls_w s2. 
Proof.     
  intros S p w.
  induction w as [|x w' IHw'] using rev_ind; intros ls_w H_reach s1 s2 H_s1_in_s_w H_exists. 
  - inversion H_reach; subst.
    unfold s0. unfold LTS_participant_subset_construction.
    unfold initial_subset_construction_state.
    unfold s0 in H_s1_in_s_w. unfold LTS_participant_subset_construction in H_s1_in_s_w.
    unfold initial_subset_construction_state in H_s1_in_s_w.
    destruct H_s1_in_s_w as [v1 [H_reach_s1 H_about_v1]].
    destruct H_exists as [v2 [H_reach_v2 H_about_v2]].
    exists (v1 ++ v2). split.
    eapply Reachable_app. exact H_reach_s1. exact H_reach_v2.
    rewrite wproj_split_app. rewrite H_about_v1. rewrite H_about_v2. reflexivity.
    apply app_nil in H. easy.
  - apply Reachable_app_inv in H_reach. 
    destruct H_reach as [ls_w' [H_reach_ls_w' H_reach_ls_w]].
    apply Reachable_singleton_inv in H_reach_ls_w.
    (* Because s1 is in ls_w, either there is
       s1_pre -x-> s1 in S, or
       s1_pre -x-> s1_inter -epsilon->* s1 in S with s1_pre in ls_w' *)
    (* From the induction hypothesis on ls_w', s1_inter is also in ls_w' *)
    (* From H_exists we know that
       s1 -epsilon->* s2 in S *)
    assert (H_useful := subset_construction_reachable_transition_fact S p w' x ls_w' ls_w H_reach_ls_w' H_reach_ls_w s1 H_s1_in_s_w).
    destruct H_reach_ls_w as [_ [_ [_ H_about_transition]]].
    spec H_about_transition s2. 
    destruct H_about_transition as [_ H_about_transition]. 
    apply H_about_transition.
    destruct H_useful as [H_immediate | H_epsilon].
    * destruct H_immediate as [s1_pre [H_s1_pre_in_ls_w' H_transition]].
      right. exists s1_pre, s1.
      split. assumption. split. assumption.
      assumption.
    * destruct H_epsilon as [s1_pre [s1_inter [H_s1_pre_in_ls_w' [H_transition H_epsilon]]]].
      right. exists s1_pre, s1_inter. split. assumption. split. assumption.
      destruct H_exists as [v1 [H_reach_v1 H_about_v1]].
      destruct H_epsilon as [v2 [H_reach_v2 H_about_v2]].
      exists (v2 ++ v1). split.
      eapply Reachable_app. exact H_reach_v2.  exact H_reach_v1.
      rewrite wproj_split_app. rewrite H_about_v1. rewrite H_about_v2.
      easy.
Qed.

Lemma strengthened_conjunct3 :
  forall (S : @LTS SyncAlphabet State)
    (p : participant)
    (w : FinAsyncWord)
    (ls : State -> Prop),
    lts.Reachable (LTS_participant_subset_construction S p) (s0 (LTS_participant_subset_construction S p)) w ls ->
    forall (s : State),
      ls s ->
      ∃ w' : FinSyncWord,
        lts.Reachable S (s0 S) w' s /\ wproj (split w') p = w.
Proof. 
  intros S p w.
  induction w as [|x w' IHw'] using rev_ind; intros ls H_reach s H_state. 
  - inversion H_reach; subst.
    unfold s0 in H_state.
    unfold LTS_participant_subset_construction in H_state.
    unfold initial_subset_construction_state in H_state.
    assumption.
    apply app_nil in H. 
    destruct H as [_ H_false].
    inversion H_false.
  - (* Induction hypothesis: for every state ls_w' such that
       s0 -w'-> ls_w' in the subset construction,
       and for every t in ls_w',
       there exists a synchronous word v' such that
       s0 -v'-> t in S, and wproj (split v') p = w' *)
    (* Now we have some state ls such that
       s0 -w'x-> ls in the subset construction,
       and some s,
       and we want to show that there exists a synchronous word v such that
       s0 -v-> s in S, and wproj (split v) p = w'x *)
    (* First we peel back the run on w'x to obtain a state reachable on w' in the subset construction *)
    apply Reachable_app_inv in H_reach.
    destruct H_reach as [ls_w' [H_reach_ls_w' H_reach_ls_w]].
    (* s1 -w'-> ls_w' -x-> ls_w in the subset construction *) 
    (* Now we want to obtain the state in ls_w' that allowed the transition on x in the subset construction to occur *)
    apply Reachable_singleton_inv in H_reach_ls_w.
    unfold transition in H_reach_ls_w. 
    unfold LTS_participant_subset_construction in H_reach_ls_w.
    unfold subset_construction_transition_relation in H_reach_ls_w. 
    destruct H_reach_ls_w as [H_active [_ [_ H_about_ls]]].
    spec H_about_ls s.
    destruct H_about_ls as [H_about_ls _].
    spec H_about_ls H_state. 
    destruct H_about_ls as [H_immediate | H_epsilon].
    * destruct H_immediate as [s_w' [H_s_w'_in_ls_w' H_transition_s_w']]. 
      (* Now we get that s_w' is in ls_w', and
       s_w' -v_x-> s in S, and v_x gives us x *)
      (* Now we want to instantiate the induction hypothesis with this particular state, s_w' *)
      spec IHw' ls_w' H_reach_ls_w'.
      spec IHw' s_w' H_s_w'_in_ls_w'.
      destruct IHw' as [v_w' [H_reach_s_w' H_about_v_w']].
      (* Now we are ready to stitch our run together *)
      exists (v_w' ++ [async_to_sync x]).
      split.
      eapply Reachable_app.
      exact H_reach_s_w'.
      apply Reachable_singleton.
      exact H_transition_s_w'. 
      rewrite wproj_split_app.
      rewrite H_about_v_w'.
      rewrite wproj_split_async_to_sync_eq. assumption.
      reflexivity. 
    * destruct H_epsilon as [s_w' [s_inter [H_s_w'_in_ls_w' [H_transition [v_epsilon [H_reach_v_epsilon H_about_v_epsilon]]]]]]. 
      spec IHw' ls_w' H_reach_ls_w'.
      spec IHw' s_w' H_s_w'_in_ls_w'.
      destruct IHw' as [v_w' [H_reach_s_w' H_about_v_w']].
      exists (v_w' ++ [async_to_sync x] ++ v_epsilon).
      split. eapply Reachable_app. exact H_reach_s_w'.
      eapply Reachable_app.
      apply Reachable_singleton. exact H_transition.
      assumption. 
      repeat rewrite wproj_split_app.
      rewrite H_about_v_w'.
      rewrite H_about_v_epsilon.
      rewrite app_nil_r.
      rewrite wproj_split_async_to_sync_eq. assumption.
      reflexivity. 
Qed.

(* This property is not inductive! *) 
Lemma conjunct3 :
  forall (S : @LTS SyncAlphabet State)
    (p : participant)
    (w : FinAsyncWord), 
     (@is_trace AsyncAlphabet (State -> Prop) (LTS_participant_subset_construction S p) w
      → ∃ w' : FinSyncWord,
          @is_trace SyncAlphabet State S w'
          ∧ wproj (split w') p = w).
Proof. 
  intros S p w H_trace.
  assert (H_useful := strengthened_conjunct3).
  destruct H_trace as [ls_w H_reach_ls_w].
  spec H_useful S p w ls_w H_reach_ls_w.
  assert (H_useful2 := subset_construction_reachable_state_nonempty).
  spec H_useful2 S p ls_w w H_reach_ls_w.
  destruct H_useful2 as [s_w H_s_w_in_ls_w].
  spec H_useful s_w H_s_w_in_ls_w.
  destruct H_useful as [v [H_reach_v H_about_v]].
  exists v. split. exists s_w. assumption. assumption.
Qed.

Lemma strengthened_conjunct4 :
  forall (S : @LTS SyncAlphabet State)
    (p : participant)
    (w : FinSyncWord)
    (s : State),
    lts.Reachable S (s0 S) w s ->
    ∃ (w' : FinAsyncWord) (ls_w' : State -> Prop),
        lts.Reachable (LTS_participant_subset_construction S p) (s0 (LTS_participant_subset_construction S p)) w' ls_w' /\ ls_w' s /\ wproj (split w) p = w'.
Proof.
  intros S p w. 
  induction w as [|x w' IHw'] using rev_ind; intros s H_reach. 
  - exists [], (s0 (LTS_participant_subset_construction S p)).
    split.
    apply lts.Reachable_refl.
    split. inversion H_reach. subst.
    unfold s0.
    unfold LTS_participant_subset_construction.
    unfold initial_subset_construction_state.
    exists []. split. easy. easy.
    apply app_nil in H. 
    destruct H as [_ H_false].
    inversion H_false.
    easy.
  - (* Induction hypothesis: for every state s_w' such that
       s0 -w'-> s_w' in S,
       there exists an asynchronous p-word v' and a state ls_v' such that 
       s0 -v'-> ls_v' in the subset construction, and
       s_w' is in ls_v', and 
       wproj (split w') p = v' *)
    (* Now we have some state s such that
       s0 -w'x-> ls in S,
       and we want to show that there exists an asynchronous p-word v
       and a state ls_v such that 
       s0 -v-> ls_v in the subset construction, and
       s is in ls_v, and 
       wproj (split w) p = v *)
    (* First we peel back the run on w'x to obtain a state reachable on w' in S *)
    apply Reachable_app_inv in H_reach.
    destruct H_reach as [s_w' [H_reach_s_w' H_reach_s]].
    (* s0 -w'-> s_w' -x-> s in S *) 
    (* Now we want to obtain the state in ls_w' that allowed the transition on x in the subset construction to occur *)
    spec IHw' s_w' H_reach_s_w'.
    destruct IHw' as [v_w' [ls_v_w' [H_reach_ls_v_w' [H_s_w'_in_ls_w' H_about_v_w']]]].
    (* Having obtained state s_w', we do case analysis on whether p is active in x *)
    destruct x as [x H_neq_x].
    destruct x as [p' q' m'].
    destruct (classic (p' = p \/ q' = p)).
    * (* In the case that p is active in x *)
      destruct H as [H_p_sender | H_p_receiver].
      ** (* In the case that p is the sender, then append the send event to v_w' *)
        assert (H_neq_x_async : sender_receiver_neq_async (Snd p' q' m')) by easy.
        remember (exist _ (Snd p' q' m') H_neq_x_async) as v_x.
        exists (v_w' ++ [v_x]). 
        (* Now I need to construct ls_w' *) 
        exists (fun s2 => (exists (s1 : State), ls_v_w' s1 /\ transition S s1 (Event p' q' m' ↾ H_neq_x) s2) \/
                    (exists (s1 s_inter : State), ls_v_w' s1 /\ transition S s1 (Event p' q' m' ↾ H_neq_x) s_inter /\ 
                                               exists (v_epsilon : list SyncAlphabet), lts.Reachable S s_inter v_epsilon s2 /\ wproj (split v_epsilon) p = [])).
        repeat split.
        *** (* Showing that this constructed state is indeed reachable *)
          eapply Reachable_app.
          exact H_reach_ls_v_w'.
          eapply Reachable_singleton.
          unfold transition. unfold LTS_participant_subset_construction.
          unfold subset_construction_transition_relation.
          split.
          { subst. easy. }
          split.
          { exists s_w'. easy. }
          split.
          { exists s. left. exists s_w'. split. assumption.
            apply Reachable_singleton_inv in H_reach_s. exact H_reach_s. }
          intro s'. split.
          intro. 
          replace (async_to_sync v_x) with (Event p' q' m' ↾ H_neq_x). 
          easy. unfold async_to_sync; subst. simpl. f_equal. apply proof_irrelevance.
          intro. replace (async_to_sync v_x) with (Event p' q' m' ↾ H_neq_x) in H.
          destruct H. left. easy. right.  easy.
          unfold async_to_sync; subst; simpl. f_equal. apply proof_irrelevance.
        *** (* Showing that tis constructed state contains s *)
          left. exists s_w'. split. assumption.
          apply Reachable_singleton_inv in H_reach_s. assumption.
        *** (* Showing that the projections match up *)
          rewrite wproj_split_app. 
          rewrite H_about_v_w'.
          subst. clean.
          rewrite wproj_symbol_sender_eq. easy.
          rewrite wproj_symbol_receiver_neq. easy.
          clean.
          repeat f_equal. apply proof_irrelevance.
      ** (* In the case that p is the receiver, then append the receive event to v_w' *)
        assert (H_neq_x_async : sender_receiver_neq_async (Rcv p' q' m')) by easy.
        remember (exist _ (Rcv p' q' m') H_neq_x_async) as v_x.
        exists (v_w' ++ [v_x]). 
        (* Now I need to construct ls_w' *) 
        exists (fun s2 => (exists (s1 : State), ls_v_w' s1 /\ transition S s1 (Event p' q' m' ↾ H_neq_x) s2) \/
                    (exists (s1 s_inter : State), ls_v_w' s1 /\ transition S s1 (Event p' q' m' ↾ H_neq_x) s_inter /\ 
                                               exists (v_epsilon : list SyncAlphabet), lts.Reachable S s_inter v_epsilon s2 /\ wproj (split v_epsilon) p = [])).
        repeat split.
        *** (* Showing that this constructed state is indeed reachable *)
          eapply Reachable_app.
          exact H_reach_ls_v_w'.
          eapply Reachable_singleton.
          unfold transition. unfold LTS_participant_subset_construction.
          unfold subset_construction_transition_relation.
          split.
          { subst. easy. }
          split.
          { exists s_w'. easy. }
          split.
          { exists s. left. exists s_w'. split. assumption.
            apply Reachable_singleton_inv in H_reach_s. exact H_reach_s. }
          intro s'. split.
          intro. 
          replace (async_to_sync v_x) with (Event p' q' m' ↾ H_neq_x). 
          easy. unfold async_to_sync; subst. simpl. f_equal. apply proof_irrelevance.
          intro. replace (async_to_sync v_x) with (Event p' q' m' ↾ H_neq_x) in H.
          destruct H. left. easy. right.  easy.
          unfold async_to_sync; subst; simpl. f_equal. apply proof_irrelevance.
        *** (* Showing that tis constructed state contains s *)
          left. exists s_w'. split. assumption.
          apply Reachable_singleton_inv in H_reach_s. assumption.
        *** (* Showing that the projections match up *)
          rewrite wproj_split_app. 
          rewrite H_about_v_w'.
          subst. clean.
          rewrite wproj_symbol_sender_neq. easy.
          rewrite wproj_symbol_receiver_eq. easy.
          clean.
          repeat f_equal. apply proof_irrelevance.
    * (* In the case that p is not active in x *)
          exists v_w', ls_v_w'.
          split. assumption.
          assert (H_about_x : wproj (split ([Event p' q' m' ↾ H_neq_x])) p = []).
      { simpl.
        clean. rewrite wproj_symbol_sender_neq. tauto. 
        rewrite wproj_symbol_receiver_neq; tauto. }
      split.
      (* We know that s is in the epsilon-closure of s_w' *)
      assert (H_useful := subset_construction_states_epsilon_closed S p v_w' ls_v_w' H_reach_ls_v_w').
      spec H_useful s_w' s. apply H_useful.
      assumption.
      exists [Event p' q' m' ↾ H_neq_x].
      split. exact H_reach_s.
      assumption.
      rewrite wproj_split_app. rewrite H_about_v_w'.
      rewrite H_about_x. now rewrite app_nil_r.
Qed. 
  
Lemma conjunct4 :
  forall (S : @LTS SyncAlphabet State)
    (p : participant)
    (w : FinSyncWord), 
    (* Every finite word in S has a corresponding finite word in S_p *)
    @is_trace SyncAlphabet State S w →
    @is_trace AsyncAlphabet (State -> Prop) (LTS_participant_subset_construction S p) (wproj (split w) p).  
Proof.
  intros S p w H_trace.
  assert (H_useful := strengthened_conjunct4).
  destruct H_trace as [s_w H_reach_s_w].
  spec H_useful S p w s_w H_reach_s_w.
  destruct H_useful as [v [ls_v [H_reach_v [H_about_ls_v H_about_v]]]]. 
  exists ls_v. rewrite H_about_v. assumption.
Qed. 

(* 2/4 done *)

Lemma conjunct1 : 
  ∀ (S : @LTS SyncAlphabet State)
    (p : participant)
    (w : FinAsyncWord), 
    @is_finite_word AsyncAlphabet (State -> Prop) (LTS_participant_subset_construction S p) w
    → ∃ w' : FinSyncWord,
      @is_finite_word SyncAlphabet State S w'
      ∧ wproj (split w') p = w.  
Proof.    
  intros S p w H_word.
  assert (H_useful := strengthened_conjunct3).
  destruct H_word as [ls_w [H_reach_ls_w H_final_ls_w]].
  unfold final in H_final_ls_w. 
  unfold LTS_participant_subset_construction in H_final_ls_w.
  unfold final_subset_construction_state in H_final_ls_w.
  destruct H_final_ls_w as [_ H_exists_final].
  destruct H_exists_final as [s_fin [H_s_fin_in_ls_w H_fin]].
  spec H_useful S p w ls_w H_reach_ls_w.
  spec H_useful s_fin H_s_fin_in_ls_w.
  destruct H_useful as [v [H_reach_v H_about_v]].
  exists v. split. exists s_fin. split; assumption. assumption.
Qed.

Lemma conjunct2 :
  ∀ (S : @LTS SyncAlphabet State)
    (p : participant)
    (w : FinSyncWord), 
    @is_finite_word SyncAlphabet State S w → 
    @is_finite_word AsyncAlphabet (State -> Prop) (LTS_participant_subset_construction S p) (wproj (split w) p). 
Proof.
  intros S p w H_word.
  assert (H_useful := strengthened_conjunct4).
  destruct H_word as [s_w [H_reach_s_w H_final_s_w]].
  spec H_useful S p w s_w H_reach_s_w.
  destruct H_useful as [v [ls_v [H_reach_v [H_s_w_in_ls_v H_about_v]]]]. 
  exists ls_v. split. rewrite H_about_v. assumption.
  unfold final. unfold LTS_participant_subset_construction.
  unfold final_subset_construction_state. split.
  exists s_w. assumption. exists s_w. easy.
Qed.

(* And now we can prove that the projection LTS is indeed a naive canonical implementation *)

Lemma subset_construction_canonical_naive :
  forall (S : LTS) (p : participant),
    canonical_implementation_local_naive S p (LTS_participant_subset_construction S p). 
Proof.
  intros S p.
  repeat split; intros. 
  - now eapply conjunct1.
  - now eapply conjunct2.
  - now eapply conjunct3.
  - now eapply conjunct4. 
Qed.     

Lemma subset_construction_deterministic :
  forall (S : LTS) (p : participant),
    deterministic (LTS_participant_subset_construction S p).  
Proof.
  intros S p. 
  unfold deterministic.
  intros ls1 ls2 ls3 a H_trans1 H_trans2.
  apply functional_extensionality_dep.
  intro s.
  apply propositional_extensionality.
  split; intro. 
  - (* Proving that ls2 s implies ls3 s *)
    destruct H_trans1 as [_ [_ [_ H_trans1]]]. 
    destruct H_trans2 as [_ [_ [_ H_trans2]]]. 
    apply H_trans2.
    spec H_trans1 s. 
    destruct H_trans1 as [H_trans1 _].
    spec H_trans1 H.
    assumption. 
  - (* Proving that ls2 s implies ls3 s *)
    destruct H_trans1 as [_ [_ [_ H_trans1]]]. 
    destruct H_trans2 as [_ [_ [_ H_trans2]]].
    apply H_trans1.
    spec H_trans2 s. 
    destruct H_trans2 as [H_trans2 _].
    spec H_trans2 H.
    assumption.  
Qed. 

Lemma  subset_construction_canonical_implementation :
  forall (S : LTS) (p : participant),
    deadlock_free S -> 
    canonical_implementation_local S p (LTS_participant_subset_construction S p). 
Proof.
  intros.
  apply canonical_implementation_local_naive_sufficient.
  assumption.
  eapply subset_construction_canonical_naive.
Qed.

Definition canonical_implementation_implementations (S : @LTS SyncAlphabet State) :=
  fun (p : participant) => LTS_participant_subset_construction S p. 

Definition canonical_implementation_c0 (S : @LTS SyncAlphabet State) : @Configuration (State -> Prop) :=
  (fun (p : participant) => s0 (LTS_participant_subset_construction S p),
     fun (p q : participant) => []). 

Program Definition LTS_canonical_implementation (S : @LTS SyncAlphabet State) (H_df : deadlock_free S) : @CLTS (State -> Prop) := 
  mkCLTS (State -> Prop) (canonical_implementation_implementations S) (canonical_implementation_c0 S)_ _ _. 
Next Obligation.
  intros. 
  split. intros. unfold canonical_implementation_implementations. easy.
  intros. unfold canonical_implementation_implementations. easy.
Defined.
Next Obligation.
  intros.
  unfold canonical_implementation_implementations.
  apply subset_construction_deterministic.
Defined.
Next Obligation.
  intros.
  eapply about_subset_construction_alphabet.
  exact H.
Defined.

Lemma canonical_implementation_exists :
  forall (S : @LTS SyncAlphabet State),
    deadlock_free S -> 
  exists (T : CLTS),
    @canonical_implementation (State -> Prop) S T.
Proof. 
  intros S H_df.
  exists (LTS_canonical_implementation S H_df).
  unfold canonical_implementation. intros.
  apply subset_construction_canonical_implementation. assumption.
Qed. 

End Canonical. 
