export async function processToken(token) {
  token = token.replace(/≡/g, "#").trim().toUpperCase();
  if (!token) return null;

  try {
    OCL.Molecule.fromSmiles(token);
    return token;
  } catch (err) {
    console.warn(`Token is not a valid SMILES string: "${token}". Attempting name lookup.`);
  }

  try {
    const smilesFromName = await nameToSMILES_RDKit(token);
    if (smilesFromName) {
      OCL.Molecule.fromSmiles(smilesFromName);
      return smilesFromName;
    }
    console.error(`No SMILES found for token: "${token}"`);
  } catch (lookupError) {
    console.error(`Name lookup failed for "${token}":`, lookupError);
  }

  return null;
}

async function nameToSMILES_RDKit(name) {
  const url = `https://pubchem.ncbi.nlm.nih.gov/rest/pug/compound/name/${encodeURIComponent(name)}/property/CanonicalSMILES/JSON`;
  const response = await fetch(url);
  if (!response.ok) {
    throw new Error(`PubChem lookup failed: ${response.statusText}`);
  }
  const data = await response.json();
  if (
    !data ||
    !data.PropertyTable ||
    !data.PropertyTable.Properties ||
    data.PropertyTable.Properties.length === 0
  ) {
    throw new Error("No SMILES data found for the given name");
  }
  return data.PropertyTable.Properties[0].CanonicalSMILES;
}
