// scene.js
// Sets up Three.js scene, camera, renderer, controls, etc.

export let scene, camera, renderer, controls;
let moleculeGroup = null;

export function getMoleculeGroup() {
  return moleculeGroup;
}
export function setMoleculeGroup(group) {
  moleculeGroup = group;
}





export function initScene() {
  scene = new THREE.Scene();
  // Set a default background for on-screen viewing.
  scene.background = new THREE.Color(0xf0f0f0);

  camera = new THREE.PerspectiveCamera(
    75,
    window.innerWidth / window.innerHeight,
    0.1,
    1000
  );
  camera.position.set(0, 0, 5);

  // Create the renderer with alpha enabled so that we can capture transparent images.
  renderer = new THREE.WebGLRenderer({ antialias: true, alpha: true });
  // When capturing PNG, the clear color (alpha) will be used. Here we set it to fully transparent.
  renderer.setClearColor(0x000000, 0);
  renderer.setSize(window.innerWidth, window.innerHeight);
  renderer.setPixelRatio(window.devicePixelRatio);
  document.body.appendChild(renderer.domElement);

  controls = new THREE.OrbitControls(camera, renderer.domElement);
  controls.enableDamping = true;
  controls.dampingFactor = 0.1;

  // Lights
  const light = new THREE.PointLight(0xffffff, 1.2, 100);
  light.position.set(10, 10, 10);
  scene.add(light);

  scene.add(new THREE.AmbientLight(0x606060));

  // Handle window resize
  window.addEventListener("resize", () => {
    camera.aspect = window.innerWidth / window.innerHeight;
    camera.updateProjectionMatrix();
    renderer.setSize(window.innerWidth, window.innerHeight);
  });
}

export function animate() {
  requestAnimationFrame(animate);
  if (controls) controls.update();
  renderer.render(scene, camera);
}

export function centerMolecule() {
  if (!moleculeGroup) return;
  const box = new THREE.Box3().setFromObject(moleculeGroup);
  const center = box.getCenter(new THREE.Vector3());
  moleculeGroup.position.sub(center);
  camera.lookAt(moleculeGroup.position);

  const sphere = new THREE.Sphere();
  box.getBoundingSphere(sphere);
  
  const fov = camera.fov * Math.PI / 180;
  const marginFactor = 1.2;
  const targetDistance = sphere.radius / Math.sin(fov / 2) * marginFactor;
  
  const currentDistance = camera.position.distanceTo(moleculeGroup.position);
  const diff = Math.abs(targetDistance - currentDistance);
  const threshold = 0.3;
  
  if (diff > threshold) {
    animateCameraZoom(targetDistance);
  }
}

function animateCameraZoom(targetDistance) {
  const currentDistance = camera.position.distanceTo(moleculeGroup.position);
  const diff = targetDistance - currentDistance;
  const steps = 20;
  const stepIncrement = diff / steps;
  let count = 0;
  
  function zoomStep() {
    if (count < steps) {
      const direction = new THREE.Vector3().subVectors(camera.position, moleculeGroup.position).normalize();
      const currentPosDistance = camera.position.distanceTo(moleculeGroup.position);
      const newDistance = currentPosDistance + stepIncrement;
      camera.position.copy(direction.multiplyScalar(newDistance).add(moleculeGroup.position));
      count++;
      requestAnimationFrame(zoomStep);
    } else {
      const direction = new THREE.Vector3().subVectors(camera.position, moleculeGroup.position).normalize();
      camera.position.copy(direction.multiplyScalar(targetDistance).add(moleculeGroup.position));
    }
  }
  zoomStep();
}
