
export function equalizeBondLengthsPreservingAngles(molecule, targetLength) {
  const numAtoms = molecule.getAllAtoms();

  // Store original positions as THREE.Vector3 objects.
  const originalPositions = [];
  for (let i = 0; i < numAtoms; i++) {
    originalPositions.push(new THREE.Vector3(
      molecule.getAtomX(i),
      molecule.getAtomY(i),
      molecule.getAtomZ(i)
    ));
  }

  // Build an adjacency list for the molecule graph.
  const adj = Array.from({ length: numAtoms }, () => []);
  for (let i = 0; i < molecule.getAllBonds(); i++) {
    const a = molecule.getBondAtom(0, i);
    const b = molecule.getBondAtom(1, i);
    adj[a].push(b);
    adj[b].push(a);
  }

  // Create a spanning tree (BFS) from atom 0.
  const tree = Array.from({ length: numAtoms }, () => []);
  const visited = new Array(numAtoms).fill(false);
  const queue = [0];
  visited[0] = true;

  while (queue.length > 0) {
    const current = queue.shift();
    for (const neighbor of adj[current]) {
      if (!visited[neighbor]) {
        visited[neighbor] = true;
        tree[current].push(neighbor);
        queue.push(neighbor);
      }
    }
  }

  // Helper: find the bond index connecting two atoms (returns -1 if not found).
  function getBondIndex(mol, a1, a2) {
    const nbonds = mol.getAllBonds();
    for (let bIdx = 0; bIdx < nbonds; bIdx++) {
      const bA1 = mol.getBondAtom(0, bIdx);
      const bA2 = mol.getBondAtom(1, bIdx);
      if ((bA1 === a1 && bA2 === a2) || (bA1 === a2 && bA2 === a1)) {
        return bIdx;
      }
    }
    return -1;
  }

  // Array to store new positions for atoms (some might remain undefined if disconnected).
  const newPositions = new Array(numAtoms);

  // Fix the root atom (0) at its original position.
  newPositions[0] = originalPositions[0].clone();

  // Recursively assign positions down the BFS tree, preserving directions
  // but adjusting bond lengths depending on single vs. double/triple.
  function assignPositions(parentIndex) {
    for (const childIndex of tree[parentIndex]) {
      // Original direction from parent to child.
      const direction = originalPositions[childIndex]
        .clone()
        .sub(originalPositions[parentIndex])
        .normalize();

      // Default bond length = targetLength for single bonds.
      let bondLength = targetLength;

      // Check if this bond is double/triple, etc.
      const bIdx = getBondIndex(molecule, parentIndex, childIndex);
      if (bIdx >= 0) {
        const order = molecule.getBondOrder(bIdx);
        if (order > 1) {
          // Double or triple => shorten by 0.75×
          bondLength = 0.88 * targetLength;
        }
      }

      // child’s new position = parent’s new position + (bondLength * direction)
      newPositions[childIndex] = newPositions[parentIndex]
        .clone()
        .add(direction.multiplyScalar(bondLength));

      // Recurse down to the child’s children
      assignPositions(childIndex);
    }
  }
  assignPositions(0);

  // Update the molecule with the new positions
  for (let i = 0; i < numAtoms; i++) {
    if (newPositions[i]) {
      molecule.setAtomX(i, newPositions[i].x);
      molecule.setAtomY(i, newPositions[i].y);
      molecule.setAtomZ(i, newPositions[i].z);
    }
  }
}



// -------------------------------
// 3. Returns all single-bond directions (unit vectors) from an atom (ignores double/triple).
// -------------------------------
export function getSingleBondDirections(atomIndex, ignoreBondIndex, molecule) {
  const dirs = [];
  const posA = new THREE.Vector3(
    molecule.getAtomX(atomIndex),
    molecule.getAtomY(atomIndex),
    molecule.getAtomZ(atomIndex)
  );
  for (let b = 0; b < molecule.getAllBonds(); b++) {
    if (b === ignoreBondIndex) continue;
    const a1 = molecule.getBondAtom(0, b);
    const a2 = molecule.getBondAtom(1, b);
    const order = molecule.getBondOrder(b);
    if (order === 1) {  // only single bonds
      let posB;
      if (a1 === atomIndex) {
        posB = new THREE.Vector3(
          molecule.getAtomX(a2),
          molecule.getAtomY(a2),
          molecule.getAtomZ(a2)
        );
      } else if (a2 === atomIndex) {
        posB = new THREE.Vector3(
          molecule.getAtomX(a1),
          molecule.getAtomY(a1),
          molecule.getAtomZ(a1)
        );
      }
      if (posB) {
        const dir = posB.sub(posA).normalize();
        dirs.push(dir);
      }
    }
  }
  return dirs;
}

/**
 * Computes the ideal tetrahedral double-bond offset directions for an atom.
 * The method uses two bond directions attached to the atom (if available) to
 * define a plane through the atom. In that plane, the two ideal double-bond directions are:
 *
 *    offset1 = –(1/√3)*s + (√(2/3))*n  
 *    offset2 = –(1/√3)*s – (√(2/3))*n  
 *
 * where s is the normalized sum of the two chosen bond directions (pointing toward their mid–point)
 * and n is the normalized vector perpendicular to that plane.
 */

export function getDoubleBondOffsetTetrahedral(atomIndex, molecule, spacing, currentBondPartner = null) {
  let bonds = [];
  // Gather all bonds attached to atomIndex (regardless of order).
  for (let b = 0; b < molecule.getAllBonds(); b++) {
    const a1 = molecule.getBondAtom(0, b);
    const a2 = molecule.getBondAtom(1, b);
    let otherAtom = null;
    if (a1 === atomIndex) {
      otherAtom = a2;
    } else if (a2 === atomIndex) {
      otherAtom = a1;
    }
    if (otherAtom !== null) {
      const posAtom = new THREE.Vector3(
        molecule.getAtomX(atomIndex),
        molecule.getAtomY(atomIndex),
        molecule.getAtomZ(atomIndex)
      );
      const posOther = new THREE.Vector3(
        molecule.getAtomX(otherAtom),
        molecule.getAtomY(otherAtom),
        molecule.getAtomZ(otherAtom)
      );
      let dir = posOther.clone().sub(posAtom).normalize();
      bonds.push({ other: otherAtom, dir: dir });
    }
  }
  
  // If a current bond partner is given and the atom has more than one bond,
  // try to exclude it—unless that would leave us with no bonds.
  if (currentBondPartner !== null && bonds.length > 1) {
    let filtered = bonds.filter(b => b.other !== currentBondPartner);
    if (filtered.length >= 1) {
      // If filtering gives at least one, use the filtered list if it has two or more.
      if (filtered.length >= 2) {
        bonds = filtered;
      }
      // Otherwise (only one left), keep both so the current bond is also used.
    }
  }
  
  // Choose two directions.
  let v1, v2;
  if (bonds.length >= 2) {
    v1 = bonds[0].dir.clone();
    v2 = bonds[1].dir.clone();
  } else if (bonds.length === 1) {
    v1 = bonds[0].dir.clone();
    // Pick an arbitrary vector perpendicular to v1.
    let arbitrary = new THREE.Vector3(0, 1, 0);
    if (Math.abs(v1.dot(arbitrary)) > 0.9) {
      arbitrary.set(1, 0, 0);
    }
    v2 = new THREE.Vector3().crossVectors(v1, arbitrary).normalize();
  } else {
    // Fallback: use two default perpendicular directions.
    v1 = new THREE.Vector3(1, 0, 0);
    v2 = new THREE.Vector3(0, 1, 0);
  }
  
  // s: direction toward the midpoint of the two bonds.
  let s = v1.clone().add(v2).normalize();
  // n: perpendicular to the plane defined by v1 and v2.
  let n = new THREE.Vector3().crossVectors(v1, v2).normalize();
  
  // Ideal tetrahedral factors.
  const factor1 = -1 / Math.sqrt(3);
  const factor2 = Math.sqrt(2 / 3);
  
  let offset1 = s.clone().multiplyScalar(factor1)
                    .add(n.clone().multiplyScalar(factor2))
                    .normalize().multiplyScalar(spacing);
  let offset2 = s.clone().multiplyScalar(factor1)
                    .add(n.clone().multiplyScalar(-factor2))
                    .normalize().multiplyScalar(spacing);
  return { offset1, offset2 };
}


// Returns 4 default tetrahedral directions (approximately 109.5° apart)
function getTetrahedralDirections() {
  const vectors = [
    new THREE.Vector3( 1,  1,  1),
    new THREE.Vector3( 1, -1, -1),
    new THREE.Vector3(-1,  1, -1),
    new THREE.Vector3(-1, -1,  1)
  ];
  for (let v of vectors) {
    v.normalize();
  }
  return vectors;
}


// Computes the rotated tetrahedral directions for an atom based on its actual single bonds.
// This is essentially the same rotation used for your tube visualization.
function getRotatedTetrahedralDirections(atomIndex, molecule) {
  const defaultDirs = getTetrahedralDirections();
  const bondDirs = getSingleBondDirections(atomIndex, -1, molecule);
  let Q = new THREE.Quaternion();
  if (bondDirs.length === 0) {
    Q.identity();
  } else if (bondDirs.length === 1) {
    Q.setFromUnitVectors(defaultDirs[0].clone(), bondDirs[0].clone().normalize());
  } else {
    let bestDot0 = -Infinity, bestBondIndex0 = -1, bestDefaultIndex0 = -1;
    let bestDot1 = -Infinity, bestBondIndex1 = -1, bestDefaultIndex1 = -1;
    for (let i = 0; i < bondDirs.length; i++) {
      const bDir = bondDirs[i].clone().normalize();
      for (let j = 0; j < defaultDirs.length; j++) {
        const d = defaultDirs[j].clone();
        const dot = bDir.dot(d);
        if (dot > bestDot0) {
          bestDot0 = dot;
          bestBondIndex0 = i;
          bestDefaultIndex0 = j;
        }
      }
    }
    for (let i = 0; i < bondDirs.length; i++) {
      if (i === bestBondIndex0) continue;
      const bDir = bondDirs[i].clone().normalize();
      for (let j = 0; j < defaultDirs.length; j++) {
        if (j === bestDefaultIndex0) continue;
        const d = defaultDirs[j].clone();
        const dot = bDir.dot(d);
        if (dot > bestDot1) {
          bestDot1 = dot;
          bestBondIndex1 = i;
          bestDefaultIndex1 = j;
        }
      }
    }
    const Q0 = new THREE.Quaternion().setFromUnitVectors(
      defaultDirs[bestDefaultIndex0].clone(),
      bondDirs[bestBondIndex0].clone().normalize()
    );
    const d1Rotated = defaultDirs[bestDefaultIndex1].clone().applyQuaternion(Q0);
    const axis = bondDirs[bestBondIndex0].clone().normalize();
    function projectOntoPlane(v, axis) {
      return v.clone().sub(axis.clone().multiplyScalar(v.dot(axis))).normalize();
    }
    const projD1 = projectOntoPlane(d1Rotated, axis);
    const projB1 = projectOntoPlane(bondDirs[bestBondIndex1].clone().normalize(), axis);
    let angle = Math.acos(THREE.MathUtils.clamp(projD1.dot(projB1), -1, 1));
    const cross = new THREE.Vector3().crossVectors(projD1, projB1);
    if (cross.dot(axis) < 0) {
      angle = -angle;
    }
    const Q1 = new THREE.Quaternion().setFromAxisAngle(axis, angle);
    Q.multiplyQuaternions(Q1, Q0);
  }
  const rotatedDirs = defaultDirs.map(dir => dir.clone().applyQuaternion(Q));
  return rotatedDirs;
}
function getSpExtraBondOffset(mainAxis, chooseAlternate = false) {
  // Choose an arbitrary vector not parallel to mainAxis.
  const defaultVec = Math.abs(mainAxis.dot(new THREE.Vector3(0,1,0))) < 0.99 
                       ? new THREE.Vector3(0,1,0) 
                       : new THREE.Vector3(1,0,0);
  const perp1 = new THREE.Vector3().crossVectors(mainAxis, defaultVec).normalize();
  if (chooseAlternate) {
    const rot90 = new THREE.Quaternion().setFromAxisAngle(mainAxis, THREE.MathUtils.degToRad(90));
    return perp1.clone().applyQuaternion(rot90);
  } else {
    return perp1;
  }
}

export function getExtraBondOffset(atomIndex, mainAxis, molecule) {
  const singleBondDirs = getSingleBondDirections(atomIndex, -1, molecule);
  if(singleBondDirs.length === 0) {
    // For atoms with no single bonds (likely sp, as in the central carbon of C=C=C)
    const numBonds = molecule.getAllConnAtoms(atomIndex);
    if(numBonds === 2) {
      // Alternate the extra offset based on the sign of mainAxis.x (for example)
      if(mainAxis.x >= 0) {
        return getSpExtraBondOffset(mainAxis, false);
      } else {
        return getSpExtraBondOffset(mainAxis, true);
      }
    }
    return getSpExtraBondOffset(mainAxis, false);
  }
  const tetDirs = getRotatedTetrahedralDirections(atomIndex, molecule);
  let bestDir = null;
  let bestVal = Infinity;
  for(let d of tetDirs) {
    const dp = Math.abs(d.dot(mainAxis));
    if(dp < bestVal) {
      bestVal = dp;
      bestDir = d.clone();
    }
  }
  return bestDir;
}

