// molecule-utils.js
// Contains CPK colors, van der Waals radii, and helper functions for molecule construction.

export const cpk_colors = {
  H:  [1.00, 1.00, 1.00, 1.00],
  He: [0.85, 1.00, 1.00, 1.00],
  Li: [0.80, 0.50, 1.00, 1.00],
  Be: [0.76, 1.00, 0.00, 1.00],
  B:  [1.00, 0.70, 0.70, 1.00],
  C:  [0.00, 0.00, 0.00, 1.00],
  N:  [0.00, 0.00, 1.00, 1.00],
  O:  [1.00, 0.00, 0.00, 1.00],
  F:  [0.50, 1.00, 0.50, 1.00],
  Ne: [0.70, 0.89, 0.96, 1.00],
  Na: [0.67, 0.36, 0.95, 1.00],
  Mg: [0.54, 1.00, 0.00, 1.00],
  Al: [0.75, 0.65, 0.65, 1.00],
  Si: [0.94, 0.78, 0.63, 1.00],
  P:  [1.00, 0.50, 0.00, 1.00],
  S:  [1.00, 1.00, 0.19, 1.00],
  Cl: [0.12, 0.94, 0.12, 1.00],
  Ar: [0.50, 0.82, 0.89, 1.00],
  K:  [0.56, 0.25, 0.83, 1.00],
  Ca: [0.24, 1.00, 0.00, 1.00],
  Sc: [0.90, 0.90, 0.90, 1.00],
  Ti: [0.75, 0.76, 0.78, 1.00],
  V:  [0.65, 0.65, 0.67, 1.00],
  Cr: [0.54, 0.60, 0.78, 1.00],
  Mn: [0.61, 0.48, 0.78, 1.00],
  Fe: [0.88, 0.40, 0.20, 1.00],
  Co: [0.94, 0.56, 0.63, 1.00],
  Ni: [0.31, 0.82, 0.31, 1.00],
  Cu: [0.79, 0.50, 0.20, 1.00],
  Zn: [0.49, 0.50, 0.69, 1.00],
  Ga: [0.76, 0.56, 0.56, 1.00],
  Ge: [0.40, 0.56, 0.56, 1.00],
  As: [0.74, 0.50, 0.89, 1.00],
  Se: [1.00, 0.63, 0.00, 1.00],
  Br: [0.65, 0.16, 0.16, 1.00],
  Kr: [0.36, 0.72, 0.82, 1.00],
  I:  [0.58, 0.00, 0.58, 1.00],
};

export const vdw_radii = {
  H: 1.20,
  C: 1.70,
  N: 1.55,
  O: 1.52,
  S: 1.80,
  P: 1.80,
  Cl: 1.75,
  F: 1.47,
  Br: 1.85,
  I: 1.98,
  Fe: 2.00,
  Na: 2.27,
  K:  2.75,
  Ca: 2.00,
};

export function getCPKColor(elementSymbol) {
  const key = elementSymbol.toUpperCase();
  if (!cpk_colors[key]) {
    return new THREE.Color(0.8, 0.8, 0.8);
  }
  const [r, g, b] = cpk_colors[key];
  return new THREE.Color(r, g, b);
}

export function createCurvedBondMesh(start, end, offset, radius, material) {
  const mid = new THREE.Vector3().addVectors(start, end).multiplyScalar(0.5);
  const control = new THREE.Vector3().copy(mid).add(offset.clone().multiplyScalar(4));
  const curve = new THREE.QuadraticBezierCurve3(
    start.clone().add(offset),
    control,
    end.clone().add(offset)
  );
  const tubeGeometry = new THREE.TubeGeometry(curve, 20, radius, 8, false);
  if (!material) {
    material = new THREE.MeshStandardMaterial({ color: 0xaaaaaa });
  }
  return new THREE.Mesh(tubeGeometry, material);
}

export function createCurvedRidgedBondMeshRelaxed(p1, p2, mid, segment = 16, radius = 0.06, color = 0xaaaaaa) {
  const waveCount = 25;
  const waveAmplitude = 0.01;
  const smoothRatio = 0.2;

  const curve = new THREE.QuadraticBezierCurve3(p1.clone(), mid.clone(), p2.clone());

  function ridgedCurve(u, v, target) {
    const positionOnCurve = curve.getPointAt(u);
    const tangent = curve.getTangentAt(u).normalize();
    let reference = new THREE.Vector3(0, 1, 0);
    if (Math.abs(tangent.dot(reference)) > 0.94) {
      reference.set(1, 0, 0);
    }
    const normal = new THREE.Vector3().crossVectors(tangent, reference).normalize();
    const binormal = new THREE.Vector3().crossVectors(tangent, normal).normalize();
    const useSmoothTube = (u < smoothRatio || u > 1 - smoothRatio);
    const angle = v * 2 * Math.PI;
    const r = useSmoothTube ? radius * 1.5 : (radius + waveAmplitude * Math.sin(waveCount * 2 * Math.PI * u));
    const x = r * Math.cos(angle);
    const y = r * Math.sin(angle);
    const finalPosition = positionOnCurve.clone()
      .addScaledVector(binormal, x)
      .addScaledVector(normal, y);
    target.set(finalPosition.x, finalPosition.y, finalPosition.z);
  }

  const radialSegments = 20;
  const lengthSegments = 100;
  const paramGeom = new THREE.ParametricGeometry(ridgedCurve, lengthSegments, radialSegments);
  paramGeom.computeVertexNormals();

  const materialObj = new THREE.MeshStandardMaterial({
    color: 0x808080,
    metalness: 0.0,
    roughness: 0.5,
    side: THREE.DoubleSide
  });

  return new THREE.Mesh(paramGeom, materialObj);
}

// Helper Functions

export function createTextLabel(text) {
  const canvas = document.createElement("canvas");
  const context = canvas.getContext("2d");
  canvas.width = 256;
  canvas.height = 64;

  let fontSize = 24;
  context.font = `${fontSize}px Arial`;

  const maxWidth = canvas.width * 0.9;
  let measuredWidth = context.measureText(text).width;

  if (measuredWidth > maxWidth) {
    fontSize = fontSize * (maxWidth / measuredWidth);
    context.font = `${fontSize}px Arial`;
  }

  context.fillStyle = "black";
  context.textAlign = "center";
  context.textBaseline = "middle";
  context.clearRect(0, 0, canvas.width, canvas.height);
  context.fillText(text, canvas.width / 2, canvas.height / 2);

  const texture = new THREE.CanvasTexture(canvas);
  texture.minFilter = THREE.LinearFilter;
  const material = new THREE.SpriteMaterial({ map: texture, transparent: true });
  const sprite = new THREE.Sprite(material);
  sprite.scale.set(2.5, 0.6, 1);
  return sprite;
}

export function rotateVector(vec, axis, angle) {
  const rotated = vec.clone();
  rotated.applyAxisAngle(axis.clone().normalize(), angle);
  return rotated;
}

export function adjustBondEndpoints(p1, p2, r1, r2) {
  const d = new THREE.Vector3().subVectors(p2, p1).normalize();
  const newP1 = p1.clone().add(d.clone().multiplyScalar(r1));
  const newP2 = p2.clone().add(d.clone().multiplyScalar(-r2));
  return [newP1, newP2];
}

const ATOM_RADIUS = 0.14;
export function getAtomVisualRadius(atomSymbol) {
  return atomSymbol === 'H' ? ATOM_RADIUS * 0.6 : ATOM_RADIUS;
}

export function calculateDoubleBondMidpoint(p1, p2, center1, center2, offsetScale = 0.3) {
  const midCenter = center1.clone().add(center2).multiplyScalar(0.5);
  const midBond = p1.clone().add(p2).multiplyScalar(0.5);
  const direction = midBond.clone().sub(midCenter).normalize();
  return midBond.clone().add(direction.multiplyScalar(offsetScale));
}

export function quadraticBezier(p0, p1, p2, t) {
  const oneMinusT = 1 - t;
  return p0.clone().multiplyScalar(oneMinusT * oneMinusT)
    .add(p1.clone().multiplyScalar(2 * oneMinusT * t))
    .add(p2.clone().multiplyScalar(t * t));
}

export function createCylinderBetweenPoints(point1, point2, radius, color) {
  const direction = new THREE.Vector3().subVectors(point2, point1);
  const length = direction.length();
  const geometry = new THREE.CylinderGeometry(radius, radius, length, 8, 1);
  const material = new THREE.MeshStandardMaterial({ color });
  const cylinder = new THREE.Mesh(geometry, material);
  cylinder.position.copy(point1.clone().add(direction.multiplyScalar(0.5)));
  const axis = new THREE.Vector3(0, 1, 0);
  cylinder.quaternion.setFromUnitVectors(axis, direction.clone().normalize());
  return cylinder;
}

export function drawCurvedBondRelaxed(p1, p2, mid, steps, radius = 0.06, color = 0xaaaaaa) {
  const bondGroup = new THREE.Group();
  for (let i = 0; i < steps; i++) {
    const t1 = i / steps;
    const t2 = (i + 1) / steps;
    const point1 = quadraticBezier(p1, mid, p2, t1);
    const point2 = quadraticBezier(p1, mid, p2, t2);
    const segment = createCylinderBetweenPoints(point1, point2, radius, color);
    bondGroup.add(segment);
  }
  return bondGroup;
}

export function forceRotateTerminalHydrogen(molecule, carbonIndex, axis, angle = Math.PI / 2) {
  const carbonPos = new THREE.Vector3(
    molecule.getAtomX(carbonIndex),
    molecule.getAtomY(carbonIndex),
    molecule.getAtomZ(carbonIndex)
  );
  const numNeighbors = molecule.getConnAtoms(carbonIndex);
  for (let i = 0; i < numNeighbors; i++) {
    const neighborIndex = molecule.getConnAtom(carbonIndex, i);
    if (molecule.getAtomLabel(neighborIndex) === 'H') {
      const hydrogenPos = new THREE.Vector3(
        molecule.getAtomX(neighborIndex),
        molecule.getAtomY(neighborIndex),
        molecule.getAtomZ(neighborIndex)
      );
      const hVec = hydrogenPos.clone().sub(carbonPos);
      const hVecRotated = rotateVector(hVec, axis, angle);
      const newPos = carbonPos.clone().add(hVecRotated);
      if (typeof molecule.setAtomPosition === 'function') {
        molecule.setAtomPosition(neighborIndex, newPos.x, newPos.y, newPos.z);
      } else {
        console.warn("setAtomPosition not implemented; hydrogen", neighborIndex, "may not update correctly.");
      }
    }
  }
}
