import { getMoleculeGroup } from "./scene.js";
import { renderer, scene, camera } from "./scene.js";
import { getLastValidSmilesArray } from "./state.js";
import { createMoleculeGroupFromSmiles } from "./molecule-generator.js";
import { getCurrentSmiles } from "./state.js";

export function downloadGLTF() {
  const mg = getMoleculeGroup();
  if (!mg) {
    alert("No molecule to download!");
    return;
  }
  const exporter = new THREE.GLTFExporter();
  exporter.parse(
    mg,
    gltf => {
      const blob = new Blob([JSON.stringify(gltf)], { type: "application/json" });
      const link = document.createElement("a");
      link.href = URL.createObjectURL(blob);
      link.download = getCurrentSmiles() ? getCurrentSmiles() + ".gltf" : "molecules.gltf";
      link.click();
      URL.revokeObjectURL(link.href);
    },
    { binary: false }
  );
}

export function downloadUSDZ() {
  const mg = getMoleculeGroup();
  if (!mg) {
    alert("No molecule to download!");
    return;
  }
  const exporter = new THREE.USDZExporter();

  // USDZExporter returns a Promise with an ArrayBuffer.
  exporter.parse(mg).then((usdzArrayBuffer) => {
    const blob = new Blob([usdzArrayBuffer], { type: "model/vnd.usdz+zip" });
    const link = document.createElement("a");
    link.href = URL.createObjectURL(blob);
    link.download = getCurrentSmiles() ? getCurrentSmiles() + ".usdz" : "molecule.usdz";
    link.click();
    URL.revokeObjectURL(link.href);
  });
}


export function downloadImage() {
  if (!renderer) {
    alert("Renderer not ready!");
    return;
  }
  const originalBackground = scene.background;
  scene.background = null;
  renderer.render(scene, camera);
  const dataURL = renderer.domElement.toDataURL("image/png");
  scene.background = originalBackground;

  const link = document.createElement("a");
  link.href = dataURL;
  link.download = getCurrentSmiles() ? getCurrentSmiles() + ".png" : "molecules.png";
  link.click();
}


export async function downloadZip() {
  if (!getLastValidSmilesArray() || getLastValidSmilesArray().length === 0) {
    alert("No molecule to download!");
    return;
  }
  const zip = new JSZip();
  const representations = ["stick", "spacefilling", "building kit"];
  for (let i = 0; i < getLastValidSmilesArray().length; i++) {
    const smiles = getLastValidSmilesArray()[i];
    const folderName = smiles.replace(/[^a-zA-Z0-9]/g, '_');
    const folder = zip.folder(folderName);
    for (const rep of representations) {
      const generatedMolecule = await createMoleculeGroupFromSmiles(smiles, rep);
      const group = generatedMolecule[0];
      const usdzString = await exportUSDZAsync(group);
      const fileName = `${folderName}_${rep}.usdz`;
      folder.file(fileName, usdzString);
    }
  } 

  zip.generateAsync({ type: "blob" }).then(function(content) {
    const link = document.createElement("a");
    link.href = URL.createObjectURL(content);
    link.download = "molecules.zip";
    link.click();
    URL.revokeObjectURL(link.href);
  });
}


export async function exportUSDZAsync(object) {
  const exporter = new THREE.USDZExporter();
  // exporter.parse returns a promise that resolves with a Blob or ArrayBuffer.
  const usdzData = await exporter.parse(object);
  return usdzData;
}

