import tkinter as tk
from tkinter import ttk, filedialog, messagebox, scrolledtext, simpledialog
from PIL import Image, ImageTk
import os
import pickle
import re
import threading
import random
from rdflib import Graph, Namespace, Literal
from rdflib.namespace import RDF

try:
    import master_generator
except ImportError:
    master_generator = None

# --- Classi del Modello Dati ---
TPIX = Namespace("http://tripix.org/schema/")
INST = Namespace("http://tripix.org/data/scene/")
def create_safe_uri_string(text):
    return re.sub(r'[^a-zA-Z0-9_]', '_', text)

class Asset:
    def __init__(self, filepath):
        self.filepath = filepath
        self.id = create_safe_uri_string(os.path.basename(filepath))
        self.image = Image.open(filepath).convert("RGBA")
        self.name = os.path.splitext(os.path.basename(filepath))[0]

class SceneObject:
    def __init__(self, asset, x, y):
        safe_name = create_safe_uri_string(asset.name)
        self.id = f"{safe_name.lower()}_{random.randint(1000, 9999)}"
        self.asset_id, self.x, self.y = asset.id, x, y
        self.width, self.height = asset.image.width, asset.image.height
    def get_bbox(self): return (self.x, self.y, self.x + self.width, self.y + self.height)

class Scene:
    def __init__(self, name="Nuova Scena"): self.name, self.objects = name, []
    def add_object(self, obj): self.objects.append(obj)
    def remove_object(self, obj): self.objects.remove(obj)
    def to_rdf_graph(self, assets_dict):
        g = Graph(); g.bind("tpix", TPIX); g.bind("inst", INST)
        for obj in self.objects:
            obj_uri = INST[obj.id]; asset = assets_dict.get(obj.asset_id)
            if asset:
                safe_class_name = create_safe_uri_string(asset.name.capitalize())
                g.add((obj_uri, RDF.type, TPIX[safe_class_name]))
            g.add((obj_uri, TPIX.hasPosition, Literal(f"({obj.x}, {obj.y})")))
            g.add((obj_uri, TPIX.hasSize, Literal(f"({obj.width}, {obj.height})")))
        for i in range(len(self.objects)):
            for j in range(i + 1, len(self.objects)):
                g.add((INST[self.objects[i].id], TPIX.isOccludedBy, INST[self.objects[j].id]))
        return g

class Project:
    def __init__(self): self.assets, self.scenes, self.active_scene_index = {}, [Scene()], 0
    def add_asset(self, filepath):
        try:
            asset = Asset(filepath)
            if asset.id not in self.assets: self.assets[asset.id] = asset
            return self.assets[asset.id]
        except Exception as e: messagebox.showerror("Errore", f"Impossibile caricare: {e}")
    def get_active_scene(self):
        if not self.scenes: self.new_scene()
        return self.scenes[self.active_scene_index]
    def new_scene(self):
        scene_name = f"Scena {len(self.scenes) + 1}"; self.scenes.append(Scene(name=scene_name))
        self.active_scene_index = len(self.scenes) - 1

class GeneratorWindow(tk.Toplevel):
    def __init__(self, parent, project_assets):
        super().__init__(parent); self.parent = parent; self.project_assets = project_assets
        self.title("Generatore Dataset Automatico"); self.geometry("500x550"); self.create_widgets()
    def create_widgets(self):
        frame = ttk.Frame(self, padding="10"); frame.pack(fill=tk.BOTH, expand=True)
        self.entries = {}; params = {"num_scenes": "Numero Scene Base:", "min_objects": "Min Oggetti/Scena:", "max_objects": "Max Oggetti/Scena:", "rot_step": "Passo Rotazione (gradi):"}
        for i, (key, label) in enumerate(params.items()):
            ttk.Label(frame, text=label).grid(row=i, column=0, sticky=tk.W, pady=2)
            self.entries[key] = ttk.Entry(frame); self.entries[key].grid(row=i, column=1, sticky=tk.EW, pady=2)
            defaults = {"num_scenes": "10", "min_objects": "2", "max_objects": "4", "rot_step": "90"}; self.entries[key].insert(0, defaults.get(key, ""))
        self.bool_vars = {}; bool_params = {"allow_occlusion": "Permetti Occlusione", "allow_shear": "Permetti Deformazione"}
        for i, (key, label) in enumerate(bool_params.items(), start=len(params)):
            self.bool_vars[key] = tk.BooleanVar(value=True); ttk.Checkbutton(frame, text=label, variable=self.bool_vars[key]).grid(row=i, column=0, columnspan=2, sticky=tk.W, pady=5)
        log_frame = ttk.LabelFrame(frame, text="Log di Generazione", padding=5); log_frame.grid(row=len(params)+len(bool_params), column=0, columnspan=2, sticky="nsew", pady=10)
        self.log_text = scrolledtext.ScrolledText(log_frame, height=8, wrap=tk.WORD, state=tk.DISABLED); self.log_text.pack(fill=tk.BOTH, expand=True)
        frame.rowconfigure(len(params)+len(bool_params), weight=1)
        self.run_button = ttk.Button(frame, text="Avvia Generazione", command=self.run_generation_thread); self.run_button.grid(row=len(params)+len(bool_params)+1, column=0, columnspan=2, pady=10)
    def log(self, message):
        self.log_text.config(state=tk.NORMAL); self.log_text.insert(tk.END, str(message) + "\n"); self.log_text.see(tk.END); self.log_text.config(state=tk.DISABLED); self.update_idletasks()
    def run_generation_thread(self):
        if not master_generator: self.log("ERRORE: Modulo 'master_generator.py' non trovato."); return
        self.run_button.config(state=tk.DISABLED); self.log_text.config(state=tk.NORMAL); self.log_text.delete("1.0", tk.END); self.log_text.config(state=tk.DISABLED)
        try:
            # --- BLOCCO DI CODICE CORRETTO ---
            assets_for_generator = {}
            for asset in self.project_assets.values():
                # La chiave del dizionario deve essere il nome pulito dell'asset (es. 'dog')
                safe_name_key = create_safe_uri_string(asset.name)
                # Il valore deve essere un dizionario con 'filepath' e 'class'
                assets_for_generator[safe_name_key] = {
                    "filepath": asset.filepath,
                    "class": asset.name.capitalize() # Es. 'Dog', 'Tree'
                }

            config = {
                "num_scenes": int(self.entries["num_scenes"].get()),
                "rot_step": int(self.entries["rot_step"].get()),
                "min_objects": int(self.entries["min_objects"].get()),
                "max_objects": int(self.entries["max_objects"].get()),
                "allow_occlusion": self.bool_vars["allow_occlusion"].get(),
                "allow_shear": self.bool_vars["allow_shear"].get(),
                "brightness_levels": [0.7, 1.0, 1.3],
                "scale_levels": [0.8, 1.0, 1.2],
                "OUTPUT_BASE_DIR": "generated_dataset",
                "ASSETS": assets_for_generator, # <-- Usa la struttura corretta
                "TPIX": TPIX,
                "INST": Namespace("http://tripix.org/data/generated/"),
                "SCENE_WIDTH": 800,
                "SCENE_HEIGHT": 600,
            }
            
            
        except ValueError: self.log("ERRORE: Campi numerici non validi."); self.run_button.config(state=tk.NORMAL); return
        thread = threading.Thread(target=self.generation_task, args=(config,)); thread.start()
    def generation_task(self, config):
        try:
            summary = master_generator.run_generation(config, status_callback=self.log)
            messagebox.showinfo("Completato", summary, parent=self)
        except Exception as e: self.log(f"ERRORE FATALE: {e}"); messagebox.showerror("Errore", f"La generazione è fallita.\n\n{e}", parent=self)
        finally: self.run_button.config(state=tk.NORMAL)

class App(tk.Tk):
    # ... (Il resto del codice è invariato) ...
    def __init__(self):
        super().__init__(); self.title("TRIPIX Weaver Studio"); self.geometry("1600x900")
        self.project = Project(); self.drag_info, self.selected_object_index = None, None
        self.create_menu(); self.create_widgets(); self.bind_events()
    def create_menu(self):
        self.menu_bar = tk.Menu(self); self.config(menu=self.menu_bar)
        file_menu = tk.Menu(self.menu_bar, tearoff=0); self.menu_bar.add_cascade(label="File", menu=file_menu)
        file_menu.add_command(label="Nuovo Progetto", command=self.new_project); file_menu.add_command(label="Apri Progetto...", command=self.open_project); file_menu.add_command(label="Salva Progetto...", command=self.save_project); file_menu.add_separator()
        file_menu.add_command(label="Esporta Scena come PNG...", command=lambda: self.export_scene('png')); file_menu.add_command(label="Esporta Scena come JPG...", command=lambda: self.export_scene('jpg')); file_menu.add_command(label="Esporta Semantica Scena (RDF)...", command=self.export_scene_rdf); file_menu.add_separator(); file_menu.add_command(label="Esci", command=self.quit)
        self.scene_menu = tk.Menu(self.menu_bar, tearoff=0); self.menu_bar.add_cascade(label="Scene", menu=self.scene_menu); self.update_scene_menu()
        tools_menu = tk.Menu(self.menu_bar, tearoff=0); self.menu_bar.add_cascade(label="Tools", menu=tools_menu); tools_menu.add_command(label="Generatore Dataset...", command=self.open_generator_window)
        help_menu = tk.Menu(self.menu_bar, tearoff=0); self.menu_bar.add_cascade(label="Help", menu=help_menu); help_menu.add_command(label="About", command=self.show_about)
    def create_widgets(self):
        main_frame = ttk.Frame(self, padding=5); main_frame.pack(fill=tk.BOTH, expand=True)
        left_panel = ttk.LabelFrame(main_frame, text="Libreria Asset", padding=10, width=250); left_panel.pack(side=tk.LEFT, fill=tk.Y, padx=(0, 10)); left_panel.pack_propagate(False)
        add_asset_button = ttk.Button(left_panel, text="Carica Asset...", command=self.load_assets); add_asset_button.pack(pady=5, fill=tk.X)
        self.asset_canvas = tk.Canvas(left_panel, bg="lightgrey"); self.asset_canvas.pack(fill=tk.BOTH, expand=True)
        self.scene_canvas = tk.Canvas(main_frame, bg="white"); self.scene_canvas.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        right_panel = ttk.LabelFrame(main_frame, text="Grafo Semantico della Scena (RDF)", padding=10, width=400); right_panel.pack(side=tk.RIGHT, fill=tk.Y, padx=(10, 0)); right_panel.pack_propagate(False)
        self.rdf_viewer = scrolledtext.ScrolledText(right_panel, wrap=tk.WORD, state=tk.DISABLED, font=("Courier New", 9)); self.rdf_viewer.pack(fill=tk.BOTH, expand=True)
        self.draw_scene()
    def bind_events(self): self.asset_canvas.bind("<Button-1>", self.on_asset_press); self.scene_canvas.bind("<Button-1>", self.on_scene_press); self.bind("<B1-Motion>", self.on_drag); self.bind("<ButtonRelease-1>", self.on_release); self.bind("<Delete>", self.on_delete_key)
    def update_asset_library(self):
        self.asset_canvas.delete("all"); self.thumbnail_images = []
        x, y, max_x = 10, 10, 240 - 80
        for asset_id, asset in self.project.assets.items():
            thumb = asset.image.copy(); thumb.thumbnail((60, 60)); img = ImageTk.PhotoImage(thumb); self.thumbnail_images.append(img); tag = f"asset_{asset_id}"
            self.asset_canvas.create_image(x, y, anchor=tk.NW, image=img, tags=tag); self.asset_canvas.create_text(x + 30, y + 70, text=asset.name, width=60, anchor=tk.CENTER)
            x += 80
            if x > max_x: x, y = 10, y + 90
    def new_project(self):
        if messagebox.askokcancel("Nuovo", "Cancellare progetto?"): self.project = Project(); self.update_asset_library(); self.update_scene_menu(); self.draw_scene()
    def save_project(self):
        filepath = filedialog.asksaveasfilename(defaultextension=".trp", filetypes=[("TRIPIX Project", "*.trp")]);
        if not filepath: return
        try:
            with open(filepath, "wb") as f: pickle.dump(self.project, f); messagebox.showinfo("Salva", "Progetto salvato!")
        except Exception as e: messagebox.showerror("Errore", f"Errore: {e}")
    def open_project(self):
        filepath = filedialog.askopenfilename(filetypes=[("TRIPIX Project", "*.trp")]);
        if not filepath: return
        try:
            with open(filepath, "rb") as f: self.project = pickle.load(f)
            self.update_asset_library(); self.update_scene_menu(); self.draw_scene(); messagebox.showinfo("Apri", "Progetto caricato!")
        except Exception as e: messagebox.showerror("Errore", f"Errore: {e}")
    def update_scene_menu(self):
        self.scene_menu.delete(0, tk.END)
        self.scene_variable = tk.StringVar(value=self.project.get_active_scene().name)
        for i, scene in enumerate(self.project.scenes): self.scene_menu.add_radiobutton(label=scene.name, variable=self.scene_variable, value=scene.name, command=lambda i=i: self.select_scene(i))
        self.scene_menu.add_separator(); self.scene_menu.add_command(label="Nuova Scena...", command=self.add_new_scene)
    def add_new_scene(self): self.project.new_scene(); self.update_scene_menu(); self.draw_scene()
    def select_scene(self, index): self.project.active_scene_index = index; self.draw_scene()
    def export_scene(self, file_format):
        scene = self.project.get_active_scene();
        if not scene.objects: messagebox.showwarning("Scena Vuota", "Impossibile esportare."); return
        final_image = Image.new("RGBA", (self.scene_canvas.winfo_width(), self.scene_canvas.winfo_height()), (0,0,0,0))
        for scene_obj in scene.objects:
            asset = self.project.assets.get(scene_obj.asset_id)
            if asset: final_image.paste(asset.image.copy().resize((scene_obj.width, scene_obj.height)), (scene_obj.x, scene_obj.y), mask=asset.image.copy().resize((scene_obj.width, scene_obj.height)))
        filepath = None
        if file_format == 'png': filepath = filedialog.asksaveasfilename(defaultextension=".png", filetypes=[("PNG", "*.png")]);
        elif file_format == 'jpg': filepath = filedialog.asksaveasfilename(defaultextension=".jpg", filetypes=[("JPEG", "*.jpg")]);
        if not filepath: return
        if file_format == 'jpg': background = Image.new("RGBA", final_image.size, (255,255,255,255)); final_image = Image.alpha_composite(background, final_image).convert("RGB")
        final_image.save(filepath); messagebox.showinfo("Esporta", "Scena esportata!")
    def export_scene_rdf(self):
        scene = self.project.get_active_scene();
        if not scene.objects: messagebox.showwarning("Scena Vuota", "Impossibile esportare."); return
        filepath = filedialog.asksaveasfilename(defaultextension=".ttl", filetypes=[("Turtle RDF", "*.ttl")]);
        if not filepath: return
        graph = scene.to_rdf_graph(self.project.assets)
        graph.serialize(destination=filepath, format="turtle"); messagebox.showinfo("Esporta RDF", "Grafo salvato!")
    def open_generator_window(self):
        if not self.project.assets: messagebox.showwarning("Asset Mancanti", "Carica almeno un asset."); return
        generator_window = GeneratorWindow(self, self.project.assets); generator_window.grab_set()
    def on_delete_key(self, event):
        if self.selected_object_index is not None and messagebox.askyesno("Elimina", "Eliminare oggetto?"):
            scene = self.project.get_active_scene(); del scene.objects[self.selected_object_index]
            self.selected_object_index = None; self.draw_scene()
    def show_about(self): messagebox.showinfo("About", "TRIPIX Weaver Studio\n\nAutore: Luigi Usai\nData: 18 giugno 2025\nLuogo: Quartucciu (CA), Italy")
    def load_assets(self):
        filepaths = filedialog.askopenfilenames(title="Seleziona immagini", filetypes=[("PNG", "*.png")])
        if not filepaths: return
        for fp in filepaths: self.project.add_asset(fp)
        self.update_asset_library()
    def update_rdf_viewer(self):
        scene = self.project.get_active_scene()
        graph = scene.to_rdf_graph(self.project.assets)
        rdf_text = "Il grafo è vuoto." if len(graph) == 0 else graph.serialize(format="turtle")
        self.rdf_viewer.config(state=tk.NORMAL); self.rdf_viewer.delete("1.0", tk.END)
        self.rdf_viewer.insert(tk.END, rdf_text); self.rdf_viewer.config(state=tk.DISABLED)
    def draw_scene(self):
        self.scene_canvas.delete("all"); self.scene_images = []
        scene = self.project.get_active_scene()
        for i, scene_obj in enumerate(scene.objects):
            asset = self.project.assets.get(scene_obj.asset_id)
            if not asset: continue
            try:
                img = asset.image.copy().resize((scene_obj.width, scene_obj.height))
                img_tk = ImageTk.PhotoImage(img)
                self.scene_images.append(img_tk)
                self.scene_canvas.create_image(scene_obj.x, scene_obj.y, anchor=tk.NW, image=img_tk, tags=f"obj_{i}")
            except ValueError: pass
        if self.selected_object_index is not None and self.selected_object_index < len(scene.objects):
            obj = scene.objects[self.selected_object_index]
            x1, y1, x2, y2 = obj.get_bbox()
            self.scene_canvas.create_rectangle(x1, y1, x2, y2, outline='cyan', width=2)
            handle_size = 8; self.scene_canvas.create_rectangle(x2-handle_size, y2-handle_size, x2, y2, fill='cyan', outline='black', tags="resize_handle")
        self.update_rdf_viewer()
    def on_asset_press(self, event):
        items = self.asset_canvas.find_closest(event.x, event.y)
        if not items: return
        tags = self.asset_canvas.gettags(items[0]); asset_tag = next((t for t in tags if t.startswith("asset_")), None)
        if asset_tag:
            asset_id = asset_tag.replace("asset_", ""); asset = self.project.assets.get(asset_id)
            if asset: self.drag_info = {"type": "new", "asset": asset}
    def on_scene_press(self, event):
        self.selected_object_index = None; self.drag_info = None
        items = self.scene_canvas.find_overlapping(event.x-2, event.y-2, event.x+2, event.y+2)
        if not items: self.draw_scene(); return
        item_id = items[-1]; tags = self.scene_canvas.gettags(item_id)
        obj_index, drag_type = -1, None
        if "resize_handle" in tags:
            for i, obj in enumerate(self.project.get_active_scene().objects):
                 x1, y1, x2, y2 = obj.get_bbox()
                 if x2-8 <= event.x <= x2+2 and y2-8 <= event.y <= y2+2: obj_index, drag_type = i, "resize"; break
        else:
            obj_tag = next((t for t in tags if t.startswith("obj_")), None)
            if obj_tag: obj_index, drag_type = int(obj_tag.replace("obj_", "")), "move"
        if obj_index != -1:
            self.selected_object_index = obj_index
            obj = self.project.get_active_scene().objects[obj_index]
            self.drag_info = {"type": drag_type, "item_index": obj_index, "start_x": event.x, "start_y": event.y, "initial_x": obj.x, "initial_y": obj.y, "initial_w": obj.width, "initial_h": obj.height}
        self.draw_scene()
    def on_drag(self, event):
        if not self.drag_info: return
        drag_type = self.drag_info.get("type")
        if drag_type == "move":
            dx, dy = event.x - self.drag_info["start_x"], event.y - self.drag_info["start_y"]
            obj = self.project.get_active_scene().objects[self.drag_info["item_index"]]
            obj.x, obj.y = self.drag_info["initial_x"] + dx, self.drag_info["initial_y"] + dy
            self.draw_scene()
        elif drag_type == "resize":
            obj = self.project.get_active_scene().objects[self.drag_info["item_index"]]
            dx, dy = event.x - self.drag_info["start_x"], event.y - self.drag_info["start_y"]
            new_w, new_h = self.drag_info["initial_w"] + dx, self.drag_info["initial_h"] + dy
            if new_w > 20 and new_h > 20: obj.width, obj.height = new_w, new_h; self.draw_scene()
    def on_release(self, event):
        if not self.drag_info: return
        if self.drag_info.get("type") == "new":
            asset = self.drag_info.get("asset")
            if asset: 
                canvas_x, canvas_y = event.x - self.scene_canvas.winfo_rootx(), event.y - self.scene_canvas.winfo_rooty()
                if 0 < canvas_x < self.scene_canvas.winfo_width() and 0 < canvas_y < self.scene_canvas.winfo_height():
                    new_obj = SceneObject(asset, canvas_x - asset.image.width//2, canvas_y - asset.image.height//2)
                    self.project.get_active_scene().add_object(new_obj)
                    self.selected_object_index = len(self.project.get_active_scene().objects) - 1
                    self.draw_scene()
        self.drag_info = None

if __name__ == "__main__":
    app = App()
    app.mainloop()