import tkinter as tk
from tkinter import ttk, scrolledtext, filedialog, messagebox
from rdflib import Graph

class SparqlQueryApp:
    def __init__(self, root):
        self.root = root
        self.root.title("TRIPIX - SPARQL Query Interface")
        self.root.geometry("1000x750")
        
        self.graph = Graph()
        self.style = ttk.Style()
        self.style.theme_use('clam')
        
        self.create_widgets()

    def create_widgets(self):
        # --- Frame principale ---
        main_frame = ttk.Frame(self.root, padding="10")
        main_frame.pack(fill=tk.BOTH, expand=True)
        main_frame.rowconfigure(2, weight=1)
        main_frame.columnconfigure(0, weight=1)

        # --- Frame di Controllo (Carica File) ---
        control_frame = ttk.Frame(main_frame)
        control_frame.grid(row=0, column=0, sticky="ew", pady=(0, 10))
        control_frame.columnconfigure(1, weight=1)
        
        self.load_button = ttk.Button(control_frame, text="1. Carica File RDF (knowledge_base.ttl)", command=self.load_rdf_file)
        self.load_button.grid(row=0, column=0, padx=(0, 10))
        
        self.file_label = ttk.Label(control_frame, text="Nessun file caricato.", relief=tk.SUNKEN, padding=5)
        self.file_label.grid(row=0, column=1, sticky="ew")

        # --- Frame della Query ---
        query_frame = ttk.LabelFrame(main_frame, text="2. Inserisci la tua Query SPARQL", padding="10")
        query_frame.grid(row=1, column=0, sticky="ew", pady=5)
        query_frame.rowconfigure(0, weight=1)
        query_frame.columnconfigure(0, weight=1)
        
        self.query_input = scrolledtext.ScrolledText(query_frame, height=10, wrap=tk.WORD)
        self.query_input.grid(row=0, column=0, sticky="nsew")
        self.query_input.insert(tk.END, self.get_example_query()) # Inserisci una query di esempio
        
        self.execute_button = ttk.Button(main_frame, text="3. Esegui Query", command=self.execute_query)
        self.execute_button.grid(row=2, column=0, sticky="ew", pady=10)
        self.execute_button.grid_remove() # Nascondi il pulsante finché un file non è caricato

        # --- Frame dei Risultati ---
        results_frame = ttk.LabelFrame(main_frame, text="4. Risultati", padding="10")
        results_frame.grid(row=3, column=0, sticky="nsew", pady=5)
        results_frame.rowconfigure(0, weight=1)
        results_frame.columnconfigure(0, weight=1)

        # Usiamo un Treeview per mostrare i risultati in una tabella
        self.results_tree = ttk.Treeview(results_frame, show='headings')
        self.results_tree.grid(row=0, column=0, sticky="nsew")

        # Aggiungi scrollbar alla tabella
        vsb = ttk.Scrollbar(results_frame, orient="vertical", command=self.results_tree.yview)
        vsb.grid(row=0, column=1, sticky='ns')
        hsb = ttk.Scrollbar(results_frame, orient="horizontal", command=self.results_tree.xview)
        hsb.grid(row=1, column=0, sticky='ew')
        self.results_tree.configure(yscrollcommand=vsb.set, xscrollcommand=hsb.set)

    def get_example_query(self):
        """Restituisce una query SPARQL di esempio utile."""
        return 
        
        
        
        """
PREFIX tpix: <http://tripix.org/schema/>
PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>

# Esempio: Trova tutte le scene in cui un cane
# si sovrappone a una roccia.

SELECT ?dog ?rock
WHERE {
    ?dog rdf:type tpix:Dog .
    ?rock rdf:type tpix:Rock .
    ?dog tpix:isOverlappingWith ?rock .
}
LIMIT 25
"""

    def load_rdf_file(self):
        """Apre una finestra di dialogo per selezionare il file .ttl."""
        file_path = filedialog.askopenfilename(
            title="Seleziona il tuo Knowledge Graph",
            filetypes=(("Turtle files", "*.ttl"), ("All files", "*.*"))
        )
        if not file_path:
            return

        try:
            self.file_label.config(text=f"Caricamento in corso: {os.path.basename(file_path)}...")
            self.root.update_idletasks()
            
            self.graph.parse(file_path, format="turtle")
            
            self.file_label.config(text=f"Caricato: {os.path.basename(file_path)} ({len(self.graph)} triple)")
            self.execute_button.grid() # Mostra il pulsante Esegui
            
        except Exception as e:
            messagebox.showerror("Errore di Caricamento", f"Impossibile caricare o analizzare il file RDF:\n{e}")
            self.file_label.config(text="Caricamento fallito.")

    def execute_query(self):
        """Esegue la query SPARQL scritta dall'utente e mostra i risultati."""
        query = self.query_input.get("1.0", tk.END)
        
        if not query.strip():
            messagebox.showwarning("Query Vuota", "Per favore, inserisci una query SPARQL.")
            return

        try:
            results = self.graph.query(query)
            
            # Pulisci la tabella precedente
            for i in self.results_tree.get_children():
                self.results_tree.delete(i)
            self.results_tree["columns"] = []

            if not results:
                messagebox.showinfo("Nessun Risultato", "La query non ha prodotto risultati.")
                return

            # Imposta le colonne della tabella
            self.results_tree["columns"] = results.vars
            for col in results.vars:
                self.results_tree.heading(col, text=str(col))
                self.results_tree.column(col, width=200, anchor=tk.W)

            # Inserisci le righe dei risultati
            for row in results:
                # Converte i risultati (che sono URI) in stringhe leggibili
                values = [self.graph.namespace_manager.normalizeUri(item) for item in row]
                self.results_tree.insert("", "end", values=values)

        except Exception as e:
            messagebox.showerror("Errore di Query", f"La query SPARQL non è valida:\n{e}")

if __name__ == "__main__":
    import os
    root = tk.Tk()
    app = SparqlQueryApp(root)
    root.mainloop()