// File: prepare_training_data.py (versione corretta)

import os
import csv
from rdflib import Graph, URIRef, Namespace, Literal
from rdflib.namespace import RDF, RDFS

# --- CONFIGURAZIONE ---
TPIX = Namespace("http://tripix.org/schema/")
# Assumiamo che il generatore sia master_generator o dataset_generator_v6
# Entrambi creano una cartella di output e un file .ttl al suo interno.
# Modifichiamo il nome per coerenza.
DATASET_DIR = "master_dataset_v6" # O "generated_dataset" se usi weaver_studio
KNOWLEDGE_BASE_FILE = "knowledge_base_master.ttl" # O "knowledge_base.ttl"
OUTPUT_CSV = os.path.join(DATASET_DIR, "metadata.csv")

def find_knowledge_base(directory):
    """Cerca il file del grafo di conoscenza principale."""
    kb_path = os.path.join(directory, KNOWLEDGE_BASE_FILE)
    if os.path.exists(kb_path):
        return kb_path
    # Prova con l'altro nome per flessibilità
    kb_path_alt = os.path.join(directory, "knowledge_base.ttl")
    if os.path.exists(kb_path_alt):
        return kb_path_alt
    return None

def main():
    print(f"Inizio preparazione dati da '{DATASET_DIR}'...")
    
    rdf_path = find_knowledge_base(DATASET_DIR)
    if not rdf_path:
        print(f"ERRORE: File del grafo di conoscenza non trovato in '{DATASET_DIR}'.")
        print(f"Assicurati che esista '{KNOWLEDGE_BASE_FILE}' o 'knowledge_base.ttl'.")
        return

    print(f"Caricamento del grafo di conoscenza da: {rdf_path}")
    g = Graph()
    try:
        g.parse(rdf_path, format="turtle")
    except Exception as e:
        print(f"ERRORE: Impossibile analizzare {rdf_path}. Errore: {e}")
        return
        
    training_data = []
    
    # Query SPARQL per estrarre tutte le informazioni necessarie in un solo colpo
    # Questo è molto più efficiente che iterare sul grafo più volte.
    query = """
    PREFIX tpix: <http://tripix.org/schema/>
    PREFIX inst: <http://tripix.org/data/master_dataset/scene_>
    PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
    PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>

    SELECT ?scene_variant ?rotation ?brightness ?scale (GROUP_CONCAT(DISTINCT ?object_type; SEPARATOR=",") AS ?objects)
    WHERE {
        ?scene_variant rdf:type tpix:SceneVariant .
        
        # Estrai le proprietà della variante di scena
        OPTIONAL { ?scene_variant tpix:hasRotation ?rotation . }
        OPTIONAL { ?scene_variant tpix:hasBrightness ?brightness . }
        OPTIONAL { ?scene_variant tpix:hasScale ?scale . }

        # Trova tutti gli oggetti in quella scena e ottieni il loro tipo
        # (assumendo che gli oggetti siano condivisi tra le varianti della stessa scena base)
        ?any_obj tpix:hasPosition ?pos . # Un modo per trovare oggetti
  		?any_obj rdfs:label ?object_name . # Assumiamo che gli oggetti abbiano una label
        ?any_obj rdf:type ?object_class_uri .
        FILTER (STRSTARTS(STR(?any_obj), REPLACE(STR(?scene_variant), "_.+", ""))) # Corrispondenza base scena
        
        # Pulisci il nome della classe per la caption
        BIND(REPLACE(STR(?object_class_uri), "http://tripix.org/schema/", "") AS ?object_type)
    }
    GROUP BY ?scene_variant ?rotation ?brightness ?scale
    ORDER BY ?scene_variant
    """

    results = g.query(query)
    
    print(f"Trovate {len(results)} varianti di scena nel grafo. Elaborazione in corso...")

    for row in results:
        scene_uri = row.scene_variant
        
        # Ricostruisci il percorso del file dall'URI della scena
        # Es: inst:scene_0001_rot90_br1.3_sc0.8 -> scene_0001/rot90_br1.3_sc0.8.jpg
        uri_str = str(scene_uri)
        parts = uri_str.split('/')[-1].split('_')
        scene_folder_name = f"{parts[0]}_{parts[1]}" # "scene_0001"
        filename_base = "_".join(parts[2:]) # "rot90_br1.3_sc0.8"
        
        relative_image_path = os.path.join(scene_folder_name, f"{filename_base}.jpg")
        full_image_path = os.path.join(DATASET_DIR, relative_image_path)
        
        if not os.path.exists(full_image_path):
            print(f"ATTENZIONE: File immagine non trovato per la scena {scene_uri}: {full_image_path}")
            continue

        # Crea la caption
        caption_parts = []
        if row.rotation is not None: caption_parts.append(f"rotazione {row.rotation} gradi")
        if row.brightness is not None: caption_parts.append(f"luminosità {row.brightness}")
        if row.scale is not None: caption_parts.append(f"scala {row.scale}")
        
        if row.objects:
            # Pulisce e ordina gli oggetti per una caption consistente
            object_list = sorted([o.lower() for o in str(row.objects).split(',')])
            caption_parts.append(f"oggetti {' e '.join(object_list)}")

        final_caption = ". ".join(caption_parts) + "." if caption_parts else "scena generica."
        training_data.append([relative_image_path, final_caption])

    if not training_data:
        print("ERRORE: Nessun dato di training generato. Controlla il grafo e la struttura delle cartelle.")
        return
        
    with open(OUTPUT_CSV, 'w', newline='', encoding='utf-8') as f:
        writer = csv.writer(f)
        writer.writerow(['file_name', 'text'])
        writer.writerows(training_data)

    print(f"Preparazione completata. Creato il file '{OUTPUT_CSV}' con {len(training_data)} campioni.")

if __name__ == "__main__":
    main()