import os
import random
import time
import sys
import re
from PIL import Image, ImageEnhance
from rdflib import Graph, URIRef, Namespace, Literal
from rdflib.namespace import RDF, RDFS, XSD

def create_safe_uri_string(text):
    """Pulisce una stringa per renderla un frammento di URI valido."""
    s = re.sub(r'[^a-zA-Z0-9_]', '_', text)
    return s

def create_base_graph(config):
    g = Graph(); g.bind("tpix", config["TPIX"]); g.bind("inst", config["INST"]); g.bind("rdfs", RDFS)
    return g
def apply_shear(img, factor): return img.transform(img.size, Image.AFFINE, (1, factor, 0, 0, 1, 0), fillcolor=(0,0,0,0))
def apply_rotation(img, angle): return img.rotate(angle, expand=True, fillcolor=(0,0,0,0))
def apply_brightness(img, factor): return ImageEnhance.Brightness(img).enhance(factor)
def apply_scale(img, factor): return img.resize((int(img.width * factor), int(img.height * factor)))

def run_generation(config, status_callback=print, progress_callback=None):
    start_time = time.time()
    NUM_BASE_SCENES = config.get("num_scenes", 1); ROTATION_STEP = config.get("rot_step", 360)
    BRIGHTNESS_LEVELS = config.get("brightness_levels", [1.0]); SCALE_LEVELS = config.get("scale_levels", [1.0])
    MIN_OBJECTS = config.get("min_objects", 2); MAX_OBJECTS = config.get("max_objects", 3)
    ALLOW_SHEAR = config.get("allow_shear", False); ALLOW_OCCLUSION = config.get("allow_occlusion", False)
    OUTPUT_BASE_DIR = config["OUTPUT_BASE_DIR"]; ASSETS = config["ASSETS"]
    TPIX = config["TPIX"]; INST = config["INST"]
    SCENE_WIDTH, SCENE_HEIGHT = config["SCENE_WIDTH"], config["SCENE_HEIGHT"]

    os.makedirs(OUTPUT_BASE_DIR, exist_ok=True)
    unified_graph = create_base_graph(config)
    total_variants_generated = 0
    
    rot_range = range(0, 360, ROTATION_STEP) if ROTATION_STEP > 0 else [0]
    total_images_to_generate = NUM_BASE_SCENES * len(rot_range) * len(BRIGHTNESS_LEVELS) * len(SCALE_LEVELS)
    status_callback(f"Inizio generazione. Previste {total_images_to_generate} immagini...")

    for i in range(1, NUM_BASE_SCENES + 1):
        scene_id = f"{i:04d}"; scene_dir = os.path.join(OUTPUT_BASE_DIR, f"scene_{scene_id}")
        os.makedirs(scene_dir, exist_ok=True); status_callback(f"--- Creando scena base {scene_id} ---")
        
        base_scene_image = Image.new('RGBA', (SCENE_WIDTH, SCENE_HEIGHT), (0, 0, 0, 0))
        base_graph = create_base_graph(config)
        
        num_objects = random.randint(MIN_OBJECTS, MAX_OBJECTS)
        available_assets = list(ASSETS.keys())
        if len(available_assets) < num_objects:
            status_callback(f"ATTENZIONE: Asset insufficienti."); continue
        
        object_keys = random.sample(available_assets, num_objects)
        placed_objects = []

        for j, key in enumerate(object_keys):
            asset_filepath = ASSETS[key]["filepath"]
            try:
                asset_img = Image.open(asset_filepath).convert("RGBA")
            except FileNotFoundError:
                status_callback(f"ERRORE: Asset '{asset_filepath}' non trovato."); continue

            shear_factor = 0.0
            if ALLOW_SHEAR and random.random() > 0.5:
                shear_factor = random.uniform(-0.3, 0.3)
                asset_img = apply_shear(asset_img, shear_factor)

            size = (random.randint(150, 300), random.randint(150, 300))
            asset_img = asset_img.resize(size)
            pos = (random.randint(0, SCENE_WIDTH-size[0]), random.randint(0, SCENE_HEIGHT-size[1]))
            if ALLOW_OCCLUSION and j > 0 and random.random() > 0.3:
                prev_obj = placed_objects[j-1]
                offset_x = random.randint(-prev_obj['size'][0]//4, prev_obj['size'][0]//4)
                offset_y = random.randint(-prev_obj['size'][1]//4, prev_obj['size'][1]//4)
                pos = (max(0, min(prev_obj['pos'][0] + offset_x, SCENE_WIDTH - size[0])),
                       max(0, min(prev_obj['pos'][1] + offset_y, SCENE_HEIGHT - size[1])))

            base_scene_image.paste(asset_img, pos, mask=asset_img)
            safe_key = create_safe_uri_string(key)
            obj_uri = INST[f"{safe_key}_{scene_id}"]
            placed_objects.append({'key': key, 'uri': obj_uri, 'pos': pos, 'size': size, 'shear': shear_factor})

        for obj in placed_objects:
            asset_class_name = create_safe_uri_string(ASSETS[obj['key']]["class"])
            base_graph.add((obj['uri'], RDF.type, TPIX[asset_class_name]))
            base_graph.add((obj['uri'], TPIX.hasPosition, Literal(str(obj['pos']))))
            if obj['shear'] != 0.0: base_graph.add((obj['uri'], TPIX.hasShear, Literal(obj['shear'], datatype=XSD.float)))
        
        for j in range(len(placed_objects)):
            for k in range(j + 1, len(placed_objects)):
                rect_j, rect_k = (*placed_objects[j]['pos'], *placed_objects[j]['size']), (*placed_objects[k]['pos'], *placed_objects[k]['size'])
                overlap_x = max(0, min(rect_j[0]+rect_j[2], rect_k[0]+rect_k[2]) - max(rect_j[0], rect_k[0]))
                overlap_y = max(0, min(rect_j[1]+rect_j[3], rect_k[1]+rect_k[3]) - max(rect_j[1], rect_k[1]))
                if overlap_x > 0 and overlap_y > 0:
                     base_graph.add((placed_objects[j]['uri'], TPIX.isPartiallyOccludedBy, placed_objects[k]['uri']))

        status_callback(f"Generando varianti per la scena {scene_id}...")
        for angle in rot_range:
            rotated_image = apply_rotation(base_scene_image, angle)
            for br in BRIGHTNESS_LEVELS:
                bright_image = apply_brightness(rotated_image, br)
                for sc in SCALE_LEVELS:
                    final_image_transformed = apply_scale(bright_image, sc)
                    background = Image.new('RGBA', final_image_transformed.size, (255, 255, 255, 255))
                    final_image_merged = Image.alpha_composite(background, final_image_transformed).convert("RGB")
                    variant_graph = create_base_graph(config); variant_graph += base_graph
                    filename = f"rot{angle}_br{br:.1f}_sc{sc:.1f}"
                    scene_uri = INST[f"scene_{scene_id}_{filename}"]
                    variant_graph.add((scene_uri, RDF.type, TPIX.SceneVariant))
                    variant_graph.add((scene_uri, TPIX.hasRotation, Literal(angle, datatype=XSD.integer)))
                    variant_graph.add((scene_uri, TPIX.hasBrightness, Literal(br, datatype=XSD.float)))
                    variant_graph.add((scene_uri, TPIX.hasScale, Literal(sc, datatype=XSD.float)))
                    final_image_merged.save(os.path.join(scene_dir, f"{filename}.jpg"))
                    unified_graph += variant_graph
                    total_variants_generated += 1
                    if progress_callback: progress_callback(total_variants_generated / total_images_to_generate * 100)
    
    unified_graph_path = os.path.join(OUTPUT_BASE_DIR, "knowledge_base.ttl")
    unified_graph.serialize(destination=unified_graph_path, format="turtle")
    duration = time.time() - start_time
    summary = (f"Generazione completata in {duration:.2f}s.\n"
               f"Immagini: {total_variants_generated}\nGrafo: {len(unified_graph)} triple.\nSalvato in '{unified_graph_path}'")
    status_callback(summary)
    return summary

if __name__ == "__main__":
    print("Questo script è progettato per essere importato.")