// filename alloc.cc
#include <cuda.h>
#include <rmm/device_buffer.hpp>
#include <rmm/mr/device/pool_memory_resource.hpp>
#include <iostream>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <fstream>
#include <semaphore.h>
#include <map>
#include <pthread.h>

struct SharedData {
    bool request_received;  // Flag to indicate a request
    bool response_ready;    // Flag to indicate the response is ready
    bool is_alloc;
    uint64_t size;
    uint64_t pool_offset;
    cudaIpcMemHandle_t ipchandle;   // The memory address returned by the sender
    pthread_mutex_t lock;
} __attribute__((__packed__));

static std::map<void *, uint64_t> ofs_map;
static std::mutex mtx;

extern "C" {

void* my_malloc(ssize_t size, int device, cudaStream_t stream) {
   if (size == 0)
      return nullptr;
   mtx.lock();  // Wait (lock) the semaphore
   int shm_fd = shm_open("/exploitshm", O_RDWR, 0666);
   if (shm_fd == -1) {
      std::cerr << "Error opening shared memory" << std::endl;
   }

   // Step 2: Map shared memory into address space
   SharedData* shared_data = (SharedData*)mmap(0, sizeof(SharedData), PROT_READ | PROT_WRITE, MAP_SHARED, shm_fd, 0);
   if (shared_data == MAP_FAILED) {
      std::cerr << "Error mapping shared memory" << std::endl;
   }

   pthread_mutex_lock(&shared_data->lock);
   // Step 3: Send a request to the senders
   shared_data->is_alloc = true;
   shared_data->size = size;
   shared_data->request_received = true;  // Signal the sender that a request has been made

   pthread_mutex_unlock(&shared_data->lock);
   // std::cout << "Request sent to sender..." << std::endl;

   // Step 4: Wait for the response (memory address)
   while (!shared_data->response_ready) {
      usleep(1);  // Sleep to simulate waiting
   }
   pthread_mutex_lock(&shared_data->lock);

   // Step 5: Retrieve the memory address from the shared memory
   void *address;
   cudaIpcOpenMemHandle(&address, shared_data->ipchandle, cudaIpcMemLazyEnablePeerAccess);
   // std::cout << "Received memory address: " << address << std::endl;
   address =(uint8_t*) address  + shared_data->pool_offset;
   // std::cout << "Received memory address: " << address << std::endl;
   if (ofs_map.find(address) != ofs_map.end())
      std::cerr << "Addr already exist " << address << std::endl;
   ofs_map[address] = shared_data->pool_offset;
   // Step 6: Clean up
   shared_data->response_ready = false;
   pthread_mutex_unlock(&shared_data->lock);
   munmap(shared_data, sizeof(SharedData));
   close(shm_fd);
   
   mtx.unlock();  // Signal (unlock) the semaphore
   
   return address;
}

void my_free(void* ptr, ssize_t size, int device, cudaStream_t stream) {
   // Step 4: Lock the semaphore to ensure only one request is handled at a time
   // std::cerr << "Freed " << ptr << ' ' << size << std::endl;
   if (ptr == nullptr)
      return;
   int i = 0;
   while (ofs_map.find(ptr) == ofs_map.end())
   {
      std::cerr << "Did not find " << ptr << ' ' << size << std::endl;
      usleep(1000);
      i++;
      if (i == 50)
      {
         std::cerr << "Skipping " << ptr << ' ' << size << std::endl;
         return;
      }
   }
   
   mtx.lock();  // Wait (lock) the semaphore

   int shm_fd = shm_open("/exploitshm", O_RDWR, 0666);
   if (shm_fd == -1) {
      std::cerr << "Error opening shared memory" << std::endl;
   }

   // Step 2: Map shared memory into address space
   SharedData* shared_data = (SharedData*)mmap(0, sizeof(SharedData), PROT_READ | PROT_WRITE, MAP_SHARED, shm_fd, 0);
   if (shared_data == MAP_FAILED) {
      std::cerr << "Error mapping shared memory" << std::endl;
   }

   pthread_mutex_lock(&shared_data->lock);
   // Step 3: Send a request to the sender
   shared_data->is_alloc = false;
   shared_data->size = size;
   shared_data->pool_offset = ofs_map[ptr];
   shared_data->request_received = true;  // Signal the sender that a request has been made
   // std::cout << "Request sent to sender..." << std::endl;
   pthread_mutex_unlock(&shared_data->lock);
   // Step 4: Wait for the response (memory address)
   while (!shared_data->response_ready) {
      usleep(1);  // Sleep to simulate waiting
   }
   pthread_mutex_lock(&shared_data->lock);
   
   ofs_map.erase(ptr);
   // std::cerr << "Freed " << ptr << std::endl;
   // Step 6: Clean up
   shared_data->response_ready = false;
   pthread_mutex_unlock(&shared_data->lock);
   munmap(shared_data, sizeof(SharedData));
   close(shm_fd);
   
   mtx.unlock();  // Signal (unlock) the semaphore
}

}