# This routines enables creating movies from a series of images
# That is perhaps not structly a GMG routine, but still part of the workflow
# as images, generated by Paraview, often need to be merged into a movie.
# that can be done with FFMPEG, but determining the optimal parameters can be oftentimes tricky
# (while ensuring that the movie works well on mac/windows etc)

using FFMPEG

export movie_from_images

"""
    movie_from_images(; dir=pwd(), file=nothing, outfile=nothing, framerate=10, copy_to_current_dir=true, type=:mp4_default, collect=true)

The typical way to create animations with `Paraview` is to use the `Save Animation` option to save a series of `*.png` images.

This function combines these images to an `*.mp4` movie.

Optional options
===
- `dir`:    directory where the images are stored.
- `file`:   filename of the image series without extension and numbers. Required if >1 image series is stored in the same directory. By default we reconstruct this name from the available files.
- `outfile`:  filename of the resulting movie without extension; if not specified, `file` is used.
- `framerate`: number of frames/second.
- `copy_to_current_dir`: copies the final movie to the current directory if `true` (default); otherwise it will be stored in `dir`.
- `type`: type of movie that is created; possible options are:
    -  `:mp4_default`: Default option that saves a well-compressed `mp4` movie that works well for us on ipad and embedded in a powerpoint presentation.
    -  `:mov_hires`: Higher-resolution quicktime movie (larger filesize & not compatible with windows)
- `collect`: suppresses output of `FFMPEG` if `true` (default).
"""
function movie_from_images(; dir = pwd(), file = nothing, outfile = nothing, framerate = 10, copy_to_current_dir = true, type = :mp4_default)
    curdir = pwd()
    cd(dir)
    files = split.(readdir(), ".")
    files = files[length.(files) .== 3]     # names should be filename.0001.png or something like that

    filenames = [f[1] for f in files]
    fileext = files[1][3]
    le = length(files[1][2])

    # try to determine the filename automatically (works if we have only one name in the director)
    if length(unique(filenames)) > 1 && isnothing(file)
        error("you have more than one image series in the directory $dir; please specify the filename `file`.")
    elseif isnothing(file)
        file = unique(filenames)[1]
    end

    if isnothing(outfile)
        outfile = file  #use same name as images
    end

    if type == :mp4_default
        # this produces an *.mp4 movie that looks good on an ipad
        outfile_ext = outfile * ".mp4"
        cmd = `-y -framerate $framerate -f image2 -i $file.%0$(le)d.$fileext  -vf pad="""width=ceil(iw/2)*2:height=ceil(ih/2)*2""" -f mp4 -vcodec libx264 -pix_fmt yuv420p $outfile_ext`
    elseif type == :mov_hires
        outfile_ext = outfile * ".mov"
        cmd = `-y -f image2 -framerate $framerate -i $file.%0$(le)d.$fileext  -c:v prores_ks -profile:v 1  $outfile_ext`
    else
        error("unknown movie type $type")
    end

    # run
    FFMPEG.exe(cmd, collect = true)

    result = joinpath(pwd(), outfile_ext)
    if copy_to_current_dir
        # copy result
        result = joinpath(curdir, outfile_ext)
        cp(outfile_ext, result, force = true)
    end
    println("created movie: $result")

    cd(curdir)  # return to original directory

    return outfile_ext
end
