/********************************************************************************
 *   Copyright (c) : Université de Lyon 1, CNRS/IN2P3, UMR5822,                 *
 *                   IP2I, F-69622 Villeurbanne Cedex, France                   *
 *   Contibutor(s) :                                                            *
 *      Jérémie Dudouet jeremie.dudouet@cnrs.fr [2023]                          *
 *                                                                              *
 *    This software is governed by the CeCILL-B license under French law and    *
 *    abiding by the  rules of distribution of free  software.  You can use,    *
 *    modify  and/ or  redistribute  the  software under  the  terms of  the    *
 *    CeCILL-B license as circulated by CEA, CNRS and INRIA at the following    *
 *    URL \"http://www.cecill.info\".                                           *
 *                                                                              *
 *    As a counterpart to the access  to the source code and rights to copy,    *
 *    modify  and redistribute granted  by the  license, users  are provided    *
 *    only with a limited warranty  and the software's author, the holder of    *
 *    the economic  rights, and the  successive licensors have  only limited    *
 *    liability.                                                                *
 *                                                                              *
 *    In this respect, the user's attention is drawn to the risks associated    *
 *    with loading,  using, modifying  and/or developing or  reproducing the    *
 *    software by the user in light of its specific status of free software,    *
 *    that  may mean that  it is  complicated to  manipulate, and  that also    *
 *    therefore  means that it  is reserved  for developers  and experienced    *
 *    professionals having in-depth  computer knowledge. Users are therefore    *
 *    encouraged  to load  and test  the software's  suitability  as regards    *
 *    their  requirements  in  conditions  enabling the  security  of  their    *
 *    systems  and/or data to  be ensured  and, more  generally, to  use and    *
 *    operate it in the same conditions as regards security.                    *
 *                                                                              *
 *    The fact that  you are presently reading this means  that you have had    *
 *    knowledge of the CeCILL-B license and that you accept its terms.          *
 ********************************************************************************/

#include "CXBgdFit.h"

#include <iostream>
#include <iomanip>
#include <sstream>

#include "TF1.h"
#include "TH1.h"
#include "TFitResultPtr.h"
#include "TFitResult.h"
#include "Math/MinimizerOptions.h"

#include "CXHist1DPlayer.h"
#include "CXArrow.h"
#include "CXMainWindow.h"
#include "CXBashColor.h"
#include "CXWSManager.h"

using namespace std;

CXBgdFit::CXBgdFit(TH1 *hist, TVirtualPad *pad, CXHist1DPlayer *player, CXWorkspace *_workspace) : TObject()
{
    fHistogram = hist;
    fPad = pad;
    fPlayer = player;

    fWorkspace = _workspace;

    fListOfArrows = new TList;
    fListOfArrows->SetOwner();
}

CXBgdFit::~CXBgdFit()
{
    Clear(fPad);

    delete fListOfArrows;

    delete fBackFunction;

    fPlayer->GetMainWindow()->RefreshPads();
}

void CXBgdFit::AddArrow(Double_t Energy)
{
    Int_t Bin = fHistogram->FindBin(Energy);
    Double_t Value = fHistogram->GetBinContent(Bin);

    for(int i=fHistogram->FindBin(Energy)-2 ; i<=fHistogram->FindBin(Energy)+2 ; i++){
        if(i>0 && fHistogram->GetBinContent(i)>Value){
            Value = fHistogram->GetBinContent(i);
            Bin = i;
        }
    }

    Energy = fHistogram->GetBinCenter(Bin);
    Double_t MaxGlob = fHistogram->GetMaximum();
    auto *arrow = new CXArrow(this,Energy,(Value + MaxGlob/100.) ,(Value +MaxGlob/15.),0.01,0.03,"<|");
    arrow->SetAngle(30);
    arrow->SetLineWidth(2);
    arrow->Draw();

    fListOfArrows->Add(arrow);
    Update();
}

void CXBgdFit::RemoveArrow(CXArrow *arrow)
{
    if(arrow == nullptr) {
        fListOfArrows->RemoveLast();
        fPad->GetListOfPrimitives()->RemoveLast();
    }
    else {
        fListOfArrows->Remove(arrow);
        fPad->GetListOfPrimitives()->Remove(arrow);
    }

    Update();
}

void CXBgdFit::Update()
{
    fBackgd.clear();

    fListOfArrows->Sort();
    TList back,Ener;

    if(!fPlayer->DoNewBgdFit && (fListOfArrows->GetEntries()%2)==1) {
        Clear(fPad);
        gbash_color->WarningMessage("Sets of two ranges are needed for a bgd fit --> command ignored");
        return;
    }

    for(int i=0 ; i<fListOfArrows->GetEntries() ; i++) {
        auto *arr = dynamic_cast<CXArrow*>(fListOfArrows->At(i));
        back.Add(arr);
    }

    for(int i=0 ; i<back.GetEntries() ; i++) {
        auto *arr = dynamic_cast<CXArrow*>(back.At(i));
        Double_t E = arr->GetEnergy();
        Double_t MaxGlob = fHistogram->GetMaximum();
        Double_t Value = fHistogram->GetBinContent(fHistogram->FindBin(E));
        arr->Set(E,(Value + MaxGlob/100.), (Value+MaxGlob/15.));
        arr->SetLineColor(kBlue);
        arr->SetFillColor(kBlue);
        fBackgd.push_back(E);
    }

    fPlayer->GetMainWindow()->RefreshPads();
}

void CXBgdFit::Clear(TVirtualPad *pad)
{
    fPlayer->EndFit();

    if(fPad==nullptr) {
        gbash_color->WarningMessage("No selected pad, ignored");
        return;
    }
    if(pad == nullptr) pad = fPad;

    TList *list = pad->GetListOfPrimitives();

    list->Remove(fBackFunction);

    for(int i=0 ; i<fListOfArrows->GetEntries() ; i++)
        list->Remove(fListOfArrows->At(i));

    fPlayer->RemoveBgdFit(this);
    fPlayer->GetMainWindow()->RefreshPads();
}

void CXBgdFit::Fit()
{
    if(fListOfArrows->GetEntries()<4 || (fListOfArrows->GetEntries()%2) !=0) {
        gbash_color->WarningMessage("At least two pairs of arrows are needed to define the background to fit (two before and two after the peak)");
        return;
    }

    if(fPad==nullptr) {
        gbash_color->WarningMessage("No selected pad, ignored");
        return;
    }
    fPad->cd();

    if(fPlayer == nullptr) {
        gbash_color->WarningMessage("1DPlayer not defined, ignored");
        return;
    }

    if(fHistogram==nullptr || fHistogram->InheritsFrom("TH2")) {
        gbash_color->WarningMessage("No 1D histogram found, ignored");
        return;
    }

    ROOT::Math::MinimizerOptions::SetDefaultMinimizer(fPlayer->GetMinimizer(),fPlayer->GetAlgorithm());
    ROOT::Math::MinimizerOptions::SetDefaultTolerance(fPlayer->GetTolerance());
    ROOT::Math::MinimizerOptions::SetDefaultPrintLevel(fPlayer->GetPrintLevel());

    Int_t NPars = 3;

    delete fBackFunction;
    fBackFunction = new TF1("MyFit", this, &CXBgdFit::FuncBackground, fBackgd.front(), fBackgd.back(), NPars, "CXBgdFit", "FuncBackground");

    fBackFunction->SetParName(0, "BkgConst");
    fBackFunction->SetParName(1, "BkgSlope");
    fBackFunction->SetParName(2, "BkgExp");

    fBackFunction->SetNpx(1000);
    fBackFunction->SetLineColor(kRed);

    // Copy only the ranges contains within the arrows
    TH1 *HistoToFit = dynamic_cast<TH1*>(fHistogram->Clone());
    HistoToFit->Reset();

    for(size_t i=0; i<fBackgd.size() ; i+=2) {
        Int_t binmin = fHistogram->GetXaxis()->FindBin(fBackgd.at(i));
        Int_t binmax = fHistogram->GetXaxis()->FindBin(fBackgd.at(i+1));

        for(int ibin=binmin ; ibin<=binmax ; ibin++) {
            HistoToFit->SetBinContent(ibin,fHistogram->GetBinContent(ibin));
            HistoToFit->SetBinError(ibin,fHistogram->GetBinError(ibin));
        }
    }

    HistoToFit->GetXaxis()->SetRangeUser(fBackgd.front(), fBackgd.back());

    //Calc Bckd
    fBackFunction->SetParameter(0, HistoToFit->GetBinContent(HistoToFit->FindBin(fBackgd.front())));
    fBackFunction->SetParLimits(0, 0.,HistoToFit->GetMaximum());

    fBackFunction->SetParameter(1, 0);
    fBackFunction->SetParLimits(1, -50., 0.);
    fBackFunction->SetParameter(2, 0.);
//    fBackFunction->SetParLimits(2, -1., 0.);

    if(!fPlayer->fUseBgdPol1)
        fBackFunction->FixParameter(1,0);
    if(!fPlayer->fUseBgdExp)
        fBackFunction->FixParameter(2,0);

    TString FitOpt = "R0S";
    if(fPlayer->GetPrintLevel()>0) FitOpt +="V";
    TFitResultPtr r = HistoToFit->Fit(fBackFunction,FitOpt.Data(),FitOpt.Data());
    ostringstream text;

    if(r.Get() == nullptr) {
        gbash_color->WarningMessage("Oups... Error in fitting histogram");
        return;
    }

    text << "Fit results :";
    cout<<text.str()<<endl;fPlayer->PrintInListBox(text.str(),kPrint);text.str("");
    text << "Status: ";
    if(r->IsValid())
        text << " Successeful" << endl;
    else
        text << " Failed" << endl;
    cout<<text.str();
    if(r->IsValid())
        fPlayer->PrintInListBox(text.str(),kPrint);
    else
        fPlayer->PrintInListBox(text.str(),kError);
    text.str("");

    double Area = fHistogram->Integral(fHistogram->GetXaxis()->FindBin(fBackgd.front()),fHistogram->GetXaxis()->FindBin(fBackgd.back()));
    double BgdArea = fBackFunction->Integral(fBackgd.front(),fBackgd.back(),1e-6)/fHistogram->GetBinWidth(1);
    double CorrArea = Area-BgdArea;
    double CorrAreaErr  = sqrt(4*Area + 4*BgdArea);

    double Mean = fHistogram->GetXaxis()->GetBinCenter(fHistogram->GetMaximumBin());

    Double_t Area_eff = 0.;
    Double_t Area_eff_err = 0.;
    if(fWorkspace && fWorkspace->fEfficiencyFunction) {
        double eff = fWorkspace->fEfficiencyFunction->Eval(Mean);
        Area_eff = CorrArea * eff;
        double error = 0.;
        if(fWorkspace->fEfficiencyErrors) error = fWorkspace->fEfficiencyErrors->GetBinError(fWorkspace->fEfficiencyErrors->FindBin(Mean));
        Area_eff_err = Area_eff * sqrt(CorrAreaErr*CorrAreaErr/(CorrArea*CorrArea) + error*error/(eff*eff));
    }

    text<<left<<setw(11)<<"Integral"<<": "<<setprecision(7)<<setw(10)<<Area<<" ("<<setprecision(7)<<setw(10)<<sqrt(Area)<<")";
    cout<<text.str()<<endl;
    fPlayer->PrintInListBox(text.str(),kInfo);
    text.str("");

    text<<left<<setw(11)<<"Bgd Area"<<": "<<setprecision(7)<<setw(10)<<BgdArea<<" ("<<setprecision(7)<<setw(10)<<sqrt(BgdArea)<<")";
    cout<<text.str()<<endl;
    fPlayer->PrintInListBox(text.str(),kInfo);
    text.str("");

    text<<left<<setw(11)<<"Peak Area"<<": "<<setprecision(7)<<setw(10)<<CorrArea<<" ("<<setprecision(7)<<setw(10)<<sqrt(CorrArea)<<")";
    cout<<text.str()<<endl;
    fPlayer->PrintInListBox(text.str(),kInfo);
    text.str("");

    if(Area_eff>0.) {
        text<<left<<setw(11)<<"Mean"<<": "<<setprecision(7)<<setw(10)<<Mean;
        cout<<text.str()<<endl;
        fPlayer->PrintInListBox(text.str(),kInfo);
        text.str("");
        text<<left<<setw(11)<<Form("%s area",fWorkspace->GetName())<<": "<<setprecision(7)<<setw(10)<<Area_eff<<" ("<<setprecision(7)<<setw(10)<<Area_eff_err<<")";
        cout<<text.str()<<endl;
        fPlayer->PrintInListBox(text.str(),kInfo);
        text.str("");
    }

    fBackFunction->Draw("same");

    fPlayer->GetMainWindow()->RefreshPads();

    delete HistoToFit;
}

Double_t CXBgdFit::FuncBackground(Double_t*xx,Double_t*pp)
{
    Double_t x   = xx[0];


    Double_t Back_const = pp[0];
    Double_t Back_slope = pp[1];
    Double_t Back_Exp = pp[2];

    Double_t f_tot = (Back_const + (x-fBackgd.front())*Back_slope)*exp((x-fBackgd.front())*Back_Exp);

    return f_tot;
}
