#!/usr/bin/env python3
"""
cross_diff.py

Compare outputs from Lean, Coq, and GPT proof scripts line by line.
Generate a summary of differences.
"""

import subprocess
import json
import difflib
import sys
from pathlib import Path

def run_lean(file_path):
    """Run Lean on the given .lean file and capture its proof state."""
    result = subprocess.run(["lean", file_path], capture_output=True, text=True)
    return result.stdout

def run_coq(file_path):
    """Run Coq on the given .v file and capture its message output."""
    result = subprocess.run(["coqc", file_path], capture_output=True, text=True)
    return result.stdout, result.stderr

def run_gpt_verify(file_path):
    """Run GPT verification script to produce a JSON report."""
    result = subprocess.run(["python3", "gpt_verify.py", file_path], capture_output=True, text=True)
    return json.loads(result.stdout)

def compare_outputs(lean_out, coq_out, gpt_report):
    """Compare Lean, Coq, and GPT outputs and print differences."""
    print("=== Lean Output ===")
    print(lean_out)
    print("=== Coq Output ===")
    print(coq_out[0] + "\n" + coq_out[1])
    print("=== GPT Report ===")
    print(json.dumps(gpt_report, indent=2))

    # Simple textual diff between Lean and Coq outputs
    lean_lines = lean_out.splitlines()
    coq_lines = (coq_out[0] + coq_out[1]).splitlines()
    diff = difflib.unified_diff(lean_lines, coq_lines, fromfile="lean", tofile="coq", lineterm="")
    print("=== Diff between Lean and Coq ===")
    for line in diff:
        print(line)

if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Usage: cross_diff.py <proof_file_base>")
        sys.exit(1)

    base = sys.argv[1]
    lean_file = f"{base}.lean"
    coq_file = f"{base}.v"

    lean_out = run_lean(lean_file)
    coq_out = run_coq(coq_file)
    gpt_report = run_gpt_verify(base)

    compare_outputs(lean_out, coq_out, gpt_report)
