#include <glsc3d_3.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#define draw
#define capture

#define LX (2.0)//(10.0)
#define NX (600)

#define Max_t (20000)
#define dt (0.001)
#define cut (100)

#define mu (5.0)
#define seed (500000)
#define kappa (0.1)

#define axis_max (10.0)//(20.0)//(80.0)
#define axis_max2 (2.0)
#define axis_bottom2 (-2.0)


double absv(double x){
	double v;
	v=x;
	if(v<0)v=-v;
	return v;
}

double peri_d(double x){
	double d;
	d = absv(x);
	if(d > LX/2.0) d = LX - d;
	return d;
}

double kj(double x, double dj){
	return 1/( 2.0 * sqrt( dj ) * sinh( LX/2.0 / sqrt( dj ) ) ) * cosh( ( LX/2.0 - absv(x) ) / sqrt( dj ) );
}

double attr_p(double d){
	double v, R;
	v=0.0;
	R=1.0;
	if( d < R )v = ( R - d );
	return v;
}

double W(double x, double y){

	//Gauss_potential// //set mu=5.0//
	return exp( -5 * peri_d( x - y ) * peri_d( x - y ) );
	
	//Mexican-hat//
//	return kj( peri_d( x - y ), 0.1  ) - kj( peri_d( x - y ), 3.0  );

	//Cell adhesion//
	//attractive potential//
//	return attr_p(peri_d( x - y ));
}

void abc_peri_def(double d, double r, double a[], double b[], double c[], int n){
	int i;
	
	for(i=0;i<n;i++){
		a[i]=1+2.0*d*r;
		b[i]=-d*r;
		c[i]=-d*r;
	}
}

void p_def(double dx, double p[], double u[], int n){
	
	int i,j;
	double inte1, inte2;
	
	for(i=0;i<n;i++)
	{
		inte1 = 0.0;
		inte2 = 0.0;
		for(j=0;j<n;j++){
			inte1 = inte1 + u[j] * ( W( (i+1)*dx, j*dx ) -  W( i*dx, j*dx ) ); 
			inte2 = inte2 + u[j] * ( W( (i-1)*dx, j*dx ) -  W( i*dx, j*dx ) );
		}
		if(i <NX-1)inte1 = inte1/dx * ( u[i+1 ]+u[i] )/2.0 ;
		if(i==NX-1)inte1 = inte1/dx * ( u[ 0  ]+u[i] )/2.0 ;
		if( i==0  )inte2 = inte2/dx * ( u[NX-1]+u[i] )/2.0 ;
		if( i > 0 )inte2 = inte2/dx * ( u[i-1 ]+u[i] )/2.0 ;
		
		p[i]=u[i] - dt*mu*(inte1+inte2);
	}
}

void LU_peri_division(double a[], double b[], double c[], double L[][3], double U[][2], int n){
	
	int i;
	double sum;
	L[0][0]=a[0];
	U[0][0]=c[0]/L[0][0];
	U[0][1]=b[0]/L[0][0];
	
	for(i=1;i<=n-3;i++){
		L[i][1] = b[i];
		L[i][0] = a[i]-U[i-1][0]*L[i][1];
		U[i][0] = c[i]/L[i][0];
		U[i][1] = -L[i][1]*U[i-1][1]/L[i][0];
	}
	
	i=n-2;
	L[i][1]=b[i];
	L[i][0]=a[i]-U[i-1][0]*L[i][1];
	U[i][1]= ( c[i] - L[i][1]*U[i-1][1] )/L[i][0];
	L[0][2]=c[n-1];
	
	for(i=1;i<=n-3;i++){
		L[i][2] = - U[i-1][0]*L[i-1][2];
	}	
	
	i=n-2;
	L[i][2]=b[n-1] -U[i-1][0]*L[i-1][2];
	sum=0.0;
	for(i=0;i<=n-2;i++){
		sum=sum+L[i][2]*U[i][1];
			
	}
	
	L[n-1][2]=a[n-1]-sum;
}

void Lu_peri_solve(double p[], double L[][3], double U[][2], double x[], int n){
	int i;

	double y[n], sum;
	
	y[0]=p[0]/L[0][0];
	
	for(i=1;i<=n-2;i++){
		y[i] = ( p[i] - L[i][1]*y[i-1] )/ L[i][0];
	}

	sum=0.0;
	for(i=0;i<=n-2;i++){
		sum = sum + L[i][2] * y[i];
	}
	y[n-1] = ( p[n-1] -sum )/L[n-1][2];

	x[n-1] = y[n-1];
	x[n-2] = y[n-2] - U[n-2][1] * x[n-1];
	for(i=n-3;i>=0;i--){
		
		x[i] = y[i] - U[i][0]*x[i+1] - U[i][1]*x[n-1];
	}
}

void plot_set(void){
	
	//highdpi mode for retina display//
	g_enable_highdpi();
	g_set_antialiasing(4);
	
	g_init("WINDOW", 400, 400);//800, 400); 
	g_scr_color(1,1,1);
	g_cls();	
	
	#ifdef capture
		g_capture_set("");	
	#endif			
	
	g_def_scale_2D(1, 0, LX, 0.0, axis_max, 40.0, 40.0, 320.0, 320.0);
	g_def_scale_2D(2, 0, LX, axis_bottom2, axis_max2, 440.0, 40.0, 320.0, 320.0);
}

void text(int n){

	//text//
	g_text_size(15);
	g_text_standard(30,30, "time=%.3lf,",n*dt);
	g_text_standard(450,30, "dt=%.3lf,",dt);
	g_text_standard(530,30, "mu=%.2lf,",mu);
		g_text_standard(630,30, "seed=%d,",seed);
	
	//sel(1)//
	g_text_size(12);
	g_text_standard(20,50, "%.0lf", axis_max);
	g_text_standard(31,359, "0");
	g_text_standard(40,371, "0");
	g_text_standard(347,371, "%.0lf",LX);
	
	//sel(2)//
	g_text_standard(415,50, "%.2lf", axis_max2);
	g_text_standard(410,359, "%.2lf", axis_bottom2);
	g_text_standard(440,371, "0");
	g_text_standard(747,371, "%.0lf",LX);
}

void plot(double dx, double u[], int n){
	
	int i;
	g_sel_scale(1);
	g_line_width(2);
	g_line_color(0,0,0,1);
	g_box_2D(0, LX, 0.0, axis_max, G_YES, G_NO);

	g_line_color(1,0,0,1);

	for(i=0;i<NX;i++){
		g_move_2D(i*dx, u[i]);
		if(i<NX-1)g_plot_2D((i+1)*dx, u[i+1]);
		if(i==NX-1)g_plot_2D((i+1)*dx, u[0]);
	}

	g_sel_scale(2);
	g_line_width(2);
	g_line_color(0,0,0,1);
	g_box_2D(0, LX, axis_bottom2, axis_max2, G_YES, G_NO);

	g_line_color(0,0,1,1);
	for(i=0;i<NX;i++){
		g_move_2D(i*dx, W(i*dx - LX/2.0, 0.0  ) );
		g_plot_2D((i+1)*dx, W( (i+1)*dx - LX/2.0, 0.0 ) );
	}
	g_line_color(0,0,0,1);
	g_move_2D(0.0, 0.0);
	g_plot_2D(LX, 0.0);
	
	text(n);
}



int main(void){
	
	int i, n;
	double dx;
	
	double u[NX],u_new[NX];
	double L[NX][3], U[NX-1][2], p[NX], x[NX], m1[NX], m2[NX], m3[NX];
	
	dx=LX/NX;
	srand(seed);
	
	for(i=0;i<NX;i++){
		u[i] = 1.0 + 0.1*sin(12*i*dx)*rand()/RAND_MAX;;		//Guass potential & Mexican hat//
//		u[i] = 1.0 + 0.1*sin(10*i*dx)*rand()/RAND_MAX;; 	//attra_p//
		u_new[i] = 0.0;
	}
	
	#ifdef draw
		plot_set();
		plot(dx, u, 0);
		g_finish(); 
		#ifdef capture
				g_capture();
		#endif
		g_cls();
	#endif	
	
	abc_peri_def( 1.0,  dt/(dx*dx),  m1,  m2,  m3,  NX);
	LU_peri_division( m1,  m2,  m3,  L,  U, NX);
	
	for(n=1;n<=Max_t;n++){
		p_def(dx, p, u, NX);
		Lu_peri_solve( p, L, U, x, NX);
		for(i=0;i<NX;i++){
			u[i]=x[i];
			if(n==Max_t)printf("u[%d]=%lf\n",i,u[i]);
		}

	#ifdef draw
		if(n%cut==0){
			plot(dx, u, n);
			g_finish();
			#ifdef capture
					g_capture();
			#endif
			g_cls();
		}
	#endif
	}
}





























