#include <glsc3d_3.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#define draw
#define capture

#define axis_max (10.0)//(80.0)
#define axis_max2 (2.0)
#define axis_bottom2 (-2.0)
#define axis_max3 (2.0)
#define axis_bottom3 (-2.0)
#define axis_max4 (350.0)
#define axis_bottom4 (-350.0)

#define LX (2.0)//(10.0)
#define NX (600)

#define Max_t (20000)
#define dt (0.001)
#define cut (100)

#define L1 (1.0)
#define L2 (cosh(LX/2.0))
#define eta ( - ( L2 + L1 ) / 2.0 )

#define M (7)//(2) //Number of vj//
#define num (M-1)
#define d0 (1000000.)	//For Lagrange polynomial//

//Mexican hat//
//#define d0 (0.1)
// #define d1 (3.0)
//
// #define a0 (1.0)
// #define a1 (-1.0)

#define eps (0.001)
#define seed (500000)
#define mu1 (5.)


double absv(double x){
	double v;
	v=x;
	if(v<0)v=-v;
	return v;
}

double peri_d(double x){
	double d;
	d = absv(x);
	if(d > LX/2.0) d = LX - d;
	return d;
}

double kj(double x, double dj){
	return 1/( 2.0 * sqrt( dj ) * sinh( LX/2.0 / sqrt( dj ) ) ) * cosh( ( LX/2.0 - absv(x) ) / sqrt( dj ) );
}

double convolution_kj(double dx, double x, double dj, double u[]){
	int i;
	double v;
	v=0.0;
	
	for(i=0;i<NX;i++){
		v = v + kj(peri_d( x - i*dx ),dj) * u[i];
	}
	v = v*dx;
	
	return v;
}

void abc_peri_def(double d, double r, double a[], double b[], double c[], int n){
	int i;
	
	for(i=0;i<n;i++){
		a[i]=1+2.0*d*r;
		b[i]=-d*r;
		c[i]=-d*r;
	}
}

double deriv_f_p(int i, double dx, double u[]){
	double v;
	if(i < NX-1) v = (u[i+1]-u[i])/dx;
	if(i==NX-1 ) v = (u[ 0 ]-u[i])/dx;
	
	return v;
}

double deriv_b_p(int i, double dx, double u[]){
	double v;
	if(i == 0 ) v = (u[i] - u[NX-1])/dx;
	if(i  > 0 ) v = (u[i] - u[i-1] )/dx;
	
	return v;
}

void p_def_u(double dx, double p[], double u[], double v[][NX], double a[], int n){
	
	int i,j;
	double aggre, r, rho1, rho2, sum1, sum2;
	
	r = dt/(dx);
	
	for(i=0;i<n;i++)
	{
		if(i <NX-1)rho1 =   (u[i+1 ]+u[i])/2.0  ;
		if(i==NX-1)rho1 =   (u[ 0  ]+u[i])/2.0  ;
		if( i==0  )rho2 =   (u[NX-1]+u[i])/2.0  ;
		if( i > 0 )rho2 =   (u[i-1 ]+u[i])/2.0  ;
		
		sum1=0.0;
		sum2=0.0;
		for(j=0;j<M;j++){
			sum1 = sum1 + a[j]*( deriv_f_p(i, dx, v[j]) );
			sum2 = sum2 + a[j]*( deriv_b_p(i, dx, v[j]) );
		}

		aggre = r * ( rho1 * sum1 - rho2 * sum2   );
		
		p[i]=u[i] -  mu1 * aggre;
	}
}

void p_def_v(double p[], double u[], double v[], int n){

	int i;
	
	for(i=0;i<n;i++){
		p[i] = v[i] + dt/eps * ( - v[i]+ u[i] );
	}
}

void LU_peri_division(double a[], double b[], double c[], double L[][3], double U[][2], int n){
	
	int i;
	double sum;
	L[0][0]=a[0];
	U[0][0]=c[0]/L[0][0];
	U[0][1]=b[0]/L[0][0];
	
	for(i=1;i<=n-3;i++){
		L[i][1] = b[i];
		L[i][0] = a[i]-U[i-1][0]*L[i][1];
		U[i][0] = c[i]/L[i][0];
		U[i][1] = -L[i][1]*U[i-1][1]/L[i][0];
	}
	
	i=n-2;
	L[i][1]=b[i];
	L[i][0]=a[i]-U[i-1][0]*L[i][1];
	U[i][1]= ( c[i] - L[i][1]*U[i-1][1] )/L[i][0];
	L[0][2]=c[n-1];
	
	for(i=1;i<=n-3;i++){
		L[i][2] = - U[i-1][0]*L[i-1][2];
	}	
	
	i=n-2;
	L[i][2]=b[n-1] -U[i-1][0]*L[i-1][2];
	sum=0.0;
	for(i=0;i<=n-2;i++){
		sum=sum+L[i][2]*U[i][1];
	}
	
	L[n-1][2]=a[n-1]-sum;
}

void Lu_peri_solve(double p[], double L[][3], double U[][2], double x[], int n){

	int i;

	double y[n], sum;
	
	y[0]=p[0]/L[0][0];
	
	for(i=1;i<=n-2;i++){
		y[i] = ( p[i] - L[i][1]*y[i-1] )/ L[i][0];
	}

	sum=0.0;
	for(i=0;i<=n-2;i++){
		sum = sum + L[i][2] * y[i];
	}
	y[n-1] = ( p[n-1] -sum )/L[n-1][2];

	x[n-1] = y[n-1];
	x[n-2] = y[n-2] - U[n-2][1] * x[n-1];
	for(i=n-3;i>=0;i--){
		x[i] = y[i] - U[i][0]*x[i+1] - U[i][1]*x[n-1];
	}
}

double W(double x){
	
	return exp( -5 * x * x ); //- exp( -  x * x );
}

double f(double x){
	return W( LX/2.0 - log(x + sqrt( x*x - 1  ))  );
}

//Chebyshev node//
double r(int n, int j){
	return (L2+L1)/2.0 + (L2-L1)/2.0 * cos( (2*j+1)*M_PI / (2*n) );
}

double prod(int n, int j){
	int i;
	double v;
	v=1;
	for(i=0;i<=n;i++){
		if(i!=j)v = v * ( r(n+1,j) - r(n+1,i) ) ;
	}
	
	return v;
}

double zeta(int n, int j){
	return f( r(n+1,j) ) / prod( n, j);
}

int factorial(int n){
	int i, v;
	v=1;
	for(i=1;i<=n; i++){
		v=v*i;
	}
	return v;
}

double binomial(double n, int k){
	int i;
	double v=1.0;
	
	for(i=0;i<k;i++){
		v = v*(n-i);
	}
	return v/factorial(k);
}

double C(int n, int k){
	return n/2.0 * pow(-1, k) * factorial(n-k-1)/( factorial(k) * factorial(n - 2 * k ) ) * pow(2, n-2*k);
}

double mu (int n, int k, int j){
	return C(n,k)*pow( 2/(L2-L1), n - 2*k )* binomial(n-2*k,j)*pow(eta,j);
}

int Guass_sy(double n){
	int v;
	v=n;
	return v;
}

double xi(int n, int k){
	int i, terminal;
	double sum;
	sum=0.0;
	
	if(n%2==0)terminal = Guass_sy(n/2.0) - Guass_sy((k+1)/2.0);
	if(n%2==1)terminal = Guass_sy(n/2.0) - Guass_sy((k)/2.0);

	for(i=0;i<=terminal ; i++  ){
		sum = sum + mu(n, i, n - 2*i - k);
	}
	return sum;
}

double beta(int n, int l, int j){
	int i;
	double sum;
	sum=0.0;
	
	for(i=l;i<=n-1;i++){
		sum=sum + pow(r(n,j), i-l ) * xi( n, i+1);
	}
	return sum;
}

double b(int n, int l){
	int i;
	double sum=0.0;
	
	for(i=0;i<=n;i++){
		sum=sum+zeta(n,i)*beta(n+1,l,i);
	}
	sum = sum /pow(2,n)*pow( (L2 - L1)/2.0, n+1 );
	return sum;
}

//Lagrange polynomial//
double Lp(double x){
	int i;
	double sum=0.0;
	
	for(i=0;i<=num;i++){
		sum = sum + b(num,i)*pow(x,i);
	}
	return sum;
}

double delta(int n, int j){
	double v;
	v = binomial (n, (n-j)/2.0 )*pow(1.0/2.0, n-1);
	if (j == 0) v = binomial (n, (n-j)/2.0 )*pow(1.0/2.0, n);
	return v;
}

double alpha(int n, int j){
	int i;
	double sum=0.0;
	
	for(i=j;i<=n; i+=2){
		sum =sum + b(n,i)*delta(i,j);
	}
	return sum;
}

double HCSE(double x){
	int i;
	double sum=0.0;

	for(i=0;i<=num;i++){
		sum = sum + alpha(num,i) * cosh(i*(LX/2 - absv(x) ));
	}
	return sum;
}

void plot_set(void){
	
	//highdpi mode for retina display//
	g_enable_highdpi();
	g_set_antialiasing(4);
	
	g_init("WINDOW", 400, 400);//800, 800); 
	g_scr_color(1,1,1);
	g_cls();	
	
	#ifdef capture
		g_capture_set("");	
	#endif			
	
	g_def_scale_2D(1, 0, LX, 0.0, axis_max, 40.0, 40.0, 320.0, 320.0);
	g_def_scale_2D(2, -LX/2, LX/2, axis_bottom2, axis_max2, 440.0, 40.0, 320.0, 320.0);
	g_def_scale_2D(3, L1, L2, axis_bottom3, axis_max3, 440.0, 440.0, 320.0, 320.0);
	g_def_scale_2D(4, 0, num, axis_bottom4, axis_max4, 40.0, 440.0, 320.0, 320.0);
}

void text(int n){

	//text//
	g_text_size(15);
	g_text_standard(30,30, "time=%.3lf,",n*dt);

	g_text_standard(30,412, "mu1=%.2lf,",mu1);
	g_text_standard(120,412, "eps=%.3lf,",eps);
	g_text_standard(200,412, "seed=%d,",seed);
	g_text_standard(320,412, "dt=%.5lf,",dt);
	
	//sel(1)//
	g_text_size(12);
	g_text_standard(20,50, "%.0lf", axis_max);
	g_text_standard(31,359, "0");
	g_text_standard(40,371, "0");
	g_text_standard(347,371, "%.0lf",LX);

	//sel(2)//
	g_text_standard(420,50, "%.1lf", axis_max2);
	g_text_standard(416,359, "%.1lf", axis_bottom2);
	g_text_standard(440,371, "%.1lf",-LX/2);
	g_text_standard(747,371, "%.1lf",LX/2);
	
	//sel(3)//
	g_text_standard(420,450, "%.1lf", axis_max3);
	g_text_standard(416,759, "%.1lf", axis_bottom3);
	g_text_standard(440,771, "%.1lf",L1);
	g_text_standard(747,771, "%.3lf",L2);
	
	//sel(4)//
	g_text_size(12);
	g_text_standard(10,450, "%.0lf",  axis_max4);
	g_text_standard(10,759, "%.0lf",  axis_bottom4);
	g_text_standard(40,771, "0");
	g_text_standard(347,771, "%.d",num);
}

void plot(double dx, double u[], double v[][NX], double a[], double r1[], double r2[], double r3[], int n){
	
	int i,j;
	double dx2;
	
	dx2=(L2-L1)/NX;
	
	g_sel_scale(1);
	g_line_width(2);
	g_line_color(0,0,0,1);
	g_box_2D(0, LX, 0.0, axis_max, G_YES, G_NO);

	g_line_color(1,0,0,1);

	for(i=0;i<NX;i++){
		g_line_color(1,0,0,1);
		g_move_2D(i*dx, u[i]);
		if(i<NX-1)g_plot_2D((i+1)*dx, u[i+1]);
		if(i==NX-1)g_plot_2D((i+1)*dx, u[0]);
		
		for(j=0;j<M;j++){
			g_line_color(r1[j], r2[j] , r3[j],1);
			if(j==0)g_line_color(0,1,0,1);
			if(j==1)g_line_color(0,0,1,1);
			g_move_2D(i*dx, v[j][i]);
			if(i<NX-1)g_plot_2D((i+1)*dx, v[j][i+1]);
			if(i==NX-1)g_plot_2D((i+1)*dx, v[j][0]);
		}
	}

	g_sel_scale(2);
	g_line_width(2);
	g_line_color(0,0,0,1);
	g_box_2D(-LX/2, LX/2, axis_bottom2, axis_max2, G_YES, G_NO);

	for(i=0;i<NX;i++){
		g_line_color(0,0,1,1);
		g_move_2D(-LX/2+i*dx,     W(-LX/2+i*dx));
		g_plot_2D(-LX/2+(i+1)*dx, W(-LX/2+(i+1)*dx));

		g_line_color(1, 120.0/238.0,0,1);
		g_move_2D(-LX/2+i*dx,     HCSE(-LX/2+i*dx));
		g_plot_2D(-LX/2+(i+1)*dx, HCSE(-LX/2+(i+1)*dx));
	}
	
	g_sel_scale(3);
	g_line_width(2);
	g_line_color(0,0,0,1);
	g_box_2D(L1, L2, axis_bottom3, axis_max3, G_YES, G_NO);
	
	for(i=0;i<NX;i++){
		g_line_color(0,0,1,1);
		g_move_2D(L1+i*dx2, f(L1+i*dx2));
		g_plot_2D(L1+(i+1)*dx2, f(L1+(i+1)*dx2));
		
		g_line_color(1, 120.0/238.0,0,1);
		g_move_2D(L1+i*dx2, Lp(L1+i*dx2));
		g_plot_2D(L1+(i+1)*dx2, Lp(L1+(i+1)*dx2));
	}
	
	g_marker_size(10.0);
	g_marker_type(1);
	g_marker_color(0,0,0,1);
	for(i=0;i<=num+1;i++){
		g_marker_2D(r(num+1,i), f(r(num+1,i)));
	}
	
	g_sel_scale(4);
	g_line_width(2);
	g_line_color(0,0,0,1);
	g_box_2D(0, num, axis_bottom4, axis_max4, G_YES, G_NO);
	g_move_2D(0.0,0.0);
	g_plot_2D(M,0.0);
	
	for(i=0;i<=num;i++){
		g_marker_color(0,0,0,1);
		g_marker_2D(i, alpha(num,i));

		g_marker_color(1,0,0,1);
		g_marker_2D(i, a[i]);
	}
	
	text(n);
}


int main(void){
	
	int i, j, n;
	double dx;
	
	double u[NX],u_new[NX];
	double L[NX][3], U[NX-1][2], p[NX], x[NX], m1[NX], m2[NX], m3[NX];
	
	double v[M][NX], v_new[M][NX], d[M], a[M];
	double Lv[M][NX][3], Uv[M][NX-1][2], pv[M][NX], xv[M][NX], m1v[M][NX], m2v[M][NX], m3v[M][NX];

	double r1[num],r2[num],r3[num];
	//v_j rgb//
	for(i=0;i<num;i++){
		r1[i]=1.0*rand()/RAND_MAX;
		r2[i]=1.0*rand()/RAND_MAX;
		r3[i]=1.0*rand()/RAND_MAX;
	}
	
	dx=LX/NX;

	//Approximation_Guass//
	d[0]=d0;
	a[0]=2*(LX/2) * alpha(num,0);
	for(i=1; i<M; i++){
		d[i]=1.0/(i*i);
		a[i]=2*alpha(num,i)*sinh(i*LX/2)/i;		
	}
	
	//Mexican-hat (M=2)//
//	d[0]=d0;
//	d[1]=d1;
//	a[0]=a0;
//	a[1]=a1;
	
	
	for(i=0;i<M;i++){
		printf("alpha[%d]=%lf\n",i, alpha(num,i));
		printf("a[%d]=%lf\n",i, a[i]);
	}
	
	srand(seed);
	
	for(i=0;i<NX;i++){
		u[i] = 1.0 + 0.1*sin(12*i*dx)*rand()/RAND_MAX;;
		u_new[i] = 0.0;
	}
	
	for(i=0;i<NX;i++){
		for(j=0;j<M;j++){
			v[j][i] = convolution_kj( dx, i*dx, d[j], u);
			v_new[j][i]=0.0;
		}
	}
	
	#ifdef draw
		plot_set();
		plot(dx, u, v, a, r1, r2, r3, 0);
		g_finish(); 
		#ifdef capture
				g_capture();
		#endif
		g_cls();
	#endif	
	
	abc_peri_def( 1.0,  dt/(dx*dx),  m1,  m2,  m3,  NX);
	LU_peri_division( m1,  m2,  m3,  L,  U, NX);
	
	for(j=0;j<M;j++){
		abc_peri_def( d[j],  dt/(eps*dx*dx),  m1v[j],  m2v[j],  m3v[j],  NX);
		LU_peri_division( m1v[j],  m2v[j],  m3v[j],  Lv[j],  Uv[j], NX);
	}
	
	for(n=1;n<=Max_t;n++){
		
		p_def_u(dx, p, u, v, a, NX);
		Lu_peri_solve( p, L, U, x, NX);
		for(i=0;i<NX;i++){
			u[i]=x[i];
		}

		for(j=0;j<M;j++){
			p_def_v( pv[j], u, v[j], NX);
			Lu_peri_solve( pv[j], Lv[j], Uv[j], xv[j], NX);
			for(i=0;i<NX;i++){
				v[j][i]=xv[j][i];
			}
		}
		
	#ifdef draw
		if(n%cut==0){
			plot(dx, u, v, a, r1, r2, r3, n);
			g_finish();
			#ifdef capture
					g_capture();
			#endif
			g_cls();
		}
	#endif
	}
}






























