#------------------------------------------------------------------------------
#********* TopEros: An integrated TOPMODEL - MUSLE modelling platform**********
#------------------------------------------------------------------------------

## This is a python implementation of TopEros.
## The program implements a Monte carlo Simulation (MCS) to determine 5
## parameters of TOPMODEL: This TOPMODEL assumes (LnT0-LnTe) = 0.
## It then calculates non-gulley soil erosion at pixel scale within the
## sub-daily routine using MUSLE; Gulley Erosion by equation of detachment by
## channel flow as well as detachment by rain drop.
## The hydrological model also works when all water in UZ leaks to saturated
## zone within a single time step
## The module for infiltration excess overland flow is omitted
## Units of length and time are consistent with units of input observation data
## TOPMODEL component is adopted from Fortran Code (Keith Beven); MATLAB code 
## (Hiromu Okazawa, 2015) and Python Code (Okiria Emmanuel, 2018)
## Written by Okiria Emmanuel (2024) during PhD esearch at Gifu University

# IMPORTING NECESSARY LIBRARIES
import platform
import os
import time # time library to help keep count of code run time
start_time = time.time()
import numpy as np
import pandas as pd
import rasterio
from rasterio.features import geometry_mask
import geopandas as gpd
import matplotlib.pyplot as plt
import matplotlib.colors as mcolors
import matplotlib.dates as mdates
from datetime import datetime, timedelta
import pygame


#****************************SECTION 1: FUNCTIONS******************************

#### Dynamic assignment of PC_NAME based on OS
if platform.system().lower() == 'windows':
    PC_NAME='okiri'
    #PC_NAME='emmanuel'
elif platform.system().lower() ==' darwin':
    PC_NAME = 'Emma'
else:
    PC_NAME = 'default'

#### Function to generate file paths
def generate_file_path(pc_name, *path_parts):
    if platform.system() == 'Windows':
        return os.path.join(f'C:\\Users\\{pc_name}\\Dropbox\\NIES\\NIES\\useful_data\\TopEros', *path_parts)
    else:
        return os.path.join(f'/Users/{pc_name}/Dropbox/NIES/NIES/useful_data/TopEros', *path_parts)
    
#### Function to read _xlsx file ( load dataframes) with parse_dates to handle date parsing
def read_xlsx_w_date(file_path, sheet_name, parse_dates):
    return pd.read_excel(file_path, sheet_name=sheet_name, parse_dates=parse_dates)


#### Function to read out excel _csv files
def read_csv(file_path):
    return pd.read_csv(file_path, low_memory=False).values

#### Function to read out raster values into a Numpy array
def read_raster(file_path):
    with rasterio.open(file_path) as ds:
        return ds.read()[0]

#### Plot the raster layer over  shapefile: Continuous values
def plot_data_array_over_shapefile(shapefile_path, data_array, num_classes, cmap='viridis', alpha=1, cbar_label='', plot_title=''): # Made alpha 1 from 0.5
    """
    Plots numerical data on top of a shapefile with automatically set class ranges.

    Parameters:
    - shapefile_path: str, path to the shapefile.
    - data_array: 2D array, numerical values to overlay on the shapefile.
    - num_classes: int, number of classes for overlay.
    - cmap: str, colormap for the data values.
    - alpha: float, transparency of the overlay.
    - cbar_label: str, label for the colorbar.
    - plot_title: str, title for the plot.

    Returns:
    None
    """

    # Open the shapefile using geopandas
    gdf = gpd.read_file(shapefile_path)

    # Get the bounding box of the shapefile for setting the plot extent
    bounds = gdf.bounds

    # Create a mask for NaN values in the data_array
    nan_mask_data = np.isnan(data_array)

    # Apply NaN mask to data_array
    data_array = np.ma.masked_array(data_array, mask=nan_mask_data)

    # Get the minimum and maximum values from the data_array
    min_val, max_val = np.nanmin(data_array), np.nanmax(data_array)

    # Create a custom colormap for reclassified values
    cmap_category = plt.get_cmap(cmap)

    # Mask NaN values for transparency
    data_array = np.ma.masked_where(nan_mask_data, data_array)

    # Plot the shapefile using geopandas with hollow polygons
    fig, ax = plt.subplots(figsize=(10, 8))
    ax.set_facecolor('white')  # Set the background color of the axes to white
    gdf.plot(ax=ax, alpha=alpha, edgecolor='none', facecolor='none')

    # Overlay the data_array values on top of the shapefile
    im = ax.imshow(data_array, extent=[bounds.minx.values[0], bounds.maxx.values[0],
                                       bounds.miny.values[0], bounds.maxy.values[0]],
                   cmap=cmap_category, alpha=alpha, interpolation='none', aspect='auto',
                   vmin=min_val, vmax=max_val)  # Set vmin and vmax to limit color map range

    # Add colorbar for data_array values with custom label
    cbar = plt.colorbar(im, label=cbar_label)

    # Add title to the plot
    plt.title(plot_title)

    # Show the plot
    plt.show()


#### Plot the raster layer over  shapefile: Discrete values
def plot_data_array_over_shapefile_2(shapefile_path, data_array, num_classes, cmap='viridis', alpha=0.5,
                                   cbar_label='', plot_title=''):
    """
    Plots numerical data on top of a shapefile with automatically set class ranges.

    Parameters:
    - shapefile_path: str, path to the shapefile.
    - data_array: 2D array, numerical values to overlay on the shapefile.
    - num_classes: int, number of classes for overlay.
    - cmap: str, colormap for the data values.
    - alpha: float, transparency of the overlay.
    - cbar_label: str, label for the colorbar.
    - plot_title: str, title for the plot.

    Returns:
    None
    """

    # Open the shapefile using geopandas
    gdf = gpd.read_file(shapefile_path)

    # Get the bounding box of the shapefile for setting the plot extent
    bounds = gdf.bounds

    # Create a mask for NaN values in the data_array
    nan_mask_data = np.isnan(data_array)

    # Apply NaN mask to data_array
    data_array = np.ma.masked_array(data_array, mask=nan_mask_data)

    # Get the unique values from the data_array
    unique_values = np.unique(data_array[~nan_mask_data])

    # Create a custom colormap for discrete values
    colors = plt.get_cmap(cmap, num_classes)
    cmap_category = mcolors.ListedColormap(colors.colors[:num_classes])

    # Mask NaN values for transparency
    data_array = np.ma.masked_where(nan_mask_data, data_array)

    # Plot the shapefile using geopandas with hollow polygons
    fig, ax = plt.subplots(figsize=(10, 8))
    ax.set_facecolor('white')  # Set the background color of the axes to white
    gdf.plot(ax=ax, alpha=alpha, edgecolor='none', facecolor='none')

    # Overlay the data_array values on top of the shapefile
    im = ax.imshow(data_array, extent=[bounds.minx.values[0], bounds.maxx.values[0],
                                       bounds.miny.values[0], bounds.maxy.values[0]],
                   cmap=cmap_category, alpha=alpha, interpolation='none', aspect='auto')

    # Add colorbar for discrete values with custom label
    cbar = plt.colorbar(im, ticks=unique_values, label=cbar_label)

    # Add title to the plot
    plt.title(plot_title)

    # Show the plot
    plt.show()


# Read out files & Data into a Numpy array
# Add pd. suffix to function to create a dataframe

rain = pd.read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', 'Namatala', 'Rain_D__.csv'))) # Daily rainfall
rain_hr = read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', 'Namatala', 'Rain_H__.csv'))) # Hourly rainfall
Qobs = read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', 'Namatala', 'Qobs_D__.csv'))) # Daily observed discharge
Qobs_hr = read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', 'Namatala', 'Qobs_H__.csv'))) # Hourly observed discharge
Qobs_hr = Qobs_hr/24 # Coverting units from [mm/day] to [mm/hr]
ET0 = read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', 'Namatala', 'ET0_D___.csv'))) # Daily Evapotranspiration
ET0_hr = read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', 'Namatala', 'ET0_H___.csv'))) # Hourly Evapotranspiration
TI = read_csv((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'TI______.csv'))) # Topographic Inderx
freq = read_csv((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'freq____.csv'))) # Frequency distribution of Topographic Index

K_raw = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'K_50m___.tif'))) # Soil erodibility factor & its spatial resoltion.
LSF = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'LSp_50m_.tif'))) # A physically derived slope-length factor.
P = np.ones(LSF.shape) # Since no human SWC interventions, P is kept at 1 (Li et al. (2023) guides on values for arable land where human intervention is observed
NDVI_raw = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', '2015____', 'NDVI50m_.tif')))
TIM = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'TI______.tif'))) # Gridded Topographic Index
Deg_S = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'SlopeDeg.tif'))) # Degree Slope
Rad_S=np.radians(Deg_S) # Convert slope to Radians
Fac = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'FacMfd__.tif'))) # Flow accumulation
Sa_raw = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'SndPc50m.tif'))) # Percent sand
Or_raw = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'OrgPc50m.tif'))) # Percent Organic matter
Rod_raw = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'Ro_50m__.tif'))) # Bulk Density [ ]
Veg_pc_raw =  read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'VegPc50m.tif'))) # Vegetation Cover Percentage

catchment_shapefile = gpd.read_file(generate_file_path(PC_NAME, 'Dat', 'Others__', 'Namatala', 'CatPoly_.shp'))


# CLEAN UP RASTER VALUES: THRESHOLDING OPERATIONS
#********************** Harmonising Raster shapes******************************
target_shape=TIM.shape # Target array shape of rasters

# Slicing the larger arrays to match target_shape
K=K_raw[:target_shape[0], :target_shape[1]]
NDVI=NDVI_raw[:target_shape[0], :target_shape[1]]
Sa=Sa_raw[:target_shape[0], :target_shape[1]]
Or=Or_raw[:target_shape[0], :target_shape[1]]
Rod=Rod_raw[:target_shape[0], :target_shape[1]]
Veg_pc=Veg_pc_raw[:target_shape[0], :target_shape[1]]
#******************************************************************************

#***********Cleaning up K varray to eliminate NaN(s): Mask Nan values**********
nan_value_K=-3.40282e+38
K_masked=np.ma.masked_values(K, nan_value_K)

# Cleaning up TIM array
nan_value_TIM=-128
TIM_masked=np.ma.masked_values(TIM, nan_value_TIM)

# Cleaning up LSF array
nan_value_LSF=-3.4028235e+38
LSF_masked=np.ma.masked_values(LSF, nan_value_LSF)

# Cleaning up Rad_S array
nan_value_Rad_S=-5.93905e+36
Rad_S_masked=np.ma.masked_values(Rad_S, nan_value_Rad_S)

# Cleaning up Fac array
nan_value_Fac=-3.4028235e+38
Fac_masked=np.ma.masked_values(Fac, nan_value_Fac)
#******************************************************************************

















#*********************SECTION 2: PRELIMINARY CALCULATIONS**********************
# RANDOM NUMBER GENERATION FOR THE PARAMETER CALIBRATION using MCS
np.random.seed(0) # Resets random number generator to seed 0 per randomisation
MM=100 # Number of parameter sets generated by MCS
RNDM=np.zeros((MM,9)) #,dtype=np.float16) # Creates array, whose elements are 
# zeros, in shape (MM,7)

## m; PARAMETER CONTROLLING THE RATE OF DECLINE OF TRANSMISSIVITY IN THE SOIL 
## PROFILE: RATE OF CHANGE OF RECESSION CURVEs
a0=0 # Minimum m value (changeable)
b0=50  # Maximum m value (changeable)
RNDM[:,:1]=np.random.uniform(a0,b0,(MM,1)) # Random m values; column 0, size=MM

## Te; THE AREAL VALUE OF T0 [L*L/T]
a1=1000 # Minimum Te value (changeable)
b1=10000 # Maximum Te value (changeable)
RNDM[:,1:2]=np.random.uniform(a1,b1,(MM,1)) # Random Te values, column 1,
# size = MM

## td; SATURATED ZONE TIME DELAY PER UNIT STORAGE DEFICIT [T/L]
a2=0 # Minimum td value (changeable)
b2=.03 # Maximum td value (changeable)
RNDM[:,2:3]=np.random.uniform(a2,b2,(MM,1)) # Random td values, column 2,
# size = MM

## SRZinitial; initial root zone storage deficit [L] #Eliminated from parameters
a3=0 # Minimum value (changeable)
b3=0 # Maximum value (changeable)
RNDM[:,3:4]=np.random.uniform(a3,b3,(MM,1)) # Random SRZinitial values, column 3
# , size = MM

## SRmax; MAXIMUM ROOTZONE STORAGE DEFICIT [L] #Rot depth of < 100 is modelrately deep
a4=0 # Minimum value (changeable)
b4=10 # Maximum value (changeable)
RNDM[:,4:5]=np.random.uniform(a4,b4,(MM,1)) # Random SRmax values, column 4,
# size = MM

# READING OUT EXCEL DATASETS FOR TOPMODEL CALIBRATION & TopEros RUN
# "WE WANT THIS OUT OF THE "FOR" LOOP TO SAVE COMPUTATION TIME"


# ASSIGNING THE VALUE OF INITIAL SIMULATED DISCHARGE
Q1=Qobs[0,0] # Initial base flow is set to equal observed dischare at initial
# time step

# ***************NUMBER OF ROWS & COLUMNS IN SELECT ARRAYS*********************

rows_h,cols_h = rain_hr.shape

# Finding number of cells in Area of Interest
nan_mask = np.isclose(TIM, nan_value_TIM, rtol=1e-05, atol=1e-08)
nan_count=np.sum(nan_mask)
rows,cols = TIM.shape
Num_of_catchment_cells = (rows*cols)-nan_count

#What fraction of a catchment is each cell within the catchnent bounds?
freq_1=1/(Num_of_catchment_cells)

#******************************************************************************

# CREATION OF N-D ARRAY STRUCTURES FOR THE CALCULATION OF DISTRIBUTED VARIABLES
zero_matrix=(np.zeros(TI.shape))@(np.transpose(np.zeros(rain.shape))) # Matrix
# multiplication
print('2-D shape is', zero_matrix.shape)
zero_matrix_2=np.zeros((rows_h,rows,cols)) #3-D array: [depth,r,c]
print('3-D shape is', zero_matrix_2.shape)
zero_matrix_2_1=np.zeros((rows_h,len(TI),cols_h))
print('3-D_2 shape is', zero_matrix_2_1.shape)
zero_matrix_3=np.zeros((rows_h,rows,cols,cols_h)) #4-D array: [batch,r,c,depth]
print('4-D shape is', zero_matrix_3.shape)

# ASSIGNING THE VALUE OF INITIAL SIMULATED DISCHARGE
Q1_hr=Qobs_hr[0,0] # Initial obbserved doscharge is set to equal observed 
# discharge at initial time step

#SETTING UP INDICES TO ALLOW FOR WARMUP PERIOD
start_index =0
end_index = len(rain)

# CALCULATING THE AREAL AVERAGE OF TI
Lambda=np.sum(TI*freq)/np.sum(freq) # Lambda is catchment scale areal average
# of TI

#******************************* SECTION 3*************************************
# TOPMODEL IMPLEMENTATION / CALIBRATION
for M in range(MM): # M iteration variable takes on values 0 to (MM-1) for each
# iteration

    # PREALLOCATE ARRAYS　
    Qv=np.zeros(rain.shape) # Qv is lumped daily vertical flux from unsaturated
    #zone (UZ) to saturated zone (SZ) [L/T]: drainage flux into the water table
    #from the unsaturated zone
    Qv_hr=np.zeros(rain_hr.shape) #Lumped hourly vertical flux from UZ to SZ
    Qover=np.zeros(rain.shape) # Qover is lumped daily overland flow for
    Qover_hr=np.zeros(rain_hr.shape) # Lumped hourly overland flow
    Qsub=np.zeros(rain.shape) # Qsub (base flow) is lumped saturated subsurface
    # flow [L/T]
    Qsub_hr=np.zeros(rain_hr.shape) # Lumped hourly subsirface flow: Tricky
    # (Perhaps Qsub_hr changes at daily time scale)
    Qsim=np.zeros(rain.shape) # Qsim is simulated discharge: Qover + Qsub [L/T]
    Qsim_hr=np.zeros(rain_hr.shape)
    ETa=np.zeros(rain.shape) # ETa is actual evepotranspiration [L]
    ETa_hr=np.zeros(rain_hr.shape) # Actual hourly evapotranspiration
    SRZ=np.zeros(rain.shape) # Storage deficit in root zone [L]
    SRZ_hr=np.zeros(rain_hr.shape) # Lumped hourly storage deficit in rootzone
    Sbar=np.zeros(rain.shape) # Areal average of S [L]
    Sbar_hr=np.zeros(rain_hr.shape) # Hourly areal average of S [L]
    S_hr=np.zeros(zero_matrix_2_1.shape)
    S=np.zeros(zero_matrix.shape) # Pixel scale storage deficit until
    # saturation [L]
    SUZ=np.zeros(zero_matrix.shape) # Storage in unsaturated zone [L]
    SUZ_hr=np.zeros(zero_matrix_2_1.shape)
    EX=np.zeros(zero_matrix.shape) # Redundant or excess water amount in root
    # zone [L]
    EX_hr=np.zeros(zero_matrix_2_1.shape)
    UZ=np.zeros(zero_matrix.shape) # Vertical flux from unsaturated zone to
    # saturated zone [L]
    UZ_hr=np.zeros(zero_matrix_2_1.shape)

    #　ALLOCATE ROOMS TO PARAMETERS IN THE RNDM STORE
    m=RNDM[M,0]
    Te=RNDM[M,1]
    td=RNDM[M,2] # Represents the effective permeability of the soil
    SRZinitial=RNDM[M,3]
    SRmax=RNDM[M,4]
    
    # CALCUALTE THE MEAN STORAGE DEFICIT OF CATCHMENT
    Sbar_hr[0,0]=-m*np.log((Q1_hr)/(Te*np.exp(-Lambda)))  # Mean storage deficit of
    # watershed [m/dt]

    # CALCULATE FLUXES　AT EACH STEP
    for t in range(rows_h): # Iterate through the days in the year
        for ha in range(cols_h): # Iterate through the hours in a day
            # ITERATE THROUGH THE TI CLASSES
            for ia in range(len(TI)):

                # CALCULATE THE LOCAL STORAGE DEFICIT
                S_hr[t,ia,ha]=Sbar_hr[t,ha]+(m*(Lambda-TI[ia,:]))
                if S_hr[t,ia,ha]<0:
                    S_hr[t,ia,ha]=0

            # ROOTZONE CALCULATIONS
            SRZ_hr[0,0]=SRZinitial
            SRZ_hr[t,ha]=SRZ_hr[t,ha]-rain_hr[t,ha]
            if SRZ_hr[t,ha]<0.0:
                SUZ_hr[t,:,ha]=SUZ_hr[t,:,ha]-SRZ_hr[t,ha]
                SRZ_hr[t,ha]=0

            # UNSATURATED ZONE CALCULATIONS
            for ib in range (len(TI)):
                if SUZ_hr[t,ib,ha]>S_hr[t,ib,ha]:
                    EX_hr[t,ib,ha]=SUZ_hr[t,ib,ha]-S_hr[t,ib,ha]
                    SUZ_hr[t,ib,ha]=S_hr[t,ib,ha]
                # CALCULATE DRAINAGE FROM SUZ
                if S_hr[t,ib,ha]>0.0:
                    UZ_hr[t,ib,ha]=SUZ_hr[t,ib,ha]/(td*S_hr[t,ib,ha])
                    if UZ_hr[t,ib,ha]>SUZ_hr[t,ib,ha]:
                        UZ_hr[t,ib,ha]=SUZ_hr[t,ib,ha]
                    SUZ_hr[t,ib,ha]=SUZ_hr[t,ib,ha]-UZ_hr[t,ib,ha]
                    if SUZ_hr[t,ib,ha]<0.0000001:
                        SUZ_hr[t,ib,ha]=0

            # CALCULATE ACTUAL EVAPOTRANSPIRATION
            if ET0_hr[t,ha]>0.0:
                ETa_hr[t,ha]=ET0_hr[t,ha]*(1-(SRZ_hr[t,ha]/SRmax))
                if ETa_hr[t,ha]>(SRmax-SRZ_hr[t,ha]):
                    ETa_hr[t,ha]=SRmax-SRZ_hr[t,ha]
                SRZ_hr[t,ha]=SRZ_hr[t,ha]+ETa_hr[t,ha]


            # CALCULATE HOURLY DISCHARGES AT CATCHMENT OUTLET
            Qsub_hr[t,ha]=Te*np.exp(-Lambda)*np.exp((-1)*Sbar_hr[t,ha]/m)
            Qover_hr[t,ha]=np.sum(freq*EX_hr[t:t+1,:,ha:ha+1])
            Qv_hr[t,ha]=np.sum(freq*UZ_hr[t:t+1,:,ha:ha+1])
            Qsim_hr[t,ha]=Qover_hr[t,ha]+Qsub_hr[t,ha]


            # UPDATE VARIABLES FOR THE NEXT HOUR TIME STEP
            if ha<(cols_h-1):
                Sbar_hr[t,ha+1]=Sbar_hr[t,ha]-Qv_hr[t,ha]+Qsub_hr[t,ha]
                SRZ_hr[t,ha+1]=SRZ_hr[t,ha] # SRZ value at beginning of next time
                # step is set as SRZ value at end of previous time step;
                # done because value was not updating from previous version of
                #code
                #SUZ_hr[t,:,ha+1]=SUZ_hr[t,:,ha] # No need since SUZ is 0 at
                # beginning of each time step, given small td value # please
                # confirm this statement

        # UPDATING VARIABLES FOR THE NEXT DAY TIME STEP
        if t<(rows_h-1):
            Sbar_hr[t+1,0]=Sbar_hr[t,23]-Qv_hr[t,23]+Qsub_hr[t,23]
            SRZ_hr[t+1,0]=SRZ_hr[t,23]
            #SUZ_hr[t+1,:,0]=SUZ_hr[t,:,23]

        # CALCULATING DAILY FLUXES AT THE CATCHMENT OUTLET
        Qsub[t,:]=(np.mean(Qsub_hr[t,:]))*24 #Sbar; Qsub, are things whose changes are
        #detectable at daily time scale? # Find literature to cite. : *******How is it if we take the mean?******
        #Qsub[t,:]=np.mean(Qsub_hr[t,:])
        Qover[t,:]=(np.mean(Qover_hr[t,:]))*24
        Qv[t,:]=(np.mean(Qv_hr[t,:]))*24
        Qsim[t,:]=Qover[t,:]+Qsub[t,:]

    # OBJECTIVE FUNCTION CALCULATION
    Er=np.sum(Qobs-Qsim)
    SSE=np.sum((Qobs-Qsim)**2)
    # Calculation of error mean or error variance
    Qm=np.mean(Qobs) # Mean of observed discharge
    Qs=np.sum(Qobs)
    SSU=np.sum((Qobs-Qm)**2) # Mean variance
    STD=np.sqrt(SSU/rain.size)
    Check_NS=1-(SSE/SSU) # NS /likelihooh value/model performance efficiency
    Check_RMSE=np.sqrt(SSE/(rain.size))
    Check_RSR=Check_RMSE/STD
    Check_PBIAS=Er*100/Qs #Yapo et al.(1996) & Sorooshian et al. (1993); Gupta et al. (1999). NB:Sorooshin & Yapo have different formulations of PBIAS from Gupta et al. (1999). I adopted Gupta
    RNDM[M,5]=Check_NS
    RNDM[M,6]=Check_RMSE
    RNDM[M,7]=Check_RSR
    RNDM[M,8]=Check_PBIAS

# DECISION OF OPTIMUM PARAMETERS
w=RNDM[:,5].argsort() # Sorting by column 5 in ascending order while
# maintaining array intergrity
RNDM=RNDM[w[::-1]] # Sort w by descending order; (::-1) reverses w order
#np.savetxt('RNDM1.csv',RNDM,fmt='%.4f',delimiter=',',header="m,Te,td,\
#SRZinitial,SRmax,NS,RMSE,RSR") # Save RNDM1 as .csv format in local storage


#*******************************SECTION 4*************************************
# TopEros IMPLEMENTATION USING SO-CALLED OPTIMUM TOPMODEL PARAMETERS: SOME
# COMMENTS ARE AS ABOVE

## PREALLOCATE ARRAYS
Qv=np.zeros(rain.shape)
Qv_hr=np.zeros(rain_hr.shape) #Lumped hourly vertical flux from UZ to SZ
Qover=np.zeros(rain.shape)
Qover_hr=np.zeros(rain_hr.shape) # Lumped hourly overland flow
Qsub=np.zeros(rain.shape)
Qsub_hr=np.zeros(rain_hr.shape) # Lumped hourly subsurface flow
Qsim=np.zeros(rain.shape)
Qsim_hr=np.zeros(rain_hr.shape)
ETa=np.zeros(rain.shape)
ETa_hr=np.zeros(rain_hr.shape) # Actual hourly evapotranspiration
SRZ=np.zeros(rain.shape)
SRZ_hr=np.zeros(rain_hr.shape) # Lumped hourly storage deficit in rootzone
Sbar=np.zeros(rain.shape)
Sbar_hr=np.zeros(rain_hr.shape) # Hourly areal average of S [L]
RESULTS=np.zeros((rows_h,11)) # Array  of zeros created to be store for result
S=np.zeros(zero_matrix_2.shape)
S_hr=np.zeros(zero_matrix_3.shape) # Hourly S
SUZ=np.zeros(zero_matrix_2.shape)
SUZ_hr=np.zeros(zero_matrix_3.shape)
EX_D=np.zeros(zero_matrix_2.shape)
EX_hr=np.zeros(zero_matrix_3.shape)
q=np.zeros(zero_matrix_2.shape) # Discharge per unit width [m^3/h/m]
Dr_hr = np.zeros(zero_matrix_3.shape) # Soil detachement by rain drop.
Dr_D=np.zeros(zero_matrix_2.shape) # Daily pixel scale detachment by raindrop
UZ=np.zeros(zero_matrix_2.shape)
UZ_hr=np.zeros(zero_matrix_3.shape)
n=np.zeros(K.shape) # Manning's n
Sf=np.zeros(zero_matrix_2.shape) # Friction slope
Qpeak=np.zeros(zero_matrix_2.shape)
Dof=np.zeros(zero_matrix_2.shape) # Daily pixel scale detachment by overland flow
Dof_tot=np.zeros(zero_matrix_2.shape)
Dof_tot_ds=np.zeros(zero_matrix_2.shape) #Total detachment in cell with overland flow
Dof_tot_ds=np.zeros(zero_matrix_2.shape) #Total detachment in cell with overland flow that is tranported to the downstream cell
Dch=np.zeros(zero_matrix_2.shape) # Daily pixel scale detachment capacity by concentrated flow
Dch_no_nan=np.zeros(zero_matrix_2.shape)
Df=np.zeros(zero_matrix_2.shape) # Detachment by concentrated flow
Df_tot=np.zeros(zero_matrix_2.shape) # Total detachment in cell with concentrated flow
Df_tot_ds=np.zeros(zero_matrix_2.shape) # Total detachment in cell with concentrated flow.
Dof_cc=np.zeros(zero_matrix_2.shape) # Sheet erosion on non-channel section of Channel erosion cell.
Dcc=np.zeros(zero_matrix_2.shape) # Detachment from cell with channel feature
Dcc_ds=np.zeros(zero_matrix_2.shape) #Detached soil from channel feature transported to downstream cell.
Tau_bar=np.zeros(zero_matrix_2.shape) # Average shear stress.
h=np.zeros(zero_matrix_2.shape) # Cahnnel flow depth
PSY_2=np.zeros(zero_matrix_2.shape)
Tc = np.zeros(zero_matrix_2.shape) # Transport capacity of flow
PGE=np.zeros(zero_matrix_2.shape) #Daily gross soil erosion at pixel scale
PSY=np.zeros(zero_matrix_2.shape) #Daily Pixel scale sediment yield
PSY_masked_by_catchment=np.zeros(zero_matrix_2.shape)
PSY_yr=np.zeros(K.shape) # Yearly pixel scale sediment scale
PSY_th=np.zeros(zero_matrix_2.shape) # daily pixel scale threshold values
PSY_th_masked_by_catchment=np.zeros(zero_matrix_2.shape) # Daily pixel thresholds masked by catchment bounds
PSY_yr_th=np.zeros(K.shape) # Threshold values for yearly sediment scale
PSD=np.zeros(zero_matrix_2.shape) #Pixel scale net daily soil deposition
PSD_masked_by_catchment=np.zeros(zero_matrix_2.shape)
CSY=np.zeros(rain.shape) # Daily sediment yield at catchment outlet [Mg]
CGE=np.zeros(rain.shape) #Gross catchment sediment yield
CGE_Dr=np.zeros(rain.shape) # Catchment sediment yield from rain drop detachment
CGE_Dcc=np.zeros(rain.shape) #Cathcment sediment detachment from cell with concentrated flow
CGE_Dcc_ds=np.zeros(rain.shape)
CGE_Dch=np.zeros(rain.shape) #Cathcment sediment yield from concentrated flow
CGE_Dof_tot=np.zeros(rain.shape) #Cathcment sediment detachment from cell with no channel erosion
CGE_Dof_tot_ds=np.zeros(rain.shape) #Catchment-scale sediment yield from non-gulley areas
CSD=np.zeros(rain.shape) #Sediment deposition at catchment scale
SDR=np.zeros(rain.shape) # Sediment delivery ratio at catchment outlet
u=np.zeros(zero_matrix_2.shape)
Ai=np.zeros(Fac_masked.shape) # Upstream contributing area per unit contour length
ai=np.zeros(Fac_masked.shape) # Upstream contributing area per unit contour length
EPI=np.zeros(Fac_masked.shape) # Erosion power index: Christened by me [(as * tanBea)]
Wi=np.zeros(TIM.shape) # Width of imaginary channel in a cell
Tau_cr=np.zeros(TIM.shape)
C=np.zeros(K.shape) # Cover Management Factor

X=0
RS=50**2 # Raster resolution [m^2]
L=50 #Length of imaginary channel
catchment_area=Num_of_catchment_cells*RS/1000000 #[sq. km]
filename = r'C:\Users'+'\\'+pc_name+r'\Dropbox\Gifu_dai\PhD_Prog\Project_TopEros\TopEros_2\TopEros_2\50_m\NamTIKrigMfd50mRecl.tif'
filename_2 = r'C:\Users'+'\\'+pc_name+r'\Dropbox\Gifu_dai\PhD_Prog\Project_TopEros\TopEros_2\TopEros_2\Nam_poly.shp'
output_folder = r'C:\Users'+'\\'+pc_name+r'\Dropbox\Gifu_dai\PhD_Prog\Data\Simulation_out_put\2015\Namatala'

# CALCULATING UPSTREAM CONTRIBUTING AREA FOR A GIVEN CELL
for k in range(rows):
    for l in range(cols):
        if TIM[k,l]!=nan_value_TIM: # Excludes cells outside the watershed bounds
            if Fac[k,l]<1:
                Fac[k,l]=0
            Ai[k,l] = (Fac[k,l]+1)*RS #Upstream contributin area of a given cell
            ai[k,l]=(Fac[k,l]+1)*L #Upstream contributing area per unit contour length
            EPI[k,l] =ai[k,l]*np.tan(Rad_S[k,l]) # Erosive Power Index of concentrated flow
            
Zeta = 0.5
Wriv=8.4 # River width at catchment outlet
Epsilon=Wriv/((np.max(Ai))**Zeta)
Gamma=9.81*998# [kg/m^2/s^2] #*3600*3600 # Specific density of water in kg/m^2/h
Alpha = 2
Beta = 1
u=50#200 #360#1800 #4461 # Flow velocity [m/h]: How did you find flow velocity? (From the unit hydrograph?)

# FINDING NUMBER OF CELLS WITH CONCENTRATED FLOW BY TI & EPI THRESHOLD METHOD of Moore et al. (1988)
EG_TI_threshold = 15 #Suggested by Moore et al. (1988), who adoptd 6.8
EG_EPI_threshold =35 #Suggested by Moore et al. (1988), who adopted 18
EG_cells = np.nansum((TIM_masked > EG_TI_threshold) & (EPI > EG_EPI_threshold)) # Cells with EG erosion as in Moore et al. (1988)
OF_cells=(Num_of_catchment_cells)-(EG_cells) # Cells with sheet erosion

# CALCULATING C factor & Wi
for k in range(rows):
    for l in range(cols):
        if TIM[k,l]!=nan_value_TIM:
            C[k,l]=np.exp(-Alpha*NDVI[k,l]/(Beta-NDVI[k,l])) # Azizian et al.#(2021) as in van der Knijff et al. (1999)
            Wi[k,l] = Epsilon*(Ai[k,l]**Zeta)


# ALLOCATE THE 'OPTIMUM PARAMETERS'
m=RNDM[X,0]
Te=RNDM[X,1]
td=RNDM[X,2]
SRZinitial=RNDM[X,3]
SRmax=RNDM[X,4]

## INITIALISE VALUES
Sbar_hr[0,0]=-m*np.log((Q1_hr)/(Te*np.exp(-Lambda)))

for t in range(rows_h):
    for ha in range(cols_h):
        for ia in range(rows):
            for ja in range(cols):
                # CALCULATE THE LOCAL STORAGE DEFICIT & RAINDROP DETACHMENT
                if TIM[ia,ja]!=nan_value_TIM:
                    S_hr[t,ia,ja,ha]=Sbar_hr[t,ha]+(m*(Lambda-TIM[ia,ja]))
                    if S_hr[t,ia,ja,ha]<0:
                        S_hr[t,ia,ja,ha]=0
                        
                    # SOIL DETACHMENT BY RAIN DROP
                    if ((0<=C[ia,ja]<=1) and (0<=K[ia,ja]<=1)): #"or" changed to "and"
                        Dr_hr[t,ia,ja,ha]=0.0138*K[ia,ja]*C[ia,ja]*((rain_hr[t,ha]*0.001)**2)*((2.96*(np.tan(Rad_S[ia,ja]))**0.79)+0.56) #Eqn 19  (Wang et al., 2010) [kg/h/m^2]
                        Dr_hr[t,ia,ja,ha]=Dr_hr[t,ia,ja,ha]*RS/1000 # [Mg/h]: else Dr_hr[t,ia,ja,ha]=0*****
                    
        # ROOTZONE CALCULATIONS
        SRZ_hr[0,0]=SRZinitial
        SRZ_hr[t,ha]=SRZ_hr[t,ha]-rain_hr[t,ha]
        if SRZ_hr[t,ha]<0.0:
            SUZ_hr[t,:,:,ha]=SUZ_hr[t,:,:,ha]-SRZ_hr[t,ha]
            SRZ_hr[t,ha]=0

        # UNSATURATED ZONE CALCULATIONS
        for ib in range(rows):
            for jb in range(cols):
                if TIM[ib,jb]!=nan_value_TIM:
                    if SUZ_hr[t,ib,jb,ha]>S_hr[t,ib,jb,ha]:
                        EX_hr[t,ib,jb,ha]=SUZ_hr[t,ib,jb,ha]-S_hr[t,ib,jb,ha] 
                        SUZ_hr[t,ib,jb,ha]=S_hr[t,ib,jb,ha]

                    # CALCULATE DRAINAGE FROM SUZ
                    if S_hr[t,ib,jb,ha]>0.0:
                        UZ_hr[t,ib,jb,ha]=SUZ_hr[t,ib,jb,ha]/(td*S_hr[t,ib,jb,ha]) 
                        if UZ_hr[t,ib,jb,ha]>SUZ_hr[t,ib,jb,ha]:
                            UZ_hr[t,ib,jb,ha]=SUZ_hr[t,ib,jb,ha]
                        SUZ_hr[t,ib,jb,ha]=SUZ_hr[t,ib,jb,ha]-UZ_hr[t,ib,jb,ha]
                        if SUZ_hr[t,ib,jb,ha]<0.0000001:
                            SUZ_hr[t,ib,jb,ha]=0
                            
                # EXTRACTING PEAK DISCHARGE PER HOUR
                Qpeak[t,ib,jb]=np.max(EX_hr[t,ib,jb,:])

        # CALCULATE ACTUAL EVAPOTRANSPIRATION
        if ET0_hr[t,ha]>0.0:
            ETa_hr[t,ha]=ET0_hr[t,ha]*(1-(SRZ_hr[t,ha]/SRmax))
            if ETa_hr[t,ha]>(SRmax-SRZ_hr[t,ha]):
                ETa_hr[t,ha]=SRmax-SRZ_hr[t,ha]
            SRZ_hr[t,ha]=SRZ_hr[t,ha]+ETa_hr[t,ha]

        # CALCULATE HOURLY DISCHARGES AT CATCHMENT OUTLET
        Qsub_hr[t,ha]=Te*np.exp(-Lambda)*np.exp((-1)*Sbar_hr[t,ha]/m)
        Qover_hr[t,ha]=np.sum(freq_1*EX_hr[t:t+1,:,:,ha:ha+1]) #Be conscious of the fact that only catchment cells are to be summed
        Qv_hr[t,ha]=np.sum(freq_1*UZ_hr[t:t+1,:,:,ha:ha+1]) 
        Qsim_hr[t,ha]=Qover_hr[t,ha]+Qsub_hr[t,ha]

        #UPDATING VARIABLES FOR THE NEXT HOUR TIME STEP
        if ha<(cols_h-1):
            Sbar_hr[t,ha+1]=Sbar_hr[t,ha]-Qv_hr[t,ha]+Qsub_hr[t,ha]
            SRZ_hr[t,ha+1]=SRZ_hr[t,ha]
            #for iy in range(rows):
                #for jy in range(cols):
                    #if TIM[iy,jy]!=nan_value_TIM: # For me it's strange how performance improves with the elimination of this code!
                        #SUZ_hr[t,iy,jy,ha+1]=SUZ_hr[t,iy,jy,ha]
                        

    # UPDATING VARIABLES FOR THE NEXT DAY TIME STEP
    if t<(rows_h-1):
        Sbar_hr[t+1,0]=Sbar_hr[t,23]-Qv_hr[t,23]+Qsub_hr[t,23] 
        SRZ_hr[t+1,0]=SRZ_hr[t,23]
        #for iz in range(rows):
            #for jz in range(cols):
                #if TIM[iz,jz]!=nan_value_TIM:
                    #SUZ_hr[t+1,iz,jz,0]=SUZ_hr[t,iz,jz,23]

    # CALCULATING DAILY FLUXES AT THE CATCHMENT OUTLET
    Qsub[t,:]=(np.mean(Qsub_hr[t,:]))*24
    Qover[t,:]=(np.mean(Qover_hr[t,:]))*24
    Qv[t,:]=(np.mean(Qv_hr[t,:]))*24
    Qsim[t,:]=Qover[t,:]+Qsub[t,:]
    
# CACULATING DAILY PIXEL Dr
Dr_D = np.sum(Dr_hr, axis=-1) # [Mg/day]
    
#**************** CALCULATING SOIL DETACHMENT BY RUNOFF************************    
       
# CALCULATIONS RELATED TO CONCENTRATED FLOW
for ie in range(rows):
    for je in range(cols):
        if TIM[ie,je]!=nan_value_TIM:
            n[ie,je]=0.06 # Chow (1959); Average value
            
            #******************* Extracting gulley pixels**********************
            # WE have already extracted gulleys by TI: Why this again?
            #if not LSF_masked.mask[ie,je]:
            #if LSF[ie,je]>1000:
                #LSF[ie,je]=0
            if LSF[ie,je]<0:
                LSF[ie,je]=0
            if K[ie,je]<0:
                K[ie,je]=0
                
            #*********************Critical Shear Stress************************
            Tau_cr[ie,je]=3.23-(5.6*Sa[ie,je])-(24.4*Or[ie,je])+(0.00088*Rod[ie,je]*1000) # Flanagan & Livingston (1995) [Pa]:The values look normal. Soil contents are ratios
            
            #******************************************************************
#******************************************************************************

EX_D = np.sum(EX_hr, axis=-1)
for day in range(rows_h):
     for id in range(rows):
         for jd in range(cols):
             if TIM[id,jd]!=nan_value_TIM:
                 h[day,id,jd]=(EX_D[day,id,jd]*RS/24000)/(Wi[id,jd]*u) #Vol/(wi*250) [m at hour timestep]
                 #u[day,id,jd]=(9.81*h[day,id,jd]*np.tan(Rad_S[id,jd])*3600*3600)**0.5 # Fondriest Environmental, Inc. (2014); [m/hr]
                 #n[day,id,jd]=0.05 #0.0903+(0.000791*Veg_pc[id,jd])-(0.04*EX_D[day,id,jd]*RS/86400) #Etedali et al. (2011), Eq. 5; What is the meaning of n=0?  Not feasible
                 
                 #Transport capacity: Beasley et al. (1980) & Wang etal. (2010)
                 q[day,id,jd]=RS*EX_D[day,id,jd]/(24000*60*L)    
                 if q[day, id, jd] <= 0.046:
                     Tc[day,id,jd]=(146*(np.tan(Rad_S[id,jd]))*(q[day, id, jd])**0.5) # [kg/m/min]
                     Tc[day,id,jd]=Tc[day,id,jd]*60*24*L/1000 #[Mg/day]
                 elif q[day, id, jd] > 0.046:
                     Tc[day,id,jd]=(14600*(np.tan(Rad_S[id,jd]))*(q[day, id, jd])**2) # # [kg/m^2/min]*(24/1000)*Wi[id,jd] # [Mg/day]
                     Tc[day,id,jd]=Tc[day,id,jd]*24*60*L/1000 # [Mg/day]
                     
                                  
                 #*****************Concentrated flow Erosion***********************
                 if (TIM[id,jd]>EG_TI_threshold) and (EPI[id,jd]>EG_EPI_threshold):
                     
                     # Sheet erosion phase in cell with assumed channel
                     if (0<=C[id,jd]<=1) and (0<=K[id,jd]<=1) and (0<=P[id,jd]<=1):
                         Dof_cc[day,id,jd]=11.8*(((EX_D[day,id,jd]*(L*(L-Wi[id,jd]))*0.001)*Qpeak[day,id,jd]/3600)**0.56)*K[id,jd]*C[id,jd]*LSF[id,jd]*P[id,jd]
                     else:
                         Dof_cc[day,id,jd]=0
                     if h[day,id,jd]==0:
                         Sf[day,id,jd]=0
                     else:
                         Sf[day,id,jd]=(n[id,jd]**2)*(u**2)/(((Wi[id,jd]*h[day,id,jd])/((2*h[day,id,jd])+Wi[id,jd]))**(4/3)) # Molls et al. (1998): Eqn. 6
                         #Sf[day,id,jd]=(n[id,jd]**2)*(u**2)/(h[day,id,jd]**(4/3)) # Wang et al. (2010), Eqn. 27#h[day,id,jd]=(EX_D[day,id,jd]*RS/24000)/(Wi[id,jd]*u)
                     Tau_bar[day,id,jd]=Gamma*h[day,id,jd]*Sf[day,id,jd]
                     if (Tau_bar[day,id,jd]>Tau_cr[id,jd]) and (0<=K[id,jd]<=1): 
                         Dch[day,id,jd] = Wi[id,jd]*K[id,jd]*((1.35*Tau_bar[day,id,jd])-Tau_cr[id,jd])**1.05 # Wang et al. (2010) Eqn. 23. [kg/h/m^2]
                     else:
                         Dch[day,id,jd] = 0
                     Dch[day,id,jd] = Dch[day,id,jd]*24*h[day,id,jd]*Wi[id,jd]/1000 # [Mg/day/channel in a pixel]
                     if (Dr_D[day,id,jd]+Dof_cc[day,id,jd])<=Tc[day,id,jd]: # Detachment ability of a flow, considering it is already ladden with sediment (must work best with flow received from upstream cells).
                         Df[day,id,jd]=Dch[day,id,jd]*(1-((Dr_D[day,id,jd]+Dof_cc[day,id,jd])/Tc[day,id,jd]))
                     else:
                         Df[day,id,jd]=0
                     Dcc[day,id,jd]=Df[day,id,jd]+Dr_D[day,id,jd]+Dof_cc[day,id,jd]
                     
                     # Comparing detached soil with transport capacity of the flow
                     if Dcc[day,id,jd]>Tc[day,id,jd]:
                         PSD[day,id,jd]=Dcc[day,id,jd]-Tc[day,id,jd]
                         Dcc_ds[day,id,jd]=Tc[day,id,jd]
                     else:
                         Dcc_ds[day,id,jd] = Dcc[day,id,jd] 
                         PSD[day,id,jd] =0
                     
                     #*********************************************************
                 
                 #****************Overland Flow Erosion by MUSLE***************
                 elif (0<=TIM[id,jd]<=EG_TI_threshold) or (0<=EPI[id,jd]<=EG_EPI_threshold):
                     if ((0<=C[id,jd]<=1) and (0<=K[id,jd]<=1) and (0<=P[id,jd]<=1)):
                         Dof[day,id,jd]=11.8*(((EX_D[day,id,jd]*RS*0.001)*Qpeak[day,id,jd]/3600)**0.56)*K[id,jd]*C[id,jd]*LSF[id,jd]*P[id,jd]
                         # Jackson et al. (1986); [Mg]...Williams used the CIA method to estimate peak doscharge
                     else:
                         Dof[day,id,jd]=0
                     Dof_tot[day,id,jd]=Dof[day,id,jd]+Dr_D[day,id,jd] #Total detachment in cell with overland flow
                     
                     # Comparing detached soil with transport capacity of the flow
                     if Dof_tot[day,id,jd]>Tc[day,id,jd]:
                         PSD[day,id,jd]=Dof_tot[day,id,jd]-Tc[day,id,jd]
                         Dof_tot_ds[day,id,jd]=Tc[day,id,jd]
                     else:
                         Dof_tot_ds[day,id,jd]=Dof_tot[day,id,jd]
                         PSD[day,id,jd] = 0
                         
             else:
                 Dcc[day,id,jd]=np.nan
                 Dof_tot[day,id,jd]=np.nan
                 PSD[day,id,jd]=np.nan
                 PSY[day,id,jd]=np.nan
                 PGE[day,id,jd]=np.nan
                 
                 #*************************************************************
               
#******************************************************************************
# Load the raster profile (metadata) from the original raster file
raster_file = r'C:\Users'+'\\'+pc_name+r'\Dropbox\Gifu_dai\PhD_Prog\Project_TopEros\TopEros_2\TopEros_2\50_m\NamTIKrigMfd50mRecl.tif'#r'C:\Users'+'\\'+pc_name+r'\Dropbox\Gifu_dai\PhD_Prog\RD\musle\NDVI\2015_Nam_NDVI_250.tif'
with rasterio.open(raster_file) as src:
    profile = src.profile
    
#Create a mask for catchment area
catchment_mask = geometry_mask(catchment_shapefile.geometry, out_shape=PSY_yr.shape, transform=profile['transform'], invert=True)

# FURTHER PROCESSING SOIL EROSION DATA FOR PLOTTING AND GRAPHING
for day in range(rows_h):
    for id in range(rows):
        for jd in range(cols):
            if TIM[id,jd]!=nan_value_TIM:
                
                #************* Gross pixel-scale soil erosion*********************
                PSY[day,id,jd]=Dcc_ds[day,id,jd]+Dof_tot_ds[day,id,jd] #One of concentrated or sheet erosion happens in a cell
                PGE[day,id,jd]=Dcc[day,id,jd]+Dof_tot[day,id,jd] # Amount of detachment soil prior to deposition
                    
                # CATEGORISING DAILY SEDIMENT YIELD AT PIXEL SCALE [Mg/cell/year] (Jiang et al., 2014)
                if 0<=PSY[day,id,jd]<(2*RS/3650000):
                    PSY_th[day,id,jd]=0
                elif (2*RS/3650000)<=PSY[day,id,jd]<(10*RS/3650000):
                    PSY_th[day,id,jd]=1
                elif (10*RS/3650000)<=PSY[day,id,jd]<(50*RS/3650000):
                    PSY_th[day,id,jd]=2
                elif (50*RS/3650000)<=PSY[day,id,jd]<(100*RS/3650000):
                    PSY_th[day,id,jd]=3
                elif PSY[day,id,jd]>=(100*RS/3650000):
                    PSY_th[day,id,jd]=4
                

    # Apply the mask to the erosion values: Potential memory intensive zone (check how to make it efficient)
    PSY_th_masked_by_catchment[day,:,:] = np.where(catchment_mask, PSY_th[day,:,:], np.nan)
    PSY_masked_by_catchment[day,:,:] = np.where(catchment_mask, PSY[day,:,:], np.nan)
    PSD_masked_by_catchment[day,:,:] = np.where(catchment_mask, PSD[day,:,:], np.nan)

    #**************************************************************************


    CSY[day,:]=np.nansum(PSY[day,:,:]) # Catchment-scale yield [Mg/day]
    CSD[day,:]=np.nansum(PSD[day,:,:])# Catchment sediment deposition [Mg/day]
    CGE[day,:]=np.nansum(PGE[day,:,:]) #Mg/day
    CGE_Dof_tot[day,:]=np.nansum(Dof_tot[day,:,:]) #Mg/day
    CGE_Dof_tot_ds=np.nansum(Dof_tot_ds[day,:,:])
    CGE_Dcc[day,:]=np.nansum(Dcc[day,:,:]) #Mg/day 
    CGE_Dcc_ds[day,:]=np.nansum(Dcc_ds[day,:,:])
    CGE_Dr[day,:]=np.sum(Dr_D[day,:,:])

    # Graphing daily sediment fluxes
    plot_data_array_over_shapefile(filename_2, PSY_masked_by_catchment[day,:,:], num_classes=500, cmap='viridis', alpha=1,cbar_label='Soil Erosion (Mg)', plot_title='Grid-Scale Daily Soil Erosion')
    plot_data_array_over_shapefile_2(filename_2, PSY_th_masked_by_catchment[day,:,:], num_classes=5, cmap='viridis', alpha=1,cbar_label='Sediment Yield Class', plot_title='Grid-Scale Daily Sediment Yield Class')
    plot_data_array_over_shapefile(filename_2, PSD_masked_by_catchment[day,:,:], num_classes=500, cmap='viridis', alpha=1,cbar_label='Soil Deposition (Mg)', plot_title='Grid-Scale Daily Sediment Deposition')

# CALCULATE YEARLY PIXEL-SCALE SEDIMENT YIELD & OR DEPOSITION
PSY_yr = np.nansum(PSY, axis=0)
PSD_yr = np.nansum(PSD, axis=0)

# Apply the mask to the yearly erosion values
PSY_yr_masked_by_catchment = np.where(catchment_mask, PSY_yr, np.nan)
PSD_yr_masked_by_catchment=np.where(catchment_mask, PSD_yr, np.nan)

# Graphing yearly sediment fluxes
plot_data_array_over_shapefile(filename_2, PSY_yr_masked_by_catchment[:,:], num_classes=500, cmap='viridis', alpha=1,cbar_label='Soil Erosion (Mg)', plot_title='Grid-Scale Yearly Sediment Yield')
plot_data_array_over_shapefile(filename_2, PSD_yr_masked_by_catchment[:,:], num_classes=500, cmap='viridis', alpha=1,cbar_label='Sediment Deposition (Mg)', plot_title='Grid-Scale Yearly Sediment Deposition')

#******************************************************************************

# CATEGORISING YEARLY SEDIMENT YIELD AT PIXEL SCALE
for ig in range(rows):
    for jg in range(cols):
        if TIM[ig,jg]!=nan_value_TIM:
            if 0<=PSY_yr[ig,jg]<(2*RS/10000):
                PSY_yr_th[ig,jg]=0
            elif (2*RS/10000)<=PSY_yr[ig,jg]<(10*RS/10000):
                PSY_yr_th[ig,jg]=1
            elif (10*RS/10000)<=PSY_yr[ig,jg]<(50*RS/10000):
                PSY_yr_th[ig,jg]=2
            elif (50*RS/10000)<=PSY_yr[ig,jg]<(100*RS/10000):
                PSY_yr_th[ig,jg]=3
            elif PSY_yr[ig,jg]>=(100*RS/10000):
                PSY_yr_th[ig,jg]=4
            if np.isnan(K[ig, jg]):
                PSY_yr_th[ig, jg] = np.nan


# Apply the mask to the erosion values
PSY_yr_th_masked_by_catchment = np.where(catchment_mask, PSY_yr_th, np.nan)
plot_data_array_over_shapefile_2(filename_2, PSY_yr_th_masked_by_catchment[:,:], num_classes=5, cmap='viridis', alpha=1,cbar_label='Sediment yield Class', plot_title='Grid-Scale Yearly Sediment Yield Class')

#*******************************SECTION 5*************************************
# Plot the fluxes
# Define the start date
start_date = datetime(2015, 2, 24)  # Adjust this to your actual start date

# Create a date range based on the number of days in the data
date_range = [start_date + timedelta(days=i) for i in range(len(CSY))]

#*************************** Plotting Erosion fluxes***************************
# SEDIMENT YIELD ATTHE CATCHEMNT OUTLET
plt.figure(figsize=(12, 6))
plt.bar(date_range, CSY[:, 0], color='blue')  # Use CSY[:, 0] to access the single column
#plt.xlabel('Date')
plt.ylabel('Sediment Yield (Mg/day)')
plt.title('Daily Sediment Yield at Catchment Outlet')
plt.grid(axis='y')

# Format the x-axis as dates
plt.gca().xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m-%d'))

# Use a DayLocator with a frequency of every 30 days
plt.gca().xaxis.set_major_locator(mdates.DayLocator(interval=30))

# Rotate x-axis labels for better readability
plt.gcf().autofmt_xdate()

# Show the plot
plt.show()

# EROSION RATES
bar_width = 0.4
plt.figure(figsize=(12, 6))
plt.bar(date_range, CGE_Dcc[:, 0], color='blue', label='CGE_Dcc',width=bar_width)
plt.bar(date_range, CGE_Dof_tot[:, 0], color='red', bottom=CGE_Dcc[:, 0], label='CGE_Dof',width=bar_width)
plt.bar(date_range, CGE_Dr[:, 0], color='black', bottom=CGE_Dcc[:, 0] + CGE_Dof_tot[:, 0], label='CGE_Dr',width=bar_width)
bar_width_2 = 0.4  # Adjust as needed
plt.bar(date_range, -CSD[:, 0], color='orange', label='CSD', width=bar_width_2, align='edge') #CSD is taken as negative & net erosion as positive
plt.ylabel('Soil Erosion (Mg/day)')
plt.title('Sediment Deposition and Erosion Contribution by Erosion Type')
plt.grid(axis='y')
plt.gca().xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m-%d'))
plt.gca().xaxis.set_major_locator(mdates.DayLocator(interval=30))
plt.gcf().autofmt_xdate()
plt.legend()
plt.show()

# EROSION RATES & SOIL DEPOSTION
bar_width = 0.4
plt.figure(figsize=(12, 6))
plt.bar(date_range, CGE[:, 0], color='blue', label='CGE',width=bar_width)
bar_width_2 = 0.4  # Adjust as needed
plt.bar(date_range, -CSD[:, 0], color='orange', label='CSD', width=bar_width_2, align='edge') #CSD is taken as negative & net erosion as positive
plt.ylabel('Soil Erosion (Mg/day)')
plt.title('Soil Erosion and Sediment Deposition')
plt.grid(axis='y')
plt.gca().xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m-%d'))
plt.gca().xaxis.set_major_locator(mdates.DayLocator(interval=30))
plt.gcf().autofmt_xdate()
plt.legend()
plt.show()

# DEPOSITION RATES
bar_width = 0.4
plt.figure(figsize=(12, 6))
plt.bar(date_range, CSD[:, 0], color='blue', label='CSD',width=bar_width)
plt.ylabel('Soil Erosion (Mg/day)')
plt.title('Daily Sediment Deposition')
plt.grid(axis='y')
plt.gca().xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m-%d'))
plt.gca().xaxis.set_major_locator(mdates.DayLocator(interval=30))
plt.gcf().autofmt_xdate()
plt.legend()
plt.show()


#************************ Plotting Discahrge fluxes****************************
# Plotting
fig, ax1 = plt.subplots(figsize=(12, 6))

# Plotting observed and simulated discharge as line graphs
ax1.plot(date_range, Qobs, label='Observed Discharge', color='red', marker='') #marker='o'
ax1.plot(date_range, Qsim, label='Simulated Discharge', color='black', marker='',linestyle='dotted')

# Set y-axis label for discharge
ax1.set_ylabel('Discharge (mm/day)', color='black')
ax1.tick_params('y', colors='black')

# Create a second y-axis for rainfall as a bar graph with reversed values
ax2 = ax1.twinx()
#rain_neg=-rain
ax2.bar(date_range, -rain[:,0], label='Rainfall', color='blue', alpha=0.2, width=1.0, zorder=10)

# Set y-axis label for reversed rainfall
ax2.set_ylabel('Rainfall (mm/day)', color='black')
ax2.tick_params('y', colors='black')

# Format the x-axis as dates with a spacing of 30 days
ax1.xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m-%d'))
ax1.xaxis.set_major_locator(mdates.DayLocator(interval=30))

# Set y-axis limits for the second y-axis to make rainfall bars run from top to bottom
ax2.set_ylim(-max(rain), 0)#ax2.set_ylim(0, -max(rain[:, 0]))

# Manually set positive tick labels for the y-axis
yticks = ax2.get_yticks()
ax2.set_yticks(yticks)
ax2.set_yticklabels([str(int(abs(x))) for x in ax2.get_yticks()])

# Rotate x-axis labels for better readability
plt.gcf().autofmt_xdate()

# Adjust layout to create space for the legend
plt.subplots_adjust(right=0.85) #How stretchy is the plot along x-axis

# Set a common title
plt.title('Rainfall, Observed and Simulated Discharges')

# Show legend
ax1.legend(loc='upper left')
ax2.legend(loc='upper right')

# Show the plot
plt.show()

#*******************************SECTION 6*************************************
# OUTPUT OBSEREVED & SIMULATED VALUES
RESULTS[:,:1]=rain
RESULTS[:,1:2]=ET0
RESULTS[:,2:3]=ETa
RESULTS[:,3:4]=CSY
RESULTS[:,4:5]=Qobs
RESULTS[:,5:6]=Qsim
RESULTS[:,6:7]=Qover
RESULTS[:,7:8]=Qsub
RESULTS[:,8:9]=Qv
RESULTS[:,9:10]=Sbar
RESULTS[:,10:11]=SRZ

# CALCUALTE THE OBJECTIVE FUNCTIONS
Er=np.sum(Qobs-Qsim)
SSE=np.sum((Qobs-Qsim)**2) # Calculation of error mean or error variance
Qm=np.mean(Qobs) # Mean of observed discharge
Qs=np.sum(Qobs)
SSU=np.sum((Qobs-Qm)**2) # Mean variance
STD=np.sqrt(SSU/rain.size)
Check_NS=1-(SSE/SSU) # NS value/likelihooh value/model performance efficiency
Check_RMSE=np.sqrt(SSE/(rain.size))
Check_RSR=Check_RMSE/STD
Check_PBIAS=Er*100/Qs
print('NS = ',Check_NS, 'RSR = ',Check_RSR, '& PBIAS =', Check_PBIAS)

# EXPORTING OUTPUT DATA TO EXCEL
#np.savetxt('FinalResult.csv',RESULTS,fmt='%.4f',delimiter=',',header="rain,ET0,\
#CSY,Qobs,Qsim,Qover,Qsub,Qv,Sbar,SRZ")
print ("TOPMODEL code ran for", (time.time() - start_time)/60, "minutes")

# CODE END ALARM
audio_file_path = r'C:\Users'+'\\'+pc_name+r'\Dropbox\Gifu_dai\PhD_Prog\Audio_files\The Witchs Aria.mp3'
play_audio(audio_file_path)