#------------------------------------------------------------------------------
#********* TopEros: An integrated TOPMODEL - Erosion modelling platform********
#------------------------------------------------------------------------------

## This is a python implementation of TopEros.
## Pre-calibrated parameters are used to run the LSM, TOPMODEL.
## It calculates non-gulley soil erosion at pixel scale within the
## sub-daily routine using MUSLE; Gulley Erosion by the equation(s) of detachm-
## ent due to concentrated flow as well as detachment by rain drop.
## Units of length and time are consistent with units of input observation data
## TOPMODEL component is adopted from Fortran Code (Keith Beven); MATLAB code 
## (Hiromu Okazawa, 2015) and Python Code (Okiria Emmanuel, 2018)
## Written by Okiria Emmanuel (2024) during PhD esearch at Gifu University

# IMPORTING NECESSARY LIBRARIES
import platform
import os
import time # time library to help keep count of code run time
start_time = time.time()
import numpy as np
import pandas as pd
import rasterio
from rasterio.features import geometry_mask
import geopandas as gpd
import matplotlib.pyplot as plt
import matplotlib.colors as mcolors
import matplotlib.dates as mdates
from datetime import datetime, timedelta


#****************************SECTION 1: FUNCTIONS******************************

#### Dynamic assignment of PC_NAME based on OS
if platform.system().lower() == 'windows':
    PC_NAME='okiri'
    #PC_NAME='emmanuel'
elif platform.system().lower() ==' darwin':
    PC_NAME = 'Emma'
else:
    PC_NAME = 'default'

#### Function to generate file paths
def generate_file_path(pc_name, *path_parts):
    if platform.system() == 'Windows':
        return os.path.join(f'C:\\Users\\{pc_name}\\Dropbox\\NIES\\NIES\\WRM\\TopEros', *path_parts)
    else:
        return os.path.join(f'/Users/{pc_name}/Dropbox/NIES/NIES/WRM/TopEros', *path_parts)
    
#### Function to read _xlsx file ( load dataframes) with parse_dates to handle date parsing
def read_xlsx_w_date(file_path, sheet_name, parse_dates):
    return pd.read_excel(file_path, sheet_name=sheet_name, parse_dates=parse_dates)


#### Function to read out excel _csv files
def read_csv(file_path):
    return pd.read_csv(file_path, low_memory=False).values

#### Function to read out raster values into a Numpy array
def read_raster(file_path):
    with rasterio.open(file_path) as ds:
        return ds.read()[0]

#### Plot the raster layer over  shapefile: Continuous values
#figsize=(10,6) # For single column layout (approx. 5:3 ratio)
figsize=(8.5, 5.1) # For a 2 column layout
fontsize=10
labelsize=10

def plt_tif_over_shp(shapefile_path, data_array, num_classes, cmap='', alpha=1, cbar_label='', plot_title='', dir_path=''):
    
    """
    Plots a raster layer (continuous values) over a shapefile.

    Parameters:
    shapefile_path (str): Path to the shapefile.
    data_array (np.array): 2D array representing raster values.
    cmap (str): Colormap for raster visualization.
    alpha (float): Transparency level for raster layer (0 to 1).
    cbar_label (str): Label for the colorbar.
    plot_title (str): Title for the plot.
    dir_path (str): Folder where output will bve saved
    """

    # Load the shapefile
    gdf = gpd.read_file(shapefile_path)

    # Get the bounding box of the shapefile
    bounds = gdf.total_bounds  # [minx, miny, maxx, maxy]

    # Create a mask for NaN values in the data_array
    data_array = np.ma.masked_invalid(data_array)  # More concise than using np.isnan()

    # Determine min and max values for consistent color scaling
    min_val, max_val = np.nanmin(data_array), np.nanmax(data_array)

    # Initialise the plot
    fig, ax = plt.subplots(figsize=figsize, dpi=300)
    ax.set_facecolor('white')  # Background colour

    # Plot the shapefile with visible edges
    gdf.plot(ax=ax, alpha=alpha, edgecolor='black', facecolor='none', linewidth=0.8)

    # Overlay the raster data: <# [minx, maxx, miny, maxy]> working with <bounds>
    im = ax.imshow(data_array, extent=[bounds[0], bounds[2], bounds[1], bounds[3]], 
                   cmap=cmap, alpha=alpha, interpolation='none', aspect='auto',
                   vmin=min_val, vmax=max_val)

    # Add colorbar
    cbar = plt.colorbar(im, ax=ax, shrink=0.7, label=cbar_label)
    cbar.set_label(cbar_label, fontsize=fontsize)
    cbar.ax.tick_params(labelsize=labelsize)

    # Set title
    ax.set_title(plot_title, fontsize=fontsize)
    
    # Improve overall appearance
    plt.tight_layout()
    
    # Save the plot
    if dir_path:
        os.makedirs(dir_path, exist_ok=True)  
        save_path = os.path.join(dir_path)
        plt.savefig(save_path, bbox_inches='tight', dpi=300)

    # Show the plot
    plt.show()


#### Plot the raster layer over  shapefile: Discrete values
def plt_tif_over_shp_2(shapefile_path, data_array, num_classes, cmap='RdYlGn_r', alpha=0.5, 
                                     cbar_label='', plot_title='', dir_path=''):
    
    """
    Plots discrete categorical data over a shapefile.

    Parameters:
    - shapefile_path: str, path to the shapefile.
    - data_array: 2D numpy array, categorical values to overlay.
    - num_classes: int, number of unique classes for colormap.
    - cmap: str, colormap to use.
    - alpha: float, transparency of the overlay (0 to 1).
    - cbar_label: str, label for the colorbar.
    - plot_title: str, title for the plot.
    """

    # Load the shapefile
    gdf = gpd.read_file(shapefile_path)
    bounds = gdf.total_bounds  # [minx, miny, maxx, maxy]

    # Mask NaN values
    data_array = np.ma.masked_invalid(data_array)

    # Get unique values, ensuring they fit within num_classes
    unique_values = np.unique(data_array.compressed())  # Remove NaNs
    if len(unique_values) > num_classes:
        unique_values = np.linspace(unique_values.min(), unique_values.max(), num_classes, dtype=int)

    # Create a colormap for discrete values
    base_cmap = plt.get_cmap(cmap, num_classes)  
    cmap_category = mcolors.ListedColormap(base_cmap(np.linspace(0, 1, num_classes)))

    # Initialize the figure
    fig, ax = plt.subplots(figsize=figsize, dpi=300)
    ax.set_facecolor('white')  

    # Plot the shapefile with visible edges
    gdf.plot(ax=ax, edgecolor='black', facecolor='none', linewidth=0.8, alpha=1)

    # Overlay categorical raster data
    im = ax.imshow(data_array, extent=[bounds[0], bounds[2], bounds[1], bounds[3]], 
                   cmap=cmap_category, alpha=alpha, interpolation='none', aspect='auto')

    # Add colorbar with discrete values
    cbar = plt.colorbar(im, ax=ax, ticks=unique_values, label=cbar_label)
    cbar.ax.set_yticklabels(unique_values)  # Ensure correct labels
    cbar.set_label(cbar_label, fontsize=fontsize)
    cbar.ax.tick_params(labelsize=labelsize)

    # Set title
    ax.set_title(plot_title, fontsize=fontsize)
    
    # Improve overall appearance
    plt.tight_layout()
    
    # Save the plot
    # Save the plot
    if dir_path:
        os.makedirs(dir_path, exist_ok=True)  
        save_path = os.path.join(dir_path)
        plt.savefig(save_path, bbox_inches='tight', dpi=300)
    

    # Show plot
    plt.show()

#**************************Load & cleanup the data*****************************
# Constant
cat_ = 'Namatala'

# Read out files & Data into a Numpy array
# Add pd. suffix to function to create a dataframe
rain = read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', cat_, 'Rain_D__.csv'))) # Daily rainfall
rain_hr = read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', cat_, 'Rain_H__.csv'))) # Hourly rainfall
Qobs = read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', cat_, 'Qobs_D__.csv'))) # Daily observed discharge
Qobs_hr = read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', cat_, 'Qobs_H__.csv'))) # Hourly observed discharge

#--- If Q is in cumecs, activate the code below ---
Qobs_hr=Qobs_hr*3600*24*1000/(153.63*1000000) # 153.63 is the catchment area
#--- 

Qobs_hr = Qobs_hr/24 # Coverting units from [mm/day] to [mm/hr]
ET0 = read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', cat_, 'ET0_D___.csv'))) # Daily Evapotranspiration
ET0_hr = read_csv((generate_file_path(PC_NAME, 'Dat', 'Met_dat_', '2015____', cat_, 'ET0_H___.csv'))) # Hourly Evapotranspiration
TI = read_csv((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'TI______.csv'))) # Topographic Inderx
freq = read_csv((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'freq____.csv'))) # Frequency distribution of Topographic Index
CalPars = read_csv((generate_file_path(PC_NAME, 'Out', cat_, 'CalPar__.csv'))) # TOPMODEL calibrated parameters
SYobs = read_csv((generate_file_path(PC_NAME, 'Dat', 'Others', cat_, '2015____', 'Eros____.csv'))) # Observed plot erosion

K_raw = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'K_50m___.tif'))) # Soil erodibility factor & its spatial resoltion.
LSF = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'LSp_50m_.tif'))) # A physically derived slope-length factor.
P = np.ones(LSF.shape) # Since no human SWC interventions, P is kept at 1 (Li et al. (2023) guides on values for arable land where human intervention is observed
NDVI_raw = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, '2015____', 'NDVI50m_.tif')))
TIM = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'TI______.tif'))) # Gridded Topographic Index
Deg_S = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'SlopeDeg.tif'))) # Degree Slope
Rad_S=np.radians(Deg_S) # Convert slope to Radians
Fac = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'FacMfd__.tif'))) # Flow accumulation
Sa_raw = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'SndPc50m.tif'))) # Percent sand
Or_raw = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'OrgPc50m.tif'))) # Percent Organic matter
Rod_raw = read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'Ro_50m__.tif'))) # Bulk Density [ ]
Veg_pc_raw =  read_raster((generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'VegPc50m.tif'))) # Vegetation Cover Percentage

CatPolyPath = generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'CatPoly_.shp') # Path to shape file
catchment_shapefile = gpd.read_file(CatPolyPath)


# CLEAN UP RASTER VALUES: THRESHOLDING OPERATIONS
#********************** Harmonising Raster shapes******************************
target_shape=TIM.shape # Target array shape of rasters

# Slicing the larger arrays to match target_shape
K=K_raw[:target_shape[0], :target_shape[1]]
NDVI=NDVI_raw[:target_shape[0], :target_shape[1]]
Sa=Sa_raw[:target_shape[0], :target_shape[1]]
Or=Or_raw[:target_shape[0], :target_shape[1]]
Rod=Rod_raw[:target_shape[0], :target_shape[1]]
Veg_pc=Veg_pc_raw[:target_shape[0], :target_shape[1]]
#******************************************************************************

#***********Cleaning up K varray to eliminate NaN(s): Mask Nan values**********
nan_value_K=-3.40282e+38
K_masked=np.ma.masked_values(K, nan_value_K)

# Cleaning up TIM array
nan_value_TIM=-128
TIM_masked=np.ma.masked_values(TIM, nan_value_TIM)

# Cleaning up LSF array
nan_value_LSF=-3.4028235e+38
LSF_masked=np.ma.masked_values(LSF, nan_value_LSF)

# Cleaning up Rad_S array
nan_value_Rad_S=-5.93905e+36
Rad_S_masked=np.ma.masked_values(Rad_S, nan_value_Rad_S)

# Cleaning up Fac array
nan_value_Fac=-3.4028235e+38
Fac_masked=np.ma.masked_values(Fac, nan_value_Fac)
#******************************************************************************


# ASSIGNING THE VALUE OF INITIAL SIMULATED DISCHARGE
Q1=Qobs[0,0] # Initial base flow is set to equal observed dischare at initial
# time step

# ***************NUMBER OF ROWS & COLUMNS IN SELECT ARRAYS*********************

rows_h,cols_h = rain_hr.shape

# Finding number of cells in Area of Interest
nan_mask = np.isclose(TIM, nan_value_TIM, rtol=1e-05, atol=1e-08)
nan_count=np.sum(nan_mask)
rows,cols = TIM.shape
Num_of_catchment_cells = (rows*cols)-nan_count

#What fraction of a catchment is each cell within the catchnent bounds?
freq_1=1/(Num_of_catchment_cells)

#******************************************************************************

# CREATION OF N-D ARRAY STRUCTURES FOR THE CALCULATION OF DISTRIBUTED VARIABLES
zero_matrix=(np.zeros(TI.shape))@(np.transpose(np.zeros(rain.shape))) # Matrix
# multiplication
print('2-D shape is', zero_matrix.shape)
zero_matrix_2=np.zeros((rows_h,rows,cols)) #3-D array: [depth,r,c]
print('3-D shape is', zero_matrix_2.shape)
zero_matrix_2_1=np.zeros((rows_h,len(TI),cols_h))
print('3-D_2 shape is', zero_matrix_2_1.shape)
zero_matrix_3=np.zeros((rows_h,rows,cols,cols_h)) #4-D array: [batch,r,c,depth]
print('4-D shape is', zero_matrix_3.shape)

# ASSIGNING THE VALUE OF INITIAL SIMULATED DISCHARGE
Q1_hr=Qobs_hr[0,0] # Initial obbserved doscharge is set to equal observed 
# discharge at initial time step

#SETTING UP INDICES TO ALLOW FOR WARMUP PERIOD
start_index =0
end_index = len(rain)

# CALCULATING THE AREAL AVERAGE OF TI
Lambda=np.sum(TI*freq)/np.sum(freq) # Lambda is catchment scale areal average
# of TI


#*******************************SECTION 4*************************************
# TopEros IMPLEMENTATION USING SO-CALLED OPTIMUM TOPMODEL PARAMETERS: SOME
# COMMENTS ARE AS ABOVE

## PREALLOCATE ARRAYS
Qv=np.zeros(rain.shape)
Qv_hr=np.zeros(rain_hr.shape) #Lumped hourly vertical flux from UZ to SZ
Qover=np.zeros(rain.shape)
Qover_hr=np.zeros(rain_hr.shape) # Lumped hourly overland flow
Qsub=np.zeros(rain.shape)
Qsub_hr=np.zeros(rain_hr.shape) # Lumped hourly subsurface flow
Qsim=np.zeros(rain.shape)
Qsim_hr=np.zeros(rain_hr.shape)
ETa=np.zeros(rain.shape)
ETa_hr=np.zeros(rain_hr.shape) # Actual hourly evapotranspiration
SRZ=np.zeros(rain.shape)
SRZ_hr=np.zeros(rain_hr.shape) # Lumped hourly storage deficit in rootzone
Sbar=np.zeros(rain.shape)
Sbar_hr=np.zeros(rain_hr.shape) # Hourly areal average of S [L]
RESULTS=np.zeros((rows_h,11)) # Array  of zeros created to be store for result
S=np.zeros(zero_matrix_2.shape)
S_hr=np.zeros(zero_matrix_3.shape) # Hourly S
SUZ=np.zeros(zero_matrix_2.shape)
SUZ_hr=np.zeros(zero_matrix_3.shape)
EX_D=np.zeros(zero_matrix_2.shape)
EX_hr=np.zeros(zero_matrix_3.shape)
q=np.zeros(zero_matrix_2.shape) # Discharge per unit width [m^3/h/m]
Dr_hr = np.zeros(zero_matrix_3.shape) # Soil detachement by rain drop.
Dr_D=np.zeros(zero_matrix_2.shape) # Daily pixel scale detachment by raindrop
UZ=np.zeros(zero_matrix_2.shape)
UZ_hr=np.zeros(zero_matrix_3.shape)
n=np.zeros(K.shape) # Manning's n
Sf=np.zeros(zero_matrix_2.shape) # Friction slope
Qpeak=np.zeros(zero_matrix_2.shape)
Dof=np.zeros(zero_matrix_2.shape) # Daily pixel scale detachment by overland flow
Dof_tot=np.zeros(zero_matrix_2.shape)
Dof_tot_ds=np.zeros(zero_matrix_2.shape) #Total detachment in cell with overland flow
Dof_tot_ds=np.zeros(zero_matrix_2.shape) #Total detachment in cell with overland flow that is tranported to the downstream cell
Dch=np.zeros(zero_matrix_2.shape) # Daily pixel scale detachment capacity by concentrated flow
Dch_no_nan=np.zeros(zero_matrix_2.shape)
Df=np.zeros(zero_matrix_2.shape) # Detachment by concentrated flow
Df_tot=np.zeros(zero_matrix_2.shape) # Total detachment in cell with concentrated flow
Df_tot_ds=np.zeros(zero_matrix_2.shape) # Total detachment in cell with concentrated flow.
Dof_cc=np.zeros(zero_matrix_2.shape) # Sheet erosion on non-channel section of Channel erosion cell.
Dcc=np.zeros(zero_matrix_2.shape) # Detachment from cell with channel feature
Dcc_ds=np.zeros(zero_matrix_2.shape) #Detached soil from channel feature transported to downstream cell.
Tau_bar=np.zeros(zero_matrix_2.shape) # Average shear stress.
h=np.zeros(zero_matrix_2.shape) # Cahnnel flow depth
PSY_2=np.zeros(zero_matrix_2.shape)
Tc = np.zeros(zero_matrix_2.shape) # Transport capacity of flow
PGE=np.zeros(zero_matrix_2.shape) #Daily gross soil erosion at pixel scale
PSY=np.zeros(zero_matrix_2.shape) #Daily Pixel scale sediment yield
PSY_masked_by_catchment=np.zeros(zero_matrix_2.shape)
PSY_yr=np.zeros(K.shape) # Yearly pixel scale sediment scale
PSY_th=np.zeros(zero_matrix_2.shape) # daily pixel scale threshold values
PSY_th_masked_by_catchment=np.zeros(zero_matrix_2.shape) # Daily pixel thresholds masked by catchment bounds
PSY_yr_th=np.zeros(K.shape) # Threshold values for yearly sediment scale
PSD=np.zeros(zero_matrix_2.shape) #Pixel scale net daily soil deposition
PSD_masked_by_catchment=np.zeros(zero_matrix_2.shape)
CSY=np.zeros(rain.shape) # Daily sediment yield at catchment outlet [Mg]
CGE=np.zeros(rain.shape) #Gross catchment sediment yield
CGE_Dr=np.zeros(rain.shape) # Catchment sediment yield from rain drop detachment
CGE_Dcc=np.zeros(rain.shape) #Cathcment sediment detachment from cell with concentrated flow
CGE_Dcc_ds=np.zeros(rain.shape)
CGE_Dch=np.zeros(rain.shape) #Cathcment sediment yield from concentrated flow
CGE_Dof_tot=np.zeros(rain.shape) #Cathcment sediment detachment from cell with no channel erosion
CGE_Dof_tot_ds=np.zeros(rain.shape) #Catchment-scale sediment yield from non-gulley areas
CSD=np.zeros(rain.shape) #Sediment deposition at catchment scale
SDR=np.zeros(rain.shape) # Sediment delivery ratio at catchment outlet
u=np.zeros(zero_matrix_2.shape)
Ai=np.zeros(Fac_masked.shape) # Upstream contributing area per unit contour length
ai=np.zeros(Fac_masked.shape) # Upstream contributing area per unit contour length
EPI=np.zeros(Fac_masked.shape) # Erosion power index: Christened by me [(as * tanBea)]
Wi=np.zeros(TIM.shape) # Width of imaginary channel in a cell
Tau_cr=np.zeros(TIM.shape)
C=np.zeros(K.shape) # Cover Management Factor

X=0
RS=50**2 # Raster resolution [m^2]
L=50 #Length of imaginary channel
catchment_area=Num_of_catchment_cells*RS/1000000 #[sq. km]

folder_path = generate_file_path(PC_NAME, 'Pst', cat_)  # Output folder
os.makedirs(folder_path, exist_ok=True) # Ensure the folder exists
#output_folder = r'C:\Users'+'\\'+pc_name+r'\Dropbox\Gifu_dai\PhD_Prog\Data\Simulation_out_put\2015\Namatala'

# CALCULATING UPSTREAM CONTRIBUTING AREA FOR A GIVEN CELL
for k in range(rows):
    for l in range(cols):
        if TIM[k,l]!=nan_value_TIM: # Excludes cells outside the watershed bounds
            if Fac[k,l]<1:
                Fac[k,l]=0
            Ai[k,l] = (Fac[k,l]+1)*RS #Upstream contributin area of a given cell
            ai[k,l]=(Fac[k,l]+1)*L #Upstream contributing area per unit contour length
            EPI[k,l] =ai[k,l]*np.tan(Rad_S[k,l]) # Erosive Power Index of concentrated flow
            
Zeta = 0.5
Wriv=8.4 # River width at catchment outlet
Epsilon = Wriv/((np.max(Ai))**Zeta)
Gamma = 9.81*998 # [kg/m^2/s^2] #*3600*3600 # Specific density of water in kg/m^2/h
Alpha = 2
Beta = 1
u=50 #200 #360#1800 #4461 # Flow velocity [m/h]: How did you find flow velocity? (From the unit hydrograph?)

# FINDING NUMBER OF CELLS WITH CONCENTRATED FLOW BY TI & EPI THRESHOLD METHOD of Moore et al. (1988)
EG_TI_threshold = 15 #Suggested by Moore et al. (1988), who adoptd 6.8
EG_EPI_threshold =35 #Suggested by Moore et al. (1988), who adopted 18
EG_cells = np.nansum((TIM_masked > EG_TI_threshold) & (EPI > EG_EPI_threshold)) # Cells with EG erosion as in Moore et al. (1988)
OF_cells=(Num_of_catchment_cells)-(EG_cells) # Cells with sheet erosion

# CALCULATING C factor & Wi
for k in range(rows):
    for l in range(cols):
        if TIM[k,l]!=nan_value_TIM:
            C[k,l]=np.exp(-Alpha*NDVI[k,l]/(Beta-NDVI[k,l])) # Azizian et al.#(2021) as in van der Knijff et al. (1999)
            Wi[k,l] = Epsilon*(Ai[k,l]**Zeta)


# ALLOCATE THE 'OPTIMUM PARAMETERS'
m = CalPars[X,0]
Te = CalPars[X,1]
td = CalPars[X,2]
SRZinitial = CalPars[X,3]
SRmax = CalPars[X,4]

## INITIALISE VALUES
Sbar_hr[0,0]=-m*np.log((Q1_hr)/(Te*np.exp(-Lambda)))

for t in range(rows_h):
    for ha in range(cols_h):
        for ia in range(rows):
            for ja in range(cols):
                # CALCULATE THE LOCAL STORAGE DEFICIT & RAINDROP DETACHMENT
                if TIM[ia,ja]!=nan_value_TIM:
                    S_hr[t,ia,ja,ha]=Sbar_hr[t,ha]+(m*(Lambda-TIM[ia,ja]))
                    if S_hr[t,ia,ja,ha]<0:
                        S_hr[t,ia,ja,ha]=0
                        
                    # SOIL DETACHMENT BY RAIN DROP
                    if ((0<=C[ia,ja]<=1) and (0<=K[ia,ja]<=1)): #"or" changed to "and"
                        Dr_hr[t,ia,ja,ha]=0.0138*K[ia,ja]*C[ia,ja]*((rain_hr[t,ha]*0.001)**2)*((2.96*(np.tan(Rad_S[ia,ja]))**0.79)+0.56) #Eqn 19  (Wang et al., 2010) [kg/h/m^2]
                        Dr_hr[t,ia,ja,ha]=Dr_hr[t,ia,ja,ha]*RS/1000 # [Mg/h]: else Dr_hr[t,ia,ja,ha]=0*****
                    
        # ROOTZONE CALCULATIONS
        SRZ_hr[0,0]=SRZinitial
        SRZ_hr[t,ha]=SRZ_hr[t,ha]-rain_hr[t,ha]
        if SRZ_hr[t,ha]<0.0:
            SUZ_hr[t,:,:,ha]=SUZ_hr[t,:,:,ha]-SRZ_hr[t,ha]
            SRZ_hr[t,ha]=0

        # UNSATURATED ZONE CALCULATIONS
        for ib in range(rows):
            for jb in range(cols):
                if TIM[ib,jb]!=nan_value_TIM:
                    if SUZ_hr[t,ib,jb,ha]>S_hr[t,ib,jb,ha]:
                        EX_hr[t,ib,jb,ha]=SUZ_hr[t,ib,jb,ha]-S_hr[t,ib,jb,ha] 
                        SUZ_hr[t,ib,jb,ha]=S_hr[t,ib,jb,ha]

                    # CALCULATE DRAINAGE FROM SUZ
                    if S_hr[t,ib,jb,ha]>0.0:
                        UZ_hr[t,ib,jb,ha]=SUZ_hr[t,ib,jb,ha]/(td*S_hr[t,ib,jb,ha]) 
                        if UZ_hr[t,ib,jb,ha]>SUZ_hr[t,ib,jb,ha]:
                            UZ_hr[t,ib,jb,ha]=SUZ_hr[t,ib,jb,ha]
                        SUZ_hr[t,ib,jb,ha]=SUZ_hr[t,ib,jb,ha]-UZ_hr[t,ib,jb,ha]
                        if SUZ_hr[t,ib,jb,ha]<0.0000001:
                            SUZ_hr[t,ib,jb,ha]=0
                            
                # EXTRACTING PEAK DISCHARGE PER HOUR
                Qpeak[t,ib,jb]=np.max(EX_hr[t,ib,jb,:])

        # CALCULATE ACTUAL EVAPOTRANSPIRATION
        if ET0_hr[t,ha]>0.0:
            ETa_hr[t,ha]=ET0_hr[t,ha]*(1-(SRZ_hr[t,ha]/SRmax))
            if ETa_hr[t,ha]>(SRmax-SRZ_hr[t,ha]):
                ETa_hr[t,ha]=SRmax-SRZ_hr[t,ha]
            SRZ_hr[t,ha]=SRZ_hr[t,ha]+ETa_hr[t,ha]

        # CALCULATE HOURLY DISCHARGES AT CATCHMENT OUTLET
        Qsub_hr[t,ha]=Te*np.exp(-Lambda)*np.exp((-1)*Sbar_hr[t,ha]/m)
        Qover_hr[t,ha]=np.sum(freq_1*EX_hr[t:t+1,:,:,ha:ha+1]) #Be conscious of the fact that only catchment cells are to be summed
        Qv_hr[t,ha]=np.sum(freq_1*UZ_hr[t:t+1,:,:,ha:ha+1]) 
        Qsim_hr[t,ha]=Qover_hr[t,ha]+Qsub_hr[t,ha]

        #UPDATING VARIABLES FOR THE NEXT HOUR TIME STEP
        if ha<(cols_h-1):
            Sbar_hr[t,ha+1]=Sbar_hr[t,ha]-Qv_hr[t,ha]+Qsub_hr[t,ha]
            SRZ_hr[t,ha+1]=SRZ_hr[t,ha]
            #for iy in range(rows):
                #for jy in range(cols):
                    #if TIM[iy,jy]!=nan_value_TIM: # For me it's strange how performance improves with the elimination of this code!
                        #SUZ_hr[t,iy,jy,ha+1]=SUZ_hr[t,iy,jy,ha]
                        

    # UPDATING VARIABLES FOR THE NEXT DAY TIME STEP
    if t<(rows_h-1):
        Sbar_hr[t+1,0]=Sbar_hr[t,23]-Qv_hr[t,23]+Qsub_hr[t,23] 
        SRZ_hr[t+1,0]=SRZ_hr[t,23]
        #for iz in range(rows):
            #for jz in range(cols):
                #if TIM[iz,jz]!=nan_value_TIM:
                    #SUZ_hr[t+1,iz,jz,0]=SUZ_hr[t,iz,jz,23]

    # CALCULATING DAILY FLUXES AT THE CATCHMENT OUTLET
    Qsub[t,:]=(np.mean(Qsub_hr[t,:]))*24
    Qover[t,:]=(np.mean(Qover_hr[t,:]))*24
    Qv[t,:]=(np.mean(Qv_hr[t,:]))*24
    Qsim[t,:]=Qover[t,:]+Qsub[t,:]
    
# CACULATING DAILY PIXEL Dr
Dr_D = np.sum(Dr_hr, axis=-1) # [Mg/day]
    
#**************** CALCULATING SOIL DETACHMENT BY RUNOFF************************    
       
# CALCULATIONS RELATED TO CONCENTRATED FLOW
for ie in range(rows):
    for je in range(cols):
        if TIM[ie,je]!=nan_value_TIM:
            n[ie,je]=0.06 # Chow (1959); Average value
            
            #******************* Extracting gulley pixels**********************
            # WE have already extracted gulleys by TI: Why this again?
            #if not LSF_masked.mask[ie,je]:
            #if LSF[ie,je]>1000:
                #LSF[ie,je]=0
            if LSF[ie,je]<0:
                LSF[ie,je]=0
            if K[ie,je]<0:
                K[ie,je]=0
                
            #*********************Critical Shear Stress************************
            Tau_cr[ie,je]=3.23-(5.6*Sa[ie,je])-(24.4*Or[ie,je])+(0.00088*Rod[ie,je]*1000) # Flanagan & Livingston (1995) [Pa]:The values look normal. Soil contents are ratios
            
            #******************************************************************
#******************************************************************************

EX_D = np.sum(EX_hr, axis=-1)
for day in range(rows_h):
     for id in range(rows):
         for jd in range(cols):
             if TIM[id,jd]!=nan_value_TIM:
                 h[day,id,jd]=(EX_D[day,id,jd]*RS/24000)/(Wi[id,jd]*u) #Vol/(wi*250) [m at hour timestep]
                 #u[day,id,jd]=(9.81*h[day,id,jd]*np.tan(Rad_S[id,jd])*3600*3600)**0.5 # Fondriest Environmental, Inc. (2014); [m/hr]
                 #n[day,id,jd]=0.05 #0.0903+(0.000791*Veg_pc[id,jd])-(0.04*EX_D[day,id,jd]*RS/86400) #Etedali et al. (2011), Eq. 5; What is the meaning of n=0?  Not feasible
                 
                 #Transport capacity: Beasley et al. (1980) & Wang etal. (2010)
                 q[day,id,jd]=RS*EX_D[day,id,jd]/(24000*60*L)    
                 if q[day, id, jd] <= 0.046:
                     Tc[day,id,jd]=(146*(np.tan(Rad_S[id,jd]))*(q[day, id, jd])**0.5) # [kg/m/min]
                     Tc[day,id,jd]=Tc[day,id,jd]*60*24*L/1000 #[Mg/day]
                 elif q[day, id, jd] > 0.046:
                     Tc[day,id,jd]=(14600*(np.tan(Rad_S[id,jd]))*(q[day, id, jd])**2) # # [kg/m^2/min]*(24/1000)*Wi[id,jd] # [Mg/day]
                     Tc[day,id,jd]=Tc[day,id,jd]*24*60*L/1000 # [Mg/day]
                     
                                  
                 #*****************Concentrated flow Erosion***********************
                 if (TIM[id,jd]>EG_TI_threshold) and (EPI[id,jd]>EG_EPI_threshold):
                     
                     # Sheet erosion phase in cell with assumed channel
                     if (0<=C[id,jd]<=1) and (0<=K[id,jd]<=1) and (0<=P[id,jd]<=1):
                         Dof_cc[day,id,jd]=11.8*(((EX_D[day,id,jd]*(L*(L-Wi[id,jd]))*0.001)*Qpeak[day,id,jd]/3600)**0.56)*K[id,jd]*C[id,jd]*LSF[id,jd]*P[id,jd]
                     else:
                         Dof_cc[day,id,jd]=0
                     if h[day,id,jd]==0:
                         Sf[day,id,jd]=0
                     else:
                         Sf[day,id,jd]=(n[id,jd]**2)*(u**2)/(((Wi[id,jd]*h[day,id,jd])/((2*h[day,id,jd])+Wi[id,jd]))**(4/3)) # Molls et al. (1998): Eqn. 6
                         #Sf[day,id,jd]=(n[id,jd]**2)*(u**2)/(h[day,id,jd]**(4/3)) # Wang et al. (2010), Eqn. 27#h[day,id,jd]=(EX_D[day,id,jd]*RS/24000)/(Wi[id,jd]*u)
                     Tau_bar[day,id,jd]=Gamma*h[day,id,jd]*Sf[day,id,jd]
                     if (Tau_bar[day,id,jd]>Tau_cr[id,jd]) and (0<=K[id,jd]<=1): 
                         Dch[day,id,jd] = Wi[id,jd]*K[id,jd]*((1.35*Tau_bar[day,id,jd])-Tau_cr[id,jd])**1.05 # Wang et al. (2010) Eqn. 23. [kg/h/m^2]
                     else:
                         Dch[day,id,jd] = 0
                     Dch[day,id,jd] = Dch[day,id,jd]*24*h[day,id,jd]*Wi[id,jd]/1000 # [Mg/day/channel in a pixel]
                     if (Dr_D[day,id,jd]+Dof_cc[day,id,jd])<=Tc[day,id,jd]: # Detachment ability of a flow, considering it is already ladden with sediment (must work best with flow received from upstream cells).
                         Df[day,id,jd]=Dch[day,id,jd]*(1-((Dr_D[day,id,jd]+Dof_cc[day,id,jd])/Tc[day,id,jd]))
                     else:
                         Df[day,id,jd]=0
                     Dcc[day,id,jd]=Df[day,id,jd]+Dr_D[day,id,jd]+Dof_cc[day,id,jd]
                     
                     # Comparing detached soil with transport capacity of the flow
                     if Dcc[day,id,jd]>Tc[day,id,jd]:
                         PSD[day,id,jd]=Dcc[day,id,jd]-Tc[day,id,jd]
                         Dcc_ds[day,id,jd]=Tc[day,id,jd]
                     else:
                         Dcc_ds[day,id,jd] = Dcc[day,id,jd] 
                         PSD[day,id,jd] =0
                     
                     #*********************************************************
                 
                 #****************Overland Flow Erosion by MUSLE***************
                 elif (0<=TIM[id,jd]<=EG_TI_threshold) or (0<=EPI[id,jd]<=EG_EPI_threshold):
                     if ((0<=C[id,jd]<=1) and (0<=K[id,jd]<=1) and (0<=P[id,jd]<=1)):
                         Dof[day,id,jd]=11.8*(((EX_D[day,id,jd]*RS*0.001)*Qpeak[day,id,jd]/3600)**0.56)*K[id,jd]*C[id,jd]*LSF[id,jd]*P[id,jd]
                         # Jackson et al. (1986); [Mg]...Williams used the CIA method to estimate peak doscharge
                     else:
                         Dof[day,id,jd]=0
                     Dof_tot[day,id,jd]=Dof[day,id,jd]+Dr_D[day,id,jd] #Total detachment in cell with overland flow
                     
                     # Comparing detached soil with transport capacity of the flow
                     if Dof_tot[day,id,jd]>Tc[day,id,jd]:
                         PSD[day,id,jd]=Dof_tot[day,id,jd]-Tc[day,id,jd]
                         Dof_tot_ds[day,id,jd]=Tc[day,id,jd]
                     else:
                         Dof_tot_ds[day,id,jd]=Dof_tot[day,id,jd]
                         PSD[day,id,jd] = 0
                         
             else:
                 Dcc[day,id,jd]=np.nan
                 Dof_tot[day,id,jd]=np.nan
                 PSD[day,id,jd]=np.nan
                 PSY[day,id,jd]=np.nan
                 PGE[day,id,jd]=np.nan
                 
                 #*************************************************************
               
#******************************************************************************
# Load the raster profile (metadata) from the original raster file
TIM_path = generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'TI______.tif')

with rasterio.open(TIM_path) as src:
    profile = src.profile
    
#Create a mask for catchment area
catchment_mask = geometry_mask(catchment_shapefile.geometry, out_shape=PSY_yr.shape, transform=profile['transform'], invert=True)

# Creating directories for storing plots and figures
yr='2015'
Fig_dir = generate_file_path(PC_NAME, 'Pst', cat_, 'Fig_____', 'Eros____', yr) # Soil erosion
os.makedirs(Fig_dir, exist_ok=True)
Fig_dir_2 = generate_file_path(PC_NAME, 'Pst', cat_, 'Fig_____', 'Dep____', yr) # Sediment deposition
os.makedirs(Fig_dir_2, exist_ok=True)
Fig_dir_3 = generate_file_path(PC_NAME, 'Pst', cat_, 'Fig_____', 'SYC____', yr) # Sediment Yield class
os.makedirs(Fig_dir_3, exist_ok=True)
Fig_dir_4 = generate_file_path(PC_NAME, 'Pst', cat_, 'Fig_____', 'SY_____', yr) # Sediment Yield class
os.makedirs(Fig_dir_4, exist_ok=True)
Plt_dir = generate_file_path(PC_NAME, 'Pst', cat_, 'Plt_____', 'Eros____', yr)
os.makedirs(Plt_dir, exist_ok=True)
Plt_dir_2 = generate_file_path(PC_NAME, 'Pst', cat_, 'Plt_____', 'Eros____', yr)
os.makedirs(Plt_dir_2, exist_ok=True)

# FURTHER PROCESSING SOIL EROSION DATA FOR PLOTTING AND GRAPHING
for day in range(rows_h):
    for id in range(rows):
        for jd in range(cols):
            if TIM[id,jd]!=nan_value_TIM:
                
                #************* Gross pixel-scale soil erosion*********************
                PSY[day,id,jd]=Dcc_ds[day,id,jd]+Dof_tot_ds[day,id,jd] #One of concentrated or sheet erosion happens in a cell
                PGE[day,id,jd]=Dcc[day,id,jd]+Dof_tot[day,id,jd] # Amount of detachment soil prior to deposition
                    
                # CATEGORISING DAILY SEDIMENT YIELD AT PIXEL SCALE [Mg/cell/year] (Jiang et al., 2014)
                if 0<=PSY[day,id,jd]<(2*RS/3650000):
                    PSY_th[day,id,jd]=0
                elif (2*RS/3650000)<=PSY[day,id,jd]<(10*RS/3650000):
                    PSY_th[day,id,jd]=1
                elif (10*RS/3650000)<=PSY[day,id,jd]<(50*RS/3650000):
                    PSY_th[day,id,jd]=2
                elif (50*RS/3650000)<=PSY[day,id,jd]<(100*RS/3650000):
                    PSY_th[day,id,jd]=3
                elif PSY[day,id,jd]>=(100*RS/3650000):
                    PSY_th[day,id,jd]=4
                

    # Apply the mask to the erosion values: Potential memory intensive zone (check how to make it efficient)
    PSY_th_masked_by_catchment[day,:,:] = np.where(catchment_mask, PSY_th[day,:,:], np.nan)
    PSY_masked_by_catchment[day,:,:] = np.where(catchment_mask, PSY[day,:,:], np.nan)
    PSD_masked_by_catchment[day,:,:] = np.where(catchment_mask, PSD[day,:,:], np.nan)

    #**************************************************************************


    CSY[day,:]=np.nansum(PSY[day,:,:]) # Catchment-scale yield [Mg/day]
    CSD[day,:]=np.nansum(PSD[day,:,:])# Catchment sediment deposition [Mg/day]
    CGE[day,:]=np.nansum(PGE[day,:,:]) #Mg/day
    CGE_Dof_tot[day,:]=np.nansum(Dof_tot[day,:,:]) #Mg/day
    CGE_Dof_tot_ds=np.nansum(Dof_tot_ds[day,:,:])
    CGE_Dcc[day,:]=np.nansum(Dcc[day,:,:]) #Mg/day 
    CGE_Dcc_ds[day,:]=np.nansum(Dcc_ds[day,:,:])
    CGE_Dr[day,:]=np.sum(Dr_D[day,:,:])

    # Graphing daily sediment fluxes
    plt_tif_over_shp(CatPolyPath, PSY_masked_by_catchment[day,:,:], num_classes=500, cmap='RdYlGn_r', alpha=1,cbar_label='Soil Erosion (Mg/day)', plot_title='', dir_path=Fig_dir)
    plt_tif_over_shp_2(CatPolyPath, PSY_th_masked_by_catchment[day,:,:], num_classes=5, cmap='RdYlGn_r', alpha=1,cbar_label='Sediment Yield Class', plot_title='', dir_path=Fig_dir_3)
    plt_tif_over_shp(CatPolyPath, PSD_masked_by_catchment[day,:,:], num_classes=500, cmap='RdYlGn_r', alpha=1,cbar_label='Soil Deposition (Mg/day)', plot_title='', dir_path=Fig_dir_3)

# CALCULATE YEARLY PIXEL-SCALE SEDIMENT YIELD & OR DEPOSITION
PSY_yr = np.nansum(PSY, axis=0)
PSD_yr = np.nansum(PSD, axis=0)

# Apply the mask to the yearly erosion values
PSY_yr_masked_by_catchment = np.where(catchment_mask, PSY_yr, np.nan)
PSD_yr_masked_by_catchment=np.where(catchment_mask, PSD_yr, np.nan)

# Graphing yearly sediment fluxes
plt_tif_over_shp(CatPolyPath, PSY_yr_masked_by_catchment[:,:], num_classes=500, cmap='RdYlGn_r', alpha=1,cbar_label='Soil Erosion (Mg/yr)', plot_title='', dir_path=Plt_dir)
plt_tif_over_shp(CatPolyPath, PSD_yr_masked_by_catchment[:,:], num_classes=500, cmap='RdYlGn_r', alpha=1,cbar_label='Sediment Deposition (Mg/yr)', plot_title='', dir_path=Plt_dir_2)

#******************************************************************************

# CATEGORISING YEARLY SEDIMENT YIELD AT PIXEL SCALE
for ig in range(rows):
    for jg in range(cols):
        if TIM[ig,jg]!=nan_value_TIM:
            if 0<=PSY_yr[ig,jg]<(2*RS/10000):
                PSY_yr_th[ig,jg]=0
            elif (2*RS/10000)<=PSY_yr[ig,jg]<(10*RS/10000):
                PSY_yr_th[ig,jg]=1
            elif (10*RS/10000)<=PSY_yr[ig,jg]<(50*RS/10000):
                PSY_yr_th[ig,jg]=2
            elif (50*RS/10000)<=PSY_yr[ig,jg]<(100*RS/10000):
                PSY_yr_th[ig,jg]=3
            elif PSY_yr[ig,jg]>=(100*RS/10000):
                PSY_yr_th[ig,jg]=4
            if np.isnan(K[ig, jg]):
                PSY_yr_th[ig, jg] = np.nan


# Apply the mask to the erosion values
PSY_yr_th_masked_by_catchment = np.where(catchment_mask, PSY_yr_th, np.nan)
plt_tif_over_shp_2(CatPolyPath, PSY_yr_th_masked_by_catchment[:,:], num_classes=5, cmap='RdYlGn_r', alpha=1,cbar_label='Sediment yield Class', plot_title='', dir_path=Fig_dir_4)

#*******************************SECTION 5*************************************
# Plot the fluxes
figsize=(7, 3.5) # Two-column journal format
labelsize=10
fontsize=10

# Define the start date
start_date = datetime(2015, 2, 24)  # Adjust this to your actual start date
#start_date = datetime(2016, 5, 1)  # Adjust this to your actual start date

# Create a date range based on the number of days in the data
date_range = [start_date + timedelta(days=i) for i in range(len(CSY))]

#*************************** Plotting Erosion fluxes***************************
# SEDIMENT YIELD ATTHE CATCHEMNT OUTLET

# Create figure with optimal size and high resolution
fig, ax = plt.subplots(figsize=figsize, dpi=300)

# # 1) Simulated erosion as bars
ax.bar(date_range, CSY[:, 0], color='blue', edgecolor='black', alpha=0.8)  # Black edges for clarity

# 2) Observed erosion as points
#    Ensure SYobs is the same length as CSY[:,0] or slice appropriately
ax.scatter(date_range, SYobs,
           color='red', marker='o', s=20,      # small red circles
           label='Observed')


# Labels and title
ax.set_ylabel('Sediment Yield (Mg/day)', fontsize=fontsize)
ax.set_title('', fontsize=fontsize, fontweight='bold')

# Grid
ax.grid(axis='y', linestyle='--', alpha=0.6)

# Format x-axis as dates
ax.xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m-%d'))
ax.xaxis.set_major_locator(mdates.MonthLocator(bymonthday=1))
ax.xaxis.set_minor_locator(mdates.MonthLocator(bymonthday=15))
ax.tick_params(axis='x', labelsize=labelsize, rotation=30)  # Improve readability

# # Set date ticks every 30 days
# ax.xaxis.set_major_locator(mdates.DayLocator(interval=30))

# # Rotate x-axis labels for better readability
# plt.setp(ax.xaxis.get_majorticklabels(), rotation=45, ha='right')

# Adjust layout to prevent clipping
plt.tight_layout()

# Saving to a Directory
# Ensure the save path exists
Hyd_plot = generate_file_path(PC_NAME, 'Pst', cat_, 'Plot____', yr)
os.makedirs(Hyd_plot, exist_ok=True)

# Save figure correctly
#plt.savefig(os.path.join(Hyd_plot, 'CSY_____.png'), dpi=300, bbox_inches="tight")

# Show the plot
plt.show()


# EROSION RATES
bar_width = 0.4
plt.figure(figsize=figsize, dpi=300)
plt.bar(date_range, CGE_Dcc[:, 0], color='blue', label='CGE_Dcc',width=bar_width)
plt.bar(date_range, CGE_Dof_tot[:, 0], color='red', bottom=CGE_Dcc[:, 0], label='CGE_Dof',width=bar_width)
plt.bar(date_range, CGE_Dr[:, 0], color='black', bottom=CGE_Dcc[:, 0] + CGE_Dof_tot[:, 0], label='CGE_Dr',width=bar_width)
bar_width_2 = 0.4  # Adjust as needed
plt.bar(date_range, -CSD[:, 0], color='orange', label='CSD', width=bar_width_2, align='edge') #CSD is taken as negative & net erosion as positive
plt.ylabel('Soil Erosion (Mg/day)')
plt.title('Sediment Deposition and Erosion Contribution by Erosion Type')
plt.grid(axis='y')
plt.gca().xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m-%d'))
plt.gca().xaxis.set_major_locator(mdates.MonthLocator(bymonthday=1))
plt.gca().xaxis.set_minor_locator(mdates.MonthLocator(bymonthday=15))
plt.gca().tick_params(axis='x', labelsize=labelsize, rotation=30)  # Improve readability
#plt.gca().xaxis.set_major_locator(mdates.DayLocator(interval=30))
plt.gcf().autofmt_xdate()
plt.legend()

# Saving to a Directory
# Ensure the save path exists
Hyd_plot = generate_file_path(PC_NAME, 'Pst', cat_, 'Plot____', yr)
os.makedirs(Hyd_plot, exist_ok=True)

# Save figure correctly
#plt.savefig(os.path.join(Hyd_plot, 'ErDpCnt_.png'), dpi=300, bbox_inches="tight")

plt.show()

# ---
# EROSION RATES & SOIL DEPOSTION
# --- Original, without obs erosion

bar_width = 0.4
plt.figure(figsize=figsize, dpi=300)
plt.bar(date_range, CGE[:, 0], color='blue', label='CGE',width=bar_width)
bar_width_2 = 0.4  # Adjust as needed
plt.bar(date_range, -CSD[:, 0], color='orange', label='CSD', width=bar_width_2, align='edge') #CSD is taken as negative & net erosion as positive
plt.ylabel('Soil Erosion (Mg/day)')
plt.title('')
plt.grid(axis='y')
plt.gca().xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m-%d'))
plt.gca().xaxis.set_major_locator(mdates.MonthLocator(bymonthday=1))
plt.gca().xaxis.set_minor_locator(mdates.MonthLocator(bymonthday=15))
plt.gca().tick_params(axis='x', labelsize=labelsize, rotation=30)  # Improve readability

plt.gcf().autofmt_xdate()

plt.legend()

Hyd_plot = generate_file_path(PC_NAME, 'Pst', cat_, 'Plot____', yr)
os.makedirs(Hyd_plot, exist_ok=True)

# Save figure correctly
#plt.savefig(os.path.join(Hyd_plot, 'ErosDep_.png'), dpi=300, bbox_inches="tight")

plt.show()


#---

# DEPOSITION RATES
bar_width = 0.4
plt.figure(figsize=figsize, dpi=300)
plt.bar(date_range, CSD[:, 0], color='blue', label='CSD',width=bar_width)
plt.ylabel('Soil Erosion (Mg/day)')
plt.title('')
plt.grid(axis='y')
plt.gca().xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m-%d'))
plt.gca().xaxis.set_major_locator(mdates.MonthLocator(bymonthday=1))
plt.gca().xaxis.set_minor_locator(mdates.MonthLocator(bymonthday=15))
plt.gca().tick_params(axis='x', labelsize=labelsize, rotation=30)  # Improve readability

#plt.gca().xaxis.set_major_locator(mdates.DayLocator(interval=30))
plt.gcf().autofmt_xdate()
plt.legend()

Hyd_plot = generate_file_path(PC_NAME, 'Pst', cat_, 'Plot____', yr)
os.makedirs(Hyd_plot, exist_ok=True)

# Save figure correctly
#plt.savefig(os.path.join(Hyd_plot, 'CSD_____.png'), dpi=300, bbox_inches="tight")

plt.show()


#************************ Plotting Discahrge fluxes****************************
# Define figure size explicitly
figsize = (7, 3.5)  # Two-column format

# Plotting
fig, ax1 = plt.subplots(figsize=figsize)
ax1.margins(x=0.02, y=0.05)  # Small margins for clarity

# Plot observed and simulated discharge
ax1.plot(date_range, Qobs, label='Qobs', color='red', linewidth=1.5)
ax1.plot(date_range, Qsim, label='Qsim', color='black', linestyle='dotted', linewidth=1.5)

# Set y-axis for discharge
ax1.set_ylabel('Discharge (mm/day)', fontsize=fontsize)
ax1.tick_params(axis='y', labelsize=labelsize)
ax1.set_ylim(0, max(Qobs.max(), Qsim.max()) * 1.1)  # Add 10% margin for readability

# Create secondary y-axis for rainfall
ax2 = ax1.twinx()
ax2.bar(date_range, -rain[:, 0], label='Rainfall', color='blue', alpha=0.2, width=1.0, edgecolor="black", zorder=10)

# Set y-axis for rainfall
ax2.set_ylabel('Rainfall (mm/day)', fontsize=fontsize)
ax2.tick_params(axis='y', labelsize=labelsize)
ax2.set_ylim(-max(rain[:, 0]) * 1.1, 0)  # Add margin for balance

# Format x-axis as dates with spacing
ax1.xaxis.set_major_formatter(mdates.DateFormatter('%Y-%m-%d'))
ax1.xaxis.set_major_locator(mdates.MonthLocator(bymonthday=1))
ax1.xaxis.set_minor_locator(mdates.MonthLocator(bymonthday=15))
ax1.tick_params(axis='x', labelsize=labelsize, rotation=30)  # Improve readability

# Fix tick labels for rainfall axis
yticks = ax2.get_yticks()
ax2.set_yticks(yticks)  # Ensure the same tick locations are used
ax2.set_yticklabels([f"{int(abs(x))}" for x in yticks], fontsize=fontsize)

# Adjust layout for better spacing
plt.gcf().autofmt_xdate()
plt.tight_layout()
plt.subplots_adjust(right=0.85)  # Prevent overlap with legend

# Add title (if needed)
# plt.title('Rainfall, Observed and Simulated Discharges', fontsize=12)

# Improve legend placement
ax1.legend(loc='upper left', fontsize=10, frameon=False, bbox_to_anchor=(0, 1.02))
ax2.legend(loc='upper right', fontsize=10, frameon=False, bbox_to_anchor=(1, 1.02))

# Ensure the save path exists
Hyd_plot = generate_file_path(PC_NAME, 'Pst', cat_, 'Plot____', yr)
os.makedirs(Hyd_plot, exist_ok=True)

# Save figure correctly
#plt.savefig(os.path.join(Hyd_plot, 'Hyd_____.png'), dpi=300, bbox_inches="tight")

# Show plot
plt.show()


#*******************************SECTION 6*************************************
# OUTPUT OBSEREVED & SIMULATED VALUES
RESULTS[:,:1]=rain
RESULTS[:,1:2]=ET0
RESULTS[:,2:3]=ETa
RESULTS[:,3:4]=CSY
RESULTS[:,4:5]=Qobs
RESULTS[:,5:6]=Qsim
RESULTS[:,6:7]=Qover
RESULTS[:,7:8]=Qsub
RESULTS[:,8:9]=Qv
RESULTS[:,9:10]=Sbar
RESULTS[:,10:11]=SRZ

# CALCUALTE THE OBJECTIVE FUNCTIONS
Er=np.sum(Qobs-Qsim)
SSE=np.sum((Qobs-Qsim)**2) # Calculation of error mean or error variance
Qm=np.mean(Qobs) # Mean of observed discharge
Qs=np.sum(Qobs)
SSU=np.sum((Qobs-Qm)**2) # Mean variance
STD=np.sqrt(SSU/rain.size)
Check_NS=1-(SSE/SSU) # NS value/likelihooh value/model performance efficiency
Check_RMSE=np.sqrt(SSE/(rain.size))
Check_RSR=Check_RMSE/STD
Check_PBIAS=Er*100/Qs
print('NS = ',Check_NS, 'RSR = ',Check_RSR, '& PBIAS =', Check_PBIAS)

# EXPORTING OUTPUT DATA TO EXCEL
#np.savetxt('FinalResult.csv',RESULTS,fmt='%.4f',delimiter=',',header="rain,ET0,\
#CSY,Qobs,Qsim,Qover,Qsub,Qv,Sbar,SRZ")
print ("TOPMODEL code ran for", (time.time() - start_time)/60, "minutes")

