import platform  
import os
import pandas as pd 
import geopandas as gpd
import matplotlib.pyplot as plt
from matplotlib_scalebar.scalebar import ScaleBar
from mpl_toolkits.axes_grid1.inset_locator import inset_axes
from pathlib import Path
import logging


#### Dynamic assignment of PC_NAME based on OS
if platform.system().lower() == 'windows':
    PC_NAME='okiri'
    #PC_NAME='emmanuel'
elif platform.system().lower() ==' darwin':
    PC_NAME = 'Emma'
else:
    PC_NAME = 'default'

#### Function to generate file paths
def generate_file_path(pc_name, *path_parts):
    if platform.system() == 'Windows':
        return os.path.join(f'C:\\Users\\{pc_name}\\Dropbox\\NIES\\NIES\\WRM\\TopEros', *path_parts)
    else:
        return os.path.join(f'/Users/{pc_name}/Dropbox/NIES/NIES/WRM/TopEros', *path_parts)
    
#### Function to read out excel _csv files
def read_csv(file_path):
    return pd.read_csv(file_path, low_memory=False).values





# Constants
cat_ = 'Namatala'



# 1. Load data

# --- Shapefiles (Vector data)
UG_shp = generate_file_path(PC_NAME, 'Org', 'UG_shape_file','Uganda_Districts-2020---136-wgs84.shp') # Path to Uganda
gdf_uganda = gpd.read_file(UG_shp)
if gdf_uganda.crs is None:
    # shapefile uses WGS84 per filename
    gdf_uganda.set_crs(epsg=4326, inplace=True)
    logging.info("Set CRS of Uganda boundary to EPSG:4326")

Cat_pol = generate_file_path(PC_NAME, 'Dat', 'Others__', cat_, 'CatPoly_.shp') # Path to Namatala catchment shape file
gdf_catch = gpd.read_file(Cat_pol)
if gdf_catch.crs is None:
    # assume same WGS84
    gdf_catch.set_crs(epsg=32636, inplace=True)

# --- Station locations & Convert to gdf
stn_ = generate_file_path(PC_NAME, 'Org', 'Stn_loc_.csv') # Monitoring station locations
stn_=Path(stn_)
if stn_.suffix.lower() == '.shp':
    gdf_stations = gpd.read_file(str(stn_))
else:
    df_stations = pd.read_csv(stn_)
    expected_cols = {'X', 'Y'}
    if not expected_cols.issubset(df_stations.columns):
        raise ValueError(f"CSV must contain columns {expected_cols}")
    gdf_stations = gpd.GeoDataFrame(
        df_stations,
        geometry=gpd.points_from_xy(df_stations['X'], df_stations['Y']),
        crs='EPSG:32636'
    )

# Reproject all to UTM Zone 36N
target_crs = 'EPSG:32636'
gdf_uganda = gdf_uganda.to_crs(target_crs)
gdf_catch = gdf_catch.to_crs(target_crs)
gdf_stations = gdf_stations.to_crs(target_crs)

# Create A4 figure
fig, ax = plt.subplots(figsize=(8.27, 11.69), dpi=300)

# Plot catchment and stations
gdf_catch.plot(ax=ax, edgecolor='black', facecolor='none', linewidth=1)
gdf_stations.plot(ax=ax, markersize=30, color='red', marker='^', label='Stations')
# --- Labe stations
for idx, row in gdf_stations.iterrows():
    ax.annotate(
        row['Station'],
        xy=(row.geometry.x, row.geometry.y),
        xytext=(3, 3), textcoords='offset points',
        fontsize=8, color='black'
    )

# Add scale bar and north arrow
scalebar = ScaleBar(1, units="m", location='lower left')
ax.add_artist(scalebar)

ax.annotate('N', xy=(0.92, 0.15), xycoords='axes fraction', fontsize=12,
            ha='center', va='center', bbox=dict(boxstyle='circle', fc='white'))
ax.arrow(0.92, 0.17, 0, 0.05, transform=ax.transAxes,
         linewidth=2, head_width=0.015, head_length=0.025)

# Title and turn off axis
title = ''
ax.set_title(title, fontsize=10)
#ax.set_axis_off()
# Enable grid with minor ticks
ax.grid(True, which='both', linestyle='--', linewidth=0.5)
# Set axis labels for projected coordinates
ax.set_xlabel('Easting (m)', fontsize=8)
ax.set_ylabel('Northing (m)', fontsize=8)
# Optionally hide top and right spines
#ax.spines['top'].set_visible(False)
#ax.spines['right'].set_visible(False)

# 5. Inset showing location in Uganda
axins = inset_axes(ax, width='30%', height='30%', loc='upper left', borderpad=1)
# Plot Uganda boundary
gdf_uganda.plot(ax=axins, edgecolor='gray', facecolor='#f0f0f0')
# Transform catchment to WGS84 & compute centroid
gdf_catch_wgs = gdf_catch.to_crs('EPSG:4326')
centroid_wgs = gdf_catch_wgs.geometry.centroid.iloc[0]
# Reproject centroid back to target CRS for plotting
gdf_cent = gpd.GeoSeries([centroid_wgs], crs='EPSG:4326').to_crs(target_crs)
axins.scatter(
    gdf_cent.x, gdf_cent.y,
    transform=axins.transData,
    s=100, marker='*', label='Catchment'
)
axins.set_title('', fontsize=8)
axins.set_axis_off()

# Save
# Ensure the save path exists
Map_plt = generate_file_path(PC_NAME, 'Pst', cat_, 'Plot____')
os.makedirs(Map_plt, exist_ok=True)
plt.savefig(os.path.join(Map_plt, 'loc_map.png'), dpi=300, bbox_inches="tight")

# #plt.savefig(output_png, bbox_inches='tight')
# print(f"Map saved to {output_png}")
