#ifndef TRANSPORT_IFACE_H_
#define TRANSPORT_IFACE_H_

#include <cstddef>
#include <cstdint>
#include <iosfwd>
#include <memory>
#include <optional>
#include <string>

#include "macro.h"
#include "util.h"

struct ConnectionContext {
  std::string client_hostname, server_hostname;
  std::uint16_t client_port, server_port;

  // SSH specific
  std::optional<std::string> ssh_server_pubkey, ssh_server_privkey;

  // TLS specific
  std::optional<std::string> tls_server_key, tls_server_cert;

  // CurveZMQ specific
  std::optional<std::string> curvezmq_client_cert, curvezmq_server_cert;

  // Snow specific
  std::optional<std::string> snow_noise_params;

  friend std::ostream &operator<<(std::ostream &stream,
                                  const ConnectionContext &cc) {
    stream << "Server hostname:\t" << cc.server_hostname << "\n";
    stream << "Server port:\t" << cc.server_port << "\n";
    stream << "SSH server public key:\t"
           << string_of_string_option(cc.ssh_server_pubkey) << "\n";
    stream << "SSH server private key:\t"
           << string_of_string_option(cc.ssh_server_privkey) << "\n";
    stream << "TLS server key:\t" << string_of_string_option(cc.tls_server_key)
           << "\n";
    stream << "TLS server cert:\t"
           << string_of_string_option(cc.tls_server_cert) << "\n";
    stream << "CurveZMQ client cert:\t"
           << string_of_string_option(cc.curvezmq_client_cert) << "\n";
    stream << "CurveZMQ server cert:\t"
           << string_of_string_option(cc.curvezmq_server_cert) << "\n";
    stream << "Snow noise parameters:\t"
           << string_of_string_option(cc.snow_noise_params);

    return stream;
  }
};

class ClientIFace {
 public:
  ClientIFace(std::shared_ptr<const ConnectionContext> conn_ctx);
  virtual ~ClientIFace() = 0;
  DISALLOW_COPY_AND_ASSIGN(ClientIFace);

  virtual void cleanup() = 0;
  virtual bool try_init() = 0;
  virtual void connect() = 0;
  virtual void recv(std::size_t nbytes) = 0;
  virtual void send(std::size_t nbytes) = 0;

 protected:
  std::shared_ptr<const ConnectionContext> conn_ctx_;
};

inline ClientIFace::ClientIFace(
    std::shared_ptr<const ConnectionContext> conn_ctx)
    : conn_ctx_{conn_ctx} {}

inline ClientIFace::~ClientIFace() {}

class ServerIFace {
 public:
  ServerIFace(std::shared_ptr<const ConnectionContext> conn_ctx);
  virtual ~ServerIFace() = 0;

  virtual void cleanup() = 0;
  virtual bool try_init() = 0;
  virtual void listen() = 0;
  virtual void accept() = 0;
  virtual void recv(std::size_t nbytes) = 0;
  virtual void send(std::size_t nbytes) = 0;

  DISALLOW_COPY_AND_ASSIGN(ServerIFace);

 protected:
  std::shared_ptr<const ConnectionContext> conn_ctx_;
};

inline ServerIFace::ServerIFace(
    std::shared_ptr<const ConnectionContext> conn_ctx)
    : conn_ctx_{conn_ctx} {}

inline ServerIFace::~ServerIFace() {}

#endif  // TRANSPORT_IFACE_H_
