#ifndef TLS_H_
#define TLS_H_

#include <cassert>
#include <memory>

#include "macro.h"
#include "transport_iface.h"

/*
 * Call this function to initialize TLS before use.
 */
void tls_global_init();

enum class TLSVersion { k1_2, k1_3 };

class TLS12Client : public ClientIFace {
public:
  TLS12Client(std::shared_ptr<const ConnectionContext> conn_ctx,
              TLSVersion version);
  virtual ~TLS12Client();

  void cleanup() override;
  bool try_init() override;
  void connect() override;
  void recv(std::size_t nbytes) override;
  void send(std::size_t nbytes) override;

  DISALLOW_COPY_AND_ASSIGN(TLS12Client);

private:
  struct TLS12ClientInternals;

  enum State { kUninit, kInit, kDead, kInvalid, kConnected };

  inline TLS12ClientInternals &internal() {
    assert(internal_);
    return *this->internal_;
  }

  std::unique_ptr<TLS12ClientInternals> internal_{nullptr};
  State state_;
};

class TLS12Server : public ServerIFace {
public:
  TLS12Server(std::shared_ptr<const ConnectionContext> conn_ctx,
              TLSVersion version);
  virtual ~TLS12Server();

  void cleanup() override;
  bool try_init() override;
  void listen() override;
  void accept() override;
  void recv(std::size_t nbytes) override;
  void send(std::size_t nbytes) override;

  DISALLOW_COPY_AND_ASSIGN(TLS12Server);

private:
  struct TLS12ServerInternals;

  enum State { kUninit, kInit, kDead, kInvalid, kConnected, kListening };

  static const int kBacklog{1};

  inline TLS12ServerInternals &internal() {
    assert(internal_);
    return *this->internal_;
  }

  std::unique_ptr<TLS12ServerInternals> internal_{nullptr};
  State state_;
};

#endif // TLS_H_
