#ifndef NET_UTIL_H_
#define NET_UTIL_H_

#include <stdexcept>

#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "log.h"
#include "util.h"

static inline struct addrinfo *addrinfo_ipv4_tcp(const char *hostname, const char *port) {
  struct addrinfo hints{}, *servinfo{nullptr};

  std::memset(&hints, 0, sizeof(struct addrinfo));
  hints.ai_family = AF_INET;
  hints.ai_socktype = SOCK_STREAM;
  hints.ai_flags = AI_NUMERICSERV;

  const auto rc = getaddrinfo(hostname, port, &hints, &servinfo);

  if (rc != 0) {
    LOG_AND_THROW_ERROR("getaddrinfo().", std::runtime_error);
  }

  assert(servinfo);

  return servinfo;
}

static inline void bind_socket_from_addrinfo(int sock, const struct addrinfo &info) {
  const auto rc = bind(sock, info.ai_addr, info.ai_addrlen);

  if (rc == -1) {
    LOG_AND_THROW_ERROR("bind().", std::runtime_error);
  }
}

static inline void connect_socket_from_addrinfo(int sock, const struct addrinfo &info) {
  const auto rc = connect(sock, info.ai_addr, info.ai_addrlen);

  if (rc == -1) {
    LOG_AND_THROW_ERROR("connect().", std::runtime_error);
  }
}

static inline void set_socket_reuseaddr(int sock) {
  const int yes{1};

  auto rc = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof(int));

  if (rc == -1) {
    LOG_AND_THROW_ERROR("setsockopt().", std::runtime_error);
  }
}

static inline int socket_from_addrinfo(const struct addrinfo &info) {
  const int sock{socket(info.ai_family, info.ai_socktype, info.ai_protocol)};

  if (sock == -1) {
    LOG_AND_THROW_ERROR("socket().", std::runtime_error);
  }

  set_socket_reuseaddr(sock);

  return sock;
}

#endif // NET_UTIL_H_
