#include "driver_requester.h"

#include <cassert>
#include <string>
#include <type_traits>
#include <utility>
#include <zmq.hpp>

#include "log.h"

struct DriverRequester::DriverRequesterInternals {
  zmq::context_t ctx;
  zmq::socket_t client_sock, server_sock;
};

DriverRequester::DriverRequester(std::string client_addr,
                                 std::uint16_t client_port,
                                 std::string server_addr,
                                 std::uint16_t server_port)
    : internal_{new DriverRequesterInternals{}},
      client_addr_{client_addr},
      server_addr_{server_addr},
      client_port_{client_port},
      server_port_{server_port} {}

DriverRequester::~DriverRequester() {}

bool DriverRequester::init() {
  {
    internal().client_sock =
        zmq::socket_t{internal().ctx, zmq::socket_type::req};
    std::string connect_addr =
        "tcp://" + client_addr_ + ":" + std::to_string(client_port_);
    LOG(info) << "Connecting to " << connect_addr << ".";
    internal().client_sock.connect(connect_addr);
  }

  {
    internal().server_sock =
        zmq::socket_t{internal().ctx, zmq::socket_type::req};
    std::string connect_addr =
        "tcp://" + client_addr_ + ":" + std::to_string(server_port_);
    LOG(info) << "Connecting to " << connect_addr << ".";
    internal().server_sock.connect(connect_addr);
  }

  return internal().client_sock && internal().server_sock;
}

void DriverRequester::send_message(const Command &req, Endpoint ep) {
  zmq::socket_t *sock{nullptr};

  switch (ep) {
    case Endpoint::CLIENT:
      sock = &internal().client_sock;
      break;
    case Endpoint::SERVER:
      sock = &internal().server_sock;
      break;
  }

  assert(sock);

  LOG(info) << "Sending " << req << ".";

  LOG(debug) << "Begin sending message.";
  send_command_synch(req, sock);
  LOG(debug) << "End sending message.";

  LOG(debug) << "Begin receiving message.";
  auto rep = recv_command_synch(sock);
  LOG(debug) << "End receiving message.";

  LOG(info) << "Recieved " << *rep << ".";
}
