#include "driver_common.h"

#include <google/protobuf/util/json_util.h>

#include <cassert>
#include <cstdint>
#include <ostream>
#include <zmq.hpp>

#include "log.h"
#include "transcript.pb.h"

std::ostream &operator<<(std::ostream &stream, const Command &cmd) {
  stream << "< " << Command::Kind_Name(cmd.kind()) << ", ";

  stream << "(";
  switch (cmd.kind()) {
    case Command::SEND:
      stream << cmd.nbytes();
      break;
    case Command::RECV:
      stream << cmd.nbytes();
      break;
    case Command::ACK:
      stream << Command::Kind_Name(cmd.ack());
      break;
    default:
      break;
  }
  stream << ")";

  stream << " >";

  return stream;
}

template <typename T>
static std::optional<std::string> json_of_pb_object_(const T &t) {
  std::string retval{};

  google::protobuf::util::JsonOptions options{};
  auto status =
      google::protobuf::util::MessageToJsonString(t, &retval, options);

  if (!status.ok()) {
    LOG(error) << "Converting to JSON failed.";
    return {};
  }

  return retval;
}

template <typename T>
static std::optional<std::unique_ptr<T>> pb_object_of_json_(const char *str) {
  assert(str);

  std::unique_ptr<T> retval{new T};

  google::protobuf::util::JsonParseOptions options{};
  auto status =
      google::protobuf::util::JsonStringToMessage(str, retval.get(), options);

  if (!status.ok()) {
    LOG(error) << "Parsing JSON failed.";
    return {};
  }

  return retval;
}

void zmq_message_of_command(const Command &cmd, zmq::message_t *p_out) {
  assert(p_out);
  auto json = *json_of_pb_object_(cmd);
  zmq::message_t zmsg{json.cbegin(), json.cend()};
  p_out->copy(zmsg);
}

std::unique_ptr<Command> command_of_zmq_message(const zmq::message_t &zmsg) {
  std::string s{static_cast<const char *>(zmsg.data()), zmsg.size()};
  return *pb_object_of_json_<Command>(s.c_str());
}

void send_command_synch(const Command &cmd, zmq::socket_t *sock) {
  LOG(info) << "In send_message_synch.";
  assert(sock);
  zmq::message_t zmsg{};
  zmq_message_of_command(cmd, &zmsg);
  LOG(info) << "Sending " << zmsg.size() << " bytes.";
  sock->send(zmsg, zmq::send_flags::none);
}

std::unique_ptr<Command> recv_command_synch(zmq::socket_t *sock) {
  LOG(info) << "In recv_message_synch.";
  assert(sock);
  zmq::message_t zmsg{};
  auto rv = sock->recv(zmsg);
  LOG(info) << "Received " << rv.value() << " bytes.";
  return command_of_zmq_message(zmsg);
}

std::optional<std::unique_ptr<Transcript>> transcript_of_json(const char *str) {
  assert(str);
  return pb_object_of_json_<Transcript>(str);
}

std::optional<std::string> json_of_transcript(const Transcript &transcript) {
  return json_of_pb_object_(transcript);
}
