#include <boost/program_options.hpp>
#include <cassert>
#include <cstdint>
#include <cstdio>
#include <cstdlib>
#include <exception>
#include <iostream>
#include <memory>
#include <optional>
#include <string>
#include <type_traits>
#include <utility>

#include "driver_common.h"
#include "driver_requester.h"
#include "log.h"
#include "util.h"

class ProgramOptions {
 public:
  ProgramOptions() {}

  static ProgramOptions make_or_exit(int argc, char **argv) {
    namespace po = boost::program_options;

    ProgramOptions retval{};

    // clang-format off
    po::options_description desc{"Program options"};
    desc.add_options()
      ("help,h", "produce help message")
      ("client-hostname,c", po::value<std::string>(&retval.client_addr_)->required(), "client hostname")
      ("client-port,p", po::value<std::uint16_t>(&retval.client_port_)->required(), "client control port")
      ("server-hostname,s", po::value<std::string>(&retval.server_addr_)->required(), "server hostname")
      ("server-port,P", po::value<std::uint16_t>(&retval.server_port_)->required(), "server control port")
      ("transcript-filepath,t", po::value<std::string>(&retval.transcript_filepath_)->required(), "path to transcript file")
      ("log-level,l", po::value<std::string>(&retval.log_level_)->default_value("info"), "trace|debug|info|warning|error|fatal");
    // clang-format on

    po::positional_options_description p;
    p.add("client-hostname", 1);
    p.add("client-port", 1);
    p.add("server-hostname", 1);
    p.add("server-port", 1);
    p.add("transcript-filepath", 1);

    po::variables_map vm{};

    auto exit = [&](int exit_code) {
      std::ostream *os{nullptr};

      if (exit_code == EXIT_SUCCESS) {
        os = &std::cout;
      } else {
        os = &std::cerr;
      }

      *os << "Usage:\t" << argv[0]
          << " [OPTIONS] CLIENT_HOSTNAME CLIENT_PORT SERVER_HOSTNAME "
             "SERVER_PORT TRANSCRIPT_FILEPATH"
          << std::endl;
      *os << desc << std::endl;
      std::exit(exit_code);
    };

    try {
      po::store(
          po::command_line_parser(argc, argv).options(desc).positional(p).run(),
          vm);

      po::notify(vm);

      if (vm.count("help")) {
        exit(EXIT_SUCCESS);
      }

      const auto &ll = retval.log_level_;
      if (ll.compare("trace") != 0 && ll.compare("debug") != 0 &&
          ll.compare("info") != 0 && ll.compare("warning") != 0 &&
          ll.compare("error") != 0 && ll.compare("fatal") != 0) {
        throw po::invalid_option_value(ll);
      }

    } catch (const std::exception &ex) {
      std::cerr << "ERROR:\t" << ex.what() << "\n" << std::endl;
      exit(EXIT_FAILURE);
    }

    return retval;
  }

  friend std::ostream &operator<<(std::ostream &stream,
                                  const ProgramOptions &po) {
    stream << "Client address:\t" << po.client_addr() << std::endl;
    stream << "Client port:\t" << po.client_port() << std::endl;
    stream << "Server address:\t" << po.server_addr() << std::endl;
    stream << "Server port:\t" << po.server_port() << std::endl;
    stream << "Transcript filepath:\t" << po.transcript_filepath() << std::endl;
    stream << "Log level:\t" << po.log_level() << "\n";
    return stream;
  }

  const std::string &client_addr() const { return client_addr_; }
  const std::string &server_addr() const { return server_addr_; }
  std::uint16_t client_port() const { return client_port_; }
  std::uint16_t server_port() const { return server_port_; }
  const std::string &log_level() const { return log_level_; }
  const std::string &transcript_filepath() const {
    return transcript_filepath_;
  }

 private:
  std::string client_addr_, server_addr_;
  std::uint16_t client_port_, server_port_;

  std::string log_level_{"null"};
  std::string transcript_filepath_{"/dev/null"};
};

int main(int argc, char **argv) {
  auto program_options = ProgramOptions::make_or_exit(argc, argv);

  log_init(program_options.log_level());

  LOG(info) << "Program options:\n" << program_options;

  auto controller = DriverRequester{
      program_options.client_addr(), program_options.client_port(),
      program_options.server_addr(), program_options.server_port()};

  controller.init();

  controller.send_message(make_no_data_command(Command::SERVER_LISTEN),
                          Endpoint::SERVER);

  controller.send_message(make_no_data_command(Command::CLIENT_CONNECT),
                          Endpoint::CLIENT);

  const auto transcript_text =
      read_file(program_options.transcript_filepath().c_str());
  const auto transcript = *transcript_of_json(transcript_text->c_str());
  const auto transcript_cmd = make_transcript_command(*transcript);

  controller.send_message(transcript_cmd, Endpoint::CLIENT);
  controller.send_message(transcript_cmd, Endpoint::SERVER);

  controller.send_message(make_no_data_command(Command::STOP),
                          Endpoint::CLIENT);

  controller.send_message(make_no_data_command(Command::STOP),
                          Endpoint::SERVER);

  return 0;
}
