#!/usr/bin/env python3

import asyncio
import logging
import sys

import time

import common as cmn
from ptadapter import ClientAdapter, ServerAdapter



class SynchServer:
    def __init__(self, pt_exec, transport, options, forward_host, forward_port):
        self._pt_exec = pt_exec
        self._transport = transport
        self._options = options
        self._forward_host = forward_host
        self._forward_port = forward_port

    async def init(self):
        self._srv_adapter = ServerAdapter(
            self._pt_exec, "./server_state", self._forward_host, self._forward_port
        )

        # TODO: setup my port first, then feed in the port number.
        self._srv_adapter.add_transport(
            self._transport, "127.0.0.1", 9000, options=self._options
        )

        logging.info("Server has initialized.")

    async def start_impl(self):
        await self._srv_adapter.start()
        logging.info("Server has started.")

    async def stop_impl(self):
        await self._srv_adapter.stop()
        logging.info("Server has stopped.")
        await self._srv_adapter.wait()
        logging.info("Server has exited.")

    def start(self, loop):
        t = loop.create_task(self.start_impl())
        loop.run_until_complete(t)


class SynchClient:
    def __init__(self, pt_exec, transports, options, connect_host, connect_port):
        self._client_adapter = None
        self._pt_exec = pt_exec
        self._transports = transports
        self._options = options
        self._connect_host = connect_host
        self._connect_port = connect_port

    async def init(self):
        self._client_adapter = ClientAdapter(self._pt_exec, "./state", self._transports)
        logging.info("Client has initialized.")
        await self._client_adapter.start()
        logging.info("Client has started.")

    async def start(self):
        logging.info("Client connecting...")

        reader, writer = await self._client_adapter.open_transport_connection(
            "obfs4", self._connect_host, self._connect_port, args=self._options
        )

        self._reader = reader
        self._writer = writer

        writer.write("abcdefghijklmnop\n".encode())
        await writer.drain()

        logging.info("Client has connected.")


def main():
    logging.basicConfig(level=logging.INFO, stream=sys.stderr)
    loop = asyncio.new_event_loop()

    cfg = cmn.parse_config("./test_cfg.ini")

    logging.info(f"Config: {cfg}")

    pt_exec = [cfg["obfs4"]["exe_path"], "-enableLogging"]
    state = None
    transports = ["obfs4"]

    server_options = {
        "node-id": cfg["obfs4"]["server_node-id"],
        "private-key": cfg["obfs4"]["server_private-key"],
        "public-key": cfg["obfs4"]["server_public-key"],
        "drbg-seed": cfg["obfs4"]["server_drbg-seed"],
        "iat-mode": cfg["obfs4"]["server_iat-mode"],
    }

    server = SynchServer(pt_exec, "obfs4", server_options, "127.0.0.1", 55555)

    loop.run_until_complete(server.init())
    loop.run_until_complete(server.start_impl())

    time.sleep(3)

    client_options = {
        "cert": cfg["obfs4"]["client_cert"],
        "iat-mode": cfg["obfs4"]["client_iat-mode"],
    }
    client = SynchClient(pt_exec, transports, client_options, "127.0.0.1", 9000)

    loop.run_until_complete(client.init())

    time.sleep(3)

    loop.run_until_complete(client.start())

    # async with ClientAdapter(pt_exec, state, transports) as adapter:
    #    args = {"cert": cfg["obfs4"]["client_cert"]}
    #    reader, writer = await adapter.open_transport_connection(
    #        "obfs4", "127.0.0.1", 9000, args
    #    )

    time.sleep(3)

    loop.run_until_complete(server.stop_impl())

    quit()

    # async with ClientAdapter(pt_exec, state, transports) as adapter:
    #     args = {"cert": cfg["obfs4"]["client_cert"]}
    #     reader, writer = await adapter.open_transport_connection(
    #         "obfs4", "127.0.0.1", 7900, args
    #     )

    # async with ClientAdapter(pt_exec, state, transports) as adapter:
    #     args = {"cert": cfg["obfs4"]["client_cert"]}
    #     reader, writer = await adapter.open_transport_connection(
    #         "obfs4", "127.0.0.1", 7900, args
    #     )


if __name__ == "__main__":
    main()
