#!/usr/bin/env python3

import argparse
import logging
import sys

import numpy as np

from transcript_pb2 import Transcript
from google.protobuf.json_format import MessageToJson


def make_download_transcript():
    rounds = []

    round = Transcript.Round()
    round.client_send_nbytes = 1000
    round.server_send_nbytes = 0
    round.sleep_usec = 0

    rounds.append(round)

    for _ in range(10):
        round = Transcript.Round()
        round.client_send_nbytes = 0
        round.server_send_nbytes = 10000
        round.sleep_usec = 0
        rounds.append(round)

        round = Transcript.Round()
        round.client_send_nbytes = 256
        round.server_send_nbytes = 0
        round.sleep_usec = 0
        rounds.append(round)

    transcript = Transcript()
    transcript.rounds.extend(rounds)
    return transcript


def make_upload_transcript():
    rounds = []

    round = Transcript.Round()
    round.client_send_nbytes = 1000
    round.server_send_nbytes = 0
    round.sleep_usec = 0

    rounds.append(round)

    for _ in range(10):
        round = Transcript.Round()
        round.client_send_nbytes = 10000
        round.server_send_nbytes = 0
        round.sleep_usec = 0
        rounds.append(round)

        round = Transcript.Round()
        round.client_send_nbytes = 0
        round.server_send_nbytes = 256
        round.sleep_usec = 0
        rounds.append(round)

    transcript = Transcript()
    transcript.rounds.extend(rounds)
    return transcript


def make_exchange_transcript():
    rounds = []

    round = Transcript.Round()
    round.client_send_nbytes = 10000
    round.server_send_nbytes = 10000
    round.sleep_usec = 0

    rounds.append(round)

    transcript = Transcript()
    transcript.rounds.extend(rounds)
    return transcript


def make_worst_transcript():
    rounds = []

    round = Transcript.Round()
    round.client_send_nbytes = 1
    round.server_send_nbytes = 0
    round.sleep_usec = 5000000

    rounds.append(round)

    round = Transcript.Round()
    round.client_send_nbytes = 0
    round.server_send_nbytes = 1
    round.sleep_usec = 5000000

    rounds.append(round)

    transcript = Transcript()
    transcript.rounds.extend(rounds)
    return transcript


def make_interactive_transcript():
    rounds = []

    # <https://userinterfaces.aalto.fi/136Mkeystrokes/resources/chi-18-analysis.pdf>
    iki_mu = 238660
    iki_sigma = 111600

    for _ in range(100):
        delay = max(0, int(np.random.default_rng().normal(iki_mu, iki_sigma, 1)[0]))

        round = Transcript.Round()
        round.client_send_nbytes = 1
        round.server_send_nbytes = 0
        round.sleep_usec = 0

        rounds.append(round)

        round = Transcript.Round()
        round.client_send_nbytes = 0
        round.server_send_nbytes = 1
        round.sleep_usec = delay

        rounds.append(round)

    transcript = Transcript()
    transcript.rounds.extend(rounds)
    return transcript


def main(args: argparse.Namespace):
    logging.basicConfig(level=logging.INFO, stream=sys.stderr)
    logging.info(f"Program arguments: {args}")

    dispatch = {
        "download": make_download_transcript,
        "upload": make_upload_transcript,
        "exchange": make_exchange_transcript,
        "worst": make_worst_transcript,
        "interactive": make_interactive_transcript,
    }

    transcript = dispatch[args.pattern]()

    json_obj = MessageToJson(transcript)

    print(json_obj)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "pattern", choices=["download", "upload", "exchange", "worst", "interactive"]
    )
    return parser.parse_args()


if __name__ == "__main__":
    main(parse_args())
