#!/usr/bin/env python3

import argparse
from dataclasses import dataclass
import json
import logging
import random
import sys
from typing import NewType

CLIENT = 0
SERVER = 1

@dataclass
class Burst:
    endpoint: int
    nbytes: int

    def to_json(self):
        d = { "value0" : self.endpoint,
              "value1" : self.nbytes }

        return json.dumps(d)

@dataclass
class TraceSpec:
    data: list[Burst]

    def to_json(self):
        inner = ",".join(map(lambda x: x.to_json(), TraceSpec._simplify_data(self.data)))
        outer_begin = '{"value0": {"value0": ['
        outer_end = ']}}'
        return outer_begin + inner + outer_end

    @staticmethod
    def _simplify_data(data):
        if len(data) == 0:
            return []

        new_data = [data[0]]

        for burst in data[1:]:
            last_burst = new_data[-1]

            if last_burst.endpoint == burst.endpoint:
                last_burst.nbytes += burst.nbytes
            else:
                new_data.append(burst)

        return new_data


Pattern = NewType("Pattern", str)

def compute_noisy_uniform_chunks(nbytes, nchunks):
    assert nchunks > 0

    nleft = nbytes

    chunk_mean = nbytes / nchunks

    chunks = []

    while nleft > 0:
        chunk_nbytes = min(int(random.gauss(chunk_mean, 0.1 * chunk_mean)), nleft)
        chunk_nbytes = max(1, chunk_nbytes)
        chunks.append(chunk_nbytes)
        nleft -= chunk_nbytes

    assert sum(chunks) == nbytes

    return chunks

def build_trace_spec(nbytes_send: int, nbytes_recv: int, pattern: Pattern) -> TraceSpec:
    ts = TraceSpec([])

    nbytes_send_left = nbytes_send
    nbytes_recv_left = nbytes_recv

    if pattern == "download":
        initial_request_nbytes = int(random.uniform(.7 * nbytes_send, .9 * nbytes_send))
        nbytes_send_left -= initial_request_nbytes

        ts.data.append(Burst(CLIENT, initial_request_nbytes))

        nchunks = max(1, int(random.expovariate(lambd=0.1)))

        download_chunks = compute_noisy_uniform_chunks(nbytes_recv_left, nchunks)
        download_chunks = [Burst(SERVER, x) for x in download_chunks]

        nchunks = len(download_chunks)

        nacks = nchunks - 1

        if nacks == 0:
            ack_chunks = [nbytes_send_left]
        else:
            ack_chunks = compute_noisy_uniform_chunks(nbytes_send_left, nacks)

        ack_chunks = [Burst(CLIENT, x) for x in ack_chunks]

        ts.data.extend(interleave(download_chunks, ack_chunks))

    return ts

def interleave(xs: list, ys: list) -> list:
    retval = []

    for (x, y) in zip(xs, ys):
        retval.append(x)
        retval.append(y)

    if len(xs) > len(ys):
        retval.extend(xs[len(ys):])
    elif len(ys) > len(xs):
        retval.extend(ys[len(xs):])

    return retval

def main(args: argparse.Namespace):

    logging.basicConfig(level=logging.INFO, stream=sys.stderr)
    logging.info(f"Program arguments: {args}")

    ts = build_trace_spec(args.nbytes_send, args.nbytes_recv, args.pattern)

    total_sent = sum([b.nbytes for b in ts.data if b.endpoint == CLIENT])
    total_recv = sum([b.nbytes for b in ts.data if b.endpoint == SERVER])

    assert total_sent == args.nbytes_send
    assert total_recv == args.nbytes_recv

    print(ts.to_json())

def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("nbytes_send", type=int)
    parser.add_argument("nbytes_recv", type=int)
    parser.add_argument("pattern", choices = ["download", "upload", "interactive"])
    return parser.parse_args()

if __name__ == "__main__":
    main(parse_args())
