#!/usr/bin/env python3

import argparse
import logging
import sys

import zmq

from command_pb2 import Command
from transcript_pb2 import Transcript

import common as cmn


def main(args: argparse.Namespace):
    logging.basicConfig(level=logging.INFO, stream=sys.stderr)
    logging.info(f"Command line arguments: {args}")

    cfg = cmn.parse_config(args.cfg_filepath)

    logging.info(f"Config: {cfg}")

    transcript_contents = cmn.read_file(args.transcript_filepath)
    transcript = cmn.parse_transcript(transcript_contents)

    context = zmq.Context()
    server_socket = context.socket(zmq.REQ)
    server_socket.connect(f"tcp://127.0.0.1:{cfg['control']['server_port']}")

    client_socket = context.socket(zmq.REQ)
    client_socket.connect(f"tcp://127.0.0.1:{cfg['control']['client_port']}")

    cmn.send_command_synch(
        server_socket, cmn.make_no_data_command(Command.Kind.SERVER_LISTEN)
    )
    cmn.recv_command_synch(server_socket)

    cmn.send_command_synch(
        client_socket, cmn.make_no_data_command(Command.Kind.CLIENT_CONNECT)
    )
    cmn.recv_command_synch(client_socket)

    transcript_cmd = cmn.make_transcript_command(transcript)

    cmn.send_command_synch(client_socket, transcript_cmd)
    cmn.recv_command_synch(client_socket)

    cmn.send_command_synch(server_socket, transcript_cmd)
    cmn.recv_command_synch(server_socket)

    cmn.send_command_synch(server_socket, cmn.make_no_data_command(Command.Kind.STOP))
    cmn.recv_command_synch(server_socket)

    cmn.send_command_synch(client_socket, cmn.make_no_data_command(Command.Kind.STOP))
    cmn.recv_command_synch(client_socket)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("cfg_filepath", type=str)
    parser.add_argument("transcript_filepath", type=str)
    return parser.parse_args()


if __name__ == "__main__":
    main(parse_args())
