#!/usr/bin/env python3

import logging
import re
import sys

from configobj import ConfigObj
from configobj.validate import Validator
from command_pb2 import *
from transcript_pb2 import Transcript
from google.protobuf.json_format import Parse, MessageToJson
import zmq


def parse_config(filepath) -> ConfigObj:
    cfg = ConfigObj(filepath)
    return cfg


def send_command_synch(sock: zmq.Socket, cmd: Command):
    s = MessageToJson(cmd)
    sock.send_string(s)


def recv_command_synch(sock: zmq.Socket) -> Command:
    msg = sock.recv()
    cmd = Parse(msg.decode(), Command())
    logging.info(f"Received command {cmd}")
    return cmd


def make_client_connect_command(pluggable_transport: PluggableTransport,
                                ip_addr: str, port: int,
                                options: ProteusOptions, pt_exec: str):
    cmd = Command()

    params = ClientParams()
    params.pluggable_transport = pluggable_transport
    params.ip_addr = ip_addr
    params.port = port
    params.pt_exec = pt_exec

    if pluggable_transport == PluggableTransport.PROTEUS:
        params.proteus_options.CopyFrom(options)

    cmd.kind = Command.Kind.CLIENT_START
    cmd.client_params.CopyFrom(params)

    return cmd


def make_server_listen_command(pluggable_transport: PluggableTransport,
                               port: int, options: ProteusOptions,
                               pt_exec: str):
    cmd = Command()

    params = ServerParams()
    params.pluggable_transport = pluggable_transport
    params.port = port
    params.pt_exec = pt_exec

    if pluggable_transport == PluggableTransport.PROTEUS:
        params.proteus_options.CopyFrom(options)

    cmd.kind = Command.Kind.SERVER_START
    cmd.server_params.CopyFrom(params)

    return cmd


def make_start_tcpdump_command(filepath: str, port: int):
    cmd = Command()

    params = TCPDumpParams()
    params.filepath = filepath
    params.port = port

    cmd.kind = Command.Kind.START_TCPDUMP
    cmd.tcp_dump_params.CopyFrom(params)

    return cmd


def make_no_data_command(kind: Command.Kind) -> Command:
    cmd = Command()
    cmd.kind = kind
    return cmd


def make_ack_command(kind: Command.Kind) -> Command:
    cmd = Command()
    cmd.kind = Command.Kind.ACK
    cmd.ack = kind
    return cmd


def make_data_command(kind: Command.Kind, nbytes: int) -> Command:
    assert kind in (Command.Kind.SEND, Command.Kind.RECV)

    cmd = Command()
    cmd.kind = kind
    cmd.nbytes = nbytes
    return cmd


def make_transcript_command(transcript: Transcript) -> Command:
    cmd = Command()
    cmd.kind = Command.Kind.TRANSCRIPT
    cmd.transcript.CopyFrom(transcript)
    return cmd


def read_file(filepath):
    with open(filepath, "r") as in_f:
        return in_f.read()


def parse_transcript(transcript_contents):
    transcript = Parse(transcript_contents, Transcript())
    return transcript


def parse_transport_options(cfg, transport, role):

    opts = dict()

    for k, v in cfg[transport].items():
        m = re.match(f"{role}_(.*)", k)
        if m is not None:
            opts[m.group(1)] = v

    return opts
