use libp2p_core::transport::Transport;
use libp2p_core::{identity, upgrade, Multiaddr, PeerId};
use libp2p_secio::{SecioConfig, SecioMiddleware};

use async_std::future::Future;
use async_std::net::{TcpListener, TcpStream};
use async_std::prelude::*;
use async_std::task;
use std::thread;

use futures_util::sink::{Sink, SinkExt};
use futures_util::stream::{Stream, StreamExt};

// Client //////////////////////////////////////////////////////////////////////
pub struct SecioClient {
    server_hostname: String,
    server_port: u16,
    secio_middle_ware: Option<SecioMiddleware<TcpStream>>,
}

impl SecioClient {
    pub fn new(config: crate::interface::Config) -> SecioClient {
        SecioClient {
            server_hostname: config.server_hostname,
            server_port: config.server_port,
            secio_middle_ware: None,
        }
    }
}

impl crate::interface::ClientIFace for SecioClient {
    fn try_init(&mut self) -> bool {
        log::info!("Client init...");
        true
    }

    fn connect(&mut self) {
        async fn impl_(
            server_hostname: &String,
            server_port: u16,
        ) -> std::io::Result<SecioMiddleware<TcpStream>> {
            let endpoint = format!("{}:{}", server_hostname, server_port);
            log::info!("Connecting to endpoint {}.", endpoint);
            let stream = TcpStream::connect(endpoint).await?;

            let local_keys = identity::Keypair::generate_secp256k1();
            let config = SecioConfig::new(local_keys);
            let (mw, _, _) = SecioMiddleware::handshake(stream, config).await.unwrap();

            Ok(mw)
        }

        loop {
            match task::block_on(impl_(&self.server_hostname, self.server_port)) {
                Ok(secio_middle_ware) => {
                    self.secio_middle_ware = Some(secio_middle_ware);
                    break;
                }
                Err(_) => {
                    std::thread::sleep(std::time::Duration::from_secs(1));
                    log::info!("Connection failed. Retrying...");
                    continue;
                }
            }
        }
    }

    fn send(&mut self, nbytes: usize) {
        async fn impl_(mw: &mut SecioMiddleware<TcpStream>, nbytes: usize) -> std::io::Result<()> {
            let data = vec![0xFF; nbytes];
            let _ = mw.feed(data).await;
            let _ = mw.flush().await;
            Ok(())
        }

        let _ = task::block_on(impl_(self.secio_middle_ware.as_mut().unwrap(), nbytes));
    }

    fn recv(&mut self, nbytes: usize) {
        async fn impl_(mw: &mut SecioMiddleware<TcpStream>, nbytes: usize) -> std::io::Result<()> {
            let mut nbytes_recv_acc = 0;

            while nbytes_recv_acc < nbytes {
                let msg = mw.next().await.unwrap().unwrap();
                nbytes_recv_acc += msg.len();
                log::info!("Received {} bytes.", msg.len());
            }

            Ok(())
        }

        let _ = task::block_on(impl_(self.secio_middle_ware.as_mut().unwrap(), nbytes));
    }

    fn cleanup(&mut self) {
        let mw = std::mem::replace(&mut self.secio_middle_ware, None);
        let mut mw = mw.unwrap();
        let _ = mw.close();
    }
}

use crate::interface::ClientIFace;

impl SecioClient {
    pub fn into_interface_ffi(self) -> crate::ffi::ClientInterfaceFFI {
        fn try_init(p_this: *mut core::ffi::c_void) -> bool {
            let ref_this = unsafe { &mut *(p_this as *mut SecioClient) };
            ref_this.try_init()
        }

        fn connect(p_this: *mut core::ffi::c_void) {
            let ref_this = unsafe { &mut *(p_this as *mut SecioClient) };
            ref_this.connect()
        }

        fn send(p_this: *mut core::ffi::c_void, nbytes: usize) {
            let ref_this = unsafe { &mut *(p_this as *mut SecioClient) };
            ref_this.send(nbytes)
        }

        fn recv(p_this: *mut core::ffi::c_void, nbytes: usize) {
            let ref_this = unsafe { &mut *(p_this as *mut SecioClient) };
            ref_this.recv(nbytes)
        }

        fn cleanup(p_this: *mut core::ffi::c_void) {
            let ref_this = unsafe { &mut *(p_this as *mut SecioClient) };
            ref_this.cleanup()
        }

        fn free(p_this: *mut core::ffi::c_void) {
            let p_this = p_this as *mut SecioClient;
            let boxed = unsafe { Box::from_raw(p_this) };
            drop(boxed);
        }

        let p_this = Box::into_raw(Box::new(self)) as *mut core::ffi::c_void;

        crate::ffi::ClientInterfaceFFI {
            p_this,
            try_init,
            free,
            connect,
            send,
            recv,
            cleanup,
        }
    }
}

// Server //////////////////////////////////////////////////////////////////////
pub struct SecioServer {
    server_port: u16,
    listener: Option<TcpListener>,
    secio_middle_ware: Option<SecioMiddleware<TcpStream>>,
}

impl SecioServer {
    pub fn new(config: crate::interface::Config) -> SecioServer {
        SecioServer {
            server_port: config.server_port,
            listener: None,
            secio_middle_ware: None,
        }
    }
}

impl crate::interface::ServerIFace for SecioServer {
    fn try_init(&mut self) -> bool {
        log::info!("Server init...");
        true
    }

    fn listen(&mut self) {
        async fn impl_(server_port: u16) -> std::io::Result<TcpListener> {
            let endpoint = format!("localhost:{}", server_port);
            log::info!("Listening on endpoint {}.", endpoint);
            let listener = TcpListener::bind(endpoint).await?;
            Ok(listener)
        }

        self.listener = Some(task::block_on(impl_(self.server_port)).unwrap());
    }

    fn accept(&mut self) {
        async fn impl_(listener: &mut TcpListener) -> std::io::Result<SecioMiddleware<TcpStream>> {
            let local_keys = identity::Keypair::generate_secp256k1();
            let config = SecioConfig::new(local_keys);

            let (stream, _) = listener.accept().await?;
            let (mw, _, _) = SecioMiddleware::handshake(stream, config).await.unwrap();
            Ok(mw)
        }

        self.secio_middle_ware =
            Some(task::block_on(impl_(self.listener.as_mut().unwrap())).unwrap());
    }

    fn send(&mut self, nbytes: usize) {
        async fn impl_(mw: &mut SecioMiddleware<TcpStream>, nbytes: usize) -> std::io::Result<()> {
            let data = vec![0xFF; nbytes];
            let _ = mw.feed(data).await;
            let _ = mw.flush().await;
            Ok(())
        }

        let _ = task::block_on(impl_(self.secio_middle_ware.as_mut().unwrap(), nbytes));
    }

    fn recv(&mut self, nbytes: usize) {
        async fn impl_(mw: &mut SecioMiddleware<TcpStream>, nbytes: usize) -> std::io::Result<()> {
            let mut nbytes_recv_acc = 0;

            while nbytes_recv_acc < nbytes {
                let msg = mw.next().await.unwrap().unwrap();
                nbytes_recv_acc += msg.len();
                log::info!("Received {} bytes.", msg.len());
            }

            Ok(())
        }

        let _ = task::block_on(impl_(self.secio_middle_ware.as_mut().unwrap(), nbytes));
    }

    fn cleanup(&mut self) {
        let mw = std::mem::replace(&mut self.secio_middle_ware, None);
        let mut mw = mw.unwrap();
        let _ = mw.close();
    }
}

use crate::interface::ServerIFace;

impl SecioServer {
    pub fn into_interface_ffi(self) -> crate::ffi::ServerInterfaceFFI {
        fn try_init(p_this: *mut core::ffi::c_void) -> bool {
            let ref_this = unsafe { &mut *(p_this as *mut SecioServer) };
            ref_this.try_init()
        }

        fn listen(p_this: *mut core::ffi::c_void) {
            let ref_this = unsafe { &mut *(p_this as *mut SecioServer) };
            ref_this.listen()
        }

        fn accept(p_this: *mut core::ffi::c_void) {
            let ref_this = unsafe { &mut *(p_this as *mut SecioServer) };
            ref_this.accept()
        }

        fn send(p_this: *mut core::ffi::c_void, nbytes: usize) {
            let ref_this = unsafe { &mut *(p_this as *mut SecioServer) };
            ref_this.send(nbytes)
        }

        fn recv(p_this: *mut core::ffi::c_void, nbytes: usize) {
            let ref_this = unsafe { &mut *(p_this as *mut SecioServer) };
            ref_this.recv(nbytes)
        }

        fn cleanup(p_this: *mut core::ffi::c_void) {
            let ref_this = unsafe { &mut *(p_this as *mut SecioServer) };
            ref_this.cleanup()
        }

        fn free(p_this: *mut core::ffi::c_void) {
            let p_this = p_this as *mut SecioServer;
            let boxed = unsafe { Box::from_raw(p_this) };
            drop(boxed);
        }

        let p_this = Box::into_raw(Box::new(self)) as *mut core::ffi::c_void;

        crate::ffi::ServerInterfaceFFI {
            p_this,
            try_init,
            free,
            listen,
            accept,
            send,
            recv,
            cleanup,
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_secio() {
        let cfg = crate::interface::Config {
            client_hostname: "localhost".to_string(),
            server_hostname: "localhost".to_string(),
            client_port: 50000,
            server_port: 8080,
            snow_noise_params: None,
        };

        let mut client = SecioClient::new(cfg.clone());
        let mut server = SecioServer::new(cfg);

        client.try_init();
        server.try_init();

        let server_thread = std::thread::spawn(move || {
            server.listen();
            server.accept();
            server.recv(100);
            server.send(100);
            server.cleanup();
        });

        std::thread::sleep(std::time::Duration::from_millis(100));

        let client_thread = std::thread::spawn(move || {
            client.connect();
            std::thread::sleep(std::time::Duration::from_millis(100));
            client.send(100);
            client.recv(100);
            client.cleanup();
        });

        let _ = client_thread.join();
        let _ = server_thread.join();
    }
}
