% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cross_validate_cube.R
\name{cross_validate_cube}
\alias{cross_validate_cube}
\title{Leave-one-out cross-validation for data cubes}
\usage{
cross_validate_cube(
  data_cube,
  fun,
  ...,
  grouping_var,
  out_var = "taxonKey",
  crossv_method = c("loo", "kfold"),
  k = ifelse(crossv_method == "kfold", 5, NA),
  max_out_cats = 1000,
  progress = FALSE
)
}
\arguments{
\item{data_cube}{A data cube object (class 'processed_cube' or 'sim_cube',
see \code{b3gbi::process_cube()}) or a dataframe (from \verb{$data} slot of
'processed_cube' or 'sim_cube'). To limit runtime, we recommend using a
dataframe with custom function as \code{fun}.}

\item{fun}{A function which, when applied to \code{data_cube} returns the
statistic(s) of interest. This function must return a dataframe with a column
\code{diversity_val} containing the statistic of interest.}

\item{...}{Additional arguments passed on to \code{fun}.}

\item{grouping_var}{A character vector specifying the grouping variable(s)
for \code{fun}. The output of \code{fun(data_cube)} returns a row per group.}

\item{out_var}{A string specifying the column by which the data should be
left out iteratively. Default is \code{"taxonKey"} which can be used for
leave-one-species-out CV.}

\item{crossv_method}{Method of data partitioning.
If \code{crossv_method = "loo"} (default),
\verb{S = number of unique values in out_var} training partitions are created
containing \code{S - 1} rows each.
If \code{crossv_method = "kfold"}, the aggregated data is split the data into
\code{k} exclusive partitions containing \code{S / k} rows each. K-fold CV is
experimental and results should be interpreted with caution.}

\item{k}{Number of folds (an integer). Used only if
\code{crossv_method = "kfold"}. Default 5.}

\item{max_out_cats}{An integer specifying the maximum number of unique
categories in \code{out_var} to leave out iteratively. Default is \code{1000}.
This can be increased if needed, but keep in mind that a high number of
categories in \code{out_var} may significantly increase runtime.}

\item{progress}{Logical. Whether to show a progress bar. Set to \code{TRUE} to
display a progress bar, \code{FALSE} (default) to suppress it.}
}
\value{
A dataframe containing the cross-validation results with the
following columns:
\itemize{
\item Cross-Validation id (\code{id_cv})
\item The grouping variable \code{grouping_var} (e.g., year)
\item The category left out during each cross-validation iteration
(specified \code{out_var} with suffix '_out' in lower case)
\item The computed statistic values for both training (\code{rep_cv}) and true
datasets (\code{est_original})
\item Error metrics: error (\code{error}), squared error (\code{sq_error}),
absolute difference (\code{abs_error}), relative difference (\code{rel_error}), and
percent difference (\code{perc_error})
\item Error metrics summarised by \code{grouping_var}: mean relative difference
(\code{mre}), mean squared error (\code{mse}) and root mean squared error (\code{rmse})
}

See Details section on how these error metrics are calculated.
}
\description{
This function performs leave-one-out (LOO) or k-fold (experimental)
cross-validation (CV) on a biodiversity data cube to assess the performance
of a specified indicator function. It partitions the data by a specified
variable, calculates the specified indicator on training data, and compares
it with the true values to evaluate the influence of one or more categories
on the final result.
}
\details{
This function assesses the influence of each category in \code{out_var} on the
indicator value by iteratively leaving out one category at a time, similar to
leave-one-out cross-validation. K-fold CV works in a similar fashion but is
experimental and will not be covered here.
\enumerate{
\item \strong{Original Sample Data}:
\eqn{\mathbf{X} = \{X_{11}, X_{12}, X_{13}, \ldots, X_{sn}\}}
\itemize{
\item The initial set of data points, where there are \eqn{s}
different categories in \code{out_var} and \eqn{n} total samples across all
categories (= the sample size). \eqn{n} corresponds to the number of cells
in a data cube or the number of rows in tabular format.
}
\item \strong{Statistic of Interest}: \eqn{\theta}
\itemize{
\item The parameter or statistic being estimated, such as the mean
\eqn{\bar{X}}, variance \eqn{\sigma^2}, or a biodiversity indicator. Let
\eqn{\hat{\theta}} denote the estimated value of \eqn{\theta} calculated
from the complete dataset \eqn{\mathbf{X}}.
}
\item \strong{Cross-Validation (CV) Sample}: \eqn{\mathbf{X}_{-s_j}}
\itemize{
\item The full dataset \eqn{\mathbf{X}} excluding all samples belonging to
category \eqn{j}. This subset is used to investigate the influence of
category \eqn{j} on the estimated statistic \eqn{\hat{\theta}}.
}
\item \strong{CV Estimate for Category} \eqn{\mathbf{j}}: \eqn{\hat{\theta}_{-s_j}}
\itemize{
\item The value of the statistic of interest calculated from
\eqn{\mathbf{X}_{-s_j}}, which excludes category \eqn{j}. For example, if
\eqn{\theta} is the sample mean,
\eqn{\hat{\theta}_{-s_j} = \bar{X}_{-s_j}}.
}
\item \strong{Error Measures}:
\itemize{
\item The \strong{Error} is the difference between the statistic estimated without
category \eqn{j} (\eqn{\hat{\theta}_{-s_j}}) and the statistic calculated
on the complete dataset (\eqn{\hat{\theta}}).
}

\deqn{\text{Error}_{s_j} = \hat{\theta}_{-s_j} - \hat{\theta}}
\itemize{
\item The \strong{Relative Error} is the absolute error, normalised by the true
estimate \eqn{\hat{\theta}} and a small error term
\eqn{\epsilon = 10^{-8}} to avoid division by zero.
}

\deqn{\text{Rel. Error}_{s_j} = \frac{|\hat{\theta}_{-s_j} - \hat{\theta}|}{\hat{\theta} +\epsilon}}
\itemize{
\item The \strong{Percent Error} is the relative error expressed as a percentage.
}

\deqn{\text{Perc. Error}_{s_j} = \text{Rel. Error}_{s_j} \times 100 \%}
\item \strong{Summary Measures}:
\itemize{
\item The \strong{Mean Relative Error (MRE)} is the average of the relative errors
over all categories.
}

\deqn{\text{MRE} = \frac{1}{s} \sum_{j=1}^s \text{Rel. Error}_{s_j}}
\itemize{
\item The \strong{Mean Squared Error (MSE)} is the average of the squared errors.
}

\deqn{\text{MSE} = \frac{1}{s} \sum_{j=1}^s (\text{Error}_{s_j})^2}
\itemize{
\item The \strong{Root Mean Squared Error (RMSE)} is the square root of the MSE.
}

\deqn{\text{RMSE} = \sqrt{\text{MSE}}}
}
}
\examples{
\dontrun{
# After processing a data cube with b3gbi::process_cube()

# Function to calculate statistic of interest
# Mean observations per year
mean_obs <- function(data) {
  out_df <- aggregate(obs ~ year, data, mean) # Calculate mean obs per year
  names(out_df) <- c("year", "diversity_val") # Rename columns
  return(out_df)
}
mean_obs(processed_cube$data)

# Perform leave-one-species-out CV
cv_mean_obs <- cross_validate_cube(
  data_cube = processed_cube$data,
  fun = mean_obs,
  grouping_var = "year",
  out_var = "taxonKey",
  crossv_method = "loo",
  progress = FALSE
)
head(cv_mean_obs)
}
}
\concept{data_exploration}
