# DanLing
# Copyright (C) 2022-Present  DanLing

# This file is part of DanLing.

# DanLing is free software: you can redistribute it and/or modify
# it under the terms of the following licenses:
# - The Unlicense
# - GNU Affero General Public License v3.0 or later
# - GNU General Public License v2.0 or later
# - BSD 4-Clause "Original" or "Old" License
# - MIT License
# - Apache License 2.0

# DanLing is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the LICENSE file for more details.

from __future__ import annotations

import os
import sys
from collections.abc import Mapping
from contextlib import suppress
from datetime import datetime
from enum import auto
from functools import wraps
from math import isnan
from typing import Any
from warnings import warn

from danling.utils import base62

try:
    from enum import StrEnum  # type: ignore[attr-defined]
except ImportError:
    from strenum import LowercaseStrEnum as StrEnum  # type: ignore[no-redef]


class RunnerMeta(type):
    def __call__(cls, *args: Any, **kwargs: Any) -> Any:
        instance = super().__call__(*args, **kwargs)
        instance.__post_init__()
        return instance


class RunnerMode(StrEnum):  # pylint: disable=too-few-public-methods
    r"""
    `RunnerMode` is an enumeration of running modes.

    Attributes:
        train: Training mode.
        evaluate: Evaluation mode.
        infer: Inference mode.
    """

    train = auto()
    evaluate = auto()
    infer = auto()


def get_time_str() -> str:
    time = datetime.now()
    time_tuple = time.isocalendar()[1:] + (
        time.hour,
        time.minute,
        time.second,
        time.microsecond,
    )
    return "".join(base62.encode(i) for i in time_tuple)


def get_git_hash() -> str | None:
    try:
        from git.exc import InvalidGitRepositoryError
        from git.repo import Repo

        try:
            return Repo(search_parent_directories=True).head.object.hexsha
        except ImportError:
            pass  # handle at last
        except (InvalidGitRepositoryError, ValueError):
            path = os.path.dirname(os.path.abspath(sys.argv[0]))
            with suppress(InvalidGitRepositoryError, ValueError):
                hexsha = Repo(path=path, search_parent_directories=True).head.object.hexsha
                warn(
                    "Unable to get git hash from CWD, fallback to git hash of top-level code environment.",
                    category=RuntimeWarning,
                    stacklevel=2,
                )
                return hexsha
    except ImportError:
        warn(
            "GitPython is not installed, unable to fetch git hash",
            category=RuntimeWarning,
            stacklevel=2,
        )
    return None


def on_main_process(func):
    """
    Decorator to run func only on main process.
    """

    @wraps(func)
    def wrapper(self, *args, **kwargs) -> Any | None:
        if self.is_main_process or not self.distributed:
            return func(self, *args, **kwargs)
        return None

    return wrapper


def on_local_main_process(func):
    """
    Decorator to run func only on local main process.
    """

    @wraps(func)
    def wrapper(self, *args, **kwargs) -> Any | None:
        if self.is_local_main_process or not self.distributed:
            return func(self, *args, **kwargs)
        return None

    return wrapper


def format_result(result, format_spec: str = ".4f", depth: int = 0) -> str:
    longest_key = max(len(k) for k in result.keys())
    repr_list = [_format_result(result, format_spec)]
    for k, v in result.items():
        if isinstance(v, Mapping):
            initials = " " * (longest_key - len(k)) + "\t" * depth
            repr_list.append(f"{initials}{k}: {format_result(v, format_spec, depth + 1)}")
    return "\n".join(repr_list)


def _format_result(result, format_spec: str = ".4f") -> str:
    repr_str = ""
    for k, v in result.items():
        if isinstance(v, (Mapping,)):
            continue
        padding = 1
        if isinstance(v, (float,)):
            is_negative = v < 0 if not isnan(v) else False
            v = format(v, format_spec) if not isnan(v) else "  NaN  "
            padding = padding if is_negative else padding + 1
        repr_str += f"\t{k}:{' ' * padding}{v}"
    return repr_str
