Require Import Basic.
Require Import ImpDefs Trace SecurityDefs.
Require Import Tactics.
Require Import BasicTheories ImpTheories TraceTheories.

From Coq Require Import Equality List Relations RelationClasses Ensembles.

Import ListNotations.

Module Type SecurityTheories (B : Basic) (ID : ImpDefs) (TD : TraceDefs B ID) (SD : SecurityDefs B ID TD)
    (Tac : SimpleTactics B ID TD SD) (BT : BasicTheories B) (IT : ImpTheories B ID TD SD Tac BT) (TT : TraceTheories B ID TD SD Tac BT IT).
  Import B ID TD SD Tac BT IT TT.

  Section LowSets.

    Lemma high_set_up_closed : forall D `{LowSet D} l1 l2, ~ In Label D l1 -> flows_to l1 l2 -> ~ In Label D l2.
      intros ? ? ? ? l2 l1NotInD ? ?. contradict l1NotInD. eauto using down_closed.
    Qed.

    #[global] Instance dec_in_union_inst D0 D1 `{DecideIn Label D0} `{DecideIn Label D1} : DecideIn (Union Label D0 D1).
      split ; intro a.
      destruct (H.(dec_in) a) ; [| destruct (H0.(dec_in) a)] ; auto using Union.
      right ; intro InUnion ; destruct InUnion ; exfalso ; auto.
    Defined.

    #[global] Instance low_set_union_inst D0 D1 `{LowSet D0} `{LowSet D1} : LowSet (Union Label D0 D1).
      split.
      intros ? ? InUnionl1 ?.
      destruct InUnionl1 ; [apply Union_introl | apply Union_intror] ; eapply down_closed ; eauto.
    Defined.

    Proposition attacker_symmetry : forall (P T : Ensemble Label) `{LowSet P} `{LowSet T}, Attacker P T -> Attacker T P.
      intros ? ? ? ? ? ? AtkPT. split ; intros l ?.
      assert (In Label (Union Label P T) l) as [|] ; auto using Union, AtkPT.(non_compromised_low).
    Qed.

  End LowSets.

  Import ImpNotations.

  Open Scope imp_scope.

  Section BasicProperties.

    Variable G : Varname -> option Label.
    Variable D : Ensemble Label.
    Notation deq_evt := (deq_evt G D).
    Notation deq_evt_lst := (deq_evt_lst G D).
    Notation deq_store := (deq_store G D).

    Lemma deq_evt_refl : forall evt, deq_evt evt evt.
      constructor.
    Qed.

    Lemma deq_evt_symm : forall evt0 evt1, deq_evt evt0 evt1 -> deq_evt evt1 evt0.
      intros evt0 evt1 deq. induction deq ; econstructor ; eauto.
    Qed.

    Lemma deq_store_refl : forall s, deq_store s s.
      unfold deq_store. reflexivity.
    Qed.

    Lemma deq_store_symm : forall s1 s2, deq_store s1 s2 -> deq_store s2 s1.
      unfold deq_store. intros. symmetry. eauto.
    Qed.

    #[global] Instance deq_store_eq_inst : Equivalence deq_store.
      split ; unfold Reflexive, Symmetric, Transitive ; try solve [apply deq_store_refl | apply deq_store_symm].
      unfold deq_store. intros ? ? ? SEq01 SEq12 x0 l InG InD.
      rewrite -> (SEq01 x0 l InG InD) ; eauto.
    Defined.

    Lemma deq_evt_lst_refl : forall lst, deq_evt_lst lst lst.
      induction lst ; constructor ; assumption.
    Qed.

    Lemma deq_evt_lst_symm : forall lst0 lst1, deq_evt_lst lst0 lst1 -> deq_evt_lst lst1 lst0.
      intros ? ? DEq. induction DEq ; constructor ; auto.
    Qed.

    #[global] Instance deq_evt_lst_eq_inst : Equivalence deq_evt_lst.
      split ; unfold Reflexive, Symmetric, Transitive ; try solve [apply deq_evt_lst_refl | apply deq_evt_lst_symm].
      intros ? ? z DEq01. revert z.
      induction DEq01 ; intros ? DEq12 ; dependent induction DEq12
      ; eauto using DEqEmpty, DEqSame, DEqNoEvtL, DEqNoEvtR.
    Defined.

    Lemma deq_pfx_refl : forall pfx, pfx =[G, D] pfx.
      intros [] ; constructor ; reflexivity.
    Qed.

    Lemma deq_pfx_symm : forall pfx0 pfx1, pfx0 =[G, D] pfx1 -> pfx1 =[G, D] pfx0.
      intros ? ? DEq. induction DEq ; constructor ; symmetry ; assumption.
    Qed.

    #[global] Instance deq_pfx_eq_inst : Equivalence (deq_pfx G D).
      split ; unfold Reflexive, Symmetric, Transitive ; try solve [apply deq_pfx_refl | apply deq_pfx_symm].
      intros [] [s1 lst1] [] DEq01 DEq12. dependent destruction DEq01. dependent destruction DEq12.
      constructor ; [transitivity s1 | transitivity lst1] ; assumption.
    Defined.

    Lemma dle_pfx_refl : forall pfx, pfx <=[G, D] pfx.
      unfold dle_pfx. eauto using deq_pfx_refl.
    Qed.

  End BasicProperties.

  #[global] Hint Resolve deq_evt_refl : deq.
  #[global] Hint Resolve deq_evt_symm : deq.
  #[global] Hint Resolve deq_store_refl : deq.
  #[global] Hint Resolve deq_store_symm : deq.
  #[global] Hint Resolve DEqEmpty : deq.
  #[global] Hint Resolve DEqSame : deq.
  #[global] Hint Resolve deq_evt_lst_refl : deq.
  #[global] Hint Resolve deq_evt_lst_symm : deq.
  #[global] Hint Resolve DEqPfx_intro : deq.
  #[global] Hint Resolve deq_pfx_refl : deq.
  #[global] Hint Resolve deq_pfx_symm : deq.
  #[global] Hint Resolve dle_pfx_refl : deq.

  Section DEqLists.

    Variable G : Varname -> option Label.
    Variable D : Ensemble Label.
    Notation deq_evt := (deq_evt G D).
    Notation deq_evt_lst := (deq_evt_lst G D).
    Notation deq_store := (deq_store G D).

    #[local] Hint Resolve DEqEmpty : deq.
    #[local] Hint Resolve DEqSame : deq.
    #[local] Hint Resolve DEqNoEvtL : deq.
    #[local] Hint Resolve DEqNoEvtR : deq.

    Lemma deq_evt_lst_append_same : forall a lst0 lst1,
        deq_evt_lst lst0 lst1 -> deq_evt_lst (lst0 ++ [a]) (lst1 ++ [a]).
      intros a lst0 lst1 DEq. dependent induction DEq ; simpl ; auto with deq.
    Qed.

    Lemma deq_evt_lst_append_high_l : forall a lst0 lst1,
        deq_evt_lst lst0 lst1 -> deq_evt a NoEvt -> deq_evt_lst (lst0 ++ [a]) lst1.
      intros a lst0 lst1 DEq HighA. dependent induction DEq ; simpl ; auto with deq.
    Qed.

    Lemma deq_evt_lst_append_high_r : forall a lst0 lst1,
        deq_evt_lst lst0 lst1 -> deq_evt a NoEvt -> deq_evt_lst lst0 (lst1 ++ [a]).
      intros a lst0 lst1 DEq HighA. dependent induction DEq ; simpl ; auto with deq.
    Qed.

    Inductive deq_evt_lst_rev : relation (list Event) :=
      | DEqRevEmpty : deq_evt_lst_rev [] []
      | DEqRevSame : forall a lst1 lst2, deq_evt_lst_rev lst1 lst2 -> deq_evt_lst_rev (lst1 ++ [a]) (lst2 ++ [a])
      | DEqRevNoEvtL : forall a lst1 lst2, deq_evt a NoEvt -> deq_evt_lst_rev lst1 lst2 -> deq_evt_lst_rev (lst1 ++ [a]) lst2
      | DEqRevNoEvtR : forall a lst1 lst2, deq_evt a NoEvt -> deq_evt_lst_rev lst1 lst2 -> deq_evt_lst_rev lst1 (lst2 ++ [a]).

    Lemma deq_evt_lst_rev_refl : forall lst, deq_evt_lst_rev lst lst.
      induction lst using rev_ind ; auto using deq_evt_lst_rev.
    Qed.

    Lemma deq_evt_lst_rev_prepend_same : forall a lst0 lst1,
        deq_evt_lst_rev lst0 lst1 -> deq_evt_lst_rev (a :: lst0) (a :: lst1).
      intros a lst0 lst1 DEq. dependent induction DEq ; repeat rewrite -> app_comm_cons ; auto using deq_evt_lst_rev_refl, deq_evt_lst_rev.
    Qed.

    Lemma deq_evt_lst_rev_prepend_high_l : forall a lst0 lst1,
        deq_evt_lst_rev lst0 lst1 -> deq_evt a NoEvt -> deq_evt_lst_rev (a :: lst0) lst1.
      intros a lst0 lst1 DEq HighA. dependent induction DEq ; repeat rewrite -> app_comm_cons ; auto using deq_evt_lst_rev.
      rewrite <- app_nil_l at 1. auto using deq_evt_lst_rev.
    Qed.

    Lemma deq_evt_lst_rev_prepend_high_r : forall a lst0 lst1,
        deq_evt_lst_rev lst0 lst1 -> deq_evt a NoEvt -> deq_evt_lst_rev lst0 (a :: lst1).
      intros a lst0 lst1 DEq HighA. dependent induction DEq ; repeat rewrite -> app_comm_cons ; auto using deq_evt_lst_rev.
      rewrite <- app_nil_l. auto using deq_evt_lst_rev.
    Qed.

    Lemma deq_lst_iff_rev : forall lst0 lst1, deq_evt_lst lst0 lst1 <-> deq_evt_lst_rev lst0 lst1.
      intros lst0 lst1. split ; intros DEq.
      * induction DEq ; auto using DEqRevEmpty, deq_evt_lst_rev_prepend_same, deq_evt_lst_rev_prepend_high_l, deq_evt_lst_rev_prepend_high_r with deq.
      * induction DEq ; auto using deq_evt_lst_append_same, deq_evt_lst_append_high_l, deq_evt_lst_append_high_r with deq.
    Qed.

    Lemma deq_lst_app_eq_nil : forall lst0 lst1, deq_evt_lst (lst0 ++ lst1) [] -> deq_evt_lst lst0 [] /\ deq_evt_lst lst1 [].
      induction lst0 ; intros lst1 DEqApp.
      * simpl in *. auto with deq.
      * simpl in DEqApp. inversion DEqApp as [| | ? ? ? ? DEqApp' |] ; subst.
        apply IHlst0 in DEqApp'. destruct DEqApp'. auto with deq.
    Qed.

    Lemma deq_cons_l_inv : forall lst0 lst1 a, deq_evt a NoEvt -> deq_evt_lst (a :: lst0) lst1 -> deq_evt_lst lst0 lst1.
      intros ? ? ? ? DEqLst. dependent induction DEqLst ; eauto using DEqNoEvtR.
    Qed.

    Lemma deq_cons_r_inv : forall lst0 lst1 a, deq_evt a NoEvt -> deq_evt_lst lst0 (a :: lst1) -> deq_evt_lst lst0 lst1.
      intros ? ? ? ? DEqLst. dependent induction DEqLst ; eauto using DEqNoEvtL.
    Qed.

    Lemma deq_cons_same_inv : forall lst0 lst1 a, deq_evt_lst (a :: lst0) (a :: lst1) -> deq_evt_lst lst0 lst1.
      intros ? ? ? DEqLst. dependent induction DEqLst ; eauto using deq_cons_l_inv, deq_cons_r_inv.
    Qed.

    Lemma deq_app_l_inv : forall lst0 lst1 a, deq_evt a NoEvt -> deq_evt_lst (lst0 ++ [a]) lst1 -> deq_evt_lst lst0 lst1.
      intros ? ? ? ? DEqLst. apply deq_lst_iff_rev in DEqLst ; dependent induction DEqLst
      ; try lazymatch goal with
        | [H : _ ++ _ :: _ = nil |- _] => symmetry in H ; apply app_cons_not_nil in H ; inversion H
        | [H : nil = _ ++ _ :: _ |- _] => apply app_cons_not_nil in H ; inversion H
        | [H : _ ++ _ :: nil = _ ++ _ :: nil |- _] => apply app_inj_tail in H ; destruct H ; subst
      end ; apply deq_lst_iff_rev in DEqLst
      ; eauto using deq_evt_lst_append_high_r.
    Qed.

    Lemma deq_app_r_inv : forall lst0 lst1 a, deq_evt a NoEvt -> deq_evt_lst lst0 (lst1 ++ [a]) -> deq_evt_lst lst0 lst1.
      intros ? ? ? ? DEqLst. apply deq_lst_iff_rev in DEqLst ; dependent induction DEqLst
      ; try lazymatch goal with
        | [H : _ ++ _ :: _ = nil |- _] => symmetry in H ; apply app_cons_not_nil in H ; inversion H
        | [H : nil = _ ++ _ :: _ |- _] => apply app_cons_not_nil in H ; inversion H
        | [H : _ ++ _ :: nil = _ ++ _ :: nil |- _] => apply app_inj_tail in H ; destruct H ; subst
      end ; apply deq_lst_iff_rev in DEqLst
      ; eauto using deq_evt_lst_append_high_l.
    Qed.

    Lemma deq_app_same_inv : forall lst0 lst1 a, deq_evt_lst (lst0 ++ [a]) (lst1 ++ [a]) -> deq_evt_lst lst0 lst1.
      intros ? ? ? DEqLst. apply deq_lst_iff_rev in DEqLst ; dependent induction DEqLst
      ; repeat lazymatch goal with
        | [H : _ ++ _ :: _ = nil |- _] => symmetry in H ; apply app_cons_not_nil in H ; inversion H
        | [H : nil = _ ++ _ :: _ |- _] => apply app_cons_not_nil in H ; inversion H
        | [H : _ ++ _ :: nil = _ ++ _ :: nil |- _] => apply app_inj_tail in H ; destruct H ; subst
      end ; apply deq_lst_iff_rev in DEqLst
      ; eauto using deq_app_l_inv, deq_app_r_inv.
    Qed.

    Lemma deq_app : forall lst0 lst0' lst1 lst1', deq_evt_lst lst0 lst1 -> deq_evt_lst lst0' lst1'
        -> deq_evt_lst (lst0 ++ lst0') (lst1 ++ lst1').
      intros ? ? ? ? DEq. induction DEq ; intros DEq' ; simpl in * ; try constructor ; auto.
    Qed.

    Lemma deq_app_nil_l : forall lst0 lst1 lst2, deq_evt_lst lst0 (lst1 ++ lst2) -> deq_evt_lst lst1 [] -> deq_evt_lst lst0 lst2.
      intros ? ? ? DEqApp DEqNil. dependent induction DEqNil ; simpl in * ; eauto using deq_cons_r_inv.
    Qed.

    Lemma deq_app_nil_r : forall lst0 lst1 lst2, deq_evt_lst lst0 (lst1 ++ lst2) -> deq_evt_lst lst2 [] -> deq_evt_lst lst0 lst1.
      intros ? ? ? DEqApp DEqNil. dependent induction DEqApp
      ; repeat lazymatch goal with
        | [H : [] = _ ++ _ |- _] => symmetry in H ; apply app_eq_nil in H ; destruct H ; subst
        | [H : _ :: _ = ?lst ++ _ |- _] => induction lst ; simpl in * ; subst
        | [H : _ :: _ = _ :: _ |- _] => injection H ; intros ; subst ; clear H
        | [H : deq_evt_lst (_ :: _) [] |- _] => inversion H ; subst ; clear H
        | [H: deq_evt ?a NoEvt |- deq_evt_lst (?a :: _) _] => apply DEqNoEvtL
        | [H: deq_evt ?a NoEvt |- deq_evt_lst _ (?a :: _)] => apply DEqNoEvtR
        | [H0 : deq_evt_lst ?lst0 ?lst1, H1 : deq_evt_lst ?lst1 [] |- deq_evt_lst ?lst0 []] => transitivity lst1 ; assumption
      end ; try constructor ; eauto.
    Qed.

    Lemma deq_app_inv : forall lst0 lst1 lst0' lst1', deq_evt_lst lst0 lst1
        -> deq_evt_lst (lst0 ++ lst0') (lst1 ++ lst1')
        -> deq_evt_lst lst0' lst1'.
      intros ? ? ? ? DEq DEqApp. induction DEq ; simpl in * ; eauto using deq_cons_l_inv, deq_cons_r_inv, deq_cons_same_inv.
    Qed.

    Lemma deq_app_opt_inv : forall lst0 lst1 lst2, deq_evt_lst lst0 (lst1 ++ lst2)
        -> exists lst0', Prefix lst0' lst0 /\ deq_evt_lst lst0' lst1.
      intros ? ? ? DEqApp. dependent induction DEqApp
      ; repeat lazymatch goal with
        | [H : [] = _ ++ _ |- _] => symmetry in H ; apply app_eq_nil in H ; destruct H ; subst
        | [H : _ :: _ = ?lst ++ _ |- _] => induction lst ; simpl in *
        | [H : _ :: _ = _ :: _ |- _] => injection H ; intros ; subst ; clear H
        | [IH : forall lst2 lst3, ?lst0 ++ ?lst1 = lst2 ++ lst3 -> _ |- _]
          => pose proof (IH lst0 lst1 eq_refl) as (lst0' & ? & ?) ; clear IH
      end ; eauto using DEqEmpty, DEqSame, DEqNoEvtL, DEqNoEvtR.
    Qed.

    Lemma deq_mstep_stop : forall c s cs lst0, (c, s) ==>*[lst0] cs
        -> forall lst1, deq_evt_lst lst0 (lst1 ++ [StopEvt])
        -> exists lst0', lst0 = lst0' ++ [StopEvt].
      intros c s cs lst0 MStep. dependent induction MStep using mstep_rev_ind
      ; intros lst1 DEq ; invert_tail deq_lst_iff_rev DEq.
      assert (exists lst0, lst = lst0 ++ [StopEvt]) as [lst0 ?]
        by (apply IHMStep with c s lst1 ; auto) ; subst.
      destruct cs1 as [c1 ?].
      assert (c1 = Stop) by eauto using mstep_to_stop_evt ; subst.
      handle_simple_contradict.
    Qed.

    Lemma step_noevt_store_eq : forall c s c' s' a, (c, s) -->[a] (c', s') -> deq_evt a NoEvt -> deq_store s s'.
      intros ? ? ? ? ? Step DEqNil. dependent induction Step ; try handle_simple_contradict ; eauto with deq.
      inversion DEqNil as [| ? ? ? Gx | | | | | | |] ; subst.
      unfold deq_store ; intros x' ? Gx' ?.
      destruct (eq_dec x x') ; subst ; [| reflexivity].
      rewrite -> Gx' in Gx ; injection Gx ; intros ; subst ; exfalso ; auto.
    Qed.

    Lemma mstep_noevt_store_eq : forall c s c' s' lst, (c, s) ==>*[lst] (c', s') -> deq_evt_lst lst [] -> deq_store s s'.
      intros c ? c' ? ? MStep DEqNil ; dependent induction MStep ; eauto with deq.
      destruct cs1 as [c1 s1].
      inversion DEqNil ; subst.
      transitivity s1 ; eauto using step_noevt_store_eq.
    Qed.

    Lemma mstep_deq_stop : forall lst1 lst0 cs0 cs1, deq_evt_lst (lst0 ++ [StopEvt]) lst1 -> cs0 ==>*[lst1] cs1
        -> exists lst1', lst1 = lst1' ++ [StopEvt].
      intros lst1 lst0 cs0 cs1 DEqLst MStep.
      dependent induction MStep using mstep_rev_ind ;
      invert_tail deq_lst_iff_rev DEqLst.
      assert (exists lst', lst = lst' ++ [StopEvt]) as [lst' ?] by eauto ; subst.
      destruct cs1 as [c1 ?].
      assert (c1 = Stop) by eauto using mstep_to_stop_evt ; subst.
      handle_simple_contradict.
    Qed.

    Lemma nil_step_store_eq : forall c s a c' s', (c, s) -->[a] (c', s') -> deq_evt a NoEvt -> deq_store s s'.
      intros c s a c' s' Step DEqANone.
      dependent induction Step ; eauto with deq.
      inversion DEqANone as [| ? ? ? Gx | | | | | | |] ; subst.
      unfold deq_store.
      intros x' ? Gx' ?.
      destruct (eq_dec x x') ; [subst | reflexivity].
      rewrite -> Gx' in Gx ; injection Gx ; intros ; subst ; exfalso ; auto.
    Qed.

    Lemma nil_mstep_store_eq : forall c s lst c' s', (c, s) ==>*[lst] (c', s') -> deq_evt_lst lst [] -> deq_store s s'.
      intros c s lst c' s' MStep DEqNil.
      dependent induction MStep ; auto with deq.
      inversion DEqNil as [| | ? ? ? DEqANone DEqLstNil |] ; subst.
      destruct cs1 as [c1 s1].
      transitivity s1 ; eauto using nil_step_store_eq.
    Qed.

    Lemma same_step_same_state : forall a c0 s0 c0' s0', (c0, s0) -->[a] (c0', s0')
        -> forall c1 s1 c1' s1', (c1, s1) -->[a] (c1', s1')
        -> deq_store s0 s1
        -> deq_store s0' s1'.
      intros a c0 s0 c0' s0' Step0 c1 s1 c1' s1' Step1 DEqStore.
      dependent induction Step0 ; eauto with deq
      ; try lazymatch type of Step1 with
        | _ -->[NoEvt] _ => transitivity s1 ; eauto using nil_step_store_eq with deq
      end.
      * assert (c1' = Stop) by eauto using step_stop_evt_impl_stop ; subst.
        inversion Step1 ; subst ; assumption.
      * dependent induction Step1 ; eauto.
        unfold deq_store in *.
        intros x' m l'.
        destruct (eq_dec x x') ; [reflexivity | intro ; eauto].
      * dependent induction Step1 ; eauto.
    Qed.

    Proposition deq_traces_stores : forall c0 s0 lst0 c0' s0', (c0, s0) ==>*[lst0] (c0', s0')
        -> forall c1 s1 lst1 c1' s1', (c1, s1) ==>*[lst1] (c1', s1')
        -> (s0, lst0) =[G, D] (s1, lst1)
        -> deq_store s0' s1'.
      intros c0 s0 lst0 c0' s0' MStep0 c1 s1 lst1 c1' s1' MStep1 DEqPfx.
      dependent induction MStep0 ; inversion DEqPfx as [? ? ? ? ? DEqEvtLst] ; subst.
      * transitivity s1 ; eauto using nil_mstep_store_eq with deq.
      * destruct cs1 as [c0'' s0''].
        dependent induction MStep1.
        - transitivity s0 ; eauto using nil_mstep_store_eq, MultiStep_some with deq.
        - destruct cs1 as [c1'' s1''].
          inversion DEqEvtLst ; subst.
          + apply IHMStep0 with c0'' s0'' c0' c1'' s1'' lst0 c1' ; eauto using DEqPfx_intro, same_step_same_state.
          + apply IHMStep0 with c0'' s0'' c0' c1 s1 (a0 :: lst0) c1' ; eauto using MultiStep_some.
            apply DEqPfx_intro ; [transitivity s0 |] ; eauto using nil_step_store_eq with deq.
          + assert (deq_store s0 s1'') by (transitivity s1 ; eauto using nil_step_store_eq).
            apply IHMStep1 with c1'' s1'' c1' ; auto using DEqPfx_intro.
    Qed.

  End DEqLists.

  Section ListErasure.

    Context (G : Varname -> option Label) (D : Ensemble Label) `{DecideIn Label D}.

    Lemma deq_evt_dec a : {deq_evt G D a NoEvt} + {~ deq_evt G D a NoEvt}.
      destruct a 
      ; try destruct (G x) eqn:Gx0
      ; try destruct (dec_in l)
      ; try solve [left ; eauto using deq_evt]
      ; right ; intro DEq ; inversion DEq as [| ? ? ? Gx | | | | | | |] ; eauto
      ; rewrite -> Gx0 in Gx ; solve [discriminate | injection Gx ; intros ; subst ; auto].
    Qed.

    Fixpoint erase_evt_lst (lst : list Event) : list Event :=
      match lst with
        | nil => nil
        | a :: rst => if (deq_evt_dec a)
                      then (erase_evt_lst rst)
                      else a :: (erase_evt_lst rst)
      end.

    Lemma erase_nil : erase_evt_lst [] = [].
      simpl. reflexivity.
    Qed.

    Lemma erase_no_add : forall lst a, List.In a (erase_evt_lst lst) -> List.In a lst.
      induction lst as [| a' lst] ; intros a InErase ; simpl in * ; [inversion InErase |].
      destruct (deq_evt_dec a') ; eauto.
      simpl in InErase.
      destruct InErase ; eauto.
    Qed.

    Lemma erase_lst_impl_deq : forall lst0 lst1, (erase_evt_lst lst0) = (erase_evt_lst lst1) -> deq_evt_lst G D lst0 lst1.
      induction lst0 ; induction lst1 ; intro EraseToNil ; simpl in *
      ; try destruct (deq_evt_dec a) ; try (constructor ; auto ; fail)
      ; try lazymatch goal with
        | [H : [] = _ :: _ |- _] => inversion H
        | [H : _ :: _ = [] |- _] => inversion H
      end.
      destruct (deq_evt_dec a0) ; [| injection EraseToNil ; intros ; subst] ; auto using DEqNoEvtR with deq.
    Qed.

    Lemma deq_impl_erase_lst : forall lst0 lst1, deq_evt_lst G D lst0 lst1 -> (erase_evt_lst lst0) = (erase_evt_lst lst1).
      intros ? ? DEqLsts. induction DEqLsts ; simpl in * ; try rewrite -> IHDEqLsts ; auto
      ; destruct (deq_evt_dec a) ; auto ; handle_simple_contradict.
    Qed.

    Lemma erase_app_distr : forall lst0 lst1,
        erase_evt_lst (lst0 ++ lst1) = erase_evt_lst (lst0) ++ erase_evt_lst (lst1).
      induction lst0 ; intro ; simpl ; [| destruct (deq_evt_dec a) ; rewrite -> IHlst0] ; auto.
    Qed.

    Lemma non_erase_low : forall a lst, ~ deq_evt G D a NoEvt -> erase_evt_lst (a :: lst) = a :: (erase_evt_lst lst).
      intros a ? ?. simpl. destruct (deq_evt_dec a) ; auto ; handle_simple_contradict.
    Qed.

    Lemma prefix_erase : forall lst0 lst1, Prefix lst0 lst1 -> Prefix (erase_evt_lst lst0) (erase_evt_lst lst1).
      intros ? ? Pfx. induction Pfx ; simpl ; try destruct (deq_evt_dec a) ; eauto.
    Qed.

    Lemma deq_cons_a_inv : forall lst0 lst1 a, deq_evt_lst G D lst0 lst1 -> deq_evt_lst G D lst0 (a :: lst1) -> deq_evt G D a NoEvt.
      intros ? ? ? DEqLst DEqConsLst.
      apply deq_impl_erase_lst with lst0 lst1 in DEqLst.
      apply deq_impl_erase_lst with lst0 (a :: lst1) in DEqConsLst.
      rewrite -> DEqLst in DEqConsLst.
      simpl in DEqConsLst.
      destruct (deq_evt_dec a) ; [assumption |].
      contradiction (cons_neq (erase_evt_lst lst1) a).
    Qed.

    Lemma deq_app_nil : forall lst0 lst1 lst2, deq_evt_lst G D lst0 lst1 -> deq_evt_lst G D lst0 (lst1 ++ lst2) -> deq_evt_lst G D lst2 [].
      intros.
      apply erase_lst_impl_deq.
      apply lst_eq_app_impl_nil with (erase_evt_lst lst1).
      rewrite <- erase_app_distr.
      apply deq_impl_erase_lst.
      transitivity lst0 ; auto with deq.
    Qed.

    Lemma deq_lst_nil_dec : forall lst, {deq_evt_lst G D lst []} + {~ deq_evt_lst G D lst []}.
      intros lst. induction lst ; auto with deq.
      destruct IHlst ; [destruct (deq_evt_dec a) |] ; [auto using DEqNoEvtL | |]
      ; right ; intro DEq ; inversion DEq ; auto.
    Qed.

  End ListErasure.

  Section DEqPfx.

    Variable G : Varname -> option Label.
    Variable D : Ensemble Label.

    #[local] Hint Resolve DEqEmpty : deq.
    #[local] Hint Resolve DEqSame : deq.
    #[local] Hint Resolve DEqNoEvtL : deq.
    #[local] Hint Resolve DEqNoEvtR : deq.

    Lemma pfx_deq_inv_list : forall s0 lst0 s1 lst1, (s0, lst0) =[G, D] (s1, lst1) -> deq_evt_lst G D lst0 lst1.
      intros ? ? ? ? DEqPfx. inversion DEqPfx ; assumption.
    Qed.

    Lemma pfx_deq_impl_dle : forall pfx0 pfx1, pfx0 <=, pfx1 -> forall pfx2, pfx1 =[G, D] pfx2 -> pfx0 <=[G, D] pfx2.
      intros ? ? [lst0 lst1 PrefixLst s]. dependent induction PrefixLst ; intros [s' ?] DEq ; inversion DEq as [? ? ? ? ? DEqLst] ; subst.
      * exists (s', []). auto using LePfx_intro with deq.
      * dependent induction DEqLst
        ; lazymatch goal with
          | [H : deq_evt_lst G D ?lst1 ?lst2, IH : forall s pfx2, (s, ?lst1) =[G, D] pfx2 -> _ |- (?s, ?a :: ?lst0) <=[G, D] _]
            => assert ((s, lst0) <=[G, D] (s', lst2)) as (pfx2' & PfxPfx & DEqPfx2') by eauto using DEqPfx_intro
          | [H : deq_evt_lst G D (?a :: ?lst1) ?lst2 |- (?s, ?a :: ?lst0) <=[G, D] _]
            => assert ((s, a :: lst0) <=[G, D] (s', lst2)) as (pfx2' & PfxPfx & DEqPfx2') by eauto with deq
        end
        ; inversion PfxPfx as [lst2'] ; subst ; inversion DEqPfx2' ; subst
        ; lazymatch goal with
          | [|- _ <=[G, D] (?s', ?a :: _)] => exists (s', (a :: lst2'))
          | [|- _ <=[G, D] (?s', _)] => exists (s', lst2')
        end ; auto using LePfx_intro with deq.
    Qed.

    Lemma deq_dle_pfx : forall pfx0 pfx1 pfx2, pfx0 <=[G, D] pfx1 -> pfx1 =[G, D] pfx2 -> pfx0 <=[G, D] pfx2.
      intros ? ? ? (pfx1' & Pfx1 & DEq01) DEq12.
      assert (pfx1' <=[G, D] pfx2) as (pfx2' & ? & ?) by eauto using pfx_deq_impl_dle.
      exists pfx2'. split ; [| transitivity pfx1'] ; assumption.
    Qed.

    Lemma dle_pfx_trans : forall pfx0 pfx1 pfx2, pfx0 <=[G, D] pfx1 -> pfx1 <=[G, D] pfx2 -> pfx0 <=[G, D] pfx2.
      unfold dle_pfx. intros pfx0 pfx1 pfx2 (pfx1' & ? & ?) (pfx2' & PfxOf2 & ?).
      assert (pfx1' <=[G, D] pfx2') as (pfx2'' & PfxOf2' & ?) by eauto using pfx_deq_impl_dle.
      exists pfx2''. split.
      * destruct pfx2 as [s2 lst2].
        inversion PfxOf2 as [lst2' ? Prefix2] ; subst.
        inversion PfxOf2' as [lst2'' ? Prefix2'] ; subst.
        apply LePfx_intro. generalize lst2' Prefix2' lst2 Prefix2. clear.
        induction lst2'' ; intros ; auto.
        inversion Prefix2' ; subst ; inversion Prefix2 ; subst ; eauto.
      * transitivity pfx1' ; assumption.
    Qed.

    Lemma deq_dlt_pfx : forall pfx0 pfx1 pfx2, pfx0 =[G, D] pfx1 -> pfx1 <[G, D] pfx2 -> pfx0 <[G, D] pfx2.
      unfold dlt_pfx, dle_pfx.
      intros pfx0 pfx1 pfx2 DEqPfx01 [(pfx2' & ? & ?) NDEqPfx12].
      split ; [exists pfx2' ; split ; [| transitivity pfx1] | intro ; contradict NDEqPfx12 ; transitivity pfx0]
      ; auto with deq.
    Qed.

    Lemma deq_nil_prefix : forall lst0 lst1, deq_evt_lst G D lst0 [] -> Prefix lst1 lst0 -> deq_evt_lst G D lst1 [].
      intros lst0 lst1 DEqLst PfxLst. induction PfxLst ; [| inversion DEqLst ; subst] ; eauto with deq.
    Qed.

    Lemma dle_pfx_antisym : forall pfx0 pfx1, pfx0 <=[G, D] pfx1 -> pfx1 <=[G, D] pfx0 -> pfx0 =[G, D] pfx1.
      intros ? ? (? & PfxLe0 & PfxDEq0).
      dependent destruction PfxDEq0.
      generalize dependent pfx1.
      dependent induction H0 ; intros ; inversion PfxLe0 ; subst
      ; repeat lazymatch goal with
        | [H : _ <=[G, D] _ |- _] => destruct H as (? & PfxLe & PfxDEq) ; inversion PfxDEq ; subst ; inversion PfxLe ; subst
        | [H : Prefix _ [] |- _] => inversion H ; subst ; clear H
        | [H : Prefix (_ :: _) _ |- _] => inversion H ; subst ; clear H
        | [H : Prefix _ (_ :: _) |- _] => inversion H ; subst ; clear H
      end ; apply DEqPfx_intro ; auto with deq
      ; lazymatch goal with
        | [|- deq_evt_lst G D (?a :: _) (?a :: _)] => apply DEqSame
        | [H : deq_evt G D ?a NoEvt |- deq_evt_lst G D (?a :: _) _] => apply DEqNoEvtL ; [assumption |]
        | [H : deq_evt G D ?a NoEvt |- deq_evt_lst G D _ (?a :: _)] => apply DEqNoEvtR ; [assumption |]
      end ; repeat lazymatch goal with
        | [H : deq_evt_lst G D (_ :: _) [] |- _] => inversion H ; subst ; clear H
        | [DEqLst0 : deq_evt_lst G D ?lst0 ?lst1', DEqNil : deq_evt_lst G D ?lst1 [],
           Pfx : Prefix ?lst1' ?lst1 |- deq_evt_lst G D ?lst0 ?lst1]
          => transitivity lst1' ; [| transitivity ([] : list Event)] ; eauto using deq_nil_prefix with deq
        | [IH : forall pfx, (?s1, _) <=, pfx -> pfx <=[G, D] (?s0, ?lst0) -> _,
           Pref1 : Prefix ?lst0' ?lst0 |- deq_evt_lst G D ?lst0 ?lst1]
          => assert ((s0, lst0) =[G, D] (s1, lst1)) as IHres
              by (apply IH ; [| exists (s0, lst0')] ; eauto using LePfx_intro, deq_cons_same_inv, deq_cons_l_inv, deq_cons_r_inv with deq)
             ; inversion IHres ; assumption
      end.
    Qed.

    Lemma deq_or_dle : forall {s0 lst0 s1 lst1 a}, (s0, lst0) <=[G, D] (s1, lst1 ++ [a])
        -> deq_evt_lst G D lst0 (lst1 ++ [a]) \/ (s0, lst0) <=[G, D] (s1, lst1).
      unfold dle_pfx. intros s0 lst0 s1 lst1 a (pfx1' & PfxLe & PfxDEq).
      inversion PfxLe as [lst1' ? Pfx] ; subst ; inversion PfxDEq ; subst.
      assert (lst1' = lst1 ++ [a] \/ Prefix lst1' lst1) as [|] by eauto using eq_or_prefix
      ; subst ; eauto using LePfx_intro.
    Qed.

  End DEqPfx.

  Section DEqPfxDecD.

    Context (G : Varname -> option Label) (D : Ensemble Label) `{DecideIn Label D}.

    (* Ltac unfold_all_dle_pfx dec_d :=
      repeat lazymatch goal with
        | [H : _ <=[D] _ |- _] =>
            unfold dle_pfx in H ; destruct H as (? & PfxLe & PfxDeq)
            ; dependent destruction PfxDeq ; let pfx := fresh "Pfx" in inversion PfxLe as [? ? pfx] ; subst ; clear PfxLe
            ; lazymatch type of pfx with
              | Prefix ?lst0 ?lst1 => apply prefix_erase with D dec_d lst0 lst1 in pfx
            end
        | [H : deq_evt_lst D ?lst0 ?lst1 |- _] => apply deq_impl_erase_lst with D dec_d lst0 lst1 in H
      end.

    Lemma dle_pfx_antisym : DecideIn D -> forall pfx0 pfx1, pfx0 <=[D] pfx1 -> pfx1 <=[D] pfx0 -> pfx0 =[D] pfx1.
      intros dec_d [] [] ? ?.
      unfold_all_dle_pfx dec_d.
      apply DEqPfx_intro ; auto.
      apply erase_lst_impl_deq with dec_d.
      repeat lazymatch goal with
      | [H : erase_evt_lst D dec_d _ = erase_evt_lst D dec_d _ |- _] => rewrite -> H in * ; clear H
      end.
      apply prefix_antisymm ; auto.
    Qed. *)

    Lemma dle_pfx_erase_prefix : forall s0 lst0 s1 lst1, (s0, lst0) <=[G, D] (s1, lst1)
        -> Prefix (erase_evt_lst G D lst0) (erase_evt_lst G D lst1).
      intros ? lst0 ? lst1 (pfx1' & Pfx1'Pfx & DEqPfx).
      inversion Pfx1'Pfx as [lst1' ? PfxLsts] ; subst.
      inversion DEqPfx as [? ? ? ? ? DEqLsts] ; subst.
      eapply deq_impl_erase_lst in DEqLsts.
      rewrite -> DEqLsts.
      apply prefix_erase. assumption.
    Qed.

    Ltac resolve_prefix_of_erase :=
      lazymatch goal with
        | [Pfx : Prefix ?lst0 (?a :: erase_evt_lst G ?D ?lst1) |- _]
          => let PfxN := fresh "Pfx" in
             induction lst0 ; [eauto using Prefix_empty |] ; inversion Pfx as [| ? ? ? PfxN] ; subst ; clear Pfx
        | [|- _] => idtac
      end ; lazymatch goal with
        | [IH : forall lst0, Prefix lst0 (erase_evt_lst G ?D ?lst1)
                -> exists lst1', Prefix lst1' ?lst1 /\ lst0 = erase_evt_lst G ?D lst1',
            Pfx : Prefix ?lst0 (erase_evt_lst G ?D ?lst1)
            |- exists lst1', Prefix lst1' (?a :: ?lst1) /\ _ = erase_evt_lst G ?D lst1']
            => let lstN := fresh "lst" in
                pose proof (IH lst0 Pfx) as (lstN & ? & ?) ; subst ; exists (a :: lstN) ; simpl
                ; destruct (deq_evt_dec G D a) ; auto ; handle_simple_contradict
      end.

    Lemma prefix_of_erase : forall lst1 lst0, Prefix lst0 (erase_evt_lst G D lst1)
        -> exists lst1', Prefix lst1' lst1 /\ lst0 = erase_evt_lst G D lst1'.
      induction lst1 ; intros lst0 Pfx.
      * simpl in *. inversion Pfx ; subst. exists []. auto.
      * simpl in Pfx ; destruct (deq_evt_dec G D a) ; resolve_prefix_of_erase.
    Qed.

    Lemma erase_prefix_dle_pfx : forall s0 lst0 s1 lst1, deq_store G D s0 s1
        -> Prefix (erase_evt_lst G D lst0) (erase_evt_lst G D lst1)
        -> (s0, lst0) <=[G, D] (s1, lst1).
      intros s0 lst0. remember (erase_evt_lst G D lst0) as lst.
      generalize dependent lst0.
      induction lst as [| a lst] ; intros ? ? s1 ? ? ErasePfx
      ; [exists (s1, []) | apply prefix_of_erase in ErasePfx ; destruct ErasePfx as (lst1' & ? & ?) ; rewrite -> Heqlst in * ; exists (s1, lst1')]
      ; eauto using LePfx_intro, DEqPfx_intro, erase_lst_impl_deq.
    Qed.

    Lemma dlt_pfx_erase_prefix : forall s0 lst0 s1 lst1, (s0, lst0) <[G, D] (s1, lst1)
        -> PropPrefix (erase_evt_lst G D lst0) (erase_evt_lst G D lst1).
      intros ? lst0 ? lst1 ((pfx1' & Pfx1'Pfx & DEqPfx) & PfxNEq).
      inversion Pfx1'Pfx as [lst1' ? PfxLsts] ; subst.
      inversion DEqPfx as [? ? ? ? ? DEqLsts] ; subst.
      unfold PropPrefix. split.
      * eapply deq_impl_erase_lst in DEqLsts.
        rewrite -> DEqLsts.
        apply prefix_erase. assumption.
      * intro EqErase. contradict PfxNEq. eauto using DEqPfx_intro, erase_lst_impl_deq.
    Qed.

    Lemma erase_prop_prefix_dlt_pfx : forall s0 lst0 s1 lst1, deq_store G D s0 s1
        -> PropPrefix (erase_evt_lst G D lst0) (erase_evt_lst G D lst1)
        -> (s0, lst0) <[G, D] (s1, lst1).
      intros s0 lst0.
      remember (erase_evt_lst G D lst0) as lst. rewrite -> Heqlst.
      generalize dependent lst0.
      induction lst as [| a lst] ; intros lst0 lstVal s1 lst1 ? [ErasePfx NEqErase]
      ; [| apply prefix_of_erase in ErasePfx ; destruct ErasePfx as (lst1' & ? & ?)]
      ; split ; [exists (s1, []) | | exists (s1, lst1') |]
      ; eauto using LePfx_intro, DEqPfx_intro, erase_lst_impl_deq
      ; intro DEqPfx ; inversion DEqPfx ; subst ; contradict NEqErase ; auto using deq_impl_erase_lst.
    Qed.

    Lemma not_dlt_empty : forall lst, deq_evt_lst G D lst [] -> forall pfx s, ~ (pfx <[G, D] (s, lst)).
      intros lst DEqLstNil pfx s [(pfx' & Pfx'OfLst & DEqPfx) NDEqPfx].
      contradict NDEqPfx.
      inversion Pfx'OfLst as [lst' ? PrefixLst] ; subst ; inversion DEqPfx ; subst.
      apply DEqPfx_intro ; auto.
      transitivity lst' ; auto.
      eapply erase_lst_impl_deq.
      eapply deq_impl_erase_lst in DEqLstNil.
      eapply prefix_erase in PrefixLst.
      rewrite -> DEqLstNil in *.
      simpl in PrefixLst.
      dependent destruction PrefixLst.
      auto.
    Qed.

    Lemma prefix_dle_eq_or_pfxle : forall pfx0 pfx1, (pfx0 <=, pfx1 \/ pfx1 <=, pfx0)
        -> pfx0 <=[G, D] pfx1
        -> pfx0 =[G, D] pfx1 \/ pfx0 <=, pfx1.
      intros pfx0 pfx1 PfxOfPfx DLePfx.
      destruct PfxOfPfx as [| Pfx1Of0] ; [right ; assumption | left].
      inversion Pfx1Of0 as [lst0 lst1 Pfx s] ; subst.
      apply dle_pfx_antisym ; eauto using erase_prefix_dle_pfx, prefix_erase with deq.
    Qed.

    Lemma dlt_impl_input_deq : forall s0 s1 lst0 lst1, (s0, lst0) <[G, D] (s1, lst1) -> deq_store G D s0 s1.
      intros s0 s1 ? ? [(pfx1' & PfxLt & DEq) ?].
      inversion PfxLt ; subst ; inversion DEq ; subst.
      assumption.
    Qed.

  End DEqPfxDecD.

End SecurityTheories.
