Require Import Basic.
Require Import ImpDefs Trace SecurityDefs.
Require Import Tactics.
Require Import InferenceDefs InferenceTheories.

From Coq Require Import Equality Relations RelationClasses Ensembles.
From Coq Require FunctionalExtensionality.

(* Prove that the less general label model is a special case of the more general one. *)
Section LabelModel.

  Context (Conf Integ : Set) (conf_flow : relation Conf) (integ_flow : relation Integ).
  Definition Label : Set := Conf * Integ.
  Context `{PreOrder Conf conf_flow} `{PreOrder Integ integ_flow}.
  Context (bot : Label) (bot_least : forall l, conf_flow (fst bot) (fst l) /\ integ_flow (snd bot) (snd l)).
  Context (voice : Conf -> Integ) (voice_antimon : forall c0 c1, conf_flow c0 c1 -> integ_flow (voice c1) (voice c0)).
  Context (view : Integ -> Conf) (view_antimon : forall i0 i1, integ_flow i0 i1 -> conf_flow (view i1) (view i0)).

  Instance lab_order : LabelOrder Label.
    (* Construct flows_to and reflect. *)
    exists (* flows_to *) (fun l0 l1 => conf_flow (fst l0) (fst l1) /\ integ_flow (snd l0) (snd l1))
           (* reflect *)  (fun l => (view (snd l), voice (fst l))).
    * (* reflexivity of flows_to *)
      split ; reflexivity.
    * (* transitivity of flows_to *)
      intros ? [c i] ? [? ?] [? ?].
      split ; [transitivity c | transitivity i] ; assumption.
    * (* Lower bounds exists *)
      intros. exists bot. auto.
    * (* anti-monotonicity of reflect *)
      intros ? ? [] ; simpl ; auto.
  Defined.

  Section AttackerModel.

    Context (P : Ensemble Conf) (T : Ensemble Integ) {DecP : DecideIn P} {DecT : DecideIn T}.
    Context (p_down_closed : forall c0 c1, In Conf P c1 -> conf_flow c0 c1 -> In Conf P c0).
    Context (t_down_closed : forall i0 i1, In Integ T i1 -> integ_flow i0 i1 -> In Integ T i0).

    (* Show that, assuming voice and view form an antitone Galois connection,
      then the two ways of saying that an attacker to read anything they can write
      are equivalent. *)
    Proposition comp_refl_equivalence (voice_view_galois : forall c i, conf_flow c (view i) <-> integ_flow i (voice c)) :
        (forall c, ~ In Conf P c -> In Integ T (voice c)) <-> (forall i, ~ In Integ T i -> In Conf P (view i)).
      split
      ; intros ? l NotIn
      ; [destruct (dec_in (view l)) | destruct (dec_in (voice l))] ; try assumption
      ; contradict NotIn
      ; [apply t_down_closed with (voice (view l)) | apply p_down_closed with (view (voice l))]
      ; eauto ; apply voice_view_galois ; reflexivity.
    Qed.

    Context (untrust_refl_pub : forall i, ~ In Integ T i -> In Conf P (view i)).

    (* Because of the module type structure, it's very difficult to directly
       claim that this is an instance of the Attacker class.
       Instead we just prove that Pub and Trust are downward-closed and
       non-compromised labels are always in their union. *)
    Definition Pub : Ensemble Label := fun l => In Conf P (fst l).
    Definition Trust : Ensemble Label := fun l => In Integ T (snd l).

    Proposition pub_down_closed : forall l0 l1, In Label Pub l1 -> flows_to l0 l1 -> In Label Pub l0.
      intros [] [] ? []. eapply p_down_closed ; eauto.
    Qed.

    Proposition trust_down_closed : forall l0 l1, In Label Trust l1 -> flows_to l0 l1 -> In Label Trust l0.
      intros [] [] ? []. eapply t_down_closed ; eauto.
    Qed.

    (* Proposition 5.8 *)
    Proposition non_comp_low : forall l, flows_to l (reflect l) -> In Label (Union Label Pub Trust) l.
      unfold Pub, Trust, In. intros [c i] []. simpl in *.
      destruct (DecP.(dec_in) c)
      ; [| destruct (DecT.(dec_in) i)]
      ; [apply Union_introl | apply Union_intror | exfalso] ; eauto.
    Qed.

  End AttackerModel.

End LabelModel.

Module Type LowEquivalence (B : Basic) (ID : ImpDefs) (TD : TraceDefs B ID) (SD : SecurityDefs B ID TD).
  Import ID SD.

  Context (G : Varname -> option Label) (D : Ensemble Label).

  (* Prove that the use of Silent is equivalent to being event-equal to bullet,
     meaning the definitions of low-equivalence match up. *)
  Section EventListDEqEquivalence.

    Inductive Silent : Event -> Prop :=
      | Bullet_intro: Silent NoEvt
      | HighAssign_intro: forall x n l, G x = Some l -> ~(In Label D l) -> Silent (AssignEvt x n)
      | HighPDown_intro: forall l, ~(In Label D l) -> Silent (PDownEvt l).

    Proposition hidden_iff_deq_none: forall e, Silent e <-> deq_evt G D e NoEvt.
      intro e ; destruct e ; split ; intro H ; inversion H ; subst
      ; econstructor ; eauto.
    Qed.

  End EventListDEqEquivalence.

End LowEquivalence.

(* Prove that the traditional NMIF downgrade condition can only downgrade
  compromised labels to other compromised labels, assuming the labels form
  a distributive lattice. *)
Module Type CompromisedDowngrades (ID : ImpDefs) (InfD : InferenceDefs ID) (InfT : InferenceTheories ID InfD).
  Import ID InfD InfT.

  Context (distr_lattice : forall l0 l1 l2, (join (meet l0 l1) l2) = (meet (join l0 l2) (join l1 l2))).

  Proposition Traditional_Downgrade_to_Non_compromised : forall pc nt l, flows_to l (reflect l)
      -> flows_to nt (join l (reflect (join pc nt)))
      -> flows_to nt (reflect nt).
    intros pc nt l ? ?.
    transitivity (join (meet l (join pc nt)) (reflect (join pc nt))).
    * rewrite -> distr_lattice.
      apply meet_glb ; auto using join_trans_bound_l, join_bound_r.
    * apply join_lub.
      - apply reflect_galois.
        transitivity (join (reflect l) (reflect (join pc nt))).
        + transitivity (join l (reflect (join pc nt))) ; [assumption |].
          apply join_lub ; auto using join_trans_bound_l, join_bound_r.
        + apply join_lub ; apply reflect_homomorphism ; auto using meet_bound_l, meet_bound_r.
      - apply reflect_homomorphism. apply join_bound_r.
  Qed.

End CompromisedDowngrades.
