Require Import ImpDefs.
Require Import InferenceDefs.

From Coq Require Import Equality Ensembles Relations.

Module Type InferenceTheories (ID : ImpDefs) (InfD : InferenceDefs ID).
  Import ID InfD.
  Import ImpNotations.

  #[local] Hint Resolve join_bound_l : lab.
  #[local] Hint Resolve join_bound_r : lab.
  #[local] Hint Resolve join_lub : lab.
  #[local] Hint Resolve meet_bound_l : lab.
  #[local] Hint Resolve meet_bound_r : lab.
  #[local] Hint Resolve meet_glb : lab.

  Section JoinMeetProps.

    Proposition join_comm : forall l1 l2, (join l1 l2) = (join l2 l1).
      intros l1 l2. apply flows_to_antisymm ; auto using join_lub, join_bound_l, join_bound_r.
    Qed.

    Proposition join_assoc : forall l1 l2 l3, (join (join l1 l2) l3) = (join l1 (join l2 l3)).
      intros l1 l2 l3. apply flows_to_antisymm ; repeat apply join_lub
      ; try lazymatch goal with
        | [|- flows_to ?l (join _ (join ?l ?l'))] => transitivity (join l l')
        | [|- flows_to ?l (join _ (join ?l' ?l))] => transitivity (join l' l)
        | [|- flows_to ?l (join (join ?l ?l') _)] => transitivity (join l l')
        | [|- flows_to ?l (join (join ?l' ?l) _)] => transitivity (join l' l)
      end
      ; auto using join_bound_l, join_bound_r.
    Qed.

    Proposition join_idempotent : forall l, join l l = l.
      intro l. apply flows_to_antisymm ; auto using join_lub, join_bound_l.
    Qed.

    Proposition meet_comm : forall l1 l2, (meet l1 l2) = (meet l2 l1).
      intros l1 l2. apply flows_to_antisymm ; auto using meet_glb, meet_bound_l, meet_bound_r.
    Qed.

    Proposition meet_assoc : forall l1 l2 l3, (meet (meet l1 l2) l3) = (meet l1 (meet l2 l3)).
      intros l1 l2 l3. apply flows_to_antisymm ; repeat apply meet_glb
      ; try lazymatch goal with
        | [|- flows_to (meet _ (meet ?l ?l')) ?l] => transitivity (meet l l')
        | [|- flows_to (meet _ (meet ?l' ?l)) ?l] => transitivity (meet l' l)
        | [|- flows_to (meet (meet ?l ?l') _) ?l] => transitivity (meet l l')
        | [|- flows_to (meet (meet ?l' ?l) _) ?l] => transitivity (meet l' l)
      end
      ; auto using meet_bound_l, meet_bound_r.
    Qed.

    Proposition meet_idempotent : forall l, meet l l = l.
      intro l. apply flows_to_antisymm ; auto using meet_glb, meet_bound_l.
    Qed.

    Lemma join_bot_l : forall l, join bot l = l.
      intro. apply flows_to_antisymm ; auto with lab.
    Qed.

    Lemma join_l_bot : forall l, join l bot = l.
      intro. apply flows_to_antisymm ; auto with lab.
    Qed.

    Lemma join_assoc_comm : forall l0 l1 l2, join (join l0 l1) l2 = join (join l0 l2) l1.
      intros l0 l1 l2. rewrite -> join_assoc. rewrite join_comm with l1 l2. rewrite <- join_assoc. reflexivity.
    Qed.

    Lemma join_distr : forall l0 l1 l2, join (join l0 l1) l2 = join (join l0 l2) (join l1 l2).
      intros l0 l1 l2. apply flows_to_antisymm ; repeat apply join_lub
      ; try lazymatch goal with
        | [|- flows_to ?l (join (join ?l ?l') _)] => transitivity (join l l')
        | [|- flows_to ?l (join (join ?l' ?l) _)] => transitivity (join l' l)
        | [|- flows_to ?l (join _ (join ?l ?l'))] => transitivity (join l l')
      end ; auto with lab.
    Qed.

    Lemma meet_distr : forall l0 l1 l2, meet (meet l0 l1) l2 = meet (meet l0 l2) (meet l1 l2).
      intros l0 l1 l2. apply flows_to_antisymm ; repeat apply meet_glb
      ; try lazymatch goal with
        | [|- flows_to (meet (meet ?l ?l') _) ?l] => transitivity (meet l l')
        | [|- flows_to (meet (meet ?l' ?l) _) ?l] => transitivity (meet l' l)
        | [|- flows_to (meet _ (meet ?l ?l')) ?l] => transitivity (meet l l')
      end ; auto with lab.
    Qed.

    Lemma join_trans_bound_l : forall l l0 l1, flows_to l l0 -> flows_to l (join l0 l1).
      intros l l0 l1 Flow. transitivity l0 ; auto using join_bound_l.
    Qed.

    Lemma join_trans_bound_r : forall l l0 l1, flows_to l l1 -> flows_to l (join l0 l1).
      intros l l0 l1 Flow. transitivity l1 ; auto using join_bound_r.
    Qed.

    Lemma meet_trans_bound_l : forall l0 l1 l, flows_to l0 l -> flows_to (meet l0 l1) l.
      intros l0 l1 l Flow. transitivity l0 ; auto using meet_bound_l.
    Qed.

    Lemma meet_trans_bound_r : forall l0 l1 l, flows_to l1 l -> flows_to (meet l0 l1) l.
      intros l0 l1 l Flow. transitivity l1 ; auto using meet_bound_r.
    Qed.

    Lemma reflect_lattice_hom_join : forall l1 l2, reflect (join l1 l2) = meet (reflect l1) (reflect l2).
      intros l1 l2. apply flows_to_antisymm.
      * apply meet_glb ; apply reflect_homomorphism ; auto using join_bound_l, join_bound_r.
      * apply reflect_galois.
        apply join_lub ; apply reflect_galois ; auto using meet_bound_l, meet_bound_r.
    Qed.

  End JoinMeetProps.

  Ltac resolve_refl_flow :=
    lazymatch goal with
      | [H0 : flows_to ?l ?l', H1 : flows_to ?l' (reflect ?l') |- flows_to ?l (reflect ?l)]
        => transitivity l' ; [| transitivity (reflect l')] ; auto using reflect_homomorphism
    end.

  #[local] Hint Resolve join_trans_bound_l : lab.
  #[local] Hint Resolve join_trans_bound_r : lab.
  #[local] Hint Resolve meet_trans_bound_l : lab.
  #[local] Hint Resolve meet_trans_bound_r : lab.

  Inductive CanInfer (G : Varname -> option Label) : Label -> Cmd -> (PartialInferCmd * Label * Label) -> Prop :=
    | CanInfSkip : forall pc, CanInfer G pc Skip (PartSkip, top, bot)
    | CanInfAssign : forall pc x e l0 l1, G x = Some l0
        -> expr_label G e = Some l1
        -> flows_to (join pc l1) l0
        -> CanInfer G pc (Assign x e) (PartAssign x e, l0, bot)
    | CanInfIfClean : forall pc e c1 c2 l, expr_label G e = Some l
        -> forall c1' b1 nt1, CanInfer G (join pc l) c1 (c1', b1, nt1)
        -> forall c2' b2 nt2, CanInfer G (join pc l) c2 (c2', b2, nt2)
        -> flows_to (join nt1 nt2) (reflect (join nt1 nt2))
        -> CanInfer G pc (If e c1 c2) (PartIf e l c1' c2', meet b1 b2, join nt1 nt2)
    | CanInfIfPDown : forall pc e c1 c2 l, expr_label G e = Some l
        -> forall c1' b1 nt1, CanInfer G (join pc l) c1 (c1', b1, nt1)
        -> forall c2' b2 nt2, CanInfer G (join pc l) c2 (c2', b2, nt2)
        -> ~ flows_to (join nt1 nt2) (reflect (join nt1 nt2))
        -> CanInfer G pc (If e c1 c2) (PartIf e l (PartProgDown c1') c2', meet b1 b2, nt2)
    | CanInfSeqClean : forall pc c1 c2,
        forall c1' b1 nt1, CanInfer G pc c1 (c1', b1, nt1)
        -> forall c2' b2 nt2, CanInfer G pc c2 (c2', b2, nt2)
        -> flows_to nt1 b2
        -> CanInfer G pc (Seq c1 c2) (PartSeq c1' nt1 c2', meet b1 b2, join nt1 nt2)
    | CanInfSeqPDown : forall pc c1 c2,
        forall c1' b1 nt1, CanInfer G pc c1 (c1', b1, nt1)
        -> forall c2' b2 nt2, CanInfer G pc c2 (c2', b2, nt2)
        -> ~ flows_to nt1 b2
        -> CanInfer G pc (Seq c1 c2) (PartSeq (PartProgDown c1') bot c2', meet b1 b2, join pc nt2)
    | CanInfWhileClean : forall pc e c l, expr_label G e = Some l
        -> flows_to (join pc l) (reflect (join pc l))
        -> forall c' b nt, CanInfer G (join pc l) c (c', b, nt)
        -> flows_to nt b
        -> CanInfer G pc (While e c) (PartWhile e (join l nt) c', meet b (reflect (join pc l)), join nt (join pc l))
    | CanInfWhilePDown : forall pc e c l, expr_label G e = Some l
        -> flows_to (join pc l) (reflect (join pc l))
        -> forall c' b nt, CanInfer G (join pc l) c (c', b, nt)
        -> ~ flows_to nt b
        -> CanInfer G pc (While e c) (PartWhile e l (PartProgDown c'), meet b (reflect (join pc l)), join pc l).

  Lemma can_infer_impl_infer : forall {G pc c c' b nt}, CanInfer G pc c (c', b, nt) -> infer_pdown_locs G pc c = Some (c', b, nt).
    intros G pc c c' b nt Infer.
    dependent induction Infer ; simpl
    ; repeat lazymatch goal with
      | [IH : forall c' b' nt', (?c, ?b, ?nt) = (c', b', nt') -> ?Val = Some _ |- context[?Val] ] => specialize (IH c b nt eq_refl)
      | [H : _ = Some _ |- _] => rewrite -> H ; clear H
      | [H : flows_to ?l0 ?l1 |- context[flows_to_dec ?l0 ?l1]]
        => destruct (flows_to_dec l0 l1) as [| BadNFlow] ; [| contradict BadNFlow ; assumption]
      | [H : ~ flows_to ?l0 ?l1 |- context[flows_to_dec ?l0 ?l1]]
        => destruct (flows_to_dec l0 l1) ; [contradict H ; assumption |]
    end ; auto.
  Qed.

  Ltac simplify_infer_hypo G :=
    repeat lazymatch goal with
      | [H : match G ?x with | Some _ => _ | None => None end = Some _ |- _] => destruct_with_eqn (G x)
      | [H : match expr_label G ?e with | Some _ => _ | None => None end = Some _ |- _]
        => destruct_with_eqn (expr_label G e)
      | [H : match infer_pdown_locs G ?pc ?c with | Some _ => _ | None => None end = Some _ |- _]
        => destruct_with_eqn (infer_pdown_locs G pc c)
      | [H : (let (_, _) := ?p in _) = Some _ |- _] => destruct p
      | [H : (if flows_to_dec ?l0 ?l1 then _ else _) = Some _ |- _] => destruct (flows_to_dec l0 l1)
      | [H : match set_pdown_labs G ?l ?c with | Some _ => _ | None => None end = Some _ |- _]
        => destruct_with_eqn (set_pdown_labs G l c)
      | [H : Some _ = Some _ |- _] => injection H ; intros ; subst ; clear H
    end ; try discriminate.

  Lemma infer_impl_can_infer : forall {G c pc c' b nt}, infer_pdown_locs G pc c = Some (c', b, nt) -> CanInfer G pc c (c', b, nt).
    intro G ; induction c ; simpl ; intros ; simplify_infer_hypo G ; eauto using CanInfer.
  Qed.

  Inductive LEqPartialCmd (l : Label) : relation PartialInferCmd :=
    | LEqPartSkip : LEqPartialCmd l PartSkip PartSkip
    | LEqPartAssign : forall x e, LEqPartialCmd l (PartAssign x e) (PartAssign x e)
    | LEqPartIf : forall e l' c1 c1' c2 c2',
        LEqPartialCmd l c1 c1'
        -> LEqPartialCmd l c2 c2'
        -> LEqPartialCmd l (PartIf e l' c1 c2) (PartIf e l' c1' c2')
    | LEqPartSeq : forall c1 c1' nt nt' c2 c2',
        LEqPartialCmd l c1 c1'
        -> join nt l = join nt' l
        -> LEqPartialCmd l c2 c2'
        -> LEqPartialCmd l (PartSeq c1 nt c2) (PartSeq c1' nt' c2')
    | LEqPartWhile : forall e nt nt' c c',
        LEqPartialCmd l c c'
        -> join nt l = join nt' l
        -> LEqPartialCmd l (PartWhile e nt c) (PartWhile e nt' c')
    | LEqPartProgDown : forall c c',
        LEqPartialCmd l c c' -> LEqPartialCmd l (PartProgDown c) (PartProgDown c').

  Ltac split_individual_flows :=
    repeat lazymatch goal with
      | [H : context[reflect (join ?l0 ?l1)] |- _] => rewrite -> reflect_lattice_hom_join in H
      | [H : flows_to (join ?l0 ?l1) ?l2 |- _]
          => assert (flows_to l0 l2) by (transitivity (join l0 l1) ; [apply join_bound_l | assumption])
            ; assert (flows_to l1 l2) by (transitivity (join l0 l1) ; [apply join_bound_r | assumption])
            ; clear H
      | [H : flows_to ?l0 (meet ?l1 ?l2) |- _]
          => assert (flows_to l0 l1) by (transitivity (meet l1 l2) ; [assumption | apply meet_bound_l])
            ; assert (flows_to l0 l2) by (transitivity (meet l1 l2) ; [assumption | apply meet_bound_r])
            ; clear H
    end.

  Ltac resolve_refl_join_flow :=
    lazymatch goal with
      | [|- flows_to _ _]
        => repeat rewrite -> reflect_lattice_hom_join ; repeat apply join_lub ; repeat apply meet_glb ; auto with lab
           ; try lazymatch goal with
             | [H0 : flows_to ?l0 ?l1, H1 : flows_to ?l1 ?l2 |- flows_to ?l0 ?l2] => transitivity l1 ; assumption
             | [H0 : flows_to ?l0 ?l1, H1 : flows_to ?l2 (reflect ?l1) |- flows_to ?l2 (reflect ?l0)] => transitivity (reflect l1) ; auto using reflect_homomorphism
           end
           ; try (apply reflect_galois ; assumption) ; try resolve_refl_flow
    end.

  Ltac apply_infer_ih IH l :=
    lazymatch type of IH with
      | forall c' b' nt', (?c, ?b, ?nt) = (c', b', nt') -> forall l', _
        => pose proof (IH c b nt eq_refl l) as IHres
        ; lazymatch type of IHres with
          | _ -> _ -> exists c1' b' nt', CanInfer ?G (join ?pc l) ?cIn (c1', b', nt') /\ ?LabProps
            => let c' := fresh "c" in
               let b := fresh "b" in
               let nt := fresh "nt" in
               assert (exists c1' b' nt', CanInfer G (join pc l) cIn (c1', b', nt') /\ LabProps)
                  as (c' & b & nt & ? & ? & ? & ? & ? & [[? ?] | [? ?]])
                  by (eapply IHres ; eauto using flows_to_trans with lab)
               ; subst ; clear IHres
        end
    end.

  Ltac apply_all_infer_ih l :=
    repeat lazymatch goal with
      | [IH : forall c' b' nt', (?c, ?b, ?nt) = (c', b', nt') -> forall l', flows_to l' (reflect l') -> _ |- _]
        => apply_infer_ih IH l ; clear IH
    end ; repeat rewrite join_bot_l in * ; repeat rewrite join_l_bot in *.

  Lemma infer_increase_pc : forall {G pc c c' b nt}, CanInfer G pc c (c', b, nt)
      -> forall {l}, flows_to l (reflect l)
      -> flows_to l b
      -> exists c1' b' nt', CanInfer G (join pc l) c (c1', b', nt')
            /\ LEqPartialCmd l c' c1'
            /\ flows_to (meet b (reflect l)) b'
            /\ flows_to b' b
            /\ flows_to nt' (reflect nt')
            /\ (nt = bot /\ nt' = bot \/ flows_to pc nt /\ nt' = join nt l).
    intros G pc c c' b nt CanInf l FTlrefl FTlb. dependent induction CanInf.
    * do 3 eexists. repeat split ; try constructor ; auto.
    * exists (PartAssign x e). exists b. exists bot.
      repeat split ; try econstructor ; try rewrite join_assoc_comm ; eauto with lab.
    * apply_all_infer_ih l0 ; exists (PartIf e l c0 c) ; exists (meet b3 b0)
      ; [set (nt' := bot) | set (nt' := join nt1 l0) | set (nt' := join nt2 l0) | set (nt' := join (join nt1 nt2) l0)]
      ; exists nt'
      ; [| | | assert (flows_to nt' (reflect nt')) by (subst nt' ; split_individual_flows ; resolve_refl_join_flow)]
      ; (repeat split ; [| | rewrite -> meet_distr | | | ] ; auto using LEqPartIf with lab ; try (right ; split ; [transitivity (join pc l) |] ; auto with lab))
      ; [rewrite <- join_l_bot with nt' | rewrite <- join_l_bot with nt' | rewrite <- join_bot_l with nt' | subst nt' ; rewrite -> join_distr with nt1 nt2 l0]
      ; apply CanInfIfClean ; try rewrite -> join_assoc_comm ; try rewrite join_bot_l ; try rewrite join_l_bot ; auto.
      rewrite <- join_assoc_comm ; rewrite <- join_distr ; assumption.
    * apply_all_infer_ih l0.
      4: exists (PartIf e l (PartProgDown c0) c) ; exists (meet b3 b0) ; exists (join nt l0)
        ; repeat split ; [| | rewrite -> meet_distr | | | right ; split ; [transitivity (join pc l) |]] ; eauto using LEqPartialCmd with lab
        ; eapply CanInfIfPDown ; try rewrite -> join_assoc_comm ; eauto ; intro.
      all: contradiction H0 ; split_individual_flows ; resolve_refl_join_flow.
    * apply_infer_ih IHCanInf2 l ; (apply_infer_ih IHCanInf1 l ; [set (nt1' := bot) | set (nt1' := join nt1 l)])
      ; exists (PartSeq c0 nt1' c) ; subst nt1' ; exists (meet b3 b0)
      ; repeat rewrite join_bot_l in * ; repeat rewrite join_l_bot in *
      ; [set (nt' := bot) | set (nt' := join nt1 l) | set (nt' := join nt2 l) | set (nt' := join (join nt1 nt2) l)]
      ; exists nt'
      ; repeat split ; try rewrite -> meet_distr ; auto using LEqPartSeq with lab
      ; try (subst nt' ; apply LEqPartSeq ; auto ; rewrite -> join_assoc ; rewrite -> join_idempotent ; reflexivity)
      ; assert (nt' = join nt' bot) as nt'BotBot by auto using join_l_bot
      ; [rewrite -> nt'BotBot at 2 | rewrite -> nt'BotBot at 2 | rewrite <- join_bot_l with nt' | |]
      ; subst nt' ; [| | | rewrite -> join_distr |] ; try apply CanInfSeqClean ; auto
      ; try solve [apply join_lub ; transitivity (meet b2 (reflect l)) ; split_individual_flows ; auto with lab].
      apply_infer_ih IHCanInf2 (join nt1 l) ; repeat rewrite join_bot_l ; repeat rewrite join_l_bot ; auto.
      split_individual_flows ; resolve_refl_join_flow.
    * apply_all_infer_ih l ; try solve [exfalso ; auto]
      ; exists (PartSeq (PartProgDown c0) bot c)
      ; exists (meet b3 b0)
      ; [exists (join pc l) | exists (join (join pc nt2) l)]
      ; repeat split ; try rewrite -> meet_distr ; eauto using LEqPartialCmd with lab
      ; [rewrite <- join_l_bot at 2 | | rewrite -> join_distr |]
      ; try solve [split_individual_flows ; resolve_refl_join_flow]
      ; apply CanInfSeqPDown with (join nt1 l) ; auto
      ; intro ; contradict H ; (transitivity (join nt1 l) ; [| transitivity b0]) ; auto with lab.
    * apply_all_infer_ih l0
      ; [exists (PartWhile e (join l bot) c0) | exists (PartWhile e (join l (join nt0 l0)) c0)]
      ; exists (meet (meet b1 (reflect (join pc l))) (reflect l0))
      ; [exists (join (join pc l) l0) | exists (join (join nt0 (join pc l)) l0)]
      ; repeat split
      ; try (try rewrite -> join_l_bot ; apply LEqPartWhile)
      ; auto 6 using flows_to_antisymm with lab
      ; assert (meet (meet b1 (reflect (join pc l))) (reflect l0) = meet b1 (reflect (join (join pc l0) l))) as bEqual
          by (apply flows_to_antisymm ; resolve_refl_join_flow)
      ; try (split_individual_flows ; resolve_refl_join_flow).
      - rewrite -> bEqual.
        rewrite <- join_bot_l with (join (join pc l) l0).
        rewrite -> join_assoc_comm with pc l l0.
        apply CanInfWhileClean ; [| | rewrite join_assoc_comm |] ; auto.
        split_individual_flows ; resolve_refl_join_flow.
      - rewrite -> meet_distr. auto with lab.
      - rewrite -> bEqual.
        rewrite -> join_distr with nt0 (join pc l) l0. rewrite -> join_assoc_comm with pc l l0.
        apply CanInfWhileClean ; [| | rewrite join_assoc_comm |] ; auto
        ; split_individual_flows ; resolve_refl_join_flow
        ; transitivity (meet b0 (reflect l0)) ; auto with lab.
      - transitivity (meet b0 (reflect l0)) ; auto with lab.
    * apply_all_infer_ih l0 ; [contradict H1 ; auto |].
      exists (PartWhile e l (PartProgDown c0)).
      exists (meet (meet b1 (reflect (join pc l))) (reflect l0)).
      exists (join (join pc l) l0).
      repeat split ; [| | rewrite meet_distr | | |] ; auto using LEqPartialCmd with lab ; [| split_individual_flows ; resolve_refl_join_flow].
      assert (meet (meet b1 (reflect (join pc l))) (reflect l0) = meet b1 (reflect (join (join pc l0) l))) as bEqual
        by (apply flows_to_antisymm ; resolve_refl_join_flow).
      rewrite -> bEqual.
      rewrite -> join_assoc_comm with pc l l0.
      econstructor ; [| split_individual_flows ; resolve_refl_join_flow | rewrite join_assoc_comm |] ; eauto.
      intro ; contradict H1. transitivity (join nt0 l0) ; [| transitivity b1] ; auto with lab.
  Qed.

  Lemma infer_increase_pc_flow : forall G pc c c' b nt, CanInfer G pc c (c', b, nt)
      -> forall l c1' b' nt', CanInfer G (join pc l) c (c1', b', nt')
      -> flows_to l b.
    intros ? ? ? ? ? ? CanInf ? ? ? ? CanInf'. dependent induction CanInf ; inversion CanInf' ; subst
    ; repeat lazymatch goal with
      | [H0 : ?P = Some ?l, H1 : ?P = Some ?l' |- _] => rewrite -> H0 in H1 ; injection H1 ; intros ; subst ; clear H1
    end
    ; try rewrite join_assoc_comm in * ; split_individual_flows ; resolve_refl_join_flow ; eauto with lab.
  Qed.

  Lemma infer_decrease_pc : forall G pc c c' b nt, CanInfer G pc c (c', b, nt)
      -> forall pc', flows_to pc' pc
      -> exists c'' b' nt', CanInfer G pc' c (c'', b', nt').
    intros G pc c c' b nt CanInf. dependent induction CanInf ; intros pc' FTpc
    ; lazymatch goal with
      | [CanInf : CanInfer G (join ?pc ?l) _ (_, _, _), Flow : flows_to ?pc' ?pc |- _]
        => assert (flows_to (join pc' l) (join pc l)) by auto with lab
      | [|- _] => idtac
    end
    ; repeat lazymatch goal with
      | [IH : forall c b nt, (?c', ?b', ?nt') = (c, b, nt) -> forall pc', flows_to pc' ?pc -> _,
         FT : flows_to ?pc' ?pc,
         H : CanInfer G ?pc _ (?c', ?b', ?nt') |- _]
        => let b := fresh "b" in
           let nt := fresh "nt" in
           pose proof (IH c' b' nt' eq_refl pc' FT) as (? & b & nt & ?) ; clear IH
      end
    ; lazymatch goal with
      | [H0 : flows_to (join ?pc ?l) ?b, H1 : flows_to ?pc' ?pc |- context[Assign _ _]]
        => assert (flows_to (join pc' l) b) by (transitivity (join pc l) ; auto with lab)
      | [H1 : CanInfer G _ ?c1 (_, _, ?nt1), H2 : CanInfer G _ ?c2 (_, _, ?nt2) |- context[If _ ?c1 ?c2]]
        => destruct (flows_to_dec (join nt1 nt2) (reflect (join nt1 nt2)))
      | [H1 : CanInfer G _ ?c1 (_, _, ?nt1), H2 : CanInfer G _ ?c2 (_, ?b2, _) |- context[Seq ?c1 ?c2]]
        => destruct (flows_to_dec nt1 b2)
      | [Flow : flows_to ?pc' ?pc,
         FlowRefl : flows_to (join ?pc ?l) (reflect (join ?pc ?l)),
         H : CanInfer G (join ?pc' ?l) ?c (_, ?b, ?nt)
         |- context[CanInfer _ ?pc' (While _ ?c) _]]
        => assert (flows_to (join pc' l) (reflect (join pc' l))) by resolve_refl_flow
           ; destruct (flows_to_dec nt b)
      | [|- context[Skip]] => idtac
    end ; repeat eexists ; eauto using CanInfer.
  Qed.

  Corollary nt_bot_or_pc_flow : forall G pc c c' b nt, CanInfer G pc c (c', b, nt) -> nt <> bot -> flows_to pc nt.
    intros ? ? ? ? ? ? CanInf ?.
    pose proof (infer_increase_pc CanInf (bot_least (reflect bot)) (bot_least b))
      as (? & ? & ? & ? & ? & ? & ? & ? & [[? ?] | [? ?]]) ; [exfalso |] ; auto.
  Qed.

  Corollary infer_nt_ft_refl : forall {G pc c c' b nt}, CanInfer G pc c (c', b, nt) -> flows_to nt (reflect nt).
    intros G pc c c' b nt CanInf.
    pose proof (infer_increase_pc CanInf (bot_least (reflect bot)) (bot_least b))
      as (? & ? & ? & ? & ? & ? & ? & ? & [[? ?] | [? ?]]) ; subst ; rewrite -> join_l_bot in * ; assumption.
  Qed.

  Lemma erase_leq_part : forall {c1 c2 l}, LEqPartialCmd l c1 c2 -> forall l1 l2, erase_pdown (set_pdown_labs l1 c1) = erase_pdown (set_pdown_labs l2 c2).
    intros c1 c2 l LEqPart. dependent induction LEqPart ; intros ; simpl ; auto.
    * specialize (IHLEqPart1 (join l1 l') (join l2 l')) ; rewrite -> IHLEqPart1.
      specialize (IHLEqPart2 (join l1 l') (join l2 l')) ; rewrite -> IHLEqPart2.
      reflexivity.
    * specialize (IHLEqPart1 l1 l2) ; rewrite -> IHLEqPart1.
      specialize (IHLEqPart2 (join l1 nt) (join l2 nt')) ; rewrite -> IHLEqPart2.
      reflexivity.
    * specialize (IHLEqPart (join l1 nt) (join l2 nt')) ; rewrite -> IHLEqPart ; reflexivity.
  Qed.

  Theorem infer_no_modify : forall G c pc c' b nt, CanInfer G pc c (c', b, nt) -> c = (erase_pdown (set_pdown_labs pc c')).
    intro G ; induction c ; intros pc c' b nt Infer
    ; inversion Infer ; subst ; simpl ; try (rewrite -> join_l_bot)
    ; repeat lazymatch goal with
      | [IH : forall pc c' b nt, CanInfer _ pc ?c (c', b, nt) -> ?c = erase_pdown (set_pdown_labs pc c'),
          H : CanInfer _ ?pc ?c (?c', ?b, ?nt) |- context[erase_pdown (set_pdown_labs ?pc ?c')]] => specialize (IH pc c' b nt H) ; rewrite <- IH ; clear IH
    end ; auto.
    * pose proof (infer_increase_pc H6 (infer_nt_ft_refl H4) H7) as (c2'' & b2' & ? & CanInfc2'' & LEqPart & ?).
      assert (c2 = erase_pdown (set_pdown_labs (join pc nt1) c2'')) as Erasec2'' by eauto.
      pose proof (erase_leq_part LEqPart (join pc nt1) (join pc nt1)) as EraseEq.
      rewrite -> EraseEq ; rewrite <- Erasec2'' ; reflexivity.
    * assert (flows_to nt0 (reflect nt0)) as Nt0Refl by (pose proof (infer_nt_ft_refl Infer) as OutNtRefl ; split_individual_flows ; assumption).
      pose proof (infer_increase_pc H7 Nt0Refl H8) as (c' & b' & nt & CanInfc' & LEqPart & ?).
      specialize (IHc (join (join pc l) nt0) c' b' nt CanInfc').
      pose proof (erase_leq_part LEqPart (join pc (join l nt0)) (join (join pc l) nt0)) as EraseEq.
      rewrite -> EraseEq ; rewrite <- IHc ; reflexivity.
  Qed.

  Section SoundInference.

    Ltac simplify_infer_goal G :=
      repeat (lazymatch goal with
        | [H : Some ?l0 = Some ?l1 |- _] => inversion H ; subst ; clear H
        | [|- context[match expr_label G ?e with | Some _ => _ | None => None end]]
          => destruct_with_eqn (expr_label G e)
        | [IH : forall pc partc b nt, CanInfer G pc ?c (partc, b, nt) -> _,
          H : CanInfer G ?pc ?c (?part, _, ?nt) |- context[set_pdown_labs G ?pc ?part]]
          => let pdown := fresh in
             assert (exists c', set_pdown_labs G pc part = Some (c', nt)) as [? pdown] by eauto ; rewrite -> pdown ; clear pdown
        | [IH : forall pc c nt, set_pdown_labs G pc ?c' = Some (c, nt) -> _,
          H0 : set_pdown_labs G ?pc0 ?c' = Some _,
          H1 : LEqPartialCmd ?c ?c' |- context[set_pdown_labs G ?pc1 ?c]]
          => let pdown := fresh in
             assert (exists c' nt', set_pdown_labs G pc1 c = Some (c', nt')) as (? & ? & pdown) by eauto ; rewrite -> pdown ; clear pdown
      end ; try discriminate ; try rewrite -> join_l_bot ; try rewrite -> join_idempotent).

    Lemma set_pdown_lequiv_eq : forall c0 c1 l,
        LEqPartialCmd l c0 c1 -> forall pc, set_pdown_labs (join pc l) c0 = set_pdown_labs (join pc l) c1.
      intros c0 c1 l LEq. induction LEq ; intro pc ; simpl ; auto
      ; repeat lazymatch goal with
        | [|- context[join (join ?pc l) ?nt]] => rewrite -> join_assoc_comm with pc l nt
      end
      ; repeat lazymatch goal with
        | [IH : forall pc, set_pdown_labs _ _ = set_pdown_labs _ _ |- _] => rewrite <- IH ; clear IH
      end
      ; repeat lazymatch goal with
        | [H : join ?nt l = join ?nt' l |- context[join (join ?pc ?nt') l]] => rewrite -> join_assoc with pc nt l ; rewrite -> H ; rewrite <- join_assoc
      end ; reflexivity.
    Qed.

    Theorem expr_type_inference : forall G e l, expr_label G e = Some l -> ExprType G e l.
      intro G. induction e ; intros l ExprLab ; simpl in * ; eauto using ExprType.
      simplify_infer_hypo G. apply OpT ; eauto using EVarianceT with lab.
    Qed.

    Theorem sound_inference_help : forall G c pc c' b nt, CanInfer G pc c (c', b, nt) -> G;; pc |- set_pdown_labs pc c' -| nt.
      intro G. induction c ; intros pc c' b nt CanInf
      ; inversion CanInf as [| | ? ? ? ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? ? ? ? CanInfc_w
                              | ? ? ? ? ? ? ? ? ? CanInfc_b]
      ; subst ; simpl in *
      ; [| | | | | rewrite -> join_l_bot | rewrite <- join_assoc |]
      ; try lazymatch goal with
          | [CanInfc : CanInfer G ?pc ?c (?c', ?b, ?nt),
             FTb : flows_to ?l ?b
             |- context[set_pdown_labs (join ?pc ?l) ?c']]
            => assert (flows_to l (reflect l)) as FTrefl by eauto using infer_nt_ft_refl, infer_impl_can_infer
              ; pose proof (infer_increase_pc CanInfc FTrefl FTb) as (c'' & ? & nt' & CanInfc' & ? & ? & ? & ? & ntVal)
              ; assert (set_pdown_labs (join pc l) c' = set_pdown_labs (join pc l) c'') as SetLabsEq by (apply set_pdown_lequiv_eq ; auto)
              ; rewrite -> SetLabsEq
        end
        ; repeat lazymatch goal with
        | [IH : forall pc' c' nt, match infer_pdown_locs G pc' ?c with Some _ => _ | None => None end = Some(c', nt) -> G ;; pc' |- c' -| nt,
           H : infer_pdown_locs G ?pc ?c = Some(?c', _, ?nt) |- _]
          => assert (G;; pc |- (set_pdown_labs pc c') -| nt) by (apply IH ; rewrite -> H ; reflexivity) ; clear IH
        end.
      * apply SkipT.
      * apply VarianceT with b bot ; [| transitivity (join pc l1) |] ; auto with lab.
        apply AssignT ; [| apply EVarianceT with l1 ; [| transitivity (join pc l1)]] ; auto using expr_type_inference with lab.
      * apply VarianceT with (join pc l) (join nt1 nt2) ; auto with lab.
        apply IfT ; eauto using EVarianceT, VarianceT, expr_type_inference with lab.
      * apply VarianceT with (join pc l) nt ; auto with lab.
        apply IfT ; eauto using EVarianceT, expr_type_inference with lab.
        assert (flows_to nt1 (reflect nt1)) by eauto using infer_nt_ft_refl.
        apply VarianceT with (join pc l) (join pc l) ; [apply ProgDownT with nt1 | |] ; eauto.
        eapply nt_bot_or_pc_flow ; eauto.
        intro ; contradict H0 ; subst. rewrite -> join_l_bot. assumption.
      * apply SeqT with (join pc nt1) nt1 ; eauto with lab.
        apply VarianceT with (join pc nt1) nt' ; eauto with lab.
        destruct ntVal as [[? ?] | [? ?]] ; subst ; rewrite join_comm ; auto.
      * apply SeqT with pc pc ; [apply ProgDownT with nt1 | | | apply VarianceT with pc nt2 |]
        ; eauto using infer_nt_ft_refl with lab.
      * apply VarianceT with (join (join pc l) nt0) (join (join pc l) nt0) ; auto with lab.
        apply WhileT ; eauto using EVarianceT, expr_type_inference with lab.
        apply VarianceT with (join (join pc l) nt0) nt' ; eauto with lab.
        destruct ntVal as [[? ?] | [? ?]] ; subst ; auto with lab.
      * apply VarianceT with (join pc l) (join pc l) ; auto with lab.
        apply WhileT ; eauto using EVarianceT, expr_type_inference with lab.
        apply ProgDownT with nt0 ; eauto using infer_nt_ft_refl.
    Qed.

  End SoundInference.

  Section CompleteInference.

    Theorem complete_expr_inference : forall G e l, ExprType G e l -> exists l', expr_label G e = Some l'.
      intros G e l eType. induction eType ; simpl in * ; eauto.
      destruct IHeType1 as [? eInfType1] ; rewrite -> eInfType1.
      destruct IHeType2 as [? eInfType2] ; rewrite -> eInfType2.
      eauto.
    Qed.

    Lemma min_expr_lab_inference : forall G e l0 l1, expr_label G e = Some l0 -> ExprType G e l1 -> flows_to l0 l1.
      intro G. induction e ; intros l0 l1 ExprInf eType ; simpl in *.
      * inversion ExprInf. auto.
      * dependent induction eType.
        - rewrite -> ExprInf in H ; inversion H ; subst. auto.
        - transitivity l ; eauto.
      * destruct_with_eqn (expr_label G e1) ; destruct_with_eqn (expr_label G e2) ; try discriminate.
        injection ExprInf ; intro ; subst.
        dependent induction eType.
        - apply join_lub ; auto.
        - transitivity l0 ; eauto.
    Qed.

    Lemma wt_if_inv_c1 : forall G pc e c1 c2 nt, G;; pc |- If e c1 c2 -| nt -> G;; pc |- c1 -| nt.
      intros G pc e c1 c2 nt WTc. dependent induction WTc ; [| apply VarianceT with pc' nt'] ; eauto.
    Qed.

    Lemma wt_if_inv_c2 : forall G pc e c1 c2 nt, G;; pc |- If e c1 c2 -| nt -> G;; pc |- c2 -| nt.
      intros G pc e c1 c2 nt WTc. dependent induction WTc ; [| apply VarianceT with pc' nt'] ; eauto.
    Qed.

    Lemma wt_if_e_l : forall G pc e c1 c2 nt, G;; pc |- If e c1 c2 -| nt -> forall l, expr_label G e = Some l -> G;; (join pc l) |- If e c1 c2 -| nt.
      intros G pc e c1 c2 nt WTc. dependent induction WTc ; intros l eType.
      * assert (join pc l = pc) as EqPc by (apply flows_to_antisymm ; eauto using min_expr_lab_inference with lab).
        rewrite -> EqPc. apply IfT ; auto.
      * apply VarianceT with (join pc' l) nt' ; auto with lab.
    Qed.

    Lemma wt_seq_inv : forall G pc c1 c2 nt, G;; pc |- Seq c1 c2 -| nt
        -> exists nt' pc', G;; pc |- c1 -| nt' /\ G;; pc' |- c2 -| nt /\ flows_to pc pc' /\ flows_to nt' pc' /\ flows_to nt' nt.
      intros G pc c1 c2 nt WTc. dependent induction WTc ; eauto 7.
      pose proof (IHWTc c1 c2 eq_refl) as (nt0 & pc0 & ? & ? & ? & ? & ?).
      exists nt0 ; exists pc0.
      repeat split ; eauto using VarianceT, flows_to_trans.
    Qed.

    Lemma wt_while_l : forall G pc e c nt, G;; pc |- While e c -| nt -> forall l, expr_label G e = Some l -> G;; (join pc l) |- While e c -| nt.
      intros G pc e c nt WT. dependent induction WT ; intros l eType.
      * assert (join pc l = pc) as EqPc by (apply flows_to_antisymm ; eauto using min_expr_lab_inference with lab).
        rewrite -> EqPc. apply WhileT ; auto.
      * apply VarianceT with (join pc' l) nt' ; eauto with lab.
    Qed.

    Lemma wt_while_pc_nt_flow : forall G pc e c nt, G;; pc |- While e c -| nt -> flows_to pc nt.
      intros G pc e c nt WT. dependent induction WT ; eauto using flows_to_trans.
    Qed.

    Lemma wt_while_inv : forall G pc e c nt, G;; pc |- While e c -| nt -> G;; pc |- c -| nt.
      intros G pc e c nt WT. dependent induction WT ; eauto using VarianceT.
    Qed.

    Lemma wt_pdown_inv : forall G pc l c nt, G;; pc |- (ProgDown l c) -| nt -> exists nt', G;; pc |- c -| nt' /\ flows_to nt' (reflect nt').
      intros G pc l c nt WT. dependent induction WT ; [| pose proof (IHWT l c eq_refl) as (? & ? & ?)] ; eauto using VarianceT.
    Qed.

    Lemma wt_pdown_pc_flow : forall G pc l c nt, G;; pc |- (ProgDown l c) -| nt -> flows_to pc nt.
      intros G pc l c nt WT. dependent induction WT ; eauto using flows_to_trans.
    Qed.

    Lemma set_pdown_eq_struct : forall c pc pc', EqCmdStructure (set_pdown_labs pc c) (set_pdown_labs pc' c).
      induction c ; intros pc pc' ; eauto using EqCmdStructure.
    Qed.

    Theorem complete_inf_help : forall G pc c nt, flows_to nt (reflect nt) -> G;; pc |- c -| nt
        -> exists c' b nt', CanInfer G pc (erase_pdown c) (c', b, nt').
      intros G pc c nt FTntRefl WTc. induction WTc ; simpl in * ; eauto using CanInfer
      ; repeat lazymatch goal with
        | [FT1 : flows_to ?l ?l', FTrefl : flows_to ?l' (reflect ?l') |- _]
          => assert (flows_to l (reflect l)) by resolve_refl_flow ; clear FT1
        | [FT : flows_to ?l (reflect ?l),
           IH : flows_to ?l (reflect ?l) -> exists c' b nt', _ |- _]
          => let c := fresh "c" in
             let b := fresh "b" in
             let nt := fresh "nt" in
             pose proof (IH FT) as (c & b & nt & ?) ; clear IH
        end.
      * assert (exists l', expr_label G e = Some l') as [l' ?] by (apply complete_expr_inference with l ; assumption).
        assert (flows_to l' l) by (apply min_expr_lab_inference with G e ; auto).
        eauto 6 using CanInfAssign with lab.
      * assert (exists l, expr_label G e = Some l) as [l ?] by (apply complete_expr_inference with pc ; assumption).
        assert (join pc l = pc) as PcJoinL
          by (apply flows_to_antisymm ; try apply join_lub ; [| apply min_expr_lab_inference with G e |] ; auto with lab).
        destruct (flows_to_dec (join nt1 nt0) (reflect (join nt1 nt0))).
        - exists (PartIf e l c0 c). exists (meet b1 b0). exists (join nt1 nt0).
          apply CanInfIfClean ; try rewrite -> PcJoinL ; assumption.
        - exists (PartIf e l (PartProgDown c0) c). exists (meet b1 b0). exists nt0.
          apply CanInfIfPDown with nt1 ; try rewrite -> PcJoinL ; assumption.
      * assert (exists c2' b2' nt2', CanInfer G pc (erase_pdown c2) (c2', b2', nt2')) as (? & b2 & ? & ?) by eauto using infer_decrease_pc.
        destruct (flows_to_dec nt1 b2) ; repeat eexists ; eauto using CanInfer.
      * assert (exists l, expr_label G e = Some l) as [l ?] by (apply complete_expr_inference with pc ; assumption).
        assert (join pc l = pc) as PcJoinL
          by (apply flows_to_antisymm ; try apply join_lub ; [| apply min_expr_lab_inference with G e |] ; auto with lab).
        destruct (flows_to_dec nt b0).
        - exists (PartWhile e (join l nt) c0). repeat eexists.
          eapply CanInfWhileClean ; try rewrite -> PcJoinL ; eauto.
        - exists (PartWhile e l (PartProgDown c0)). repeat eexists.
          eapply CanInfWhilePDown ; try rewrite -> PcJoinL ; eauto.
      * eauto using infer_decrease_pc.
    Qed.

  End CompleteInference.

  Section MinimalInference.

    Lemma wt_while_inv_full : forall G pc e c nt, G;; pc |- While e c -| nt
        -> forall l, expr_label G e = Some l
        -> exists nt', G;; join (join pc l) nt' |- c -| nt' /\ flows_to nt' nt.
      intros G pc e c nt WT. dependent induction WT ; intros l eType.
      * assert (join pc l = pc) as EqPc by (apply flows_to_antisymm ; eauto using min_expr_lab_inference with lab).
        rewrite -> EqPc. exists pc. rewrite join_idempotent. auto.
      * assert (exists nt0, G;; join (join pc' l) nt0 |- c -| nt0 /\ flows_to nt0 nt') as (nt0 & ? & ?) by eauto.
        exists nt0. split ; [apply VarianceT with (join (join pc' l) nt0) nt0 | transitivity nt'] ; auto with lab.
    Qed.

    Lemma erase_struct_pdle : forall c0 c1, PDownLe c0 c1 -> (erase_pdown c0) = (erase_pdown c1).
      intros c0 c1 PdLe. induction PdLe ; simpl
      ; repeat lazymatch goal with
        | [IH : erase_pdown ?c = erase_pdown ?c' |- _] => rewrite -> IH ; clear IH
      end ; auto.
    Qed.

    Lemma erase_infer : forall G pc c c' b nt, CanInfer G pc c (c', b, nt) -> forall l, (erase_pdown (set_pdown_labs l c')) = c.
      intros G pc c c' b nt CanInf l. dependent induction CanInf ; simpl
      ; repeat lazymatch goal with
        | [IH : forall c' b' nt', (?c, ?b, ?nt) = (c', b', nt') -> forall l, erase_pdown ?c0 = ?c1 |- _] =>
          specialize (IH c b nt eq_refl) ; rewrite -> IH ; clear IH
      end ; auto.
    Qed.

    Lemma min_nt_inference : forall G c pc c' b nt, CanInfer G pc c (c', b, nt)
        -> forall c1 nt', G;; pc |- c1 -| nt'
        -> EqCmdStructure (set_pdown_labs pc c') c1
        -> flows_to nt nt'.
      unfold infer. intro G. induction c as [| x e | e c1' ? c2' ? | c1' ? c2' ? | e c_w ? | l c_p ? |]
      ; intros pc c' ? nt CanInf c1 nt' WTc1 EqCmdStruct
      ; inversion CanInf as [| | ? ? ? ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? ? ? ? CanInfc_w
                              | ? ? ? ? ? ? ? ? ? CanInfc_b]
      ; subst ; simpl in * ; inversion EqCmdStruct ; subst ; auto.
      * apply join_lub.
        - eapply IHc1 with (c1 := c1'1) ; eauto using wt_if_inv_c1, wt_if_e_l.
        - eapply IHc2 with (c1 := c2'1) ; eauto using wt_if_inv_c2, wt_if_e_l.
      * eapply IHc2 with (c1 := c2'1) ; eauto using wt_if_inv_c2, wt_if_e_l.
      * lazymatch goal with
          | [H : CanInfer G ?pc ?c (?c', ?b, ?nt),
            FTb : flows_to ?l ?b,
            H' : context[set_pdown_labs (join ?pc ?l) ?c'] |- _]
            => assert (flows_to l (reflect l)) as FTrefl by eauto using infer_nt_ft_refl
              ; let CanInf := fresh "CanInf" in
                let newC := fresh "c" in
                let newNt := fresh "nt" in
                pose proof (infer_increase_pc H FTrefl FTb) as (newC & ? & newNt & CanInf & ? & ? & ? & ? & ntVal)
              ; assert (set_pdown_labs (join pc l) c' = set_pdown_labs (join pc l) newC) as SetLabsEq by (apply set_pdown_lequiv_eq ; auto)
        end.
        rewrite -> SetLabsEq in *.
        pose proof (wt_seq_inv G pc c1'1 c2'1 nt' WTc1) as (nt1' & pc0 & ? & ? & ? & ? & ?).
        assert (flows_to nt1 nt1') by eauto.
        destruct ntVal as [[? ?] | [? ?]] ; subst.
        - rewrite -> join_l_bot. transitivity nt1' ; auto.
        - rewrite -> join_comm with nt2 nt1 in CanInf0.
          eapply IHc2 with (c1 := c2'1) ; eauto using VarianceT, join_lub, flows_to_trans.
      * rewrite -> join_l_bot in *.
        inversion H2 ; subst.
        pose proof (wt_seq_inv G pc (ProgDown l' c') c2'1 nt' WTc1) as (nt0 & pc0 & ? & ? & ? & ? & ?).
        apply join_lub ; [transitivity nt0 | eapply IHc2 with (c1 := c2'1)] ; eauto using wt_pdown_pc_flow, VarianceT.
      * apply join_lub.
        - eapply IHc with (c1 := c')
          ; [| | transitivity (set_pdown_labs (join pc (join l nt0)) c'0)]
          ; eauto using wt_while_inv, wt_while_l, set_pdown_eq_struct.
        - eauto using wt_while_pc_nt_flow, wt_while_l.
      * eauto using wt_while_pc_nt_flow, wt_while_l.
    Qed.

    Lemma eq_cmd_pdownle : forall c0 c1, EqCmdStructure c0 c1 -> PDownLe c0 c1.
      intros c0 c1 EqStruc. induction EqStruc ; auto using PDownLe.
    Qed.

    Theorem min_inference : forall G c pc cInt b nt, CanInfer G pc c (cInt, b, nt)
        -> forall c' nt', PDownLe c' (set_pdown_labs pc cInt)
        -> flows_to nt' (reflect nt')
        -> G;; pc |- c' -| nt'
        -> EqCmdStructure (set_pdown_labs pc cInt) c'.
      intro G. induction c ; intros pc cIn b nt CanInf c' nt' PDownLec FTnt'refl WTc'
      ; inversion CanInf as [| | ? ? ? ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? CanInf1 ? ? ? CanInf2
                              | ? ? ? ? ? ? ? ? ? CanInfc_w
                              | ? ? ? ? ? ? ? ? ? CanInfc_b]
      ; subst ; simpl in * ; inversion PDownLec ; subst
      ; try lazymatch goal with
        | [H : PDownLe ?c (ProgDown _ (set_pdown_labs _ _)) |- _] => inversion H ; subst
      end ; subst ; simpl in *
      ; auto using EqCmdStructure.
      * apply EqIf ; eauto using wt_if_inv_c1, wt_if_inv_c2, wt_if_e_l.
      * apply EqIf ; eauto using wt_if_inv_c1, wt_if_inv_c2, wt_if_e_l.
        apply EqPDown.
        assert (exists nt0, G;; join pc l |- c -| nt0 /\ flows_to nt0 (reflect nt0)) as (nt0 & ? & ?) by eauto using wt_pdown_inv, wt_if_e_l, wt_if_inv_c1.
        apply IHc1 with b1 nt1 nt0 ; auto.
      * contradict H0.
        enough (flows_to (join nt1 nt) nt') by resolve_refl_flow.
        apply join_lub.
        - apply min_nt_inference with G c1 (join pc l) c1' b1 c0
          ; eauto using sound_inference_help, wt_if_e_l, wt_if_inv_c1.
        - apply min_nt_inference with G c2 (join pc l) c2' b2 c3
          ; eauto using sound_inference_help, wt_if_e_l, wt_if_inv_c2.
      * pose proof (wt_seq_inv G pc c0 c3 nt' WTc') as (nt0 & pc0 & WTc0 & WTc3 & FTpc0 & FTnt0pc0 & FTnt0).
        assert (flows_to nt0 (reflect nt0)) by resolve_refl_flow.
        apply EqSeq ; eauto.
        transitivity (set_pdown_labs pc c2') ; auto using set_pdown_eq_struct.
        apply IHc2 with b2 nt2 nt' ; auto.
        - transitivity (set_pdown_labs (join pc nt1) c2') ; auto using set_pdown_eq_struct, eq_cmd_pdownle.
        - apply VarianceT with pc0 nt' ; auto.
      * pose proof (wt_seq_inv G pc (ProgDown l c) c3 nt' WTc') as (nt0 & pc0 & WTc0 & WTc3 & FTpc0 & FTnt0pc0 & FTnt0).
        pose proof (wt_pdown_inv G pc l c nt0 WTc0) as (nt0' & WTc0' & ?).
        rewrite -> join_l_bot in *.
        apply EqSeq ; [apply EqPDown |] ; eauto using VarianceT.
      * contradict H.
        pose proof (wt_seq_inv G pc c0 c3 nt' WTc') as (nt0 & pc0 & WTc0 & WTc3 & FTpc0 & FTnt0pc0 & FTnt0).
        transitivity nt0.
        - apply min_nt_inference with G c1 pc c1' b1 c0 ; eauto using sound_inference_help.
          eapply IHc1 with _ _ nt0 ; try resolve_refl_flow ; eauto.
        - assert (exists c2'' b2' nt2', CanInfer G (join pc nt0) (erase_pdown c3) (c2'', b2', nt2')) as (c2'' & b2' & nt2' & ?)
            by (apply complete_inf_help with nt' ; eauto using VarianceT with lab).
          rewrite -> join_l_bot in *.
          apply infer_increase_pc_flow with G pc c2 c2' nt2 c2'' b2' nt2' ; auto.
          enough ((erase_pdown c3) = c2) by (subst ; assumption).
          transitivity (erase_pdown (set_pdown_labs pc c2')) ; eauto using erase_struct_pdle, erase_infer.
      * pose proof (wt_while_inv_full G pc e c0 nt' WTc' l H) as (nt0' & ? & ?).
        apply EqWhile.
        transitivity (set_pdown_labs (join pc l) c'0) ; auto using set_pdown_eq_struct.
        apply IHc with b0 nt0 nt0' ; try resolve_refl_flow ; auto.
        - transitivity (set_pdown_labs (join pc (join l nt0)) c'0) ; auto using set_pdown_eq_struct, eq_cmd_pdownle.
        - apply VarianceT with (join (join pc l) nt0') nt0' ; auto with lab.
      * pose proof (wt_while_inv_full G pc e (ProgDown l0 c1) nt' WTc' l H) as (nt0' & WTc0 & ?).
        pose proof (wt_pdown_inv G (join (join pc l) nt0') l0 c1 nt0' WTc0) as (nt1' & WTc1' & ?).
        eauto using EqCmdStructure, VarianceT with lab.
      * contradict H1.
        pose proof (wt_while_inv_full G pc e c0 nt' WTc' l H) as (nt0' & ? & ?).
        assert (flows_to nt0' (reflect nt0')) by resolve_refl_flow.
        transitivity nt0'.
        - apply min_nt_inference with G c (join pc l) c'0 b0 c0
          ; eauto using sound_inference_help, VarianceT with lab.
        - assert (exists c1 b1 nt1, CanInfer G (join (join pc l) nt0') (erase_pdown c0) (c1, b1, nt1)) as (c1 & b1 & nt1 & ?)
          by (apply complete_inf_help with nt0' ; auto).
          apply infer_increase_pc_flow with G (join pc l) c c'0 nt0 c1 b1 nt1 ; auto.
          enough ((erase_pdown c0) = c) by (subst ; assumption).
          transitivity (erase_pdown (set_pdown_labs (join pc l) c'0)) ; eauto using erase_struct_pdle, erase_infer.
    Qed.

  End MinimalInference.

End InferenceTheories.
