Require Import Basic.
Require Import ImpDefs Trace SecurityDefs.
Require Import Tactics.
Require Import BasicTheories.

From Coq Require Import Basics Equality Relations RelationClasses List PeanoNat.

Import ListNotations.

Module Type ImpTheories (B : Basic) (ID : ImpDefs) (TD : TraceDefs B ID) (SD : SecurityDefs B ID TD)
    (Tac : SimpleTactics B ID TD SD) (BT : BasicTheories B).
  Import B ID Tac BT.
  Import ImpNotations.

  Open Scope imp_scope.

  Section StepTheories.

    Lemma seq_type_c1_type : forall G pc c1 c2 nt, G;; pc |- (Seq c1 c2) -| nt -> G;; pc |- c1 -| nt.
      intros G pc c1 c2 nt WellTyped. dependent induction WellTyped ; eauto using VarianceT.
    Qed.

    Lemma while_cmd_type_inv : forall G pc e c nt, G;; pc |- While e c -| nt -> G;; pc |- c -| nt.
      intros ? ? ? ? ? WTc. dependent induction WTc ; eauto using VarianceT.
    Qed.

    Lemma wt_step_to_stop_impl_skip : forall G pc c nt,
        G;; pc |- c -| nt
        -> forall s a s', (c, s) -->[a] (Stop, s')
        -> c = Skip.
      intros G pc c nt WellTyped. induction WellTyped ; intros s a s' Step
      ; try (apply IHWellTyped with s a s' ; assumption)
      ; inversion Step ; try handle_simple_contradict ; auto.
    Qed.

    Lemma wt_step_to_stop_impl_stop_evt : forall G pc c nt,
        G;; pc |- c -| nt
        -> forall s a s', (c, s) -->[a] (Stop, s')
        -> a = StopEvt.
      intros G pc c nt WellTyped. induction WellTyped ; intros s a s' Step
      ; try (apply IHWellTyped with s s' ; assumption)
      ; inversion Step ; try handle_simple_contradict ; auto.
    Qed.

    Lemma step_stop_evt_impl_stop : forall cs c' s', cs -->[StopEvt] (c', s') -> c' = Stop.
      intros cs c' s' Step. dependent induction Step ; try reflexivity ; contradict H ; apply IHStep with s' ; auto.
    Qed.

    Lemma deterministic_step : forall cs cs1 a1,
        cs -->[a1] cs1 -> forall cs2 a2, cs -->[a2] cs2 -> cs1 = cs2 /\ a1 = a2.
      intros cs cs1 a1 Step1. dependent induction Step1 ; intros cs2 a2 Step2 ; dependent destruction Step2 ; auto ; try handle_simple_contradict.
      * apply IHStep1 in Step2. destruct Step2 as [CmdStoreEq EvtEq]. injection CmdStoreEq ; intros ; subst. auto.
      * rewrite -> H0 in H. inversion H. rewrite -> H2 in H0. inversion H0. auto.
      * apply IHStep1 in Step2. destruct Step2 as [CmdStoreEq EvtEq]. inversion CmdStoreEq. auto.
    Qed.

  End StepTheories.

  Ltac subst_eq_steps :=
    repeat lazymatch goal with
      | [H : (_, _) = (_, _) |- _] => injection H ; intros ; subst ; clear H
      | [H0 : ?cs -->[?a0] ?cs0, H1 : ?cs -->[?a1] ?cs1 |- _]
        => assert (cs1 = cs0 /\ a1 = a0) as [? ?] by eauto using deterministic_step ; subst ; clear H1
    end.

  Section TypeSoundness.

    Lemma cmd_stop_dec : forall c, {c = Stop} + {c <> Stop}.
      induction c ; try (left ; reflexivity) ; right ; discriminate.
    Qed.

    Lemma no_downgrade_skip : forall D G pc nt (Pf : CmdTypeProof G pc Skip nt), ~ HasDowngrade D Pf.
      intros D G pc nt Pf HasDown. dependent induction HasDown ; eauto.
    Qed.

    Lemma seq_from_while_down : forall D G pc e c (eType : ExprType G e pc) (Pf : CmdTypeProof G pc c pc),
        HasDowngrade D (SeqTPf G pc pc c (While e c) pc pc Pf (flows_to_refl pc) (flows_to_refl pc)
                        (WhileTPf G pc e c eType Pf) (flows_to_refl pc))
        -> HasDowngrade D (WhileTPf G pc e c eType Pf).
      intros D G pc e c eType Pf HasDown. dependent induction HasDown ; auto using HasDownWhile.
    Qed.

    Lemma type_pres_pf_with_downgrade : forall G pc c nt s a (Pf : CmdTypeProof G pc c nt) c' s',
        (c, s) -->[a] (c', s')
        -> c' <> Stop
        -> exists (Pf' : CmdTypeProof G pc c' nt), forall D, HasDowngrade D Pf' -> HasDowngrade D Pf.
      intros G pc c nt s a Pf. induction Pf ; intros c' s' Step c'NotStop.
      * handle_simple_contradict.
      * inversion Step ; subst.
        exists (SkipTPf G l nt). intros ? BadDown. inversion BadDown.
      * inversion Step ; subst ; eauto using HasDownIf1, HasDownIf2.
      * inversion Step as [| ? c1' ? ? ? ? c1'NotStop c1Step | | | | | | |] ; subst.
        - pose proof (IHPf1 c1' s' c1Step c1'NotStop) as IHres.
          destruct IHres as [Pf1' NoDown].
          exists (SeqTPf G pc pc' c1' c2 nt nt' Pf1' f f0 Pf2 f1).
          intros D HasDownPf'. dependent induction HasDownPf' ; auto using HasDowngrade.
        - exists (VarianceTPf G pc pc' c' nt nt Pf2 f (flows_to_refl nt)).
          intros D HasDownPf'. dependent induction HasDownPf' ; auto using HasDownSeq2.
      * inversion Step ; subst.
        exists (IfTPf G pc e (Seq c (While e c)) Skip pc e0
                (SeqTPf G pc pc c (While e c) pc pc Pf (flows_to_refl pc) (flows_to_refl pc)
                  (WhileTPf G pc e c e0 Pf) (flows_to_refl pc))
                (SkipTPf G pc pc)).
        intros D HasDownPf'. dependent induction HasDownPf'.
        - apply seq_from_while_down ; auto.
        - inversion HasDownPf'.
      * inversion Step as [| | | | | | | ? c1 ? ? ? ? c1NotStop cStep |] ; subst.
        - pose proof (IHPf c1 s' cStep c1NotStop) as IHres.
          destruct IHres as [Pf1 NoDown].
          exists (ProgDownTPf G pc c1 nt l Pf1 f f0).
          intros D HasDownPf'. dependent induction HasDownPf' ; auto using HasDowngrade.
        - exists (SkipTPf G pc l).
          intros D HasDownPf'. inversion HasDownPf'.
      * pose proof (IHPf c' s' Step c'NotStop) as IHres.
        destruct IHres as [Pf' IHNoDown].
        exists (VarianceTPf G pc pc' c' nt nt' Pf' f f0).
        intros D HasDownPf'. dependent induction HasDownPf'.
        apply HasDownVar. auto.
    Qed.

    Theorem type_pres_not_stop : forall G pc c nt s a, G;; pc |- c -| nt
        -> forall c' s', (c, s) -->[a] (c', s')
        -> c' <> Stop
        -> G;; pc |- c' -| nt.
      intros G pc c nt s a WTc c' s' Step c'NotStop.
      apply wt_impl_proof in WTc. destruct WTc as [Pf ?].
      pose proof (type_pres_pf_with_downgrade G pc c nt s a Pf c' s' Step c'NotStop) as [? ?].
      auto using wt_proof_impl_prop.
    Qed.

    Corollary type_preservation : forall G pc c nt s a, G;; pc |- c -| nt
        -> forall c' s', (c, s) -->[a] (c', s')
        -> c' = Stop \/ (G;; pc |- c' -| nt).
      intros G pc c nt s a WTc c' s' Step.
      destruct (cmd_stop_dec c') ; [left | right ; eapply type_pres_not_stop] ; eauto.
    Qed.

    Lemma dom_subset_preservation : forall (G : Varname -> option Label) c s c' s' a,
        dom_subset G s -> (c, s) -->[a] (c', s') -> dom_subset G s'.
      intros G c s c' s' a WellFormed Step. dependent induction Step ; eauto.
      unfold dom_subset in *. intros x' NotinS.
      destruct (eq_dec x x') ; [discriminate | eauto].
    Qed.

    Lemma mstep_type_pres_with_downgrade : forall c s c' s' lst,
        (c, s) ==>*[lst] (c', s')
        -> forall G pc nt (Pf : CmdTypeProof G pc c nt),
        c' <> Stop
        -> exists (Pf' : CmdTypeProof G pc c' nt), forall D, HasDowngrade D Pf' -> HasDowngrade D Pf.
      intros c s c' s' lst MStep. dependent induction MStep ; intros G pc nt Pf c'NotStop.
      * exists Pf. auto.
      * destruct cs1 as [c1 s1].
        assert (c1 <> Stop) as c1NotStop
          by (intro ; subst ; inversion MStep as [| ? ? ? ? ? Step]
              ; [contradict c'NotStop ; auto | inversion Step]).
        pose proof (type_pres_pf_with_downgrade G pc c nt s a Pf c1 s1 H c1NotStop) as OneStepRes.
        destruct OneStepRes as [Pf1 OneStepImpl].
        assert (exists (Pf' : CmdTypeProof G pc c' nt),
                forall D, HasDowngrade D Pf' -> HasDowngrade D Pf1) as IHres by eauto.
        destruct IHres as [Pf' IHImpl]. eauto.
    Qed.

    Lemma mstep_type_pres_not_stop : forall c s c' s' lst,
        (c, s) ==>*[lst] (c', s')
        -> forall G pc nt, G;; pc |- c -| nt
        -> c' <> Stop
        -> G;; pc |- c' -| nt.
      intros c s c' s' lst MStep G pc nt WTc c'NotStop.
      apply wt_impl_proof in WTc. destruct WTc as [Pf H] ; clear H.
      pose proof (mstep_type_pres_with_downgrade c s c' s' lst MStep G pc nt Pf c'NotStop) as H.
      destruct H. apply wt_proof_impl_prop. assumption.
    Qed.

    Lemma mstep_type_preservation : forall c s c' s' lst,
        (c, s) ==>*[lst] (c', s')
        -> forall G pc nt, G;; pc |- c -| nt
        -> c' = Stop \/ G;; pc |- c' -| nt.
      intros ? ? c'. intros.
      destruct (cmd_stop_dec c') ; [left | right ; eapply mstep_type_pres_not_stop] ; eauto.
    Qed.

    Lemma mstep_wf_store_preservation : forall c s c' s' lst, (c, s) ==>*[lst] (c', s')
        -> forall (G : Varname -> option Label), dom_subset G s
        -> dom_subset G s'.
      intros c s c' s' lst MStep. dependent induction MStep ; intros G WellFormedS.
      * assumption.
      * destruct cs1 as [c1 s1]. apply IHMStep with c1 s1 c' ; auto. apply dom_subset_preservation with c s c1 a ; auto.
    Qed.

    Lemma expr_type_eval_some : forall G e s l, dom_subset G s -> ExprType G e l -> exists n, evalExpr e s = Some n.
      unfold evalExpr. intros G e s l WellFormed WellTyped. induction WellTyped ; eauto using dom_subset_some.
      destruct IHWellTyped1 as [n1 EvalExprE1]. rewrite -> EvalExprE1.
      destruct IHWellTyped2 as [n2 EvalExprE2]. rewrite -> EvalExprE2. simpl. eauto.
    Qed.

    Ltac eval_e_s_as e s n EvalSome G pc :=
      assert (exists n, evalExpr e s = Some n) as eEvalSome by (apply expr_type_eval_some with G pc ; auto)
      ; destruct eEvalSome as [n EvalSome].

    Ltac progress_seq_pdown_step c :=
      lazymatch goal with
        | [ WT : ?G;; ?pc |- c -| ?nt,
            IHStep : exists c' s' a, (c, ?s) -->[a] (c', s') |- _ ] =>
          destruct IHStep as (c' & s' & a & ?) ; induction c'
          ; lazymatch goal with
            | [ H : (c, s) -->[a] (Stop, s') |- _ ]
                => assert (c = Skip) as cSkip by (apply wt_step_to_stop_impl_skip with G pc nt s a s' ; auto)
                  ; rewrite -> cSkip ; repeat eexists
                  ; match goal with
                    | [ |- ((Seq _ _), _) -->[_] _ ] => apply SeqSkipE
                    | [ |- ((ProgDown _ _), _) -->[_] _ ] => apply PDownSkipE
                  end
            | _ => match goal with
                    | [ |- exists c' s' a, ((Seq _ _), _) -->[_] _ ]
                        => repeat eexists ; apply SeqCE ; eauto ; discriminate
                    | [ |- exists c' s' a, ((ProgDown _ _), _) -->[_] _ ]
                        => repeat eexists ; apply PDownCE ; eauto ; discriminate
                  end
          end
      end.

    Theorem cmd_progress : forall G pc c nt s, G;; pc |- c -| nt -> dom_subset G s
        -> exists c' s' a, (c, s) -->[a] (c', s').
      intros G pc c nt s WellTyped WellFormedS. induction WellTyped ; eauto using OneStep.
      * assert (exists n, s x = Some n) as [n ?] by eauto using dom_subset_some.
        eval_e_s_as e s n' EvalSome G l. eauto using AssignE.
      * eval_e_s_as e s n EvalSome G pc. destruct n ; eauto using If0E, IfNE.
      * progress_seq_pdown_step c1.
      * progress_seq_pdown_step c.
    Qed.

    Theorem type_soundness : forall {G pc c nt s}, G;; pc |- c -| nt -> dom_subset G s -> never_stuck c s.
      unfold never_stuck. intros G pc c nt s WTc WFs c1 s1 lst MStep. dependent induction MStep.
      * assert (exists c s a, (c1, s1) -->[a] (c, s)) as (? & ? & ?) ; eauto using cmd_progress.
      * destruct cs1 as [c1' s1'].
        assert (c1' = Stop \/ G;; pc |- c1' -| nt) as [c1'Stop | WTc1'] by (apply type_preservation with c s a s1' ; assumption).
        - left. rewrite -> c1'Stop in MStep.
          inversion MStep as [ | ? ? ? ? ? StopStep] ; [reflexivity | inversion StopStep].
        - apply IHMStep with c1' s1' ; eauto using dom_subset_preservation.
    Qed.

  End TypeSoundness.

  Section MStepTheories.

    (*
      We include two equivalent versions of the multi-step semantics.

      The primary one forms a 'cons' list where there is a first step and a rest of
      the steps. For most proofs, this is the simplest. Some proofs, however, need a
      'snoc' list where there is all but the last step followed by the last one.

      We therefore include both and prove them equivalent so we can freely switch.
    *)
    Inductive MultiStepTl : Cmd * Store -> Cmd * Store -> list Event -> Prop :=
      | MultiStepTl_refl : forall cs, MultiStepTl cs cs []
      | MultiStepTl_some : forall cs0 cs1 cs2 a lst,
          MultiStepTl cs0 cs1 lst -> cs1 -->[a] cs2 -> MultiStepTl cs0 cs2 (lst ++ [a]).

    Lemma mstep_append : forall cs0 cs1 lst,
        cs0 ==>*[lst] cs1 -> forall cs2 a, cs1 -->[a] cs2 -> cs0 ==>*[lst ++ [a]] cs2.
      intros ? ? ? MStep. induction MStep ; intros ; eauto using MultiStep.
    Qed.

    Lemma mstep_tl_prepend : forall cs1 cs2 lst,
        MultiStepTl cs1 cs2 lst -> forall cs0 a, cs0 -->[a] cs1 -> MultiStepTl cs0 cs2 (a :: lst).
      intros cs1 cs2 lst MStep. induction MStep as [cs1 | cs1 cs2 cs2' a' MStep IHMStep] ; intros cs0 a Step.
      * rewrite <- app_nil_l with Event [a]. apply MultiStepTl_some with cs0. apply MultiStepTl_refl. assumption.
      * rewrite -> app_comm_cons. apply MultiStepTl_some with cs2 ; auto.
    Qed.

    Proposition mstep_iff_msteptl : forall cs0 cs1 lst, cs0 ==>*[lst] cs1 <-> MultiStepTl cs0 cs1 lst.
      intros cs0 cs1 lst. split ; intro MStep ; induction MStep.
      * apply MultiStepTl_refl.
      * apply mstep_tl_prepend with cs1 ; assumption.
      * apply MultiStep_refl.
      * apply mstep_append with cs1 ; assumption.
    Qed.

    Proposition mstep_rev_ind : forall (P : Cmd * Store -> Cmd * Store -> list Event -> Prop),
        (forall cs, P cs cs []) ->
        (forall cs0 cs1 cs2 a lst, cs0 ==>*[lst] cs1 -> P cs0 cs1 lst -> cs1 -->[a] cs2 -> P cs0 cs2 (lst ++ [a]))
        -> forall cs0 cs1 lst, cs0 ==>*[lst] cs1 -> P cs0 cs1 lst.
      intros P BaseCase IndCase cs0 cs2 lst MStep. apply mstep_iff_msteptl in MStep. induction MStep.
      * auto.
      * apply IndCase with cs1 ; try assumption. apply mstep_iff_msteptl. assumption.
    Qed.

    Proposition mstep_strong_ind : forall (P : Cmd * Store -> Cmd * Store -> list Event -> Prop),
        (forall cs, P cs cs [])
        -> (forall cs0 cs2 lst,
            (forall cs1 a lst1 lst2, lst = a :: lst1 ++ lst2
            -> cs0 ==>*[a :: lst1] cs1
            -> cs1 ==>*[lst2] cs2
            -> P cs1 cs2 lst2)
          -> cs0 ==>*[lst] cs2 -> P cs0 cs2 lst)
        -> forall cs0 cs1 lst, cs0 ==>*[lst] cs1 -> P cs0 cs1 lst.
      intros P BaseCase IndCase cs0 cs1 lst MStep.
      enough (forall cs' lst0 lst1, cs0 ==>*[lst0] cs' -> cs' ==>*[lst1] cs1 -> lst0 ++ lst1 = lst -> P cs' cs1 lst1)
        by (apply H with [] ; auto using MultiStep_refl).
      induction MStep ; intros cs' lst0 lst1 MStep0 MStep1 LstVal.
      * apply app_eq_nil in LstVal as [] ; subst.
        inversion MStep1 ; auto.
      * destruct lst0 ; simpl in * ; subst
        ; [apply IndCase ; auto ; intros ? ? lst0 ? LstVal MStep0' ? |]
        ; injection LstVal ; intros ; subst ; clear LstVal
        ; apply IHMStep with lst0 ; auto
        ; inversion MStep0 ; subst
        ; [inversion MStep0' ; subst |]
        ; subst_eq_steps
        ; assumption.
    Qed.

    (* Proposition mstep_strong_rev_ind : forall (P : Cmd * Store -> Cmd * Store -> list Event -> Prop),
        (forall cs, P cs cs [])
        -> (forall cs0 cs2 lst,
            (forall cs1 a lst1 lst2, lst = lst1 ++ lst2 ++ [a]
            -> cs0 ==>*[lst1] cs1
            -> cs1 ==>*[lst2 ++ [a]] cs2
            -> P cs0 cs1 lst1)
          -> cs0 ==>*[lst] cs2 -> P cs0 cs2 lst)
        -> forall cs0 cs1 lst, cs0 ==>*[lst] cs1 -> (P cs0 cs1 lst).
      intros P BaseCase IndCase cs0 cs1 lst MStep.
      enough (forall cs' lst0 lst1, cs0 ==>*[lst0] cs' -> cs' ==>*[lst1] cs1 -> lst0 ++ lst1 = lst -> P cs0 cs' lst0) by eauto.
      apply mstep_iff_msteptl in MStep ; induction MStep ; intros cs' lst0 lst1 MStep0 MStep1 LstVal.
      * apply app_eq_nil in LstVal as [] ; subst.
        inversion MStep1 ; auto.
      * destruct lst1 using rev_ind ; try rewrite -> app_nil_r in * ; subst
        ; [apply IndCase ; auto ; intros ? ? ? lst1 LstVal ? MStep1' |]
        ; rewrite -> app_assoc in LstVal ; apply app_inj_tail in LstVal as [] ; subst
        ; apply IHMStep with lst1 ; auto
        ; [inversion MStep1 ; subst |]
        ; apply mstep_iff_msteptl in MStep
        ; eauto using step_concat_inv.
    Qed. *)

    Inductive LoopsN : Expr -> Cmd -> Store -> Store -> nat -> Prop :=
      | LoopsN_none : forall e c s, LoopsN e c s s 0
      | LoopsN_some : forall e c s s' s'' lst n n',
          LoopsN e c s s' n
          -> evalExpr e s' = Some (S n')
          -> (c, s') ==>*[lst] (Skip, s'')
          -> LoopsN e c s s'' (S n).

    Inductive EvalCtx : (Cmd -> Cmd) -> Prop :=
      | SeqCtx : forall c2, EvalCtx (fun c1 => Seq c1 c2)
      | PDownCtx : forall l, EvalCtx (fun c => ProgDown l c).

    Lemma step_under_ctx : forall {c s c' s' lst},
        (c, s) ==>*[lst] (c', s')
        -> c' <> Stop
        -> forall {E}, EvalCtx E -> (E c, s) ==>*[lst] (E c', s').
      intros c s c' s' lst MStep. dependent induction MStep ; intros c'NotStop E CtxE.
      * apply MultiStep_refl.
      * destruct cs1 as [c1 s1]. apply MultiStep_some with (E c1, s1) ; auto.
        destruct CtxE ; [apply SeqCE | apply PDownCE] ; auto
        ; intro ; inversion MStep ; subst ; auto ; handle_simple_contradict.
    Qed.

    Lemma step_under_seq : forall c1 s c1' s' lst,
        (c1, s) ==>*[lst] (c1', s')
        -> c1' <> Stop
        -> forall c2, ((Seq c1 c2), s) ==>*[lst] ((Seq c1' c2), s').
      intros ? ? ? ? ? MStep c'NotStop c2.
      exact (step_under_ctx MStep c'NotStop (SeqCtx c2)).
    Qed.

    Lemma step_under_seq_skip : forall c1 s s' lst, (c1, s) ==>*[lst] (Skip, s')
        -> forall c2, (Seq c1 c2, s) ==>*[lst] (Seq Skip c2, s').
      intros. apply step_under_seq ; [assumption | discriminate].
    Qed.

    Lemma step_under_pdown : forall c s c' s' lst,
        (c, s) ==>*[lst] (c', s')
        -> c' <> Stop
        -> forall l, (ProgDown l c, s) ==>*[lst] (ProgDown l c', s').
      intros ? ? ? ? ? MStep c'NotStop l.
      exact (step_under_ctx MStep c'NotStop (PDownCtx l)).
    Qed.

    Lemma step_under_pdown_skip : forall c s s' lst, (c, s) ==>*[lst] (Skip, s')
        -> forall l, (ProgDown l c, s) ==>*[lst] (ProgDown l Skip, s').
      intros. apply step_under_pdown ; [assumption | discriminate].
    Qed.

    Lemma wt_mstep_under_ctx : forall G pc E c nt, EvalCtx E -> G;; pc |- E c -| nt
        -> forall s c' s' lst, (E c, s) ==>*[lst] (E c', s')
        -> c' <> Stop.
      intros G pc E c nt CtxE WTc s ? ? ? MStep ?. subst.
      assert (E Stop = Stop \/ G;; pc |- E Stop -| nt) as [| WTStop] by eauto using mstep_type_preservation
      ; destruct CtxE ; try discriminate
      ; revert WTStop ; clear ; intro WTStop ; dependent induction WTStop ; solve [eauto | handle_simple_contradict].
    Qed.

    Lemma step_concat : forall cs0 cs1 lst0,
        cs0 ==>*[lst0] cs1
        -> forall cs2 lst1, cs1 ==>*[lst1] cs2
        -> cs0 ==>*[lst0 ++ lst1] cs2.
      intros cs0 cs1 lst0 MStep0. dependent induction MStep0 ; intros cs2' lst1 MStep1 ; eauto using MultiStep_some.
    Qed.

    Lemma step_concat_inv : forall cs0 cs1 cs2 lst0 lst1, cs0 ==>*[lst0] cs1 -> cs0 ==>*[lst0 ++ lst1] cs2 -> cs1 ==>*[lst1] cs2.
      intros cs0 cs1 cs2 lst0 lst1 MStep0 MStepConcat.
      dependent induction MStep0 ; simpl in * ; auto.
      apply IHMStep0.
      inversion MStepConcat ; subst ; subst_eq_steps. assumption.
    Qed.

    Lemma seq_mstep_inv : forall {c1 c2 c' s s' lst}, (Seq c1 c2, s) ==>*[lst] (c', s')
        -> (exists c1', c' = Seq c1' c2 /\ (c1, s) ==>*[lst] (c1', s'))
            \/ (exists lst1' lst2' s'', (c1, s) ==>*[lst1'] (Skip, s'') /\ (c2, s'') ==>*[lst2'] (c', s')).
      intros ? ? ? ? ? ? MStep. dependent induction MStep ; eauto using MultiStep_refl.
      inversion H as [| ? c1' ? ? s'' | | | | | | |] ; subst.
      * pose proof (IHMStep c1' c2 c' s'' s' eq_refl eq_refl) as [(? & ? & ?) | (? & ? & ? & ? & ?)]
        ; [left ; subst | right] ; eauto 6 using MultiStep_some.
      * right. eauto using MultiStep_refl.
    Qed.

    Lemma wt_mstep_to_stop : forall G pc c nt s s' lst a,
        G;; pc |- c -| nt
        -> (c, s) ==>*[lst ++ [a]] (Stop, s')
        -> a = StopEvt.
      intros G pc c nt s s' lst a WTc MStep.
      apply mstep_iff_msteptl in MStep. inversion MStep ; subst.
      * apply app_cons_not_nil in H3. inversion H3.
      * apply mstep_iff_msteptl in H0. apply app_inj_tail in H. destruct H as [EqLst EQa]. subst.
        destruct cs1 as [c1 s1]. apply wt_step_to_stop_impl_stop_evt with G pc c1 nt s1 s' ; auto.
        assert (c1 = Stop \/ G;; pc |- c1 -| nt) as c1StopOrWt by (apply mstep_type_preservation with c s s1 lst ; auto).
        destruct c1StopOrWt as [c1Stop |] ; [subst ; inversion H3 | assumption].
    Qed.

    Lemma wt_mstep_to_stop_impl_skip : forall G pc c nt, G;; pc |- c -| nt
        -> forall s s' lst, (c, s) ==>*[lst] (Stop, s')
        -> exists lst', (c, s) ==>*[lst'] (Skip, s').
      intros G pc c nt WTc s s' lst MStep.
      dependent induction MStep using mstep_rev_ind ; [handle_simple_contradict |].
      destruct cs1 as [c' ?].
      enough (c' = Skip) by (inversion H ; subst ; eauto).
      apply wt_step_to_stop_impl_skip with G pc nt s0 a s' ; auto.
      apply mstep_type_pres_not_stop with c s s0 lst ; auto ; intro ; handle_simple_contradict.
    Qed.

    Lemma conv_with_step_to_skip : forall G pc c nt s, G;; pc |- c -| nt
        -> forall lst s', (c, s) ==>*[lst] (Stop, s') -> exists lst', (c, s) ==>*[lst'] (Skip, s').
      intros G pc c nt s WTc lst s' MStep.
      dependent induction MStep using mstep_rev_ind.
      * handle_simple_contradict.
      * destruct cs1 as [c1 s1].
        enough (c1 = Skip) by (subst ; inversion H ; subst ; eauto).
        eapply wt_step_to_stop_impl_skip ; try eapply mstep_type_pres_not_stop ; eauto.
        intro ; subst ; inversion H.
    Qed.

    Lemma conv_step_to_skip : forall G pc c nt s, G;; pc |- c -| nt
        -> converge c s -> exists lst s', (c, s) ==>*[lst] (Skip, s').
      intros ? ? ? ? ? ? (? & ? & Conv).
      eapply conv_with_step_to_skip in Conv as [? ?] ; eauto.
    Qed.

    Lemma deterministic_mstep : forall cs cs0 lst0, cs ==>*[lst0] cs0
        -> forall cs1 lst1, cs ==>*[lst1] cs1
        -> length lst0 = length lst1
        -> cs0 = cs1 /\ lst0 = lst1.
      intros cs cs0 lst0 MStep0. dependent induction MStep0
      ; intros cs1' lst1 MStep1 LstLenEq ; dependent destruction MStep1 ; simpl in *
      ; inversion LstLenEq as [LstLenEq']; auto.
      subst_eq_steps.
      pose proof (IHMStep0 cs4 lst0 MStep1 LstLenEq') as [? ?] ; subst.
      auto.
    Qed.

    Lemma deterministic_conv_state : forall cs0 s0 s1 lst0, cs0 ==>*[lst0] (Stop, s0)
        -> forall lst1, cs0 ==>*[lst1] (Stop, s1) -> s0 = s1.
      intros ? ? ? ? MStep0. dependent induction MStep0 ; intros ? MStep1 ; dependent destruction MStep1
      ; try subst_eq_steps ; try apply f_equal ; eauto
      ; try lazymatch goal with
        | [H : (Stop, _) -->[_] _ |- _] => inversion H
      end.
    Qed.

    Lemma desterministic_conv_trace : forall cs0 s0 s1 lst0, cs0 ==>*[lst0] (Stop, s0)
        -> forall lst1, cs0 ==>*[lst1] (Stop, s1) -> lst0 = lst1.
      intros ? ? ? ? MStep0. dependent induction MStep0 ; intros ? MStep1 ; dependent destruction MStep1
      ; try subst_eq_steps ; try apply f_equal ; eauto
      ; lazymatch goal with
        | [H : (Stop, _) -->[_] _ |- _] => inversion H
      end.
    Qed.

    Lemma prefix_mstep : forall lst0 lst1, Prefix lst0 lst1
        -> forall cs0 cs1, cs0 ==>*[lst1] cs1
        -> exists cs1', cs0 ==>*[lst0] cs1'.
      intros lst0 lst1 Pfx.
      induction Pfx ; intros cs0 cs1 MStep ; eauto using MultiStep.
      inversion MStep as [| ? cs0' ? ? ? ? MStep'] ; subst.
      destruct (IHPfx cs0' cs1 MStep').
      eauto using MultiStep_some.
    Qed.

    Lemma mstep_prefix : forall cs cs0 lst0, cs ==>*[lst0] cs0
        -> forall cs1 lst1, cs ==>*[lst1] cs1
        -> (exists lst0', cs0 ==>*[lst0'] cs1) \/ (exists lst1', cs1 ==>*[lst1'] cs0).
      intros cs cs0 lst0 MStep0. dependent induction MStep0 ; intros cs1' lst1 MStep1 ; eauto.
      dependent destruction MStep1 ; eauto using MultiStep_some.
      subst_eq_steps.
      pose proof (IHMStep0 cs4 lst0 MStep1) as [|] ; [left | right] ; assumption.
    Qed.

    Lemma mstep_prefix_lst : forall cs cs0 lst0, cs ==>*[lst0] cs0
        -> forall cs1 lst1, cs ==>*[lst1] cs1
        -> Prefix lst0 lst1 \/ Prefix lst1 lst0.
      intros cs cs0 lst0 MStep0. dependent induction MStep0
      ; intros cs1' lst1 MStep1 ; dependent destruction MStep1 ; simpl ; eauto.
      subst_eq_steps.
      pose proof (IHMStep0 cs4 lst0 MStep1) as [? | ?] ; auto.
    Qed.

    Lemma mstep_prefix_stop : forall cs cs0 lst0, cs ==>*[lst0] cs0
        -> forall s1 lst1, cs ==>*[lst1] (Stop, s1)
        -> exists lst0', cs0 ==>*[lst0'] (Stop, s1) /\ lst0 ++ lst0' = lst1.
      intros cs cs0 lst MStep0. dependent induction MStep0 ; intros s1 lst1 MStep1
      ; dependent destruction MStep1
      ; try handle_simple_contradict ; try subst_eq_steps ; eauto using MultiStep.
      pose proof (IHMStep0 s1 lst0 MStep1) as (lst0' & ? & ?) ; subst ; simpl ; eauto.
    Qed.

    Lemma mstep_split : forall lst0 cs0 cs1 lst1, cs0 ==>*[lst0 ++ lst1] cs1 ->
        exists cs2, cs0 ==>*[lst0] cs2 /\ cs2 ==>*[lst1] cs1.
      induction lst0 ; intros cs0 cs1 lst1 MStep ; simpl in *.
      * eauto using MultiStep_refl.
      * inversion MStep as [| ? cs2 ? ? ? ? MStep'] ; subst.
        pose proof (IHlst0 cs2 cs1 lst1 MStep') as (cs2' & ? & ?).
        eauto using MultiStep.
    Qed.

    Lemma mstep_seq_inv : forall {c1 c2 s c' s' lst}, (Seq c1 c2, s) ==>*[lst] (c', s')
        -> (exists c1', (c1, s) ==>*[lst] (c1', s') /\ c' = Seq c1' c2)
            \/ (exists lst0 lst1 s'', (c1, s) ==>*[lst0] (Skip, s'') /\ (c2, s'') ==>*[lst1] (c', s') /\ lst = lst0 ++ NoEvt :: lst1).
      intros c1 c2 s c' s' lst MStep. dependent induction MStep.
      * left. eauto using MultiStep_refl.
      * inversion H as [| ? c1' ? ? s''| | | | | | |] ; subst.
        - destruct (IHMStep c1' c2 s'' c' s' eq_refl eq_refl) as [(? & ? & ?) | (? & ? & ? & ? & ? & ?)] ; subst
          ; [left | right]
          ; eauto 7 using MultiStep_some.
        - right. eauto 6 using MultiStep_refl.
    Qed.

    Lemma mstep_pdown_inv : forall {l c_in s c' s' lst}, (ProgDown l c_in, s) ==>*[lst] (c', s')
        -> (exists c_in', (c_in, s) ==>*[lst] (c_in', s') /\ c' = ProgDown l c_in')
            \/ (exists lst', (c_in, s) ==>*[lst'] (Skip, s') /\ (lst = lst' ++ [PDownEvt l] \/ lst = lst' ++ [PDownEvt l ; StopEvt])).
      intros l c_in s c' s' lst MStep. dependent induction MStep.
      * left. eauto using MultiStep_refl.
      * inversion H as [| | | | | | | ? c_in' ? s'' |] ; subst.
        - destruct (IHMStep l c_in' s'' c' s' eq_refl eq_refl) as [(? & ? & ?) | (lst' & ? & [|])]
          ; [left | right | right] ; subst ; eauto using MultiStep_some.
        - right.
          inversion MStep as [| ? ? ? ? ? Step MStep'] ; subst ; eauto using MultiStep_refl.
          inversion Step ; subst.
          inversion MStep' ; subst ; try handle_simple_contradict.
          eauto using MultiStep_refl.
    Qed.

    Lemma mstep_to_stop_evt : forall cs c s lst, cs ==>*[lst ++ [StopEvt]] (c, s) -> c = Stop.
      intros ? ? ? ? MStep. dependent induction MStep using mstep_rev_ind.
      * apply app_cons_not_nil in x ; inversion x.
      * apply app_inj_tail in x ; destruct x ; subst.
        apply step_stop_evt_impl_stop with cs1 s ; eauto.
    Qed.

    Lemma wt_mstep_stop_lst : forall {G pc c nt}, G;; pc |- c -| nt -> forall {s s' lst}, (c, s) ==>*[lst] (Stop, s')
        -> exists lst', lst = lst' ++ [StopEvt].
      intros G pc c nt WTc s s' lst MStep. dependent induction MStep.
      * handle_simple_contradict.
      * destruct cs1 as [c1 s1].
        assert (c1 = Stop \/ G;; pc |- c1 -| nt) as [| WTc1] by eauto using type_preservation ; [subst |].
        - assert (a = StopEvt) by eauto using wt_step_to_stop_impl_stop_evt ; subst.
          inversion MStep as [| ? ? ? ? ? Step] ; [| inversion Step] ; subst.
          exists []. auto.
        - pose proof (IHMStep c1 WTc1 s1 s' eq_refl eq_refl) as [lst' ?].
          exists (a :: lst'). subst. auto.
    Qed.

    Lemma stop_evt_at_conv : forall cs c' s' lst, cs ==>*[lst] (c', s') -> c' <> Stop -> ~ In StopEvt lst.
      intros ? c' s' ? MStep c'NotStop. dependent induction MStep ; simpl ; auto.
      destruct cs1 as [c1 s1].
      intro H' ; destruct H' as [|] ; [subst | contradiction IHMStep with c' s' ; reflexivity].
      contradict c'NotStop.
      assert (c1 = Stop) by eauto using step_stop_evt_impl_stop ; subst.
      inversion MStep ; auto ; handle_simple_contradict.
    Qed.

    Lemma mstep_only_last_stop : forall cs0 cs1 lst a, cs0 ==>*[lst ++ [a]] cs1 -> ~ In StopEvt lst.
      intros ? ? ? ? MStep.
      invert_tail mstep_iff_msteptl MStep.
      induction H0 ; unfold In ; auto.
      intro IsIn ; destruct IsIn as [| BadIn] ; subst.
      * destruct cs2 as [c ?].
        assert (c = Stop) by eauto using step_stop_evt_impl_stop ; subst.
        inversion H0 ; subst ; handle_simple_contradict.
      * apply IHMultiStep ; eauto using mstep_append.
    Qed.

    Lemma never_stuck_step : forall {c s}, never_stuck c s -> forall {c' s' a}, (c, s) -->[a] (c', s') -> never_stuck c' s'.
      unfold never_stuck. intros. eauto using MultiStep_some.
    Qed.

    Lemma never_stuck_mstep : forall {c s}, never_stuck c s -> forall {c' s' lst}, (c, s) ==>*[lst] (c', s') -> never_stuck c' s'.
      unfold never_stuck. intros. eauto using step_concat.
    Qed.

    Lemma never_stuck_ctx : forall {E c s}, EvalCtx E -> never_stuck (E c) s -> never_stuck c s.
      unfold never_stuck. intros E c s ECtx NStuck c' s' lst MStep.
      destruct (cmd_stop_dec c') as [| c'NotStop] ; [auto |].
      assert ((E c, s) ==>*[lst] (E c', s')) as EMStep by auto using step_under_ctx.
      destruct (NStuck (E c') s' lst EMStep) as [| (? & ? & Step)]
      ; dependent destruction ECtx ; solve [discriminate | inversion Step ; eauto using StopE].
    Qed.

    Lemma never_stuck_seq_c1 : forall {c1 c2 s}, never_stuck (Seq c1 c2) s -> never_stuck c1 s.
      intros ? c2 ? NStuck.
      exact (never_stuck_ctx (SeqCtx c2) NStuck).
    Qed.

    Lemma nstuck_stop_step_seq : forall c1 c2 s a s', (c1, s) -->[a] (Stop, s')
        -> never_stuck (Seq c1 c2) s
        -> c1 = Skip.
      induction c1 ; intros ? ? ? ? Step NStuck ; auto ; inversion Step ; subst
      ; lazymatch goal with
        | [Step : (If ?e ?c1' ?c2', _) --> (Stop, _) |- _] => set (c1 := If e c1' c2')
        | [Step : (Seq Skip Stop, _) --> (Stop, _) |- _] => set (c1 := Seq Skip Stop)
      end
      ; unfold never_stuck in NStuck
      ; (specialize (NStuck (Seq c1 c2) s' [] (MultiStep_refl (Seq c1 c2, s')))
          as [| (? & a & BadStep)]
        ; [discriminate |])
      ; inversion BadStep ; subst ; subst c1
      ; lazymatch goal with
        | [H : ?c <> Stop, Step0 : ?cs -->[?a0] (Stop, ?s0), Step1 : ?cs -->[?a1] (?c, ?s1) |- _]
          => contradict H ; assert ((Stop, s0) = (c, s1) /\ (a0 = a1)) as [csEq ?] by eauto using deterministic_step
             ; injection csEq ; auto
      end.
    Qed.

    Lemma nstuck_stop_mstep_seq : forall c1 c2 s lst s', never_stuck (Seq c1 c2) s
        -> (c1, s) ==>*[lst] (Stop, s')
        -> exists lst', (c1, s) ==>*[lst'] (Skip, s').
      intros c1 c2 s lst s' NStuck MStep.
      dependent induction MStep ; [| destruct cs1 as [c1' s1]].
      * unfold never_stuck in NStuck.
        specialize (NStuck (Seq Stop c2) s' [] (MultiStep_refl (Seq Stop c2, s'))).
        destruct NStuck as [| (? & ? & SeqStep)] ; [discriminate | inversion SeqStep ; handle_simple_contradict].
      * assert (c1 = Skip \/ c1 <> Skip) as [| c1NotSkip] by (induction c1 ; try (left ; reflexivity) ; right ; discriminate).
        - subst. inversion H ; subst.
          inversion MStep ; subst ; [exists [] ; apply MultiStep_refl | handle_simple_contradict].
        - assert (exists lst, (c1', s1) ==>*[lst] (Skip, s')) as [? ?]
            by (apply IHMStep ; auto
                ; eapply never_stuck_step ; [| apply SeqCE] ; eauto
                ; intro ; subst
                ; eauto using nstuck_stop_step_seq).
          eauto using MultiStep.
    Qed.

    Lemma never_stuck_under_pdown : forall {l c s}, never_stuck (ProgDown l c) s -> never_stuck c s.
      intros l ? ? NStuck.
      exact (never_stuck_ctx (PDownCtx l) NStuck).
    Qed.

    Lemma nstuck_stop_step_pdown : forall c l s a s', (c, s) -->[a] (Stop, s')
        -> never_stuck (ProgDown l c) s
        -> c = Skip.
      induction c ; intros ? ? ? ? Step NStuck ; auto ; inversion Step ; subst
      ; lazymatch goal with
        | [Step : (If ?e ?c1' ?c2', _) --> (Stop, _) |- _] => set (c := If e c1' c2')
        | [Step : (Seq Skip Stop, _) --> (Stop, _) |- _] => set (c := Seq Skip Stop)
      end
      ; unfold never_stuck in NStuck
      ; (specialize (NStuck (ProgDown l c) s' [] (MultiStep_refl (ProgDown l c, s'))) as [| (? & a & BadStep)]
        ; [discriminate |])
      ; inversion BadStep ; subst ; subst c
      ; lazymatch goal with
        | [H : ?c <> Stop, Step0 : ?cs -->[?a0] (Stop, ?s0), Step1 : ?cs -->[?a1] (?c, ?s1) |- _]
          => contradict H ; assert ((Stop, s0) = (c, s1) /\ (a0 = a1)) as [csEq ?] by eauto using deterministic_step
             ; injection csEq ; auto
      end.
    Qed.

    Lemma nstuck_stop_mstep_pdown : forall l c s lst s', never_stuck (ProgDown l c) s
        -> (c, s) ==>*[lst] (Stop, s')
        -> exists lst', (c, s) ==>*[lst'] (Skip, s').
      intros l c s lst s' NStuck MStep.
      dependent induction MStep ; [| destruct cs1 as [c1 s1]].
      * unfold never_stuck in NStuck.
        specialize (NStuck (ProgDown l Stop) s' [] (MultiStep_refl (ProgDown l Stop, s'))) as [| (? & ? & BadStep)]
        ; [discriminate | inversion BadStep ; handle_simple_contradict].
      * assert (c = Skip \/ c <> Skip) as [| cNotSkip] by (induction c ; try (left ; reflexivity) ; right ; discriminate).
        - subst. inversion H ; subst.
          inversion MStep ; subst ; [exists [] ; apply MultiStep_refl | handle_simple_contradict].
        - assert (exists lst, (c1, s1) ==>*[lst] (Skip, s')) as [? ?]
            by (apply IHMStep ; auto
                ; eapply never_stuck_step ; [| apply PDownCE] ; eauto
                ; intro ; subst
                ; eauto using nstuck_stop_step_pdown).
          eauto using MultiStep.
    Qed.

  End MStepTheories.

  Section LoopTheories.

    Lemma loops_lt : forall n e c s s', LoopsN e c s s' n -> forall m, m <= n -> exists s'', LoopsN e c s s'' m.
      induction n ; intros e c s s' Loops m LeM ; inversion LeM ; subst ; eauto.
      inversion Loops ; subst ; eauto.
    Qed.

    Inductive LoopsNLst : Expr -> Cmd -> Store -> Store -> list Event -> nat -> Prop :=
      | LoopsNLst_none : forall e c s, LoopsNLst e c s s [] 0
      | LoopsNLst_some : forall e c s s' s'' lst lst' n n',
          LoopsNLst e c s s' lst n
          -> evalExpr e s' = Some (S n')
          -> (c, s') ==>*[lst'] (Skip, s'')
          -> LoopsNLst e c s s'' (lst ++ NoEvt :: NoEvt :: lst' ++ [NoEvt]) (S n).

    Lemma loopsn_loopsnlst : forall {e c s s' n}, LoopsN e c s s' n -> exists lst, LoopsNLst e c s s' lst n.
      intros ? ? ? ? ? Loops. dependent induction Loops ; [| destruct IHLoops] ; eauto using LoopsNLst.
    Qed.

    Lemma loopsnlst_loopsn : forall {e c s s' lst n}, LoopsNLst e c s s' lst n -> LoopsN e c s s' n.
      intros ? ? ? ? ? ? Loops. dependent induction Loops ; eauto using LoopsN.
    Qed.

    Lemma loops_lst_len : forall {e c s lst s' n}, LoopsNLst e c s s' lst n -> n <= length lst.
      intros ? ? ? ? ? ? Loops. dependent induction Loops ; try rewrite -> app_length ; simpl ; auto.
      rewrite -> Nat.add_succ_r.
      apply le_n_S.
      assert (forall m, n <= length lst + m) by (induction m ; [rewrite -> Nat.add_0_r | rewrite -> Nat.add_succ_r] ; auto).
      auto.
    Qed.

    Lemma loops_lst_step : forall {e c s lst s' n}, LoopsNLst e c s s' lst n -> (While e c, s) ==>*[lst] (While e c, s').
      intros n e c s s' lst Loops. dependent induction Loops ; auto using MultiStep_refl.
      eapply step_concat ; eauto.
      do 2 (eapply MultiStep_some ; eauto using WhileE, IfNE).
      apply step_concat with (Seq Skip (While e c), s'') ; eauto using MultiStep, SeqSkipE.
      apply step_under_seq ; [assumption | discriminate].
    Qed.

    Lemma loops_step : forall {n e c s s'}, LoopsN e c s s' n -> exists lst, (While e c, s) ==>*[lst] (While e c, s').
      intros ? ? ? ? ? Loops. destruct (loopsn_loopsnlst Loops) ; eauto using loops_lst_step.
    Qed.

    Lemma loopsN_prepend : forall n e c s s' lst s'' n', evalExpr e s = Some (S n')
        -> (c, s) ==>*[lst] (Skip, s'')
        -> LoopsN e c s'' s' n
        -> LoopsN e c s s' (S n).
      induction n ; intros e c s s' lst s'' n' EvalExpr MStep Loops ; inversion Loops ; subst ; eauto using LoopsN.
    Qed.

  End LoopTheories.

  Section ConvergeDiverge.

    Lemma diverge_impl_never_stuck : forall c s, diverge c s -> never_stuck c s.
      unfold diverge, never_stuck. intros ? ? Divg ? ? ? MStep.
      apply Divg in MStep.
      destruct MStep as ([? ?] & ? & ?).
      eauto.
    Qed.

    Lemma converge_impl_never_stuck : forall c s, converge c s -> never_stuck c s.
      unfold converge, never_stuck. intros c s (lst & sFin & ConvStep) c' s' lst' MStep.
      destruct (mstep_prefix (c, s) (Stop, sFin) lst ConvStep (c', s') lst' MStep) as [[? MStep'] | [? MStep']]
      ; inversion MStep' ; eauto ; try handle_simple_contradict.
    Qed.

    Lemma diverge_step : forall c0 s0 c1 s1 a, diverge c0 s0 -> (c0, s0) -->[a] (c1, s1) -> diverge c1 s1.
      unfold diverge. intros. eauto using MultiStep_some.
    Qed.

    Lemma diverge_mstep : forall c0 s0 c1 s1 lst, diverge c0 s0 -> (c0, s0) ==>*[lst] (c1, s1) -> diverge c1 s1.
      unfold diverge. intros. eauto using step_concat.
    Qed.

    Lemma step_diverge : forall c0 s0 c1 s1 a, diverge c1 s1 -> (c0, s0) -->[a] (c1, s1) -> diverge c0 s0.
      unfold diverge. intros c0 s0 c1 s1 a Divgc1 Step cs1 lst MStep.
      dependent induction MStep ; [| subst_eq_steps ; apply Divgc1 in MStep ; destruct MStep as (? & ? & ?)] ; eauto.
    Qed.

    Lemma converge_step : forall c0 s0 c1 s1 a, converge c0 s0 -> (c0, s0) -->[a] (c1, s1) -> converge c1 s1.
      unfold converge. intros ? ? ? ? ? (? & ? & MStep) ?.
      inversion MStep ; subst ; [handle_simple_contradict | subst_eq_steps] ; eauto.
    Qed.

    Lemma step_converge : forall c0 s0 c1 s1 a, converge c1 s1 -> (c0, s0) -->[a] (c1, s1) -> converge c0 s0.
      unfold converge. intros ? ? ? ? ? (? & ? & ?) ?. eauto using MultiStep_some.
    Qed.

    Lemma converge_impl_not_diverge : forall c s, converge c s -> ~ diverge c s.
      unfold converge, diverge. intros ? ? (lst & s' & Conv) Divg.
      pose proof (Divg (Stop, s') lst Conv) as (? & ? & StopSteps). inversion StopSteps.
    Qed.

    Lemma diverge_impl_not_converge : forall c s, diverge c s -> ~ converge c s.
      unfold converge, diverge. intros ? ? Divg (lst & s' & Conv).
      pose proof (Divg (Stop, s') lst Conv) as (? & ? & StopSteps). inversion StopSteps.
    Qed.

    Lemma nstuck_not_converge_impl_diverge : forall c s, never_stuck c s -> ~ converge c s -> diverge c s.
      unfold never_stuck, converge, diverge.
      intros c s NStuck NConv [c' s'] lst MStep.
      destruct (NStuck c' s' lst MStep) as [| (? & ? & ?)] ; [subst ; contradict NConv |] ; eauto.
    Qed.

    Lemma while_divg_step_to_seq : forall e c s, diverge (While e c) s -> (While e c, s) ==>*[[NoEvt ; NoEvt]] (Seq c (While e c), s).
      intros e c s Divg.
      apply MultiStep_some with (If e (Seq c (While e c)) Skip, s) ; auto using WhileE.
      apply MultiStep_some with (Seq c (While e c), s) ; auto using MultiStep_refl.
      unfold diverge in Divg.
      assert (exists cs a, (If e (Seq c (While e c)) Skip, s) -->[a] cs) as (? & ? & IfStep) by eauto using MultiStep, WhileE.
      inversion IfStep ; subst ; [assumption |].
      assert (exists cs a, (Stop, s) -->[a] cs) as (? & ? & StopStep) by eauto using MultiStep, OneStep.
      inversion StopStep.
    Qed.

    Lemma while_divg_never_stuck : forall e c s, diverge (While e c) s
        -> forall n s', LoopsN e c s s' n -> never_stuck c s'.
      unfold diverge, never_stuck. intros e c s Divg n s' Loops c' s'' lst MStep.
      destruct (cmd_stop_dec c') as [| c'NotStop] ; [left ; assumption | right].
      assert (exists lst, (While e c, s) ==>*[lst] (While e c, s')) as [lst' MStepWhiles] by eauto using loops_step.
      assert (exists cs a, (Seq c' (While e c), s'') -->[a] cs) as (? & ? & SeqStep)
        by (apply Divg with (lst' ++ [NoEvt ; NoEvt] ++ lst)
            ; apply step_concat with (While e c, s') ; auto
            ; apply step_concat with (Seq c (While e c), s')
            ; [apply while_divg_step_to_seq ; unfold diverge ; intros ; eauto using step_concat
              | apply step_under_seq ; auto ]).
      dependent destruction SeqStep ; eauto using StopE.
    Qed.

    Lemma ctx_conv_to_skip : forall {E c s}, EvalCtx E -> converge (E c) s -> exists lst s', (c, s) ==>*[lst] (Skip, s').
      unfold converge. intros E c s CtxE (? & ? & MStep).
      dependent induction MStep.
      * destruct CtxE ; discriminate.
      * destruct cs1 as [? s'].
        destruct CtxE
        ; (inversion H as [| ? c' | | | | | | ? c' |] ; subst
            ; [assert (exists lst s1, (c', s') ==>*[lst] (Skip, s1)) as (? & ? & ?) by eauto using EvalCtx |])
        ; eauto using MultiStep.
    Qed.

    Lemma ctx_converge : forall {E c s}, EvalCtx E -> converge (E c) s -> converge c s.
      intros ? ? ? CtxE Conv. unfold converge.
      destruct (ctx_conv_to_skip CtxE Conv) as (? & ? & ?).
      eauto using mstep_append, StopE.
    Qed.

    Lemma seq_conv_to_skip : forall c1 c2 s, converge (Seq c1 c2) s -> exists lst s', (c1, s) ==>*[lst] (Skip, s').
      intros ? c2 ? Conv. exact (ctx_conv_to_skip (SeqCtx c2) Conv).
    Qed.

    Lemma seq_converge_c1 : forall c1 c2 s, converge (Seq c1 c2) s -> converge c1 s.
      intros ? c2 ? Conv. exact (ctx_converge (SeqCtx c2) Conv).
    Qed.

    Lemma pdown_converge : forall l c s, converge (ProgDown l c) s -> converge c s.
      intros l ? ? Conv. exact (ctx_converge (PDownCtx l) Conv).
    Qed.

    Lemma ctx_diverge : forall {E c s}, EvalCtx E -> diverge c s
        -> forall {lst cs}, (E c, s) ==>*[lst] cs
        -> exists cs', (c, s) ==>*[lst] cs'.
      intros ? ? ? CtxE Divg ? ? MStep.
      dependent induction MStep ; eauto using MultiStep_refl.
      destruct CtxE ; inversion H ; subst
      ; try (assert (exists cs a, (Stop, s) -->[a] cs) as (? & ? & ?) by eauto using MultiStep, StopE ; handle_simple_contradict)
      ; [set (E := fun c => Seq c c2) | set (E := fun c => ProgDown l c)]
      ; lazymatch goal with
        | [H : (c, s) -->[_] (?c', ?s') |- _]
          => assert (exists cs', (c', s') ==>*[lst] cs') as [cs' ?]
              by (apply IHMStep with E ; eauto using diverge_step, EvalCtx)
            ; eauto using MultiStep_some
      end.
    Qed.

    Lemma one_loop_two_loop : forall e c s, diverge (While e c) s
        -> forall s' n, LoopsN e c s s' n
        -> converge c s'
        -> exists s'', LoopsN e c s s'' (S n).
      unfold converge. intros e c s Divg s' n Loops (lst0 & s'' & ?).
      assert (exists lst, (While e c, s) ==>*[lst] (While e c, s')) as [lst' ?] by eauto using loops_step.
      assert (diverge (While e c) s') by (unfold diverge in * ; eauto using step_concat).
      assert (exists lst, (c, s') ==>*[lst] (Skip, s'')) as [lst MStep]
        by (apply nstuck_stop_mstep_seq with (While e c) lst0
            ; [eapply never_stuck_mstep |]
            ; eauto using diverge_impl_never_stuck, while_divg_step_to_seq).
      assert ((While e c, s') ==>*[[NoEvt ; NoEvt]] (Seq c (While e c), s')) as LastTwoSteps by auto using while_divg_step_to_seq.
      inversion LastTwoSteps as [| ? [c1 s1] ? ? ? WhileStep LastStep] ; subst.
      inversion WhileStep ; subst.
      inversion LastStep as [| ? [c2 s2] ? ? ? IfStep] ; subst.
      inversion IfStep ; subst.
      * exists s''. apply LoopsN_some with s2 lst n0 ; auto.
      * assert (exists cs a, (Stop, s2) -->[a] cs) as (? & ? & StopStep) by (unfold diverge in * ; eauto using MultiStep, OneStep).
        inversion StopStep.
    Qed.

    Lemma fin_loops_converge : forall {e c s s' n}, LoopsN e c s s' n -> evalExpr e s' = Some 0 -> converge (While e c) s.
      intros e c s s' n Loops ?.
      pose proof (loops_step Loops) as [lst MStep].
      unfold converge.
      exists (lst ++ [NoEvt ; NoEvt ; StopEvt]) ; exists s'.
      eapply step_concat ; eauto using step_concat, MultiStep, WhileE, If0E, StopE.
    Qed.

    Lemma conv_while_fin_loops : forall e c s, converge (While e c) s -> exists n s', LoopsN e c s s' n /\ evalExpr e s' = Some 0.
      unfold converge. intros e c s (lst & s' & MStep).
      dependent induction MStep using mstep_strong_ind.
      inversion MStep as [| ? ? ? ? ? WhileStep MStep'] ; subst.
      inversion WhileStep ; subst ; inversion MStep' as [| ? ? ? ? ? IfStep] ; subst ; inversion IfStep ; subst
      ; eauto using LoopsN_none.
      assert (exists lst s', (c, s) ==>*[lst] (Skip, s')) as (lst' & s'' & ?)
        by (apply seq_conv_to_skip with (While e c) ; unfold converge ; eauto using MultiStep_some, IfNE, WhileE).
      assert ((While e c, s) ==>*[(NoEvt :: NoEvt :: lst') ++ [NoEvt]] (While e c, s''))
        by (apply mstep_append with (Seq Skip (While e c), s'') ; auto using SeqSkipE
            ; apply MultiStep_some with (If e (Seq c (While e c)) Skip, s) ; auto using WhileE
            ; apply MultiStep_some with (Seq c (While e c), s) ; eauto using IfNE
            ; apply step_under_seq ; auto ; discriminate).
      simpl in *.
      assert (exists lst'', (While e c, s'') ==>*[lst''] (Stop, s') /\ (NoEvt :: NoEvt :: lst' ++ [NoEvt]) ++ lst'' = NoEvt :: NoEvt :: lst)
          as (lst'' & ? & LstVal) by eauto using mstep_prefix_stop.
      simpl in LstVal ; rewrite <- app_assoc in LstVal ; injection LstVal ; clear LstVal ; intro LstVal.
      assert (exists n s', LoopsN e c s'' s' n /\ evalExpr e s' = Some 0) as (n' & sFin & ? & ?)
        by (apply H with (While e c, s'') NoEvt (NoEvt :: lst' ++ [NoEvt]) lst'' s' ; auto
            ; simpl ; repeat apply f_equal ; rewrite <- app_assoc ; simpl ; auto).
      exists (S n'). exists sFin. split ; auto.
      eauto using loopsN_prepend.
    Qed.

  End ConvergeDiverge.

End ImpTheories.
