Require Import Basic.
Require Import ImpDefs SecurityDefs Trace.
Require Import Tactics.
Require Import BasicTheories ImpTheories TraceTheories SecurityTheories ConvergeDiverge Containment.

From Coq Require Import Basics Equality List Ensembles.

Import ListNotations.

Module Type BridgeStep (B : Basic) (ID : ImpDefs) (TD : TraceDefs B ID) (SD : SecurityDefs B ID TD)
    (Tac : SimpleTactics B ID TD SD) (BT : BasicTheories B) (IT : ImpTheories B ID TD SD Tac BT) (TT : TraceTheories B ID TD SD Tac BT IT)
    (ST : SecurityTheories B ID TD SD Tac BT IT TT) (CD : ConvergeDiverge B ID TD SD Tac BT IT)
    (Con : Containment B ID TD SD Tac BT IT TT ST CD).
  Import B ID SD TD Tac BT IT ST Con.
  Import ImpNotations.

  Section BStepSec.

    Variable G : Varname -> option Label.
    Variable D : Ensemble Label.

    Inductive BridgeStep : Cmd * Store -> Cmd * Store -> Event -> nat -> Prop :=
      | BStep_0 : forall cs0 cs1 a, cs0 -->[a] cs1 -> ~ deq_evt G D a NoEvt -> BridgeStep cs0 cs1 a 0
      | BStep_n : forall cs0 cs1 cs2 a0 a1 n,
          cs0 -->[a0] cs1 -> deq_evt G D a0 NoEvt -> BridgeStep cs1 cs2 a1 n -> BridgeStep cs0 cs2 a1 (S n).

    Lemma deterministic_bstep : forall cs cs0 a0 n0,
        BridgeStep cs cs0 a0 n0
        -> forall cs1 a1 n1, BridgeStep cs cs1 a1 n1
        -> cs0 = cs1 /\ a0 = a1 /\ n0 = n1.
      intros cs cs0 a0 n0 BStep0. dependent induction BStep0 ; intros cs1' a1' n1 BStep1.
      * inversion BStep1 ; subst.
        - assert (cs1 = cs1' /\ a = a1') as [? ?] by eauto using deterministic_step ; subst ; auto.
        - assert (cs1 = cs3 /\ a = a0) as [? ?] by eauto using deterministic_step ; subst.
          handle_simple_contradict.
      * inversion BStep1 as [| ? ? ? ? ? ? ? ? BStep'] ; subst.
        - assert (cs1 = cs1' /\ a0 = a1') as [? ?] by eauto using deterministic_step ; subst.
          handle_simple_contradict.
        - assert (cs1 = cs4 /\ a0 = a2) as [? ?] by eauto using deterministic_step ; subst.
          pose proof (IHBStep0 cs1' a1' n0 BStep') as (? & ? & ?) ; subst. auto.
    Qed.

    Lemma mstep_lst_from_bstep : forall {cs0 cs1 a n}, BridgeStep cs0 cs1 a n
        -> exists lst, cs0 ==>*[lst ++ [a]] cs1 /\ deq_evt_lst G D lst [].
      intros cs0 cs1 a n BStep. dependent induction BStep.
      * exists []. simpl. eauto using MultiStep with deq.
      * destruct IHBStep as (lst & ? & ?).
        exists (a0 :: lst). eauto using MultiStep_some, DEqNoEvtL.
    Qed.

    Corollary mstep_from_bstep : forall cs0 cs1 a n, BridgeStep cs0 cs1 a n
        -> exists lst, cs0 ==>*[lst ++ [a]] cs1.
      intros cs0 cs1 a n BStep. pose proof (mstep_lst_from_bstep BStep) as (? & ? & ?). eauto.
    Qed.

    Lemma bstep_from_mstep : forall cs0 cs1 lst, cs0 ==>*[lst] cs1
        -> deq_evt_lst G D lst []
        -> forall cs2 a, cs1 -->[a] cs2
        -> ~ deq_evt G D a NoEvt
        -> exists n, BridgeStep cs0 cs2 a n.
      intros cs0 cs1 lst MStep.
      induction MStep as [| cs0 cs1' cs1 a0 lst Step1] ; intros DEqLst cs2 a Step2 NDEqA
      ; [exists 0 | inversion DEqLst ; assert (exists n, BridgeStep cs1' cs2 a n) as [n ?] by eauto ; exists (S n)]
      ; econstructor ; eauto.
    Qed.

    Lemma bstep_mstep : forall n cs0 cs1 lst, cs0 ==>*[lst] cs1
        -> deq_evt_lst G D lst []
        -> forall cs2 a, BridgeStep cs0 cs2 a n
        -> exists n', BridgeStep cs1 cs2 a n' /\ n' = n - length lst.
      induction n ; intros cs0 cs1 lst MStep ; induction MStep as [ | cs0 cs1' cs1 a' ] ; intros DEqLstNil cs2 a BStep.
      * exists 0. auto.
      * inversion BStep ; subst_eq_steps.
        inversion DEqLstNil. handle_simple_contradict.
      * exists (S n). auto.
      * inversion DEqLstNil. inversion BStep ; subst_eq_steps.
        pose proof (IHn cs1' cs1 lst MStep H4 cs2 a H11) as (n' & ? & ?).
        exists n'. split ; auto.
    Qed.

    Lemma mstep_bstep : forall cs0 cs1 lst, cs0 ==>*[lst] cs1
        -> deq_evt_lst G D lst []
        -> forall n cs2 a, BridgeStep cs1 cs2 a n
        -> BridgeStep cs0 cs2 a (n + length lst).
      intros ? ? ? MStep. induction MStep ; intros DEqNil n ? ? ?.
      * simpl. rewrite <- plus_n_O. assumption.
      * inversion DEqNil ; subst.
        assert (BridgeStep cs1 cs3 a0 (n + length lst)) as IHres by (apply IHMStep ; auto).
        simpl. rewrite <- plus_n_Sm. econstructor ; eauto.
    Qed.

    Lemma bstep_stop : forall pc c nt n s s' a, G;; pc |- c -| nt -> BridgeStep (c, s) (Stop, s') a n -> a = StopEvt.
      intros pc c nt n s s' a WTc BStep. dependent induction BStep.
      * apply wt_step_to_stop_impl_stop_evt with G pc c nt s s' ; auto.
      * destruct cs1 as [c1 s1]. eapply IHBStep ; auto. apply type_pres_not_stop with c s a0 s1; auto.
        intro c1Stop ; subst. inversion BStep as [? ? ? StopStep | ? ? ? ? ? ? StopStep] ; inversion StopStep.
    Qed.

    Lemma bstep_stop_evt : forall c s c' s' n, BridgeStep (c, s) (c', s') StopEvt n -> c' = Stop.
      intros c s c' s' n BStep. dependent induction BStep.
      * apply step_stop_evt_impl_stop with (c, s) s'. assumption.
      * destruct cs1 as [c1 s1]. apply IHBStep with c1 s1 s' ; auto.
    Qed.

    Lemma step_silent : forall c0 s0 c1 s1 a, silent G D c1 s1 -> (c0, s0) -->[a] (c1, s1) -> deq_evt G D a NoEvt -> silent G D c0 s0.
      unfold silent. intros c0 s0 c1 s1 a c1Silent Step HighA cs' lst MStep. induction MStep ; try subst_eq_steps ; eauto using deq_evt_lst.
    Qed.

    Lemma seq_diverge : forall c1 c2 s, diverge c1 s
        -> forall lst c' s', ((Seq c1 c2), s) ==>*[lst] (c', s')
        -> exists c1', c' = Seq c1' c2 /\ (c1, s) ==>*[lst] (c1', s').
      intros c1 c2 s c1Divg lst c' s' MStepSeq. dependent induction MStepSeq.
      * eauto using MultiStep_refl.
      * inversion H as [| ? c1' ? ? s0 | | | | | | |] ; subst.
        - assert (exists c1'', c' = (Seq c1'' c2) /\ (c1', s0) ==>*[lst] (c1'', s')) as (? & ? & ?)
            by (apply IHMStepSeq ; try apply diverge_step with c1 s a ; auto).
          eauto using MultiStep_some.
        - contradiction converge_impl_not_diverge with Skip s.
          unfold converge. eauto using StopE, MultiStep.
    Qed.

    Lemma pdown_diverge : forall c s, diverge c s
        -> forall l lst cp' s', (ProgDown l c, s) ==>*[lst] (cp', s')
        -> exists c', cp' = ProgDown l c' /\ (c, s) ==>*[lst] (c', s').
      intros c s cDivg l lst cp' s' MStepPDown. dependent induction MStepPDown.
      * eauto using MultiStep_refl.
      * inversion H as [| | | | | | | ? c0 ? s0 |] ; subst.
        - assert (exists c', cp' = ProgDown l c' /\ (c0, s0) ==>*[lst] (c', s')) as (? & ? & ?)
            by (apply IHMStepPDown ; try apply diverge_step with c s a ; auto).
          eauto using MultiStep_some.
        - contradiction converge_impl_not_diverge with Skip s.
          unfold converge. eauto using StopE, MultiStep.
    Qed.

    Lemma seq_1_silent : forall c1 c2 s, silent G D c1 s -> silent G D (Seq c1 c2) s.
      unfold silent. intros c1 c2 s c1Silent cs' lst MStep. destruct cs' as [c' s']. dependent induction MStep.
      * apply DEqEmpty.
      * destruct cs1 as [c1' s1']. inversion H ; subst.
        - assert (deq_evt_lst G D [a] []) as DEqANil by (apply c1Silent with (c0', s1') ; eauto using MultiStep).
          inversion DEqANil as [| | ? ? ? DEqANo |] ; subst.
          apply DEqNoEvtL ; [assumption |]. apply IHMStep with c0' c2 s1' c' s' ; auto.
          intros cs' lst0 MStep0.
          assert ((c1, s) ==>*[a :: lst0] cs') as MStep0' by (eauto using MultiStep_some).
          apply c1Silent in MStep0'. inversion MStep0'. assumption.
        - assert ((Skip, s1') ==>*[[StopEvt]] (Stop, s1')) as StepToStop by (eauto using MultiStep, StopE).
          apply c1Silent in StepToStop. inversion StepToStop as [| | ? ? ? BadDEq |]. inversion BadDEq.
    Qed.

    Lemma seq_2_silent : forall c1 s s' lst,
        (c1, s) ==>*[lst] (Skip, s')
        -> deq_evt_lst G D lst []
        -> forall c2, silent G D c2 s'
        -> silent G D (Seq c1 c2) s.
      intros c1 s s' lst MStep. dependent induction MStep ; intros DEqLstNil c2 c2Silent.
      * unfold silent in *. intros cs' lst MStep'. inversion MStep' as [| ? ? ? ? ? Step] ; subst.
        - apply DEqEmpty.
        - inversion Step ; [inversion H6 ; subst ; contradict H5 ; reflexivity | subst].
          apply DEqNoEvtL ; auto with deq.
          apply c2Silent with cs'. assumption.
      * destruct cs1 as [c1' s1']. inversion DEqLstNil as [| | ? ? ? HighA DEqSublstNil |]; subst.
        assert (silent G D (Seq c1' c2) s1'). apply IHMStep with s' ; auto.
        apply step_silent with (Seq c1' c2) s1' a ; auto.
        apply SeqCE ; auto.
        intro c1'Stop. rewrite -> c1'Stop in MStep. inversion MStep. inversion H1.
    Qed.

    Lemma pdown_silent : forall l c s, silent G D c s -> silent G D (ProgDown l c) s.
      unfold silent. intros l c s cSilent cs' lst MStep. destruct cs' as [c' s']. dependent induction MStep.
      * apply DEqEmpty.
      * destruct cs1 as [c1 s1]. inversion H ; subst.
        - assert (deq_evt_lst G D [a] []) as DEqANil by (apply cSilent with (c'0, s1) ; eauto using MultiStep).
          inversion DEqANil as [| | ? ? ? DEqANo |] ; subst.
          apply DEqNoEvtL ; [assumption |]. apply IHMStep with l c'0 s1 c' s' ; auto.
          intros cs' lst0 MStep0.
          assert ((c, s) ==>*[a :: lst0] cs') as MStep0' by (eauto using MultiStep_some).
          apply cSilent in MStep0'. inversion MStep0'. assumption.
        - assert ((Skip, s1) ==>*[[StopEvt]] (Stop, s1)) as StepToStop by (eauto using MultiStep, StopE).
          apply cSilent in StepToStop. inversion StepToStop as [| | ? ? ? BadDEq |]. inversion BadDEq.
    Qed.

    Lemma bstep_low_a : forall cs0 cs1 a n, BridgeStep cs0 cs1 a n -> ~ deq_evt G D a NoEvt.
      intros cs0 cs1 a n BStep. induction BStep ; assumption.
    Qed.

    Ltac resolve_inductive_bstep_simpl :=
      lazymatch goal with
        | [ BadLowA : ~ deq_evt D NoEvt NoEvt |- _ ] => handle_simple_contradict
        | [ H : _ -->[?a] _ |- BridgeStep ((Seq ?c1 ?c2), ?s) ((Seq ?c1' ?c2), ?s') ?a 0 ]
          => apply BStep_0 ; auto ; apply SeqCE ; auto
        | [ H : _ -->[?a] _ |- BridgeStep ((ProgDown _ ?c), ?s) ((ProgDown _ ?c'), ?s') ?a 0 ]
          => apply BStep_0 ; auto ; apply PDownCE ; auto
        | [ Step : (?c1, ?s) -->[?a0] ?cs1,
            BStep : BridgeStep ?cs1 _ ?a _
            |- BridgeStep (?c, ?s) (?c', ?s') ?a (S _) ]
          => destruct cs1 as [c1a s1a]
          ; lazymatch c with
              | (Seq _ ?c2) => apply BStep_n with ((Seq c1a c2), s1a) a0 ; auto ; apply SeqCE
              | (ProgDown ?l _) => apply BStep_n with ((ProgDown l c1a), s1a) a0 ; auto ; apply PDownCE
            end
            ; auto ; intro c1aStop
            ; lazymatch goal with
                | [ CStop : c1a = Stop |- _ ] => rewrite -> CStop in BStep ; inversion BStep
            end
            ; lazymatch goal with
                | [ H : (Stop, _) -->[_] _ |- _ ] => inversion H
            end
        | [ H : (?c, ?s) -->[StopEvt] (?c', ?s') |- ?c' = Stop ] => apply step_stop_evt_impl_stop with (c, s) s' ; auto
        | [ BStep : BridgeStep ?cs (?c', ?s') StopEvt _,
            IH : LowSet D -> forall c1 c1' s1 s1', ?cs = (c1, s1) -> (?c', _) = (c1', _) -> _ -> _ -> c1' = Stop
            |- ?c' = Stop ]
          => destruct cs as [cMid sMid] ; apply IH with cMid sMid s' ; auto
      end.

    Lemma seq_bstep : forall c1 c1' s s' a n, BridgeStep (c1, s) (c1', s') a n
        -> c1' <> Stop -> forall c2, BridgeStep ((Seq c1 c2), s) ((Seq c1' c2), s') a n.
      intros c1 c1' s s' a n BStepc1 ?.
      assert (~ deq_evt G D a NoEvt) as LowA by (eapply bstep_low_a ; eauto).
      induction a ; try intro c2 ; dependent induction BStepc1 ; resolve_inductive_bstep_simpl.
    Qed.

    Lemma seq_bstep_inv : forall c1 c2 c' s s' a n, BridgeStep (Seq c1 c2, s) (c', s') a n
        -> exists c1' s'' a' n', BridgeStep (c1, s) (c1', s'') a' n' /\
           ((Seq c1' c2 = c' /\ c1' <> Stop /\ s'' = s' /\ a' = a /\ n' = n) \/ (c1' = Stop /\ n' < n)).
      intros c1 c2 c' s s' a n BStep. dependent induction BStep.
      * inversion H as [| ? c1' ? ? ? ? c1'NotStop Step | | | | | | |] ; subst.
        - do 4 eexists. split ; [| left] ; eauto using BStep_0.
        - handle_simple_contradict.
      * destruct cs1 as [ca sa]. inversion H as [| ? c1' ? ? ? ? c1'NotStop Step | | | | | | |] ; subst.
        - pose proof (IHBStep c1' c2 c' sa s' eq_refl eq_refl) as IHres.
          destruct IHres as [c1'' IHres]. destruct IHres as [s'' IHres]. destruct IHres as [a' IHres].
          destruct IHres as [n' IHres]. destruct IHres as [BStep1 IHres].
          exists c1''. exists s''. exists a'. exists (S n'). split.
          + apply BStep_n with (c1', sa) a0 ; auto.
          + destruct IHres as [IHres | IHres] ; destruct IHres as [? IHres]
            ; [do 3 destruct IHres as [? IHres] ; subst ; left | ] ; auto using le_n_S.
        - exists Stop. repeat eexists ; auto using le_n_S, le_0_n. apply BStep_0. apply StopE.
          intro DEqStopNo. inversion DEqStopNo. 
    Qed.

    Lemma pdown_bstep : forall c c' s s' a n , BridgeStep (c, s) (c', s') a n
        -> c' <> Stop -> forall l, BridgeStep (ProgDown l c, s) (ProgDown l c', s') a n.
      intros c c' s s' a n BStepc c'NotStop.
      assert (~ deq_evt G D a NoEvt) as LowA by (eapply bstep_low_a ; eauto).
      induction a ; try intro l' ; dependent induction BStepc ; resolve_inductive_bstep_simpl.
    Qed.

    Lemma pdown_bstep_inv : forall l c cFin s s' a n, BridgeStep (ProgDown l c, s) (cFin, s') a n
        -> exists c' a' n', BridgeStep (c, s) (c', s') a' n' /\
           ((ProgDown l c' = cFin /\ c' <> Stop /\ a' = a /\ n' = n) \/ (c' = Stop /\ n' <= n)).
      intros l c cFin s s' a n BStep. dependent induction BStep.
      * inversion H as [| | | | | | | ? c' ? ? ? ? c'NotStop cStep |] ; subst.
        - repeat eexists ; [| left] ; eauto using BStep_0.
        - exists Stop. exists StopEvt. exists 0. split ; [| auto]. apply BStep_0.
          + apply StopE.
          + intro DEq. inversion DEq.
      * destruct cs1 as [c1 s1]. inversion H as [| | | | | | | ? c1' ? ? ? ? c1'NotStop cStep |] ; subst.
        - pose proof (IHBStep l c1' cFin s1 s' eq_refl eq_refl) as IHres.
          destruct IHres as [c' IHres]. destruct IHres as [a' IHres]. destruct IHres as [n' IHres].
          destruct IHres as [BStepc1' IHres].
          exists c'. exists a'. exists (S n'). split.
          + apply BStep_n with (c1', s1) a0 ; auto.
          + destruct IHres as [IHres | IHres] ; destruct IHres as [? IHres]
            ; [do 2 destruct IHres as [? IHres] ; subst |] ; auto using le_n_S.
        - inversion BStep as [? ? ? Step | ? ? ? ? ? ? Step HighA] ; inversion Step ; subst.
          + exists Stop. exists StopEvt. exists 0. auto.
          + inversion HighA.
    Qed.

    Lemma bstep_containment : forall `(LowSet D) pc c nt s,
        G;; pc |- c -| nt
        -> never_stuck c s
        -> ~ In Label D pc
        -> silent G D c s \/ exists s' n, BridgeStep (c, s) (Stop, s') StopEvt n.
      intros ? ? pc c nt s WTc WFs HighPc.
      assert (silent G D c s \/ exists lst s', (c, s) ==>*[lst ++ [StopEvt]] (Stop, s') /\ (s, lst) =[G, D] (s', []))
          as [| (lst & s' & ConvWith & LstEqStop)]
        by (eauto using mstep_containment)
      ; [left ; assumption | right].
      inversion LstEqStop as [? ? ? ? DEqStore DEqLstNil] ; subst.
      apply mstep_iff_msteptl in ConvWith.
      inversion ConvWith as [| ? ? ? ? ? MStep1 Step] ; [apply app_cons_not_nil in H4 ; inversion H4 |].
      apply app_inj_tail in H3 as [? ?]. subst.
      apply mstep_iff_msteptl in MStep1.
      exists s'. apply bstep_from_mstep with cs1 lst ; auto.
      intro BadDEq. inversion BadDEq.
    Qed.

    Lemma contained_bstep : forall `(LowSet D) pc c nt s c' s' a n,
        G;; pc |- c -| nt
        -> BridgeStep (c, s) (c', s') a n
        -> ~ In Label D pc
        -> c' = Stop /\ a = StopEvt.
      intros ? ? pc c nt s c' s' a n WTc BStep HighPc. dependent induction BStep.
      * assert (a = StopEvt \/ deq_evt G D a NoEvt) as [|] by eauto using step_containment
        ; subst ; [eauto using step_stop_evt_impl_stop | handle_simple_contradict].
      * destruct cs1 as [c1 s1]. eapply IHBStep with ?[H] c1 s1 s' ; auto.
        assert (c1 = Stop \/ G;; pc |- c1 -| nt) as c1StopOrWt by (apply type_preservation with c s a0 s1 ; auto).
        destruct c1StopOrWt as [c1Stop | WTc1] ; auto.
        assert (a0 = StopEvt) as a0Stop by (subst ; apply wt_step_to_stop_impl_stop_evt with G pc c nt s s1 ; auto).
        handle_simple_contradict.
    Qed.

    Lemma wt_mstep_to_stop_skip : forall pc c nt,
        G;; pc |- c -| nt
        -> forall s lst a s', (c, s) ==>*[lst ++ [a]] (Stop, s')
        -> (c, s) ==>*[lst] (Skip, s').
      intros pc c nt WellTyped s lst a s' MStep. dependent induction MStep using mstep_rev_ind.
      * apply app_cons_not_nil in x. inversion x.
      * destruct cs1 as [c1 s1]. assert (c1 = Skip). apply wt_step_to_stop_impl_skip with G pc nt s1 a0 s' ; auto.
        assert (c1 = Stop \/ G;; pc |- c1 -| nt) as c1StopOrWt by (apply mstep_type_preservation with c s s1 lst0 ; auto).
        destruct c1StopOrWt as [c1Stop | WTc1] ; [rewrite -> c1Stop in * ; inversion H | assumption].
        apply app_inj_tail in x. destruct x. subst. inversion H. subst. assumption.
    Qed.

    Ltac match_bstep_downtype_case :=
      lazymatch goal with
        | [H : ?a = StopEvt /\ ~ In Label D ?l \/ _ |- ?a = StopEvt /\ ~ In Label D ?l \/ _ ]
          => destruct H ; [left ; assumption | right ; match_bstep_downtype_case]
        | [H : exists l, ?a = PDownEvt l /\ ?DownTypeSubExpr
           |- exists l, ?a = PDownEvt l /\ ?DownTypExpr ]
          => let l := fresh in destruct H as (l & ? & ?) ; exists l ; split ; eauto using HasDowngrade
      end.

    Ltac match_bstep_if_step :=
      lazymatch goal with
        | [Eqes0s1 : evalExpr ?e ?s0 = evalExpr ?e ?s1,
           EvalRes : evalExpr ?e ?s0 = Some ?n,
           IHres : ?BStepIH \/ silent G D ?cN ?s1 /\ _
           |- ?BStep \/ silent G D ?c ?s1 /\ _]
            => rewrite -> Eqes0s1 in EvalRes ; destruct IHres as [s1Stop | s1Silent]
               ; [ left ; destruct s1Stop as [s1' s1Stop] ; destruct s1Stop as [n' s1Stop] ; destruct s1Stop
                   ; exists s1' ; exists (S n') ; split ; auto ; apply BStep_n with (cN, s1) NoEvt ; eauto using IfNE, If0E
                 | right ; repeat destruct s1Silent as [? s1Silent] ; split ; [eapply step_silent ; eauto using IfNE, If0E |]
                   ; match_bstep_downtype_case]
      end.

    Ltac match_bstep_resolve_variance :=
      lazymatch goal with
        | [BStep0 : BridgeStep (?c, ?s0) (?c', ?s0') ?a _,
           IH : forall s0 c' s0' a, BridgeStep (?c, s0) (c', s0') a _
                -> forall s1, deq_store G D s0 s1
                -> never_stuck ?c s1
                -> _,
           Flow : flows_to ?nt' ?nt,
           DEqStore : deq_store G D ?s0 ?s1,
           NStucks1 : never_stuck ?c ?s1
          |- _ ] => pose proof (IH s0 c' s0' a BStep0 s1 DEqStore NStucks1) as [s1BStep | [c1Silent dvgProps]]
            ; [left ; assumption | right ; split ; auto]
            ; destruct dvgProps as [aStop | aPDown]
            ; [ left ; destruct aStop ; eauto using high_set_up_closed | right ; match_bstep_downtype_case]
      end.

    Theorem matching_bridge_step : forall `{LowSet D} {n pc c nt} (WTc : CmdTypeProof G pc c nt),
        forall {s0 c' s0' a}, BridgeStep (c, s0) (c', s0') a n
        -> forall {s1}, deq_store G D s0 s1
        -> never_stuck c s1
        -> (exists s1' n', BridgeStep (c, s1) (c', s1') a n' /\ deq_store G D s0' s1')
           \/ (silent G D c s1 /\
               ((a = StopEvt /\ ~ In Label D nt)
                \/ (exists l, a = PDownEvt l /\ HasDowngrade D WTc))).
      intros ? LowD.
      induction n as [n IHn] using strong_ind.
      intros pc c nt WTc. induction WTc ; intros s0 c' s0' a BStep s1 DEqStore NStucks1
      (* intros G pc c nt WTc. induction WTc using CmdType_prop_ind ; intros s0 c' s0' a BStep s1 DEqStore WFs0 WFs1 *)
      ; inversion BStep as [? ? ? Step LowA | ? ? ? ? ? ? Step DEqANo BStep'] ; subst
      ; try assert (n0 < S n0) as n0LtSn0 by auto.
      * left. inversion Step ; subst.
        exists s1. exists 0. split ; [constructor |] ; auto using StopE with deq.
      * inversion Step ; subst. inversion DEqANo.
      * left. inversion Step ; subst.
        assert (In Label D l) as LowL by (destruct (dec_in l) ; [| contradict LowA] ; eauto using DEqHighAssignNoEvt).
        eexists ; exists 0 ; split.
        - constructor ; auto.
          apply AssignE.
          assert (evalExpr e s0 = evalExpr e s1) as SameEval by eauto using deq_expr_eval.
          rewrite <- SameEval.
          assumption.
        - unfold deq_store in * ; intros x0 ? ? ?.
          destruct (eq_dec x x0) ; eauto.
      * left. inversion Step as [| | | ? ? ? ? EvalExprSome | | | | |] ; subst.
        assert (~ In Label D l) as HighL
          by (inversion DEqANo ; subst
              ; lazymatch goal with | [H0 : G ?x = Some _, H1 : G ?x = Some _ |- _] => rewrite -> H0 in H1 ; injection H1 ; intros ; subst end ; assumption).
        assert (a = StopEvt /\ c' = Stop /\ s0' = (fun y => if eq_dec x y then Some n else s0 y)) as (? & ? & ?) by
          (inversion BStep' ; subst
            ; lazymatch goal with | [H : (Skip, _) -->[_] _ |- _] => inversion H ; subst end ; eauto ; handle_simple_contradict) ; subst.
        assert (exists n, evalExpr e s1 = Some n) as []
          by (destruct (NStucks1 (Assign x e) s1 [] (MultiStep_refl (Assign x e, s1))) as [| (? & ? & StepS1)]; [discriminate | inversion StepS1 ; eauto]).
        eexists ; exists 1 ; split.
        - econstructor ; eauto using AssignE, DEqHighAssignNoEvt, expr_type_eval_some.
          constructor ; [eapply StopE | intro ; handle_simple_contradict].
        - unfold deq_store in * ; intros x1 ? ? ?.
          destruct (eq_dec x x1) ; subst ; eauto.
          lazymatch goal with | [H0 : G ?x = Some _, H1 : G ?x = Some _ |- _] => rewrite -> H0 in H1 ; injection H1 ; intros ; subst end.
          handle_simple_contradict.
      * contradict LowA. inversion Step ; apply deq_evt_refl.
      * destruct cs1 as [cIn sIn]. destruct (dec_in pc) as [LowPc | HighPc].
        - pose proof (deq_expr_eval G D s0 s1 DEqStore e pc e0 LowPc) as Eqes0s1.
          inversion Step ; subst
          ; [set (c := If e cIn c2) | set (c := If e c1 cIn)]
          ; assert ((c, s1) --> (cIn, s1)) as StepS1 by (econstructor ; rewrite <- Eqes0s1 ; eauto)
          ; [set (WTc := WTc1) | set (WTc := WTc2)]
          ; pose proof (IHn n0 n0LtSn0 pc cIn nt WTc sIn c' s0' a BStep' s1 DEqStore (never_stuck_step NStucks1 StepS1)) as IHres
          ; match_bstep_if_step.
        - assert (c' = Stop /\ a = StopEvt) as [c'Stop aStop]
            by (apply contained_bstep with pc (If e c1 c2) nt s0 s0' (S n0) ; auto using IfT, wt_proof_impl_prop) ; subst.
          assert (silent G D (If e c1 c2) s1 \/ exists s1' n', BridgeStep (If e c1 c2, s1) (Stop, s1') StopEvt n') as [| (s1' & n' & s1Conv)]
            by (apply bstep_containment with pc nt ; auto using IfT, wt_proof_impl_prop)
          ; [right | left].
          + split ; [| left ; split ] ; auto.
            eapply high_set_up_closed with ?[H] pc ; auto.
            apply diverge_pc_ft_nt with G (If e c1 c2) s1 ; eauto using IfT, wt_silent_impl_diverge, wt_proof_impl_prop.
          + exists s1'. exists n'. split ; [assumption |].
            apply mstep_from_bstep in BStep. destruct BStep as [lst0 BStep].
            apply mstep_from_bstep in s1Conv. destruct s1Conv as [lst1 s1Conv].
            transitivity s0 ; [| transitivity s1] ; auto ; [apply deq_store_symm |]
            ; eauto using mstep_contain_store, IfT, wt_proof_impl_prop.
      * inversion Step as [| ? c0' ? ? ? ? c0'NotStop c1Step | | | | | | |] ; subst.
        - assert (BridgeStep (c1, s0) (c0', s0') a 0) as BStepc1 by (constructor ; auto).
          pose proof (IHWTc1 s0 c0' s0' a BStepc1 s1 DEqStore (never_stuck_seq_c1 NStucks1)) as [(? & ? & ? & ?) | [? dvgProps]]
          ; [left ; eauto using seq_bstep | right ; split].
          + apply seq_1_silent ; assumption.
          + destruct dvgProps as [aStop | aPDown]
            ; [ left ; destruct aStop ; eauto using high_set_up_closed | right ; match_bstep_downtype_case].
        - inversion BStep ; handle_simple_contradict.
      (* The inductive sequence case is extremely complicated. *)
      * pose proof (seq_bstep_inv c1 c2 c' s0 s0' a (S n0) BStep) as (c1' & s0'' & a' & n' & BStepc1s0 & [(? & ? & ? & ? & ?) | [? n'Ltn0]]) ; subst.
        - destruct cs1 as [c1a s0a].
          pose proof (IHWTc1 s0 c1' s0' a BStepc1s0 s1 DEqStore (never_stuck_seq_c1 NStucks1)) as [(? & ? & ? & ?) | [? DivgProps]] ; [left ; eauto using seq_bstep  | right].
          split.
          + apply seq_1_silent. assumption.
          + destruct DivgProps as [[] | aPDown]
            ; [ left ; eauto using high_set_up_closed | right ; match_bstep_downtype_case ].
        - assert (a' = StopEvt) as a'Stop by (apply bstep_stop with pc c1 nt' n' s0 s0'' ; auto using wt_proof_impl_prop) ; subst.
          pose proof (mstep_lst_from_bstep BStepc1s0) as (lst & MStepc1 & DEqLstNil).
          assert ((c1, s0) ==>*[lst] (Skip, s0'')) as c1MStepSkip
            by (apply wt_mstep_to_stop_skip with pc nt' StopEvt ; auto using wt_proof_impl_prop).
          assert (exists n0', BridgeStep (c2, s0'') (c', s0') a n0' /\ n0' = S n0 - length (lst ++ [NoEvt]))
              as (n0' & BStepc2 & n0'val)
            by (apply bstep_mstep with (Seq c1 c2, s0) ; auto
                ; [apply mstep_iff_msteptl ; apply MultiStepTl_some with (Seq Skip c2, s0'')
                  ; [apply mstep_iff_msteptl ; apply step_under_seq | apply SeqSkipE]
                  | apply deq_evt_lst_append_high_l] ; auto with deq ; discriminate).
          pose proof (IHn n' n'Ltn0 pc c1 nt' WTc1 s0 Stop s0'' StopEvt BStepc1s0 s1 DEqStore (never_stuck_seq_c1 NStucks1))
            as [(s1'' & n1' & BStepc1s1 & DEqStore'') | s1Silent].
          + assert (n0' < S n0) as n0'Ltn0 by (rewrite -> n0'val ; rewrite -> last_length ; simpl ; apply le_n_S ; apply le_minus).
            pose proof (mstep_lst_from_bstep BStepc1s1) as (lst1 & ? & ?).
            assert ((c1, s1) ==>*[lst1] (Skip, s1'')) as c1s1MStepSkip
              by (apply wt_mstep_to_stop_skip with pc nt' StopEvt ; auto using wt_proof_impl_prop).
            assert (never_stuck c2 s1'') as NStucks1' by eauto using never_stuck_mstep, mstep_append, step_under_seq_skip, SeqSkipE.
            pose proof (IHn n0' n0'Ltn0 pc' c2 nt WTc2 s0'' c' s0' a BStepc2 s1'' DEqStore'' NStucks1')
                as [(s1' & n1'' & ? & ?) | [? ?]]
            ; [left | right].
            ** exists s1'. exists ((S n1'') + length lst1). split ; [| assumption].
               apply mstep_bstep with (Seq Skip c2, s1'')
                 ; [apply step_under_seq | | econstructor]; auto using SeqSkipE with deq ; discriminate.
            ** split ; [apply seq_2_silent with s1'' lst1 ; auto | match_bstep_downtype_case].
          + destruct s1Silent as [s1Silent [[? Highnt'] | (? & ? & ?)]] ; [| discriminate]. right.
            split ; [apply seq_1_silent ; assumption |].
            pose proof (mstep_lst_from_bstep BStepc2) as (lst2 & MStep2 & DEqLst2Nil).
            eapply mstep_contain_evt_lst in MStep2 ; eauto using wt_proof_impl_prop, high_set_up_closed.
            destruct MStep2 as [BadNil | (? & ?)].
            ** apply deq_lst_app_eq_nil in BadNil as [? BadDEq].
               exfalso ; eapply bstep_low_a ; [apply BStep | inversion BadDEq ; auto].
            ** left ; split ; [| eauto using high_set_up_closed].
               assert (deq_evt_lst G D [a] [StopEvt]) as DEqStop by eauto using deq_app_nil_l, deq_evt_lst_symm.
               inversion DEqStop ; subst ; [reflexivity | exfalso ; eapply bstep_low_a ; eauto | handle_simple_contradict].
      * inversion Step ; handle_simple_contradict.
      * inversion Step ; subst.
        assert (exists (WTWhile : CmdTypeProof G pc (If e (Seq c (While e c)) Skip) pc),
          forall D', HasDowngrade D' WTWhile -> HasDowngrade D' (WhileTPf G pc e c e0 WTc)) as [WTWhile ?]
          by (eapply type_pres_pf_with_downgrade ; eauto ; discriminate).
        assert ((While e c, s1) --> (If e (Seq c (While e c)) Skip, s1)) as StepS1 by eauto using WhileE.
        pose proof (IHn n0 n0LtSn0 pc (If e (Seq c (While e c)) Skip) pc WTWhile s0 c' s0' a BStep' s1 DEqStore (never_stuck_step NStucks1 StepS1)) as IHres.
        destruct IHres as [(s1' & n' & ? & ?) | [? ?]] ; [left | right].
        - exists s1'. exists (S n'). split ; [| assumption]. econstructor ; eauto.
        - split ; [apply step_silent with (If e (Seq c (While e c)) Skip) s1 NoEvt ; auto using WhileE |].
          match_bstep_downtype_case.
      * inversion Step as [| | | | | | | ? c0 ? ? ? ? c0NotStop cStep |] ; subst.
        - assert (BridgeStep (c, s0) (c0, s0') a 0) as BStepc by (constructor ; auto).
          pose proof (IHWTc s0 c0 s0' a BStepc s1 DEqStore (never_stuck_under_pdown NStucks1))
            as [(s1' & n' & ? & ?) | [? dvgProps]] ; subst ; [left | right].
          + exists s1'. exists n'. auto using pdown_bstep.
          + split ; [auto using pdown_silent |].
            destruct dvgProps as [[? ?] | aPDown] ; subst.
            ** contradict c0NotStop. eauto using step_stop_evt_impl_stop.
            ** right. match_bstep_downtype_case.
        - left. exists s1. exists 0. split ; auto. constructor ; auto using PDownSkipE.
      * pose proof (pdown_bstep_inv l c c' s0 s0' a (S n0) BStep)
          as (c0 & a' & n0' & BStepcs0 & [(? & c0NotStop & ? & ?) | [? n0'LeSn0]]) ; subst.
        - destruct cs1 as [c1 s0a].
          pose proof (IHWTc s0 c0 s0' a BStepcs0 s1 DEqStore (never_stuck_under_pdown NStucks1)) as IHres.
          destruct IHres as [s1Conv | s1Divg] ; [left | right].
          + destruct s1Conv as [s1' s1Conv]. destruct s1Conv as [n' s1Conv]. destruct s1Conv as [s1BStep DEqStore'].
            exists s1'. exists n'. split ; [apply pdown_bstep |] ; auto.
          + destruct s1Divg as [s1Silent DivgProps]. split.
            ** apply pdown_silent. assumption.
            ** destruct DivgProps as [[? ?] | aPDown] ; subst ; [| right ; match_bstep_downtype_case].
               contradict c0NotStop. apply bstep_stop_evt with c s0 s0' (S n0). assumption.
        - assert (a' = StopEvt) by (apply bstep_stop with pc c nt n0' s0 s0' ; auto using wt_proof_impl_prop) ; subst.
          pose proof (mstep_lst_from_bstep BStepcs0) as (lst0 & ? & ?).
          assert ((c, s0) ==>*[lst0] (Skip, s0')) as cs0MStepSkip by (apply wt_mstep_to_stop_skip with pc nt StopEvt ; auto using wt_proof_impl_prop).
          assert (exists n0'', BridgeStep (ProgDown l Skip, s0') (c', s0') a n0'' /\ n0'' = (S n0) - length (lst0)) as (n0'' & PDownSkipToA & ?)
            by (apply bstep_mstep with (ProgDown l c, s0) ; auto ; apply step_under_pdown ; [assumption | discriminate]).
          assert ((exists s1' n', BridgeStep (c, s1) (Stop, s1') StopEvt n' /\ deq_store G D s0' s1') \/
                   silent G D c s1 /\ (StopEvt = StopEvt /\ ~ In Label D nt \/
                   (exists l, StopEvt = PDownEvt l /\ HasDowngrade D WTc)))
            as IHres
            by (inversion n0'LeSn0 as [n0'EqSn0 | n0'LeSSn0] ; subst
                ; [apply IHWTc with s0
                  | assert (n0' < S n0) as n0'LtSn0 by auto using le_n_S ; apply IHn with n0' s0] ; eauto using never_stuck_under_pdown).
          destruct IHres as [(s1' & n' & s1BStep & ?) | [? aProps]] ; [left | right].
          + pose proof (mstep_lst_from_bstep s1BStep) as (lst1 & ? & ?).
            assert ((c, s1) ==>*[lst1] (Skip, s1')) as cs1MStepSkip
              by (apply wt_mstep_to_stop_skip with pc nt StopEvt ; auto using wt_proof_impl_prop).
            inversion PDownSkipToA as [? ? ? FinStep LowA | ? ? ? ? ? ? FinStep HighA FinBStep] ; subst.
            ** inversion FinStep ; subst ; [handle_simple_contradict |].
               exists s1'. exists (0 + (length lst1)). split ; [| assumption].
               apply mstep_bstep with (ProgDown l Skip, s1') ; [apply step_under_pdown | | constructor] ; auto using PDownSkipE ; discriminate.
            ** inversion FinStep ; subst ; [handle_simple_contradict |].
               inversion FinBStep as [? ? ? SkipStep LowA | ? ? ? ? ? ? SkipStep LowA] ; inversion SkipStep ; subst.
               -- exists s1'. exists (1 + (length lst1)). split ; [| assumption].
                  apply mstep_bstep with (ProgDown l Skip, s1') ; [apply step_under_pdown | | econstructor ]
                  ; eauto using BStep_0, PDownSkipE, StopE ; try discriminate.
               -- inversion LowA.
          + split ; [apply pdown_silent ; assumption |].
            inversion PDownSkipToA as [? ? ? FinStep LowA | ? ? ? ? ? ? FinStep HighA FinBStep] ; subst ; [right | left].
            ** inversion FinStep ; subst ; [handle_simple_contradict |].
               assert (In Label D l) by (destruct (dec_in l) ; [assumption | contradict LowA ; auto using DEqHighPDownNoEvt]).
               destruct aProps as [[] | (? & ? & ?)] ; [eauto using HasDownPHasDownHere | discriminate].
            ** inversion FinStep ; subst ; [handle_simple_contradict |].
               inversion HighA ; subst.
               inversion FinBStep as [? ? ? SkipStep | ? ? ? ? ? ? SkipStep]
               ; inversion SkipStep ; [auto | handle_simple_contradict].
      * match_bstep_resolve_variance.
      * match_bstep_resolve_variance.
    Qed.

    Lemma wt_bstep_erase : forall `{DecideIn Label D} {pc c nt s cs2 lst a elst},
        G;; pc |- c -| nt
        -> (c, s) ==>*[lst] cs2
        -> a :: elst = erase_evt_lst G D lst
        -> exists cs1 n lst1, BridgeStep (c, s) cs1 a n /\ cs1 ==>*[lst1] cs2 /\ erase_evt_lst G D lst1 = elst.
      intros DecD pc c nt s cs2 lst a elst WTc MStep EraseEq.
      dependent induction MStep ; simpl in EraseEq ; [inversion EraseEq |].
      destruct cs1 as [c1 s1].
      assert (c1 = Stop \/ G;; pc |- c1 -| nt) as [c1Stop | WTc1] by eauto using type_preservation.
      * subst. assert (a0 = StopEvt) by (eapply wt_step_to_stop_impl_stop_evt ; eauto) ; subst.
        destruct (deq_evt_dec G D StopEvt) ; [handle_simple_contradict |].
        inversion MStep as [| ? ? ? ? ? Step] ; [subst | inversion Step].
        simpl in EraseEq ; injection EraseEq ; intros ; subst.
        eauto 7 using MultiStep_refl, BStep_0.
      * destruct (deq_evt_dec G D a0)
        ; [pose proof (IHMStep DecD c1 s1 WTc1 eq_refl EraseEq) as (cs1 & ? & ? & ? & ? & ?) | inversion EraseEq ; subst ]
        ; eauto 7 using BridgeStep, deq_evt.
    Qed.

    Lemma bstep_split_mstep : forall {cs cs0 lst}, cs ==>*[lst] cs0
        -> forall {cs1 a n}, BridgeStep cs cs1 a n
        -> (deq_evt_lst G D lst [] /\ exists lst', cs0 ==>*[lst'] cs1)
            \/ (exists lstF lstR, lst = lstF ++ a :: lstR /\ deq_evt_lst G D lstF [] /\ cs1 ==>*[lstR] cs0).
      intros cs cs0 lst MStep. dependent induction MStep ; intros cs1' a' n BStep.
      * left. split ; [auto with deq |].
        assert (exists lst', cs ==>*[lst' ++ [a']] cs1') as BuiltMStep by (apply mstep_from_bstep with n; auto).
        destruct BuiltMStep. eauto.
      * dependent induction BStep ; subst_eq_steps.
        - right. exists nil. simpl. eauto with deq.
        - pose proof (IHMStep cs4 a1 n BStep) as [[? ?] | (lstF & ? & ? & ? & ?)] ; subst ; auto using DEqNoEvtL.
          right. exists (a :: lstF). simpl. eauto using DEqNoEvtL.
    Qed.

    Lemma bstep_type_preservation : forall {pc c nt s c' s' a n}, G;; pc |- c -| nt
        -> BridgeStep (c, s) (c', s') a n
        -> c' = Stop \/ G;; pc |- c' -| nt.
      intros pc c nt s c' s' a n WTc BStep.
      assert (exists lst, (c, s) ==>*[lst ++ [a]] (c', s')) as eMStep by (eapply mstep_from_bstep ; eauto).
      destruct eMStep.
      eapply mstep_type_preservation ; eauto.
    Qed.

    Lemma bstep_nstuck : forall {c s c' s' a n}, BridgeStep (c, s) (c', s') a n -> never_stuck c s -> never_stuck c' s'.
      intros c s c' s' a n BStep NStuck.
      assert (exists lst, (c, s) ==>*[lst ++ [a]] (c', s')) as [] by eauto using mstep_from_bstep.
      eauto using never_stuck_mstep.
    Qed.

    Lemma conv_divg_deq : forall `{LowSet D} lst0 {pc c nt}, G;; pc |- c -| nt
        -> forall s0 s1, never_stuck c s0 -> never_stuck c s1
        -> forall s0', (c, s0) ==>*[lst0 ++ [StopEvt]] (Stop, s0')
        -> forall lst1, (exists cs1, (c, s1) ==>*[lst1] cs1)
        -> diverge c s1
        -> (s0, lst0) =[G, D] (s1, lst1)
        -> ~ In Label D nt.
      intros DecD LowD lst0. remember (erase_evt_lst G D lst0) as lst eqn:LstVal.
      generalize dependent lst0.
      induction lst as [| a lst] ; intros lst0 LstVal pc c nt WTc s0 s1 NStuck0 NStuck1 s0' MStep0 lst1 [cs1 MStep1] DivgS1 DEqPfx
      ; inversion DEqPfx as [? ? ? ? DEqStore DEqLst] ; subst
      ; invert_tail mstep_iff_msteptl MStep0
      ; lazymatch goal with
        | [H : _ -->[StopEvt] _ |- _] => inversion H ; subst
      end
      ; assert (exists _ : CmdTypeProof G pc c nt, True) as [Pf Junk] by auto using wt_impl_proof ; clear Junk.
      * assert (exists n, BridgeStep (c, s0) (Stop, s0') StopEvt n) as [n0 BStep0]
          by (apply bstep_from_mstep with (Skip, s0') lst0 ; eauto using erase_lst_impl_deq
              ; destruct (deq_evt_dec G D StopEvt) ; [handle_simple_contradict | assumption]).
        pose proof (matching_bridge_step Pf BStep0 DEqStore NStuck1)
          as [(s1' & n1 & BStep1 & DEqStore') | [? [[] | (? & ? & ?)]]]
        ; [| assumption | discriminate].
        assert (exists lst, (c, s1) ==>*[lst ++ [StopEvt]] (Stop, s1')) as [] by eauto using mstep_from_bstep.
        contradiction diverge_impl_not_converge with c s1.
        unfold converge ; eauto.
      * pose proof (wt_bstep_erase WTc H0 LstVal) as ([c' s0''] & ? & lst0' & BStep0 & ? & ?).
        pose proof (matching_bridge_step Pf BStep0 DEqStore NStuck1) as [(s1' & n1 & BStep1 & DEqStore') | [SilentS1 ?]]
        ; [pose proof (bstep_split_mstep MStep1 BStep1) as [(? & ? & ?) | (? & lst1' & ? & ? & ?)] | apply SilentS1 in MStep1]
        ; try lazymatch goal with
          | [H : deq_evt_lst G D lst1 [] |- _]
            => eapply deq_impl_erase_lst in DEqLst, H ; simpl in *
               ; rewrite <- LstVal in DEqLst
               ; rewrite -> H in DEqLst
               ; inversion DEqLst
        end.
        assert (exists lst, (c, s1) ==>*[lst ++ [a]] (c', s1')) as [] by eauto using mstep_from_bstep.
        apply IHlst with lst0' pc c' s0'' s1' s0' lst1' ; eauto using bstep_nstuck, mstep_append, StopE, diverge_mstep.
        - eapply mstep_type_pres_not_stop with c s1 s1' _ ; eauto.
          intro ; subst.
          contradiction diverge_impl_not_converge with c s1.
          unfold converge ; eauto.
        - apply DEqPfx_intro ; auto.
          repeat lazymatch goal with
            | [H : deq_evt_lst G D _ _ |- _] => eapply deq_impl_erase_lst in H
          end ; simpl in * ; subst.
          rewrite <- LstVal in DEqLst.
          rewrite -> erase_app_distr in DEqLst.
          rewrite -> H4 in DEqLst.
          simpl in DEqLst.
          destruct (deq_evt_dec G D a) ; [exfalso ; eapply bstep_low_a ; eauto |].
          injection DEqLst.
          apply erase_lst_impl_deq.
    Qed.

  End BStepSec.

End BridgeStep.
