From Coq Require Import Basics Equality List Ensembles Relations RelationClasses.

Import ListNotations.

Class EqDec (A : Type) := {
  eq_dec : forall (a b : A), {a = b} + {a <> b}
}.

Class LabelOrder (Label : Set) := {
  flows_to : Label -> Label -> Prop ;
  flows_to_refl : forall l, flows_to l l ;
  flows_to_trans : forall l1 l2 l3, flows_to l1 l2 -> flows_to l2 l3 -> flows_to l1 l3 ;

  e_lower_bound : forall l1 l2, exists l0, flows_to l0 l1 /\ flows_to l0 l2 ;

  reflect : Label -> Label ;
  reflect_homomorphism: forall l1 l2, (flows_to l1 l2) -> (flows_to (reflect l2) (reflect l1)) ;
}.

#[global] Hint Resolve flows_to_refl : core.

#[global] Instance flows_to_preorder (Label : Set) (LabOrder : LabelOrder Label) : PreOrder flows_to.
  split.
  * unfold Reflexive. exact flows_to_refl.
  * unfold Transitive. exact flows_to_trans.
Defined.

Module Type ImpDefs.

  Declare Scope imp_scope.
  Open Scope imp_scope.

  Parameter Varname : Set.
  Parameter var_eq_dec : EqDec Varname.
  #[global] Instance var_eq_dec_inst : EqDec Varname.
    exact var_eq_dec.
  Defined.

  Parameter Label : Set.
  Parameter label_order : LabelOrder Label.
  #[global] Instance label_order_inst : LabelOrder Label.
    exact label_order.
  Defined.

  Implicit Types (n : nat) (G : Varname -> option Label) (x : Varname) (l pc nt : Label).

  Inductive Expr :=
    | Nat n
    | Var x
    | Op (f : (nat -> nat -> nat)) (e1 : Expr) (e2 : Expr).

  Inductive Cmd :=
    | Skip
    | Assign x (e : Expr)
    | If (e : Expr) (c1 : Cmd) (c2 : Cmd)
    | Seq (c1 : Cmd) (c2 : Cmd)
    | While (e : Expr) (c : Cmd)
    | ProgDown l (c : Cmd)
    | Stop.

  Definition Store := Varname -> option nat.

  Implicit Types (e : Expr) (c : Cmd) (s : Store).

  Section Semantics.

    Fixpoint evalExpr e s : option nat :=
      match e with
        | Nat n => Some n
        | Var x => (s x)
        | Op f e1 e2 =>
          match (evalExpr e1 s) with
            | Some n1 => (option_map (f n1)) (evalExpr e2 s)
            | None => None
          end
      end.

    Inductive Event :=
      | NoEvt
      | AssignEvt x n
      | PDownEvt l
      | StopEvt.

    #[local] Reserved Notation "cs0 '-->[' a ']' cs1" (at level 50, no associativity).
    #[local] Reserved Notation "cs0 '-->' cs1" (at level 50, no associativity).

    Inductive OneStep : Cmd * Store -> Cmd * Store -> Event -> Prop :=
      | StopE : forall s, (Skip, s) -->[StopEvt] (Stop, s)
      | SeqCE : forall c0 c0' c1 s s' a,
          c0' <> Stop -> (c0, s) -->[a] (c0', s') -> (Seq c0 c1, s) -->[a] (Seq c0' c1, s')
      | SeqSkipE : forall c s, (Seq Skip c, s) --> (c, s)
      | AssignE : forall x e s n,
          (evalExpr e s) = Some n ->
          ((Assign x e), s) -->[AssignEvt x n] (Skip, (fun y => if (eq_dec x y) then Some n else (s y)))
      | IfNE : forall e c1 c2 s n,
          (evalExpr e s) = Some (S n) -> ((If e c1 c2), s) --> (c1, s)
      | If0E : forall e c1 c2 s,
          (evalExpr e s) = Some 0 -> ((If e c1 c2), s) --> (c2, s)
      | WhileE : forall e c s,
          ((While e c), s) --> ((If e (Seq c (While e c)) Skip), s)
      | PDownCE : forall c c' s s' a l,
          c' <> Stop -> (c, s) -->[a] (c', s') -> ((ProgDown l c), s) -->[a] ((ProgDown l c'), s')
      | PDownSkipE : forall s l, ((ProgDown l Skip), s) -->[PDownEvt l] (Skip, s)

      where "cs0 '-->[' a ']' cs1" := (OneStep cs0 cs1 a) : imp_scope
      and "cs0 '-->' cs1" := (OneStep cs0 cs1 NoEvt) : imp_scope.

    #[local] Reserved Notation "cs0 '==>*[' lst ']' cs1" (at level 50, no associativity).

    Inductive MultiStep : Cmd * Store -> Cmd * Store -> list Event -> Prop :=
      | MultiStep_refl : forall cs, cs ==>*[[]] cs
      | MultiStep_some : forall cs0 cs1 cs2 a lst,
          cs0 -->[a] cs1 -> cs1 ==>*[lst] cs2 -> cs0 ==>*[a :: lst] cs2

      where "cs0 '==>*[' lst ']' cs1" := (MultiStep cs0 cs1 lst) : imp_scope.

    Definition never_stuck c s := forall c' s' lst, (c, s) ==>*[lst] (c', s') -> c' = Stop \/ exists cs a, (c', s') -->[a] cs.

    Definition converge c s : Prop := exists lst s', (c, s) ==>*[lst] (Stop, s').

    Definition diverge c s : Prop := forall cs1 lst, (c, s) ==>*[lst] cs1 -> exists cs2 a, cs1 -->[a] cs2.

  End Semantics.

  Section TypingRules.

    Inductive ExprType G : Expr -> Label -> Prop :=
      | ConstT : forall n l, ExprType G (Nat n) l
      | VarT : forall x l , G x = Some l -> ExprType G (Var x) l
      | OpT : forall op e1 e2 l, ExprType G e1 l -> ExprType G e2 l -> ExprType G (Op op e1 e2) l
      | EVarianceT : forall e l l', ExprType G e l -> flows_to l l' -> ExprType G e l'.

    #[local] Reserved Notation "G ';;' pc '|-' c '-|' nt" (at level 30, no associativity).

    Inductive CmdType G : Label -> Cmd -> Label -> Prop :=
      | SkipT : forall pc nt, G ;; pc |- Skip -| nt
      | AssignT : forall x e l nt,
        G x = Some l
        -> ExprType G e l
        -> G;; l |- Assign x e -| nt
      | IfT : forall pc e c1 c2 nt,
        (ExprType G e pc)
        -> G;; pc |- c1 -| nt
        -> G;; pc |- c2 -| nt
        -> G;; pc |- If e c1 c2 -| nt
      | SeqT : forall pc pc' c1 c2 nt nt',
        G;; pc |- c1 -| nt'
        -> flows_to pc pc'
        -> flows_to nt' pc'
        -> G;; pc' |- c2 -| nt
        -> flows_to nt' nt
        -> G;; pc |- Seq c1 c2 -| nt
      | WhileT : forall pc e c,
        ExprType G e pc
        -> G;; pc |- c -| pc
        -> G;; pc |- While e c -| pc
      | ProgDownT : forall pc c nt l,
        G;; pc |- c -| nt
        -> flows_to nt (reflect nt)
        -> flows_to pc l
        -> G;; pc |- ProgDown l c -| l
      | VarianceT : forall pc pc' c nt nt',
        G;; pc' |- c -| nt'
        -> flows_to pc pc'
        -> flows_to nt' nt
        -> G;; pc |- c -| nt

      where "G ';;' pc '|-' c '-|' nt" := (CmdType G pc c nt).

    (* We also have a separate CmdTypeProof that's in Set so we can more easily deconstruct it. *)
    Inductive CmdTypeProof G : Label -> Cmd -> Label -> Set :=
      | SkipTPf : forall pc nt, CmdTypeProof G pc Skip nt
      | AssignTPf : forall x e l nt,
        G x = Some l
        -> ExprType G e l
        -> CmdTypeProof G  l (Assign x e) nt
      | IfTPf : forall pc e c1 c2 nt,
        ExprType G e pc
        -> CmdTypeProof G pc c1 nt
        -> CmdTypeProof G pc c2 nt
        -> CmdTypeProof G pc (If e c1 c2) nt
      | SeqTPf : forall pc pc' c1 c2 nt nt',
        CmdTypeProof G pc c1 nt'
        -> flows_to pc pc'
        -> flows_to nt' pc'
        -> CmdTypeProof G pc' c2 nt
        -> flows_to nt' nt
        -> CmdTypeProof G pc (Seq c1 c2) nt
      | WhileTPf : forall pc e c,
        ExprType G e pc
        -> CmdTypeProof G pc c pc
        -> CmdTypeProof G pc (While e c) pc
      | ProgDownTPf : forall pc c nt l,
        CmdTypeProof G pc c nt
        -> flows_to nt (reflect nt)
        -> flows_to pc l
        -> CmdTypeProof G pc (ProgDown l c) l
      | VarianceTPf : forall pc pc' c nt nt',
        CmdTypeProof G pc' c nt'
        -> flows_to pc pc'
        -> flows_to nt' nt
        -> CmdTypeProof G pc c nt.

    Lemma wt_impl_proof : forall {G pc c nt}, G;; pc |- c -| nt -> exists (_ : CmdTypeProof G pc c nt), True.
      intros G pc c nt WTc.
      dependent induction WTc
        ; repeat lazymatch goal with
          | [IH : exists (_ : CmdTypeProof G _ _ _), True |- _] => destruct IH
        end ; eauto using CmdTypeProof.
    Qed.

    Lemma wt_proof_impl_prop : forall {G pc c nt} (Pf : CmdTypeProof G pc c nt), G;; pc |- c -| nt.
      intros G pc c nt Pf. dependent induction Pf ; eauto using CmdType.
    Qed.

    Definition dom_subset {A B C : Type} (f : A -> option B) (g : A -> option C) :=
      forall a, g a = None -> f a = None.

    Lemma dom_subset_some {A B C : Type} : forall (f : A -> option B) (g : A -> option C), dom_subset f g
        -> forall a b, f a = Some b -> exists (c : C), g a = Some c.
      intros f g DomSub a b FSome.
      destruct (g a) eqn:GNone ; eauto.
      assert (f a = None) as FNone by eauto.
      rewrite -> FNone in FSome ; discriminate.
    Qed.

    Inductive HasDowngrade (D : Ensemble Label) {G pc} : forall {c nt}, CmdTypeProof G pc c nt -> Prop :=
      | HasDownIf1 : forall e c1 c2 nt (eType : ExprType G e pc) (Pfc1 : CmdTypeProof G pc c1 nt) (Pfc2 : CmdTypeProof G pc c2 nt),
          HasDowngrade D Pfc1 -> HasDowngrade D (IfTPf G pc e c1 c2 nt eType Pfc1 Pfc2)
      | HasDownIf2 : forall e c1 c2 nt (eType : ExprType G e pc) (Pfc1 : CmdTypeProof G pc c1 nt) (Pfc2 : CmdTypeProof G pc c2 nt),
          HasDowngrade D Pfc2 -> HasDowngrade D (IfTPf G pc e c1 c2 nt eType Pfc1 Pfc2)
      | HasDownSeq1: forall pc' c1 c2 nt nt' (Pfc1 : CmdTypeProof G pc c1 nt')
            (FTpc : (flows_to pc pc')) (FTntpc : (flows_to nt' pc'))
            (Pfc2 : CmdTypeProof G pc' c2 nt) (FTnt : (flows_to nt' nt)),
          HasDowngrade D Pfc1 -> HasDowngrade D (SeqTPf G pc pc' c1 c2 nt nt' Pfc1 FTpc FTntpc Pfc2 FTnt)
      | HasDownSeq2: forall pc' c1 c2 nt nt' (Pfc1 : CmdTypeProof G pc c1 nt')
            (FTpc : (flows_to pc pc')) (FTntpc : (flows_to nt' pc'))
            (Pfc2 : CmdTypeProof G pc' c2 nt) (FTnt : (flows_to nt' nt)),
          HasDowngrade D Pfc2 -> HasDowngrade D (SeqTPf G pc pc' c1 c2 nt nt' Pfc1 FTpc FTntpc Pfc2 FTnt)
      | HasDownWhile : forall e c (eType : ExprType G e pc) (Pfc : CmdTypeProof G pc c pc),
          HasDowngrade D Pfc -> HasDowngrade D (WhileTPf G pc e c eType Pfc)
      | HasDownPHasDownHere : forall c nt l (Pfc : CmdTypeProof G pc c nt) (FlowsNtRefl : flows_to nt (reflect nt)) (FlowsPcL : flows_to pc l),
          ~ In Label D nt -> In Label D l -> HasDowngrade D (ProgDownTPf G pc c nt l Pfc FlowsNtRefl FlowsPcL)
      | HasDownPHasDownInd : forall c nt l (Pfc : CmdTypeProof G pc c nt) (FlowsNtRefl : flows_to nt (reflect nt)) (FlowsPcL : flows_to pc l),
          HasDowngrade D Pfc -> HasDowngrade D (ProgDownTPf G pc c nt l Pfc FlowsNtRefl FlowsPcL)
      | HasDownVar : forall pc' c nt nt' (Pfc' : CmdTypeProof G pc' c nt') (FTpc : flows_to pc pc') (FTnt : flows_to nt' nt),
          HasDowngrade D Pfc' -> HasDowngrade D (VarianceTPf G pc pc' c nt nt' Pfc' FTpc FTnt).

    End TypingRules.

    Module ImpNotations.
      Notation "G ';;' pc '|-' c '-|' nt" := (CmdType G pc c nt) (at level 30, no associativity).
      Notation "cs0 '-->[' a ']' cs1" := (OneStep cs0 cs1 a) (at level 50, no associativity).
      Notation "cs0 '-->' cs1" := (OneStep cs0 cs1 NoEvt) (at level 50, no associativity).
      Notation "cs0 '==>*[' lst ']' cs1" := (MultiStep cs0 cs1 lst) (at level 50, no associativity).
    End ImpNotations.

End ImpDefs.
