# Bibliotecas utilizadas ----
pacman::p_load(readr, 
               tidyverse, 
               janitor,
               stringr,
               hrbrthemes,
               patchwork,
               ggridges,
               ggthemes,
               ggrepel,
               scales,
               viridis,
               gghighlight)


# Função que Padroniza nomes
padroniza_nomes <- function(x) {
  x |> 
    str_to_title() |> 
    str_replace_all("(?<=\\s)(\\b\\w{1,2}\\b)(?=\\s)", tolower)
}

# Carregamento de banco de dados ----
data_relat_16_24 <- 
  read_rds("bancos_de_dados/banco_de_dados_transicao/dados_distritais_eleitorais.rds") 

data_despesas_gdf_16_24 <-
  read_rds("bancos_de_dados/banco_de_dados_transicao/data_despesas_gdf_16_24.rds") 



# Resultados ----
# . ----
## Emendas distritais ----

### 1. Montante Geral das Emendas ----
data_relat_16_24 |> 
  summarise(across(c(aprovado, autorizado, empenhado, liquidado, pago), ~ sum(.x, na.rm = TRUE))) |> 
  mutate(across(where(is.numeric), ~ .x / 10^6)) # Convertendo para milhões

## Trazer informações orçamento geral do GDF

### 2. Montante ao Longo do Tempo (Longitudinal) ----
# Fig 1 relat
data_relat_16_24 |> 
  group_by(ano) |>  
  summarise(across(c(aprovado, autorizado, empenhado, liquidado, pago), ~ sum(.x, na.rm = TRUE))) |> 
  mutate(across(where(is.numeric), ~ .x / 10^6)) |> 
  pivot_longer(-ano, names_to = "fase", values_to = "valor") |> 
  mutate(fase=str_to_title(fase)) |>
  ggplot(aes(x = ano, y = valor, color = fase, group = fase)) +
  geom_line(size = .75) +
  labs(x = "", y = "Valor em Milhões", color = "") +
  scale_color_manual(values = c(
    
    "#000000",
    "#808080",
    #"#708090",
    "#4682B4",
    #1E90FF"
    "#FF8B3B",
    "#DCD6D6"#,
    #"#F76E6E"#,
    #"#F55151"
    )) +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "top")

# salvando figura x
ggsave("resultados/relatorio_observadf/totais_ano_df.png",
       width = 20, height = 12, units = "cm",dpi = 320)

### 3. Análise da Execução Contábil (Onde há Entraves?) ----
# Fig 2 relat
data_relat_16_24 |> 
  group_by(ano) |>  
  summarise(across(c(aprovado, autorizado, empenhado, liquidado, pago), ~ sum(.x, na.rm = TRUE))) |> 
  mutate(across(where(is.numeric), ~ .x / 10^6)) |> 
  mutate(
 #   `(%) Autorizado` = (autorizado / aprovado) * 100,
    `(%) Empenhado` = (empenhado / autorizado) ,
    `(%) Liquidado` = ( liquidado / autorizado),
    `(%) Pago` = (pago / autorizado) 
  ) |> 
  pivot_longer(cols = c(`(%) Empenhado`, `(%) Liquidado`, `(%) Pago`), 
               names_to = "fase", values_to = "percentual") |> 
  ggplot(aes(x = ano, y = percentual, fill = fase, group = fase)) +
  geom_bar(stat = "identity",position = "dodge",alpha=.75) +
  scale_y_percent(limits = c(0,1)) +
  labs(x = "Ano", y = "Percentual (%)", color = "Fase da Execução",fill="") +
  scale_fill_tableau(palette = "Classic Color Blind") +
#  facet_wrap(vars(fase),nrow = 1) +
  theme_ipsum() +
  theme(legend.position = "top",
        axis.text.x = element_text(angle = 45,hjust = 1,vjust=1))

# salvando figura x
ggsave("resultados/relatorio_observadf/execucao_ano_df.png",
       width = 20, height = 12, units = "cm",dpi = 320)

# Vendo emendas com % de pago maior que o esperado
emendas_com_execucao_atipica <-
  data_relat_16_24 |> 
  mutate(
    `(%) Execução` = (pago / autorizado) 
  ) |> 
  drop_na(pago,autorizado) |> 
  filter(`(%) Execução`>1)  |> filter(`(%) Execução`<100)

## 3.1 Natureza da despesa
data_relat_16_24 |> 
  group_by(ano,categoria_economica) |>  
  summarise(liquidado = sum(liquidado, na.rm = TRUE) / 10^6 ) |> 
  filter(liquidado>0) |> 
  mutate(natureza=str_to_title(categoria_economica)) |>
  ggplot(aes(x = ano, y = liquidado, color = natureza, 
             group = natureza)) +
  geom_line(size = .75) +
  labs(x = "", y = "Valor em Milhões", color = "") +
  scale_color_manual(values = c(
    
    "#000000",
    "#808080",
    #"#708090",
    "#4682B4",
    #1E90FF"
    "#FF8B3B",
    "#DCD6D6"#,
    #"#F76E6E"#,
    #"#F55151"
  )) +
  scale_color_tableau(palette = "Classic Color Blind") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "top")

# salvando figura x
ggsave("resultados/relatorio_observadf/natureza_totais_df.png",
       width = 20, height = 12, units = "cm",dpi = 320)



## 3.2 Natureza da despesa - multiplas fases
# Fig 3 relat
data_relat_16_24 |> 
  group_by(ano,categoria_economica) |>  
  summarise(across(c(# aprovado, #tirando aprovado
                     autorizado, empenhado, liquidado, pago), ~ sum(.x, na.rm = TRUE)/10^6)) |> 
  pivot_longer(cols = -c(ano, categoria_economica), names_to = "fase", values_to = "valor") |> 
  filter(valor>0) |> 
  mutate(natureza=str_to_title(categoria_economica),
         fase=str_to_title(fase)) |>
  ggplot(aes(x = ano, y = valor, color = fase,group=fase)) +
  geom_line(size = .75) +
  labs(x = "", y = "Valor em Milhões", color = "") +
  scale_color_manual(values = c(
    
 #   "#000000",
    "#808080",
    #"#708090",
    "#4682B4",
    #1E90FF"
    "#FF8B3B",
    "#DCD6D6"#,
    #"#F76E6E"#,
    #"#F55151"
  )) +
  facet_wrap(vars(natureza)) +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "top")

# salvando figura x
ggsave("resultados/relatorio_observadf/natureza_comparacao_df.png",
       width = 20, height = 12, units = "cm",dpi = 320)


investimento_emendas_cldf <- data_relat_16_24 |> 
  group_by(ano,grupo_de_natureza_da_despesa) |>  
  summarise(liquidado= sum(liquidado, na.rm = TRUE)) |> 
  mutate(liquidado = liquidado / 10^6) |> 
  filter(grupo_de_natureza_da_despesa == "Investimentos")

data_relat_16_24 |> 
  summarise(pago= sum(pago, na.rm = TRUE)/ 10^6,
            .by = c(ano,grupo_de_natureza_da_despesa)) |> 
  mutate(base = "Emendas") |> 
#  filter(grupo_de_natureza_da_despesa == "INVESTIMENTOS") |> 
  bind_rows( 
  data_despesas_gdf_16_24 |> 
  group_by(ano,grupo_de_natureza_da_despesa) |>  
  summarise(pago= sum(pago, na.rm = TRUE)/ 10^6) |> 
  mutate(base = "Total orçamento GDF") #|>
#  filter(grupo_de_natureza_da_despesa == "INVESTIMENTOS")
  ) |> 
  ggplot(aes(x=ano,y=pago,group=base,color=base)) +
  geom_line(stat = "identity") +
  facet_grid(cols=vars(grupo_de_natureza_da_despesa))  +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "top")
  
data_relat_16_24 |> 
  summarise(pago= sum(pago, na.rm = TRUE)/ 10^6,
            .by = c(ano,grupo_de_natureza_da_despesa)) |> 
  mutate(base = "Emendas") |> 
  #  filter(grupo_de_natureza_da_despesa == "INVESTIMENTOS") |> 
  bind_rows( 
    data_despesas_gdf_16_24 |> 
      group_by(ano,grupo_de_natureza_da_despesa) |>  
      summarise(pago= sum(pago, na.rm = TRUE)/ 10^6) |> 
      mutate(base = "Total orçamento GDF") #|>
    #  filter(grupo_de_natureza_da_despesa == "INVESTIMENTOS")
  ) |> 
  mutate(percent=prop.table(pago),
         .by = c(ano,grupo_de_natureza_da_despesa)) |> 
  filter(grupo_de_natureza_da_despesa %in%
           c("INVESTIMENTOS","OUTRAS DESPESAS CORRENTES"),
         base == "Emendas") |> 
  ggplot(aes(x=ano,y=percent,group=grupo_de_natureza_da_despesa,
             color=grupo_de_natureza_da_despesa)) +
  geom_line(stat = "identity") +
  scale_y_percent() +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "top")



### 4. Para Onde Vão Esses Valores? (Políticas Públicas Beneficiadas) ----

# lista de funcoes pra trabalhar
lista_funcoes_mais_relevantes <-
  data_relat_16_24 |> 
  summarise(pago=sum(pago,na.rm = T),
            .by = funcao) |>
  slice_max(order_by = pago, n = 4) |>
#  slice_max(order_by = pago, n = 6) |>
#  slice_max(order_by = pago, n = 9) |>
  pull(funcao)

nao_politicas_publicas <- c("Essencial à Justiça","Legislativa","Judiciária","Encargos Especiais")

# 1. Padroniza nomes
padroniza_nomes <- function(x) {
  x |> 
    str_to_title() |> 
    str_replace_all("(?<=\\s)(\\b\\w{1,2}\\b)(?=\\s)", tolower)
}

# 2. Lista de funções mais relevantes já tratadas
funcoes_ordenadas <- data_relat_16_24 |> 
  mutate(funcao = padroniza_nomes(funcao)) |> 
  filter(!(funcao %in% nao_politicas_publicas)) |> 
  summarise(pago = sum(pago, na.rm = TRUE) / (9 * 10^6), .by = funcao) |> 
  arrange(funcao) |> 
  pull(funcao)

# 3. Extração da paleta diretamente do ggthemes
cores_tableau_20 <- ggthemes::tableau_color_pal("Tableau 20")(20)

# 4. Criação do vetor nomeado com as cores corretas
cores_funcoes <- setNames(cores_tableau_20[1:length(funcoes_ordenadas)], funcoes_ordenadas)


# Filtrando funcoes legislativa e judiciario pra um novo banco
funcoes_leg_jud <-
  data_relat_16_24 |> 
  filter(funcao %in% nao_politicas_publicas)

# 4.1 total
# 4.1.1.1 - função

# Fig 4 relat
cima <- data_relat_16_24 |> 
  filter(!(funcao %in% nao_politicas_publicas)) |> 
  summarise(pago = sum(pago, na.rm = TRUE)/(9*10^6),
            .by=funcao) |>
  slice_max(order_by = pago, n = 20) |>
  ggplot(aes(y = reorder(funcao, pago), x = pago, fill = funcao)) +
  geom_col(alpha=0.7) +
  geom_text(aes(label = round(pago,1)),hjust=-0.1,size=3) +
  expand_limits(x = 95) +
  labs(y = "", x = "") +
  ggthemes::scale_fill_tableau(palette = "Tableau 20") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "none",
        plot.margin = margin(0.2, 0.2, 0.2, 0.2, "cm"))


baixo <-
  data_relat_16_24 |> 
  filter(!(funcao %in% nao_politicas_publicas)) |> 
  summarise(pago = sum(pago, na.rm = TRUE)/ (9*10^6),
            .by=c(funcao,grupo_de_natureza_da_despesa)) |> 
  mutate(ordem=sum(pago,na.rm=T),
         .by = funcao) |> 
  filter(grupo_de_natureza_da_despesa %in% c("Investimentos","Outras Despesas Correntes")) |> 
  arrange(desc(ordem),grupo_de_natureza_da_despesa) |> #print(n=44)
  slice_max(order_by = pago, n = 38) |>
  ggplot(aes(y = reorder(funcao, ordem), x = pago, fill = funcao)) +
  geom_col(alpha=0.7) +
  geom_text(aes(label = round(pago,1)),hjust=-0.1,size=3) +
  expand_limits(x = 95) +
  labs(y = "", x = "Valor Pago (em milhões)") +
  facet_wrap(vars(grupo_de_natureza_da_despesa)) +
  ggthemes::scale_fill_tableau(palette = "Tableau 20") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "none",
        plot.margin = margin(0.2, 0.2, 0.2, 0.2, "cm"))

cima / baixo

# salvando figura 4
ggsave("resultados/relatorio_observadf/funcoes_totais_df.png",
       width = 24, height = 28, units = "cm",dpi = 320)



# funcoes mais relevantes por ano

data_relat_16_24 |> 
  filter(funcao %in% lista_funcoes_mais_relevantes) |> 
  mutate(
    ano = as.numeric(ano),
    natureza = grupo_de_natureza_da_despesa
  ) |>
  summarise(pago = sum(pago, na.rm = TRUE), .by = c(funcao, natureza, ano)) |> 
  mutate(pago = pago / 10^6) |> 
  filter(natureza %in% c("Investimentos", "Outras Despesas Correntes")) |> 
  mutate(ordem = sum(pago, na.rm = TRUE), .by = funcao) |> 
  
  ggplot(aes(x = ano, y = pago, color = funcao, group = funcao)) +
  
  geom_line(size = 0.3, alpha = 0.8) +
#  geom_smooth(method = "gam", formula = y ~ s(x, bs = "cs"), se = FALSE, size = 1.5) +
  geom_smooth(se=F) +
  # Rótulo ao final da linha
  geom_text_repel(
    data = ~ .x |> group_by(funcao, natureza) |> filter(ano == max(ano)),
    aes(label = funcao),
    nudge_x = 0,
    hjust = 0,
    fontface="bold",
    family="Arial Narrow",
    direction = "y",
    box.padding = 0.2,
    segment.color = NA,
    size = 3
  ) +
  
  scale_color_manual(values=cores_funcoes) +
  facet_wrap(vars(natureza),ncol=2) +
  labs(y = "Valores Pagos (em milhões)", x = NULL, color = NULL) +
  scale_x_continuous(expand = expansion(mult = c(0.01, 0.15)),breaks = seq(2016,2024,by=1)) +
  theme_ipsum() +
  theme(
    legend.position = "none",
    strip.text = element_text(face = "bold"),
    plot.margin = margin(0.2, 0.2, 0.2, 0.4, "cm"),
    strip.background = element_blank(),  # Fundo dos títulos dos facets
    panel.spacing.x = unit(1, "lines"),  # Controla o espaço horizontal entre os facets
    panel.border = element_rect(color = "black", fill = NA, size = 1)  # Adiciona borda ao redor dos painéis
  )

# salvando figura x
ggsave("resultados/relatorio_observadf/funcoes_principais_ano_df.png",
       width = 20, height = 12, units = "cm",dpi = 320)


# 4.1.1.2 - função execucao
# Fig 5 relat
data_relat_16_24 |> 
  summarise(across(c(# aprovado, #tirando aprovado
    autorizado, empenhado, liquidado, pago), ~ sum(.x, na.rm = TRUE)),
    .by=funcao) |> 
  mutate(
    #   `(%) Autorizado` = (autorizado / aprovado) * 100,
    `(%) Empenhado` = (empenhado / autorizado) ,
    `(%) Liquidado` = ( liquidado / autorizado) ,
    `(%) Pago` = (pago / autorizado),
    ordem = (empenhado / autorizado)
  ) |> 
  slice_max(order_by = pago, n = 9) |>
  dplyr::select(-c(autorizado, empenhado, liquidado, pago)) |> 
  pivot_longer(cols = c(`(%) Empenhado`, `(%) Liquidado`, `(%) Pago`), 
               names_to = "fase", values_to = "percentual") |> 
  
  ggplot(aes(y = percentual, x = reorder(str_wrap(funcao,15),-ordem), fill = fase,group = fase)) +
  geom_hline(yintercept = 0.75,linetype="dotted",color="grey50") +
  geom_col(alpha=0.7,position = "dodge") +
  scale_fill_tableau(palette = "Classic Color Blind") +
  scale_y_percent(limits = c(0,1)) +
  labs(y = "", x = "",fill="") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "top",
        plot.margin = margin(0.2, 0.2, 0.2, 0.2, "cm"))# -> cima_exec


# data_relat_16_24 |> 
#   mutate(funcao=str_to_title(funcao),  # Converte tudo para título
#          funcao = str_replace_all(funcao, "(?<=\\s)(\\b\\w{1,2}\\b)(?=\\s)", tolower),  # Minúsculas para palavras pequenas
#          natureza=str_to_title(grupo_de_natureza_da_despesa),  # Converte tudo para título
#          natureza = str_replace_all(natureza, "(?<=\\s)(\\b\\w{1,2}\\b)(?=\\s)", tolower)
#   ) |>
#   group_by(funcao,natureza) |>  
#   summarise(pago = sum(pago, na.rm = TRUE)) |> 
#   mutate(pago = pago / 10^6,
#          pago=pago/9) |> 
#   filter(natureza %in% c("Investimentos","Outras Despesas Correntes")) |> 
#   arrange(desc(pago)) |> 
#   head(20) |> 
#   ggplot(aes(y = reorder(funcao, pago), x = pago, fill = funcao)) +
#   geom_col(alpha=0.7) +
#   geom_text(aes(label = round(pago,1)),hjust=-0.1,size=3) +
#   expand_limits(x = 95) +
#   labs(y = "", x = "Valor Pago (em milhões)") +
#   facet_wrap(vars(natureza)) +
#   ggthemes::scale_fill_tableau(palette = "Tableau 20") +
#   hrbrthemes::theme_ipsum() +
#   theme(legend.position = "none",
#         plot.margin = margin(0.2, 0.2, 0.2, 0.2, "cm")) #-> baixo_exec

#cima_exec / baixo_exec

# salvando figura x
ggsave("resultados/relatorio_observadf/funcoes_execucao_df.png",
       width = 28, height = 12, units = "cm",dpi = 320)

# 4.1.1.3 - distribuicao por emenda
data_relat_16_24 |> 
  filter(funcao %in% lista_funcoes_mais_relevantes) |> 
  mutate(pago_tot = sum(pago, na.rm = TRUE),
    .by=funcao) |> 
  mutate(
    `(%) Execução` = (pago / autorizado) 
  ) |> 
  drop_na(pago,autorizado) |> 
  filter(`(%) Execução`>=0,`(%) Execução`<=1) |> 
  ggplot(aes(x=`(%) Execução`, y = reorder(funcao,pago_tot), fill = funcao,group = funcao)) +
  geom_density_ridges(alpha=0.6,aes(color=funcao)) +
  scale_x_percent() +
  labs(y = "", x = "") +
  ggthemes::scale_fill_tableau(palette = "Tableau 20") +
  ggthemes::scale_color_tableau(palette = "Tableau 20") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "none",
        plot.margin = margin(0.2, 0.2, 0.2, 0.2, "cm"))


# 4.1.2 - funcao e programa
data_relat_16_24 |> 
  unite(funcao_programa,c(funcao,programa),sep=' - ',
        remove = FALSE) |> 
  summarise(pago = sum(pago, na.rm = TRUE)/(9* 10^6),
            .by=c(funcao_programa,funcao)) |> 
  slice_max(order_by = pago, n = 20) |>
  ggplot(aes(y = reorder(funcao_programa, pago), 
             x = pago, fill = funcao)) +
  geom_col(alpha=0.7) +
  geom_text(aes(label = round(pago,1)),hjust=-0.1,size=3) +
  scale_x_continuous(expand = expansion(mult = c(0, 0.1))) +
  labs(y = "", x = "Valor Pago (em milhões)",
       fill="") +
  hrbrthemes::theme_ipsum() +
  scale_fill_manual(values=cores_funcoes) +
  scale_color_manual(values=cores_funcoes) +
  theme(legend.position = "right",
        plot.margin = margin(0.2, 0.2, 0.2, 0.2, "cm"))

# salvando figura x
ggsave("resultados/relatorio_observadf/funcao_programa_ano_df.png",
       width = 24, height = 20, units = "cm",dpi = 320)


# 4 maiores programas das 4 funcoes mais relevantes
programas_selecao <-
  data_relat_16_24 |> 
  filter(funcao %in% lista_funcoes_mais_relevantes) |> 
  summarise(pago = sum(pago, na.rm = TRUE)/(9* 10^6),
            .by=c(funcao,programa,ano)) |> 
  summarise(media = mean(pago, na.rm = TRUE),
            .by=c(funcao,programa)) |> 
  mutate(rank = rank(-media),
         .by=funcao) |> 
  filter(rank <= 4) |> 
  unite(funcao_programa,c(funcao,programa),sep=' - ',
        remove = FALSE) |> 
  pull(funcao_programa)

# funcoes mais relevantes por ano

data_relat_16_24 |> 
  mutate(
    ano = as.numeric(ano)
  ) |>
  unite(funcao_programa,c(funcao,programa),sep=' - ',
        remove = FALSE) |> 
  filter(funcao_programa %in% programas_selecao) |> 
  summarise(pago = sum(pago, na.rm = TRUE)/ 10^6, 
            .by = c(funcao, programa, ano)) |> 
  ggplot(aes(x = ano, y = pago, color = funcao, group = programa)) +
  
  geom_line(size = 0.3, alpha = 0.8) +
  #  geom_smooth(method = "gam", formula = y ~ s(x, bs = "cs"), se = FALSE, size = 1.5) +
  geom_smooth(se=F) +
  # Rótulo ao final da linha
  geom_text_repel(
    data = ~ .x |> group_by(funcao, programa) |> filter(ano == max(ano)),
    aes(label = programa),
    nudge_x = 0,
    hjust = 0,
    fontface="bold",
    family="Arial Narrow",
    direction = "y",
    box.padding = 0.2,
    segment.color = NA,
    size = 3
  ) +
  
  scale_color_manual(values=cores_funcoes) +
  facet_wrap(vars(funcao),ncol=2) +
  labs(y = "Valores Pagos (em milhões)", x = NULL, color = NULL) +
  scale_x_continuous(expand = expansion(mult = c(0.01, 0.15)),breaks = seq(2016,2024,by=1)) +
  theme_ipsum() +
  theme(
    legend.position = "none",
    strip.text = element_text(face = "bold"),
    plot.margin = margin(0.2, 0.2, 0.2, 0.4, "cm"),
    strip.background = element_blank(),  # Fundo dos títulos dos facets
    panel.spacing.x = unit(1, "lines"),  # Controla o espaço horizontal entre os facets
    panel.border = element_rect(color = "black", fill = NA, size = 1)  # Adiciona borda ao redor dos painéis
  )

# salvando figura x
ggsave("resultados/relatorio_observadf/maiores_programa_maiores_funcoes_df.png",
       width = 28, height = 24, units = "cm",dpi = 320)

# 4.1.3 - programa
data_relat_16_24 |> 
  group_by(programa) |>  
  summarise(pago = sum(pago, na.rm = TRUE)) |> 
  mutate(pago = pago / 10^6,
         pago=pago/9) |> 
  arrange(desc(pago)) |> 
  head(20) |> 
  ggplot(aes(y = reorder(programa, pago), x = pago, fill = programa)) +
  geom_col(alpha=0.7) +
  geom_text(aes(label = round(pago,1)),hjust=-0.1,size=3) +
  expand_limits(x = 95) +
  labs(y = "Programa", x = "Valor Pago (em milhões)") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "none")


# 4.1.4 - subfunção
data_relat_16_24 |> 
  group_by(subfuncao) |>  
  summarise(pago = sum(pago, na.rm = TRUE)) |> 
  mutate(pago = pago / 10^6,
         pago=pago/9) |> 
  arrange(desc(pago)) |> 
  head(20) |> 
  ggplot(aes(y = reorder(subfuncao, pago), x = pago, fill = subfuncao)) +
  geom_col(alpha=0.7) +
  geom_text(aes(label = round(pago,1)),hjust=-0.1,size=3) +
  expand_limits(x = 95) +
  labs(y = "Subfunção", x = "Valor Pago (em milhões)") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "none")

# salvando figura x
ggsave("resultados/relatorio_observadf/subfuncoes_totais_df.png",
       width = 20, height = 12, units = "cm",dpi = 320)



# 4.2 por funcoes por ano
data_relat_16_24 |> 
  dplyr::filter(funcao %in% lista_funcoes_mais_relevantes) |>
  group_by(funcao,ano) |>  
  summarise(pago = sum(pago, na.rm = TRUE)) |> 
  mutate(pago = pago / 10^6) |> 
  ggplot(aes(y = reorder(funcao, pago), x = pago, fill = funcao)) +
  geom_col(alpha=0.7) +
  geom_text(aes(label = round(pago,1)),hjust=-0.1,size=3) +
  facet_wrap(vars(ano)) +
  labs(y = "Política Pública", x = "Valor Pago (em milhões)") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "none")

# salvando figura x
ggsave("resultados/relatorio_observadf/funcoes_totais_df_ano.png",
       width = 20, height = 12, units = "cm",dpi = 320)

### 5. Quais Órgãos Recebem os Recursos? ----
data_relat_16_24 |>
  mutate(orgao = str_replace_all(orgao,"[Dd]f","DF")) |> 
  group_by(orgao) |>  
  summarise(pago = sum(pago, na.rm = TRUE)) |> 
  mutate(pago = pago / (9*10^6)) |> 
  arrange(desc(pago)) |> 
  head(10) |> 
  ggplot(aes(y = reorder(orgao, pago), x = pago, fill = orgao)) +
  geom_col(alpha=0.7) +
  geom_text(aes(label = round(pago,1)),hjust=-0.1,size=3) +
  scale_x_continuous(expand = expansion(mult = c(0, 0.1))) +
  ggthemes::scale_fill_tableau(palette = "Tableau 20") +
  labs(y = "Órgão", x = "Valor Pago (em milhões)") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "none")

ggsave("resultados/relatorio_observadf/secretarias_totais_df.png",
       width = 20, height = 12, units = "cm",dpi = 320)



# 5.2 Execução por orgao 
data_relat_16_24 |>
  mutate(orgao = str_replace_all(orgao,"[Dd]f","DF")) |> 
  summarise(across(c(# aprovado, #tirando aprovado
    autorizado, empenhado, liquidado, pago), ~ sum(.x, na.rm = TRUE)),
    .by=orgao) |> 
  mutate(
    #   `(%) Autorizado` = (autorizado / aprovado) * 100,
    `(%) Empenhado` = (empenhado / autorizado) ,
    `(%) Liquidado` = ( liquidado / autorizado) ,
    `(%) Pago` = (pago / autorizado),
    ordem = (empenhado / autorizado)
  ) |> 
  slice_max(order_by = pago, n = 10) |>
  dplyr::select(-c(autorizado, empenhado, liquidado, pago)) |> 
  pivot_longer(cols = c(`(%) Empenhado`, `(%) Liquidado`, `(%) Pago`), 
               names_to = "fase", values_to = "percentual") |> 
  
  ggplot(aes(y = percentual, x = reorder(str_wrap(orgao,15),-ordem), 
             fill = fase,group = fase)) +
  geom_hline(yintercept = 0.75,linetype="dotted",color="grey50") +
  geom_col(alpha=0.7,position = "dodge") +
  scale_fill_tableau(palette = "Classic Color Blind") +
  scale_y_percent(limits = c(0,1)) +
  labs(y = "", x = "",fill="") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "top",
        plot.margin = margin(0.2, 0.2, 0.2, 0.2, "cm"))

# salvando figura x
ggsave("resultados/relatorio_observadf/orgaos_execucao_df.png",
       width = 28, height = 12, units = "cm",dpi = 320)

# 5.3 Orgaos por natureza
data_relat_16_24 |>
  mutate(orgao = str_replace_all(orgao,"[Dd]f","DF")) |> 
  summarise(pago = sum(pago, na.rm = TRUE),
            .by = c(orgao,)) |> 
  mutate(pago = pago / (9*10^6)) |> 
  arrange(desc(pago)) |> 
  head(10) |> 
  ggplot(aes(y = reorder(orgao, pago), x = pago, fill = orgao)) +
  geom_col(alpha=0.7) +
  geom_text(aes(label = round(pago,1)),hjust=-0.1,size=3) +
  scale_x_continuous(expand = expansion(mult = c(0, 0.1))) +
  ggthemes::scale_fill_tableau(palette = "Tableau 20") +
  labs(y = "Órgão", x = "Valor Pago (em milhões)") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "none")

ggsave("resultados/relatorio_observadf/secretarias_totais_df.png",
       width = 20, height = 12, units = "cm",dpi = 320)

### 6. Quais Regiões Administrativas (RAs) são Contempladas? ----

#### 6.1 Valores localizaveis no espaço

# Criar variável indicando se o órgão é uma Administração Regional 

data_relat_16_24 |> 
  mutate(admin_regional = ifelse(str_detect(orgao, "Administração Regional"),
                                 "Sim", "Não")) |>  
  summarise(pago = sum(pago, na.rm = TRUE),
            .by=admin_regional) |> 
  mutate(pago = pago / (9*10^6))


data_relat_16_24 |> 
    group_by(RA) |>  
    summarise(pago = sum(pago, na.rm = TRUE)) |> 
    mutate(pago = pago / 10^6) |> 
    arrange(desc(pago)) |> 
    head(10) |> 
    ggplot(aes(x = reorder(RA, pago), y = pago, fill = RA)) +
    geom_col() +
    coord_flip() +
    labs(x = "Região Administrativa", y = "Valor Pago (em milhões)", title = "Top 10 RAs Beneficiadas") +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "none")

### 7. Variações Políticas na aplicação (governo e oposição)  ----
data_relat_16_24 |> 
  filter(situacao_autor_1 %in% c("Governo","Oposição"),
         apenas_um_autor==1) |> 
 # drop_na(situacao_autor_1) |> 
  summarise(across(c(# aprovado, #tirando aprovado
    autorizado, empenhado, liquidado, pago), ~ sum(.x, na.rm = TRUE)),
              .by=c(ano,situacao_autor_1)) |> 
  mutate(
 #   `(%) Autorizado` = (autorizado / aprovado) * 100,
    `(%) Empenhado` = (empenhado / autorizado) ,
    `(%) Liquidado` = ( liquidado / empenhado) ,
    `(%) Pago` = (pago / liquidado) 
  ) |> 
  dplyr::select(-c(autorizado, empenhado, liquidado, pago)) |> 
  pivot_longer(cols = c(`(%) Empenhado`, `(%) Liquidado`, `(%) Pago`), 
               names_to = "fase", values_to = "percentual") |> 
  
    ggplot(aes(x = ano, y = percentual, 
               group=situacao_autor_1,
               color = situacao_autor_1,
               fill = situacao_autor_1)) +
    geom_smooth() +
    ggthemes::scale_colour_tableau() +
    ggthemes::scale_fill_tableau() +
    scale_y_percent(limits = c(.4, 1.2), breaks = seq(0, 1, by = .1)) + 
    labs(x = "", y = "", color="",fill="") +
    facet_grid(cols=vars(fase)) +
    hrbrthemes::theme_ipsum() +
    theme(legend.position = "top",
          axis.text.x = element_text(angle = 45, hjust = 1))

# ggsave("resultados/relatorio_observadf/funcoes_percentual_oposi_gov.png",
#        width = 20, height = 12, units = "cm",dpi = 320)

# 7.1 Partido e funcao
#mandato_selecionado <- "Rollemberg I (2016-2018)"
#mandato_selecionado <- "Ibaneis I (2019-2022)" 
mandato_selecionado <- "Ibaneis II (2023-2024)"
  
data_relat_16_24 |> 
  filter(titulos == mandato_selecionado) |> 
  filter(situacao_autor_1 %in% c("Governo","Oposição"),
         apenas_um_autor==1) |> 
  
  # drop_na(situacao_autor_1) |> 
  summarise(pago = sum(pago, na.rm = TRUE),
    .by=c(#ano,
          situacao_autor_1,funcao)) |> 
  mutate(percent = prop.table(pago),
         .by = c(#ano,
                 situacao_autor_1)) |> 
  mutate(media_funcao = mean(percent,na.rm=T),
         .by=funcao) |> 
  filter(!(funcao %in% nao_politicas_publicas)) |> 
  ggplot() +
  geom_point(aes(y = reorder(funcao,percent),
                 x=percent,color=situacao_autor_1,
                 group = situacao_autor_1),
             shape=15,
             size=4,alpha=0.75) +
  
  geom_linerange(data = ~.x |> 
                   pivot_wider(id_cols = funcao,
                               names_from = situacao_autor_1,
                               values_from = percent,
                               values_fill = 0) |> 
                   rowwise() |> # operar o mutate por linhas
                   mutate(min=min(`Governo`,`Oposição`), #menor valor entre Gov e Op
                          max=max(`Governo`,`Oposição`), #maior valor entre Gov e Op
                          maior_percentual = ifelse(`Governo`>`Oposição`,"Governo","Oposição"),
                          range = max-min, # diferença percentual
                          mean=mean(c(min,max))), #posicao do rotulo da diferença
                 aes(y = reorder(funcao,max),color=maior_percentual,
                     xmin=min, xmax=max),size=1,alpha=.75, show.legend = FALSE) +
  

 
  geom_text(data = ~.x |> 
                   pivot_wider(id_cols = funcao,
                               names_from = situacao_autor_1,
                               values_from = percent,
                               values_fill = 0) |> 
                   rowwise() |> # operar o mutate por linhas
                   mutate(min=min(`Governo`,`Oposição`), #menor valor entre Gov e Op
                          max=max(`Governo`,`Oposição`), #maior valor entre Gov e Op
                          maior_percentual = ifelse(`Governo`>`Oposição`,"Governo","Oposição"),
                          range = max-min, # diferença percentual
                          mean=mean(c(min,max))), #posicao do rotulo da diferença
                 aes(y = reorder(funcao,max),x=mean,color=maior_percentual, 
                     label=ifelse(range > 0.03, paste0(#"\U0394",#delta
                                                       "+",
                                                       round(100 * range, 0),"%"),
                                                       "")),size=3.25,
            family="Arial Narrow",vjust=1.25,fontface="bold", show.legend = FALSE) +
  ggthemes::scale_colour_tableau() +
  labs(x = "% de valores pagos em Políticas Públicas", 
       y = "", color="",fill="",title = paste("Governo",mandato_selecionado)) +
  #facet_wrap(vars(mandato)) +
  scale_x_percent(breaks = seq(0, 1, by = .05),
                  expand = expansion(mult = c(0.05, 0.1)),
                  sec.axis = dup_axis(name = NULL)) +
  
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "top",
        plot.margin = margin(0.2, 0.2, 0.2, 0.2, "cm"),
        panel.grid.minor.x=element_blank())


# salvando figura com nome do mandato no arquivo
ggsave(
  filename = glue::glue("cestas_pol_pub_gov_op_{str_replace_all(str_sub(mandato_selecionado,end=-13), '\\\\s+', '_')}.png"),
  path = "resultados/relatorio_observadf",
  width = 20, 
  height = 18, 
  units = "cm",
  dpi = 320
)

# fig |> 
#   pivot_wider(id_cols = funcao,
#               names_from = situacao_autor_1,
#               values_from = percent,
#               values_fill = 0) 


# 7.2 Valores e percentuais por quantidade de autores
data_relat_16_24 |> 
  rowwise() |>  # Garante que a operação seja feita linha a linha
  mutate(qtd_autores = sum(!is.na(c_across(c(partido_autor_1, partido_autor_2, partido_autor_3))))) |> 
  ungroup() |>  # Remove agrupamento caso necessário
  #filter(qtd_autores>0) |> 
  summarise(pago = sum(pago, na.rm = TRUE),
            .by = qtd_autores) |> mutate(percent=100*prop.table(pago)) |> 
  arrange(qtd_autores)

### 8 Executivo vs Legislativo ----

# vetor de funções presentes nas emendas
funcoes_emendas <- data_relat_16_24|>
                            summarise(liquidado=sum(liquidado,na.rm=T),
                                      .by=funcao) |> 
                   filter(liquidado > 0)|> 
                   pull(funcao)

data_relat_16_24 |> 
  summarise(liquidado = sum(liquidado, na.rm = TRUE),
            .by=c(funcao,titulos)) |> 
  mutate(base = "Legislativo (Emendas)",
         percent = prop.table(liquidado),
         .by=titulos) |> 
  
  bind_rows( 
    data_despesas_gdf_16_24 |> 
      filter(funcao %in% funcoes_emendas) |> 
      summarise(liquidado = sum(liquidado, na.rm = TRUE),
                .by=c(funcao,titulos)) |>  
      mutate(base = "Executivo (Despesa)",
             percent = prop.table(liquidado),
             .by=titulos)
  ) |> 
  mutate(media_funcao = mean(percent,na.rm=T),
         .by=funcao) |> 
  filter(!(funcao %in% nao_politicas_publicas)) |>  
  ggplot(aes(y = reorder(str_wrap(funcao,15),percent),
               x=percent,color=base,
               group = base)) +
    geom_point(size=5,alpha=0.5) +
    geom_text(aes(label=round(100*percent,0)),
              color="black",size=2.5,fontface="bold") +
    ggthemes::scale_colour_tableau(palette = "Nuriel Stone") +
    labs(x = "% de valores pagos em Políticas Públicas", 
       y = "", color="",fill="") +
    facet_wrap(vars(titulos)) +
  scale_x_percent(expand = expansion(mult = c(0.05, 0.1))) +
    hrbrthemes::theme_ipsum() +
    theme(legend.position = "top",
          plot.margin = margin(0.2, 0.2, 0.2, 0.2, "cm"),
          panel.grid.major.y = element_line(color = alpha("grey50", 0.2), size = 4),
          strip.background = element_blank(),  # Fundo dos títulos dos facets
          strip.text = element_text(face = "bold"),
          panel.spacing.x = unit(1, "lines"),  # Controla o espaço horizontal entre os facets
          panel.border = element_rect(color = "black", fill = NA, size = 1)  # Adiciona borda ao redor dos painéis
    )


# salvando figura x
ggsave("resultados/relatorio_observadf/cestas_pol_pub_legis_gov.png",
       width = 20, height = 18, units = "cm",dpi = 320)


# 8.1 Cestas de PP - Natureza da despesa
#mandato_selecionado <- "Rollemberg I (2016-2018)"
#mandato_selecionado <- "Ibaneis I (2019-2022)" 
mandato_selecionado <- "Ibaneis II (2023-2024)"

bind_rows(
# Investimentos emendas  
  data_relat_16_24 |> 
  mutate(natureza = grupo_de_natureza_da_despesa,
         base = "Legislativo (Emendas)",
         grupo_de_natureza_da_despesa = NULL) |> 
  filter(natureza == "Investimentos") |> 
  summarise(liquidado = sum(liquidado, na.rm = TRUE),
            .by=c(funcao,natureza,base,titulos)) |> 
    mutate(percent = prop.table(liquidado),
           .by=c(natureza,base,titulos)),
  
# Despesas correntes emendas
  data_relat_16_24 |> 
  mutate(natureza = grupo_de_natureza_da_despesa,
         base = "Legislativo (Emendas)",
         grupo_de_natureza_da_despesa = NULL) |> 
  filter(natureza == "Outras Despesas Correntes") |> 
  summarise(liquidado = sum(liquidado, na.rm = TRUE),
            .by=c(funcao,natureza,base,titulos)) |> 
  mutate(percent = prop.table(liquidado),
         .by=c(natureza,base,titulos)),

# Investimentos executivo  
data_despesas_gdf_16_24 |> 
  filter(funcao %in% funcoes_emendas) |> 
  mutate(natureza = grupo_de_natureza_da_despesa,
         base = "Executivo (Despesa)",
         grupo_de_natureza_da_despesa = NULL) |> 
  filter(natureza == "Investimentos") |> 
  summarise(liquidado = sum(liquidado, na.rm = TRUE),
            .by=c(funcao,natureza,base,titulos)) |> 
  mutate(percent = prop.table(liquidado),
         .by=c(natureza,base,titulos)),

# Despesas correntes executivo
data_despesas_gdf_16_24 |> 
  filter(funcao %in% funcoes_emendas) |> 
  mutate(natureza = grupo_de_natureza_da_despesa,
         base = "Executivo (Despesa)",
         grupo_de_natureza_da_despesa = NULL) |> 
  filter(natureza == "Outras Despesas Correntes") |> 
  summarise(liquidado = sum(liquidado, na.rm = TRUE),
            .by=c(funcao,natureza,base,titulos)) |> 
  mutate(percent = prop.table(liquidado),
         .by=c(natureza,base,titulos))

) |> 
  mutate(filtro_funcoes = mean(percent, na.rm = TRUE),
            .by=funcao) |> 
  filter(titulos == mandato_selecionado) |> 
  filter(!(funcao %in% nao_politicas_publicas)) |> 
  ggplot(aes(y = reorder(funcao,percent),
             x=percent,color=base,
             group = base)) +
  geom_point(size=5.5,alpha=0.75,#shape="square"#aes(shape = base)
             ) +
  geom_text(aes(label = ifelse(percent > 0.024, round(100 * percent, 0), "")),
            color = "black",
            size = 2.6, fontface = "bold") + 
  ggthemes::scale_colour_tableau(palette = "Nuriel Stone") +
  labs(x = "% de valores pagos em Políticas Públicas", 
       y= "",
       title  = paste("Governo",mandato_selecionado), color="",fill="") +
  facet_wrap(vars(natureza)) +
  scale_x_percent(#breaks = seq(0, 1, by = .1),
                  expand = expansion(mult = c(0.05, 0.05))) +
  hrbrthemes::theme_ipsum() +
  theme(legend.position = "top",
        plot.margin = margin(0.2, 0.2, 0.2, 0.2, "cm"),
        panel.grid.major.y = element_line(color = alpha("grey50", 0.2), size = 4),
    strip.background = element_blank(),  # Fundo dos títulos dos facets
    strip.text = element_text(face = "bold"),
    panel.spacing.x = unit(1, "lines"),  # Controla o espaço horizontal entre os facets
    panel.border = element_rect(color = "black", fill = NA, size = 1)  # Adiciona borda ao redor dos painéis
  )


# salvando figura com nome do mandato no arquivo
ggsave(
  filename = glue::glue("cestas_pol_natureza_exec_leg_{str_replace_all(mandato_selecionado, '\\\\s+', '_')}.png"),
  path = "resultados/relatorio_observadf",
  width = 20, 
  height = 16, 
  units = "cm",
  dpi = 320
)

