library(deSolve)

## function that describes changes in total pool size of source population ie naive Tregs
sp_numbers <- function(ts_seq) {
  t0 <- 40.0
  basl <- 6.2
  nu <- 0.027
  # peripheral naive Tregs numbers
  return(10^basl * exp(-nu * (ts_seq - t0)))
}

## function to evaluate daily influx from source into target population
theta_spline <- function(ts_seq, psi) {
  # psi is the rate of daily influx
  value <- (psi * sp_numbers(ts_seq))
  return(value)
}

## ode for fast and slow compartment dynamics
ode_func <- function (Time, y, parms) {
  with(as.list(c(y, parms)),{
   
    dy1 <- theta_spline(Time, psi) - lambda_f * y1 ## fast
    dy2 <- mu * y1 - lambda_s * y1 ## slow
    
    list(c(dy1, dy2))
  })
}



### Extract parameter samples as a data frame
par_dist <- read.csv(file = "par_est_samples.csv")

## time seq for predictions
ts_pred1 <- 10^seq(log10(66), log10(300), length.out = 300)

## solution of ode across par samples
fast_df <- data.frame()
slow_df <- data.frame()
for (i in 1:nrow(par_dist)){
  
  ## initial conditions
  init_cond <- c("y1" = exp(par_dist$y3_0[i]),
                 "y2" = exp(par_dist$y4_0[i]))
  
  ## pars estimated from model fit
  params <- c("psi" = par_dist$psi[i], 
              "lambda_s" = par_dist$lambda_I[i],
              "lambda_f" = par_dist$lambda_D[i],
              "mu" = par_dist$mu[i])
  
  ## solutions
  sol_df <- ode(y = init_cond, times = ts_pred1, func = ode_func, parms = params)
  fast_df[i, 1:length(ts_pred1)] <- sol_df[ , 2]
  slow_df[i, 1:length(ts_pred1)] <- sol_df[ , 3]
}

slow_plot <- slow_df %>%
  gather(factor_key = TRUE) %>%
  group_by(key) %>%
  summarize(lb = quantile(value, probs = 0.05),
            y_pred = quantile(value, probs = 0.5),
            ub = quantile(value, probs = 0.95)) %>%
  bind_cols("timeseq" = ts_pred1) 


fast_plot <- fast_df %>%
  gather(factor_key = TRUE) %>%
  group_by(key) %>%
  summarize(lb = quantile(value, probs = 0.05),
            y_pred = quantile(value, probs = 0.5),
            ub = quantile(value, probs = 0.95)) %>%
  bind_cols("timeseq" = ts_pred1) 


## blue is fast, pink is slow
ggplot() +
geom_line(data = slow_plot,
          aes(x = timeseq, y = y_pred), col = "#BE749E") +
geom_ribbon(data = slow_plot, 
            aes(x = timeseq, ymin = lb, ymax = ub), fill = "#BE749E", alpha=0.2) +
geom_line(data = fast_plot,
          aes(x = timeseq, y = y_pred), col = "#004594") +
geom_ribbon(data = fast_plot, 
            aes(x = timeseq, ymin = lb, ymax = ub), fill = "#004594", alpha=0.2) +
scale_y_continuous(trans = "log10",
                   minor_breaks = log10minorbreaks, 
                   labels = fancy_scientific) +
scale_x_log10()+
my_theme +
guides(col="none", fill="none") +
theme(
  legend.position = c(0.18, 0.86)
)




